/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.type;

import static org.apache.lucene.util.NumericUtils.*;

import java.text.DateFormat;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.document.NumericField;
import org.apache.lucene.document.Field.Index;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.index.Term;
import org.mtzky.lucene.descriptor.LucenePropertyConfig;

/**
 * @author mtzky
 */
@LuceneFieldStrategyType(Date.class)
public class LuceneDateStrategy implements LuceneFieldStrategy {

	private final String name;
	private final Store store;
	private final Index index;
	private final boolean original;
	private final String format;

	/**
	 * @param config
	 */
	public LuceneDateStrategy(final LucenePropertyConfig config) {
		this.name = config.getName();
		this.store = config.getStore();
		this.index = config.getIndex();
		this.format = config.getFormat();
		this.original = format.isEmpty();
	}

	@Override
	public Fieldable getField(final Object value) {
		if (original) {
			final boolean index = this.index != Index.NO;
			final NumericField field = new NumericField(name, store, index);
			field.setLongValue(toLong(value));
			return field;
		}
		return new Field(name, toString(value), store, index);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(final Fieldable field) {
		final String v = field.stringValue();
		if (original) {
			return (T) new Date(Long.parseLong(v));
		}
		final DateFormat format = new SimpleDateFormat(this.format);
		return (T) format.parse(v, new ParsePosition(0));
	}

	@Override
	public Term getTerm(final Object value) {
		if (original) {
			return new Term(name, longToPrefixCoded(toLong(value)));
		}
		return new Term(name, toString(value));
	}

	/**
	 * @param value
	 * @return value converted to {@code long}
	 */
	protected long toLong(final Object value) {
		return value != null ? ((Date) value).getTime() : 0L;
	}

	/**
	 * @param value
	 * @return value converted to {@link String}
	 */
	protected String toString(final Object value) {
		final Date v = (value instanceof Date) ? ((Date) value) : new Date(0L);
		return new SimpleDateFormat(this.format).format(v);
	}

}
