/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.type;

import static org.apache.lucene.util.NumericUtils.*;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParsePosition;

import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.document.NumericField;
import org.apache.lucene.document.Field.Index;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.index.Term;
import org.mtzky.lucene.descriptor.LucenePropertyConfig;

/**
 * @author mtzky
 */
@LuceneFieldStrategyType(Float.class)
public class LuceneFloatStrategy implements LuceneFieldStrategy {

	private final String name;
	private final Store store;
	private final Index index;
	private final boolean original;
	private final String format;

	/**
	 * @param config
	 */
	public LuceneFloatStrategy(final LucenePropertyConfig config) {
		this.name = config.getName();
		this.store = config.getStore();
		this.index = config.getIndex();
		this.format = config.getFormat();
		this.original = format.isEmpty();
	}

	@Override
	public Fieldable getField(final Object value) {
		if (original) {
			final boolean index = this.index != Index.NO;
			final NumericField f = new NumericField(name, store, index);
			f.setFloatValue(toFloat(value));
			return f;
		}
		final NumberFormat format = new DecimalFormat(this.format);
		return new Field(name, format.format(toFloat(value)), store, index);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(final Fieldable field) {
		final String v = field.stringValue();
		if (original) {
			return (T) Float.valueOf(v);
		}
		final DecimalFormat format = new DecimalFormat(this.format);
		format.setParseBigDecimal(true);
		final BigDecimal d = (BigDecimal) format.parse(v, new ParsePosition(0));
		return (T) Float.valueOf((d != null) ? d.floatValue() : 0);
	}

	@Override
	public Term getTerm(final Object value) {
		if (original) {
			return new Term(name, floatToPrefixCoded(toFloat(value)));
		}
		return new Term(name, new DecimalFormat(format).format(toFloat(value)));
	}

	/**
	 * @param value
	 * @return value converted to {@code float}
	 */
	protected float toFloat(final Object value) {
		return (value != null) ? (Float) value : 0F;
	}

}
