/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene;

import static org.apache.lucene.search.BooleanClause.Occur.*;

import java.io.IOException;

import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.search.Filter;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.BooleanClause.Occur;

/**
 * <p>
 * Fluent interface request for {@link LuceneIndexSearcher}. Setter methods can
 * be called only in following: query*, filter?, limit?, and sort*.
 * </p>
 * 
 * @param <E>
 *            entity
 * @author mtzky
 */
public class FluentIndexSearcherRequest<E> extends AbstractIndexSearcherRequest {

	private final LuceneIndexSearcher<E> searcher;

	protected FluentIndexSearcherRequest(final LuceneIndexSearcher<E> searcher) {
		this.searcher = searcher;
	}

	/**
	 * <p>
	 * Adds {@link Query} as the {@link Occur#MUST AND} condition.
	 * </p>
	 * 
	 * @param query
	 *            {@link Query}
	 * @return {@link FluentIndexSearcherRequest}
	 * @throws ParseException
	 * @see #query(Query)
	 */
	public FluentIndexSearcherRequest<E> query(final String query)
			throws ParseException {
		return query(searcher.parse(query));
	}

	/**
	 * <p>
	 * Adds {@link Query} as the {@link Occur#MUST AND} condition.
	 * </p>
	 * 
	 * @param query
	 *            {@link Query}
	 * @return {@link FluentIndexSearcherRequest}
	 * @see #query(Query, org.apache.lucene.search.BooleanClause.Occur)
	 */
	public FluentIndexSearcherRequest<E> query(final Query query) {
		return query(query, MUST);
	}

	/**
	 * <p>
	 * Adds {@link Query}.
	 * </p>
	 * 
	 * @param query
	 *            {@link Query}
	 * @param occur
	 *            {@link Occur}
	 * @return {@link FluentIndexSearcherRequest}
	 * @see #query(Query, org.apache.lucene.search.BooleanClause.Occur)
	 */
	public FluentIndexSearcherRequest<E> query(final String query,
			final Occur occur) throws ParseException {
		return query(searcher.parse(query), occur);
	}

	/**
	 * <p>
	 * Adds {@link Query}.
	 * </p>
	 * 
	 * @param query
	 *            {@link Query}
	 * @param occur
	 *            {@link Occur}
	 * @return {@link FluentIndexSearcherRequest}
	 */
	public FluentIndexSearcherRequest<E> query(final Query query,
			final Occur occur) {
		addQuery(query, occur);
		return this;
	}

	/**
	 * <p>
	 * Sets {@link Filter}.
	 * </p>
	 * 
	 * @param filter
	 *            {@link Filter}
	 * @return {@link FluentIndexSearcherLimitRequest}
	 */
	public FluentIndexSearcherLimitRequest<E> filter(final Filter filter) {
		return new FluentIndexSearcherFilterRequest<E>(this).filter(filter);
	}

	/**
	 * <p>
	 * Sets {@code limit}.
	 * </p>
	 * 
	 * @param limit
	 * @return {@link FluentIndexSearcherSortRequest}
	 */
	public FluentIndexSearcherOffsetRequest<E> limit(final int limit) {
		return new FluentIndexSearcherLimitRequest<E>(this).limit(limit);
	}

	/**
	 * <p>
	 * Sets {@code offset}.
	 * </p>
	 * 
	 * @param offset
	 * @return {@link FluentIndexSearcherSortRequest}
	 */
	public FluentIndexSearcherSortRequest<E> offset(final int offset) {
		return new FluentIndexSearcherOffsetRequest<E>(this).offset(offset);
	}

	/**
	 * <p>
	 * Adds {@link SortField sort field}s as natural order.
	 * </p>
	 * 
	 * @param fields
	 *            Name of field to sort by.
	 * @return {@link FluentIndexSearcherSortRequest}
	 */
	public FluentIndexSearcherSortRequest<E> sort(final String... fields) {
		return new FluentIndexSearcherSortRequest<E>(this).sort(fields);
	}

	/**
	 * <p>
	 * Adds {@link SortField sort field}s.
	 * </p>
	 * 
	 * @param field
	 *            Name of field to sort by.
	 * @param reverse
	 *            {@code true} if natural order should be reversed.
	 * @return {@link FluentIndexSearcherSortRequest}
	 */
	public FluentIndexSearcherSortRequest<E> sort(final String field,
			final boolean reverse) {
		return new FluentIndexSearcherSortRequest<E>(this).sort(field, reverse);
	}

	/**
	 * @return {@link LuceneIndexSearcherResponse}
	 * @throws IOException
	 */
	public LuceneIndexSearcherResponse<E> execute() throws IOException {
		return searcher.find(this);
	}

	/**
	 * @param <E>
	 *            entity
	 * @author mtzky
	 * @see FluentIndexSearcherRequest
	 */
	public static class FluentIndexSearcherFilterRequest<E> extends
			FluentIndexSearcherLimitRequest<E> {

		private final FluentIndexSearcherRequest<E> ctx;

		private FluentIndexSearcherFilterRequest(
				final FluentIndexSearcherRequest<E> request) {
			super(request);
			this.ctx = request;
		}

		/**
		 * @param filter
		 *            {@link Filter}
		 * @return {@link FluentIndexSearcherLimitRequest}
		 * @see FluentIndexSearcherRequest#filter(Filter)
		 */
		public FluentIndexSearcherLimitRequest<E> filter(final Filter filter) {
			ctx.setFilter(filter);
			return new FluentIndexSearcherLimitRequest<E>(ctx);
		}

	}

	/**
	 * @param <E>
	 *            entity
	 * @author mtzky
	 * @see FluentIndexSearcherRequest
	 */
	public static class FluentIndexSearcherLimitRequest<E> extends
			FluentIndexSearcherOffsetRequest<E> {

		private final FluentIndexSearcherRequest<E> ctx;

		private FluentIndexSearcherLimitRequest(
				final FluentIndexSearcherRequest<E> request) {
			super(request);
			this.ctx = request;
		}

		/**
		 * @param limit
		 * @return {@link FluentIndexSearcherOffsetRequest}
		 * @see FluentIndexSearcherRequest#limit(int)
		 */
		public FluentIndexSearcherOffsetRequest<E> limit(final int limit) {
			ctx.setLimit(limit);
			return new FluentIndexSearcherOffsetRequest<E>(ctx);
		}

	}

	/**
	 * @param <E>
	 *            entity
	 * @author mtzky
	 * @see FluentIndexSearcherRequest
	 */
	public static class FluentIndexSearcherOffsetRequest<E> extends
			FluentIndexSearcherSortRequest<E> {

		private final FluentIndexSearcherRequest<E> ctx;

		private FluentIndexSearcherOffsetRequest(
				final FluentIndexSearcherRequest<E> request) {
			super(request);
			this.ctx = request;
		}

		/**
		 * @param offset
		 * @return {@link FluentIndexSearcherSortRequest}
		 * @see FluentIndexSearcherRequest#offset(int)
		 */
		public FluentIndexSearcherSortRequest<E> offset(final int offset) {
			ctx.setOffset(offset);
			return new FluentIndexSearcherSortRequest<E>(ctx);
		}

		@Override
		public FluentIndexSearcherSortRequest<E> sort(final String... fields) {
			ctx.sort(fields);
			return new FluentIndexSearcherSortRequest<E>(ctx);
		}

		@Override
		public FluentIndexSearcherSortRequest<E> sort(final String field,
				final boolean reverse) {
			ctx.sort(field, reverse);
			return new FluentIndexSearcherSortRequest<E>(ctx);
		}

	}

	/**
	 * @param <E>
	 *            entity
	 * @author mtzky
	 * @see FluentIndexSearcherRequest
	 */
	public static class FluentIndexSearcherSortRequest<E> extends
			FluentIndexSearcherSearchRequest<E> {

		private final FluentIndexSearcherRequest<E> ctx;

		private FluentIndexSearcherSortRequest(
				final FluentIndexSearcherRequest<E> request) {
			super(request);
			this.ctx = request;
		}

		/**
		 * @param fields
		 * @return {@link FluentIndexSearcherSortRequest}
		 * @see FluentIndexSearcherRequest#sort(String...)
		 */
		public FluentIndexSearcherSortRequest<E> sort(final String... fields) {
			ctx.addSort(fields);
			return this;
		}

		/**
		 * @param field
		 * @param reverse
		 * @return {@link FluentIndexSearcherSortRequest}
		 * @see FluentIndexSearcherRequest#sort(String, boolean)
		 */
		public FluentIndexSearcherSortRequest<E> sort(final String field,
				final boolean reverse) {
			ctx.addSort(field, reverse);
			return this;
		}

	}

	/**
	 * @param <E>
	 *            entity
	 * @author mtzky
	 * @see FluentIndexSearcherRequest
	 */
	public static class FluentIndexSearcherSearchRequest<E> {

		private final FluentIndexSearcherRequest<E> ctx;

		protected FluentIndexSearcherSearchRequest(
				final FluentIndexSearcherRequest<E> request) {
			this.ctx = request;
		}

		/**
		 * @return {@link LuceneIndexSearcherResponse}
		 * @throws IOException
		 * @see FluentIndexSearcherRequest#execute()
		 */
		public LuceneIndexSearcherResponse<E> execute() throws IOException {
			return ctx.execute();
		}

		@Override
		public String toString() {
			return ctx.toString();
		}

	}

}
