/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import java.util.HashMap;
import java.util.Map;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;

/**
 * <p>
 * Map class descriptor for {@link org.mtzky.lucene.LuceneIndex}.
 * </p>
 * 
 * @param <T>
 *            entity value type
 * @author mtzky
 */
public class MapDocumentDesc<T> extends
		AbstractLuceneDocumentDesc<Map<String, T>> {

	private final LucenePropertyDesc<Map<String, T>>[] descs;

	@SuppressWarnings("unchecked")
	public MapDocumentDesc(final LucenePropertyConfig... configs) {
		this(config(configs));
	}

	@SuppressWarnings("unchecked")
	private static <T> LucenePropertyDesc[] config(
			final LucenePropertyConfig[] configs) {
		if (configs == null) {
			throw new NullPointerException("configs");
		}
		final int len = configs.length;
		final LucenePropertyDesc[] descs = new LucenePropertyDesc[len];
		for (int i = 0; i < len; i++) {
			final LucenePropertyConfig config = configs[i];
			if (config == null) {
				throw new NullPointerException("configs[" + i + "]");
			}
			descs[i] = new MapPropertyDesc<T>(config);
		}
		return descs;
	}

	private MapDocumentDesc(final LucenePropertyDesc<Map<String, T>>[] descs) {
		super(descs);
		this.descs = descs;
	}

	@Override
	public Document createDocument(final Map<String, T> map) {
		final Document doc = new Document();
		for (final LucenePropertyDesc<Map<String, T>> desc : descs) {
			doc.add(desc.getField(map));
		}
		return doc;
	}

	@Override
	public Map<String, T> createEntity(final Document document) {
		final Map<String, T> map = new HashMap<String, T>();
		for (final LucenePropertyDesc<Map<String, T>> desc : descs) {
			final Fieldable fieldable = document.getFieldable(desc.getName());
			if (fieldable == null) {
				continue;
			}
			desc.setField(map, fieldable);
		}
		return map;
	}

}
