/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import static org.mtzky.reflect.IterableUtils.*;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.analysis.LimitTokenCountAnalyzer;
import org.apache.lucene.analysis.PerFieldAnalyzerWrapper;
import org.apache.lucene.index.Term;
import org.mtzky.lucene.analyzer.AnalyzerWrapper;
import org.mtzky.reflect.IterableUtils.Find;

/**
 * <p>
 * Descriptor for {@link org.mtzky.lucene.LuceneIndex}.
 * </p>
 * 
 * @author mtzky
 */
public abstract class AbstractLuceneDocumentDesc<E> implements
		LuceneDocumentDesc<E> {

	private final PerFieldAnalyzerWrapper analyzer;
	private final LucenePropertyDesc<E> idPropDesc;

	public AbstractLuceneDocumentDesc(final LucenePropertyDesc<E>[] descs) {
		analyzer = new PerFieldAnalyzerWrapper(new KeywordAnalyzer());
		idPropDesc = each(descs, new Find<LucenePropertyDesc<E>>() {
			@Override
			public boolean call(final LucenePropertyDesc<E> desc) {
				return desc.isId();
			}
		});
		if (idPropDesc == null) {
			throw new IllegalArgumentException("Requires the ID field");
		}
		for (final LucenePropertyDesc<E> desc : descs) {
			Analyzer a = new AnalyzerWrapper(desc);
			final int maxTokenCount = desc.getMaxTokenCount();
			if (0 < maxTokenCount) {
				a = new LimitTokenCountAnalyzer(a, maxTokenCount);
			}
			analyzer.addAnalyzer(desc.getName(), a);
		}
	}

	@Override
	public Analyzer getAnalyzer() {
		return analyzer;
	}

	@Override
	public String getIdName() {
		return idPropDesc.getName();
	}

	@Override
	public Term getIdTermByEntity(final E entity) {
		return idPropDesc.getTerm(entity);
	}

	@Override
	public Term getIdTermByValue(final Object value) {
		return idPropDesc.getTermByValue(value);
	}

}
