/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import java.io.Reader;
import java.lang.reflect.Constructor;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.document.Field.Index;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.Field.TermVector;
import org.apache.lucene.index.Term;
import org.mtzky.lucene.type.LuceneFieldStrategy;

/**
 * @param <E>
 *            entity
 * @author mtzky
 */
public abstract class AbstractPropertyDesc<E> implements LucenePropertyDesc<E> {

	private final boolean id;
	private final String name;
	private final Store store;
	private final Index index;
	private final TermVector termVector;
	private final int maxTokenCount;
	private final String format;
	private final LuceneFieldStrategy luceneFieldStrategy;
	private final Analyzer analyzer;
	private final Constructor<? extends Reader>[] normalizers;
	private final Constructor<? extends TokenFilter>[] filters;

	public AbstractPropertyDesc(final LucenePropertyConfig config) {
		this.id = config.isId();
		this.name = config.getName();
		this.store = config.getStore();
		this.index = config.getIndex();
		this.termVector = config.getTermVector();
		this.maxTokenCount = config.getMaxTokenCount();
		this.format = config.getFormat();
		this.luceneFieldStrategy = config.getLuceneFieldStrategy();
		this.analyzer = config.getAnalyzer();
		this.normalizers = config.getNormalizers();
		this.filters = config.getFilters();
	}

	@Override
	public Fieldable getFieldByValue(final Object value) {
		return luceneFieldStrategy.getField(value);
	}

	@Override
	public Object getValue(final Fieldable fieldable) {
		return luceneFieldStrategy.getValue(fieldable);
	}

	@Override
	public Term getTermByValue(final Object value) {
		return luceneFieldStrategy.getTerm(value);
	}

	@Override
	public boolean isId() {
		return id;
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public Store getStore() {
		return store;
	}

	@Override
	public Index getIndex() {
		return index;
	}

	@Override
	public TermVector getTermVector() {
		return termVector;
	}

	@Override
	public int getMaxTokenCount() {
		return maxTokenCount;
	}

	@Override
	public String getFormat() {
		return format;
	}

	@Override
	public LuceneFieldStrategy getLuceneFieldStrategy() {
		return luceneFieldStrategy;
	}

	@Override
	public Analyzer getAnalyzer() {
		return analyzer;
	}

	@Override
	public Constructor<? extends Reader>[] getNormalizers() {
		return normalizers;
	}

	@Override
	public Constructor<? extends TokenFilter>[] getFilters() {
		return filters;
	}

}
