/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import java.io.Reader;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.document.Field.Index;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.Field.TermVector;
import org.mtzky.lucene.type.LuceneFieldStrategy;

/**
 * <p>
 * Defines a {@link org.apache.lucene.document.Fieldable field} of Lucene
 * {@link org.apache.lucene.document.Document Document}.
 * </p>
 * 
 * @author mtzky
 * @see LucenePropertyConfig
 */
@Retention(RetentionPolicy.RUNTIME)
@Target( { ElementType.FIELD, ElementType.METHOD })
public @interface BeanLuceneField {

	/**
	 * @return {@code true} if the specified field is ID.
	 * @see LucenePropertyConfig#isId()
	 */
	boolean id() default false;

	/**
	 * @return The name of the specified field
	 * @see LucenePropertyConfig#getName()
	 */
	String name() default "";

	/**
	 * @return The {@link Store} of the specified field.
	 * @see LucenePropertyConfig#getStore()
	 */
	Store store() default Store.YES;

	/**
	 * @return The {@link Index} of the specified field.
	 * @see LucenePropertyConfig#getIndex()
	 */
	Index index() default Index.ANALYZED;

	/**
	 * @return The {@link TermVector} of the specified field.
	 * @see LucenePropertyConfig#getTermVector()
	 */
	TermVector termVector() default TermVector.NO;

	/**
	 * @return The maximum field token length, or 0 if unlimited (default)
	 * @see LucenePropertyConfig#getMaxTokenCount()
	 */
	int maxTokenCount() default 0;

	/**
	 * @return field value format
	 * @see LucenePropertyConfig#getFormat()
	 */
	String format() default "";

	/**
	 * @return {@link LuceneFieldStrategy} to use for the specified field
	 * @see LucenePropertyConfig#getLuceneFieldStrategy()
	 */
	Class<? extends LuceneFieldStrategy> luceneFieldStrategy() default LuceneFieldStrategy.class;

	/**
	 * @return {@link Analyzer} to use for the specified field
	 * @see LucenePropertyConfig#getAnalyzer()
	 */
	Class<? extends Analyzer> analyzer() default KeywordAnalyzer.class;

	/**
	 * @return {@link Reader}s to use for the specified field
	 * @see LucenePropertyConfig#getNormalizers()
	 */
	Class<? extends Reader>[] normalizers() default {};

	/**
	 * @return {@link TokenFilter}s to use for the specified field
	 * @see LucenePropertyConfig#getFilters()
	 */
	Class<? extends TokenFilter>[] filters() default {};

}
