/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import java.io.Reader;
import java.lang.reflect.Constructor;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.document.Field.Index;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.Field.TermVector;
import org.mtzky.lucene.type.LuceneFieldStrategy;
import org.mtzky.lucene.type.LuceneFieldStrategyType;

/**
 * <p>
 * Holds all the configuration of {@link LucenePropertyDesc}. You should
 * instantiate an implemented class, call the setters to set your configuration,
 * then pass it to {@link LucenePropertyDesc}.
 * </p>
 * 
 * @author mtzky
 */
public interface LucenePropertyConfig {

	/**
	 * @return {@code true} if the specified field is ID.
	 */
	boolean isId();

	/**
	 * @return The name of the specified field.
	 */
	String getName();

	/**
	 * @return The {@link Store} of the specified field.
	 */
	Store getStore();

	/**
	 * @return The {@link Index} of the specified field.
	 */
	Index getIndex();

	/**
	 * @return The {@link TermVector} of the specified field.
	 */
	TermVector getTermVector();

	/**
	 * @return The maximum field token length, or 0 if unlimited
	 */
	int getMaxTokenCount();

	/**
	 * <p>
	 * The specified field is treated as {@link String} forcibly if NOT empty.
	 * </p>
	 * 
	 * @return field value format
	 * @see org.mtzky.lucene.type.LuceneDateStrategy
	 */
	String getFormat();

	/**
	 * <p>
	 * {@link LuceneFieldStrategy} to use for the specified field. The class
	 * must be annotated with {@link LuceneFieldStrategyType}, and the
	 * constructor signature must be the following:
	 * </p>
	 * 
	 * <pre>
	 * <code>(L{@link LucenePropertyConfig org/mtzky/lucene/descriptor/LucenePropertyConfig};)V</code>
	 * </pre>
	 * 
	 * @return The {@link LuceneFieldStrategy} of the specified field.
	 * @see LuceneFieldStrategyType
	 */
	LuceneFieldStrategy getLuceneFieldStrategy();

	/**
	 * <p>
	 * {@link Analyzer} to use for the specified field. The constructor
	 * signature of this analyzer must be the following:
	 * </p>
	 * 
	 * <pre>
	 * <code>(L{@link org.apache.lucene.util.Version org/apache/lucene/util/Version};)V</code>
	 * or
	 * <code>()V</code>
	 * </pre>
	 * 
	 * @return The {@link Analyzer} class of the specified field.
	 * @see org.mtzky.lucene.analyzer.AnalyzerWrapper
	 */
	Analyzer getAnalyzer();

	/**
	 * <p>
	 * {@link Reader}s to use for the specified field. The constructor signature
	 * of these readers must be the following:
	 * </p>
	 * 
	 * <pre>
	 * <code>(L{@link Reader java/io/Reader};)V</code>
	 * </pre>
	 * 
	 * @return {@link Reader}s to use for the specified field
	 * @see org.mtzky.lucene.analyzer.AnalyzerWrapper
	 */
	Constructor<? extends Reader>[] getNormalizers();

	/**
	 * <p>
	 * {@link TokenFilter}s to use for the specified field. The constructor
	 * signature of these filters must be the following:
	 * </p>
	 * 
	 * <pre>
	 * <code>(L{@link org.apache.lucene.analysis.TokenStream org/apache/lucene/analysis/TokenStream};)V</code>
	 * </pre>
	 * 
	 * @return {@link TokenFilter}s to use for the specified field
	 * @see org.mtzky.lucene.analyzer.AnalyzerWrapper
	 */
	Constructor<? extends TokenFilter>[] getFilters();

}
