/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.type;

import static java.math.BigDecimal.*;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.ParsePosition;

import org.apache.lucene.document.Fieldable;
import org.mtzky.lucene.descriptor.LucenePropertyConfig;
import org.mtzky.lucene.descriptor.LucenePropertyDesc;

/**
 * @author mtzky
 */
@LuceneFieldStrategyType(BigDecimal.class)
public class LuceneBigDecimalStrategy extends LuceneStringStrategy {

	private final boolean original;
	private final String format;

	/**
	 * @param config
	 */
	public LuceneBigDecimalStrategy(final LucenePropertyDesc<?> config) {
		super(config);
		this.format = config.getFormat();
		this.original = format.isEmpty();
	}

	/**
	 * @param config
	 */
	public LuceneBigDecimalStrategy(final LucenePropertyConfig config) {
		super(config);
		this.format = config.getFormat();
		this.original = format.isEmpty();
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(final Fieldable field) {
		if (original) {
			return (T) new BigDecimal(field.stringValue());
		}
		final DecimalFormat format = new DecimalFormat(this.format);
		format.setParseBigDecimal(true);
		return (T) format.parse(field.stringValue(), new ParsePosition(0));
	}

	@Override
	protected String toString(final Object value) {
		final BigDecimal v;
		if (value instanceof Number) {
			v = new BigDecimal(value.toString());
		} else {
			v = ZERO;
		}
		if (original) {
			return v.toString();
		}
		return new DecimalFormat(this.format).format(v);
	}

}
