/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.type;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.mtzky.reflect.PackageDesc;
import org.mtzky.reflect.IterableUtils.Find;

/**
 * @author mtzky
 */
public class LuceneFieldStrategyFactory {

	/**
	 * Default {@code package} of {@link LuceneFieldStrategy}s
	 */
	public static final String DEFAULT_LUCENE_FIELD_STRATEGY_PACKAGE_NAME = LuceneFieldStrategy.class
			.getPackage().getName();

	private static final Find<Class<?>> FILTER = new Find<Class<?>>() {
		@Override
		public boolean call(final Class<?> value) {
			return (LuceneFieldStrategy.class.isAssignableFrom(value))
					&& (value.getAnnotation(LuceneFieldStrategyType.class) != null);
		}
	};

	private final Map<Class<?>, Class<? extends LuceneFieldStrategy>> types;
	private final Lock lock = new ReentrantLock();

	/**
	 * <p>
	 * Constructs {@link LuceneFieldStrategyFactory} with
	 * {@link #DEFAULT_LUCENE_FIELD_STRATEGY_PACKAGE_NAME}
	 * </p>
	 */
	public LuceneFieldStrategyFactory() {
		types = new HashMap<Class<?>, Class<? extends LuceneFieldStrategy>>();
		addPackageName(DEFAULT_LUCENE_FIELD_STRATEGY_PACKAGE_NAME);
	}

	/**
	 * @param packageName
	 */
	@SuppressWarnings("unchecked")
	public void addPackageName(final String packageName) {
		final PackageDesc desc;
		try {
			desc = new PackageDesc(packageName, FILTER);
		} catch (final Exception e) {
			throw new IllegalArgumentException(e);
		}
		try {
			lock.lock();
			for (final Class<?> c : desc.getClasses()) {
				types.put(c.getAnnotation(LuceneFieldStrategyType.class)
						.value(), (Class<? extends LuceneFieldStrategy>) c);
			}
		} finally {
			lock.unlock();
		}
	}

	/**
	 * @param type
	 * @return {@link LuceneFieldStrategy}
	 */
	public Class<? extends LuceneFieldStrategy> getLuceneFieldStrategy(
			final Class<?> type) {
		try {
			lock.lock();
			final Class<? extends LuceneFieldStrategy> c = types.get(type);
			if (c != null) {
				return c;
			}
			/* Reliefs for super classes */
			if (type.isAssignableFrom(String.class)) {
				return types.get(String.class);
			}
			if (type.isAssignableFrom(BigDecimal.class)) {
				return types.get(BigDecimal.class);
			}
			for (final Class<?> cls : types.keySet()) {
				if (type.isAssignableFrom(cls)) {
					return types.get(cls);
				}
			}
		} finally {
			lock.unlock();
		}
		throw new UnsupportedOperationException("Unsupported type: " + type);
	}

}
