/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene;

import static org.apache.lucene.index.IndexReader.*;
import static org.apache.lucene.util.Version.*;
import static org.mtzky.io.IOUtils.*;
import static org.mtzky.log.GenericMarker.*;
import static org.slf4j.LoggerFactory.*;

import java.io.IOException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.RAMDirectory;
import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;
import org.mtzky.lucene.descriptor.LuceneDocumentDesc;
import org.slf4j.Logger;

/**
 * @param <E>
 *            {@code entity}
 * @author mtzky
 */
public class LuceneIndex<E> implements Closable {

	private static final Logger LOG = getLogger(LuceneIndex.class);

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean closed = false;

	/**
	 * <p lang="ja">
	 * {@link Directory} の Unfair なロック。{@link Directory#close()} や
	 * {@link #getSearcher()} などの同期を保証するため。
	 * </p>
	 */
	private final Lock dirLock = new ReentrantLock();

	private Directory directory;
	private LuceneDocumentDesc<E> desc;
	private LuceneIndexSearcher<E> searcher = null;

	private long version;

	/**
	 * <p>
	 * Creates instance with {@link RAMDirectory}.
	 * </p>
	 * 
	 * @param desc
	 * @throws IOException
	 * @throws CorruptIndexException
	 * @see #LuceneIndex(LuceneDocumentDesc, Directory)
	 */
	public LuceneIndex(final LuceneDocumentDesc<E> desc)
			throws CorruptIndexException, IOException {
		this(desc, new RAMDirectory());
	}

	/**
	 * <p>
	 * Creates instance with the specified {@code directory}.
	 * </p>
	 * 
	 * @param desc
	 * @param dir
	 * @throws IOException
	 * @throws CorruptIndexException
	 */
	public LuceneIndex(final LuceneDocumentDesc<E> desc, final Directory dir)
			throws CorruptIndexException, IOException {
		IndexWriter w = null;
		try {
			final Analyzer analyzer = desc.getAnalyzer();
			w = new IndexWriter(dir, new IndexWriterConfig(LUCENE_31, analyzer));
		} catch (final IOException e) {
			final String msg = "FAILED to initialize directory: " + dir;
			LOG.warn(FAILED_TO_INITIALIZE, msg, e);
			throw e;
		} finally {
			closeQuietly(w);
		}

		this.desc = desc;
		this.directory = dir;
		this.version = getCurrentVersion(dir);
	}

	/**
	 * @return a current {@link Directory}
	 */
	public Directory getDirectory() {
		try {
			dirLock.lock();
			return directory;
		} finally {
			dirLock.unlock();
		}
	}

	/**
	 * <p>
	 * Switches new {@link Directory} from current {@link Directory}.
	 * </p>
	 * 
	 * @param directory
	 *            new {@link Directory} to switch from current {@link Directory}
	 * @throws IOException
	 * @throws CorruptIndexException
	 */
	public void setDirectory(final Directory directory)
			throws CorruptIndexException, IOException {
		try {
			dirLock.lock();
			final long version = getCurrentVersion(directory);
			if (version == this.version) {
				if (LOG.isDebugEnabled(CANCEL)) {
					final String fmt = "NOT switch a directory due to the same version {}";
					LOG.debug(CANCEL, fmt, version);
				}
				return;
			}
			closeQuietly(this.searcher);
			this.searcher = null;
			closeQuietly(this.directory);
			this.directory = directory;
			this.version = version;
		} finally {
			dirLock.unlock();
		}
	}

	/**
	 * <p>
	 * Requires {@link IndexWriter#close()}.
	 * </p>
	 * 
	 * @return {@link IndexWriter} of a current {@link Directory}
	 * @throws CorruptIndexException
	 * @throws LockObtainFailedException
	 * @throws IOException
	 */
	public LuceneIndexWriter<E> getWriter() throws CorruptIndexException,
			LockObtainFailedException, IOException {
		return new LuceneIndexWriter<E>(getDirectory(), desc);
	}

	/**
	 * <p>
	 * Creates a new {@link IndexWriter} for this current {@link Directory},
	 * passes it to the {@code callback}, and ensures the {@link IndexWriter} is
	 * {@link IndexWriter#close() closed} after the {@code callback} returns.
	 * </p>
	 * 
	 * @param callback
	 *            {@link WriterCallback}
	 * @throws CorruptIndexException
	 * @throws LockObtainFailedException
	 * @throws IOException
	 * @see #getWriter()
	 * @see WriterCallback
	 */
	public void withWriter(final WriterCallback<E> callback)
			throws CorruptIndexException, LockObtainFailedException,
			IOException {
		LuceneIndexWriter<E> writer = null;
		try {
			dirLock.lock();
			writer = getWriter();
			callback.call(writer);
		} finally {
			closeQuietly(writer);
			dirLock.unlock();
		}
	}

	/**
	 * @author mtzky
	 * @param <E>
	 */
	public static interface WriterCallback<E> {
		void call(LuceneIndexWriter<E> writer) throws CorruptIndexException,
				IOException;
	}

	/**
	 * <p>
	 * Requires {@link IndexSearcher#close()}.
	 * </p>
	 * 
	 * @return {@link IndexSearcher} of a current {@link Directory}
	 * @throws CorruptIndexException
	 * @throws IOException
	 */
	public LuceneIndexSearcher<E> getSearcher() throws CorruptIndexException,
			IOException {
		try {
			dirLock.lock();
			final long version = getCurrentVersion(directory);
			if (searcher == null || searcher.isClosed()) {
				this.version = version;
				searcher = createSearcher();
			} else if (version != this.version) {
				searcher.closeWhenDone();
				this.version = version;
				searcher = createSearcher();
			}
			searcher.open();
			return searcher;
		} finally {
			dirLock.unlock();
		}
	}

	/**
	 * @return {@link IndexSearcher}
	 * @see #getSearcher()
	 * @see #getDirectory()
	 */
	protected LuceneIndexSearcher<E> createSearcher()
			throws CorruptIndexException, IOException {
		return new LuceneIndexSearcher<E>(getDirectory(), desc);
	}

	@Override
	public boolean isClosed() {
		try {
			dirLock.lock();
			return closed;
		} finally {
			dirLock.unlock();
		}
	}

	@Override
	public void close() throws IOException {
		try {
			dirLock.lock();
			if (searcher != null && !searcher.isClosed()) {
				searcher.closeWhenDone();
			}
			searcher = null;
			closeQuietly(directory);
			closed = true;
		} finally {
			dirLock.unlock();
		}
	}

}
