/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.analyzer;

import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.document.Fieldable;
import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;
import org.mtzky.lucene.descriptor.LucenePropertyDesc;
import org.mtzky.reflect.InvocationTargetRuntimeException;

/**
 * @author mtzky
 */
public class AnalyzerWrapper extends Analyzer implements Closable {

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean closed = false;

	private final Analyzer analyzer;
	private final Constructor<? extends Reader>[] normalizerCtors;
	private final Constructor<? extends TokenFilter>[] filterCtors;

	public AnalyzerWrapper(final LucenePropertyDesc<?> desc) {
		super();
		this.analyzer = desc.getAnalyzer();
		this.normalizerCtors = desc.getNormalizers();
		this.filterCtors = desc.getFilters();
	}

	@Override
	public final TokenStream tokenStream(final String field, final Reader r) {
		return filter(analyzer.tokenStream(field, normalize(r)));
	}

	@Override
	public final TokenStream reusableTokenStream(final String field,
			final Reader r) throws IOException {
		return filter(analyzer.reusableTokenStream(field, normalize(r)));
	}

	private Reader normalize(Reader r) {
		try {
			for (final Constructor<? extends Reader> ctor : normalizerCtors) {
				r = ctor.newInstance(r);
			}
		} catch (final InvocationTargetException e) {
			throw new InvocationTargetRuntimeException(e.getCause());
		} catch (final Exception e) {
			throw new InvocationTargetRuntimeException(e);
		}
		return r;
	}

	private TokenStream filter(TokenStream ts) {
		try {
			for (final Constructor<? extends TokenFilter> ctor : filterCtors) {
				ts = ctor.newInstance(ts);
			}
		} catch (final InvocationTargetException e) {
			throw new InvocationTargetRuntimeException(e.getCause());
		} catch (final Exception e) {
			throw new InvocationTargetRuntimeException(e);
		}
		return ts;
	}

	@Override
	public int getPositionIncrementGap(final String fieldName) {
		return analyzer.getPositionIncrementGap(fieldName);
	}

	@Override
	public int getOffsetGap(final Fieldable field) {
		return analyzer.getOffsetGap(field);
	}

	@Override
	public boolean isClosed() {
		return closed;
	}

	@Override
	public void close() {
		analyzer.close();
		closed = true;
	}

}
