/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import static org.mtzky.reflect.IterableUtils.*;

import java.lang.reflect.Constructor;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.mtzky.lucene.type.LuceneFieldStrategyFactory;
import org.mtzky.reflect.InvocationTargetRuntimeException;
import org.mtzky.reflect.PropDesc;
import org.mtzky.reflect.PropsDesc;
import org.mtzky.reflect.IterableUtils.Wrap;
import org.mtzky.reflect.PropsDesc.WrapCallback;

/**
 * <p>
 * Bean class descriptor for {@link org.mtzky.lucene.LuceneIndex}.
 * </p>
 * 
 * @author mtzky
 */
public class BeanDocumentDesc<E> extends AbstractLuceneDocumentDesc<E> {

	protected static final LuceneFieldStrategyFactory DEFAULT_STRATEGY_FACTORY = new LuceneFieldStrategyFactory();

	private final Constructor<E> ctor;
	private final LucenePropertyDesc<E>[] descs;

	/**
	 * @param beanClass
	 */
	public BeanDocumentDesc(final Class<E> beanClass) {
		this(beanClass, DEFAULT_STRATEGY_FACTORY);
	}

	/**
	 * @param beanClass
	 * @param strategyFactory
	 *            {@link LuceneFieldStrategyFactory}
	 */
	public BeanDocumentDesc(final Class<E> beanClass,
			final LuceneFieldStrategyFactory strategyFactory) {
		this(beanClass, strategyFactory, new PropsDesc(beanClass,
				BeanLuceneField.class).wrap(new PropDescConverter<E>(
				strategyFactory)));
	}

	/**
	 * @param beanClass
	 * @param configs
	 */
	public BeanDocumentDesc(final Class<E> beanClass,
			final BeanPropertyConfig... configs) {
		this(beanClass, DEFAULT_STRATEGY_FACTORY, configs);
	}

	/**
	 * @param beanClass
	 * @param strategyFactory
	 *            {@link LuceneFieldStrategyFactory}
	 * @param configs
	 */
	public BeanDocumentDesc(final Class<E> beanClass,
			final LuceneFieldStrategyFactory strategyFactory,
			final BeanPropertyConfig... configs) {
		this(beanClass, strategyFactory, each(configs, new ConfigConverter<E>(
				beanClass, strategyFactory)));
	}

	/**
	 * @param beanClass
	 * @param strategyFactory
	 * @param descs
	 */
	BeanDocumentDesc(final Class<E> beanClass,
			final LuceneFieldStrategyFactory strategyFactory,
			final BeanPropertyDesc<E>[] descs) {
		super(descs);
		this.descs = descs;
		try {
			this.ctor = beanClass.getConstructor();
		} catch (final NoSuchMethodException e) {
			final String msg = "Requires the default constructor: " + beanClass;
			throw new IllegalArgumentException(msg, e);
		}
	}

	private static class PropDescConverter<E> implements
			WrapCallback<BeanPropertyDesc<E>> {

		private final LuceneFieldStrategyFactory factory;

		public PropDescConverter(final LuceneFieldStrategyFactory factory) {
			this.factory = factory;
		}

		@Override
		public BeanPropertyDesc<E> call(final PropDesc desc) {
			final BeanLuceneField a = desc.getAnnotation(BeanLuceneField.class);
			final String nm = a.name().isEmpty() ? desc.getName() : a.name();
			final BeanPropertyConfig config = new BeanPropertyConfig(nm, desc);
			config.id(a.id()).format(a.format());
			config.luceneFieldStrategy(a.luceneFieldStrategy());
			config.store(a.store()).index(a.index()).termVector(a.termVector());
			config.analyzer(a.analyzer());
			config.normalizers(a.normalizers()).filters(a.filters());
			return new BeanPropertyDesc<E>(
					bindStrategyIfNeeded(config, factory));
		}

	}

	private static class ConfigConverter<E> implements
			Wrap<BeanPropertyConfig, BeanPropertyDesc<E>> {

		private int i = 0;
		private final Class<E> beanClass;
		private final LuceneFieldStrategyFactory factory;

		public ConfigConverter(final Class<E> beanClass,
				final LuceneFieldStrategyFactory factory) {
			this.beanClass = beanClass;
			this.factory = factory;
		}

		@Override
		public BeanPropertyDesc<E> call(final BeanPropertyConfig config) {
			if (config == null) {
				throw new NullPointerException("configs[" + i + "]");
			}
			final Class<?> propClass = config.getPropDesc().getDeclaringClass();
			if (propClass != beanClass) {
				throw new IllegalArgumentException("beanClass: " + beanClass
						+ " but configs[" + i + "].declaringClass: "
						+ propClass);
			}
			i++;
			return new BeanPropertyDesc<E>(
					bindStrategyIfNeeded(config, factory));
		}

	}

	private static BeanPropertyConfig bindStrategyIfNeeded(
			final BeanPropertyConfig config,
			final LuceneFieldStrategyFactory factory) {
		if (config.hasLuceneFieldStrategy()) {
			return config;
		}
		final Class<?> type = config.getPropDesc().getType();
		return config.luceneFieldStrategy(factory.getLuceneFieldStrategy(type));
	}

	@Override
	public Document createDocument(final E bean) {
		final Document doc = new Document();
		for (final LucenePropertyDesc<E> desc : descs) {
			doc.add(desc.getField(bean));
		}
		return doc;
	}

	@Override
	public E createEntity(final Document document) {
		E entity;
		try {
			entity = ctor.newInstance();
		} catch (final Exception e) {
			final String msg = "Failed to create entity: " + ctor;
			throw new InvocationTargetRuntimeException(msg, e.getCause());
		}
		for (final LucenePropertyDesc<E> desc : descs) {
			final Fieldable fieldable = document.getFieldable(desc.getName());
			if (fieldable == null) {
				continue;
			}
			desc.setField(entity, fieldable);
		}
		return entity;
	}

}
