/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.descriptor;

import static org.apache.lucene.util.Version.*;
import static org.mtzky.reflect.IterableUtils.*;

import java.io.Reader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.document.Field.Index;
import org.apache.lucene.document.Field.Store;
import org.apache.lucene.document.Field.TermVector;
import org.apache.lucene.util.Version;
import org.mtzky.lucene.type.LuceneFieldStrategy;
import org.mtzky.lucene.type.LuceneFieldStrategyType;
import org.mtzky.lucene.type.LuceneStringStrategy;
import org.mtzky.reflect.InvocationTargetRuntimeException;
import org.mtzky.reflect.IterableUtils.Wrap;

/**
 * <p>
 * All setter methods return {@link SimplePropertyConfig} to allow chaining
 * settings conveniently, for example:
 * </p>
 * 
 * <pre>
 * final SimplePropertyConfig conf = new SimplePropertyConfig("id");
 * conf.id({@code true}).store({@link Store#NO})...;
 * </pre>
 * 
 * @author mtzky
 */
public class SimplePropertyConfig implements LucenePropertyConfig {

	private static final String EMPTY = "";

	private final String name;
	private boolean id = false;
	private Store store;
	private Index index;
	private TermVector termVector;
	private int maxTokenCount = 0;
	private String format;
	private Class<? extends LuceneFieldStrategy> strategy;
	private Class<? extends Analyzer> analyzer;
	private Class<? extends Reader>[] normalizers;
	private Class<? extends TokenFilter>[] filters;

	public SimplePropertyConfig(final String name) {
		if (name == null) {
			throw new NullPointerException("name");
		}
		this.name = name;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T id() {
		return (T) id(true);
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T id(final boolean id) {
		this.id = id;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T store(final Store store) {
		this.store = store;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T index(final Index index) {
		this.index = index;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T termVector(
			final TermVector termVector) {
		this.termVector = termVector;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T maxTokenCount(
			final int maxTokenCount) {
		this.maxTokenCount = maxTokenCount;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T format(final String format) {
		this.format = format;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T luceneFieldStrategy(
			final Class<? extends LuceneFieldStrategy> strategy) {
		this.strategy = strategy;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T analyzer(
			final Class<? extends Analyzer> analyzer) {
		this.analyzer = analyzer;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T normalizers(
			final Class<? extends Reader>... normalizers) {
		this.normalizers = normalizers;
		return (T) this;
	}

	@SuppressWarnings("unchecked")
	public <T extends SimplePropertyConfig> T filters(
			final Class<? extends TokenFilter>... filters) {
		this.filters = filters;
		return (T) this;
	}

	@Override
	public boolean isId() {
		return id;
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public Store getStore() {
		return store != null ? store : Store.YES;
	}

	@Override
	public Index getIndex() {
		return index != null ? index : Index.ANALYZED;
	}

	@Override
	public TermVector getTermVector() {
		return termVector != null ? termVector : TermVector.NO;
	}

	@Override
	public int getMaxTokenCount() {
		return maxTokenCount;
	}

	@Override
	public String getFormat() {
		return format != null ? format : EMPTY;
	}

	/**
	 * @return {@code true} if has {@link LuceneFieldStrategy}
	 */
	public boolean hasLuceneFieldStrategy() {
		return strategy != null
				&& strategy.getAnnotation(LuceneFieldStrategyType.class) != null;
	}

	@Override
	public LuceneFieldStrategy getLuceneFieldStrategy() {
		if (strategy == null) {
			return new LuceneStringStrategy(this);
		}
		try {
			return createCtor(strategy, LucenePropertyConfig.class)
					.newInstance(this);
		} catch (final InvocationTargetException e) {
			throw new InvocationTargetRuntimeException(e.getCause());
		} catch (final Exception e) {
			throw new UnsupportedOperationException(
					"Unsupported LuceneFieldStrategy: " + strategy.getName(), e);
		}
	}

	@Override
	public Analyzer getAnalyzer() {
		if (analyzer == null) {
			return new KeywordAnalyzer();
		}
		try {
			/* http://lucene.jugem.jp/?eid=370 */
			return createCtor(analyzer, Version.class).newInstance(LUCENE_31);
		} catch (final Exception e) {
			/* NOP */
		}
		try {
			return createCtor(analyzer).newInstance();
		} catch (final InvocationTargetException e) {
			throw new InvocationTargetRuntimeException(e.getCause());
		} catch (final Exception e) {
			throw new UnsupportedOperationException("Unsupported Analyzer: "
					+ analyzer.getName(), e);
		}
	}

	@Override
	@SuppressWarnings("unchecked")
	public Constructor<? extends Reader>[] getNormalizers() {
		if (normalizers == null) {
			return new Constructor[0];
		}
		return each(normalizers, WRAP_NORMALIZER_CTOR);
	}

	@Override
	@SuppressWarnings("unchecked")
	public Constructor<? extends TokenFilter>[] getFilters() {
		if (filters == null) {
			return new Constructor[0];
		}
		return each(filters, WRAP_FILTER_CTOR);
	}

	private static final Wrap<Class<? extends Reader>, Constructor<? extends Reader>> WRAP_NORMALIZER_CTOR = new Wrap<Class<? extends Reader>, Constructor<? extends Reader>>() {
		@Override
		public Constructor<? extends Reader> call(
				final Class<? extends Reader> normalizer) {
			return createCtor(normalizer, Reader.class);
		}
	};

	private static final Wrap<Class<? extends TokenFilter>, Constructor<? extends TokenFilter>> WRAP_FILTER_CTOR = new Wrap<Class<? extends TokenFilter>, Constructor<? extends TokenFilter>>() {
		@Override
		public Constructor<? extends TokenFilter> call(
				final Class<? extends TokenFilter> filter) {
			return createCtor(filter, TokenStream.class);
		}
	};

	private static <T> Constructor<? extends T> createCtor(
			final Class<? extends T> targetClass,
			final Class<?>... parameterTypes) {
		try {
			return targetClass.getConstructor(parameterTypes);
		} catch (final Exception e) {
			final String fmt = "Failed to get the constructor %s(%s)";
			final Object[] args = { targetClass.getName(),
					Arrays.toString(parameterTypes) };
			throw new UnsupportedOperationException(String.format(fmt, args), e);
		}
	}

}
