/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.normalizer;

import java.io.IOException;
import java.io.Reader;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.mtzky.io.Closable;
import org.mtzky.io.ClosingGuardian;

/**
 * @author mtzky
 */
public abstract class CharToStringMappingNormalizer extends Reader implements
		Closable {

	@SuppressWarnings("unused")
	private final Object guardian = new ClosingGuardian(this);
	private boolean close = false;

	private final Lock lock = new ReentrantLock();
	private final Queue<Character> buf = new LinkedList<Character>();
	private final Reader in;

	/**
	 * @param in
	 */
	public CharToStringMappingNormalizer(final Reader in) {
		super(in);
		this.in = in;
	}

	@Override
	public int read(final char[] cbuf, final int off, final int len)
			throws IOException {
		final int limit = off + len;
		int read = 0;
		try {
			lock.lock();
			for (int i = off; i < limit; i++) {
				if (!buf.isEmpty()) {
					cbuf[i] = buf.poll();
					read++;
					continue;
				}
				final int r = in.read();
				if (r < 0) {
					break;
				}
				final String str = convert(r);
				if (str == null) {
					cbuf[i] = (char) r;
					read++;
					continue;
				}
				for (final char c : str.toCharArray()) {
					buf.add(c);
				}
				if (buf.isEmpty()) {
					/* 空文字の場合は読み進ませない */
					i--;
					continue;
				}
				cbuf[i] = buf.poll();
				read++;
			}
		} finally {
			lock.unlock();
		}
		return read == 0 ? -1 : read;
	}

	/**
	 * @param c
	 *            character to convert
	 * @return a converted string, or {@code null} if unnecessary
	 */
	protected abstract String convert(int c);

	@Override
	public boolean isClosed() {
		return close;
	}

	@Override
	public void close() throws IOException {
		in.close();
		close = true;
	}

}
