/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.lucene.type;

import java.text.DateFormat;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.apache.lucene.document.Fieldable;
import org.mtzky.lucene.descriptor.LucenePropertyConfig;

/**
 * @author mtzky
 */
@LuceneFieldStrategyType(Calendar.class)
public class LuceneCalendarStrategy extends LuceneDateStrategy {

	private final boolean original;
	private final String format;

	/**
	 * @param config
	 */
	public LuceneCalendarStrategy(final LucenePropertyConfig config) {
		super(config);
		this.format = config.getFormat();
		this.original = format.isEmpty();
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(final Fieldable field) {
		final Calendar cal = Calendar.getInstance();
		final String v = field.stringValue();
		if (original) {
			cal.setTimeInMillis(Long.parseLong(v));
			return (T) cal;
		}
		final DateFormat format = new SimpleDateFormat(this.format);
		final Date date = format.parse(v, new ParsePosition(0));
		cal.setTimeInMillis((date != null) ? date.getTime() : 0L);
		return (T) cal;
	}

	@Override
	protected long toLong(final Object value) {
		return (value != null) ? ((Calendar) value).getTimeInMillis() : 0L;
	}

	@Override
	protected String toString(final Object val) {
		final Calendar cal = (val instanceof Calendar) ? (Calendar) val : null;
		final DateFormat format = new SimpleDateFormat(this.format);
		format.setTimeZone(cal.getTimeZone());
		return format.format((cal != null) ? cal.getTime() : new Date(0L));
	}

}
