/*============================================================================*/
/* 変更履歴                                                                   */
/*----------------------------------------------------------------------------*/
/* 1.0 : 新しいAPIに対応                                                      */
/*============================================================================*/

#include <musashi.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/parserInternals.h>
#include <libxml/encoding.h>
#include <iconv.h>
#include <errno.h>
#include <glob.h>
#include <stdarg.h>

#include <xmltagcntHelp.h>
struct mssComHelp comHelp={
  "xmltagcnt",    /* コマンド名       */                                      
  "1.0",          /* バージョン       */
  HELPT,          /* コマンドタイトル */
  HELPS,          /* 要約             */                                    
  HELPE,          /* 利用例           */
  HELPR,          /* 参照コマンド     */
  HELPA,          /* 作者情報         */                                    
  HELPB,          /* バグレポート情報 */
  HELPH           /* ホームページ     */
};          

#define UNDEF 0
#define MAX_NEST 32
#define LOCAL_BUF 256
#define EncMax MssFieldMaxLen  /*iconvで使う出力用文字列長*/

/*----------------------------------------------------------------------------*/
/* 構造体                                                                     */
/*----------------------------------------------------------------------------*/
/*数値タイプ列挙体*/

struct Attribute {
  char *str;
  int   cnt;
};

struct Node {
  char *element;               /*要素文字列*/
  int   cnt;                   /*ノードの出現回数*/
  struct Attribute *attribute; /*属性リスト*/
  int attCnt;                  /*属性の種類数*/
  struct Node  *parent;
  struct mssHash  *children;
};

typedef struct _XmlState {
  int level;
}XmlState;

/*----------------------------------------------------------------------------*/
/* グローバル変数                                                             */
/*----------------------------------------------------------------------------*/
extern struct mssGlobalVariables mssGV;
extern xmlParserCtxtPtr ctxt;
extern xmlParserCtxtPtr ctxt;

static struct mssFPW    *fpw; /*出力ファイル構造体*/
static struct mssHeader *hdo; /*出力ファイル用<head>タグ格納構造体*/
static iconv_t *icid;   /*iconv用 変換ハンドラ*/
static char *inEnc=NULL; /*エンコーディング*/
static char *inVer=NULL; /*バージョン*/
static char currentPath[256];
static struct Node *currentNode;
static struct Node *tree;

/*============================================================================*/
/* オプション宣言＆定義                                                       */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/* 入力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptINF optINF={
    OINF,   /* オプションタイプ                                             */
    "i",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    1,      /* 指定可能の最大ファイル数                                     */
    0,      /*1:file not foundのエラーで終了しない 0:する                   */
    INFT,   /* このオプションのタイトル(Helpで表示)                         */
    INFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 出力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptOTF optOTF={
    OOTF,   /* オプションタイプ                                             */
    "o",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    OTFT,   /* このオプションのタイトル(Helpで表示)                         */
    OTFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 圧縮出力                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLG optZIP={
    OFLG,   /* オプションタイプ                                             */
    "z",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    ZIPT,   /* このオプションのタイトル(Helpで表示)                         */
    ZIPC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* オプションをまとめる                                                       */
/*----------------------------------------------------------------------------*/
void *opt[]={&optINF,&optOTF,&optZIP,NULL};

/*----------------------------------------------------------------------------*/
/* ツリー構造の要素と属性をテーブル形式で書き出し                             */
/*----------------------------------------------------------------------------*/
void printTree(struct Node *node){
  struct mssHashNode *hnode;
  char *pos;
  int i;

  pos=currentPath+strlen(currentPath);

  /*要素名の出力*/
  if(node->element !=NULL){ /*トップノードは飛ばす */
    /*現在のノードパスを更新*/
    strcat(currentPath,"/");
    strcat(currentPath,node->element);

    mssWriteXmlContent(currentPath, icid, fpw);
    mssWriteDlm(fpw); 
    mssWriteNull(fpw); 
    mssWriteDlm(fpw); 
    mssWriteInt(node->cnt,fpw);
    mssWriteRet(fpw);
    mssGV.outCnt++;
  }

  /*属性を伴ったノードの出力*/
  for(i=0; i<node->attCnt; i++){
    mssWriteXmlContent(currentPath, icid, fpw);
    mssWriteDlm(fpw); 

    mssWriteXmlContent((node->attribute+i)->str, icid, fpw);
    mssWriteDlm(fpw); 

    mssWriteInt((node->attribute+i)->cnt,fpw);
    mssWriteRet(fpw);
    mssGV.outCnt++;
  }
  if(node->children==NULL) return;

  for(i=0; i<node->children->hashVal; i++){
    if( NULL != (hnode=*(node->children->node+i)) ){
      while(hnode!=NULL){
        printTree((struct Node *)(hnode->val.v.a));
        hnode=hnode->next;
      }
    }
  }

  /*現在のノードパスを更新*/
  *pos='\0';
}

void freeTree(struct Node *node){
  struct mssHashNode *hnode;
  int i;

  if(node->children!=NULL){
    if(node->attribute!=NULL){
      mssFree(node->attribute);
    }
    for(i=0; i<node->children->hashVal; i++){
      if( NULL != (hnode=*(node->children->node+i)) ){
        while(hnode!=NULL){
          freeTree((struct Node *)(hnode->val.v.a));
          hnode=hnode->next;
        }
      }
    }
    mssFreeHash(node->children);
  }

  if(node!=NULL) mssFree(node->element);
  mssFree(node);
}

/*----------------------------------------------------------------------------*/
/* SAX ハンドラー                                                             */
/*----------------------------------------------------------------------------*/
void start_doc(XmlState *state){
  char *fldNam[3]={"element","attribute","count"};
  char *fldNamEnc[3];
  int i;

  inEnc=mssStrdup((char *)ctxt->input->encoding);
  inVer=mssStrdup((char *)ctxt->version);
  if(inEnc==NULL) inEnc=mssStrdup("UTF-8");

  /*パラメータ用のiconvオープン                                   */
  /*xmlコマンドのパラメータ、入力データ、出力データのencoding     */
  /*は全て、入力データのエンコーディングとおなじことを前提に作る。*/
  icid=iconv_open("UTF-8","euc-jp");
  if((int)icid==-1) {
    mssShowErrMsg("encoding type error in iconv_open");
    mssEnd(mssErrorNoDefault);
  }
  fldNamEnc[0]=mssEncoding(fldNam[0], icid);
  fldNamEnc[1]=mssEncoding(fldNam[1], icid);
  fldNamEnc[2]=mssEncoding(fldNam[2], icid);

  /*パラメータ用のiconvクローズ*/
  if(icid!=NULL) iconv_close(icid);

  /*出力のiconvオープン*/
  icid=iconv_open(inEnc,"UTF-8");
  if((int)icid==-1) {
    mssShowErrMsg("encoding type error in iconv_open");
    mssEnd(mssErrorNoDefault);
  }

  /*出力ヘッダーの作成と出力   */
  hdo=mssInitSetHeader(NULL, NULL, inVer, inEnc, MssXtDefVer);

  for(i=0; i<3; i++){
    mssAddFieldsByStr(hdo->flds, mssEncoding(fldNamEnc[i],icid));
  }

  /*xmlTableヘッダーの書き出し*/
  mssWriteHeader(hdo, fpw);

  /* currentPathの初期化 */
  currentPath[0]='\0';
}

void end_doc(XmlState *state){
  printTree(tree);
  freeTree(tree);

  if(icid!=NULL) iconv_close(icid);

}

/*現在のNodeの子Nodeに指定されたelementがあれば、そのNodeを返す*/
/*なければNULLを返す                                           */
struct Node *getChildNode(struct Node *cn, char *element){
  struct mssHashNode *node;

  if( cn->element ==NULL) return(NULL);
  if( cn->children==NULL) return(NULL);
  node=(struct mssHashNode *) mssHashMember(cn->children, element);

  if( NULL == node){
    return(NULL);
  }else{
    return((struct Node *)node->val.v.a);
  }
}

void addAttribute(struct Node *node, char **atts){
  int i,j;
  int flg;

  if(atts!=NULL){
    i=0;
    while(*(atts+i)!=NULL){

      /*既に存在していればcountUp+continue*/
      flg=0;
      for(j=0; j<node->attCnt; j++){
        if( 0==strcmp(*(atts+i),(node->attribute+j)->str) ){
          (node->attribute+j)->cnt++;
          flg=1;
          break;
        }
      }
      if(flg){
        i=i+2;
        continue;
      }

      /*新規追加登録*/
      node->attribute = mssRealloc(node->attribute,
        sizeof(struct Attribute)*(node->attCnt+1),"xmltree");
      node->attribute->str=mssStrdup(*(atts+i));
      node->attribute->cnt=1;
      i=i+2;
      node->attCnt++;
    }
  }
}

struct Node *addNode(struct Node *cn, char *element, char **atts){
  MssValue node;

  /*新しいノードの確保*/
  node.v.a=mssCalloc(sizeof(struct Node),"xmltree");

  /*エレメント名,新しいノードのアドレスをchildrenに登録*/
  mssHashInsert(cn->children, element, node);

  /*新しいノードの子ノードを初期化*/
  ((struct Node *)(node.v.a))->children = mssInitHash(11);
  ((struct Node *)(node.v.a))->element  = mssStrdup(element);
  ((struct Node *)(node.v.a))->parent   = cn;
  ((struct Node *)(node.v.a))->attribute= NULL;
  ((struct Node *)(node.v.a))->attCnt   = 0;
  ((struct Node *)(node.v.a))->cnt      = 1;

  return(node.v.a);
}

/*エレメント start */
void start_element(XmlState *state, char *fullname, char **atts){
  struct Node *childNode;

  mssGV.inCnt++;
  if(state->level>MAX_NEST){
    mssShowErrMsg("nest level exceed (max=%d)" ,MAX_NEST);
    mssEnd(mssErrorNoDefault);
  }

  /*エレメントの子ノードを得る*/
  childNode=getChildNode(currentNode,fullname);

  /*エレメントの子ノードが無ければ追加する*/
  if( childNode==NULL ){
    childNode=addNode(currentNode,fullname,atts);
  }

  /*属性を追加*/
  addAttribute(childNode, atts);

  /*上の処理で得たエレメントの子ノードをカレントノードとする*/
  currentNode=childNode;
  currentNode->cnt++;

  state->level++;
}

/*エレメント end */
void end_element(XmlState *state, char *fullname, char **atts){
  state->level--;
  if(state->level<0){
    mssShowErrMsg("nest level reach to below 0");
    mssEnd(mssErrorNoDefault);
  }
  currentNode=currentNode->parent;
}

/*sax error handler*/
#include "saxerror.h"

static xmlSAXHandler SAXFunctions = {
    NULL, /* internalSubset */
    NULL, /* isStandalone */
    NULL, /* hasInternalSubset */
    NULL, /* hasExternalSubset */
    NULL, /* resolveEntity */
    NULL, /* getEntity */
    NULL, /* entityDecl */
    NULL, /* notationDecl */
    NULL, /* attributeDecl */
    NULL, /* elementDecl */
    NULL, /* unparsedEntityDecl */
    NULL, /* setDocumentLocator */
    (startDocumentSAXFunc)start_doc, /* startDocument */
    (endDocumentSAXFunc)end_doc, /* endDocument */
    (startElementSAXFunc)start_element, /* startElement */
    (endElementSAXFunc)end_element, /* endElement */
    NULL, /* reference */
    NULL, /* characters */
    NULL, /* ignorableWhitespace */
    NULL, /* processingInstruction */
    NULL, /* comment */
    (warningSAXFunc) xmlSaxErrEnd, /* xmlParserWarning */
    (errorSAXFunc) xmlSaxErrEnd, /* xmlParserError */
    (fatalErrorSAXFunc) xmlSaxErrEnd, /* xmlParserError */
    NULL, /* getParameterEntity */
};

int main(int argc, char *argv[]){

  XmlState         *state;

/*----------------------------------------------------------------------------*/
/* 前処理                                                                     */
/*----------------------------------------------------------------------------*/
  mssInit(argc,argv,&comHelp);        /* シグナル処理などの初期化     */
  mssHelpDoc(opt,&comHelp,argc,argv); /* ヘルプ                       */
  mssSetOption(opt,argc,argv);        /* コマンドオプションの設定     */

  fpw=mssOpenFPW(optOTF.str,optZIP.set,0); /*標準出力オープン*/

/*----------------------------------------------------------------------------*/
/*メインルーチン                                                              */
/*----------------------------------------------------------------------------*/
  tree=mssCalloc(sizeof(struct Node),"xmltree");
  tree->children = mssInitHash(11);
  currentNode=tree;

  state=mssCalloc(sizeof(XmlState),"xml2xt");

  if(optINF.set){
    ctxt=(xmlParserCtxtPtr)xmlCreateFileParserCtxt(optINF.str);
  }else{
    ctxt=(xmlParserCtxtPtr)xmlCreateFileParserCtxt("/dev/stdin");
  }
  if(!ctxt){
    mssShowErrMsg("not xml file\n");
    mssEnd(mssErrorNoDefault);
  }
  ctxt->sax=&SAXFunctions;

  ctxt->userData=state;
  xmlParseDocument(ctxt);
  ctxt->sax=NULL;
  xmlFreeParserCtxt(ctxt);

  if(inEnc!=NULL) mssFree(inEnc);
  if(inVer!=NULL) mssFree(inVer);
  mssFree(state);

/*----------------------------------------------------------------------------*/
/*フッター出力&終了処理                                                       */
/*----------------------------------------------------------------------------*/
  mssWriteFooter(fpw);    /*フッターの出力           */
  mssCloseFPW(fpw);       /*出力ファイルのクローズ   */
  mssFreeHeader(hdo);     /*出力ヘッダ領域開放       */
  mssFreeOption(opt);     /*オプション領域開放       */
  mssShowEndMsg();        /*完了メッセージ           */
  mssEnd(mssExitSuccess); /*終了                     */
  return(0);              /* to avoid warning message*/
}
