/*============================================================================*/
/* 変更履歴                                                                   */
/*----------------------------------------------------------------------------*/
/* 1.0 : 新規作成 2004/06/08                                                  */
/* 1.1 : 環境により座標の小数点があわないバグ修正 2004/10/09                  */
/* 1.2 : yvCnt,xvCntの型変更(BSDでのバグ原因),メモリリーク修正 2004/10/27     */
/* 1.3 : メモリリーク修正 2004/10/31                                          */
/*============================================================================*/

#include <musashi.h>
#include <stdlib.h>
#include <float.h>
#include <math.h>
#include "mssSVG.h"

#include <xtlineHelp.h>
struct mssComHelp comHelp={
  "xtline",        /* コマンド名       */
  "1.3",          /* バージョン       */
  HELPT,          /* コマンドタイトル */
  HELPS,          /* 要約             */
  HELPE,          /* 利用例           */
  HELPR,          /* 参照コマンド     */
  HELPA,          /* 作者情報         */
  HELPB,          /* バグレポート情報 */
  HELPH           /* ホームページ     */
};

#define COLMAX 20 /* 列分類数の上限 */
#define ROWMAX 20 /* 行分類数の上限 */

extern struct mssGlobalVariables mssGV;

extern double LineUnit;
extern iconv_t *icid;   /*iconv用 変換ハンドラ*/

static double CurYplus;
static double CurYminus;
static int rSet=0;
static int cSet=0;
static int eSet=0;

static char *Color[]={
"fuchsia",
"blue",
"orange",
"green",
"silver",
"salmon",
"royalblue",
"lavender",
"navy",
"yellow",
"cyan",
"darkmagenta",
"tomato",
"teal",
"purple",
"deepskyblue",
"paleturquoise",
"palegreen",
"mediumslateblue",
"red"};

double OPT_WIDTH;
double OPT_HEIGHT;
char  *OPT_TITLE;

struct Scale {
  double min;
  double max;
  double unit;
  int    splitCnt;
  int    posSplitCnt;
  int    negSplitCnt;
};

/* 凡例の構造体
               width
        +------------------+
        | +--------------+ |
        | |title         | |
        | +--------------+ |
height  | |+---+         | |
        | ||col| val     | |
        | |+---+         | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | +--------------+ |
        +------------------+
        margin
*/
struct ExNote {
  char  *id;                /* id (exNote) */
  struct SVGcoord   *co;    /* 描画領域 */
  struct SVGrect    *bg;    /* 描画領域のバックグラウンド */
  struct SVGstring  *title; /* title */
  void   **col;   /* カラーボックス */
  struct SVGstring **val;   /* カラーに対応する値 */
  int    cnt;              /* 実際に表示させる凡例の数 */
  double margin;           /* 余白の幅 */
};

/* 棒グラフの構造体
                     width
        +----------------------------------------------------------+
        | +---+------+-------------------------------------------| |
height  | |YT |YV    |                                           | |
        | |   |      |   +----+                                  | |
        | |   | Ymark+---|    |----------------------------------| |
        | |   |      |   +----+     +----+            scale line | |
        | |   |      |   |    |     |    |                       | |
        | |   |      +---|    |-----+----+-----------------------| |
        | |   |      |   |    |     |    |                       | |
        | |   |      |   |    |     +----+                       | |
        | |   |      +---+----+-----|    |-----------------------| |
        | |   |      |   |    |     |    |                       | |
        | |   |      |   |    |     |    |                       | |
        | +---+------+-+-+----+--+--+----+---+---------+---------| |
        | |          |XV                              Xmark      | |
        | |          +-------------------------------------------+ |
        | |          |XT                                         | |
        | +----------+-------------------------------------------+ |
        +----------------------------------------------------------+
        margin
*/
struct LineChart {
  char  *id;             /* id (g#-#) */
  struct SVGcoord   *co; /* 描画領域 */
  struct SVGrect    *bg; /* 領域のバックグラウンド */

  struct SVGstring  *xt; /* X title */
  struct SVGstring  *yt; /* Y title */
  struct SVGstring **xv; /* X value */
  struct SVGstring **yv; /* Y value */
  struct SVGline    *xa; /* X axis */
  struct SVGline    *ya; /* Y axis */
  struct SVGline   **xm; /* X mark */
  struct SVGline   **ym; /* Y mark */
  struct SVGline   **ys; /* scale line */

  struct SVGcoord   *g;  /* グラフ描画領域 */

  int    xvCnt;          /* X軸の値の数 */
  int    yvCnt;          /* Y軸の値の数 */

  double margin;         /* 余白の幅 */
  int ratio;             /* 構成比グラフかどうか */
  char vhFlg;            /* 横棒か縦棒か(v:縦,h:横) */

  double unit;           /* -fの値の単位あたりのピクセル数 */
  double origin;         /* 原点のy座標 */
  double dotSize;        /* ドットサイズ(ham)*/
};

/* コンテナの構造体
                     width
        +----------------------------------------------------------+
        | +------------------------------------------------------+ |
        | |Title (tWD*tHT)                                       | |
        | +-------------+---------------------------------+------+ |
height  | |             |CT     (CTWD*CTHT)               |      | |
        | |             +----------------+----------------+      | |
        | |             |CV   (CVWD*CVHT)|                |      | |
        | +------+------+----------------+----------------+------+ |
        | |RT    |RV    |                |                |EX    | |
        | |      |      | グラフ描画領域 |                |      | |
        | |RTHT* |RVHT* | G(0,0)         | G(1,0)         |EXWD* | |
        | |  RTWD|  RVWD|                |                |  EXHT| |
        | |      +------+----------------+----------------+      | |
        | |      |      |                |                |      | |
        | |      |      |                |                |      | |
        | |      |      | G(0,1)         | G(i,j)         |      | |
        | |      |      |                |                |      | |
        | +------+------+----------------+----------------+------+ |
        +----------------------------------------------------------+
        margin
*/
struct Container {
  char  *id;               /* id (root) */
  struct SVGcoord   *co;   /* 描画領域 */
  struct SVGrect    *bg;   /* 領域のバックグラウンド(margin除く) */

  struct SVGstring *title; /* タイトル */
  struct SVGstring *ct;    /* 列 title */
  struct SVGstring *rt;    /* 行 title */
  struct SVGstring **cv;   /* 列 value */
  struct SVGstring **rv;   /* 行 value */

  struct LineChart   *line[COLMAX][ROWMAX];
  struct ExNote     *exNote;

  int cvCnt;             /* 列数 */
  int rvCnt;             /* 行数 */
  int fix;               /* グラフ描画領域をX:Y=3:2に固定するかどうか */
  double margin;         /* 余白の幅 */
  double pWidth;         /* rootWindowの物理的な幅 */
  double pHeight;        /* rootWindowの物理的な高さ */
};

struct KeyInfo {
  struct mssFields *rFld;
  struct mssFields *cFld;
  struct mssFields *kFld;
  struct mssFields *eFld;
  struct mssFields *fFld;
  struct mssHash *rHash;
  struct mssHash *cHash;
  struct mssHash *kHash;
  struct mssHash *eHash;
  char **rAdd;
  char **cAdd;
  char **kAdd;
  char **eAdd;
  struct Scale *scale[COLMAX][ROWMAX];

  int rKB,cKB,kKB,eKB;
};

void setKeyBreakInfo(struct KeyInfo *keyInfo, struct mssFldRecDbl *frd){
  int fn;
  char **newPnt;
  char **oldPnt;

  /*初期化*/
  keyInfo->rKB=0;
  keyInfo->cKB=0;
  keyInfo->kKB=0;
  keyInfo->eKB=0;

  /*最初行の時はキーブレーク*/
  if(frd->firstRead){
    frd->firstRead=0;
    return;
  }

  /*最終行の時はキーブレーク*/
  if(frd->eof == 1){
    keyInfo->rKB=1;
    keyInfo->cKB=1;
    keyInfo->kKB=1;
    keyInfo->eKB=1;
    return;
  }

  newPnt=frd->pnt[frd->new];
  oldPnt=frd->pnt[frd->old];

  /* -rの比較 */
  if(rSet){
    fn=MssFlds2num(keyInfo->rFld,0);
    if( 0 != strcmp(*(newPnt+fn),*(oldPnt+fn)) ){
      keyInfo->rKB=1;
      keyInfo->cKB=1;
      keyInfo->kKB=1;
      keyInfo->eKB=1;
      return;
    }else{
      keyInfo->rKB=0;
    }
  }

  /* -cの比較 */
  if(cSet){
    fn=MssFlds2num(keyInfo->cFld,0);
    if( 0 != strcmp(*(newPnt+fn),*(oldPnt+fn)) ){
      keyInfo->cKB=1;
      keyInfo->kKB=1;
      keyInfo->eKB=1;
      return;
    }else{
      keyInfo->cKB=0;
    }
  }

  /* -kの比較 */
  fn=MssFlds2num(keyInfo->kFld,0);
  if( 0 != strcmp(*(newPnt+fn),*(oldPnt+fn)) ){
    keyInfo->kKB=1;
    keyInfo->eKB=1;
    return;
  }else{
    keyInfo->kKB=0;
  }

  /* -eの比較 */
  if(eSet){
    fn=MssFlds2num(keyInfo->eFld,0);
    if( 0 != strcmp(*(newPnt+fn),*(oldPnt+fn)) ){
      keyInfo->eKB=1;
      return;
    }else{
      keyInfo->eKB=0;
    }
  }
  return;
}

char **setKeyVal(char **list, struct mssHash *hash,char *val){
  struct mssHashNode *node;
  MssValue v;
  mssVinit(&v,INT); /* 並べ替えた後で順序番号を格納する */

  node=mssHashInsert(hash,val,v);
  if(node!=NULL){
    list=mssRealloc(list,sizeof(char *) * hash->cnt,"setKeyVal");
    *(list+hash->cnt-1) = node->str;
  }
  return(list);
}

void sortHashAdd(struct mssHash *hash,char **add, int cnt,MssOptFLD *optFld){
  int flag;
  int i;
  struct mssHashNode *node;

  if(mssIsFldOptOn(optFld,0,'n')){
    if(mssIsFldOptOn(optFld,0,'r')) flag=3;
    else                            flag=2; 
  }else{
    if(mssIsFldOptOn(optFld,0,'r')) flag=1;
    else                            flag=0; 
  }

  mssSort(add, cnt, flag);

  for(i=0; i<cnt; i++){
    node=mssHashMember(hash,*(add+i));
    if(node!=NULL){
      node->val.v.i=i;
    }
  }
}

struct Scale *setScale(double min, double max, int splitCnt){
  double unit;
  double limit;
  double unitA;
  double unitB;
  struct Scale *scale;
  struct Scale *posScale;
  struct Scale *negScale;

  scale=malloc(sizeof(struct Scale));
  scale->splitCnt = splitCnt;

  /*-------------------------------------------------*/
  /* min,maxともに正もしくは負の場合は調整           */
  /*-------------------------------------------------*/
  if(min>0 && max>0) min=0;
  if(min<0 && max<0) max=0;

  /*-------------------------------------------------*/
  /* 正と負の分割数(posSplitCnt,negSplitCnt)を決める */
  /*-------------------------------------------------*/
  scale->posSplitCnt = (int)floor((max/(max+min*(-1)))*(double)splitCnt+0.5);
  scale->negSplitCnt = splitCnt - scale->posSplitCnt;

  /* 微調整 */
  if( min <0 && scale->negSplitCnt<=0 ){
    scale->posSplitCnt--;
    scale->negSplitCnt++;
  }
  if( max >0 && scale->posSplitCnt<=0 ){
    scale->posSplitCnt++;
    scale->negSplitCnt--;
  }

  /*-------------------------------------------------*/
  /* 単位(unit)を決める                              */
  /*-------------------------------------------------*/

  /* min,maxが正負なら、それぞれ別々にスケールを求める*/
  if( min < 0 && max > 0 ){
    posScale=setScale(0,max,scale->posSplitCnt);
    negScale=setScale(min,0,scale->negSplitCnt);
    if(posScale->unit > negScale->unit) scale->unit=posScale->unit;
    else                                scale->unit=negScale->unit;
    free(posScale);
    free(negScale);

  /* min,maxのいずれかが0の場合の単位を求める*/
  }else{
 
    /* スケールは10進数表記なので、できるだけ桁数の少ない単位を設定する。
       また単位に基づいた最大値が元の最大値の1.2倍を越えない場合はunitAを
       越える場合はunitBを用いる。   例) unit  unitA unitB
                                         3456  4000  3500 
                                         0.23  0.3   0.23 */
    unit=(max-min)/(double)splitCnt;
    unitA = pow(10,floor(log10(unit)));
    unitB = unitA/10;
    unitA = unitA * ceil(unit/unitA);
    unitB = unitB * ceil(unit/unitB);
    if( max == 0 ) limit=min*(-1.2);
    else           limit=max*( 1.2);
    if(unitA*(double)splitCnt <= limit) scale->unit=unitA;
    else                                scale->unit=unitB;
  }

  /*-------------------------------------------------*/
  /* 最小、最大(min,max)を求める                     */
  /*-------------------------------------------------*/
  scale->max=scale->unit * scale->posSplitCnt;
  scale->min=scale->unit * scale->negSplitCnt * (-1);

/*
printf("min=%g max=%g splitCnt=%d\n",min,max,splitCnt);
printf("SplitCnt(pos,neg)=(%d,%d)\n",scale->posSplitCnt,scale->negSplitCnt);
printf("unit=%g min=%g max=%g\n\n",scale->unit,scale->min,scale->max);
*/

  return(scale);
}

void drawShapes(double x, double y, double w, int eNum, int swidth, struct mssFPW *fpw) {
  /********** 20 Shapes **********/ 
  /***** Shape 0 -- square *****/
  if(eNum == 0) {
    struct SVGsq *sq;
    sq=initSVGsq(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGsq(sq, fpw);
    freeSVGsq(sq);
  }
  
  /***** Shape 1 -- circle *****/
  else if(eNum == 1) { 
  struct SVGcircle *circle;
    circle=initSVGcircle(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGcircle(circle, fpw);
    freeSVGcircle(circle);
  }

  /***** Shape 2 --  rectangle *****/
  else if(eNum == 2) {
    struct SVGv_rect *v_rect; 
    v_rect=initSVGv_rect(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGv_rect(v_rect, fpw);
    freeSVGv_rect(v_rect);
  }

  /***** Shape 3 -- funnel *****/
  else if(eNum == 3) {
    struct SVGfun *fun; 
    fun=initSVGfun(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGfun(fun, fpw);
    freeSVGfun(fun);
  }

  /***** Shape 4 --  ellipse *****/
  else if(eNum == 4) {
    struct SVGell *ell;
    ell=initSVGell(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGell(ell, fpw);
    freeSVGell(ell);
  }
  
  /***** Shape 5 -- i *****/
  else if(eNum == 5) {
    struct SVGi *i;
    i=initSVGi(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGi(i, fpw);
    freeSVGi(i);
  }

  /***** Shape 6 -- diamond *****/
  else if(eNum == 6) {
    struct SVGdia *dia; 
    dia=initSVGdia(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGdia(dia, fpw);
    freeSVGdia(dia);
  }

  /***** Shape 7 -- triangle *****/
  else if(eNum == 7) { 
  struct SVGtri *tri;
    tri=initSVGtri(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGtri(tri, fpw);
    freeSVGtri(tri);
  }

  /***** Shape 8 -- cross *****/
  else if(eNum == 8) {
    struct SVGcross *cross;
    cross=initSVGcross(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGcross(cross, fpw);
    freeSVGcross(cross);
  }
  
  /***** Shape 9 -- hexagon *****/
  else if(eNum == 9) { 
    struct SVGhex *hex;
    hex=initSVGhex(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGhex(hex, fpw);
    freeSVGhex(hex);
  }

  /***** Shape 10 -- crown *****/
  else if(eNum == 10) { 
    struct SVGcrown *crown;
    crown=initSVGcrown(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGcrown(crown, fpw);
    freeSVGcrown(crown);
  }

  /***** Shape 11 -- u *****/
  else if(eNum == 11) {
    struct SVGu *u;
    u=initSVGu(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGu(u, fpw);
    freeSVGu(u);
  }

  /***** Shape 12 -- star *****/
  else if(eNum == 12) {
  struct SVGstar *star;
    star=initSVGstar(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGstar(star, fpw);
    freeSVGstar(star);
  }
  
  /***** Shape 13 -- arrow  *****/
  else if(eNum == 13) { 
    struct SVGarrow *arrow;
    arrow=initSVGarrow(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGarrow(arrow, fpw);
    freeSVGarrow(arrow);
  }

  /***** Shape 14 -- fugo *****/
  else if(eNum == 14) {
    struct SVGfugo *fugo; 
    fugo=initSVGfugo(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGfugo(fugo, fpw);
    freeSVGfugo(fugo);
  }

  /***** Shape 15 -- tree *****/
  else if(eNum == 15) {
    struct SVGtree *tree; 
    tree=initSVGtree(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGtree(tree, fpw);
    freeSVGtree(tree);
  }

  /***** Shape 16 --  house *****/
  else if(eNum == 16) {
    struct SVGhouse *house;
    house=initSVGhouse(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGhouse(house, fpw);
    freeSVGhouse(house);
  }
  
  /***** Shape 17 -- male *****/
  else if(eNum == 17) {
    struct SVGmale *male; 
    male=initSVGmale(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGmale(male, fpw);
    freeSVGmale(male);
  }

  /***** Shape 18 -- cone *****/
  else if(eNum == 18) {
    struct SVGcone *cone; 
    cone=initSVGcone(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGcone(cone, fpw);
    freeSVGcone(cone);
  }

  /***** Shape 19 -- watch *****/
  else  {
    struct SVGwatch *watch; 
    watch=initSVGwatch(x, y, w, w, swidth, Color[eNum], NULL);
    writeSVGwatch(watch, fpw);
    freeSVGwatch(watch);
  }
}
/*************** End of Dawing Shapes ***************/
/* 凡例の構造体
          TX     WD
        +-|----------------+
    TY----+------TW------+ |
        | |title        TH |
        | +--------------+ |
    HT  | |+---+ CW*CH   | |
        | ||col| val     | |
        | |+---+  VW*VH  | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | |              | |
        | +--------------+ |
        +------------------+
        MARGIN
*/
struct ExNote *setExNote(double X, double Y, double W, double H, char *title,
  struct KeyInfo *keyInfo){

  struct ExNote *exNote;
  double FontUnit;
  double x,y,w,h;
  char *s;
  char buf[20];
  int i;
  double WD      = W;   /*描画領域の幅*/
  double HT      = H;   /*描画領域の高さ(親の比率で決まる)*/
  double MARGIN  = 15;  /*余白*/

  double TWD;          /*タイトルの幅*/
  double THT;          /*タイトルの高さ*/
  double TFS    = 12;  /*タイトルのフォントサイズ*/

  double CWD;          /*color boxの値の高さ*/
  double CHT;          /*color boxの値の幅  */

  double VWD;          /* val の値の幅*/
  double VHT;          /* val の値の高さ*/
  double VFS    = 10;  /* val のフォントサイズ*/

  double CVM;          /* col と valの間の幅 */
  double LNM;          /* LINE間のマージン */

  double GWD;           /*グラフ描画領域の幅*/
  double GHT;           /*グラフ描画領域の高さ*/

  /* フォント単位の決定(FontUnit) */
  /* 固定フォントのw,hを考慮して厳密に決めるべき*/
  if(WD>HT) FontUnit=HT/100;
  else      FontUnit=WD/100;

  /*----------------------------*/
  /* 各要素の幅と高さを調整する */
  /*----------------------------*/
  GWD = W;
  GHT = H;

  LNM = H*0.01;
  THT = TFS*FontUnit;
  VHT = VFS*FontUnit;
  CHT = VFS*FontUnit;
  if(keyInfo->eHash->cnt * (VHT+LNM) > GHT-THT){
    VHT=(GHT-THT)/(double) keyInfo->eHash->cnt - LNM;
    CHT=VHT;
  }
 
  CVM = W*0.05;
  TWD = GWD;
  CWD = CHT*1.5;
  VWD = GWD-CWD-CVM-MARGIN;

  exNote=mssCalloc(sizeof(struct ExNote),"setExNote");

  exNote->cnt=keyInfo->eHash->cnt;
  exNote->col=mssCalloc(sizeof(void *)*exNote->cnt,"setExNote");
  exNote->val=mssCalloc(sizeof(struct ExNote)*exNote->cnt,"setExNote");

  /*----------------------------*/
  /* 座標                       */
  /*----------------------------*/
  exNote->co=initSVGcoord(X,Y,W,H);

  /*----------------------------*/
  /* タイトル                   */
  /*----------------------------*/
  x=X; y=Y; w=TWD; h=THT;
  s=MssFlds2name(keyInfo->eFld,0);
  exNote->title=initSVGstring(x,y,w,h,0,1,"middle","exnote",s);

  /*----------------------------*/
  /* color box                  */
  /*----------------------------*/
  for(i=0; i<exNote->cnt; i++){
    x=X+MARGIN; y=Y+MARGIN+THT+(double)i*(CHT+LNM); w=CWD*0.5;
// h=CHT;
    s=*(keyInfo->eAdd+i);
    sprintf(buf,"exCol-%d",i);

    if      (i==0)  *(exNote->col+i) = initSVGsq(x,y,w,w,0,Color[i],buf);
    else if (i==1)  *(exNote->col+i) = initSVGcircle(x,y,w,w,0,Color[i],buf);
    else if (i==2)  *(exNote->col+i) = initSVGv_rect(x,y,w,w,0,Color[i],buf);
    else if (i==3)  *(exNote->col+i) = initSVGfun(x,y,w,w,0,Color[i],buf);
    else if (i==4)  *(exNote->col+i) = initSVGell(x,y,w,w,0,Color[i],buf);
    else if (i==5)  *(exNote->col+i) = initSVGi(x,y,w,w,0,Color[i],buf);
    else if (i==6)  *(exNote->col+i) = initSVGdia(x,y,w,w,0,Color[i],buf);
    else if (i==7)  *(exNote->col+i) = initSVGtri(x,y,w,w,0,Color[i],buf);
    else if (i==8)  *(exNote->col+i) = initSVGcross(x,y,w,w,0,Color[i],buf);
    else if (i==9)  *(exNote->col+i) = initSVGhex(x,y,w,w,0,Color[i],buf);
    else if (i==10) *(exNote->col+i) = initSVGcrown(x,y,w,w,0,Color[i],buf);
    else if (i==11) *(exNote->col+i) = initSVGu(x,y,w,w,0,Color[i],buf);
    else if (i==12) *(exNote->col+i) = initSVGstar(x,y,w,w,0,Color[i],buf);
    else if (i==13) *(exNote->col+i) = initSVGarrow(x,y,w,w,0,Color[i],buf);
    else if (i==14) *(exNote->col+i) = initSVGfugo(x,y,w,w,0,Color[i],buf);
    else if (i==15) *(exNote->col+i) = initSVGtree(x,y,w,w,0,Color[i],buf);
    else if (i==16) *(exNote->col+i) = initSVGhouse(x,y,w,w,0,Color[i],buf);
    else if (i==17) *(exNote->col+i) = initSVGmale(x,y,w,w,0,Color[i],buf);
    else if (i==18) *(exNote->col+i) = initSVGcone(x,y,w,w,0,Color[i],buf);
    else            *(exNote->col+i) = initSVGwatch(x,y,w,w,0,Color[i],buf);
  }

  /*----------------------------*/
  /* val                        */
  /*----------------------------*/
  for(i=0; i<exNote->cnt; i++){
    x=X+MARGIN+CWD+CVM; y=Y+MARGIN+THT+(double)i*(VHT+LNM); w=VWD; h=VHT;
    s=*(keyInfo->eAdd+i);
    sprintf(buf,"exVal-%d",i);
    *(exNote->val+i)=initSVGstring(x,y,w,h,0,0,"start",buf,s);
  }

  /*----------------------------*/
  /* バックグラウンド領域       */
  /*----------------------------*/
  x=X; y=Y; w=WD; h=HT;
  sprintf(buf,"ex-bg");
  exNote->bg=initSVGrect(x,y,w,h,1,"none",buf);

  return(exNote);
}

void freeExNote(struct ExNote *exNote){
  int i;

  if(exNote==NULL) return;

  freeSVGcoord(exNote->co);
  freeSVGstring(exNote->title);
  freeSVGrect(exNote->bg);

  if(exNote->col!=NULL){
    for(i=0; i<exNote->cnt; i++){
      freeSVGrect(*(exNote->col+i));
    }
    mssFree(exNote->col);
  }
  if(exNote->val!=NULL){
    for(i=0; i<exNote->cnt; i++){
      freeSVGstring(*(exNote->val+i));
    }
    mssFree(exNote->val);
  }
  mssFree(exNote);
}

void writeExNote( struct ExNote *ex, struct mssFPW *fpw ){
  int i;
  /* 描画領域 */
  writeSVGrect(ex->bg,fpw);

  /* タイトル */
  writeSVGstring(ex->title,fpw);

  for(i=0; i<ex->cnt; i++){
    if      (i==0)  writeSVGsq(*(ex->col+i),fpw);
    else if (i==1)  writeSVGcircle(*(ex->col+i),fpw);
    else if (i==2)  writeSVGv_rect(*(ex->col+i),fpw);
    else if (i==3)  writeSVGfun(*(ex->col+i),fpw);
    else if (i==4)  writeSVGell(*(ex->col+i),fpw);
    else if (i==5)  writeSVGi(*(ex->col+i),fpw);
    else if (i==6)  writeSVGdia(*(ex->col+i),fpw);
    else if (i==7)  writeSVGtri(*(ex->col+i),fpw);
    else if (i==8)  writeSVGcross(*(ex->col+i),fpw);
    else if (i==9)  writeSVGhex(*(ex->col+i),fpw);
    else if (i==10) writeSVGcrown(*(ex->col+i),fpw);
    else if (i==11) writeSVGu(*(ex->col+i),fpw);
    else if (i==12) writeSVGstar(*(ex->col+i),fpw);
    else if (i==13) writeSVGarrow(*(ex->col+i),fpw);
    else if (i==14) writeSVGfugo(*(ex->col+i),fpw);
    else if (i==15) writeSVGtree(*(ex->col+i),fpw);
    else if (i==16) writeSVGhouse(*(ex->col+i),fpw);
    else if (i==17) writeSVGmale(*(ex->col+i),fpw);
    else if (i==18) writeSVGcone(*(ex->col+i),fpw);
    else            writeSVGwatch(*(ex->col+i),fpw);
  }

  for(i=0; i<ex->cnt; i++){
    writeSVGstring(*(ex->val+i),fpw);
  }
}

/**
 *                     WD
 *      +-----------------------------------------------------------+
 *    ----+---+-------+-------------------------------------------+ |
 *      | |YTHT|YVWD  |                                           | |
 *    G | | *  | *    |   +----+                                  | |
 *    H | |YTWD|YVHT  +---|    |----------------------------------| |
 *    T | |    |      |   +----+     +----+            scale line | |
 * HT   | |    |      |   |    |     |    |                       | |
 *      | |    | Ymark+---|    |-----+----+-----------------------| |
 *      | |    |      |   |    |     |    |                       | |
 *      | |    |      |   |    |     +----+                       | |
 *      | |    |      +---+----+-----|    |-----------------------| |
 *      | |    |      |   |    |     |    |                       | |
 *      | |    |      |   |    |     |    |                       | |
 *    ----+----+------+-+-+----+--+--+----+---+---------+---------| |
 *      | |           |XVWD*XVHT                       Xmark      | |
 *      | |           +-------------------------------------------+ |
 *      | |           |XTWD*XVHT                                  | |
 *      | +-----------+-------------------------------------------+ |
 *      +-------------|-------------------------------------------|-+
 *      MARGIN        |                GWD                        |
 */
struct LineChart *setLineChart(
  int cNo,
  int rNo,
  double X, double Y, double W, double H,
  struct KeyInfo *keyInfo){

  struct LineChart *line;
  double x,y,w,h;
  double x1,y1,x2,y2;
  char *s;
  int i;
  double unit,min;
  char buf[20];
  double FontUnit;

  /*幅と高さ*/
  double WD      = W;   /*描画領域の幅*/
  double HT      = H;   /*描画領域の高さ(親の比率で決まる)*/
  double MARGIN  = H*0.05; /*余白*/

  double XTWD;          /*X軸のタイトルの幅*/
  double XTHT;          /*X軸のタイトルの高さ*/
  double XTFS    = 7;   /*Y軸タイトルのフォントサイズ*/

  double YTWD;          /*Y軸のタイトルの幅*/
  double YTHT;          /*Y軸のタイトルの高さ*/
  double YTFS    = 7;   /*Y軸タイトルのフォントサイズ*/

  double XVWD;          /*X軸の値の幅*/
  double XVHT;          /*X軸の値の高さ*/
  double XVFS    = 6;   /*Y軸タイトルのフォントサイズ*/

  double YVWD;          /*Y軸の値の高さ*/
  double YVHT;          /*Y軸の値の幅 ????????? */
  double YVFS    = 6;   /*Y軸タイトルのフォントサイズ*/

  double GWD;           /*グラフ描画領域の幅*/
  double GHT;           /*グラフ描画領域の高さ*/

  double DOTU    = 3;   /*ドットのサイズ(ham)*/

  /* フォント単位の決定(FontUnit) */
  /* 固定フォントのw,hを考慮して厳密に決めるべき*/
  if(WD>HT) FontUnit=HT/100;
  else      FontUnit=WD/100;

  /*----------------------------*/
  /* 各要素の幅と高さを調整する */
  /*----------------------------*/
  XTHT = XTFS*FontUnit;
  YTHT = YTFS*FontUnit;
  XVHT = XVFS*FontUnit;
  YVWD = YVFS*FontUnit*4;

  XTWD = WD-YTHT-YVWD-MARGIN*2;
  YTWD = HT-XTHT-XVHT-MARGIN*2;
  XVWD = XTWD/keyInfo->kHash->cnt;
  YVHT = YVFS*FontUnit;

  GWD  = WD-YTHT-YVWD-MARGIN*2;
  GHT  = HT-XTHT-XVHT-MARGIN*2;

  line=mssCalloc(sizeof(struct LineChart),"setLineChart");

  /*----------------------------*/
  /* id                         */
  /*----------------------------*/
  sprintf(buf,"g%d-%d",cNo,rNo);
  line->id=mssStrdup(buf);

  /*----------------------------*/
  /* X軸タイトル                */
  /*----------------------------*/
  x=X+MARGIN+YTHT+YVWD; y=Y+MARGIN+GHT+XVHT; w=XTWD; h=XTHT;
  s=MssFlds2name(keyInfo->kFld,0);
  sprintf(buf,"g%d-%d-xt",cNo,rNo);
  line->xt=initSVGstring(x,y,w,h,0,0,"middle",buf,s);

  /*----------------------------*/
  /* Y軸タイトル                */
  /*----------------------------*/
  x=X+MARGIN; y=Y+MARGIN; w=YTWD; h=YTHT;
  s=MssFlds2name(keyInfo->fFld,0);
  sprintf(buf,"g%d-%d-yt",cNo,rNo);
  line->yt=initSVGstring(x,y,w,h,1,0,"middle",buf,s);

  /*----------------------------*/
  /* X軸値                      */
  /*----------------------------*/
  line->xvCnt=keyInfo->kHash->cnt;
  line->xv=mssMalloc(sizeof(struct SVGstring *)*keyInfo->kHash->cnt,"setLine");
  for(i=0; i<line->xvCnt; i++){
    x=X+MARGIN+YTHT+YVWD+(double)(double)i*XVWD; y=Y+MARGIN+GHT; w=XVWD; h=XVHT;
    s=*(keyInfo->kAdd+i);
    sprintf(buf,"g%d-%d-xv%d",cNo,rNo,i);
    *(line->xv+i)=initSVGstring(x,y,w,h,0,0,"middle",buf,s);
  }

  /*----------------------------*/
  /* Y軸値                      */
  /*----------------------------*/
  line->yvCnt=keyInfo->scale[cNo][rNo]->splitCnt+1;
  line->yv=mssMalloc(sizeof(struct SVGstring *)*line->yvCnt,"setLine");
  unit=keyInfo->scale[cNo][rNo]->unit;
  min =keyInfo->scale[cNo][rNo]->min;
  for(i=0; i<line->yvCnt; i++){
    x=X+MARGIN+YTHT;
    y=Y+MARGIN+GHT-(double)i*GHT/(double)(line->yvCnt-1)-YVFS*FontUnit/2;
    w=YVWD-YVWD*0.1; h=YVHT;
    s=mssFtoA(min+unit*i);
    sprintf(buf,"g%d-%d-yv%d",cNo,rNo,i);
    *(line->yv+i)=initSVGstring(x,y,w,h,0,0,"end",buf,s);
    mssFree(s); /* 追加 2004/10/31 */
  }

  /*----------------------------*/
  /* 描画領域                   */
  /*----------------------------*/
  x=X; y=Y; w=WD; h=HT;
  line->co=initSVGcoord(x,y,w,h);

  /*----------------------------*/
  /* グラフ領域                 */
  /*----------------------------*/
  x=X+MARGIN+YTHT+YVWD; y=Y+MARGIN; w=GWD; h=GHT;
  line->g=initSVGcoord(x,y,w,h);

  /*----------------------------*/
  /*  -f の単位あたりピクセル数 */
  /*----------------------------*/
  line->unit = GHT / (keyInfo->scale[cNo][rNo]->max -
                     keyInfo->scale[cNo][rNo]->min );

  /*----------------------------*/
  /*  原点のY座標               */
  /*----------------------------*/
  line->origin=line->g->y + line->g->h *
             ((double)keyInfo->scale[cNo][rNo]->posSplitCnt / 
              (double)keyInfo->scale[cNo][rNo]->splitCnt    );

  /*----------------------------*/
  /* X軸                        */
  /*----------------------------*/
  x1=line->g->x;           y1=line->origin;
  x2=line->g->x+line->g->w; y2=y1;
  sprintf(buf,"g%d-%d-xa",cNo,rNo);
  line->xa=initSVGline(x1,y1,x2,y2,1,"black",NULL,buf);

  /*----------------------------*/
  /* Y軸                        */
  /*----------------------------*/
  x1=line->g->x; y1=line->g->y;
  x2=x1;        y2=line->g->y+line->g->h;
  sprintf(buf,"g%d-%d-ya",cNo,rNo);
  line->ya=initSVGline(x1,y1,x2,y2,1,"black",NULL,buf);

  /*----------------------------*/
  /* X軸マーク                  */
  /*----------------------------*/
  line->xm=mssMalloc(sizeof(struct SVGline *)*line->xvCnt,"setLine");
  for(i=0; i<line->xvCnt; i++){
    x1 = X+MARGIN+YTHT+YVWD+(double)(i+1)*XVWD; x2=x1;
    y1=line->g->y+line->g->h - line->g->h*0.01;
    y2=line->g->y+line->g->h + line->g->h*0.01;
    s=*(keyInfo->kAdd+i);
    sprintf(buf,"g%d-%d-xm%d",cNo,rNo,i);
    *(line->xm+i)=initSVGline(x1,y1,x2,y2,1,"black",NULL,buf);
  }

  /*----------------------------*/
  /* Y軸マーク                  */
  /*----------------------------*/
  line->ym=mssMalloc(sizeof(struct SVGline *)*line->yvCnt,"setline");
  for(i=0; i<line->yvCnt; i++){
    x1=line->g->x - line->g->w*0.01;
    x2=line->g->x + line->g->w*0.01;
    y1 = Y+MARGIN+GHT-(double)i*GHT/(double)(line->yvCnt-1); y2=y1;
    s=*(keyInfo->kAdd+i);
    sprintf(buf,"g%d-%d-ym%d",cNo,rNo,i);
    *(line->ym+i)=initSVGline(x1,y1,x2,y2,1,"black",NULL,buf);
  }

  /*----------------------------*/
  /* Y軸スケール                */
  /*----------------------------*/
  line->ys=mssMalloc(sizeof(struct SVGline *)*line->yvCnt,"setLine");
  for(i=0; i<line->yvCnt; i++){
    x1=line->g->x; x2=line->g->x + line->g->w;
    y1 = Y+MARGIN+GHT-(double)i*GHT/(double)(line->yvCnt-1); y2=y1;
    s=*(keyInfo->kAdd+i);
    sprintf(buf,"g%d-%d-ys%d",cNo,rNo,i);
    *(line->ys+i)=initSVGline(x1,y1,x2,y2,0.5,"black","1,2",buf);
  }


  /*----------------------------*/
  /* バックグラウンド領域       */
  /*----------------------------*/
  x=X; y=Y; w=WD; h=HT;
  sprintf(buf,"g%d-%d-bg",cNo,rNo);
  line->bg=initSVGrect(x,y,w,h,1,"none",buf);

  /*余白のセット*/
  line->margin=MARGIN;

  /*ドットサイズのセット(ham)*/
  line->dotSize=FontUnit*DOTU;
  return(line);
}

void freeLineChart(struct LineChart *line){
  int i;

  if(line==NULL) return;

  mssFree(line->id);
  freeSVGstring(line->xt);
  freeSVGstring(line->yt);
  freeSVGcoord(line->co);
  freeSVGcoord(line->g);
  freeSVGline(line->xa);
  freeSVGline(line->ya);
  freeSVGrect(line->bg);

  for(i=0; i<line->yvCnt; i++){
    freeSVGstring(*(line->yv+i));
    freeSVGline(*(line->ys+i));
    freeSVGline(*(line->ym+i));
  }
  mssFree(line->yv);
  mssFree(line->ys);
  mssFree(line->ym);
  for(i=0; i<line->xvCnt; i++){
    freeSVGstring(*(line->xv+i));
    freeSVGline(*(line->xm+i));
  }
  mssFree(line->xv);
  mssFree(line->xm);

  mssFree(line);

}


/**
 * =============================================================================
 * ルートコンテナの設定
 *      |            WD(PWD)                                       |
 * -----+----------------------------------------------------------+
 *      | +------------------------------------------------------+ |
 *      | |Title (TWD*THT)                                       | |
 * HT   | +-------------+---------------------------------+------+ |
 *(PHT) | |             |CT     (CTWD*CTHT)               |      | |
 *      | |             +----------------+----------------+      | |
 *      | |             |CV   (CVWD*CVHT)|                |      | |
 *    ----+------+------+----------------+----------------+------+ |
 *      | |RT    |RV    |                |                |EX    | |
 *      | |      |      | グラフ描画領域 |                |      | |
 *      | |RTHT* |RVHT* | G(0,0)         | G(1,0)         |EXWD* | |
 *     G| |  RTWD|  RVWD|                |                |  EXHT| |
 *     H| |      +------+----------------+----------------+      | |
 *     T| |      |      |                |                |      | |
 *      | |      |      |                |                |      | |
 *      | |      |      | G(0,1)         | G(i,j)         |      | |
 *      | |      |      |                |                |      | |
 *    ----+------+------+----------------+----------------+------+ |
 * -----+---------------|----------------------------------------|-+
 *      MARGIN          |                GWD                     |
 *
 *  WD=GWD+RVHT+RTHT+MARGIN*2
 *  HT=GHT+CVHT+CTHT+MARGIN*2
 * =============================================================================
 * フォントサイズの単位
 * ルートコンテナのサイズに合わせて、フォントサイズは決められる
 * ルートコンテナのx,yで小さい値を100で割った数字
 * デフォルトでは1600*1200なので FontUnit=1600/10=16
 * グラフ上に表示される文字は[相対的な大きさ]*FontUnitにて
 * 最終的なフォントサイズが決まる
 * 相対的な大きさは、各種画面設定ルーチン(setContainerなど)にて定義されている。
 * この変数はsetContainer関数で定義される。
 */
struct Container *setContainer(
  char *title,  /* タイトル(NULLなら表示領域確保しない */
  int cvCnt,    /* X軸の値の数(-x が指定されていないときは1) */
  int rvCnt,    /* Y軸の値の数(-y が指定されていないときは1) */
  int fix,      /* グラフ表示領域をX:Y=3:2に固定するかどうか */
  int exFlg,    /* 凡例を表示させるかどうか */
  struct KeyInfo *keyInfo){

  struct Container *con;
  double x,y,w,h;
  char *s;
  char *id;
  int i,j;
  char buf[20];
  double FontUnit;

  /*定数*/
  double PWD     = OPT_WIDTH;     /*物理的な幅*/
  double PHT     = OPT_HEIGHT;    /*物理的な高さ*/
  double MARGIN  = 20;      /*余白*/

  double GWD     = 0;       /*グラフ描画領域の幅(ham)*/
  double GHT     = 0;       /*グラフ描画領域の高さ(ham)*/

  double TWD;               /*タイトルの幅*/
  double THT;               /*タイトルの高さ*/
  double TFS     = 6;       /*タイトルのフォントサイズ*/

  double CTWD;              /*列タイトルの幅*/
  double CTHT;              /*列タイトルの高さ*/
  double CTFS    = 4;       /*列タイトルのフォントサイズ*/

  double RTWD;              /*行タイトルの幅*/
  double RTHT;              /*行タイトルの高さ*/
  double RTFS    = 4;       /*行タイトルのフォントサイズ*/

  double CVWD;              /*列値の幅*/
  double CVHT;              /*列値の高さ*/
  double CVFS    = 3;       /*列値のフォントサイズ*/

  double RVWD;              /*行値の幅*/
  double RVHT;              /*行値の高さ*/
  double RVFS    = 3;       /*行値のフォントサイズ*/

  double EXWD    = 0;       /*凡例の幅*/
  double EXHT    = 0;       /*凡例の高さ*/

  /*----------------------------*/
  /* 各要素の幅と高さを調整する */
  /*----------------------------*/

  /* フォント単位の決定(FontUnit) */
  /* 固定フォントのw,hを考慮して厳密に決めるべき*/
  if(PWD>PHT) FontUnit=PHT/100;
  else        FontUnit=PWD/100;

  /* 各種フォントサイズから、高さを決定する */
  if(title==NULL) THT = 0;
  else            THT = TFS*FontUnit;

  if(cSet==1) { CTHT=CTFS*FontUnit; CVHT=CVFS*FontUnit; }
  else        { CTHT=0            ; CVHT=0            ; }

  if(rSet==1) { RTHT=RTFS*FontUnit; RVHT=RVFS*FontUnit; }
  else        { RTHT=0            ; RVHT=0            ; }

  /* 凡例 */
  if(exFlg==0) EXWD=0;
  else         EXWD=PWD/5;

  /* その他の幅と高さを計算 */
  if(fix) {
    if( (double)cvCnt > (double)rvCnt*(PWD/PHT) ){
      GHT=PHT-CTHT-CVHT-THT -MARGIN*2;
      GWD=(GHT/(double)rvCnt) * (PWD/PHT) * (double)cvCnt;
    }else{
      GWD=PWD-RTHT-RVHT-EXWD-MARGIN*2;
      GHT=(GWD/(double)cvCnt) * (PHT/PWD) * (double)rvCnt;
    }
  }else{
    GWD=PWD-RTHT-RVHT-EXWD-MARGIN*2;
    GHT=PHT-CTHT-CVHT-THT -MARGIN*2;
  }

  PWD=GWD+RTHT+RVHT+EXWD+MARGIN*2;
  PHT=GHT+CTHT+CVHT+THT +MARGIN*2;

  TWD=RTHT+RVHT+GWD+EXWD;
  CTWD=GWD;
  RTWD=GHT;
  CVWD=GWD/(double)cvCnt;
  RVWD=GHT/(double)rvCnt;
  EXHT=GHT;

  con=mssCalloc(sizeof(struct Container),"setContainer");

  /* id */
  con->id=mssStrdup("root");

  /* 描画領域 */
  x=0; y=0; w=PWD; h=PHT;
  con->co=initSVGcoord(x,y,w,h);

  /*X,Y軸の値の個数セット*/
  con->cvCnt=cvCnt;
  con->rvCnt=rvCnt;

  /*固定比率フラグ*/
  con->fix=fix;

  /*物理的な全体の幅と高さの設定*/
  con->pWidth  = PWD;
  con->pHeight = PHT;

  /* 線の太さの単位*/
  LineUnit=con->co->w/con->pWidth;


  /*----------------------------*/
  /* タイトル                   */
  /*----------------------------*/
  if(title==NULL){
    con->title=NULL;
  }else{
    x=MARGIN; y=MARGIN; w=TWD; h=THT;
    s=title;
    id="title";
    con->title=initSVGstring(x,y,w,h,0,0,"middle",id,title);
  }

  /*----------------------------*/
  /* 列タイトル                 */
  /*----------------------------*/
  if(cSet){
    x=MARGIN+RTHT+RVHT; y=MARGIN+THT; w=CTWD; h=CTHT;
    s=MssFlds2name(keyInfo->cFld,0);
    id="ct";
    con->ct=initSVGstring(x,y,w,h,0,1,"middle",id,s);
  }else{
    con->ct=NULL;
  }

  /*----------------------------*/
  /* 行タイトル                 */
  /*----------------------------*/
  if(rSet){
    x=MARGIN; y=MARGIN+THT+CTHT+CVHT; w=RTWD; h=RTHT;
    s=MssFlds2name(keyInfo->rFld,0);
    id="rt";
    con->rt=initSVGstring(x,y,w,h,1,1,"middle",id,s);
  }else{
    con->rt=NULL;
  }

  /*----------------------------*/
  /* 列値                       */
  /*----------------------------*/
  if(cSet){
    con->cv=mssMalloc(sizeof(struct SVGstring *)*cvCnt,"setCon");
    for(i=0; i<cvCnt; i++){
      x=MARGIN+RTHT+RVHT+(double)i*CVWD; y=MARGIN+THT+CTHT; w=CVWD; h=CVHT;
      s=*(keyInfo->cAdd+i);
      sprintf(buf,"cv%d",i);
      *(con->cv+i)=initSVGstring(x,y,w,h,0,1,"middle",buf,s);
    }
  }else{
    con->cv=NULL;
  }

  /*----------------------------*/
  /* 行値                       */
  /*----------------------------*/
  if(rSet){
    con->rv=mssMalloc(sizeof(struct SVGstring)*rvCnt,"setCon");
    for(i=0; i<rvCnt; i++){
      x=MARGIN+RTHT; y=MARGIN+THT+CTHT+CVHT+(double)i*RVWD; w=RVWD; h=RVHT;
      s=*(keyInfo->rAdd+i);
      sprintf(buf,"rv%d",i);
      *(con->rv+i)=initSVGstring(x,y,w,h,1,1,"middle",buf,s);
    }
  }else{
    con->rv=NULL;
  }

  /*----------------------------*/
  /* 凡例領域                   */
  /*----------------------------*/
  if(exFlg!=0){
    x=MARGIN+RTHT+RVHT+CTWD; y=MARGIN+THT+CTHT+CVHT; w=EXWD; h=EXHT;
    s=MssFlds2name(keyInfo->eFld,0);
    con->exNote=setExNote(x,y,w,h,s,keyInfo);
  }else{
    con->exNote=NULL;
  }

  /*----------------------------*/
  /* グラフ領域                 */
  /*----------------------------*/
  for(i=0; i<cvCnt; i++){
    for(j=0; j<rvCnt; j++){
      x=MARGIN    +RTHT+RVHT+(double)i*CVWD;
      y=MARGIN+THT+CTHT+CVHT+(double)j*RVWD;
      w=CVWD; h=RVWD;
      con->line[i][j]=setLineChart(i,j,x,y,w,h,keyInfo);
    }
  }

  return(con);
}

void freeContainer(struct Container *con){
  int i,j;

  if(con==NULL) return;

  mssFree(con->id);
  freeSVGcoord(con->co);
  freeSVGstring(con->title);
  freeSVGstring(con->ct);
  freeSVGstring(con->rt);
  if(con->cv!=NULL){
    for(i=0; i<con->cvCnt ; i++) freeSVGstring(*(con->cv+i));
    mssFree(con->cv);
  }
  if(con->rv!=NULL){
    for(i=0; i<con->rvCnt ; i++) freeSVGstring(*(con->rv+i));
    mssFree(con->rv);
  }
  freeExNote(con->exNote);
  for(i=0; i<con->cvCnt ; i++){
    for(j=0; j<con->rvCnt ; j++){
      freeLineChart(con->line[i][j]);
    }
  }
  mssFree(con);
}



void showContainer( struct Container *con ){
  int i;

  printf("===============Container Data\n");
  showSVGstring(con->title);
  showSVGstring(con->ct   );
  showSVGstring(con->rt   );

  for(i=0; i<con->cvCnt-1; i++){
    showSVGstring(*(con->cv+i) );
  }
  for(i=0; i<con->rvCnt-1; i++){
    showSVGstring(*(con->rv+i) );
  }

  printf("fix   : %d\n",con->fix);
  printf("pWidth : %g\n",con->pWidth);
  printf("pHeight: %g\n",con->pHeight);
}


void writeLineChartContainer( struct LineChart *line, struct mssFPW *fpw ){

  struct mssXmlTag *gTag; /* gタグ*/
  int i;

  gTag=mssInitXmlTag("g",NULL);
  mssAddXmlTagAttributeStr(gTag,"id", line->id, NULL);

  /* <g> */
  mssWriteXmlStartTag(gTag,NULL,fpw);
  mssWriteRet(fpw);

  /* バックグラウンド */
  writeSVGrect(line->bg,fpw);

  /* X軸&タイトル */
  writeSVGstring(line->xt,fpw);
  writeSVGline(line->xa,fpw);

  /* Y軸&タイトル */
  writeSVGstring(line->yt,fpw);
  writeSVGline(line->ya,fpw);

  /* X軸値&マーク */
  for(i=0; i<line->xvCnt; i++){
    if(line->xv!=NULL) writeSVGstring(*(line->xv+i),fpw);
    if(line->xm!=NULL) writeSVGline(*(line->xm+i),fpw);
  }

  /* Y軸値&マーク&スケール*/
  for(i=0; i<line->yvCnt; i++){
    if(line->yv!=NULL) writeSVGstring(*(line->yv+i),fpw);
    if(line->ym!=NULL) writeSVGline(*(line->ym+i),fpw);
    if(line->ys!=NULL) writeSVGline(*(line->ys+i),fpw);
  }



  /* </g> */
  mssWriteXmlEndTag(gTag,NULL,fpw);

  mssFreeXmlTag(gTag);
}

void writeContainer( struct Container *con, struct mssFPW *fpw ){
  int i,j;

  /* タイトル */
  writeSVGstring(con->title,fpw);

  if(cSet){
    /* 列タイトル */
    writeSVGstring(con->ct,fpw);

    /* 列値 */
    for(i=0; i<con->cvCnt; i++){
      writeSVGstring(*(con->cv+i),fpw);
    }
  }

  if(rSet){
    /* 行タイトル */
    writeSVGstring(con->rt,fpw);

    /* 行値 */
    for(i=0; i<con->rvCnt; i++){
      writeSVGstring(*(con->rv+i),fpw);
    }
  }

  /* グラフ描画 */
  for(i=0; i<con->cvCnt; i++){
    for(j=0; j<con->rvCnt; j++){
      writeLineChartContainer( con->line[i][j], fpw );
    }
  }

  /* 凡例描画 */
  if(eSet){
    writeExNote(con->exNote, fpw);
  }
}


/* 棒グラフの構造体
 *                              +-----------------------+
 *                              |                       |
 *                              |   +----+              |
 *                              +---|    |--------------|
 *                 CurYplus     |   +----+              |
 *                              |   |    |              |
 *                              +---|    |--------------|
 *                              |   |    |              |
 *                              |   |    |     +----+   |
 *                              +---|    |-----|    |---|
 *                              |   |    |     |    |   |
 *                              |   |    |     |    |   |
 *                 line->origin +-+-+----+--+--+----+--+|
 *                              |              |    |   |
 *                 CurYminus    |              +----+   |
 *                              +-----------------------|
 *                                  |    |
 *                                  |   *(line->w+kNum)
 *                                 *(line->x+kNum)
 *
 *  -f の単位あたりピクセル数
 *    line->unit = line->g->h / (scale->max - scale->min)
 */
void writeChart(struct Container *con,
  int cNum, int rNum, int eNum,
  int kNumO, int kNumN, double fValO, double fValN,
  struct KeyInfo *keyInfo, struct mssFPW *fpw){
  struct LineChart *line;
  struct SVGline *drawline;

  double x,y,w,h,x1,y1,x2,y2;
  char buf[30];
  int swidth = 0; /* border width */

  line=con->line[cNum][rNum];


  //w = con->pWidth*0.04;
  w = line->dotSize; //(ham)
  x = (*(line->xm+kNumO))->c2.x2 - (line->g->w/(double)line->xvCnt)*0.5 -w*0.5;

  if(fValO>=0) {
    h = fValO*line->unit;
    y = line->origin-h-w*0.5;
  }else{
    h = (-1)*(fValO*line->unit);
    y = line->origin+h-w*0.5;
  }

drawShapes(x, y, w, eNum, swidth, fpw);

//fprintf(stderr,"knum=%d,%d, %d\n",kNumN,kNumO,keyInfo->kHash->cnt-1 );
//fprintf(stderr,"comp=%d,%d\n", 1==(kNumN-kNumO), kNumN!=(keyInfo->kHash->cnt-1));
  if( 1==(kNumN-kNumO) && kNumO!=(keyInfo->kHash->cnt-1) ){
    x1=x;           y1=y;
    x2 =   (*(line->xm+kNumN))->c2.x2 - (line->g->w / line->xvCnt)*0.5 -w*0.5;
    if(fValN>=0) {
      h = fValN*line->unit;
      y2 = line->origin-h-w*0.5;
    }else{
      h = (-1)*(fValO*line->unit);
      y2= line->origin+h-w*0.5;
    }


    sprintf(buf,"line%d-%d-%d",cNum,rNum,kNumO);
    drawline=initSVGline(x1+w*0.5,y1+w*0.5,x2+w*0.5,y2+w*0.5,1,Color[eNum],NULL,buf);
    writeSVGline(drawline,fpw);
    freeSVGline(drawline);
  }
}

int main(int argc, char *argv[]){
/*----------------------------------------------------------------------------*/
/* キー項目                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLD optKEY={
    OFLD,   /* オプションタイプ                                             */
    "k",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    1,      /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    0,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    0,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    "n,r",  /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    KEYT,   /* このオプションのタイトル(Helpで表示)                         */
    KEYC,   /* このオプションのコメント(Helpで表示)                         */
    KEYF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* 凡例項目(積み上げ分類項目)                                                 */
/*----------------------------------------------------------------------------*/
  MssOptFLD optEXP={
    OFLD,   /* オプションタイプ                                             */
    "e",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    1,      /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    0,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    0,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    "n,r",  /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    EXPT,   /* このオプションのタイトル(Helpで表示)                         */
    EXPC,   /* このオプションのコメント(Helpで表示)                         */
    EXPF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* 列分類項目                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptFLD optCOL={
    OFLD,   /* オプションタイプ                                             */
    "c",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    1,      /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    0,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    0,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    NULL,   /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    COLT,   /* このオプションのタイトル(Helpで表示)                         */
    COLC,   /* このオプションのコメント(Helpで表示)                         */
    COLF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* 行分類項目                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptFLD optROW={
    OFLD,   /* オプションタイプ                                             */
    "r",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    1,      /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    0,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    0,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    "n,r",  /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    ROWT,   /* このオプションのタイトル(Helpで表示)                         */
    ROWC,   /* このオプションのコメント(Helpで表示)                         */
    ROWF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* 数値項目                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLD optFLD={
    OFLD,   /* オプションタイプ                                             */
    "f",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    1,      /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    1,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    0,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    NULL,   /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    FLDT,   /* このオプションのタイトル(Helpで表示)                         */
    FLDC,   /* このオプションのコメント(Helpで表示)                         */
    FLDF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* 入力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptINF optINF={
    OINF,   /* オプションタイプ                                             */
    "i",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    1,      /* 指定可能の最大ファイル数                                     */
    0,      /*1:file not foundのエラーで終了しない 0:する                   */
    INFT,   /* このオプションのタイトル(Helpで表示)                         */
    INFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 出力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptOTF optOTF={
    OOTF,   /* オプションタイプ                                             */
    "o",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    OTFT,   /* このオプションのタイトル(Helpで表示)                         */
    OTFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 圧縮出力                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLG optZIP={
    OFLG,   /* オプションタイプ                                             */
    "z",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    ZIPT,   /* このオプションのタイトル(Helpで表示)                         */
    ZIPC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* plain text                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptFLG optTXT={
    OFLG,   /* オプションタイプ                                             */
    "t",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    TXTT,   /* このオプションのタイトル(Helpで表示)                         */
    TXTC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* ワークファイル用ディレクトリ名                                             */
/*----------------------------------------------------------------------------*/
  MssOptSTR optTMP={
    OSTR,   /* オプションタイプ                                             */
    "T",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    MssTempDir, /* デフォルト                                               */
    1,      /* 文字列の最小長                                               */
    MssFileNameMaxLen,  /* 文字列の最大長                                   */
    TMPT,   /* このオプションのタイトル(Helpで表示)                         */
    TMPC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 表示幅                                                                     */
/*----------------------------------------------------------------------------*/
  MssOptINT optWID={
    OINT,   /* オプションタイプ                                             */
    "W",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    640,    /* デフォルト(数値として指定)                                   */
    320,    /* 最小値                                                       */
    1600,   /* 最大値                                                       */
    WIDT,   /* このオプションのタイトル(Helpで表示)                         */
    WIDC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 表示高                                                                     */
/*----------------------------------------------------------------------------*/
  MssOptINT optHIT={
    OINT,   /* オプションタイプ                                             */
    "H",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    480,    /* デフォルト(数値として指定)                                   */
    240,    /* 最小値                                                       */
    1200,   /* 最大値                                                       */
    HITT,   /* このオプションのタイトル(Helpで表示)                         */
    HITC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*//* -r,-cを指定するとき、全グラフのスケールを固定するフラグ                    *//*----------------------------------------------------------------------------*/  MssOptFLG optFIX={
    OFLG,   /* オプションタイプ                                             */
    "fix",  /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    FIXT,   /* このオプションのタイトル(Helpで表示)                         */
    FIXC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*//* -r,-cを指定するとき、個々のグラフの大きさをWID:HITに固定するフラグ         *//*----------------------------------------------------------------------------*/  MssOptFLG optADJ={
    OFLG,   /* オプションタイプ                                             */
    "adjust",  /* キーワード(複数文字は不可)                                */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    ADJT,   /* このオプションのタイトル(Helpで表示)                         */
    ADJC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* タイトル                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptSTR optTIT={
    OSTR,   /* オプションタイプ                                             */
    "title",/* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    "",     /* デフォルト                                               */
    1,      /* 文字列の最小長                                               */
    256,    /* 文字列の最大長                                   */
    TITT,   /* このオプションのタイトル(Helpで表示)                         */
    TITC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* オプションをまとめる                                                       */
/*----------------------------------------------------------------------------*/
  void *opt[]={&optKEY,&optEXP,&optCOL,&optROW,&optFLD,
               &optINF,&optOTF,&optZIP,&optTXT,&optTMP,
               &optWID,&optHIT,&optFIX,&optADJ,&optTIT,NULL};

/*============================================================================*/
/* 変数宣言＆定義                                                             */
/*============================================================================*/
  struct mssHeader    *hdi; /*入力ファイル用<head>タグ格納構造体*/
  struct mssFPR       *fpr; /*入力ファイル構造体                */
  struct mssFPW       *fpw; /*出力ファイル構造体                */
  struct mssFldRec    *fr;  /*項目-行バッファ構造体             */
  struct mssFldRecDbl *frd; /*項目-行バッファ構造体             */
  struct mssFields    *sf;  /*ソート項目構造体     */

  struct KeyInfo *keyInfo;
  double maxAL;
  double minAL;
  double accValplus;
  double accValminus;
  int i,j;
  struct Container *con;

  int cvCnt;
  int rvCnt;
  int exFlg;
  char *cStr, *rStr, *kStr,*kStrN,*kStrO,*eStr,*fStr,*fStrN,*fStrO;
  int   cNum,  rNum,  kNumO, kNumN, eNum;
  double fVal;
  double fValN,fValO;
  double maxCR[COLMAX][ROWMAX];
  double minCR[COLMAX][ROWMAX];

/*----------------------------------------------------------------------------*/
/* 前処理                                                                     */
/*----------------------------------------------------------------------------*/
  mssInit(argc,argv,&comHelp);       /* シグナル処理などの初期化              */
  mssHelpDoc(opt,&comHelp,argc,argv);/* ヘルプ                                */
  mssSetOption(opt,argc,argv);       /* コマンドオプションの設定              */
  fpr=mssOpenFPR(optINF.str,4);      /* 入力ファイルオープン                  */
  hdi=mssReadHeader(fpr);            /* ヘッダの読み込み                      */

  if(optCOL.set) cSet=1;
  if(optROW.set) rSet=1;
  if(optEXP.set) eSet=1;

           mssSetOptFld(&optKEY, hdi); /* -k 項目 */
           mssSetOptFld(&optFLD, hdi); /* -f 項目 */
  if(eSet) mssSetOptFld(&optEXP, hdi); /* -c 項目 */
  if(cSet) mssSetOptFld(&optCOL, hdi); /* -x 項目 */
  if(rSet) mssSetOptFld(&optROW, hdi); /* -r 項目 */

  /*ソート項目の作成*/
  sf=mssInitFields();
  if(rSet) mssAddFieldsByFields(sf,optROW.flds); /* -r 項目 */
  if(cSet) mssAddFieldsByFields(sf,optCOL.flds); /* -c 項目 */
  if(eSet) mssAddFieldsByFields(sf,optEXP.flds); /* -e 項目 */
           mssAddFieldsByFields(sf,optKEY.flds); /* -k 項目 */

  mssSetFieldsSortPriority(sf);         /* ソート優先順位番号を登録順にふる   */

  keyInfo=mssCalloc(sizeof(struct KeyInfo),"keyInfo");
           keyInfo->kFld=optKEY.flds;
           keyInfo->fFld=optFLD.flds;
  if(rSet) keyInfo->rFld=optROW.flds;
  if(cSet) keyInfo->cFld=optCOL.flds;
  if(eSet) keyInfo->eFld=optEXP.flds;

  keyInfo->kHash=mssInitHash(101);
  keyInfo->rHash=mssInitHash(101);
  keyInfo->cHash=mssInitHash(101);
  keyInfo->eHash=mssInitHash(101);

  if(!eSet) exFlg=0;
  else      exFlg=1;
 
/*mssShowOption(opt);*/
/*mssShowHeader(hdi);*/

  if(hdi->xmlenc!=NULL){
    icid=iconv_open("UTF-8",hdi->xmlenc);
  }else{
    icid=NULL;
  }

  OPT_WIDTH  = optWID.val;
  OPT_HEIGHT = optHIT.val;
  if(optTIT.set) OPT_TITLE    = optTIT.str;
  else           OPT_TITLE    = hdi->title;

/*----------------------------------------------------------------------------*/
/*出力ヘッダーの作成と出力                                                    */
/*----------------------------------------------------------------------------*/
  /*標準出力オープン+ヘッダーの出力*/
  fpw=mssOpenFPW(optOTF.str,optZIP.set,0);

/*----------------------------------------------------------------------------*/
/*メインルーチン                                                              */
/*----------------------------------------------------------------------------*/
  /*ソート済みファイルとしてオープン*/
  fpr=mssReopenFPRsort(fpr,4,sf,hdi->flds->cnt,optTMP.str);

  /*-------------------------------------*/
  /* 一回目の走査                        */
  /* 各次元の値の種類数,名称をhashに登録 */
  /*-------------------------------------*/
  fr=mssInitFldRec(hdi->flds->cnt);
  while( EOF != mssReadFldRec(fpr,fr) ){

    /* -kの値をhashに登録 */
    kStr=*(fr->pnt+MssFlds2num(keyInfo->kFld,0));
    if(!MssIsNull(kStr))
      keyInfo->kAdd=setKeyVal(keyInfo->kAdd,keyInfo->kHash,kStr);

    /* -rの値をhashに登録 */
    if(rSet){
      rStr=*(fr->pnt+MssFlds2num(keyInfo->rFld,0));
      if(!MssIsNull(rStr))
        keyInfo->rAdd=setKeyVal(keyInfo->rAdd,keyInfo->rHash,rStr);
    }

    /* -cの値をhashに登録 */
    if(cSet){
      cStr=*(fr->pnt+MssFlds2num(keyInfo->cFld,0));
      if(!MssIsNull(cStr))
        keyInfo->cAdd=setKeyVal(keyInfo->cAdd,keyInfo->cHash,cStr);
    }

    /* -eの値をhashに登録 */
    if(eSet){
      eStr=*(fr->pnt+MssFlds2num(keyInfo->eFld,0));
      if(!MssIsNull(eStr))
        keyInfo->eAdd=setKeyVal(keyInfo->eAdd,keyInfo->eHash,eStr);
    }

    mssGV.inCnt++;
  }
  mssFreeFldRec(fr);

  /* r,c,eが設定されていない時はダミー値(*)を登録しておく*/
  if(!rSet) keyInfo->rAdd=setKeyVal(keyInfo->rAdd,keyInfo->rHash,"DUMMY *");
  if(!cSet) keyInfo->cAdd=setKeyVal(keyInfo->cAdd,keyInfo->cHash,"DUMMY *");
  if(!eSet) keyInfo->eAdd=setKeyVal(keyInfo->eAdd,keyInfo->eHash,"DUMMY *");

/*
printf("-----------------------------------\n");
for(i=0; i<keyInfo->rHash->cnt; i++){
printf("r[%d]=%s\n",i,*(keyInfo->rAdd+i));
}
for(i=0; i<keyInfo->cHash->cnt; i++){
printf("c[%d]=%s\n",i,*(keyInfo->cAdd+i));
}
for(i=0; i<keyInfo->kHash->cnt; i++){
printf("k[%d]=%s\n",i,*(keyInfo->kAdd+i));
}
for(i=0; i<keyInfo->eHash->cnt; i++){
printf("e[%d]=%s\n",i,*(keyInfo->eAdd+i));
}
*/
  /*--------------------------------------------------*/
  /* hashに登録された値を並べ替え、順序番号を登録する */
  /*--------------------------------------------------*/
  sortHashAdd(keyInfo->kHash, keyInfo->kAdd, keyInfo->kHash->cnt, &optKEY);
  sortHashAdd(keyInfo->rHash, keyInfo->rAdd, keyInfo->rHash->cnt, &optROW);
  sortHashAdd(keyInfo->cHash, keyInfo->cAdd, keyInfo->cHash->cnt, &optCOL);
  sortHashAdd(keyInfo->eHash, keyInfo->eAdd, keyInfo->eHash->cnt, &optEXP);

  /*-------------------------*/
  /* 2回目の走査             */
  /* -f の最小値最大値       */
  /* -c -r 別に求める        */
  /*-------------------------*/
  mssSeekTopFPR(fpr);
  frd=mssInitFRD(hdi->flds->cnt);
  maxAL=-DBL_MAX;
  minAL= DBL_MAX;
  for(i=0; i<COLMAX; i++){
    for(j=0; j<ROWMAX; j++){
      maxCR[i][j]=-DBL_MAX;
      minCR[i][j]= DBL_MAX;
    }
  }

  accValplus = 0;
  accValminus= 0;
  while( EOF != mssReadFRD(fpr,frd) ){

    setKeyBreakInfo(keyInfo, frd);

    if(keyInfo->kKB){
      if(cSet) cStr=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->cFld,0));
      else     cStr="DUMMY *";
      if(rSet) rStr=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->rFld,0));
      else     rStr="DUMMY *";

      if(MssIsNull(cStr)) continue;
      if(MssIsNull(rStr)) continue;
 
      cNum=mssHashMember(keyInfo->cHash,cStr)->val.v.i;
      rNum=mssHashMember(keyInfo->rHash,rStr)->val.v.i;

      if(cNum<COLMAX && rNum<ROWMAX){
        if(maxCR[cNum][rNum] < accValplus)  maxCR[cNum][rNum] = accValplus;
        if(minCR[cNum][rNum] > accValminus) minCR[cNum][rNum] = accValminus;

        if(maxAL<accValplus)  maxAL=accValplus;
        if(minAL>accValminus) minAL=accValminus;
      }
      accValplus =0;
      accValminus=0;
    }

    /* -f の値の累積 */
    fStr = *(frd->pnt[frd->new]+MssFlds2num(optFLD.flds,0));
    if(MssIsNull(fStr)) continue;

    fVal = atof(fStr);
    if(fVal>=0) accValplus += fVal;
    else        accValminus+= fVal;

  }
  mssFreeFRD(frd);

  for(i=0; i<keyInfo->cHash->cnt; i++){
    for(j=0; j<keyInfo->rHash->cnt; j++){
      if(maxCR[i][j]==-DBL_MAX){
        maxCR[i][j]=0;
        minCR[i][j]=0;
      }
    }
  }
  for(i=0; i<keyInfo->cHash->cnt; i++){
    for(j=0; j<keyInfo->rHash->cnt; j++){
      /* -c,-r別に異なるスケールを設定する場合 */
      if(!optFIX.set){
        if(maxCR[i][j]!=-DBL_MAX){
          if(maxCR[i][j]>0 && minCR[i][j]>0){
            keyInfo->scale[i][j]=setScale(0, maxCR[i][j], 5);
          }else if(maxCR[i][j]<0 && minCR[i][j]<0){
            keyInfo->scale[i][j]=setScale(minCR[i][j], 0, 5);
          }else{
            keyInfo->scale[i][j]=setScale(minCR[i][j], maxCR[i][j], 5);
          }
        }else{
          keyInfo->scale[i][j]=NULL;
        }
      /* -c,-rの値に関係なく全て同じスケールを設定する場合 */
      }else{
        keyInfo->scale[i][j]=setScale(minAL, maxAL, 5);
      }
    }
  }


/*
printf("-----------------------------------\n");
for(i=0; i<keyInfo->rHash->cnt; i++){
printf("r[%d]=%s\n",i,*(keyInfo->rAdd+i));
}
for(i=0; i<keyInfo->cHash->cnt; i++){
printf("c[%d]=%s\n",i,*(keyInfo->cAdd+i));
}
for(i=0; i<keyInfo->kHash->cnt; i++){
printf("k[%d]=%s\n",i,*(keyInfo->kAdd+i));
}
for(i=0; i<keyInfo->eHash->cnt; i++){
printf("e[%d]=%s\n",i,*(keyInfo->eAdd+i));
}
printf("-----------------------------------\n");
*/

  /*-------------------------------------*/
  /* コンテナの設定                      */
  /*-------------------------------------*/
  cvCnt=keyInfo->cHash->cnt;
  rvCnt=keyInfo->rHash->cnt;
  con=setContainer( OPT_TITLE, cvCnt, rvCnt, optADJ.set, exFlg, keyInfo);

  //showContainer( con );

  /*-------------------------------------*/
  /* 凡例の設定                          */
  /*-------------------------------------*/
  //con->exNote=setExNote("abc",keyInfo);

  /*-------------------------------------*/
  /* グラフコンテナの設定                */
  /*-------------------------------------*/

  /*-------------------------------------*/
  /* Output XML data                     */
  /*-------------------------------------*/

  /* XML declareration : <?xml version="1.0" encoding="UTF-8"?> */
  mssWriteXmlDeclaration(MssXmlDefVer,"UTF-8",fpw);

  /* output svg start tag :
               <svg width="xxx" height="xxx" viewBox="x x x x" .... > */
  writeSVGstartTag(con->pWidth,con->pHeight,con->co->w,con->co->h,fpw);

  /*-------------------------------------*/
  /* Output line chart                    */
  /*-------------------------------------*/
  mssSeekTopFPR(fpr);
  frd=mssInitFRD(hdi->flds->cnt);
//fprintf(stderr,"xxxxxxxxxxxxxxxx\n");

  CurYplus =con->line[0][0]->origin;
  CurYminus=con->line[0][0]->origin;
  while( EOF != mssReadFRD(fpr,frd) ){

    if(frd->eof) break;

    if(frd->firstRead){
      frd->firstRead=0;
      continue;
    }
    if(cSet) cStr=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->cFld,0));
    else     cStr="DUMMY *";
    if(rSet) rStr=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->rFld,0));
    else     rStr="DUMMY *";
    if(eSet) eStr=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->eFld,0));
    else     eStr="DUMMY *";

             kStrO=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->kFld,0));
             kStrN=*(frd->pnt[frd->new]+MssFlds2num(keyInfo->kFld,0));

             fStrO=*(frd->pnt[frd->old]+MssFlds2num(keyInfo->fFld,0));
             fStrN=*(frd->pnt[frd->new]+MssFlds2num(keyInfo->fFld,0));

    if(MssIsNull(fStrO)) continue;
    if(MssIsNull(kStrO)) continue;
    if(MssIsNull(cStr)) continue;
    if(MssIsNull(rStr)) continue;
    if(MssIsNull(eStr)) continue;

    cNum=mssHashMember(keyInfo->cHash,cStr)->val.v.i;
    rNum=mssHashMember(keyInfo->rHash,rStr)->val.v.i;
    eNum=mssHashMember(keyInfo->eHash,eStr)->val.v.i;
    kNumO=mssHashMember(keyInfo->kHash,kStrO)->val.v.i;
    kNumN=mssHashMember(keyInfo->kHash,kStrN)->val.v.i;
    fValO=atof(fStrO);
    fValN=atof(fStrN);

    setKeyBreakInfo(keyInfo, frd);
    if(keyInfo->kKB){
      CurYplus =con->line[cNum][rNum]->origin;
      CurYminus=con->line[cNum][rNum]->origin;
//fprintf(stderr,"keyBreak\n");
    }

    /*通常行の処理*/
//fprintf(stderr,"xxx %s,%s : %g,%g\n",kStrO,kStrN,fValO,fValN);
    writeChart(con, cNum, rNum, eNum, kNumO, kNumN, fValO, fValN,keyInfo,fpw);

  }
  mssFreeFRD(frd);

  /*TOPコンテナ出力*/
  writeContainer(con, fpw);

  /*output SVG end tag : </svg> */
  writeSVGendTag(fpw);

  if(icid!=NULL) iconv_close(icid);

  /* 追加 2004/10/31 */
  freeContainer(con);

  for(i=0; i<keyInfo->cHash->cnt; i++){
    for(j=0; j<keyInfo->rHash->cnt; j++){
      mssFree(keyInfo->scale[i][j]);
    }
  }
  mssFreeHash(keyInfo->rHash);
  mssFreeHash(keyInfo->cHash);
  mssFreeHash(keyInfo->kHash);
  mssFreeHash(keyInfo->eHash);
  mssFree(keyInfo->rAdd);
  mssFree(keyInfo->cAdd);
  mssFree(keyInfo->kAdd);
  mssFree(keyInfo->eAdd);
  mssFree(keyInfo);

/*----------------------------------------------------------------------------*/
/*フッター出力&終了処理                                                       */
/*----------------------------------------------------------------------------*/
/*printf("allocCnt=%d\n",getAllocCnt());                */
//  mssWriteFooter(fpw);    /* フッターの出力             */
  mssCloseFPR(fpr);       /* 入力ファイルのクローズ     */
  mssCloseFPW(fpw);       /* 出力ファイルのクローズ     */
  mssFreeFields(sf);      /* ソート項目構造体の領域開放 */
  mssFreeHeader(hdi);     /* 入力ヘッダ領域開放         */
  mssFreeOption(opt);     /* オプション領域開放         */
  mssShowEndMsg();        /* 完了メッセージ             */
  mssEnd(mssExitSuccess); /* 終了                       */
  return(0);              /* to avoid warning message   */
}
