/*============================================================================*/
/* 変更履歴                                                                   */
/*----------------------------------------------------------------------------*/
/* 1.0 : 新規作成 2003/11/01 A.Fujimoto                                       */
/* 1.1 : autoconf,automake対応 2003/11/03 Y.Hamuro                            */
/* 1.2 : encoding対応 2003/11/15 Y.Hamuro                                     */
/*============================================================================*/

#include <stdio.h>
#include <signal.h>
#include <musashi.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <iconv.h>
#include <pmmldtcls.h>

#define U 0	/* 未定義 */
#define N 1	/* 数値 */
#define S 2	/* 文字列 */

struct mssComHelp comHelp={
  "pmmldtcls",    /* コマンド名       */
  "1.2",          /* バージョン       */
  HELPT,          /* コマンドタイトル */
  HELPS,          /* 要約             */
  HELPE,          /* 利用例           */
  HELPR,          /* 参照コマンド     */
  HELPA,          /* 作者情報         */
  HELPB,          /* バグレポート情報 */
  HELPH           /* ホームページ     */
};

extern struct mssGlobalVariables mssGV;

static iconv_t *icidUX; /* iconv用 UTF-8 -> XMLtable Encoding 変換ハンドラ*/
static iconv_t *icidXU; /* iconv用 XMLtable Encoding -> UTF-8 変換ハンドラ*/

/* PMML Treeのノードの構造体 */
struct PmNode{
  char *score;			/* scoreを格納 */
  char *calFormula;		/* 式を文字列で格納 */
  struct mssCal **cal;          /* calFormulaを評価しmssCal構造体にsetした式*/
                                /* surrogateの場合は複数の式になる */
                                /* ターミネータとしてNULLがセットされている */
  int depth;			/* そのノードの深さ */
  struct PmNode *parent;	/* 親ノードへのポインタ */
  struct PmNode *children;	/* 最初の子どもへのポインタ */
  struct PmNode *next;		/* 次(兄弟)のノードへのポインタ */
};

/* default値を決定するための構造体(項目の数だけ、配列として設定される) */
struct DefaultValue{
  char *fldName;		/* 項目名 */
  int   fldType;		/* 項目タイプ(N or S) */
  char *value;			/* default値を格納 */
  int  null;                    /* デフォルト値を決めれない時は１にセット*/
};

/**
 * XPath のコンテキスト(グローバル変数)
 */
xmlXPathContextPtr XpathContext;

/**
 * グローバル変数 XpathContext を登録する
 */
static void setXpathContext(xmlDocPtr doc){
  XpathContext=xmlXPathNewContext(doc);
}

/**
 * グローバル変数 XpathContext の領域開放
 */
static void freeXpathContext(void){
  xmlXPathFreeContext(XpathContext);
}

/**
 * カレントノード(node)から、XPATH(path)にマッチ
 * するノードを検索し、そのノードポインタを返す。
 * マッチするノードがなければNULLを返す。
 */
static xmlNodePtr nextNodeByXPath(
  xmlNodePtr node,  /* カレントノード */
  xmlChar *path){   /* XPath */

  xmlNodePtr next;

  xmlXPathObjectPtr result;

  /* XPathコンテキストにカレントノードを設定 */
  XpathContext->node=node;

  /* pathにマッチするノードの検索 */
  result  = xmlXPathEvalExpression(path, XpathContext);

  /* マッチするノードがなかった場合はnext=NULLにセット */
  if(xmlXPathNodeSetIsEmpty(result->nodesetval)){
    next = NULL;

  /* マッチするノードがあれば、ノードリストの0番目要素を取り出す*/
  }else{
    next = xmlXPathNodeSetItem(result->nodesetval, 0);
  }
  return(next);
}

/**
 * クラス名(結果属性名)をPMMLのMiningField要素から得る(XMLtableのencodigで)。
 * なければNULLを返す。
 *   ex) <MiningField name="whatIdo" usageType="predicted"/>
 */
static char *getClassName(xmlDocPtr doc){
  char *className=NULL;
  xmlNodePtr node;

  node = nextNodeByXPath(xmlDocGetRootElement(doc),
                         "/PMML/TreeModel/MiningSchema/MiningField");

  while(node){
    if(xmlStrcmp(node->name,"MiningField")==0){
      if(xmlStrcmp("predicted",xmlGetProp(node,"usageType"))==0){
        /* クラス名をコピー */
        className=mssEncoding(xmlGetProp(node,"name"),icidUX);
        break;
      }
    }
    node=node->next;
  }
  return(className);
}

/**
 * PMMLのDataDictionaryを調べ、項目名(fldName)の型を返す。
 * 【返値】数値:N,文字:S, 項目名がないor未定の場合:U
 * fieldはUTF-8で指定する。
 */
static int getFieldType(xmlDocPtr doc, char *field){
  int type=U;
  xmlNodePtr node;

  node = nextNodeByXPath(xmlDocGetRootElement(doc),
                         "/PMML/DataDictionary/DataField");

  while(node){
    if(xmlStrcmp(node->name,"DataField")==0){
      if(xmlStrcmp(field,xmlGetProp(node,"name"))==0){
        if(xmlStrcmp("continuous",xmlGetProp(node,"optype"))==0){
          type=N;
          break;
        }else if(xmlStrcmp("categorical",xmlGetProp(node,"optype"))==0){
          type=S;
          break;
        }
      }
    }
    node=node->next;
  }
  return(type);
}

/**
 * XMLtableを読み込み、全属性のデフォルト値を決定し、その配列を返す
 *   数値属性の場合：全行の平均値
 *   文字列属性の場合：最も出現回数の多い文字列
 *   全行がNULLの場合はNULL値をセットする
 */
static struct DefaultValue *getDefaultValue(
  xmlDocPtr doc,         /* PMML document */	
  struct mssHeader *hdi, /* XMLtable のヘッダー構造体 */
  struct mssFPR *fpr){   /* XMLtable のデータバッファ */

  struct DefaultValue *defaultValue; /* 各項目のデフォルト値を格納する構造体 */
  struct mssFldRec *fr;              /* 項目-行バッファ構造体                */
  double *sum;                       /* 数値属性の値を足し込む為の一時変数   */
  int    *cnt;                       /* 数値属性の値(nullでない)の行数       */
  struct mssHash **hash;             /* 文字列属性の値を格納する一時hash     */
  struct mssHashNode *hn;            /* 上記hashのノードを示す一時変数       */
  MssValue v;                        /* 文字列属性の値(nullでない)の行数     */
  struct mssHashNode *maxHn;         /* 最も件数の多いhashNode               */
  int maxCnt;                        /* 最多件数                             */
  int i,j;
  char *encStr;

  /* 項目数のデフォルト値領域を確保(初期化) */
  defaultValue=mssCalloc(sizeof(struct DefaultValue)*
                         (hdi->flds->cnt+1),"getDefaultValue");

  /* defaultValue配列のターミネータ(fldName==NULL) */
  (defaultValue+hdi->flds->cnt)->fldName=NULL;

  /* 項目名とタイプをセット */
  for(i=0;i< hdi->flds->cnt;i++){

    /* 項目名のセット */
    (defaultValue+i)->fldName=mssStrdup(MssFlds2name(hdi->flds,i));

    /* PMMLより、その項目のタイプをセット */
    encStr=mssEncoding((defaultValue+i)->fldName,icidXU);
    (defaultValue+i)->fldType
       =getFieldType(doc,encStr);
    mssFree(encStr);
  }

  /*FldRec構造体の初期化*/ 
  fr=mssInitFldRec(hdi->flds->cnt);

  /* データ先頭行に移動 */
  mssSeekTopFPR(fpr);

  /* 数値型用一時変数の領域確保(項目数分) */
  sum=mssCalloc(sizeof(double)*hdi->flds->cnt,"defaultValue");
  cnt=mssCalloc(sizeof(int   )*hdi->flds->cnt,"defaultValue");

  hash=mssCalloc(sizeof(struct mssHash *)*hdi->flds->cnt,"defaultValue");
  for(i=0;i< hdi->flds->cnt;i++){
    *(hash+i)=mssInitHash(101); /* hash構造体の初期化 */
  }
  mssVinit(&v,INT);      /* 件数初期値(=0) */

  /* 全項目を一度スキャン、default値決定の準備 */
  while(EOF!=mssReadFldRec(fpr,fr)){

    /* 項目数でLoop */
    for(i=0;i< hdi->flds->cnt;i++){

      /* 項目タイプ検査 */
      switch((defaultValue+i)->fldType){
      case N:                             /* 数値型の場合 */
        if(!MssIsNull(*(fr->pnt+i))){
          (*(sum+i))+=atof(*(fr->pnt+i));
          (*(cnt+i))++;
        }
        break;

      case S:                             /* 文字列型の場合 */
        if(!MssIsNull(*(fr->pnt+i))){
          /* 項目の値(文字列)をハッシュ表に登録する                      */
          /* ハッシュ表になければ、新しいノードを追加し、そのノードを返す*/
          /* ハッシュ表にあれば、そのノードを返す                        */
          hn=mssHashInsertAdd(*(hash+i),*(fr->pnt+i),v);

          /* 件数をインクリメントする */
          hn->val.v.i++;
        }
        break;
      }
    }
  }

  /* デフォルト値を決定する */
  for(i=0;i< hdi->flds->cnt;i++){

    switch((defaultValue+i)->fldType){
    /* 数値型の場合 */
    case N:                                  /* 数値型の場合 */
      /* 平均値をデフォルト値として登録する(文字列として) */
      /* 全ての行がNULLの場合はnullフラグをたてる         */
      if(*(cnt+i)!=0){
        (defaultValue+i)->value=mssFtoA((*(sum+i))/(double)(*(cnt+i)));
      }else{
        (defaultValue+i)->value=mssStrdup(MssNullStr);
      }
      break;

      case S:                             /* 文字列型の場合 */
      /* 最も件数の多い文字列項目の値をもつhashNodeを探索 */
      maxCnt=0;
      maxHn =NULL;
      for(j=0; j<(*(hash+i))->hashVal; j++){
        if( NULL != (hn=*((*(hash+i))->node+j)) ){
          while(hn!=NULL){
            if(hn->val.v.i > maxCnt){
              maxCnt=hn->val.v.i;
              maxHn =hn;
            }
            hn=hn->next;
          }
        }
      }

      /* 最も件数の多い文字列項目の値をデフォルト値として用いる */
      /* 全ての行がNULLの場合(maxCnt==0)はnullフラグをたてる */
      if(maxCnt!=0){
        (defaultValue+i)->value=mssStrdup(maxHn->str);
      }else{
        (defaultValue+i)->value=mssStrdup(MssNullStr);
      }
    }
  } 
 
  mssFreeFldRec(fr);
  return(defaultValue);
}

/**
 * デフォルト値を表示 (for debug)
 * 【出力サンプル】
 * field[0]: name=temperature, value=58.333333, null=0
 * field[1]: name=humidity, value=68.333333, null=0
 * field[2]: name=windy, value=false, null=0
 * field[3]: name=outlook, value=rain, null=0
 */
void showDefaultValue(struct DefaultValue *defaultValue){
  int i;

  if(defaultValue==NULL) return;

  /* ターミネータはfldNameがNULL */
  for(i=0;(defaultValue+i)->fldName!=NULL;i++){
    printf("field[%d]: type=%d, name=%s, value=%s\n",i,
                                    (defaultValue+i)->fldType,
                                    (defaultValue+i)->fldName,
                                    (defaultValue+i)->value);
  }
}

/**
 * デフォルト値配列の領域開放
 */
static void freeDefaultValue(struct DefaultValue *defaultValue){
  int i;

  if(defaultValue==NULL) return;

  /* ターミネータはfldNameがNULL */
  for(i=0;(defaultValue+i)->fldName!=NULL;i++){
    mssFree((defaultValue+i)->fldName);
    mssFree((defaultValue+i)->value);
  }
}

/**
 * 項目名をcalFormulaにcatする
 * 項目名は"$(項目名)"のように、括弧でくくる
 * 括弧がなければ、項目名に"-"や"+"などの演算子記号がある時に、項目名
 * の一部としてではなく、演算子として評価されてしまうため。
 */
void catFieldName(struct mssStrings *calFormula, char *fldName){
  mssCatStrings(calFormula,"$(");
  mssCatStrings(calFormula,fldName);
  mssCatStrings(calFormula,")");
}

/**
 * compoundPredicateの演算子をcatする。
 * 次の場合はなにもしない。
 *   operatorがNULLの場合(compoundPredicateでない)
 *   compoundPredicateの最初の式である場合
 */
static void catOperator(
  struct mssStrings *calFormula,
  char *operator){

  /* operatorがNULLは、compoundPredicateでないということ */
  if(operator==NULL) return;

  /* compoundPredicateの最初の式の時はoperatorは必要ないのでreturn */
  if(calFormula->str==NULL) return;
  if(*(calFormula->str+calFormula->cnt-1)=='(') return;

  /* conpoundPredicateの論理演算子(and,or,xor,surrogate)を */
  /* mssCalでの演算子に変換してcatする                     */
        if(strcmp(operator,"and")==0){
    mssCatStrings(calFormula," && ");
  }else if(strcmp(operator,"or" )==0){
    mssCatStrings(calFormula," || ");
  }else if(strcmp(operator,"xor")==0){
    mssCatStrings(calFormula," <> ");

  /* surrogateの場合は、"\n"で区切り、mssCalにコンパイルする時に */
  /* 各計算式を別々に評価し格納する                              */
  }else if(strcmp(operator,"surrogate")==0){
    mssCatStrings(calFormula,")\n(");

  /* 上記以外のoperatorが指定されていればエラー終了 */
  }else{
    mssShowErrMsg("unknown booleanOperator in ConpoundPredicate");
    mssEnd(mssErrorNoDefault);
  }
}

/**
 * simplePredicateを評価し文字列式として追加する。
 * operatorはconpoundPredicateの場合に指定され、それ以外の場合はNULL。
 */
static void catSimplePredicate(
  struct mssStrings *calFormula, /* ここにsimplePredicate式が追加される */
  xmlNodePtr node,
  xmlDocPtr doc){

  char *fldName;		/* 項目名格納用ptr */
  int   fldType;		/* 項目のタイプ */

  /* 項目名のセット */
  fldName = mssStrdup(xmlGetProp(node,"field"));

  /* 全体の式を括弧でくくる */
  mssCatStrings(calFormula,"(");

  /* isMissing: isNull()関数に変換 */
  if(xmlStrcmp("isMissing",xmlGetProp(node,"operator"))==0){
    mssCatStrings(calFormula,"isNull(");
    catFieldName(calFormula,fldName);
    mssCatStrings(calFormula,")");
  }

  /* isNotMissing: not(isNull())に変換 */
  else if(xmlStrcmp("isNotMissing",xmlGetProp(node,"operator"))==0){
    mssCatStrings(calFormula,"not(isNull(");
    catFieldName(calFormula,fldName);
    mssCatStrings(calFormula,"))");

  /* 以下missingValue関連以外の式 */
  /* "$項目名 operator Value")    */
  } else {
    /* 項目名をcat */
    catFieldName(calFormula,fldName);

    /* 項目のタイプを得る */
    fldType = getFieldType(doc,fldName);

    /* equal:数値だと==,文字列だと-eq */
    if(xmlStrcmp("equal",xmlGetProp(node,"operator"))==0) {
      if(fldType==N) mssCatStrings(calFormula," == " );
      else           mssCatStrings(calFormula," -eq ");
    }

    /* notEqual:数値だと<>,文字列だと-ne */
    else if(xmlStrcmp("notEqual",xmlGetProp(node,"operator"))==0){
      if(fldType==N) mssCatStrings(calFormula," <> " );
      else           mssCatStrings(calFormula," -ne ");
    }

    /* lessThan:数値だと<,文字列だと-lt */
    else if(xmlStrcmp("lessThan",xmlGetProp(node,"operator"))==0){
      if(fldType==N) mssCatStrings(calFormula," < "  );
      else           mssCatStrings(calFormula," -lt ");
    }

    /* lessOrEqual:数値だと<=,文字列だと-le */
    else if(xmlStrcmp("lessOrEqual",xmlGetProp(node,"operator"))==0){
      if(fldType==N) mssCatStrings(calFormula," <= " );
      else           mssCatStrings(calFormula," -le ");
    }

    /* greaterThan:数値だと>,文字列だと-gt */
    else if(xmlStrcmp("greaterThan",xmlGetProp(node,"operator"))==0){
      if(fldType==N) mssCatStrings(calFormula," > "  );
      else           mssCatStrings(calFormula," -gt ");
    }

    /* greaterOrEqual:数値だと>=,文字列だと-ge */
    else if(xmlStrcmp("greaterOrEqual",xmlGetProp(node,"operator"))==0){
      if(fldType==N) mssCatStrings(calFormula," >= " );
      else           mssCatStrings(calFormula," -ge ");
    }

    /* 文字項目の比較の場合は、値(value)はダブルクオーツでくくる */
    /* 数値項目の比較の場合は、値(value)はそのまま出力           */
    /*   ex) $(outlook) -eq "sunny"                              */
    /*       $(temperature) > 50                                 */
    if(fldType==S) mssCatStrings(calFormula,"\"");
    mssCatStrings(calFormula,xmlGetProp(node,"value"));
    if(fldType==S) mssCatStrings(calFormula,"\"");
  }

  /* 全体の式を括弧でくくる */
  mssCatStrings(calFormula,")");

  /* 領域開放 */
  mssFree(fldName);
}

/**
 * simpleSetPredicateの文字列配列(<Array>要素)を評価し文字列式として追加する。
 * ex)
 * <Array>sunny rain</Array> -> "sunny,rain"
 */
static void catArray(
  struct mssStrings *calFormula, /* ここにArrayが追加される */
  xmlNodePtr node,
  xmlDocPtr doc){

  xmlNodePtr arrayNode; /* "Array"要素 */
  xmlChar    *arrayStr; /* "Array"要素の値 */
  xmlChar    *tokElement;

  /* node以下からArrayノードを検索 */
  arrayNode = nextNodeByXPath(node,"./Array");
  if(arrayNode==NULL){
    mssShowErrMsg("threre is no array under the node: %s",
                                                     xmlGetNodePath(node));
    mssEnd(mssErrorNoDefault);
  }

  /* Arrayの値をセットする
     <Array>sunny rain</Array> -> "sunny rain" */
  arrayStr=xmlNodeListGetString(doc,arrayNode->xmlChildrenNode,1);

  /* トークンを切り出し、calFormulaにカンマ区切りでセットする */
  tokElement=strtok(arrayStr," ");
  while(tokElement!=NULL){
    mssCatStrings(calFormula,",");
    mssCatStrings(calFormula,tokElement);
    tokElement=strtok(NULL,"\n");
  }

  xmlFree(arrayStr);
}

/**
 * simpleSetPredicateを評価し文字列式として追加する。
 * operatorはconpoundPredicateの場合に指定され、それ以外の場合はNULL。
 */
static void catSimpleSetPredicate(
  struct mssStrings *calFormula, /* ここにsimplePredicate式が追加される */
  xmlNodePtr node,
  xmlDocPtr doc){

  char *fldName;		/* 項目名格納用ptr */

  /* 項目名のセット */
  fldName = mssStrdup(xmlGetProp(node,"field"));

  /* 全体の式を括弧でくくる */
  mssCatStrings(calFormula,"(");

  /* isIn: Not Supported(将来、isNot関数を実装して対応予定) */
  if(xmlStrcmp("isIn",xmlGetProp(node,"booleanOperator"))==0){
    mssCatStrings(calFormula,"isIn(");
    catFieldName(calFormula,fldName);
    catArray(calFormula,node,doc);
    mssCatStrings(calFormula,")");

  /* isNotIn: Not Supported(将来、not(isNot)関数を実装して対応予定) */
  }else if(xmlStrcmp("isNotIn",xmlGetProp(node,"booleanOperator"))==0){
    mssCatStrings(calFormula,"not(isIn(");
    catFieldName(calFormula,fldName);
    catArray(calFormula,node,doc);
    mssCatStrings(calFormula,"))");
  }

  /* 全体の式を括弧でくくる */
  mssCatStrings(calFormula,")");

  /* 領域開放 */
  mssFree(fldName);
}

/**
 * <True/>を"(1==1)"の恒真式として登録する。
 * operatorはconpoundPredicateの場合に指定され、それ以外の場合はNULL。
 */
static void catTruePredicate(struct mssStrings *calFormula){
  mssCatStrings(calFormula, "(1==1)");
}

/**
 * <False/>を"(1==2)"の恒偽式として登録する。
 */
static void catFalsePredicate(struct mssStrings *calFormula){
  mssCatStrings(calFormula, "(1==2)");
}

/**
 * nodeの子において最初に出現するPredicateノードを返す
 * Predicateノードがなければエラー終了
 */
xmlNodePtr getFirstPredicateNode(xmlNodePtr node){
  xmlNodePtr result;
  result=nextNodeByXPath(node,
                       "child::True               | \
                        child::False              | \
                        child::SimplePredicate    | \
                        child::SimpleSetPredicate | \
                        child::CompoundPredicate" );

  /* Predicateがなければエラー終了 */
  if(!result){
    mssShowErrMsg("threre is no predicate under the Node: %s",
                                                     xmlGetNodePath(node));
    mssEnd(mssErrorNoDefault);
  }
  return(result);
}

/**
 * nodeの次の兄弟ノード(followin-sibling)で最初に出現するPredicateノードを返す
 * PredicateノードがなければNULLを返す
 */
xmlNodePtr getNextPredicateNode(xmlNodePtr node){
  xmlNodePtr result;
  result=nextNodeByXPath(node,
                       "following-sibling::True               | \
                        following-sibling::False              | \
                        following-sibling::SimplePredicate    | \
                        following-sibling::SimpleSetPredicate | \
                        following-sibling::CompoundPredicate" );
  return(result);
}

static void catCompoundPredicate(
  struct mssStrings *calFormula, /* ここにcompoundPredicate式が追加される */
  xmlNodePtr node,
  xmlDocPtr doc,
  char *operator){               /* compoundPredicateの場合の演算子 */

  while(node){
    /* for debug */
    /* printf("path=%s\n",xmlGetNodePath(node)); */

    /* compoundPredicateの場合、再帰的にノードを評価して登録する */
    if(xmlStrcmp(node->name,"CompoundPredicate")==0){
      mssCatStrings(calFormula,"("); /* 括弧でくくる */
      catCompoundPredicate(calFormula,getFirstPredicateNode(node),doc,
                                        xmlGetProp(node,"booleanOperator"));
      mssCatStrings(calFormula,")"); /* 括弧でくくる */

    /* compoundPredicate以外の場合 */
    }else{
      /* 必要であればcompoundPredicateの演算子をつける */
      catOperator(calFormula, operator);

      /* True単体の場合、式を"1==1"とする */
      if(xmlStrcmp(node->name,"True")==0){
        catTruePredicate(calFormula);
      }

      /* False単体の場合、式を"1==2"とする */
      else if(xmlStrcmp(node->name,"False")==0){
        catFalsePredicate(calFormula);
      }

      /* simplePredicateの場合、式を評価して登録する */
      else if(xmlStrcmp(node->name,"SimplePredicate")==0){
        catSimplePredicate(calFormula,node,doc);
      }

      /* simpleSetPredicateの場合、式を評価して登録する */
      else if(xmlStrcmp(node->name,"SimpleSetPredicate")==0){
        catSimpleSetPredicate(calFormula,node,doc);
      }
    }

    /* 次の兄弟Predicateに移動 */
    node=getNextPredicateNode(node);
  }
}

/**
 * Predicateを文字列式に変換し、その文字列を返す。
 */
static char *getPredicate(xmlNodePtr node,xmlDocPtr doc){
  struct mssStrings *calFormula; /* predicateを文字列式で格納 */
  char *retStr;                  /* 最終的に返す文字列 */
  xmlNodePtr firstPredicateNode; /* "Node"要素下の最初のPredicateノード */

  /* Strings型の初期化 */
  calFormula=mssInitStrings();

  /* 最初のPredicateを得る */
  firstPredicateNode=getFirstPredicateNode(node);

  /* Predicateを文字列式(calFormula)として登録 */
  catCompoundPredicate(calFormula,firstPredicateNode,doc,NULL);

  /* calFormulaから文字列を取り出し、そのアドレスを返す */
  retStr=mssEncoding(calFormula->str,icidUX);
  mssFreeStrings(calFormula);
  return(retStr);
}

/**
 * PMML Treeを作る
 *              pmParent
 *               /
 *              / parent
 *             /
 *          【pmNode (depth, score, predicate, scoreDist)】
 *           //       \            \
 * children // parent  \ parent     \ parent
 *         //           \            \
 *      pmNext -----> pmNext -----> pmNext -----> NULL(兄弟ターミネータ)
 *       //\    next   //\    next   //\
 *      //  \         //  \         //  \
 */
static struct PmNode *makeTree(
  xmlDocPtr doc,
  xmlNodePtr node,
  struct PmNode *pmParent){

  xmlNodePtr next;             /* PMMLドキュメントのnodeをたぐるためのptr */
  struct PmNode *pmNode=NULL;  /* 登録されるPmNode */
  struct PmNode **pmNext=NULL; /* children or nextノードをたぐるためのptr */

  /* PMML Tree の領域確保＆初期化 */
  pmNode = mssCalloc(sizeof(struct PmNode),"makeTree");

  /* ノードの深さを登録 */
  if(pmParent==NULL) pmNode->depth = 0;
  else               pmNode->depth = pmParent->depth+1;

  /* スコア(クラス項目の値)を登録 */
  pmNode->score = mssEncoding(xmlGetProp(node,"score"),icidUX);

  /* Predicate(条件式)をmssCalの文字列式として登録 */
  pmNode->calFormula = getPredicate(node,doc);

  /* ScoreDistribution を登録 (not yet)*/
  /*pmNode->scoreDist = getScoreDist();*/

  /* カレントノードの親ノードを登録 */
  pmNode->parent = pmParent;

  /* 次の階層のNodeを全て兄弟ノードして登録 */
  next=node->children;
  pmNext=&pmNode->children;
  while(next!=NULL){
    if(xmlStrcmp(next->name,"Node")==0){
      *pmNext=makeTree(doc,next,pmNode);
      pmNext=&((*pmNext)->next);
    }
    next=next->next;
  }
  *pmNext=NULL; /* 兄弟ノードのターミネータ */
  return(pmNode);
}

/**
 * calFormulaをmssCalの内部構造にコンパイルしセットする。
 * calFormulaはsurrogateの場合は複数の式が存在するため、それぞれごとに
 * mssCal計算式を構築し、その配列を返す。
 */
struct mssCal **getCalList(char *calFormula, struct mssHeader *hdi){
  struct mssCal **cal=NULL; /* mssCal配列 */
  int cnt=0;                /* その配列要素番号 */
  char *dupCalFormula;      /* オリジナルのcatFormulaのコピー */
  char *formula;            /* トークン分割用一時変数 */

  /* オリジナルを変更しないためにコピーして利用する */
  dupCalFormula=mssStrdup(calFormula);
 
  /* surrogateを考慮して、\nでトークン分割する */
  /* surrogateでなければトークンの数は１である */
  formula=strtok(dupCalFormula,"\n");
  while(formula!=NULL){
    /* 領域確保 */
    cal=mssRealloc(cal,sizeof(struct mssCal *)*(cnt+2),"getCalList");

    /* 計算式の評価 */
    *(cal+cnt)=mssCalCompile(formula,hdi);
    cnt++;
    formula=strtok(NULL,"\n");
  }

  /* ターミネータの追加 */
  *(cal+cnt)=NULL;

  mssFree(dupCalFormula);

  return(cal);
}

/**
 * PMML treeを再帰的に走査し、文字列での式(calFormula)をmssCalにコンパイルする。
 */
static void setCal(struct PmNode *node, struct mssHeader *hdi){

  /*兄弟ノードをたぐっていく*/
  while(node!=NULL){
    /*トップノードは何も表示しない*/
    if(node->depth!=-1){
      node->cal=getCalList(node->calFormula, hdi);
    }
    /*子ノードがあれば再帰呼出*/
    if(node->children!=NULL) setCal(node->children,hdi);

    /*次の兄弟ノードに*/
    node=node->next;
  }
}

/**
 * PMML Treeを表示する(for debug)
 *   calFlg=1 でmssCalを表示 calFlg=0で非表示
 * 【出力サンプル】
 * rsl=will play, surFlg=0, depth=0
 *   +- form=1
 *  rsl=will play, surFlg=0, depth=1
 *    +- form=$(outlook) -eq "sunny"
 *    rsl=will play, surFlg=0, depth=2
 *      +- form=$(temperature) < 90 && $(temperature) > 50
 *      rsl=will play, surFlg=0, depth=3
 *        +- form=$(humidity) < 80
 *      rsl=no play, surFlg=0, depth=3
 *        +- form=$(humidity) >= 80
 *    rsl=no play, surFlg=0, depth=2
 *      +- form=$(temperature) >= 90 || $(temperature) <= 50
 *  rsl=may play, surFlg=0, depth=1
 *    +- form=$(outlook) -eq "overcast" || $(outlook) -eq "rain"
 *    rsl=may play, surFlg=0, depth=2
 *      +- form=$(temperature) > 60 && $(temperature) < 100 && $(outlook) -eq "overcast" && $(humidity) < 70 && $(windy) -eq "false"
 *    rsl=no play, surFlg=0, depth=2
 *      +- form=$(outlook) -eq "rain" && $(humidity) < 70
 */
static void showTree(struct PmNode *node, int calFlg){
  int i;
  struct mssCal **cal;

  /*兄弟ノードをたぐっていく*/
  while(node!=NULL){
    /*トップノードは何も表示しない*/
    if(node->depth!=-1){
      for(i=0; i<node->depth; i++) printf("  "); /*インデント*/
      printf("score=%s, depth=%d\n", node->score,node->depth);
      for(i=0; i<node->depth; i++) printf("  "); /*インデント*/
      printf("  +- form=%s\n",node->calFormula);
      if(calFlg){
        cal=node->cal;
        while(*cal!=NULL){
          mssCalShowTree(*cal,0);
          cal++;
        }
      }
    }
    /*子ノードがあれば再帰呼出*/
    if(node->children!=NULL) showTree(node->children, calFlg);

    /*次の兄弟ノードに*/
    node=node->next;
  }
}

/**
 * pmTree領域の開放
 */
static void freePmTree(struct PmNode *node){
  struct mssCal **cal;
  struct PmNode *next;

  /*兄弟ノードをたぐっていく*/
  while(node!=NULL){
    /*トップノードは何も表示しない*/
    if(node->depth!=-1){

      mssFree(node->score);
      mssFree(node->calFormula);

      /* mssCal領域の開放 */
      cal=node->cal;
      while(*cal!=NULL){
        mssCalFree(*cal);
        cal++;
      }
      mssFree(node->cal);
    }
    /*子ノードがあれば再帰呼出*/
    if(node->children!=NULL){
      freePmTree(node->children);
    }

    /*現在ノードを開放し、次の兄弟ノードに*/
    next=node->next;
    mssFree(node);
    node=next;
  }
}
   
struct mssFldRec *complement(
  struct mssFldRec *fr,
  struct DefaultValue *defaultValue){

  struct mssFldRec *nulfr; /* 補完用行バッファ構造体 */
  int i;

  nulfr=mssInitFldRec(fr->fldCnt);
  nulfr->chrCnt=fr->chrCnt;
  nulfr->eof=fr->eof;

  for(i=0;i<nulfr->fldCnt;i++){
    /* nullだった場合default値に置き換える */
    if(MssIsNull(*(fr->pnt+i))){
      *(nulfr->pnt+i)=(defaultValue+i)->value;
      //nulfr->chrCnt=nulfr->chrCnt-strlen(*(fr->pnt+i))+strlen(dStr[i].value);
      //nulCnt++;
    }else{
      *(nulfr->pnt+i)=*(fr->pnt+i);   
    }
  }

  return(nulfr);
}


/**
 * PMM treeをもとに与えられたデータ(fr)のクラスを予測する
 */
static char *predict(
  struct PmNode *node,                /* PMMLtree           */
  struct mssFldRec *fr,               /* 予測対象データ     */
  struct DefaultValue *defaultValue){ /* NULL値補完用データ */

  MssValue rsl;             /* 計算結果           */
  struct mssCal **cal;      /* 計算式配列         */
  struct mssFldRec *compfr; /* NULL値補完後データ */

  /* 計算実行 */
  cal=node->cal;
  while(*cal!=NULL){     /* surrogateの場合は複数回ループ,その他は１回だけ */
    rsl = mssCalculate(*cal, fr->pnt);
    if(!rsl.nul) break;  /* 結果がNULLでなければbreak */
    cal++;
  }

  /* predicateでは判定不可(NULL)の場合はNULL値を補完して再度試みる */
  if(rsl.nul==1){
    /* NULL値をデフォルト値で補完する */
    compfr=complement(fr,defaultValue);

    /* 再度、計算実行 */
    cal=node->cal;
    while(*cal!=NULL){
      rsl = mssCalculate(*cal, compfr->pnt);
      if(!rsl.nul) break;  /* 結果がNULLでなければbreak */
      cal++;
    }
  }

  /* 補完しても判定不可(NULL)の場合はNULL値(*)を返す */
  if(rsl.nul==1){
    return("*");
  }

  if(rsl.v.d==1){	       /* 判定結果が1(条件にマッチ)だった場合 */
    if(node->children==NULL){  /* leafである */
      return(node->score);
    }else{
      node=node->children;
    }
  }else{		       /* それ以外(0:条件にアンマッチ)だった場合 */
    if(node->next==NULL){      /* いずれの兄弟ノードルールにも合致しなかった */
      return(node->parent->score);
    }else{
      node=node->next;
    }
  }

  /* 再帰呼出 */
  return(predict(node,fr,defaultValue));
}

/**
 * デリミタ文字を"_"に変換し(新しい文字列領域を確保される)そのptrを返す。
 */
char *repDelim(char *str){
  char *repStr; /* 変換された文字列領域 */
  char *pos;    /* repStr走査用 */

  repStr=mssStrdup(str);

  pos=repStr;
  while(*pos!='\0'){
    if(*pos==MssFieldDelim || *pos=='\n'){
      *pos='_';
    }
    pos++;
  }
  return(repStr);
}

int main(int argc, char *argv[]){
/*============================================================================*/
/* オプション宣言＆定義                                                       */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/* 入力xtファイル                                                             */
/*----------------------------------------------------------------------------*/
  MssOptINF optINF={
    OINF,   /* オプションタイプ                                             */
    "i",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    1,      /* 指定可能の最大ファイル数                                     */
    0,      /*1:file not foundのエラーで終了しない 0:する                   */
    INFT,   /* このオプションのタイトル(Helpで表示)                         */
    INFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 入力PMMLファイル                                                           */
/*----------------------------------------------------------------------------*/
  MssOptINF optPML={
    OINF,   /* オプションタイプ                                             */
    "p",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須                                         */
    1,      /* 指定可能の最大ファイル数                                     */
    0,      /*1:file not foundのエラーで終了しない 0:する                   */
    PMLT,   /* このオプションのタイトル(Helpで表示)                         */
    PMLC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 新項目名                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptSLS optFNM={
    OSLS,   /* オプションタイプ                                             */
    "a",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    NULL,   /* デフォルト(文字列)                                           */
    1,      /* カンマで区切られる要素数の最大値                             */
    1,      /* 各要素の文字列長の最小値                                     */
    MssFieldMaxLen,/* 各要素の文字列長の最大値                              */
    0,      /* 1:要素にコロンを指定できる,0:不可  ex) aaaa:xxxxx            */
    FNMT,   /* このオプションのタイトル(Helpで表示)                         */
    FNMC    /* このオプションのコメント(Helpで表示)                         */
  }; 

/*----------------------------------------------------------------------------*/
/* 出力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptOTF optOTF={ 
    OOTF,   /* オプションタイプ                                             */ 
    "o",    /* キーワード(複数文字は不可)                                   */ 
    0,      /* 0:オプション, 1:必須                                         */ 
    OTFT,   /* このオプションのタイトル(Helpで表示)                         */
    OTFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 圧縮出力                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLG optZIP={ 
    OFLG,   /* オプションタイプ                                             */ 
    "z",    /* キーワード(複数文字は不可)                                   */ 
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */ 
    ZIPT,   /* このオプションのタイトル(Helpで表示)                         */
    ZIPC    /* このオプションのコメント(Helpで表示)                         */
  };
  
/*----------------------------------------------------------------------------*/
/* オプションをまとめる                                                       */
/*----------------------------------------------------------------------------*/
  void *opt[]={&optINF,&optPML,&optFNM,&optOTF,&optZIP,NULL};

/*============================================================================*/
/* 変数宣言＆定義                                                             */
/*============================================================================*/
  struct mssFPR    *fpr; 	/* 入力ファイル構造体                */
  struct mssFPR    *fpr2;	/* 計算用入力ファイル構造体	     */
  struct mssFPW    *fpw; 	/* 出力ファイル構造体                */
  struct mssHeader *hdi; 	/* 入力ファイル用<head>タグ格納構造体*/
  struct mssHeader *hdo; 	/* 出力ファイル用<head>タグ格納構造体*/
  struct mssFldRec *fr; 	/* 項目-行バッファ構造体             */

  xmlDocPtr	doc;		/* XML Documentのポインタ            */
  xmlNodePtr	top;		/* XML Nodeのトップポインタ          */
  xmlNodePtr	cur;		/* XML Nodeのカレントポインタ        */
  char		*className;	/* 追加するクラスの項目名            */
  int           classType;      /* クラス(結果属性)のタイプ          */
  struct	PmNode *pmTop; 	/* 新しいツリーのルートノード        */
  char *rslStr;                 /* 計算結果 			     */
  char *encStr;                 /* encoding用一時変数   	     */

  struct DefaultValue *defaultValue;    /* 値がNULL値の時に用いるデフォルト値 */
/*----------------------------------------------------------------------------*/
/* 前処理                                                                     */
/*----------------------------------------------------------------------------*/
  mssInit(argc,argv,&comHelp);       /* シグナル処理などの初期化         */
  mssHelpDoc(opt,&comHelp,argc,argv);/* ヘルプ                           */
  mssSetOption(opt,argc,argv);       /* コマンドオプションの設定         */
  fpr=mssOpenFPR(optINF.str,4);      /* 入力ファイルオープン             */
  fpr2=mssOpenFPR(optINF.str,4);     /* 計算用入力ファイルオープン       */
  hdi=mssReadHeader(fpr);            /* ヘッダの読み込み                 */
  mssSkipToBody(fpr2);		     /* 計算用構造体はヘッダ部をskip     */

  /* UTF-8 -> XMLtableのEncoding変換用iconvオープン */
  icidUX=iconv_open(hdi->xmlenc,"UTF-8");
  if((int)icidUX==-1) {
    mssShowErrMsg("encoding type error in iconv_open");
    mssEnd(mssErrorNoDefault);
  }

  /* XMLtable -> UTF-8のEncoding変換用iconvオープン */
  icidXU=iconv_open("UTF-8",hdi->xmlenc);
  if((int)icidXU==-1) {
    mssShowErrMsg("encoding type error in iconv_open");
    mssEnd(mssErrorNoDefault);
  }

  /* PMMLファイルのperse(xmlDoc構造体へのセット) */
  doc = xmlParseFile(optPML.str);
  if(doc == NULL) {
    mssShowErrMsg("It seems that %s is not a xml data.",optPML.str);
    mssEnd(mssErrorNoDefault);
  }

  /* XPathのコンテキスト(グローバル変数:XpathContext)を設定する */
  setXpathContext(doc);

  /* PMML Treeのルートノードを獲得 */
  top = xmlDocGetRootElement(doc);
  if(top==NULL){                
    xmlFreeDoc(doc);              
    mssShowErrMsg("PMML document is empty");
    mssEnd(mssErrorNoDefault);
  }      

  /* クラス(結果属性)の項目名の獲得 */
  className=getClassName(doc);
  if(className==NULL){
    mssShowErrMsg("cannot find class field in PMML file");
    mssEnd(mssErrorNoDefault);
  }

  /* クラス(結果属性)のタイプの獲得 */
  /* カテゴリ属性でなければならない */
  encStr=mssEncoding(className, icidXU);
  classType=getFieldType(doc,encStr);
  if(classType!=S){
    mssShowErrMsg("type of class field must be categorical in PMML file");
    mssEnd(mssErrorNoDefault);
  }
  mssFree(encStr);

  /* for debug */
  /* printf("className=%s %d\n",className,classType); */

  /* カレントノードを"/PMML/TreeModel/Node"に */
  cur = nextNodeByXPath(top,"/PMML/TreeModel/Node");
  if(cur==NULL){
    mssShowErrMsg("cannot find Node element in PMML file");
    mssEnd(mssErrorNoDefault);
  }

  /* PMMLのNode要素(ツリー)から、内部で利用する木(pmTree)を作る */
  pmTop=makeTree(doc,cur, NULL);

  /* pmTreeを再帰的に走査し、文字列式(calFormula)をmssCalにコンパイルする。*/
  setCal(pmTop, hdi);

  /* for debug 第二引数０で簡易表示、１で詳細表示 */
  /* showTree(pmTop,0); */

  /* NULL値の予測のための各項目のデフォルト値の取得 */
  defaultValue=getDefaultValue(doc,hdi,fpr);

  /* for debug */
  /* showDefaultValue(defaultValue); */

/*----------------------------------------------------------------------------*/
/*出力ヘッダーの作成と出力                                                    */
/*----------------------------------------------------------------------------*/
/*出力ヘッダーの初期化(タイトル等のコピ ー)*/ 
  hdo=mssInitCpyHeader(hdi);
 
  /*入力ヘッダの全項目を追加*/ 
  mssAddFieldsByFields(hdo->flds,hdi->flds);
  
  /*新項目名の追加*/ 
  if(optFNM.set){
    mssAddFieldsByStrList(hdo->flds,optFNM.strList,optFNM.cnt);
  }else{
    mssAddFieldsByStr(hdo->flds,className);
  }

  /*標準出力オープン+ヘッダーの出力*/ 
  fpw=mssOpenFPW(optOTF.str,optZIP.set,0); 
  mssWriteHeader(hdo, fpw);

/*----------------------------------------------------------------------------*/
/*メインルーチン                                                              */
/*----------------------------------------------------------------------------*/
  /*FldRec構造体の初期化*/ 
  fr=mssInitFldRec(hdi->flds->cnt);  

  /* データ先頭行に移動 */
  mssSeekTopFPR(fpr);
 
  while(EOF!=mssReadFldRec(fpr,fr)){
    /* 入力件数カウントアップ */
    mssGV.inCnt++; 

    /* 予測実行 */
    rslStr=predict(pmTop,fr,defaultValue);

    /* デリミタ文字が含まれていれば変換する */
    rslStr=repDelim(rslStr);
   
    /* 入力データ一行書き出し */ 
    mssWriteFld(fr->pnt, fr->fldCnt, MssFieldDelimStr, fpw);
  
    /* 計算結果出力 */ 
    mssWriteStr(rslStr,fpw); 
    mssWriteRet(fpw); 

    /* 領域開放 */
    mssFree(rslStr);

    /* 出力件数カウントアップ */
    mssGV.outCnt++; 
  }

  mssFreeFldRec(fr);  
  freePmTree(pmTop);
  if(icidUX!=NULL) iconv_close(icidUX);  /* icidのクローズ*/
  if(icidXU!=NULL) iconv_close(icidXU);  /* icidのクローズ*/

/*----------------------------------------------------------------------------*/
/* フッター出力&終了処理                                                      */
/*----------------------------------------------------------------------------*/
  freeXpathContext();              /* XPathコンテキストの領域開放 */
  freeDefaultValue(defaultValue);  /* デフォルト値領域の開放 */
  xmlFreeDoc(doc);        /* PMMLファイルの領域開放     */
  mssWriteFooter(fpw);    /* フッターの出力             */ 
  mssCloseFPR(fpr);       /* 入力ファイルのクローズ     */
  mssCloseFPR(fpr2);      /* 入力ファイルのクローズ     */
  mssCloseFPW(fpw);       /* 出力ファイルのクローズ     */  
  mssFreeHeader(hdi);     /* 入力ヘッダ領域開放         */
  mssFreeHeader(hdo);     /* 出力ヘッダ領域開放         */ 
  mssFreeOption(opt);     /* オプション領域の開放       */
  mssShowEndMsg();        /* 完了メッセージ             */
  mssEnd(mssExitSuccess); /* 終了                       */
  return(0);              /* to avoid warning message   */
}
