
/*============================================================================*/
/* 変更履歴                                                                   */
/*----------------------------------------------------------------------------*/
/* 1.0 : 新しいAPIに対応 2003/08/21                                           */
/*============================================================================*/

#include <musashi.h>
#include <stdlib.h>
#include <math.h>

#include <xtasruleHelp.h>
struct mssComHelp comHelp={
  "xtasrule",     /* コマンド名       */
  "1.0",          /* バージョン       */
  HELPT,          /* コマンドタイトル */
  HELPS,          /* 要約             */
  HELPE,          /* 利用例           */
  HELPR,          /* 参照コマンド     */
  HELPA,          /* 作者情報         */
  HELPB,          /* バグレポート情報 */
  HELPH           /* ホームページ     */
};

extern struct mssGlobalVariables mssGV;

int TotalCnt=0;        /*全トランザクション件数*/
int MinSupportCnt=0;   /*サポートの件数(TotalCnt*optSPP*100)*/
int DepthMax;          /*探索する同時購入アイテム数の最大値(optMAX)*/
int PMML;              /*PMMLで出力するフラグ*/

/**
 * 瀕出１アイテムセットを格納する構造体
 */
struct OneItemList {
  struct mssHash *hash;
  struct mssHashNode **index;
  int cnt;
};

/**
 * OneItemList構造体に蓄積されたアイテムを対象に、トランザクションから
 * 以下のような深さDepthMaxのツリーを作成する。
 * "Sn"がStatus構造体である。
 * 各statusには、件数(support)が計算される。
 *
 *    a      b
 * S0-+---S1-+---S4
 *    |      |
 *    |      | c
 *    |      +---S5
 *    |
 *    | b    c
 *    +---S2-----S6
 *    |
 *    | c
 *    +---S3
 * |0   |  1   |  2  | -> status->depth
 *
 * トランザクション{abc,bc}ならば、
 * S1->cnt=1, S2->cnt=2, S3->cnt=1
 * S4->cnt=1, S5->cnt=1, S6->cnt=2
 * となる。
 */
struct Status {
  struct mssHashNode *val;
  int depth;               /*そのstatusの深さ*/
  int support;             /*サポート件数*/
  struct Hash *next;       /*次のstatusへのハッシュ表*/
};

/*StatusとStatusをつなぐハッシュ表*/
struct HashNode {
  int str;
  struct Status *status;
  struct HashNode *next;
};

struct Hash {
  int hashVal;
  int cnt;                 /*挿入された値の数*/
  struct HashNode **node;  /*挿入された値へのポインタ*/
};

/**
 * 上記のツリーにトランザクション"ab"をセットすることを考える。
 * まず、初期のtargetは{S0}にセットされる。
 * そこでS0から"a"のarcをたどり、S1のsupportをカウントアップする。
 * そしてtargetに、そのS1を挿入し、target={S0,S1}となる。
 * 次に"b"の挿入においては、targetにあるS0,S1の両方から"b"のarcをたどり、
 * S2およびS4のsupportをカウントアップする。
 * この時たどりついた、S4の深さはMaxDepthを超えるので、targetには追加しない。
 * そこでtarget={S0,S1,S2}となる。
 * 以上のような手順を繰り返していく。
 */
struct Target {
  int cnt;
  struct Status **list;
};


/**
 *------------------------------------------------------------------------------
 * status間のつなぎのためのHash表関連の関数
 *------------------------------------------------------------------------------
 */

/**
 * Hash表の初期化
 */
struct Hash *initHash(int size){
  struct Hash *hash;

  hash=mssMalloc(sizeof(struct Hash),"initHash");
  hash->hashVal = size;
  hash->cnt = 0;
  hash->node=mssCalloc(sizeof(struct HashNode *)*hash->hashVal,"initHash");
  return(hash);
}

/**
 * Hash表の開放
 */
void freeHash(struct Hash *hash)
{
  int i;
  struct HashNode *node;
  struct HashNode *tmp;

  if(hash==NULL) return;
  for(i=0; i<hash->hashVal; i++){
    if( NULL != (node=*(hash->node+i)) ){
      while(node!=NULL){
        tmp=node;
        node=node->next;
        mssFree(tmp);
      }
    }
  }
  mssFree(hash->node);
  mssFree(hash);
}


/**
 * Hash値の計算
 * strは、アイテム文字列を格納したmssHash表のアドレスである。
 * そのアドレスを無理矢理int型に変換して利用している。
 */
static int getHashVal(int str,int hv){
  return(str % hv);
}

/**
 * Status->nextのHash表に次のstatus *valを挿入し、そのHashNodeアドレスを返す。
 * すでに存在する時は、そのHashNodeアドレスを返す。
 */
struct HashNode *hashInsertAdd(struct Hash *hash, int str, struct Status *val)
{
  int hv;
  struct HashNode *newNode;
  struct HashNode *node;
  struct HashNode *last=NULL;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);

  newNode=mssMalloc(sizeof(struct HashNode),"hashInsertAdd");

  if(node==NULL){
    *(hash->node+hv)=newNode;
  }else{
    while(node!=NULL){
      if( node->str==str ){
        mssFree(newNode);
        return(node);
      }else{
        last=node;
        node=node->next;
      }
    }
    last->next=newNode;
  }
  newNode->str=str;
  newNode->status=val;
  newNode->next=NULL;
  hash->cnt++;
  return(newNode);
}

/**
 * Status->nextのHash表に次のstatusがあるかを確かめ、あればそのHashNodeを返し、
 * なければNULLを返す。
 */
struct HashNode *hashMemberAdd(struct Hash *hash, int str)
{
  int hv;
  struct HashNode *node;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);
  while(node!=NULL){
    if( node->str==str ){
      return(node);
    }else{
      node=node->next;
    }
  }
  return(NULL);
}

/**
 *------------------------------------------------------------------------------
 * Status構造体&ツリー関連
 *------------------------------------------------------------------------------
 */

/**
 * Status構造体の初期化
 */
struct Status *initStatus(int depth){
  struct Status *status;
  status=mssCalloc(sizeof(struct Status),"initStatus");
  status->depth=depth;
  status->next=initHash(101);
  return(status);
}

/**
 * Status構造体の開放
 */
void freeStatus(struct Status *status){
  freeHash(status->next);
  mssFree(status);
}

/**
 * Target構造体に、指定のstatusを追加する。
 */
void addTarget(struct Target *target, struct Status *status){

  target->cnt++;
  target->list=mssRealloc(target->list, 
                          sizeof(struct Status *)*target->cnt,"addTarget");
  *(target->list+target->cnt-1)=status;
}

/**
 * 指定のstatusからchrのArcをたどった先のstatusを返す。
 * もし、そのstatusがまだ登録されていなければ、追加される。
 */
struct Status *nextStatus(struct Status *status, struct mssHashNode *chr){
  struct HashNode *node;
  struct Status *newStatus;

  node=hashMemberAdd(status->next, (int)chr);
  if(node==NULL){
    newStatus=initStatus(status->depth+1);
    newStatus->val=chr;
    node=hashInsertAdd(status->next, (int)chr, newStatus);
  }
  return(node->status);
}

/**
 * targetに登録された全statusからアイテムchrをたどり、そのStatusのsupport
 * をカウントアップする。
 * そして、そのたどり着いたstatusのMaxDepthを超えていなければ、そのstatus
 * をtargetに追加登録する。
 * もし、そのstatusがまだ登録されていなければ、追加される。
 */
void setStatus(struct Target *target, struct mssHashNode *chr){
  int i;
  int lastTargetCnt;
  struct Status *next;

  lastTargetCnt=target->cnt;
  for(i=0; i<lastTargetCnt; i++){
    next=nextStatus(*(target->list+i), chr);
    next->support++;
    if(next->depth<DepthMax){
      addTarget(target,next);
    }
  }
}

/**
 * アイテムリストstrによってツリーの各statusのsupportをカウントアップする
 */
void addTree(struct Status *top, struct mssHashNode **str, int strLen){
  struct Target target;
  int i;

  target.cnt=0;
  target.list=NULL;

  addTarget(&target,top);

  for(i=0; i<strLen; i++){
    setStatus(&target,*(str+i));
  }
}

/**
 * ツリーの表示(デバッグ用)
 */
void showTree(struct Status *status){
  int i;
  struct HashNode *node;


  for(i=0; i<status->depth; i++){
    printf("  ");
  }

  /*トップノード判定*/
  if(status->val!=NULL){
    printf("status=%s [%d]\n",status->val->str,status->support);
  }else{
    printf("[top]\n");
  }

  if(status->next->cnt==0){
    return;
  }

  for(i=0; i<status->next->hashVal; i++){
    if( NULL != (node=*(status->next->node+i)) ){
      while(node!=NULL){
        showTree(node->status);
        node=node->next;
      }
    }
  }
}

/**
 * ツリーの領域開放
 */
void freeTree(struct Status *status){
  int i;
  struct HashNode *node;

  if(status->next->cnt==0){
    freeStatus(status);
    return;
  }

  for(i=0; i<status->next->hashVal; i++){
    if( NULL != (node=*(status->next->node+i)) ){
      while(node!=NULL){
        freeTree(node->status);
        node=node->next;
      }
    }
  }
}

/**
 *------------------------------------------------------------------------------
 * 結果表示関連
 *------------------------------------------------------------------------------
 */
struct OutItem {
  struct Status **list;
  struct Status *top;
  int cnt;
};

/**
 * OutItem構造体に、指定のItem(mssHashNode)を追加する。
 */
void addOutItem(struct OutItem *outItem, struct Status *item){

  outItem->cnt++;
  outItem->list=mssRealloc(outItem->list, 
                          sizeof(struct Status *)*outItem->cnt,"addOutItem");
  *(outItem->list+outItem->cnt-1)=item;
}

/**
 * OutItem構造体から最後のItem(mssHashNode)を削除する。
 */
void delOutItem(struct OutItem *outItem){

  outItem->cnt--;
  if(outItem->cnt==0){
     mssFree(outItem->list);
     outItem->list=NULL;
  }else{
    outItem->list=mssRealloc(outItem->list, 
                          sizeof(struct Status *)*outItem->cnt,"delOutItem");
  }
}

/**
 * OutItem->listからi番目のItemを省いたItemSetの件数を返す。
 * outItemに一件のみItemがあるなら、全体の件数を返す。
 */
int  getItemCnt(struct OutItem *outItem, int omit){
  struct Status *status;
  struct HashNode *hn;
  int i;

  if(outItem->cnt==1){
    return(TotalCnt);
  }

  status=outItem->top;
  for(i=0; i<outItem->cnt; i++){
    if(i!=omit){
      hn=hashMemberAdd(status->next, (int)(*(outItem->list+i))->val);
      if(hn==NULL) {
        mssShowErrMsg("internal error: getItemCnt");
        mssEnd(mssErrorNoDefault);
      }
      status=hn->status;
    }
  }
  return(status->support);
}

/**
 * OutItem->listからi番目のItemを省いたItemSetの件数を返す。
 * outItemに一件のみItemがあるなら、全体の件数を返す。
 */
int  getOneItemCnt(struct OutItem *outItem, struct Status *status){
  struct HashNode *hn;

  hn=hashMemberAdd(outItem->top->next, (int)status->val);
  return(hn->status->support);
}

void writeRulePMML(
  struct OutItem *outItem,
  int conNo,   /*outItem->listにおける後件番号*/
  int conCnt,  /*後件部件数*/
  int antCnt,  /*前件部件数*/
  struct mssFPW *fpw){
  int i;
  int firstWrite;
  int conTotalCnt;
  double conf,supp,lift;

  /*前件部出力*/
  if(outItem->cnt==1){
    mssWriteNull(fpw);
    mssWriteDlm(fpw);
  }else{
    firstWrite=1;
    for(i=0; i<outItem->cnt; i++){
      if(i!=conNo){
        if(!firstWrite){
          mssWriteStr(",",fpw);
        }
        mssWriteStr((*(outItem->list+i))->val->str,fpw);
        firstWrite=0;
      }
    }
    mssWriteDlm(fpw);
  }
  conTotalCnt=getOneItemCnt(outItem,*(outItem->list+conNo));
  conf=(double)conCnt/(double)antCnt;
  supp=(double)conCnt/(double)TotalCnt;
  lift=conf/((double)conTotalCnt/(double)TotalCnt);

  /*後件部出力*/
  mssWriteStr( (*(outItem->list+conNo))->val->str, fpw);
  mssWriteDlm(fpw);

  /*前件部Item数(itemSize)*/
  mssWriteInt(outItem->cnt,fpw);
  mssWriteDlm(fpw);

  /*前件部件数(antecedentCnt)*/
  mssWriteInt(antCnt,fpw);
  mssWriteDlm(fpw);

  /*後件部件数(consequenctCnt)*/
  mssWriteInt(conCnt,fpw);
  mssWriteDlm(fpw);

  /*後件部全件数(consequenctTotal)*/
  mssWriteInt(conTotalCnt,fpw);
  mssWriteDlm(fpw);

  /*全件数(totalCnt)*/
  mssWriteInt(TotalCnt,fpw);
  mssWriteDlm(fpw);

  /*Support*/
  mssWriteDbl(supp,fpw);
  mssWriteDlm(fpw);

  /*Confidence*/
  mssWriteDbl(conf,fpw);
  mssWriteDlm(fpw);

  /*Lift*/
  mssWriteDbl(lift,fpw);
  mssWriteRet(fpw);
}

void writeRuleTEXT(
  struct OutItem *outItem,
  int conNo,   /*outItem->listにおける後件番号*/
  int conCnt,  /*後件部件数*/
  int antCnt,  /*前件部件数*/
  struct mssFPW *fpw){
  int i;
  int firstWrite;
  int conTotalCnt;
  double conf,supp,lift;

  /*前件部出力*/
  if(outItem->cnt==1){
    mssWriteNull(fpw);
    mssWriteDlm(fpw);
  }else{
    firstWrite=1;
    for(i=0; i<outItem->cnt; i++){
      if(i!=conNo){
        if(!firstWrite){
          mssWriteStr(",",fpw);
        }
        mssWriteStr((*(outItem->list+i))->val->str,fpw);
        firstWrite=0;
      }
    }
    mssWriteDlm(fpw);
  }
  conTotalCnt=getOneItemCnt(outItem,*(outItem->list+conNo));
  conf=(double)conCnt/(double)antCnt;
  supp=(double)conCnt/(double)TotalCnt;
  lift=conf/((double)conTotalCnt/(double)TotalCnt);

  /*後件部出力*/
  mssWriteStr( (*(outItem->list+conNo))->val->str, fpw);
  mssWriteDlm(fpw);

  /*前件部Item数(itemSize)*/
  mssWriteInt(outItem->cnt,fpw);
  mssWriteDlm(fpw);

  /*前件部件数(antecedentCnt)*/
  mssWriteInt(antCnt,fpw);
  mssWriteDlm(fpw);

  /*後件部件数(consequenctCnt)*/
  mssWriteInt(conCnt,fpw);
  mssWriteDlm(fpw);

  /*後件部全件数(consequenctTotal)*/
  mssWriteInt(conTotalCnt,fpw);
  mssWriteDlm(fpw);

  /*全件数(totalCnt)*/
  mssWriteInt(TotalCnt,fpw);
  mssWriteDlm(fpw);

  /*Support*/
  mssWriteDbl(supp,fpw);
  mssWriteDlm(fpw);

  /*Confidence*/
  mssWriteDbl(conf,fpw);
  mssWriteDlm(fpw);

  /*Lift*/
  mssWriteDbl(lift,fpw);
  mssWriteRet(fpw);
  mssGV.outCnt++;
}

/**
 * OutItem->listにセットされたItemからConf以上のルールを書き出す
 */
void calConf(struct OutItem *outItem, int support, double minConf, struct mssFPW *fpw){
  int i;
  int antCnt; /*前件部件数*/
  int conCnt; /*後件部件数*/
  double conf;

  conCnt=(*(outItem->list+outItem->cnt-1))->support;
  for(i=0; i<outItem->cnt; i++){
    /*iは前件部で省くアイテム*/
    antCnt=getItemCnt(outItem,i);
    conf = (double)conCnt/(double)antCnt;
    if(conf>=minConf){
      if(PMML){
        writeRulePMML(outItem,i,conCnt,antCnt,fpw);
      }else{
        writeRuleTEXT(outItem,i,conCnt,antCnt,fpw);
      }
    }
  }
}

void writeRslSub(struct OutItem *outItem, struct Status *status,int minSupp, double minConf, struct mssFPW *fpw){
  int i;
  struct HashNode *node;

  /*トップノードの時はとばす*/
  if(status->val!=NULL){

    if(status->support < minSupp){
      return;
    }

    calConf(outItem, status->support, minConf,fpw);

    if(status->next->cnt==0){
      return;
    }
  }

  for(i=0; i<status->next->hashVal; i++){
    if( NULL != (node=*(status->next->node+i)) ){
      while(node!=NULL){
      addOutItem(outItem, node->status);
        writeRslSub(outItem,node->status,minSupp,minConf,fpw);
        delOutItem(outItem);
        node=node->next;
      }
    }
  }
}

void writeRsl(struct Status *top, int minSupp, double minConf, struct mssFPW *fpw){

  struct OutItem *outItem;
  outItem=mssMalloc(sizeof(struct OutItem), "writeRsl");
  outItem->cnt=0;
  outItem->list=NULL;
  outItem->top=top;

  writeRslSub(outItem,top,minSupp,minConf,fpw);

  mssFree(outItem->list);
  mssFree(outItem);

}

/**
 *------------------------------------------------------------------------------
 * 瀕出１アイテムセット関連
 *------------------------------------------------------------------------------
 */

/**
 * 指定のsupport以上の瀕出１アイテムセットをハッシュ表に登録する
 */
struct OneItemList *setFrequent1ItemList(MssOptKEY *optKEY,int itemFldNo, double supp, struct mssFldRecDbl *frd,struct mssFPR *fpr){
  struct OneItemList *itemTable;
  struct mssHash *hash;
  int itemCnt;

  struct mssHashNode *node;
  MssValue v;
  int i;

  mssVinit(&v,INT);

  itemTable=mssMalloc(sizeof(struct OneItemList),"setItem");

  hash=mssInitHash(101);

  TotalCnt=1;
  while( EOF!=mssReadFRD(fpr,frd) ){

    /*キーブレイク時の処理*/
    if(mssKeyBreak(frd, optKEY)){
      if(frd->eof) break;
      TotalCnt++; /*この変数に全トランザクション件数を入れる*/
    }

    node=mssHashInsertAdd(hash,*(frd->pnt[frd->new]+itemFldNo),v);
    node->val.v.i++;
  }
  MinSupportCnt=(int)ceil(TotalCnt*supp);

  /*サポート件数より少ないノードを削除*/
  itemCnt=0;
  for(i=0; i<hash->hashVal; i++){
    if( NULL != (node=*(hash->node+i)) ){
      while(node!=NULL){
        if(node->val.v.i < MinSupportCnt){
          node=mssHashDeleteNode(hash,node);
          continue;
        }else{
          itemCnt++;
        }
        node=node->next;
      }
    }
  }

  itemTable->hash=hash;
  itemTable->cnt=itemCnt;

  return(itemTable);
}

/**
 * 一トランザクションの複数アイテムをseqテーブルに登録する
 * この時、瀕出１アイテムセットに登録されていないアイテムは省かれる
 */
struct mssHashNode **setSeqDat(
  struct mssHashNode **seq, /*シーケンスが入る配列*/
  struct mssHash *hash,     /*瀕出１アイテムセット(hash表)*/
  int *seqCnt,                /*トランザクション内で何番目のitemか*/
  char *str){               /*登録する文字列としてのitem*/

  struct mssHashNode *hn;  /*登録されたhashNodeへのポインタ*/

  if( NULL != (hn=mssHashMember(hash,str)) ){
    seq=mssRealloc(seq,sizeof(struct mssHashNode *)*(*seqCnt+1),"seq1");
    *(seq+*seqCnt)=hn;
    (*seqCnt)++;
  }
  return(seq);
}

void showSeqDat(
  struct mssHashNode **seq, /*シーケンスが入る配列*/
  struct mssHash *hash,     /*瀕出１アイテムセット(hash表)*/
  int seqCnt){              /*トランザクション内で何番目のitemか*/
  int i;
  for(i=0; i<seqCnt; i++){
    printf("%s-",(*(seq+i))->str );
  }
  printf("\n");
}

int main(int argc, char *argv[]){
/*============================================================================*/
/* オプション宣言＆定義                                                       */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/* キー項目(トランザクション識別項目)                                         */
/*----------------------------------------------------------------------------*/
  MssOptKEY optKEY={
    OKEY,   /* オプションタイプ                                             */
    "k",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    MssFieldMaxCnt, /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    1,      /* デフォルト(このオプションが指定されなかったときの動作を指定) */
            /* 1:全ての行を異るキー値として扱う                             */
            /* 2:全ての行を同じキー値として扱う)                            */
    KEYT,   /* このオプションのタイトル(Helpで表示)                         */
    KEYC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* アイテム項目                                                               */
/*----------------------------------------------------------------------------*/
  MssOptFLD optFLD={
    OFLD,   /* オプションタイプ                                             */
    "f",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    MssFieldMaxCnt, /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    1,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    1,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    NULL,   /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    FLDT,   /* このオプションのタイトル(Helpで表示)                         */
    FLDC,   /* このオプションのコメント(Helpで表示)                         */
    FLDF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* サポート件数(割合)                                                         */
/*----------------------------------------------------------------------------*/
  MssOptDBL optSPP={
    ODBL,   /* オプションタイプ                                             */
    "S",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    0.2,    /* デフォルト(数値として指定)                                   */
    0,      /* 最小値                                                       */
    1,      /* 最大値                                                       */
    SPPT,   /* このオプションのタイトル(Helpで表示)                         */
    SPPC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* confidence                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptDBL optCNF={
    ODBL,   /* オプションタイプ                                             */
    "C",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    0.2,    /* デフォルト(数値として指定)                                   */
    0,      /* 最小値                                                       */
    1,      /* 最大値                                                       */
    CNFT,   /* このオプションのタイトル(Helpで表示)                         */
    CNFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 探索する同時購入アイテム件数                                               */
/*----------------------------------------------------------------------------*/
  MssOptINT optMAX={
    OINT,   /* オプションタイプ                                             */
    "M",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    4,      /* デフォルト(数値として指定)                                   */
    1,      /* 最小値                                                       */
    10,     /* 最大値                                                       */
    MAXT,   /* このオプションのタイトル(Helpで表示)                         */
    MAXC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 新項目名                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptSLS optFNM={
    OSLS,   /* オプションタイプ                                             */
    "a",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    "antecedent,consequent,itemSize,antecedentCnt,consequentCnt,consequentTotal,totalCnt,support,confidence,lift", /* デフォルト(文字列)                   */
    10,     /* カンマで区切られる要素数の最大値                             */
    1,      /* 各要素の文字列長の最小値                                     */
    MssFieldNameMaxLen,/* 各要素の文字列長の最大値                          */
    0,      /* 1:要素にコロンを指定できる,0:不可  ex) aaaa:xxxxx            */
    FNMT,   /* このオプションのタイトル(Helpで表示)                         */
    FNMC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 入力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptINF optINF={
    OINF,   /* オプションタイプ                                             */
    "i",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須                                         */
    1,      /* 指定可能の最大ファイル数                                     */
    0,      /*1:file not foundのエラーで終了しない 0:する                   */
    INFT,   /* このオプションのタイトル(Helpで表示)                         */
    INFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 出力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptOTF optOTF={
    OOTF,   /* オプションタイプ                                             */
    "o",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    OTFT,   /* このオプションのタイトル(Helpで表示)                         */
    OTFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* PMML                                                                       */
/*----------------------------------------------------------------------------*/
  MssOptFLG optPML={
    OFLG,   /* オプションタイプ                                             */
    "p",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    PMLT,   /* このオプションのタイトル(Helpで表示)                         */
    PMLC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* plain text                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptFLG optTXT={
    OFLG,   /* オプションタイプ                                             */
    "t",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    TXTT,   /* このオプションのタイトル(Helpで表示)                         */
    TXTC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* ワークファイル用ディレクトリ名                                             */
/*----------------------------------------------------------------------------*/
  MssOptSTR optTMP={
    OSTR,   /* オプションタイプ                                             */
    "T",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    MssTempDir, /* デフォルト                                               */
    1,      /* 文字列の最小長                                               */
    MssFileNameMaxLen,  /* 文字列の最大長                                  */
    TMPT,   /* このオプションのタイトル(Helpで表示)                         */
    TMPC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* オプションをまとめる                                                       */
/*----------------------------------------------------------------------------*/
  void *opt[]={&optKEY,&optFLD,&optSPP,&optCNF,&optMAX,&optPML,&optFNM,
               &optINF,&optOTF,&optTXT,&optTMP,NULL};

/*============================================================================*/
/* 変数宣言＆定義                                                             */
/*============================================================================*/
  struct mssHeader    *hdi;      /*入力ファイル用<head>タグ格納構造体*/
  struct mssHeader    *hdo;      /*出力ファイル用<head>タグ格納構造体*/
  struct mssFPR       *fpr;      /*入力ファイル構造体                */
  struct mssFPW       *fpw;      /*出力ファイル構造体                */
  struct mssFldRecDbl *frd=NULL; /*項目-キーバッファ構造体           */
  struct mssFields     *sf;      /*ソート項目構造体                  */
  int sorted;                    /*ソート済チェック用                */

  struct OneItemList *oneItemList; /*瀕出１アイテムリスト*/
  int itemFldNo;                 /*-f項目のデータ上での位置*/
  struct mssHashNode **seq;      /*itemのシーケンスが入る配列*/
  int seqCnt;                     /*シーケンスでのカレントitem番号*/

  struct Status *topStatus;

/*----------------------------------------------------------------------------*/
/* 前処理                                                                     */
/*----------------------------------------------------------------------------*/
  mssInit(argc,argv,&comHelp);       /* シグナル処理などの初期化              */
  mssHelpDoc(opt,&comHelp,argc,argv);/* ヘルプ                                */
  mssSetOption(opt,argc,argv);       /* コマンドオプションの設定              */
  fpr=mssOpenFPR(optINF.str,4);      /* 入力ファイルオープン                  */
  hdi=mssReadHeader(fpr);            /* ヘッダの読み込み                      */
  mssSetOptKey(&optKEY, hdi);        /* -k 項目をヘッダー項目に関連づける     */
  mssSetOptFld(&optFLD, hdi);        /* -f 項目をヘッダー項目に関連づける     */

  /*-pは現在サポートされていない*/
  if(optPML.set){
    mssShowErrMsg("PMML is not supported in the current version");
    mssEnd(mssErrorNoDefault);
  }

  /*-aを指定する時、個数は10でなければならない*/
  if(optFNM.set && optFNM.cnt!=10){
    mssShowErrMsg("must specify 10 field names on -a");
    mssEnd(mssErrorNoDefault);
  }

  DepthMax=optMAX.val;
  PMML=optPML.set;

  /*ソート項目の作成*/
  sf=mssInitFields();
  mssAddFieldsByFields(sf,optKEY.flds); /* -k 項目をソート項目としてセット    */
  mssAddFieldsByFields(sf,optFLD.flds); /* -k 項目をソート項目としてセット    */
  mssSetFieldsSortPriority(sf);         /* ソート優先順位番号を登録順にふる   */
  sorted=mssChkSorted(sf,hdi);          /* ソート済かチェック                 */

/*mssShowFields(sf);*/
/*mssShowOption(opt);*/
/*mssShowHeader(hdi);*/


/*----------------------------------------------------------------------------*/
/*メインルーチン                                                              */
/*----------------------------------------------------------------------------*/
  /*ソートが必用ならばソートしてソート済みファイルとしてオープン*/
  if(!sorted){
    fpr=mssReopenFPRsort(fpr,4,sf,hdi->flds->cnt,optTMP.str);
  }


  /*------------------------------------------------*/
  /*サポート件数以上の１アイテム瀕出リストを作成する*/
  /*------------------------------------------------*/
  mssShowMsg("scanning frequent one item list");
  itemFldNo=MssFlds2num(optFLD.flds,0); /*item項目のデータ上での位置*/
  frd=mssInitFRD(hdi->flds->cnt);       /*レコード-項目構造体*/
  oneItemList = setFrequent1ItemList(&optKEY,
                              MssFlds2num(optFLD.flds,0),
                              optSPP.val,
                              frd, fpr);
  if(MinSupportCnt<=0) MinSupportCnt=1;
/*printf("totalCnt=%d, MinSupportCnt=%d\n",TotalCnt,MinSupportCnt);*/

  /*mssShowHash(oneItemList->hash);*/
  mssFreeFRD(frd);

  /*------------------------------------------------*/
  /* ツリーの作成                                   */
  /*------------------------------------------------*/
  mssShowMsg("creating tree");
  topStatus=initStatus(0); /*深さ０のトップノードstatusの作成*/
  seq=NULL;                /*各トランザクションごとのアイテムリスト*/
  seqCnt=0;                /*シーケンスでのカレントitem番号*/
  mssSeekTopFPR(fpr);
  frd=mssInitFRD(hdi->flds->cnt);
  while( EOF!=mssReadFRD(fpr,frd) ){
    /*キーブレイク時の処理*/
    if(mssKeyBreak(frd, &optKEY)){
      if(seqCnt>0){
        addTree(topStatus,seq,seqCnt);
      }
      /*showSeqDat(seq,oneItemList->hash, seqCnt);*/
      mssFree(seq);
      seq=NULL;
      seqCnt=0;
      if(frd->eof) break;
    }
   
    mssGV.inCnt++;
    /*通常行の処理(アイテムリストの作成)*/
    seq=setSeqDat(seq,oneItemList->hash, &seqCnt,
                  *(frd->pnt[frd->new]+itemFldNo));
  }
  /*showTree(topStatus);*/

  /*------------------------------------------------*/
  /* 結果の出力                                     */
  /*------------------------------------------------*/
  mssShowMsg("outputing result");

  fpw=mssOpenFPW(optOTF.str,0,0);

  /*出力ヘッダーの初期化(タイトル等のコピー)*/
  hdo=mssInitCpyHeader(hdi);

  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+0)); /*unit項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+1)); /*no項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+2)); /*len項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+3)); /*cnt項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+4)); /*totalCnt項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+5)); /*totalCnt項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+6)); /*totalCnt項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+7)); /*totalCnt項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+8)); /*totalCnt項目名の追加*/
  mssAddFieldsByStr(hdo->flds,*(optFNM.strList+9)); /*totalCnt項目名の追加*/

  if(!PMML) mssWriteHeader(hdo, fpw);

  writeRsl(topStatus,MinSupportCnt,optCNF.val,fpw);

  /*------------------------------------------------*/
  /* 各種領域開放                                   */
  /*------------------------------------------------*/
  freeTree(topStatus);
  mssFreeHash(oneItemList->hash);
  mssFreeFRD(frd);

/*----------------------------------------------------------------------------*/
/*フッター出力&終了処理                                                       */
/*----------------------------------------------------------------------------*/
  if(!PMML) mssWriteFooter(fpw);    /* フッターの出力             */
  mssCloseFPR(fpr);       /* 入力ファイルのクローズ     */
  mssCloseFPW(fpw);       /* 出力ファイルのクローズ     */
  mssFreeFields(sf);      /* ソート項目構造体の領域開放 */
  mssFreeHeader(hdi);     /* 入力ヘッダ領域開放         */
  mssFreeHeader(hdo);     /* 出力ヘッダ領域開放         */
  mssFreeOption(opt);     /* オプション領域開放         */
  mssShowEndMsg();        /* 完了メッセージ             */
  mssEnd(mssExitSuccess); /* 終了                       */
  return(0);              /* to avoid warning message   */
}
