/**
 * # CHAPTER #
 * ============================================================================
 * BONSAI関連の関数
 * ============================================================================
 */

#include <musashi.h>
#include <bonsai.h>
#include <usint.h>
#include <exHash.h>
#include <condition.h>
#include <priorityQueue.h>

/* ############ グローバル変数 ##############*/
/*パラーメータ*/
extern MssOptFLG optSEQ;
extern MssOptINT optEND;
extern MssOptINT optBGN;
extern MssOptINT optSED;
extern MssOptINT optCAN;

extern int ClassSize; /*クラスのサイズ*/

int usedRandSeed; /*利用された乱数の種(表示用としてtree.cで参照)*/

/*
 * # STRUCT #
 * 正規パターン候補を格納するプライオリティキュー構造体
 */
struct RegBest{
  struct PQnode *pqNode; /*プライオリティキュー本体*/
  int    cnt;            /*現在登録されている候補数*/
  int    maxCnt;         /*正規パターン最大候補数(optCAN)*/
  double maxVal;         /*プライオリティキューに登録されている中で最も*/
                         /*悪いGiniIndexの値*/
};

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * アルファベット−インデックス関連の関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * ローカルサーチ初期のインデックスとして適切であれば1、else 0を返す
 * ここでの適切とは、インデックスのサイズ分の種類を全て用いてるインデックス
 */
static int goodInitIdx( usint *idx, int cnt){
  int len;
  int i;
  char buf[256];

  for(i=1; i<=cnt; i++) buf[i]=-1;

  len=strlenUSI(idx);
  for(i=0; i<len; i++){
    buf[*(idx+i)]=1;
  }
  for(i=1; i<=cnt; i++){
    if(buf[i]==-1)return(0);
  }
  return(1);
}

/**
 * # FUNCTION #
 * インデックスとして適切であれば1、else 0を返す
 * ここでの「適切」とは、全てが同じでないインデックス
 */
static int goodIdx(usint *idx){
  int i;
  usint chr;

  chr=*idx;
  for(i=1; i<strlenUSI(idx); i++){
    if( chr!=*(idx+i) ) return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 * インデックスをアルファベットと同じ種類用意する
 * インデックス化なし(-s 0)の場合に用いる
 */
struct Idxs *initIdxs0(struct Data *dat){
  int i,j,k;
  int size;
  struct Idxs *idxs;
  usint *str;

  usedRandSeed=mssInitRand(optSED.val); /*乱数初期化*/

  idxs=mssMalloc(sizeof(struct Idxs),"initIdxs01");
  idxs->idx=mssMalloc(sizeof(usint **)*1,"initIdxs02");
  for(i=0; i<1; i++){
    *(idxs->idx+i)=mssMalloc(sizeof(usint *)*dat->patCnt,"initIdxs03");
  }
  idxs->cnt=1;
  idxs->patCnt=dat->patCnt;

  for(j=0; j<dat->patCnt; j++){

    size=(dat->pat+j)->map->alpSiz;
    *(*(idxs->idx+0)+j)=mssCalloc(sizeof(usint)*(size+1),"initIdxs04");
    str=*(*(idxs->idx+0)+j);

    for(k=0; k<size; k++){
      *(str+k) = 1+k;
    }
    *(str+k) = 0;
  }
  return(idxs);
}

/**
 * # FUNCTION #
 * インデックスをランダムに決定する
 */
struct Idxs *initIdxs(struct Data *dat){
  int i,j,k;
  int size;
  struct Idxs *idxs;
  int unit;
  usint num;
  usint *str;
  int numCnt;
  int digits;

  usedRandSeed=mssInitRand(optSED.val); /*乱数初期化*/

  idxs=mssMalloc(sizeof(struct Idxs),"initIdxs1");
  idxs->idx=mssMalloc(sizeof(usint **)*1,"initIdxs2");
  for(i=0; i<1; i++){
    *(idxs->idx+i)=mssMalloc(sizeof(usint *)*dat->patCnt,"initIdxs3");
  }
  idxs->cnt=1;
  idxs->patCnt=dat->patCnt;

  for(j=0; j<dat->patCnt; j++){
    unit=RAND_MAX/(dat->pat+j)->map->idxSiz;

    size=(dat->pat+j)->map->alpSiz;
    *(*(idxs->idx+0)+j)=mssCalloc(sizeof(usint)*(size+1),"initIdxs4");
    str=*(*(idxs->idx+0)+j);

    /*数値パターンの場合*/
    if((dat->pat+j)->numPat){
      numCnt=0;
      while(numCnt<(dat->pat+j)->map->idxSiz-1){
        digits=rand()/(RAND_MAX/((dat->pat+j)->map->alpSiz-1))+1;
        if( *(str+digits) != 0 ) continue;
        *(str+digits) = 9999;
        numCnt++;
      }
      num=1;
      for(i=0; i<size; i++){
        if(*(str+i)==9999) num+=1;
        *(str+i)=num;
      }

    /*文字パターンの場合*/
    }else{
      while(1){
        for(k=0; k<size; k++){
          *(str+k) = 1+rand()/unit;
        }
        *(str+k) = 0;
        if( goodInitIdx(str, (dat->pat+j)->map->idxSiz) )break;
      }
    }
  }
  return(idxs);
}

/**
 * # FUNCTION #
 * インデックスリストの開放(パターン項目数分)
 */
static void freeIdxsSub(usint **add, int cnt){
  int i;
  for(i=0; i<cnt; i++){
    mssFree(*(add+i));
  }
  mssFree(add);
}

/**
 * # FUNCTION #
 * インデックスリストの開放(全て)
 */
void freeIdxs(struct Idxs *idxs){
  int i;

  for(i=0; i<idxs->cnt; i++){
    freeIdxsSub( *(idxs->idx+i),idxs->patCnt );
  }
  mssFree(idxs->idx);
  mssFree(idxs);
}

/**
 * # FUNCTION #
 * インデックス領域の確保
 */
static struct Idxs *malNextIdxs(struct Idxs *idxs, struct Data *dat){
  int i;

  /*領域確保*/
  idxs->idx=mssRealloc(idxs->idx,sizeof(usint **)*(idxs->cnt+1),"malIdxs1");
  *(idxs->idx+idxs->cnt)=mssCalloc(sizeof(usint *)*dat->patCnt,"malIdxs2");

  /*一旦もとのalpha-indexをコピーする*/
  for(i=0; i<dat->patCnt; i++){
    *(*(idxs->idx+idxs->cnt)+i)
      =mssCalloc(sizeof(usint)*((dat->pat+i)->map->alpSiz+1),"malIdxs3");
    strcpyUSI( *(*(idxs->idx+idxs->cnt)+i), (dat->pat+i)->map->idx );
  }
  return(idxs);
}

/**
 * # FUNCTION #
 * 現在のインデックスから一桁づつ変更したインデックスのリストを作成
 */
struct Idxs *setNextIdxs(struct Data *dat){
  int i,j,k;
  struct Idxs *idxs;
  int numPat;
  char chr;
  char prv;
  char nxt;

  idxs=mssCalloc(sizeof(struct Idxs),"nextIdxs1");
  idxs->cnt=0;
  idxs->patCnt=dat->patCnt;

  /*インデックスを変更する項目で回す*/
  for(i=0; i<dat->patCnt; i++){
    numPat=(dat->pat+i)->numPat; /*数値パターンかどうか*/
    for(j=0; j<(dat->pat+i)->map->alpSiz; j++){ /*alphaの長さ回す*/
      chr=*((dat->pat+i)->map->idx+j); /*対象の文字*/
      if(!numPat){
        for(k=1; k<=(dat->pat+i)->map->idxSiz; k++){ /*indexの種類回す*/
          if(k==chr) continue;
          idxs=malNextIdxs(idxs,dat);
          *(*(*(idxs->idx+idxs->cnt)+i)+j)=k;
          if( !goodIdx( *(*(idxs->idx+idxs->cnt)+i) ) ){
            freeIdxsSub( *(idxs->idx+idxs->cnt),dat->patCnt );
            continue; /*idxs->cntをカウントアップしないので登録しないと同じ*/
          }
          idxs->cnt++;
        }
      }else{
        if(j==0) prv=0;
        else     prv=*((dat->pat+i)->map->idx+j-1);
        if(j==(dat->pat+i)->map->alpSiz-1) nxt=0;
        else nxt=*((dat->pat+i)->map->idx+j+1);
        if(prv!=chr && prv!=0){
          idxs=malNextIdxs(idxs,dat);
          *(*(*(idxs->idx+idxs->cnt)+i)+j)=prv;
          if( !goodIdx( *(*(idxs->idx+idxs->cnt)+i) ) ){
            freeIdxsSub( *(idxs->idx+idxs->cnt),dat->patCnt );
            continue; /*idxs->cntをカウントアップしないので登録しないと同じ*/
          }
          idxs->cnt++;
        }
        if(nxt!=chr && nxt!=0){
          idxs=malNextIdxs(idxs,dat);
          *(*(*(idxs->idx+idxs->cnt)+i)+j)=nxt;
          if( !goodIdx( *(*(idxs->idx+idxs->cnt)+i) ) ){
            freeIdxsSub( *(idxs->idx+idxs->cnt),dat->patCnt );
            continue; /*idxs->cntをカウントアップしないので登録しないと同じ*/
          }
          idxs->cnt++;
        }
      }
    }
  }
  return(idxs);
}

/**
 * # FUNCTION #
 * 複数インデックスのdat構造体へのコピー
 */
void cpyIdx(struct Data *dat, usint **idx){
  int j;

  for(j=0; j<dat->patCnt; j++){
    strcpyUSI( (dat->pat+j)->map->idx, *(idx+j) );
  }
}

/**
 * # FUNCTION #
 * 与えられたalphabetに対応するindexを返す
 */
char alp2idx(struct Map *map, usint alpha){
  int i;
  for(i=0; i<map->alpSiz; i++){
    if(*(map->alp+i)==alpha) return (*(map->idx+i));
  }
  mssShowErrMsg("internal error2");
  exit(mssErrorNoDefault);
  return(1);
}

/**
 * # FUNCTION #
 * 与えられたalphabet文字列に対応するindex文字列をセットする
 */
void alpStr2idxStr(usint index[],usint *alpha, struct Map *map){
  int i=0;

  while(*alpha!=0){
    index[i++]=alp2idx(map,*(alpha++));
  }
  index[i]=0;
}


/**
 * # FUNCTION #
 * dat構造体のデータをインデックス化する
 */
void datIndexing( struct Data *dat ){

  usint          alpha;            /*アルファベット文字*/
  usint          index[MaxPatLen]; /*インデックス化された文字列を入れる*/
  usint         *str;              /*パターンを操作するときの一時的ポインタ*/
  int i,j,k;

  for(i=0; i<dat->patCnt; i++){ /*パターン項目数*/
    freeStrListUSI((dat->pat+i)->patIdx);
    /*index化されたデータを入れる変数初期化*/
    (dat->pat+i)->patIdx=initStrListUSI();
  }

  for(i=0; i<dat->patCnt; i++){ /*パターン項目数*/
    for(j=0; j<dat->cnt; j++){  /*データ件数*/
      k=0;                      /*インデックスの現在文字数*/
      /*i番目項目のj行目データの文字列*/
      str=getStrListUSI((dat->pat+i)->patAlp,j);
      while(*str!='\0'){ /*NULL値の場合はいきなりブレーク*/
        alpha=*(str++);
        index[k++]=alp2idx((dat->pat+i)->map,alpha);
      }
      index[k]=0; /*ターミネータ*/
      putStrListUSI( (dat->pat+i)->patIdx, index); /*パターンの格納*/
    }
  }
}

/**
 * # FUNCTION #
 * 正規パターンを属性とするテーブルを作成
 */
void mkAtt( struct Data *dat ){

  int i,j,k;

  freeAtt(dat);

  /*パターン属性領域の確保*/
  for(i=0; i<dat->patCnt; i++){
    (dat->pat+i)->attCnt = (dat->pat+i)->regTbl->cnt;

    for(j=0; j<(dat->pat+i)->attCnt; j++){
      (dat->pat+i)->att[j]= mssCalloc(sizeof(char)*dat->cnt,"AttTblAtt");
    }
  }

  /*正規表現属性*/
  for(i=0; i<dat->patCnt; i++){ /*パターン項目数*/
    for(j=0; j<dat->cnt; j++){ /*サンプルデータ件数*/
      for(k=0; k<(dat->pat+i)->attCnt; k++){ /*正規表現数*/
      *((dat->pat+i)->att[k]+j) =
         (char)regCmp( getStrListUSI((dat->pat+i)->patIdx,j),
                     (dat->pat+i)->regTbl->reg+k );
      }
    }
  }
}

/**
 * # FUNCTION #
 * 正規パターンを属性とするテーブルの領域開放
 */
void freeAtt( struct Data *dat ){
  int i,j;

  for(i=0; i<dat->patCnt; i++){
    for(j=0; j<(dat->pat+i)->attCnt; j++){
      mssFree((dat->pat+i)->att[j]);
    }
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 正規パターン関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * Regexp構造体のコピー
 */
void cpReg(struct Regexp *to, struct Regexp *from){

  strcpyUSI(to->str,from->str);
  to->type=from->type;
  to->bgnRng=from->bgnRng;
  to->endRng=from->endRng;
  to->objVal=from->objVal;
}

/**
 * # FUNCTION #
 * 文字列strの最後のn文字内で終る文字列regがあるかどうか
 */
static int atEndRng(usint *str,usint *reg,int n){
  int l=strlenUSI(str); /*strの長さ*/
  int s=strlenUSI(reg); /*regの長さ*/
  int i;
  int start; /*strの検索開始位置*/
  int end;   /*strの検索終了位置*/

  start=l-n-s+1;
  if(start<0){start=0;end=start+l-s+1;}
  else       {        end=start+n    ;}
  for(i=start; i<end; i++){
    if(0==strncmpUSI(str+i,reg,s)) return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 * substringのマッチング(1:マッチ,0:アンマッチ)
 */
int regCmpStr(usint *str,struct Regexp *reg){
  usint *bgnPos; /*最初の検索での一致位置*/

  /*substringのmatch位置を検索*/
  if( NULL == (bgnPos=strstrUSI(str,reg->str)) ) return(0);

  /*先頭一致の指定がある場合*/
  if(reg->bgnRng!=0){
    if(bgnPos-str >= reg->bgnRng) return(0);
  }

  /*末尾一致の指定がある場合*/
  if(reg->endRng!=0){
    if(0==atEndRng(str,reg->str,reg->endRng)) return(0);
  }

  /*上記の条件に引っかからなければ一致ということ*/
  return(1);
}

/**
 * # FUNCTION #
 * subsequenceのマッチング(1:マッチ,0:アンマッチ)
 */
int regCmpSeq(usint *str,struct Regexp *reg){
  usint *strPnt=str;
  usint *regPnt=reg->str;
  usint *strEnd; /*strのend位置*/
  usint *regEnd; /*reg->strのend位置*/
  int   flg;
  int   i;

  /*subsequenceとして一致するか調べる*/
  while(*regPnt!=0 && *strPnt!=0){
    if(NULL==(strPnt=strchrUSI(strPnt,*regPnt))){
      return(0);
    }
    strPnt++;
    regPnt++;
  }

  /*正規パターンの文字を読み切っていなければ*/
  if(*regPnt!=0) return(0);

  /*先頭一致の指定があれば,正規パターンの最初の文字がbgnRng内にあるか調べる*/
  if(reg->bgnRng!=0){
    flg=0;
    for(i=0; i<reg->bgnRng; i++){
      if(*reg->str == *(str+i))flg=1;
    }
    if(!flg) return(0);
  }

  /*末尾一致の指定があれば,正規パターンの最初の文字がendRng内にあるか調べる*/
  if(reg->endRng!=0){
    flg=0;
    strEnd=     str+strlenUSI(     str)-1;
    regEnd=reg->str+strlenUSI(reg->str)-1;
    for(i=0; i<reg->endRng; i++){
      if(*regEnd == *(strEnd-i))flg=1;
    }
    if(!flg) return(0);
  }

  /*上記の条件に引っかからなければ一致ということ*/
  return(1);
}

/**
 * # FUNCTION #
 * 正規パターンが与えられた文字列にマッチするかどうか(1:マッチ,0:アンマッチ)
 * substring or subsequence , end位置,start位置 を考慮
 */
int regCmp(usint *str, struct Regexp *reg){

  switch(reg->type){
    case 0: /*substring*/
      return( regCmpStr(str,reg) );
      break;
    case 1: /*subsequence*/
      return( regCmpSeq(str,reg) );
      break;
  }
  return(0);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 正規パターン候補を求める際の高速アルゴリズム
 *  algorithm by
 *    M. Hirao, H.Hoshino, A.Shinohara, M.Takeda, and S.Arikawa,
 *    "A Practical Algorithm to Find the Best Subsequence Patterns"
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * クラス別マッチ、アンマッチ件数表から、upperBound を求める
 * upperBound : 対象となる正規パターンを拡張して得られる最良のsv
 */
static double calUpperBound( struct CndCnt *cnt){

  double svMin, sv; /*value for splitting criteria*/
  int x,y,xMax,yMax;

  x=cnt->mmCnt[0];
  y=cnt->mmCnt[1];
  xMax=cnt->mmCnt[0]+cnt->uuCnt[0];
  yMax=cnt->mmCnt[1]+cnt->uuCnt[1];

  /*f(0,y)*/
  cnt->mmCnt[0] = 0; cnt->uuCnt[0] = xMax-cnt->mmCnt[0];
  cnt->mmCnt[1] = y; cnt->uuCnt[1] = yMax-cnt->mmCnt[1];
  calCndCnt(cnt);
  calCndCntSplit(cnt);
  svMin=cnt->splitAfter;

  /*f(x,0)*/
  cnt->mmCnt[0] = x; cnt->uuCnt[0] = xMax-cnt->mmCnt[0];
  cnt->mmCnt[1] = 0; cnt->uuCnt[1] = yMax-cnt->mmCnt[1];
  calCndCnt(cnt);
  calCndCntSplit(cnt);
  sv=cnt->splitAfter;
  if(svMin>sv) svMin=sv;

  /*f(0,0)*/
  cnt->mmCnt[0] = 0; cnt->uuCnt[0] = xMax-cnt->mmCnt[0];
  cnt->mmCnt[1] = 0; cnt->uuCnt[1] = yMax-cnt->mmCnt[1];
  calCndCnt(cnt);
  calCndCntSplit(cnt);
  sv=cnt->splitAfter;
  if(svMin>sv) svMin=sv;

  /*f(x,y)*/
  cnt->mmCnt[0] = x; cnt->uuCnt[0] = xMax-cnt->mmCnt[0];
  cnt->mmCnt[1] = y; cnt->uuCnt[1] = yMax-cnt->mmCnt[1];
  calCndCnt(cnt);
  calCndCntSplit(cnt);
  sv=cnt->splitAfter;
  if(svMin>sv) svMin=sv;

  return(svMin);
}

/**
 * # FUNCTION #
 * 正規パターンのデータに対するGINIおよびupperBoundを求める
 * 正規パターンが一件もマッチしなければ1,そうでなければ0を返す
 */
static int regCnt(
  struct PQkey   *pqKey,     /*正規パターン*/
  struct StrListUSI  *patIdx,/*index化されたtraining dataのパターン項目データ*/
  struct Class   *cls,       /*クラス項目データ*/
  struct Cost    *cost){     /*コストデータ*/

  struct CndCnt  cnt; /*クラス別マッチ、アンマッチ件数表*/

  int i;

  /*cntの初期化*/
  iniCndCnt(&cnt,cost);

  /*データ行全てを正規パターンに当てはめ、CndCnt構造体に件数をセットする*/
  for(i=0; i<patIdx->lineCnt; i++){

    /*その行がマッチしていれば*/
    if(regCmp(getStrListUSI(patIdx,i),&pqKey->reg)){
      cnt.mCnt[(int)*(cls->chr+i)]+=1;

    /*その行がアンマッチならば*/
    }else{
      cnt.uCnt[(int)*(cls->chr+i)]+=1;
    }
  }

  /*件数集計*/
  calCndCnt(&cnt);

  /*Giniの計算*/
  calCndCntSplit(&cnt);
  pqKey->reg.objVal = cnt.splitAfter;

  /*upperBoundの計算*/
  pqKey->upperBound = calUpperBound(&cnt);

  if(cnt.mtCnt==0) return(1);
  else             return(0);
}

/**
 * # FUNCTION #
 * 正規パターンをベストキューに登録
 */
static void setRegBest(struct PQkey *pqKey, struct RegBest *regBest){
  struct PQnode *worst;

  worst=PQworstMember(regBest->pqNode->left);

  /* まだ一件も登録されていない、もしくは最大数に達していない場合 */
  /* 単純に登録するだけ */
  if( worst == NULL || regBest->cnt <regBest->maxCnt){
    PQinsert(pqKey, regBest->pqNode->left);
    regBest->cnt++;

  /* worstのsvより良ければ、worstを削除して新たに登録 */
  } else if ( pqKey->reg.objVal < regBest->maxVal ) {
    PQdeleteNode(worst);
    PQinsert(pqKey, regBest->pqNode->left);

    /*regBest->maxValの更新(upperboundとの比較などで利用)*/
    worst=PQworstMember(regBest->pqNode->left);
    regBest->maxVal=worst->key.reg.objVal;
  }
}

static int RegPQ2Tbl_Cnt;

/**
 * # FUNCTION #
 * キューregBestを再帰的に探りテーブルに登録
 */
static void regPQ2TblSub(
  struct PQnode *pqNode,
  struct RegTbl *regTbl){

  /*リーフ以外のノードは再帰的に呼び出される*/
  if(!PQisExternalNode(pqNode)){

    /*小さい方(left)から手繰る*/
    regPQ2TblSub(pqNode->left,regTbl);

    /*キューのノードをtableにコピーする*/
    cpReg(regTbl->reg+RegPQ2Tbl_Cnt,&pqNode->key.reg);

    /*tableの件数カウントアップ*/
    RegPQ2Tbl_Cnt++;

    /*大きい方(right)を手繰る*/
    regPQ2TblSub(pqNode->right,regTbl);
  }
}

/**
 * # FUNCTION #
 * キューregBestに登録された正規パターン候補をテーブルに変換する
 */
static struct RegTbl *regPQ2Tbl( struct RegBest *regBest ){

  struct RegTbl *regTbl;

  /*領域確保*/
  regTbl=mssMalloc(sizeof(struct RegTbl), "regPQ2Tbl");
  regTbl->reg=mssCalloc(sizeof(struct Regexp) * regBest->maxCnt, "regPQ2Tbl");

  RegPQ2Tbl_Cnt=0;
  regPQ2TblSub(regBest->pqNode->left,regTbl);
  regTbl->cnt=RegPQ2Tbl_Cnt;
  return(regTbl);
}


/**
 * # FUNCTION #
 *禁止リストに登録されているかどうかを確かめる
 *  reg->regStr="abcd"のとき
 *                "bcd","cd","d"  が禁止リストにあるか調べる
 */
static int isForbid(struct Regexp *reg,struct cell **forbidden){
  int i;
  usint *pos;

  /*先頭一致と末尾一致の場合は判断しない(禁止リストにないものとみなす)*/
  if(reg->bgnRng!=0 || reg->bgnRng!=0) return(0);

  pos=reg->str;
  for(i=1; i<strlenUSI(reg->str); i++){
    pos++;  /* 次の文字へ */

    if(EHmember(pos,forbidden)) return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 * 見込みのない正規パターンをforbidden listに登録
 * 登録する文字列は正規表現に変換した文字列(Regexp->regStr)
 */
static void putForbid(struct Regexp *reg,struct cell **forbidden){

  /*先頭一致と末尾一致の場合は禁止リストに入れない*/
  if(reg->bgnRng!=0 || reg->bgnRng!=0) return;
  /*外部ハッシュリストに登録*/
  EHinsert(reg->str,forbidden);
}

/**
 * # FUNCTION #
 * 対象となる正規パターン(pqKey)について以下のテスト上から順に行い
 * pqTop,regBest,ForbiddenListに登録する
 * 1. 部分パターンがForbiddenListにある               -> 何もしない  &return
 * 2. 正規パターンが一件もマッチしない                -> ForbidenList&return
 * 3. 正規パターンがregBestに入賞した                 -> regBest&pqTop&return
 * 4. 正規パターンのUpperBoundがregBestに入賞できない -> ForbidenList&return
 * 5. 正規パターンの長さが最大長に達した              -> 何もしない&return
 * 6. 上記条件に当てはまらない                        -> pqTop&return
 */
static void putPFB(
  struct PQkey *pqKey,       /*対象となる正規パターン*/
  struct cell  *forbidden[], /*探索禁止リスト*/
  struct Data *dat,
  struct Cost *cost,
  struct Pattern *pat,
  struct PQnode *pqTop,      /*探索対象リスト*/
  struct RegBest *regBest){  /*ベストnリスト*/
  int noMatch; /*正規パターンが一件もマッチしないフラグ*/
  double upperBound; /*fobidden listに入れるかどうかの計算用*/

  /*(1) 正規パターンの部分パターンがforbiddenリストにあればreturn*/
  if(isForbid(&pqKey->reg,forbidden)){
    return;
  }

  /*split rule,upperboundを計算*/
  noMatch=regCnt(pqKey,pat->patIdx,dat->cls,cost);

  /*(2) 一件もマッチしなければforbidenリストに登録&return*/
  if(noMatch){
    putForbid(&pqKey->reg,forbidden);
    return;
  }

  /*(3) 現在のBest nに入るならばregBestにセット*/
  if(pqKey->reg.objVal < regBest->maxVal){
    setRegBest(pqKey, regBest);
    if(strlenUSI(pqKey->reg.str)<MaxRegLen){
      PQinsert(pqKey,pqTop->left);
    }
    return;
  }

  /*upperBoundの計算はclassSizeが2の時のみ有効なので            */
  /*classSizeが2より大きい場合はupperBoundの計算が成り立たない  */
  /*ので、最も良いsvをupperBoundとして入れておく                */
  /*すなわちupperBoundによるforbiden登録のアルゴリズムを使わない*/
  if(ClassSize==2) upperBound=pqKey->upperBound;
  else             upperBound=0;

  /*(4) 正規パターンのUpperBoundがregBestに入賞できない*/
  if(upperBound > regBest->maxVal){
    putForbid(&pqKey->reg,forbidden);
    return;
  }

  /*(5) 正規パターンの長さが最大長に達した*/
  if(strlenUSI(pqKey->reg.str)>=MaxRegLen){
    return;
  }

  /*(6)上記条件に当てはまらない->キューに入れて、次回に備える*/
  PQinsert(pqKey,pqTop->left);
}

/**
 * # FUNCTION #
 *文字列(str)に一文字(chr)追加
 */
static void addStrChr(usint *str,usint chr){
  int len;

  len=strlenUSI(str);
  *(str+len)=chr;
  *(str+len+1)=0;
}

/**
 * # FUNCTION #
 * 正規パターン候補を生成し (dat->pat+patNo)->regTbl にセットする
 */
void setRegTbl(struct Data *dat, struct Cost *cost){

  struct Regexp reg;
  double upperBound;
  struct PQnode *pqTop;
  struct PQkey   pqKey;
  int i,j;
  struct RegBest regBest;
  struct cell *forbidden[EHbucket];
  int patNo;
  struct Pattern *pat;

  freeRegTbl(dat);
  for(patNo=0; patNo<dat->patCnt; patNo++){
    pat=dat->pat+patNo;

    /*hash のbucketを初期化*/
    for(j=0; j<EHbucket; j++) forbidden[j]=NULL;

    /*Best n個の正規表現が入る構造体の初期化*/
    /*モデル生成に利用する正規パターン候補が格納される*/
    regBest.pqNode               = PQmakeNode();
    regBest.pqNode->parent       = regBest.pqNode;
    regBest.pqNode->right        = NULL;
    regBest.pqNode->rank         = 0;
    regBest.pqNode->left         = PQmakeNode();
    regBest.pqNode->left->parent = regBest.pqNode;
    regBest.cnt                  = 0;
    regBest.maxVal               = 9999;
    regBest.maxCnt               = optCAN.val;

    /*priority queueの初期化*/
    /*有望な正規パターンを一時的に格納する*/
    /*このキューが空になるまで正規パターン候補を探し続ける*/
    pqTop               = PQmakeNode();
    pqTop->parent       = pqTop;
    pqTop->right        = NULL;
    pqTop->rank         = 0;
    pqTop->left         = PQmakeNode();
    pqTop->left->parent = pqTop;

    /*-------------------------------------------------------------*/
    /* メインルーチン                                              */
    /* pqTop   : 候補となる正規表現を入れておくpriority queue      */
    /* regBest : 候補となるn個の正規patternを、svの低い順に蓄える  */
    /* fobidden: 改善の見込みのない正規patternを蓄えておくハッシュ */
    /*-------------------------------------------------------------*/

    /*まず一文字だけの正規パターンを初期値としてpqTopに格納する*/
    for(i=0; i<pat->map->idxSiz; i++){
      /*通常の正規パターン*/
      pqKey.reg.str[0]=i+1;
      pqKey.reg.str[1]=0;
      pqKey.reg.bgnRng=0;
      pqKey.reg.endRng=0;
      pqKey.reg.type  =optSEQ.set;
      putPFB(&pqKey,forbidden,dat,cost,pat,pqTop,&regBest);

      /*先頭一致*/
      for(j=1; j<=optBGN.val; j++){
        pqKey.reg.bgnRng=j;
        putPFB(&pqKey,forbidden,dat,cost,pat,pqTop,&regBest);
      }
      pqKey.reg.bgnRng=0;

      /*末尾一致*/
      for(j=1; j<=optEND.val; j++){
        pqKey.reg.endRng=j;
        putPFB(&pqKey,forbidden,dat,cost,pat,pqTop,&regBest);
      }
      pqKey.reg.endRng=0;

      /*先頭、末尾両一致*/
      /*今回は実装を見送る
      for(j=1; j<=optBGN.val; j++){
        for(k=1; k<=optEND.val; k++){
          pqKey.reg.bgnRng=j;
          pqKey.reg.endRng=k;
          putPFB(&pqKey,forbidden,dat,cost,pat,pqTop,&regBest);
        }
      }
      */
    }
    pqKey.reg.bgnRng=0;
    pqKey.reg.endRng=0;

    /* priority queue(pqTop)が空になるまでループする */
    while(PQisEmpty(pqTop->left) != 1){
      /*priority queueより最優先の正規パターンを取り出し*/
      /*regとupperBoundをセット*/
      PQpop(&reg,&upperBound,pqTop->left);
      /*popしたUBがベストn以下のときだけ探索*/
      if(upperBound <= regBest.maxVal){

        /*取り出した正規表現に一文字追加*/
        for(i=0; i<pat->map->idxSiz; i++){
          cpReg(&pqKey.reg,&reg);
          addStrChr(pqKey.reg.str,i+1);
          putPFB(&pqKey,forbidden,dat,cost,pat,pqTop,&regBest);
        }
      }
    }
  
    /*プライオリティキューをテーブルに変換*/
    (dat->pat+patNo)->regTbl=regPQ2Tbl(&regBest);

    /*領域開放*/
    PQfree(pqTop);
    PQfree(regBest.pqNode);
    EHfree(forbidden);

  }/* for patNo */
}

/**
 * # FUNCTION #
 * 正規パターン候補の領域開放
 */
void freeRegTbl(struct Data *dat){
  int i;
  for(i=0; i<dat->patCnt; i++){
    if( (dat->pat+i)->regTbl == NULL) return;
    mssFree((dat->pat+i)->regTbl->reg);
    mssFree((dat->pat+i)->regTbl);
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 出力系
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 正規パターンの表示
 */
void prnRegexp(struct Regexp *reg, struct mssFPW *fpw){
  usint *str;

  if(reg->type==0) mssWriteStr("substring ",fpw);
  else             mssWriteStr("subsequence ",fpw);

  mssWriteStr("\"",fpw);

  /*先頭一致*/
  if(reg->bgnRng!=0){
    mssWriteStr("^[",fpw);
    mssWriteInt(reg->bgnRng,fpw);
    mssWriteStr("]",fpw);
  }

  str=reg->str;
  while(*str!=0){
    mssWriteInt(*str,fpw);
    if(*(str+1)!=0) mssWriteStr(" ",fpw);
    str++;
  }

  /*末尾一致*/
  if(reg->endRng!=0){
    mssWriteStr("$[",fpw);
    mssWriteInt(reg->endRng,fpw);
    mssWriteStr("]",fpw);
  }

  mssWriteStr("\"",fpw);
}
