/**
 * # CHAPTER #
 * ============================================================================
 * xtclassifyで用いられるデータ読み込み関連の関数
 * ============================================================================
 */

#include <musashi.h>
#include <readData.h>
#include <bonsai.h>
#include <string.h>
#include <iconv.h>
#include <libxml/xmlerror.h>

/* ############ グローバル変数 ##############*/
/*パラーメータ*/
extern MssOptFLD optPAT;
extern MssOptFLD optCAT;
extern MssOptSTR optDLM;
extern MssOptINT optSIZ;

extern struct mssFields *fpat; /*パターン項目*/
extern struct mssFields *fnum; /*数値項目*/
extern struct mssFields *fcat; /*カテゴリ項目*/
extern struct mssFields *fcls; /*クラス項目*/

extern int ClassSize;          /*クラスのサイズ*/
//extern iconv_t *icid;          /*Encoding*/
extern char mssXmlDomErrMsg[]; /*コスト読み込み時のエラーメッセージ*/

/**
 * # FUNCTION #
 * データ件数をカウントする関数
 */
int countDat(struct mssFPR *fpr){
  struct mssRec *rec=NULL; /*行バッファ構造体*/
  int i=0;

  rec=mssInitRec();
  mssSeekTopFPR(fpr);
  while(EOF != mssReadRec(fpr,rec)) i++;
  mssFreeRec(rec);

  return(i);
}

/**
 * # FUNCTION #
 * クラスの文字(str)から、その配列番号を返す関数
 */
static char getClassNum(struct Class *cls, char *str){
  int i;

  for(i=0; i<ClassSize; i++){
    if(0==strcmp(cls->str[i],str)) return(i);
  }
  return(-1);
}

/**
 * # FUNCTION #
 * データ上で使われているクラスの値をセットする
 * ここでは、トレーニングとテストを両方走査し、ありうるクラスの値を得る
 */
struct Class *setClass(struct mssFPR *trn, struct mssFPR *tst, struct mssHeader *hd){

  struct Class *cls;     /*クラス文字列-番号構造体*/
  struct mssFldRec *fr;  /*項目対応のレコードバッファ*/
  char *str;
  int flg;
  int i;
  int recCnt;

  cls=mssCalloc(sizeof(struct Class), "setClass");
  cls->cnt=0;
  cls->chr=NULL;

  /*トレーニングのクラスを走査*/
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(trn);
  recCnt=0;
  while( EOF != mssReadFldRec(trn,fr) ){

    /*クラス項目の値をセット*/
    str=*(fr->pnt+MssFlds2num(fcls,0));

    if( MssIsNull(str) ){
      mssShowErrMsg("class field cannot have a NULL value");
      mssEnd(mssErrorNoDefault);
    }

    /*登録されているクラス項目の文字列を検索。なければflg=1*/
    flg=1;
    for(i=0; i<cls->cnt; i++){
      if(0==strcmp(cls->str[i],str)){ flg=0; break; }
    }

    /*登録*/
    if(flg){
      cls->str[cls->cnt]=mssMalloc(sizeof(char)*(strlen(str)+1),"setClass");
      strcpy(cls->str[cls->cnt],str);
      cls->cnt++;
    }
  }
  mssFreeFldRec(fr);

  /*テストのクラスを走査*/
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(tst);
  recCnt=0;
  while( EOF != mssReadFldRec(tst,fr) ){

    /*クラス項目の値をセット*/
    str=*(fr->pnt+MssFlds2num(fcls,0));

    /*登録されているクラス項目の文字列を検索。なければflg=1*/
    flg=1;
    for(i=0; i<cls->cnt; i++){
      if(0==strcmp(cls->str[i],str)){ flg=0; break; }
    }

    /*登録*/
    if(flg){
      cls->str[cls->cnt]=mssMalloc(sizeof(char)*(strlen(str)+1),"setClass");
      strcpy(cls->str[cls->cnt],str);
      cls->cnt++;
    }
  }
  mssFreeFldRec(fr);

  /* クラスの表示(for debug) */
  /*
  for(i=0; i<cls->cnt; i++){
    printf("cls%d = %s\n",i,cls->str[i]);
  }
  */
  return(cls);
}

/**
 * # FUNCTION #
 * Class構造体領域の開放
 */
void freeClass(struct Class *cls){
  int i;
  for(i=0; i<cls->cnt; i++){
    mssFree(cls->str[i]);
  }
  mssFree(cls);
}

/**
 * # FUNCTION #
 * コストファイルの読み込み
 *
 * <?xml version="1.0" encoding="euc-jp"?>
 * <missClassificationCost>
 *   <cost class="優良" predict="非優良" value="9"/>
 *   <cost class="非優良" predict="優良" value="2.1"/>
 * </missClassificationCost>
 *
 * 優良のケースが非優良と予測されたときのコストは9。
 * 非優良のケースが優良と予測されたときのコストは2.1。
 *
 * デフォルトのコスト
 * 同じ値を同じ値と予測したときのコストは0。
 * ある値を違う値と予測したときのコストは1。
 */
struct Cost *readCost(char *fname,struct Class *cls){
        xmlDocPtr doc;
        xmlNodePtr cur;
        xmlChar *att1,*att2,*att3;
        iconv_t *icid;
  int clsNo;
  int prdNo;

  struct Cost *cost;

  int i,j;

  /*コストをデフォルト値で作成*/
  cost=mssCalloc(sizeof(struct Cost),"readCost");
  for(i=0; i<ClassSize; i++){
    for(j=0; j<ClassSize; j++){
      if(i==j) cost->tbl[i][j]=0;
      else     cost->tbl[i][j]=1;
    }
    cost->actTtl[i] = ClassSize-1;
  }
  /*Classをセット*/
  cost->cls=cls;

  /*ファイル名が指定されていなければデフォルトコストを返す*/
  if(fname==NULL) return(cost);

  /*エラー関数の設定*/
  mssXmlDomErrMsg[0]='\0';
  xmlSetGenericErrorFunc(NULL,mssXmlDomErrHandler);

  /*コストファイルのparse*/
  doc = xmlParseFile(fname);

  /*encodingの設定*/
  icid=iconv_open(doc->encoding,"UTF-8");

  if (doc == NULL ) {
    mssShowErrMsg(mssEncoding(mssXmlDomErrMsg,icid));
    exit(mssErrorNoDefault);
  }

  /*トップノードのセット*/
  cur = xmlDocGetRootElement(doc);

  /*空ドキュメントチェック*/
  if (cur == NULL) {
    xmlFreeDoc(doc);
    mssShowErrMsg("empty document");
    exit(mssErrorNoDefault);
  }

  /*ルートタグ名(missClassificationCost)チェック*/
  if (xmlStrcmp(cur->name, (const xmlChar *) "missClassificationCost")) {
    xmlFreeDoc(doc);
    mssShowErrMsg("cost file: root tag must be '<missClassificationCost>'");
    exit(mssErrorNoDefault);
  }


  /*<cost>要素を全て読み込む*/
  cur = cur->xmlChildrenNode;
  while (cur != NULL) {
    if((0==xmlStrcmp(cur->name, (const xmlChar *)"cost"))){
      att1 = xmlGetProp(cur,"class");
      att2 = xmlGetProp(cur,"predict");
      att3 = xmlGetProp(cur,"value");

      clsNo=getClassNum(cls,mssEncoding(att1,icid));
      prdNo=getClassNum(cls,mssEncoding(att2,icid));
      if(clsNo<0 || prdNo<0){
        mssShowErrMsg("cost file: invalid value on class or predict attribute");
        exit(mssErrorNoDefault);
      }
      cost->tbl[clsNo][prdNo] = atof(att3);
      xmlFree(att1);
      xmlFree(att2);
      xmlFree(att3);
    }
    cur = cur->next;
  }
  xmlFreeDoc(doc);

  /*actClassのトータルコストを計算*/
  for(i=0; i<ClassSize; i++){
    for(j=0; j<ClassSize; j++){
      cost->actTtl[i]+=cost->tbl[i][j];
    }
  }

  return(cost);
}


/**
 * # FUNCTION #
 * コストデータの表示
 */
void showCost(struct Cost *cost, struct mssFPW *fpw){
  int i,j;

  if(cost==NULL) return;

  mssWriteStr("[Cost]\n",fpw);

  for(i=0; i<ClassSize; i++){
    for(j=0; j<ClassSize; j++){
      if(i==j) continue;
      mssWriteStr("If class \"",fpw);
      mssWriteStr(cost->cls->str[i],fpw);
      mssWriteStr("\" is predicted as \"",fpw);
      mssWriteStr(cost->cls->str[j],fpw);
      mssWriteStr("\", then cost=",fpw);
      mssWriteDbl(cost->tbl[i][j],fpw);
      mssWriteRet(fpw);
    }
  }
}


/**
 * # FUNCTION #
 * 数値パターンの場合、アルファベットを元の数値によって並べ換える
 */
static void sortAlp(struct Map *map){
  struct mssHashNode *node;
  int i,j;
  int n;
  char *x;
  char **a;

  a=map->numAlp;
  n=map->alpSiz;

  for(i=0; i<n; i++){
    x=a[i];
    for(j=i-1; j>=0 && atof(a[j])>atof(x); j--){
      a[j+1]=a[j];
    }
    a[j+1]=x;
  }

  for(i=0; i<n; i++){
    node=(struct mssHashNode *)mssHashMember(map->alpOrg,a[i]);
    node->val.v.usi=(usint)(i+1);
  }
}

/**
 * # FUNCTION #
 * alphabet内部番号→alphabetの変換の為の配列を登録。
 * alpOrgハッシュ表のに登録された文字列へのポインタ配列。
 */
static char **setNumAlp( struct Map *map ){
  struct mssHashNode *node;
  char **tbl;
  int i;

  tbl=mssMalloc(sizeof(char *)*map->alpSiz,"setNumAlp");

  for(i=0; i<map->alpOrg->hashVal; i++){
    if( NULL != (node=*(map->alpOrg->node+i)) ){
      while(node!=NULL){
        *(tbl+node->val.v.usi-1)=node->str;
        node=node->next;
      }
    }
  }
  return(tbl);
}

/**
 * # FUNCTION #
 * パターン項目をデリミタ(dlm)でトークン分割し、その文字列配列を返す。
 * デリミタが'\0'の場合は、一バイトを一alphabetとしてセットする。
 */
static char **tokByChr(char *str,char dlm,int *cnt){
  char **list;
  char *buf;
  int i;
  int len;

  if(dlm=='\0'){
    len=strlen(str);
    buf=mssMalloc(sizeof(char)*len*2,"tokByChr");
    list=mssMalloc(sizeof(char *)*len,"tokByChr");
    for(i=0; i<len; i++){
      *(buf+i*2  )=*(str+i);
      *(buf+i*2+1)='\0';
      *(list+i)=buf+i*2;
    }
    *cnt=len;
  }else{
    list=mssTokByChr(str,dlm,cnt,1);
  }
  return(list);
}

/**
 * # FUNCTION #
 * データ上で使われているパターンの値を調べMap構造体にセットする
 * ここでは、トレーニングとテストを両方走査し、ありうるalphabetを得る
 * NULL値は飛ばす
 */
struct Map **setMapAlp(
  struct mssFPR *trn,
  struct mssFPR *tst,
  struct mssHeader *hd){

  struct mssFldRec *fr;  /*項目対応のレコードバッファ*/
  char         *str; /*パターンを操作するときの一時的ポインタ*/
  int i,j;
  struct Map **map;
  struct Map *mapi;
  MssValue *alpNum;
  char **list;
  int cnt;

  if(fpat->cnt==0)return(NULL);

  map=mssMalloc(sizeof(struct Map *)*fpat->cnt,"setMapAlp");
  for(i=0; i<fpat->cnt; i++){
    *(map+i)=mssCalloc(sizeof(struct Map),"setMapAlp");
    (*(map+i))->alpOrg=mssInitHash(1001);
    (*(map+i))->alp=NULL;
    (*(map+i))->idx=NULL;
    (*(map+i))->alpSiz=0;
  }

  alpNum=mssMalloc(sizeof(MssValue)*fpat->cnt,"setMapAlp");
  for(i=0; i<fpat->cnt; i++){
    mssVinit(alpNum+i,USI);
    (alpNum+i)->v.usi=1;
  }

  /*トレーニングのアルファベットを走査*/
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(trn);
  while( EOF != mssReadFldRec(trn,fr) ){
    for(i=0; i<fpat->cnt; i++){
      str=*(fr->pnt+MssFlds2num(fpat,i));
      if(MssIsNull(str)) continue;
      list=tokByChr(str,*optDLM.str,&cnt);
      mapi=*(map+i);
      for(j=0;j<cnt;j++){
        if( NULL != mssHashInsert((*(map+i))->alpOrg, *(list+j), *(alpNum+i)) ){
          mapi->alpSiz++;
          mapi->alp=mssRealloc(mapi->alp,
                      sizeof(usint)*mapi->alpSiz,"setMapAlp");
          *(mapi->alp+mapi->alpSiz-1)=(alpNum+i)->v.usi;
          (alpNum+i)->v.usi++;
        }
      }
      mssFree(*list); mssFree(list);
    }
  }
  mssFreeFldRec(fr);

  /*テストのアルファベットを走査*/
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(tst);
  while( EOF != mssReadFldRec(tst,fr) ){
    for(i=0; i<fpat->cnt; i++){
      str=*(fr->pnt+MssFlds2num(fpat,i));
      if(MssIsNull(str)) continue;
      list=tokByChr(str,*optDLM.str,&cnt);
      mapi=*(map+i);
      for(j=0;j<cnt;j++){
        if( NULL != mssHashInsert((*(map+i))->alpOrg, *(list+j), *(alpNum+i)) ){
          mapi->alpSiz++;
          mapi->alp=mssRealloc(mapi->alp,
                      sizeof(usint)*mapi->alpSiz,"setMapAlp");
          *(mapi->alp+mapi->alpSiz-1)=(alpNum+i)->v.usi;
          (alpNum+i)->v.usi++;
        }
      }
      mssFree(*list); mssFree(list);
    }
  }
  mssFreeFldRec(fr);

  for(i=0; i<fpat->cnt; i++){
    /*インデックスの領域確保*/
    (*(map+i))->idx=mssCalloc(sizeof(usint)*((*(map+i))->alpSiz+1), "setMapAlp");

    /*アルファベットの末尾にターミネータ追加*/
    (*(map+i))->alp=
      mssRealloc((*(map+i))->alp,
        sizeof(usint)*((*(map+i))->alpSiz+1),"setMapAlp");
    *((*(map+i))->alp+(*(map+i))->alpSiz)=0;

    /*hashに登録されたalphabet(文字列->内部数値数値)を、内部数値を要素とした
      配列に登録する*/
    (*(map+i))->numAlp=setNumAlp( *(map+i) );

    /*数値パターンの場合、アルファベットを数値順に並べ換える*/
    if(mssIsFldOptOn(&optPAT,0,'n')){
      sortAlp(*(map+i));
    }

    /*インデックスサイズの調整*/
    if(optSIZ.val>(*(map+i))->alpSiz) (*(map+i))->idxSiz=(*(map+i))->alpSiz;
    else                              (*(map+i))->idxSiz=optSIZ.val;
  }

  mssFree(alpNum);

  return(map);
}

/**
 * # FUNCTION #
 * Map構造体の領域開放
 */
void freeMap(struct Map **map){
  int i;
  if(map==NULL)return;
  for(i=0; i<fpat->cnt; i++){
    mssFreeHash((*(map+i))->alpOrg);
    mssFree((*(map+i))->alp);
    mssFree((*(map+i))->idx);
    mssFree((*(map+i))->numAlp);
    mssFree(*(map+i));
  }
  mssFree(map);
}

/**
 * # FUNCTION #
 * データ上で使われているカテゴリー属性の値を調べhash表にセットする
 * ここでは、トレーニングとテストを両方走査し、ありうる値をセットする
 * NULL値は飛ばす
 */
struct Category *setCatHash(
  struct mssFPR *trn,
  struct mssFPR *tst,
  struct mssHeader *hd){

  struct Category *cat;
  struct Category *cat_i;
  struct mssFldRec *fr;  /*項目対応のレコードバッファ*/
  int i;
  char *fldStr;
  MssValue val;

  mssVinit(&val,INT);

  /*カテゴリ属性の指定が無ければNULLを返す*/
  if(!optCAT.set) return(NULL);

  cat = mssMalloc(sizeof(struct Category)*fcat->cnt,"setCatHash");
  for(i=0; i<fcat->cnt; i++){
   (cat+i)->val     = NULL; /*ここではvalはセットせず、setDatでセット*/
   (cat+i)->name    = mssInitHash(101);
   (cat+i)->cnt     = 0;
   (cat+i)->valName = NULL;
  }

  /*トレーニングのcategory属性を走査*/
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(trn);
  while( EOF != mssReadFldRec(trn,fr) ){
    for(i=0; i<fcat->cnt; i++){
      cat_i=cat+i;
      fldStr=*(fr->pnt+MssFlds2num(fcat,i));

      /*値をhashテーブルにセット*/
      mssVclear(&val);
      val.v.i=cat_i->cnt;
      /*新たにhashテーブルに挿入されたなら,内部数値-値(逆引)表を登録&cnt++*/
      if(NULL != mssHashInsert(cat_i->name,fldStr,val) ){
        cat_i->valName=mssRealloc(cat_i->valName,
          sizeof(char *)*(cat_i->cnt+1),"setCatHash2");
        *(cat_i->valName+cat_i->cnt)=mssHashMemberAdd(cat_i->name,fldStr);
        cat_i->cnt++;
      }
    }
  }
  mssFreeFldRec(fr);

  /*テストのcategory属性を走査*/
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(tst);
  while( EOF != mssReadFldRec(tst,fr) ){
    for(i=0; i<fcat->cnt; i++){
      cat_i=cat+i;
      fldStr=*(fr->pnt+MssFlds2num(fcat,i));

      /*値をhashテーブルにセット*/
      mssVclear(&val);
      val.v.i=cat_i->cnt;
      /*新たにhashテーブルに挿入されたなら,内部数値-値(逆引)表を登録&cnt++*/
      if(NULL != mssHashInsert(cat_i->name,fldStr,val) ){
        cat_i->valName=mssRealloc(cat_i->valName,
          sizeof(char *)*(cat_i->cnt+1),"setCatHash2");
        *(cat_i->valName+cat_i->cnt)=mssHashMemberAdd(cat_i->name,fldStr);
        cat_i->cnt++;
      }
    }
  }
  mssFreeFldRec(fr);


  return(cat);
}

/**
 * # FUNCTION #
 * データをセットする
 */
struct Data *setDat(
  struct mssFPR *fpr,
  struct mssHeader *hd,
  struct Class  *cls,   /*クラスの文字列-数字のみ*/
  int            cnt,   /*データ件数*/
  struct Map   **map,   /*各パターン属性のマップ*/
  struct Category *cat){

  struct mssFldRec *fr;  /*項目対応のレコードバッファ*/
  int i,j,k;
  struct Data *dat;
  char *fldStr;
  MssValue val;
  char **list;
  int    listCnt;
  struct mssHashNode *node;
  usint alpha[MaxPatLen];

  mssVinit(&val,INT);
  dat=mssCalloc(sizeof(struct Data),"setDat");

  /*クラス領域の確保と、クラスの文字列-数字のコピー*/
  dat->cls=mssCalloc(sizeof(struct Class),"setDat");
  for(i=0; i<cls->cnt; i++) dat->cls->str[i]=cls->str[i];
  dat->cls->cnt=cls->cnt;
  dat->cls->chr=mssCalloc(sizeof(char)*cnt,"setDat1");

  /*数値属性領域の確保*/
  dat->numCnt=fnum->cnt;
  dat->num    =mssCalloc(sizeof(double *)*dat->numCnt,"setDat2");
  dat->numNull=mssCalloc(sizeof(char   *)*dat->numCnt,"setDat2");
  for(i=0; i<dat->numCnt; i++){
    *(dat->num+i)     = mssCalloc(sizeof(double)*cnt,"setDat3" );
    *(dat->numNull+i) = mssCalloc(sizeof(char  )*cnt,"setDat3" );
  }

  /*カテゴリー属性データ領域の確保*/
  dat->catCnt=fcat->cnt;
  dat->cat    =mssCalloc(sizeof(struct Category)*dat->catCnt,"setDat5" );
  dat->catNull=mssCalloc(sizeof(char          *)*dat->catCnt,"setDat5");
  for(i=0; i<dat->catCnt; i++){
    (dat->cat+i)->name   =(cat+i)->name;
    (dat->cat+i)->valName=(cat+i)->valName;
    (dat->cat+i)->cnt    =(cat+i)->cnt;
    (dat->cat+i)->val = mssCalloc(sizeof(int)*cnt,"setDat5" );
   *(dat->catNull+i)  = mssCalloc(sizeof(char  )*cnt,"setDat5" );
  }

  /*パターン属性領域の確保とmapのコピー*/
  dat->patCnt=fpat->cnt;
  dat->pat    =mssCalloc(sizeof(struct Pattern)*dat->patCnt ,"setDat6" );
  dat->patNull=mssCalloc(sizeof(char         *)*dat->patCnt ,"setDat6" );
  for(i=0; i<dat->patCnt; i++){
    /*mapのセット*/
    (dat->pat+i)->map=*(map+i);
    /*数値パターンであるかどうかのセット*/
    (dat->pat+i)->numPat=mssIsFldOptOn(&optPAT,0,'n');
    /*実データ領域の初期化*/
    (dat->pat+i)->patAlp=initStrListUSI();
    (dat->pat+i)->patIdx=initStrListUSI();
    /*インデックスサイズの決定*/
    if(optSIZ.val > (*(map+i))->alpSiz || optSIZ.val==0 )
      (dat->pat+i)->map->idxSiz=(*(map+i))->alpSiz;
    else
      (dat->pat+i)->map->idxSiz=optSIZ.val;
   *(dat->patNull+i)  = mssCalloc(sizeof(char  )*cnt,"setDat6" );

  }    

  /*実データをメモリに格納*/
  i=0;
  fr=mssInitFldRec(hd->flds->cnt);
  mssSeekTopFPR(fpr);
  while( EOF != mssReadFldRec(fpr,fr) ){

    /*クラス項目の値をセット*/
    /*クラス項目がNULLであることはない(setClassでチェック済)*/
    *(dat->cls->chr+i)=getClassNum(cls,*(fr->pnt+MssFlds2num(fcls,0)));

    /*数値属性の格納*/
    for(j=0; j<fnum->cnt; j++){

      /*項目の値(文字列)のアドレスをセット*/
      fldStr = *(fr->pnt+MssFlds2num(fnum,j));

      if( MssIsNull(fldStr) ){
        *(*(dat->numNull+j)+i)=1;
      }else{
        *(*(dat->numNull+j)+i)=0;
        *(*(dat->num+j)+i)=atof(fldStr);
      }
    }

    /*カテゴリー属性の格納*/
    for(j=0; j<dat->catCnt; j++){

      /*項目の値(文字列)のアドレスをセット*/
      fldStr = *(fr->pnt+MssFlds2num(fcat,j));

      if( MssIsNull(fldStr) ){
        *(*(dat->catNull+j)+i)=1;
      }else{
        *(*(dat->catNull+j)+i)=0;

        /*レコードのデータを内部数値として登録*/
        mssVclear(&val);
        val=mssHashMemberVal( (dat->cat+j)->name, fldStr );
        if(val.nul) *((dat->cat+j)->val + i) = 0;
        else        *((dat->cat+j)->val + i) = val.v.i;
      }
    }

    /*パターンの格納*/
    for(j=0; j<fpat->cnt; j++){
      /*項目の値(文字列)のアドレスをセット*/
      fldStr=*(fr->pnt+MssFlds2num(fpat,j));

      if( MssIsNull(fldStr) ){
        *(*(dat->patNull+j)+i)=1;
        alpha[0]=0;
        putStrListUSI((dat->pat+j)->patAlp,alpha);
      }else{
        *(*(dat->patNull+j)+i)=0;
 
        list=tokByChr(fldStr,*optDLM.str,&listCnt);
        /*kがある値(ex.1024)を越えたらエラーにする*/
        for(k=0;k<listCnt;k++){
          node=(struct mssHashNode *)mssHashMember((*(map+j))->alpOrg,*(list+k));
          if( NULL == node ){
            mssShowErrMsg("internal error1");
            exit(mssErrorNoDefault);
          }else{
            alpha[k]=node->val.v.usi;
          }
        }
        mssFree(*list); mssFree(list);
        alpha[k]=0; /*ターミネータ*/
        putStrListUSI((dat->pat+j)->patAlp,alpha);
      }
    }

    /*件数カウントアップ*/
    i++;
  }
  mssFreeFldRec(fr);

  /*ファイルポインタのコピー*/
  dat->fpr=fpr;

  /*データ件数のセット*/
  dat->cnt=cnt;

  return(dat);
}

/**
 * # FUNCTION #
 * データの表示(debug用)
 * Map構造体の領域開放
 */
void showDat( struct Data *dat){

  int i,j,k;

  printf("=========================================\n");
  printf(" Data of \"%s\"\n",dat->fpr->fName);
  printf("=========================================\n");
  printf(" number of lines  : %d\n",dat->cnt);
  printf(" number of patFld : %d\n",dat->patCnt);
  printf(" number of numFld : %d\n",dat->numCnt);
  printf(" number of catFld : %d\n",dat->catCnt);
  for(i=0; i<ClassSize; i++){
    printf(" cls[%d]=%s, ",i,dat->cls->str[i]);
  }
  printf("\n");
  for(i=0; i<dat->patCnt; i++){
    printf(" alphabet on pattern field[%d] numFlg=%d : ",i,(dat->pat+i)->numPat)
;

    if( (dat->pat+i)->map->idxSiz==0){
      prnstrUSI((dat->pat+i)->map->alp);
      printf("\n");
    }else{
      prnstrUSI((dat->pat+i)->map->alp);
      printf("(");
      prnstrUSI((dat->pat+i)->map->idx);
      printf(")\n");
    }
  }
  printf("-----------------------------------------\n");
  for(i=0; i<dat->cnt; i++){
    /*パターン項目*/
    for(j=0; j<dat->patCnt; j++){
      if(*(*(dat->patNull+j)+i)==1){
        printf("NULL ");
      }else{
        if((dat->pat+j)->patIdx->lineCnt==0){
          prnstrUSI(getStrListUSI((dat->pat+j)->patAlp,i));
        }else{
          prnstrUSI(getStrListUSI((dat->pat+j)->patAlp,i));
          printf("(");
          prnstrUSI(getStrListUSI((dat->pat+j)->patIdx,i));
          printf(")");
        }
      }
    }
    /*数値項目*/
    for(j=0; j<dat->numCnt; j++){
      if(*(*(dat->numNull+j)+i)==1){
        printf("NULL ");
      }else{
        printf("%g ",*(*(dat->num+j)+i) );
      }
    }
    /*カテゴリー項目*/
    for(j=0; j<dat->catCnt; j++){
      if(*(*(dat->catNull+j)+i)==1){
        printf("NULL ");
      }else{ 
        printf("%s(%d) ",*((dat->cat+j)->valName+*((dat->cat+j)->val+i)),
                         *((dat->cat+j)->val+i) );
      }
    }
    /*クラス項目*/
    printf("cls=%d\n",(int)*(dat->cls->chr+i) );
  }
  
  /*正規パターン候補の出力*/
  if(dat->pat->regTbl!=NULL){
    printf("-----------------------------------------\n");
    for(i=0; i<dat->patCnt; i++){
      printf("pattern #%d\n",i);
      for(j=0; j<(dat->pat+i)->regTbl->cnt; j++){
        printf("#%02d=",j);
        prnstrUSI(((dat->pat+i)->regTbl->reg+j)->str);
        printf("(%g)\n", ((dat->pat+i)->regTbl->reg+j)->objVal);
      }
    }
  }

  
  /*正規パターン候補を属性とした全データの出力*/
  if(dat->pat->attCnt!=0){
    printf("-----------------------------------------\n");
    for(i=0; i<dat->patCnt; i++){ /*パターン項目数*/
      printf("----- pat #%d\n",i);
      for(j=0; j<dat->cnt; j++){ /*サンプルデータ件数*/
        for(k=0; k<(dat->pat+i)->attCnt; k++){ /*正規表現数*/
          if(*((dat->pat+i)->att[k]+j) ) printf("1");
          else                           printf("0");
        }
        printf("cls=%d\n",*(dat->cls->chr+j) );
      }
    }
  }
  printf("=========================================\n");
}

/**
 * # FUNCTION #
 * データ領域を開放する
 */
void freeDat(struct Data *dat){
  int i;
  if(dat==NULL)return;
  for(i=0; i<dat->patCnt; i++){
    mssFree(*(dat->patNull+i));
    freeStrListUSI((dat->pat+i)->patAlp);
    freeStrListUSI((dat->pat+i)->patIdx);
  }
  mssFree(dat->patNull);
  mssFree(dat->pat);

  for(i=0; i<dat->catCnt; i++){
    mssFree(*(dat->catNull+i));
    mssFree((dat->cat+i)->val);
  }
  mssFree(dat->catNull);
  mssFree(dat->cat);

  for(i=0; i<dat->numCnt; i++){
    mssFree(*(dat->numNull+i));
    mssFree(*(dat->num+i));
  }
  mssFree(dat->numNull);
  mssFree(dat->num);

  mssFree(dat->cls->chr);
  mssFree(dat->cls);
  mssFree(dat);
}
