/**
 * # CHAPTER #
 * ============================================================================
 * xtclassifyで用いられる決定木関連関数
 * ============================================================================
 */

#include <musashi.h>
#include <condition.h>
#include <classTable.h>
#include <bonsai.h>
#include <tree.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <float.h>

extern MssOptFLD optPAT;
extern MssOptFLD optNUM;
extern MssOptFLD optCAT;
extern MssOptFLD optCLS;
extern MssOptINF optTST;
extern MssOptINF optCST;
extern MssOptDBL optCNF;
extern MssOptINT optMIN;

extern struct mssFields *fpat;
extern struct mssFields *fnum;
extern struct mssFields *fcat;
extern struct mssFields *fcls;

extern int   ClassSize;
extern char  catCombi[1024][10];
extern int   usedRandSeed;

extern struct mssGlobalVariables mssGV;
extern struct mssPMMLvariables mssPV;

/*ツリー生成時に、Dataを参照するためのポインタ*/
struct Pointer {
  int  recCnt;        /*データ数*/
  int *recNo;         /*先頭からの行数*/
  double *weight;     /*NULL値を考慮したrecNoに対応するケースの重み*/
};

/**
 * # FUNCTION #
 *レベルのセット関連
 *  1. レベルのセット
 *  2. 最深レベルのセット
 *  3. リーフ数のカウント
 */
static void setLevel(struct Node *node,struct Tree *tree, int level){

  /*レベルのセット*/
  node->level=level;

  /*最深レベルの更新*/
  if(level>tree->deepest) tree->deepest=level;

  /*リーフならばリターン*/
  if( 0==node->nodeType || node->pruned){
    tree->leafCnt++;
    return;
  }

  setLevel(node->uNode,tree,level+1);
  setLevel(node->mNode,tree,level+1);
}

/**
 * # FUNCTION #
 * ツリーのノードを次々に解放する再帰関数
 */
static struct Node **freeTreeTbl(
  struct Node  *node,
  struct Node **nodeTbl,
  int          *nodeCnt){

  int cnt;
  if(node->attType==2){ /*カテゴリのリストを解放*/
    mssFree( node->catList );
  } 
  if(node->nodeType==0){ /*leafなら*/
    return(nodeTbl);
  }else{
    cnt=(*nodeCnt);
    cnt=cnt+2;
    nodeTbl=mssRealloc(nodeTbl,sizeof(struct Tree *)*cnt,"freeTree");
    *(nodeTbl+cnt-2) = node->uNode;
    *(nodeTbl+cnt-1) = node->mNode;
    *nodeCnt=cnt;
  }
  nodeTbl=freeTreeTbl(node->uNode,nodeTbl,nodeCnt);
  nodeTbl=freeTreeTbl(node->mNode,nodeTbl,nodeCnt);
  return(nodeTbl);
}

/**
 * # FUNCTION #
 * サンプル構造体のコピー(to内の各種領域確保を含む)
 */
static struct Data *cpyDatMapReg(struct Data *from){

  int i,j;
  struct Data *to;

  to         = mssCalloc(sizeof(struct Data),"cpySmp");

  /*==============================================*/
  /*決定木によって不変の変数はポインタとしてコピー*/
  /*==============================================*/
  to->numCnt = from->numCnt;
  to->num    = from->num;
  to->numNull= from->numNull;
  to->catCnt = from->catCnt;
  to->cat    = from->cat;
  to->catNull= from->catNull;
  to->cls    = from->cls;



  /*====================================================*/
  /*パターン項目は決定木によって変わるので、実体をコピー*/
  /*====================================================*/
  to->patCnt = from->patCnt;
  if(to->patCnt>0){
    to->pat    = mssCalloc( to->patCnt * sizeof(struct Pattern) ,"cpyDat" );

    for(i=0; i<to->patCnt; i++){
      to->patNull= from->patNull;

      (to->pat+i)->numPat=(from->pat+i)->numPat;

      /*Mapのコピー*/
      (to->pat+i)->map=mssCalloc(sizeof(struct Map), "cpySmp");
      (to->pat+i)->map->alpOrg = (from->pat+i)->map->alpOrg;
      (to->pat+i)->map->numAlp = (from->pat+i)->map->numAlp;
      (to->pat+i)->map->alp
        =mssCalloc(sizeof(usint)*((from->pat+i)->map->alpSiz+1), "cpySmp");
      (to->pat+i)->map->idx
        =mssCalloc(sizeof(usint)*((from->pat+i)->map->alpSiz+1), "cpySmp");
      (to->pat+i)->map->alpSiz=(from->pat+i)->map->alpSiz;
      (to->pat+i)->map->idxSiz=(from->pat+i)->map->idxSiz;
      strcpyUSI((to->pat+i)->map->alp,(from->pat+i)->map->alp);
      strcpyUSI((to->pat+i)->map->idx,(from->pat+i)->map->idx);

      /*regTblのコピー*/
      (to->pat+i)->regTbl=mssMalloc(sizeof(struct RegTbl),"spySmp");
      (to->pat+i)->regTbl->reg =
        mssCalloc(sizeof(struct Regexp)*(from->pat+i)->regTbl->cnt,"scr");
      (to->pat+i)->regTbl->cnt= (from->pat+i)->regTbl->cnt;
      for(j=0; j<(to->pat+i)->regTbl->cnt; j++){
        cpReg( (to  ->pat+i)->regTbl->reg+j,
               (from->pat+i)->regTbl->reg+j);
      }
    }
  }
  return(to);
}

/**
 * # FUNCTION #
 * cpyDatMapRegでコピーされた領域の解放
 */
static void freeDatMapReg(struct Data *dat){

  int i;

  if(dat->patCnt==0) return;

  for(i=0; i<dat->patCnt; i++){
    /*Mapのfree*/
    mssFree( (dat->pat+i)->map->alp );
    mssFree( (dat->pat+i)->map->idx );
    mssFree( (dat->pat+i)->map );

    /*regTblのfree*/
    mssFree( (dat->pat+i)->regTbl->reg );
    mssFree( (dat->pat+i)->regTbl );
  }
  mssFree(dat->pat);
  mssFree(dat);
}

/**
 * # FUNCTION #
 * 一件のデータを決定木のノードに当てはめ、予測クラスを求める再帰関数
 */
static void predictTreeSub(
  double clsWeight[],
  struct Node *this,
  struct Data *dat,
  int recNo,
  double weight,
  struct Tree *tree){

  usint index[MaxPatLen];
  double mWeight;
  double uWeight;
  int isNull=0;
  int match=0;

  /*リーフに到達すれば、weightをclsWeightテーブルにセットする。*/
  if( 0==this->nodeType || this->pruned ){
    clsWeight[this->cls]+=weight;
    return;
  }

  /*現在ノードの条件項目がデータ上でNULLの場合は
    件数によるweightをかけて再帰処理*/
  switch(this->attType){
  case 0:
    isNull=*(*(dat->patNull+this->attFldNo)+recNo);
    break;
  case 1:
    isNull=*(*(dat->numNull+this->attFldNo)+recNo);
    break;
  case 2:
    isNull=*(*(dat->catNull+this->attFldNo)+recNo);
    break;
  }
  if( isNull ){
    mWeight=weight * this->trnCnt.mRatio;
    uWeight=weight * (1-this->trnCnt.mRatio);
    predictTreeSub(clsWeight, this->mNode,dat,recNo,mWeight,tree);
    predictTreeSub(clsWeight, this->uNode,dat,recNo,uWeight,tree);
  } else {

    /*条件によりmatchもしくはunmatchノードをたどる*/
    switch(this->attType){
    case 0:
      /*パターン属性だけは、alphabet-index等のデータが必要なため
        オリジナルのアルファベットパターンからインデックス化されたデータを作成し
        その後にmatch,unmatchを決定する(トレーニングだけなら問題ないが、テスト
        データを予測する際に必要となる。*/
      alpStr2idxStr(index,
                    getStrListUSI((dat->pat+this->attFldNo)->patAlp,recNo),
                    (tree->dat->pat+this->attFldNo)->map);
      match=regCmp(index,
        (tree->dat->pat+this->attFldNo)->regTbl->reg+this->attPatNo);

      /*match=(int)*((dat->pat+(this->attFldNo))->att[this->attPatNo]+recNo);*/
      break;
    case 1:
      match= *(*(dat->num+this->attFldNo)+recNo) <= this->numTH;
      break;
    case 2:
      match= this->catList[*((dat->cat+this->attFldNo)->val+recNo)]=='M';
      break;
    }
    if( match ){
      predictTreeSub(clsWeight, this->mNode,dat,recNo,weight,tree);
    }else{
      predictTreeSub(clsWeight, this->uNode,dat,recNo,weight,tree);
    } 
  }
}

/**
 * # FUNCTION #
 * データ(dat)のあるレコード(rec)が決定木(tree)で予測されるクラス値を返す関数
 */
static int predictTree( struct Tree *tree, struct Data *dat, int recNo){
  double clsWeight[MaxClsLen];
  int maxCls=0;
  double maxClsWeight=0;
  int i;

  for(i=0; i<MaxClsLen; i++) clsWeight[i]=0;
  predictTreeSub(clsWeight, tree->topNode, dat, recNo,1,tree);

  for(i=0; i<dat->cls->cnt; i++){
    if(maxClsWeight<clsWeight[i]){
      maxCls=i;
      maxClsWeight=clsWeight[i];
    }
  }

  return(maxCls);
}

/**
 * # FUNCTION #
 * データを一件ずつ読み込み、ClassificationTableに件数をセットする
 * セットされたテーブルを返す。
 */
struct ClsTbl *setClsTbl( struct Data *dat, struct Tree *tree){
  struct ClsTbl *ct;
  int predictedClass;
  int actualClass;
  int i;

  ct=mssCalloc(sizeof(struct ClsTbl),"initClsTbl");

  for(i=0; i<dat->cnt; i++){
    actualClass=*(dat->cls->chr+i);
    predictedClass=predictTree(tree,dat,i);
    ct->cnt[actualClass][predictedClass]++;
  }

  return(ct);
}
  
/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 決定木を成長させる関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 正規パターン条件別クラス別件数
 */
static void setCndCntPat(
  struct CndCnt  *cnt,
  struct Pattern *pat,
  char           *patNull,
  struct Class   *cls,
  struct Pointer *pnt,
  int             attNo){

  int i;

  for(i=0; i<pnt->recCnt; i++){
    if(*(patNull+*(pnt->recNo+i))==1){/*NULL!!*/
      cnt->nCnt[(int)(*(cls->chr+*(pnt->recNo+i)))]+=*(pnt->weight+i);
      cnt->mCount++;
      cnt->uCount++;
    }else{
      if( 0 == *(pat->att[attNo]+*(pnt->recNo+i)) ){
        cnt->uCnt[(int)*(cls->chr+*(pnt->recNo+i))]+=*(pnt->weight+i);
        cnt->utCnt+=*(pnt->weight+i);
        cnt->uCount++;
      }else{
        cnt->mCnt[(int)*(cls->chr+*(pnt->recNo+i))]+=*(pnt->weight+i);
        cnt->mtCnt+=*(pnt->weight+i);
        cnt->mCount++;
      }
    }
  }
}

/**
 * # STRUCT #
 * 数値属性を数値順にソートするための構造体
 */
struct NumCls {
  double num;
  char   cls;
  double weight; /*NULLを勘案した重み*/
};

/**
 * # FUNCTION #
 * 数値属性を数値順にソートする
 */
static int qscmp3(const void **a, const void **b){
  double x,y;
  x=((struct NumCls *)a)->num;
  y=((struct NumCls *)b)->num;
  if(x>y)      return(1);
  else if(x<y) return(-1);
  else         return(0);
}

/**
 * # FUNCTION #
 * 数値属性の条件別クラス別件数
 */
static double setCndCntNum(
  struct CndCnt  *cnt,
  double         *num,
  char           *numNull,
  struct Class   *cls,
  struct Pointer *pnt,
  struct Cost    *cost){

  struct CndCnt  tmpCnt;
  struct NumCls *tbl;
  double bestTh;
  double th;
  double bestSv;
  int i,j;

  /* NumCls構造体の領域確保*/
  tbl=mssMalloc(pnt->recCnt * sizeof(struct NumCls),"setCndCntNum");

  /*数値属性の値、クラス値をNumCls構造体にセット*/
  /*同時にtmpCnt構造体を初期化*/
  iniCndCnt(&tmpCnt,cost);
  j=0;
  for(i=0; i<pnt->recCnt; i++){
    if(*(numNull+*(pnt->recNo+i))==1){/*NULL!!*/
      tmpCnt.nCnt[(int)(*(cls->chr+*(pnt->recNo+i)))]+=*(pnt->weight+i);
      tmpCnt.mCount++;
      tmpCnt.uCount++;
      tmpCnt.nCount++;
    }else{
      (tbl+j)->num   =*(num+*(pnt->recNo+i));
      (tbl+j)->cls   =*(cls->chr+*(pnt->recNo+i));
      (tbl+j)->weight=*(pnt->weight+i);
      tmpCnt.uCnt[(int)((tbl+j)->cls)]+=*(pnt->weight+i);
      tmpCnt.utCnt+=*(pnt->weight+i);
      tmpCnt.total+=*(pnt->weight+i);
      tmpCnt.uCount++;
      j++;
    }
  }

  /*数値順に並べ変える*/
/*
  qsort(tbl,j,sizeof(struct NumCls),
                       (int (*)(const void *, const void *))qscmp3);
*/
  qsort5(tbl,j,sizeof(struct NumCls), (int (*))qscmp3);

  bestTh=0;
  th=0;
  bestSv=99999;
  for(i=0; i<j; i++){
    th=(tbl+i)->num;
    tmpCnt.uCnt[(int)(tbl+i)->cls]-=(tbl+i)->weight;
    tmpCnt.mCnt[(int)(tbl+i)->cls]+=(tbl+i)->weight;
    tmpCnt.utCnt-=(tbl+i)->weight;
    tmpCnt.mtCnt+=(tbl+i)->weight;
    tmpCnt.mCount++;
    tmpCnt.uCount--;

    if( i!=j-1 ){ /*最終レコードでないならば*/
      if(th == (tbl+i+1)->num){ /*次のレコードの値も同じならばcontinue*/
        continue;
      }
    }

    /*条件別クラス間エントロピーの計算*/
    calCndCnt(&tmpCnt);
    calCndCntSplit(&tmpCnt);

    /* 最良属性の更新 */ 
    if( tmpCnt.splitAfter < bestSv){
      cpyCndCnt(cnt,&tmpCnt);
      bestSv=tmpCnt.splitAfter;
      if( i!=pnt->recCnt-1 ) bestTh=(th+(tbl+i+1)->num)/(double)2;
      else                   bestTh=th;
    }
  }

  mssFree(tbl);
  return(bestTh);
}

#define ValClsCnt(x,y) (*(valClsCnt+(x)*cls->cnt+(y)))

/**
 * # FUNCTION #
 * カテゴリー属性の条件別クラス別件数
 * カテゴリー値別Match,Unmatch表を返す
 *   ex) MMUUUMUM
 */
static char *setCndCntCat(
  struct CndCnt   *cnt,
  struct Category *cat,
  char            *catNull,
  struct Class    *cls,
  struct Pointer  *pnt,
  struct Cost     *cost){

  int end[]={2,4,8,16,32,64,128,256,512,1024};
  struct CndCnt  tmpCnt;
  double bestSv=99999;
  int    bestCombi=0;
  char  *catList=NULL;
  int    improve;
  int    moveCat=-1;
  int i,j,k;

  double *valClsCnt;
  int *valRecCnt;    /*レコード件数を入れる*/
  int nulRec; /*nullのレコード件数*/
  int catVal,clsVal;
  double nulCnt[MaxClsLen];

  char combiNull[11];
  char *bestCombiNull=NULL;
  double total;

  for(i=0; i<ClassSize; i++){
    nulCnt[i]=0;
  }

  /* カテゴリーの値別class集計表の作成 */
  /* valClsCnt[val][class] 領域確保*/
  valClsCnt=mssCalloc(cat->cnt*cls->cnt*sizeof(double),"setCndCntCat1");
  valRecCnt=mssCalloc(cat->cnt*sizeof(int),"setCndCntCat1");
  nulRec=0;
  for(i=0; i<pnt->recCnt; i++){
    if(*(catNull+*(pnt->recNo+i))==1){/*NULL!!*/
      nulCnt[(int)(*(cls->chr+*(pnt->recNo+i)))]+=*(pnt->weight+i);
      nulRec++;
    }else{
      catVal=     *(cat->val+*(pnt->recNo+i));
      clsVal=(int)*(cls->chr+*(pnt->recNo+i));
      ValClsCnt(catVal,clsVal)+=*(pnt->weight+i);
      (*(valRecCnt+catVal))++;
    }
  }

  /*  集計表の表示 */
  /*
  for(i=0; i<cat->cnt; i++){
    printf(" val=%d : ",i);
    for(j=0; j<cls->cnt; j++){
      printf("cls[%d]=%d ",j,ValClsCnt(i,j));
    }
    printf("\n");
  }
  */

  /*tmpCnt構造体を初期化*/
  iniCndCnt(&tmpCnt,cost);

  /*=======================================================================*/
  /*カテゴリー属性の値の種類数が１０種類以下ならば、Match,Unmatchの全組合せ*/
  /*を計算し、そこから最適な分割を求める。                                 */
  /* catCombi[i][val]={M,U};                                               */
  /* i 0 1 2 .... */
  /* 0 M M M .... */
  /* 1 U M M .... */
  /* 2 M U M .... */
  /* 3 U U M .... */
  /* 4 M M U .... */
  /* 5 U M U .... */
  /* 6 M U U .... */
  /* 7 U U U .... */
  /*   実際に調べるのはi=1to7で2飛び、すなわち、1,3,5を調べる */
  /*   2飛びにするのは、0と7,1と6,2と5,3と4は同じ意味なので*/ 
  if(cat->cnt <= 10){
    for(i=0; i<end[cat->cnt-1]-1; i=i+2){
      iniCndCnt(&tmpCnt,cost);
      /*NULL件数を反映させる*/
      for(j=0; j<ClassSize; j++){
        tmpCnt.nCnt[j]=nulCnt[j];
      }
      for(j=0; j<cat->cnt+1; j++){
        combiNull[j]='0';
      }

      tmpCnt.mCount=nulRec;
      tmpCnt.uCount=nulRec;
      for(j=0; j<cat->cnt; j++){ /*カテゴリの種類数*/
        total=0; /*カテゴリーの値(j)がトータル何件かをカウントする*/

        if(catCombi[i][j]=='M'){
          tmpCnt.mCount += *(valRecCnt+j);
        }else{
          tmpCnt.uCount += *(valRecCnt+j);
        }
 
        for(k=0; k<cls->cnt; k++){ /*クラス数*/
          if(catCombi[i][j]=='M'){
            tmpCnt.mCnt[k]+=ValClsCnt(j,k);
          }else{
            tmpCnt.uCnt[k]+=ValClsCnt(j,k);
          }
          total+=ValClsCnt(j,k);
        }

        /*カテゴリの値(j)が一件もなければフラグを立てておく(表示しないため)*/
        if(total==0){
          combiNull[j]='N';
        }
      }

      /*条件別クラス間エントロピーの計算*/
      calCndCnt(&tmpCnt);
      calCndCntSplit(&tmpCnt);
      /*prnCndCnt(&tmpCnt);*/

      /* 最良属性の更新 */ 
      if( tmpCnt.splitAfter < bestSv){
        cpyCndCnt(cnt,&tmpCnt);
        bestSv=tmpCnt.splitAfter;
        bestCombi=i;
        mssFree(bestCombiNull);
        bestCombiNull=mssStrdup(combiNull);
      } 
    }

    catList=mssCalloc(sizeof(char)*cat->cnt,"setCndCntCat2");
    for(j=0; j<cat->cnt; j++){
      if(*(bestCombiNull+j)=='N'){
        *(catList+j)='N';
      }else{
        *(catList+j)=catCombi[bestCombi][j];
      }
    }
    mssFree(bestCombiNull);

  /*=======================================================================*/
  /*カテゴリー属性の値の種類数が１０種類を越えるならば、greedyに           */
  /*良い分割を求める                                                       */
  }else{
    /*初期値として、全てMatchとして扱う*/
    catList=mssCalloc(sizeof(char)*cat->cnt,"setCndCntCat3");
    /*NULL件数を反映させる*/
    for(j=0; j<ClassSize; j++){
      tmpCnt.nCnt[j]=nulCnt[j];
    }

    tmpCnt.mCount=nulRec;
    tmpCnt.uCount=nulRec;
    for(j=0; j<cat->cnt; j++){
      *(catList+j)='M';
      for(k=0; k<cls->cnt; k++) tmpCnt.mCnt[k]+=ValClsCnt(j,k);
      tmpCnt.mCount += *(valRecCnt+j);
    }
    calCndCnt(&tmpCnt);
    calCndCntSplit(&tmpCnt);
    cpyCndCnt(cnt,&tmpCnt);
    bestSv=tmpCnt.splitAfter;

    /*改善がある限り、MatchからUnmatchにカテゴリー値を移動していく*/
    improve=1;
    while(improve) {
      improve=0;
      /*Matchから一つずつ移動して良くなるか確認*/
      for(j=0; j<cat->cnt; j++){
        if(*(catList+j)=='M'){
          /*MatchからUnmatchに移動*/
          for(k=0; k<cls->cnt; k++){
            tmpCnt.mCnt[k]-=ValClsCnt(j,k);
            tmpCnt.uCnt[k]+=ValClsCnt(j,k);
          }
          tmpCnt.mCount -=*(valRecCnt+j);
          tmpCnt.uCount +=*(valRecCnt+j);
          calCndCnt(&tmpCnt);
          calCndCntSplit(&tmpCnt);
          if(tmpCnt.splitAfter < bestSv){
            bestSv=tmpCnt.splitAfter;
            moveCat=j;
            improve=1;
          }
          /*MatchからUnmatchに移動したのを元に戻す*/
          for(k=0; k<cls->cnt; k++){
            tmpCnt.mCnt[k]+=ValClsCnt(j,k);
            tmpCnt.uCnt[k]-=ValClsCnt(j,k);
          }
          tmpCnt.mCount +=*(valRecCnt+j);
          tmpCnt.uCount -=*(valRecCnt+j);
        } /*if*/
      }

      /*改善があれば、moveCatをUnmatchに正式に移動する*/
      if(improve){
        *(catList+moveCat)='U';
        for(k=0; k<cls->cnt; k++){
          tmpCnt.mCnt[k]-=ValClsCnt(moveCat,k);
          tmpCnt.uCnt[k]+=ValClsCnt(moveCat,k);
        }
        tmpCnt.mCount -=*(valRecCnt+moveCat);
        tmpCnt.uCount +=*(valRecCnt+moveCat);
      }
    } /*while*/
    calCndCnt(&tmpCnt);
    calCndCntSplit(&tmpCnt);
    cpyCndCnt(cnt,&tmpCnt);

  }

  mssFree(valClsCnt);
  mssFree(valRecCnt);
  return(catList);
}

/**
 * # FUNCTION #
 * Treeのノード作成再帰関数
 */
static struct Node *makeTree(
  struct Data     *dat,
  struct Cost     *cost,
  struct Pointer  *pointer,
  struct Node     *parent){

  struct Node  *this;
  struct CndCnt cnt;

  struct Pointer uPnt;
  struct Pointer mPnt;

  char *catList;

  double tmpTH;
  int match=0;
  int isNull=0;
  int i,j,jm,ju,jn;

  /*カレントノード領域の確保と初期化*/
  this = mssCalloc(sizeof(struct Node), "Node");
  this->parent = parent;

  /* this->xxxCntの初期化 */
  iniCndCnt(&this->trnCnt,cost);

  /*splitting ruleの値の初期化*/
  this->trnCnt.splitAfter=DBL_MAX;

  /*----------------------------------------------*/
  /*パターン項目における最小コストの属性を見付ける*/
  /*----------------------------------------------*/
  for(i=0; i<dat->patCnt; i++){
    for(j=0; j<(dat->pat+i)->attCnt; j++){
      iniCndCnt(&cnt,cost);
      setCndCntPat(&cnt,dat->pat+i,*(dat->patNull+i),dat->cls,pointer,j);
      calCndCnt(&cnt);
      calCndCntSplit(&cnt);
      /*prnCndCnt(&cnt);*/

      /* 最小コストの属性を更新*/
      if(this->trnCnt.splitAfter > cnt.splitAfter){
        cpyCndCnt(&this->trnCnt,&cnt);
        this->attType =0;
        this->attFldNo=i;
        this->attPatNo=j;
      }
    }
  }

  /*----------------------------------------------*/
  /*数値項目における最小コストの属性を見付ける    */
  /*----------------------------------------------*/
  for(i=0; i<dat->numCnt; i++){
    iniCndCnt(&cnt,cost);
    tmpTH=setCndCntNum(&cnt,*(dat->num+i),*(dat->numNull+i),dat->cls,pointer,cost);
    calCndCnt(&cnt);
    calCndCntSplit(&cnt);
    /*prnCndCnt(&cnt);*/

    if(this->trnCnt.splitAfter > cnt.splitAfter){
      cpyCndCnt(&this->trnCnt,&cnt);
      this->attType =1;
      this->attFldNo=i;
      this->numTH = tmpTH;
    }
  }

  /*------------------------------------------------*/
  /*カテゴリー項目における最小コストの属性を見付ける*/
  /*------------------------------------------------*/
  for(i=0; i<dat->catCnt; i++){
    iniCndCnt(&cnt,cost);
    catList=setCndCntCat(&cnt,dat->cat+i,*(dat->catNull+i),dat->cls,pointer,cost);
    calCndCnt(&cnt);
    calCndCntSplit(&cnt);

    if(this->trnCnt.splitAfter > cnt.splitAfter){
      cpyCndCnt(&this->trnCnt,&cnt);
      this->attType =2;
      this->attFldNo=i;
      mssFree(this->catList);
      this->catList = catList;
    }else{
      mssFree(catList);
    }
  }

  /*ノードのクラスを決定(リーフ以外でもクラスを決定するのは枝苅のため)*/
  this->cls=dominantClass(&this->trnCnt);


  /*match, unmatchのノードでいずれかが、全てNULLデータの場合はLeaf処理&return*/
  if(this->trnCnt.mCount==this->trnCnt.nCount ||
     this->trnCnt.uCount==this->trnCnt.nCount ){
    this->nodeType=0;
    return(this);
  }


  /*改善がなければLeaf処理&return*/
  if(this->trnCnt.splitAfter >= this->trnCnt.splitBefor){
    this->nodeType=0;
    return(this);
  }

  /*右左いずれかが0件の場合,Leaf処理&return*/
  if(this->trnCnt.utCnt == 0 || this->trnCnt.mtCnt == 0){
    this->nodeType=0;
    return(this);
  }

  /*上の条件に当てはまらないということはノードである*/
  this->nodeType = 1;
  /*prnCndCnt(&this->trnCnt);*/


  /*------------------------------------------------------*/
  /* 元データ(pointer)をmatchデータとunmatchデータに分ける*/
  /*------------------------------------------------------*/
  uPnt.recCnt = this->trnCnt.uCount;
  mPnt.recCnt = this->trnCnt.mCount;
  uPnt.recNo  = mssCalloc(uPnt.recCnt*sizeof(int),"uIdx");
  mPnt.recNo  = mssCalloc(mPnt.recCnt*sizeof(int),"mIdx");
  uPnt.weight = mssCalloc(uPnt.recCnt*sizeof(double),"uIdx");
  mPnt.weight = mssCalloc(mPnt.recCnt*sizeof(double),"mIdx");
  jm=0;
  ju=0;
  jn=0;

  for(i=0; i<pointer->recCnt; i++){
    switch(this->attType){
    case 0:
      isNull=*(*(dat->patNull+this->attFldNo)+*(pointer->recNo+i));
      break;
    case 1:
      isNull=*(*(dat->numNull+this->attFldNo)+*(pointer->recNo+i));
      break;
    case 2:
      isNull=*(*(dat->catNull+this->attFldNo)+*(pointer->recNo+i));
      break;
    }
    if( isNull ){
      *(mPnt.recNo +jm)=*(pointer->recNo+i);
      *(mPnt.weight+jm)=*(pointer->weight+i) * this->trnCnt.mRatio;
      *(uPnt.recNo +ju)=*(pointer->recNo+i);
      *(uPnt.weight+ju)=*(pointer->weight+i) * (1-this->trnCnt.mRatio);
      jm++;
      ju++;
      jn++;
    } else {
      switch(this->attType){
      case 0:
        match=(int)*((dat->pat+(this->attFldNo))->att[this->attPatNo]+*(pointer->recNo+i));
        break;
      case 1:
        match= *(*(dat->num+this->attFldNo)+*(pointer->recNo+i)) <= this->numTH;
        break;
      case 2:
        match= this->catList[*((dat->cat+this->attFldNo)->val+*(pointer->recNo+i))]=='M';
        break;
      }
      if( match ){
        *(mPnt.recNo +jm)=*(pointer->recNo+i);
        *(mPnt.weight+jm)=*(pointer->weight+i);
        jm++;
      }else{
        *(uPnt.recNo +ju)=*(pointer->recNo+i);
        *(uPnt.weight+ju)=*(pointer->weight+i);
        ju++;
      }
    }
  }

  /*----------------------------*/
  /*ノードを伸ばす(再帰呼び出し)*/
  /*----------------------------*/
  this->mNode = makeTree(dat,cost,&mPnt,this);
  this->uNode = makeTree(dat,cost,&uPnt,this);

  /*ポインタ領域の開放*/
  mssFree(uPnt.recNo);
  mssFree(uPnt.weight);
  mssFree(mPnt.recNo);
  mssFree(mPnt.weight);

  return(this);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 枝苅り (Error-Based Pruning)
 * ----------------------------------------------------------------------------
 */

static double CF;

double Val[] = {  0,  0.001, 0.005, 0.01, 0.05, 0.10, 0.20, 0.40, 1.00};
double Dev[] = {100.0,3.09,  2.58,  2.33, 1.65, 1.28, 0.84, 0.25, 0.00};

/**
 * # FUNCTION #
 * 枝苅り 予測分類誤差計算(refered from C4.5)
 */
static double calEBPerr(double NN, double e){

  double coeff=0;
  double val0;
  int i;

  i=1;
  while(CF>Val[i])i++;
  coeff=Dev[i-1]+(Dev[i]-Dev[i-1]) * (CF-Val[i-1]) / (Val[i]-Val[i-1]);
  coeff=coeff * coeff;

  if(e<1E-6){
    return( NN * (1-exp(log(CF)/NN)) );
  }else if(e < 0.9999 ){
    val0=NN * (1-exp(log(CF)/NN));
    return(val0+e*(calEBPerr(NN,1.0)-val0) );
  }else if(e+0.5 >=NN){
    return( 0.67 * (NN-e) );
  }else{
    return( NN * ((e+0.5+coeff/2+sqrt(coeff*((e+0.5)*(1-(e+0.5)/NN)+coeff/4)))
                 /(NN+coeff)) -e  );
  }
}

/**
 * # FUNCTION #
 * 全ノードにおける予測分類誤差(C4.5)を計算してセットする
 */
static void setEBPNodeErr(struct Node *node){
  double errCnt;
  int i;

  /*error件数のカウント*/
  errCnt=0;
  for(i=0; i<ClassSize; i++){
    if(i!=node->cls) errCnt += (double)node->trnCnt.total 
                              *(*(node->trnCnt.tShr+i));
  }
  node->estErrNode = calEBPerr((double)node->trnCnt.total,errCnt)+errCnt;

  /*リーフならばリターン*/
  if( 0==node->nodeType ) return;

  setEBPNodeErr(node->uNode);
  setEBPNodeErr(node->mNode);
}

/**
 * # FUNCTION #
 * 与えられたノード以下の全リーフにおける予測分類誤差(C4.5)を計算して返す
 */
static double setEBPLeafErr(struct Node *node){

  if( 0==node->nodeType ) return(node->estErrNode);

  node->estErrLeaf= setEBPLeafErr(node->uNode)+setEBPLeafErr(node->mNode);
  return(node->estErrLeaf);
}

/**
 * # FUNCTION #
 * 枝苅り実行
 */
static void pruneEBP(struct Node *node){

  if( 0==node->nodeType ) return;
  
  if(node->estErrLeaf > node->estErrNode ){
    node->pruned=1;
    return;
  }

  pruneEBP(node->uNode);
  pruneEBP(node->mNode);

  return;
}

/**
 * # FUNCTION #
 * 枝苅り 同じクラスを予測する二の葉を持つノードは枝苅り(共通)
 */
static void pruneSameClass(struct Node *node){

  if( 0==node->nodeType ) return;

  if( 1==node->uNode->nodeType && !node->uNode->pruned )
    pruneSameClass(node->uNode);
  if( 1==node->mNode->nodeType && !node->mNode->pruned )
    pruneSameClass(node->mNode);

  /*子が両方ともリーフもしくはPrunedNodeの場合、両方とも同じクラスなら枝苅り*/
  if( (0==node->uNode->nodeType || node->uNode->pruned) &&
      (0==node->mNode->nodeType || node->mNode->pruned) &&
      (node->uNode->cls == node->mNode->cls)            ){
    node->pruned=1;
    return;
  }
}

/**
 * # FUNCTION #
 * ノード内最小ケース数(optMIN.val)に満たないノードは枝刈り
 */
static void pruneMinCaseSize(struct Node *node){

  if( 0==node->nodeType ) return;

  if( 1==node->uNode->nodeType && !node->uNode->pruned )
    pruneMinCaseSize(node->uNode);
  if( 1==node->mNode->nodeType && !node->mNode->pruned )
    pruneMinCaseSize(node->mNode);

  /*子が両方ともリーフもしくはPrunedNodeの場合、両方とも同じクラスなら枝苅り*/
  if( node->trnCnt.total < optMIN.val){
    node->pruned=1;
  }
}

/**
 * # FUNCTION #
 * 枝苅りメイン
 */
static void pruneTree(struct Tree *tree,struct Cost *cost){

  /*ノード内最小ケース数(optMIN.val)による枝刈り*/
  if(optMIN.set){
    pruneMinCaseSize(tree->topNode);
  }

  /*Error-based Pruning*/
  CF=optCNF.val/(double)100;
  setEBPNodeErr(tree->topNode);
  setEBPLeafErr(tree->topNode);
  pruneEBP(tree->topNode);

  /*同じクラスの二のリーフを持つノードを枝苅り*/
  pruneSameClass(tree->topNode);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 表示系関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 決定木のノードを表示する再帰関数
 */
static void showNode(
  struct Tree *tree,
  struct Node *node,
  char trl,          /* match or unmatch */
  struct mssFPW *fpw){

  char *clsNam;
  char *clsVal;
  double hitCnt=0;
  double supCnt=0;
  struct Category *catAdd;
  int firstWrite;
  int i;

  for(i=0; i<node->level; i++){
    mssWriteStr("  ",fpw);
  }
  switch(trl){
  case 'm': /*match*/
    mssWriteStr("then ",fpw);
    break;
  case 'u': /*unmatch*/
    mssWriteStr("else ",fpw);
    break;
  case 't': /*top Node*/
    break;
  }

  /*Leafならばクラス別件数の表示 & 改行*/
  if(node->nodeType==0 || node->pruned==1){
    clsNam = MssFlds2name(fcls,0);
    clsVal = tree->dat->cls->str[node->cls];
    hitCnt = node->trnCnt.tCnt[node->cls],
    supCnt = node->trnCnt.total;
    mssWriteStr("$",fpw);
    mssWriteStr(clsNam,fpw);
    mssWriteStr("=\"",fpw);
    mssWriteStr(clsVal,fpw);
    mssWriteStr("\" (hit/sup)=(",fpw);
    mssWriteDbl(hitCnt,fpw);
    mssWriteStr("/",fpw);
    mssWriteDbl(supCnt,fpw);
    mssWriteStr(")\n",fpw);
    return;
  }

  /*ノードならば*/
  if(node->nodeType==1){
    switch(node->attType){
    case 0: /*pat*/
      mssWriteStr("if($",fpw);
      mssWriteStr(MssFlds2name(fpat,node->attFldNo),fpw);
      mssWriteStr(" has ",fpw);
      prnRegexp((tree->dat->pat+node->attFldNo)->regTbl->reg+node->attPatNo,fpw);
      mssWriteStr(")\n",fpw);
      break;
    case 1: /*num*/
      mssWriteStr("if($",fpw);
      mssWriteStr(MssFlds2name(fnum,node->attFldNo), fpw);
      mssWriteStr("<=",fpw);
      mssWriteDbl(node->numTH,fpw);
      mssWriteStr(")\n",fpw);
      break;
    case 2: /*category*/
      mssWriteStr("if($",fpw);
      mssWriteStr(MssFlds2name(fcat,node->attFldNo),fpw);
      mssWriteStr(" is in {",fpw);
      catAdd=tree->dat->cat+node->attFldNo;
      firstWrite=1;
      for(i=0; i<catAdd->cnt; i++){
        if(node->catList[i]=='M'){
          if(!firstWrite) mssWriteStr(",",fpw);
          else            firstWrite=0;
          mssWriteStr("\"",fpw);
          mssWriteStr(*(catAdd->valName+i),fpw);
          mssWriteStr("\"",fpw);
        }
      }
      mssWriteStr("})\n",fpw);
      break;
    }
  }
  showNode(tree,node->mNode,'m',fpw);
  showNode(tree,node->uNode,'u',fpw);
}

/**
 * # FUNCTION #
 * 正規パターンを文字列として返す
 */
char *getRegStr(struct Regexp *reg){
  char buf[4096];
  char tmp[10];
  usint *str;

  buf[0]='\0';

  /*先頭一致*/
  if(reg->bgnRng!=0){
    strcat(buf,"^[");
    sprintf(tmp,"%d",reg->bgnRng);
    strcat(buf,tmp);
    strcat(buf,"] ");
  }

  str=reg->str;
  while(*str!=0){
    sprintf(tmp,"%d",*str++);
    strcat(buf,tmp);
    if(*str!=0) strcat(buf," ");
  }

  /*末尾一致*/
  if(reg->endRng!=0){
    strcat(buf,"$[");
    sprintf(tmp,"%d",reg->endRng);
    strcat(buf,tmp);
    strcat(buf,"]");
  }
  return(mssStrdup(buf));
}

/**
 * # FUNCTION #
 * パターン属性のノード書き出し
 */
void writePatPMML(
  struct Tree *tree,
  struct Node *node,
  char trl,          /* top or match or unmatch */
  struct mssFPW *fpw){

  struct mssXmlTag *tag;

  struct Regexp *reg;
  char *regStr;

  if(node->nodeType==0 || node->pruned==1) return;

  reg=(tree->dat->pat+node->attFldNo)->regTbl->reg+node->attPatNo;
  mssPMMLextensionStart("MUSASHI","patternPredicate",NULL,0,fpw);
  tag=mssInitXmlTag("X-PatternPredicate",NULL);
  mssAddXmlTagAttributeStr(tag,"x-field",MssFlds2name(fpat,node->attFldNo),NULL);
  if(reg->type==0){
    if(trl=='m'){
      mssAddXmlTagAttributeStr(tag,"x-operator","substring",NULL);
    }else{
      mssAddXmlTagAttributeStr(tag,"x-operator","notSubstring",NULL);
    }
  }else{
    if(trl=='m'){
      mssAddXmlTagAttributeStr(tag,"x-operator","notSubsequence",NULL);
    }else{
      mssAddXmlTagAttributeStr(tag,"x-operator","subsequence",NULL);
    }
  }
  regStr=getRegStr(reg);
  mssAddXmlTagAttributeStr(tag,"x-value",regStr,NULL);
  mssFree(regStr);
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlEmptyTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssPMMLextensionEnd(fpw);
}

/**
 * # FUNCTION #
 * 数値属性のノード書き出し
 */
void writeNumPMML(
  struct Tree *tree,
  struct Node *node,
  char trl,          /* top or match or unmatch */
  struct mssFPW *fpw){

  if(node->nodeType==0 || node->pruned==1) return;

  if(trl=='m'){
    mssPMMLsimplePredicateNumEmpty( MssFlds2name(fnum,node->attFldNo),
                                    "lessOrEqual", &node->numTH, fpw);
  }else{
    mssPMMLsimplePredicateNumEmpty( MssFlds2name(fnum,node->attFldNo),
                                    "greaterThan", &node->numTH, fpw);
  }
}

/**
 * # FUNCTION #
 * カテゴリ属性のノード書き出し
 */
void writeCatPMML(
  struct Tree *tree,
  struct Node *node,
  char trl,          /* top or match or unmatch */
  struct mssFPW *fpw){

  struct Category *catAdd;
  char **str;
  int  cnt;
  int i;

  if(node->nodeType==0 || node->pruned==1) return;

  if(trl=='m'){
    mssPMMLsimpleSetPredicateStart( MssFlds2name(fcat,node->attFldNo),
                                    "isIn", fpw);
  }else{
    mssPMMLsimpleSetPredicateStart( MssFlds2name(fcat,node->attFldNo),
                                    "isNotIn", fpw);
  }


  catAdd=tree->dat->cat+node->attFldNo;
  cnt=0;
  for(i=0; i<catAdd->cnt; i++){
     if(node->catList[i]=='M') cnt++;
  }
  str=mssMalloc(sizeof(char *)*cnt,"writeCatPMML");

  cnt=0;
  for(i=0; i<catAdd->cnt; i++){
     if(node->catList[i]=='M') *(str+cnt++)=*(catAdd->valName+i);
  }
  mssPMMLarrayStr(&cnt, str, fpw);
  mssPMMLsimpleSetPredicateEnd(fpw);
  mssFree(str);
}

/**
 * # FUNCTION #
 * 決定木のノードを表示する再帰関数(PMML)
 */
static void showNodePMML(
  struct Tree *tree,
  struct Node *node,
  char trl,          /* top or match or unmatch */
  struct mssFPW *fpw){

  char *clsVal;


  /* <node score="will play" recordCount="14"> */
  clsVal = tree->dat->cls->str[node->cls];
  mssPMMLnodeStart(clsVal,&node->trnCnt.total,fpw);


  /* <predicate> */
  if(trl=='t'){
    mssPMMLtrueEmpty(fpw);
  }else{
    switch(node->parent->attType){
      case 0: writePatPMML(tree,node->parent,trl,fpw); break; /*pat*/
      case 1: writeNumPMML(tree,node->parent,trl,fpw); break; /*num*/
      case 2: writeCatPMML(tree,node->parent,trl,fpw); break; /*cat*/
    }
  }

  if(node->nodeType==0 || node->pruned==1){
    mssPMMLnodeEnd(fpw);
    return;
  }

  showNodePMML(tree,node->mNode,'m',fpw);
  showNodePMML(tree,node->uNode,'u',fpw);
  mssPMMLnodeEnd(fpw);

}


/**
 * # FUNCTION #
 * PMMLフォーマットで決定木を出力
 */
void showTreePMML(struct Tree *tree,struct Cost *cost, struct mssFPW *fpw){

  struct mssXmlTag *tag;
  struct Map *map;
  char **str;
  int cnt;
  int ival;
  int i,j,k;

  /* PMML出力初期化*/
  mssPMMLinit();

  /* <?xml version="1.0" encoding="EUC-JP"?> */
  mssWriteXmlDeclaration( MssXmlDefVer, MssXmlDefEnc ,fpw);
  mssPV.outRetCnt++;

  /* <PMML version="2.0"> */
  mssPMMLpmmlStart("2.0",fpw);

  /*   <Header copyright="MUSASHI" description="binary tree model"/> */
  mssPMMLheaderStart("MUSASHI",NULL,fpw);

  /*     <Application name="xtclassify" version="1.1"/> */
  mssPMMLapplicationEmpty(mssGV.comHelp->name,mssGV.comHelp->version,fpw);

  /*     <Timestamp>10 Jun 2003 13:16:15</Timestamp> */
  mssPMMLtimestamp(fpw);

  /*   </Header> */
  mssPMMLheaderEnd(fpw);


  /*   <DataDictionary numberOfFields="5">*/
  ival=optPAT.cnt+optNUM.cnt+optCAT.cnt+optCLS.cnt;
  mssPMMLdataDictionaryStart(&ival,fpw);

  /*     <DataField name="温度" optype="continuous">*/
  for(i=0; i<fnum->cnt; i++){
    mssPMMLdataFieldStart(MssFlds2name(fnum,i),NULL,"continuous",NULL,NULL,0,fpw);
    mssPMMLvalueStart(MssNullStr,NULL,"missing",1,fpw);
    mssPMMLdataFieldEnd(fpw);
  }

  /*     <DataField name="天気" optype="categorical">*/
  for(i=0; i<fcat->cnt; i++){
    mssPMMLdataFieldStart(MssFlds2name(fcat,i),NULL,"categorical",NULL,NULL,0,fpw);
    mssPMMLvalueStart(MssNullStr,NULL,"missing",1,fpw);
    mssPMMLdataFieldEnd(fpw);
  }

  /*     <DataField name="パターン" optype="categorical">*/
  for(i=0; i<fpat->cnt; i++){
    mssPMMLdataFieldStart(MssFlds2name(fpat,i),NULL,"categorical",NULL,NULL,0,fpw);
    mssPMMLextensionStart("MUSASHI","optype","pattern",1,fpw);
    mssPMMLvalueStart(MssNullStr,NULL,"missing",1,fpw);
    mssPMMLdataFieldEnd(fpw);
  }

  /*     <DataField name="ゴルフ" optype="categorical">*/
  for(i=0; i<fcls->cnt; i++){
    mssPMMLdataFieldStart(MssFlds2name(fcls,i),NULL,"categorical",NULL,NULL,1,fpw);
  }

  /*   </DataDictionary>*/
  mssPMMLdataDictionaryEnd(fpw);

  /*<TreeModel functionName="classification">*/
  mssPMMLtreeModelStart(NULL,"classification",NULL,"binarySplit",fpw);

  /*<MiningSchema>*/
  mssPMMLminingSchemaStart(fpw);
  mssPMMLminingFields(fnum,NULL,NULL,NULL,NULL,NULL,"asMean",fpw);
  mssPMMLminingFields(fcat,NULL,NULL,NULL,NULL,NULL,"asMean",fpw);
  for(i=0; i<fpat->cnt; i++){
    mssPMMLminingFieldStart(MssFlds2name(fpat,i),NULL,NULL,NULL,NULL,NULL,NULL,0,fpw);

    mssPMMLextensionStart("MUSASHI","patternIndex",NULL,0,fpw);
    map=(tree->dat->pat+i)->map;
    for(j=0; j<map->idxSiz; j++){
      /*</X-PatternIndex x-index="1">*/
      tag=mssInitXmlTag("X-PatternIndex",NULL);
      mssAddXmlTagAttributeInt(tag,"x-index",j+1,NULL);
      mssPV.indentLevel++;
      mssPMMLindent(fpw);
      mssWriteXmlStartTag(tag,NULL,fpw);
      mssWriteRet(fpw); mssPV.outRetCnt++;

      /*<Array n="3" type="string">aa bb cc</Array>*/
      /*一つのインデックスに含まれるアルファベットの件数をカウント*/
      cnt=0;
      for(k=0; k<map->alpSiz; k++){
        if(*(map->idx+k) == (usint)(j+1) ) cnt++;
      }
      /*一つのインデックスに含まれるアルファベットのアドレスを登録*/
      str=mssMalloc(sizeof(char *)*cnt,"showTreePMML");
      cnt=0;
      for(k=0; k<map->alpSiz; k++){
        if(*(map->idx+k) == (usint)(j+1) ){
          *(str+cnt++)=*(map->numAlp+k);
        }
      }
      mssPMMLarrayStr(&cnt, str, fpw);
      mssFree(str);

      /*</X-PatternIndex>*/
      mssPMMLindent(fpw);
      mssWriteXmlEndTag(tag,NULL,fpw);
      mssWriteRet(fpw); mssPV.outRetCnt++;
      mssPV.indentLevel--;
    }
    mssPMMLextensionEnd(fpw);
    mssPMMLminingFieldEnd(fpw);
  }
  mssPMMLminingFields(fcls,"predicted",NULL,NULL,NULL,NULL,NULL,fpw);
  mssPMMLminingSchemaEnd(fpw);

  showNodePMML(tree,tree->topNode,'t',fpw);

  mssPMMLtreeModelEnd(fpw);

  /* </PMML> */
  mssPMMLpmmlEnd(fpw);

  mssGV.outCnt=mssPV.outRetCnt;
}

/**
 * # FUNCTION #
 * テキストで決定木を出力
 *  1.Command Line
 *  2.Cost Matrix
 *  3.Alphabet-Index
 *  4.Decision Tree
 *  5.Classification Table
 *  6.Summary
 */
void showTree(struct Tree *tree,struct Cost *cost, struct mssFPW *fpw){
  int i,j,k;
  int firstWrite;
  struct Map *map;

  j=0;
  mssWriteStr("=================================\n",fpw);
  mssWriteStr("RESULT on xtclassify\n",fpw);
  mssWriteStr("=================================\n",fpw);
  mssWriteStr("[Command Line]\n",fpw);

  for(i=0; i<mssGV.argc; i++){
    mssWriteStr(*(mssGV.argv+i),fpw);
    mssWriteStr(" ",fpw);
  }
  mssWriteRet(fpw);
  mssWriteRet(fpw);

  if(optCST.set){
    showCost(cost,fpw);
    mssWriteRet(fpw);
  } 

  if(optPAT.set){
    mssWriteStr("[Random Seed]\n",fpw);
    mssWriteInt(usedRandSeed,fpw);
    mssWriteRet(fpw);
    mssWriteRet(fpw);

    mssWriteStr("[Alphabet-Index]\n",fpw);
    for(i=0; i<tree->dat->patCnt; i++){
      map=(tree->dat->pat+i)->map;
      mssWriteStr("Field Name: $",fpw);
      mssWriteStr(MssFlds2name(fpat,i),fpw);
      mssWriteRet(fpw);

      for(j=0;j<map->idxSiz;j++){
        mssWriteStr("Index[",fpw);
        mssWriteInt(j+1,fpw);
        mssWriteStr("]={",fpw);
        firstWrite=1;
        for(k=0; k<map->alpSiz; k++){
          if(*(map->idx+k) == (usint)(j+1) ){
            if(!firstWrite) mssWriteStr(",",fpw);
            else            firstWrite=0;
            mssWriteStr("\"",fpw);
            mssWriteStr(*(map->numAlp+k),fpw);
            mssWriteStr("\"",fpw);
          }
        }
        mssWriteStr("}\n",fpw);
      }
    }    
    mssWriteStr("\n",fpw);
  }

  mssWriteStr("[Decision Tree]\n",fpw);
  showNode(tree,tree->topNode,'t',fpw);
  mssWriteRet(fpw);
  mssWriteStr("numberOfLeaves=",fpw);
  mssWriteInt(tree->leafCnt,fpw);
  mssWriteRet(fpw);
  mssWriteStr("deepestLevel=",fpw);
  mssWriteInt(tree->deepest,fpw);
  mssWriteRet(fpw);
  mssWriteRet(fpw);

  mssWriteStr("[Confusion Matrix]\n",fpw);
  mssWriteStr("## TRAINING DATA ##\n",fpw);
  showClsTbl(tree->trnRsl,tree->dat,fpw);
  mssWriteRet(fpw);

  if(optTST.set){
    mssWriteStr("## TEST DATA ##\n",fpw);
    showClsTbl(tree->tstRsl,tree->dat,fpw);
    mssWriteRet(fpw);
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 決定木生成API
 *   1.decisionTree
 *   2.updateTree
 *   3.freeTree
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 決定木生成メインルーチン
 */
struct Tree *decisionTree(
  struct Data *trn,   /*トレーニングデータ*/
  struct Data *tst,   /*テストデータ*/
  struct Cost *cost){ /*コスト表*/

  struct Tree   *tree;
  struct Pointer pointer; /*ツリー生成時に利用するattTblのインデックス*/
  int i;

  tree = mssCalloc(sizeof(struct Tree),"tree");

  /*pointerをつける*/
  pointer.recCnt=trn->cnt;
  pointer.recNo =mssCalloc(trn->cnt * sizeof(int),"decTreePointer");
  pointer.weight=mssCalloc(trn->cnt * sizeof(double),"decTreePointer");
  for(i=0; i<trn->cnt; i++){
    *(pointer.recNo+i)=i;
    *(pointer.weight+i)=1;
  }
  tree->topNode   = makeTree(trn,cost,&pointer,NULL);
  mssFree(pointer.recNo);
  mssFree(pointer.weight);

  /*サンプルデータ構造体のコピー(実際はmapHash,mapTbl,cat->catListのみコピー)*/
  tree->dat=cpyDatMapReg(trn);

  /*枝苅り*/
  pruneTree(tree,cost);

  /*ツリー構造体の各種変数のセット*/
  setLevel(tree->topNode,tree,0);

  /*結果表の作成(トレーニング)*/
  tree->trnRsl=setClsTbl( trn, tree);
  calClsTbl(tree->trnRsl,cost); /*各種合計値、指標を計算*/

  /*結果表の作成(テスト)*/
  tree->tstRsl=setClsTbl( tst, tree);
  calClsTbl(tree->tstRsl,cost); /*各種合計値、指標を計算*/

  return(tree);
}

/**
 * # FUNCTION #
 * ツリー構造体の領域解放
 */
void freeTree(struct Tree *tree){
  struct Node **nodeTbl=NULL;
  int           nodeCnt=0;
  int i;

  if(tree==NULL)return;
  nodeTbl=freeTreeTbl(tree->topNode,nodeTbl,&nodeCnt);

  for(i=0; i<nodeCnt; i++){
   mssFree(*(nodeTbl+i));
  }
  mssFree(nodeTbl);
  mssFree(tree->topNode);
  freeDatMapReg(tree->dat);
  mssFree(tree->trnRsl);
  mssFree(tree->tstRsl);
  mssFree(tree);
}

/**
 * # FUNCTION #
 * よいツリーの更新
 */
struct Tree *updateTree(
  struct Tree *tree,
  struct Tree *betterTree){

  /*一方がNULLの判定*/
  if(tree==NULL){
    return(betterTree);
  }else if(betterTree==NULL){
    return(tree);

  /*コストの比較*/
  }else if(tree->trnRsl->tCst < betterTree->trnRsl->tCst){
    freeTree(betterTree);
    return(tree);
  }else if(tree->trnRsl->tCst > betterTree->trnRsl->tCst){
    freeTree(tree);
    return(betterTree);

  /*コストが同じ時はサイズが小さい方が勝ち*/
  }else{
    if(tree->leafCnt < betterTree->leafCnt){
      freeTree(betterTree);
      return(tree);
    }else{
      freeTree(tree);
      return(betterTree);
    }
  }
  return(NULL); /*to avoid warning*/
}

/**
 * # FUNCTION #
 * 途中経過表示
 */
void showProgressTree(struct Data *dat, struct Tree *tree){
  int i;
  char buf[MssRecordMaxLen];
  char *num[]={"0","1","2","3","4","5","6","7","8","9"};
  usint *str;

  buf[0]='\0';
  strcat(buf,"index:");
  for(i=0; i<dat->patCnt; i++){
    str=(dat->pat+i)->map->idx;
    while(*str!=0){
      strcat(buf,num[*str++]);
    }
    strcat(buf," ");
  }
  mssShowMsg("%scost=%f geoMean=%f leafCnt=%d",buf,tree->tstRsl->tCst,tree->tstRsl->geoMean,tree->leafCnt);
}

