#include <musashi.h>
#include <dasg.h>

extern struct mssGlobalVariables mssGV;

/*============================================================================*/
/* DASG(Directed Acyclic Subsequence Graph)の作成                             */
/*  algorithm by H.Hosino, A.Shinohara, at el., "Online construction of subsequence automata for multiple texts",DOI Technical Report,2000.                   */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/* マクロ                                                                     */
/*----------------------------------------------------------------------------*/
/*direction変数のstateが"s",アルファベットが"a"の要素を返すマクロ*/
#define Direction(s,a) (*(*(dasg->direction+(s))+(a)))
/*nsal変数のstateが"s",アルファベットが"a"の要素を返すマクロ*/
#define Nsal(s,a)      (*(*(dasg->nsal     +(s))+(a)))
#define IniStat 0  /*初期stateの番号は0*/
#define ErrStat 1  /*エラーstateの番号は1*/

/*============================================================================*/
/* サブ関数                                                                   */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/* directionのセット                                                          */
/* state="s",alphabet="a"のarcを"state"に設定する                             */
/* 同時に"state"に入るarcの本数をカウントしている                             */
/*----------------------------------------------------------------------------*/
static void setDirection(struct DASG *dasg,int s,int a,int state){
  (*(dasg->refCnt+state))++;
  (*(dasg->refCnt+Direction(s,a)))--;
  Direction(s,a)=state;
}

/*----------------------------------------------------------------------------*/
/* StateList構造体の初期化と開放                                              */
/*----------------------------------------------------------------------------*/
static struct StateList *initStateList(){
  struct StateList *sl;
  sl=mssCalloc(sizeof(struct StateList),"initStateList");
  return(sl);
}

static void freeStateList(struct StateList *sl){
  mssFree(sl->state);
  mssFree(sl);
}

/*----------------------------------------------------------------------------*/
/* StateList構造体にstateを追加する(return 1)。                               */
/* ただし、既にあれば何もせず0を返す。                                        */
/*----------------------------------------------------------------------------*/
static int addList(struct StateList *sl, int state){
  int i;
  for(i=0; i<sl->cnt; i++) if(*(sl->state+i) == state) return(0);
  sl->state=mssRealloc(sl->state,sizeof(int)*(sl->cnt+1),"addList");
  *(sl->state+sl->cnt)=state;
  sl->cnt++;
  return(1);
}

/*----------------------------------------------------------------------------*/
/* state="s",alphabet="a"のnsalに"state"を追加登録する。                      */
/* 同時に、state="s"のnsaCnt(nsaの件数)を更新する                             */
/*----------------------------------------------------------------------------*/
static void addNonSolidArcList(struct DASG *dasg,int s, int a, int state){
  if( addList(Nsal(s,a),state) ) (*(dasg->nsaCnt+s))++;
}

/*----------------------------------------------------------------------------*/
/* state="s",alphabet="a"のnsalを初期化する。                                 */
/* 同時に、state="s"のnsaCnt(nsaの件数)を更新する                             */
/*----------------------------------------------------------------------------*/
static void delNonSolidArcList(struct DASG *dasg,int s, int a){
  *(dasg->nsaCnt+s)-=Nsal(s,a)->cnt;
  Nsal(s,a)=initStateList();
}

/*----------------------------------------------------------------------------*/
/* DASGに一つの文字列を追加し終えたときの各種初期化                           */
/*----------------------------------------------------------------------------*/
static void extendDimension(struct DASG *dasg){
  int a,i;
  int r,s;
  for(i=0; i<dasg->stateCnt; i++){
    *(dasg->nsaCnt+i)=0;
  }
  for(a=0; a<dasg->alpCnt; a++){
    for(s=0; s<(*(dasg->targetStates+a))->cnt; s++){
      i=*((*(dasg->targetStates+a))->state+s);
      freeStateList(Nsal(i,a));
      Nsal(i,a)=initStateList();
    }
    r=Direction(IniStat,a);
    addNonSolidArcList(dasg,r,a,IniStat);
    freeStateList(*(dasg->targetStates+a));
    *(dasg->targetStates+a)=initStateList();
    addList(*(dasg->targetStates+a),r);
  }
  (*(dasg->match+IniStat))++;
}

/*----------------------------------------------------------------------------*/
/* alphabetテーブルから、与えられた文字(chr)の番号をcにセットする。           */
/* 新たに追加する時は1を返し、既に存在すれば0を返す                           */
/* setAlphaNoとgetNo4Alphaは効率のよいアルゴリズムに変更すること              */
/*----------------------------------------------------------------------------*/
static int setAlphaNo(struct DASG *dasg, struct mssHashNode *chr, int *c){
  int i;
  for(i=0; i<dasg->alpCnt; i++){
    if( *(dasg->alpTbl+i)==chr ){
      *c=i;
      return(0);
    }
  }

  /*ここまでくるということは、新規文字*/
  dasg->alpTbl=mssRealloc(dasg->alpTbl,sizeof(int)*(dasg->alpCnt+1),"sa");
  *(dasg->alpTbl+dasg->alpCnt)=chr;
  *c=i;
  return(1);
}

/*----------------------------------------------------------------------------*/
/* alphabetテーブルから、与えられたalphabetからその番号を返す                 */
/* ないalphabetについては-1を返す                                             */
/*----------------------------------------------------------------------------*/
static int getNo4Alpha(struct DASG *dasg, struct mssHashNode *chr){
  int i;
  for(i=0; i<dasg->alpCnt; i++){
    if( *(dasg->alpTbl+i)==chr ){
      return(i);
    }
  }
  return(-1);
}


/*----------------------------------------------------------------------------*/
/* 新しいalphabetが出現したときにDASGの各種変数領域を拡張し、初期化する       */
/*----------------------------------------------------------------------------*/
static void extendAlp(struct DASG *dasg){
  int i;

  /*--- direction領域拡張(State×Alpha)*/
  for(i=0; i<dasg->stateCnt; i++){
    *(dasg->direction+i)
      =mssRealloc(*(dasg->direction+i),sizeof(int)*(dasg->alpCnt+1),"exAl");
    Direction(i,dasg->alpCnt)=ErrStat; /*新しい文字はErrStateに接続*/
  }

  /*--- refCnt領域拡張(State)*/
  /*Stateに接続されるArcはdasg->stateCnt分の新alphabet分増加する*/
  *(dasg->refCnt+ErrStat)+=dasg->stateCnt;

  /*--- nsaCnt領域拡張(State)*/
  /*refCntと同様。ただしErrStatからErrStat分は引く(-1)*/
  *(dasg->nsaCnt+ErrStat)+=dasg->stateCnt-1;

  /*--- nsal領域拡張(State×Alpha)*/
  /*全stateの新alphabetにStateListを追加し、
    全stateからの"b"のErrStateへのarcはnsalに追加(ErrState以外)。
    そして、ErrStateをtargetStates[newChr]に追加*/
  for(i=0; i<dasg->stateCnt; i++){
    *(dasg->nsal+i)=mssRealloc(*(dasg->nsal+i),
       sizeof(struct StateList)*(dasg->alpCnt+1),"exAl");
    Nsal(i,dasg->alpCnt)=initStateList();
  }
  for(i=0; i<dasg->stateCnt; i++){
    if(i != ErrStat) addList(Nsal(ErrStat,dasg->alpCnt),i);
  }

  /*--- targetStates領域拡張(Alpha)*/
  /*newChrのtargetにErrStateを追加*/
  dasg->targetStates=mssRealloc(dasg->targetStates,
                       sizeof(struct StateList *)*(dasg->alpCnt+1),"exAl");
  *(dasg->targetStates+dasg->alpCnt)=initStateList();
  addList(*(dasg->targetStates+dasg->alpCnt),ErrStat);
}

/*----------------------------------------------------------------------------*/
/* 新しいstateを追加するときにDASGの各種変数領域を拡張し、初期化する          */
/*----------------------------------------------------------------------------*/
static void extendState(struct DASG *dasg){
  int j;

  /*--- direction領域拡張(State×Alpha)*/
  dasg->direction=mssRealloc(dasg->direction,sizeof(int *)*
                    (dasg->stateCnt+1),"exSt");
  *(dasg->direction+dasg->stateCnt)=mssMalloc(sizeof(int)*
                    (dasg->alpCnt),"exSt");
  for(j=0; j<dasg->alpCnt; j++) Direction(dasg->stateCnt,j)=0;

  /*--- match領域拡張(State)*/
  dasg->match=mssRealloc(dasg->match,sizeof(int)*(dasg->stateCnt+1),"exSt");
  *(dasg->match+dasg->stateCnt)=0;

  /*--- refCnt領域拡張(State)*/
  dasg->refCnt=mssRealloc(dasg->refCnt,sizeof(int)*(dasg->stateCnt+1),"exSt");
  *(dasg->refCnt+dasg->stateCnt)=0;

  /*--- nsaCnt領域拡張(State)*/
  dasg->nsaCnt=mssRealloc(dasg->nsaCnt,sizeof(int)*(dasg->stateCnt+1),"exSt");
  *(dasg->nsaCnt+dasg->stateCnt)=0;

  /*--- nsal領域拡張(State×Alpha)*/
  dasg->nsal=mssRealloc(dasg->nsal,sizeof(struct StateList *)*
              dasg->alpCnt*(dasg->stateCnt+1),"exSt");
  *(dasg->nsal+dasg->stateCnt)=mssMalloc(sizeof(struct StateList)*
                    (dasg->alpCnt),"exSt");
  for(j=0; j<dasg->alpCnt; j++) Nsal(dasg->stateCnt,j)=initStateList();
}

/*----------------------------------------------------------------------------*/
/* DASGに一文字追加する                                                       */
/*----------------------------------------------------------------------------*/
static void appendChar(struct DASG *dasg, int c){
  struct StateList *lastTargetStates;
  struct StateList **lastNonSolidArcs;
  int a,i,j,p;
  int r,s,t;
  int *tmpCnt;

  lastNonSolidArcs=mssMalloc(sizeof(struct StateList *)*dasg->stateCnt,"append");
  tmpCnt=mssMalloc(sizeof(int)*dasg->stateCnt,"append");

  lastTargetStates=*(dasg->targetStates+c);
  *(dasg->targetStates+c)=initStateList();

  for(s=0; s<lastTargetStates->cnt; s++){
    i=*(lastTargetStates->state+s);
    tmpCnt[i]=*(dasg->nsaCnt+i);
    lastNonSolidArcs[i]=Nsal(i,c);
    delNonSolidArcList(dasg,i,c);
  }

  for(s=0; s<lastTargetStates->cnt; s++){
    i=*(lastTargetStates->state+s);
    if(*(dasg->refCnt+i)==tmpCnt[i]){ /*既存のStateを使う*/
      p=i;
    }else{                            /*新たなStateを追加*/
      extendState(dasg);  /*新しいState用に領域を拡張*/
      p=dasg->stateCnt++;
    }

    for(a=0; a<dasg->alpCnt; a++){
      r=Direction(i,a);
      setDirection(dasg,p,a,r);
      addNonSolidArcList(dasg,r,a,p);
      addList(*(dasg->targetStates+a),r);
    }
    *(dasg->match+p)=*(dasg->match+i)+1;
    for(t=0; t<lastNonSolidArcs[i]->cnt; t++){
      j=*(lastNonSolidArcs[i]->state+t);
      setDirection(dasg,j,c,p);
    }
    freeStateList(lastNonSolidArcs[i]);
  }
  freeStateList(lastTargetStates);
  mssFree(lastNonSolidArcs);
  mssFree(tmpCnt);
}

/*============================================================================*/
/* PUBLIC関数                                                                 */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/* DASGに問い合わせ、サブシーケンス"s"が何件かを返す                          */
/*----------------------------------------------------------------------------*/
int DASGqueryCnt(struct DASG *dasg,struct mssHashNode **s, int n){
  int q,i;
  q=IniStat;
  for(i=0;i<n;i++){
    q=Direction(q,getNo4Alpha(dasg,*(s+i)));
  }
  return(*(dasg->match+q));
}

/*----------------------------------------------------------------------------*/
/* DASGを出力                                                                 */
/*----------------------------------------------------------------------------*/
void DASGshow(struct DASG *dasg){
  int i,j,k;

  printf("----------------------------\n");
  for(i=0; i<dasg->alpCnt; i++){
    printf("  targetStates(cnt:%d)=",(*(dasg->targetStates+i))->cnt);
    for(j=0; j<(*(dasg->targetStates+i))->cnt; j++){
      printf("%d,",*((*(dasg->targetStates+i))->state+j));
    }
    printf("\n");
  }

  for(i=0; i<dasg->stateCnt; i++){
    printf("state[%d]:\n",i);

    printf("  match=%d\n",*(dasg->match+i));
    printf("  cnt(nonSolid,all)=(%d,%d)\n",*(dasg->nsaCnt+i),*(dasg->refCnt+i));

    printf("  d=");
    for(j=0; j<dasg->alpCnt; j++){
      printf("%d,",Direction(i,j));
    }
    printf("\n");

    printf("  nonSolidArcList=");
    for(j=0; j<dasg->alpCnt; j++){
      for(k=0; k<Nsal(i,j)->cnt; k++){
        printf("%d,",*(Nsal(i,j)->state+k));
      }
      printf("; ");
    }
    printf("\n");
  }
}

/*----------------------------------------------------------------------------*/
/* DASGの初期化 (引数にアルファベットのサイズ)                                */
/*----------------------------------------------------------------------------*/
struct DASG *DASGinit(int size){
  struct DASG *dasg;

  dasg=mssMalloc(sizeof(struct DASG),"DASGinit");
  dasg->alpCnt      =0;
  dasg->alpTbl      =NULL;
  dasg->stateCnt    =2;
  dasg->direction   =mssCalloc(sizeof(int)*2,"main");
  dasg->match       =mssCalloc(sizeof(int)*2,"main");
  dasg->refCnt      =mssCalloc(sizeof(int)*2,"main");
  dasg->nsaCnt      =mssCalloc(sizeof(int)*2,"main");
  dasg->targetStates=NULL;
  dasg->nsal        =mssCalloc(sizeof(int)*2,"main");
  return(dasg);
}

/*----------------------------------------------------------------------------*/
/* DASGの開放                                                                 */
/*----------------------------------------------------------------------------*/
void DASGfree(struct DASG *dasg){
  int i,j;

  for(i=0; i<dasg->stateCnt; i++){
    for(j=0; j<dasg->alpCnt; j++){
      freeStateList(Nsal(i,j));
    }
  }

  for(i=0; i<dasg->alpCnt; i++){
    freeStateList(*(dasg->targetStates+i));
  }

  for(i=0; i<dasg->stateCnt; i++){
    mssFree(*(dasg->direction+i));
    mssFree(*(dasg->nsal+i));
  }
  mssFree(dasg->direction);
  mssFree(dasg->match);
  mssFree(dasg->refCnt);
  mssFree(dasg->nsaCnt);
  mssFree(dasg->targetStates);
  mssFree(dasg->nsal);
  mssFree(dasg->alpTbl);
  mssFree(dasg);
}

/*----------------------------------------------------------------------------*/
/* DASGに文字列を追加                                                         */
/*----------------------------------------------------------------------------*/
void DASGaddStr(struct DASG *dasg,struct mssHashNode **str, int n){
  int i;
  int c;
  for(i=0; i<n; i++){
    if( 1==setAlphaNo(dasg,*(str+i),&c) ){ /*オリジナルchrを番号cに変換*/
      extendAlp(dasg); /*新しい文字列ならば、各種変数の領域を拡大する*/
      dasg->alpCnt++;
    }
    appendChar(dasg,c);
  }

  extendDimension(dasg);
}

int *stateList; /*出力用のstateテーブル*/
int *alphaList; /*アルファベットテーブル*/
int statePnt=0; /*stateテーブルにおけるカレント位置*/
int outKeyCnt=0;
MssOptKEY *gKey;
char **gPnt;
/*----------------------------------------------------------------------------*/
/* DASGをXMLtable形式で出力                                                   */
/*----------------------------------------------------------------------------*/
void DASGwriteXtSub(struct DASG *dasg, int state,
                    int minDepth,int maxDepth,int support,struct mssFPW *fpw){
  int i,j;

  if(*(dasg->match+state) < support){
    return;
  }
  if(statePnt>maxDepth){ /*最小長より長い時に書き出し*/
    return;
  }
  if(statePnt>=minDepth){ /*最小長より長い時に書き出し*/

    /*書き出し*/
    for(i=0; i<statePnt; i++){
      for(j=0; j<gKey->cnt; j++){
        mssWriteStr(*(gPnt+MssFlds2num(gKey->flds,j)),fpw);
        mssWriteDlm(fpw);
      }
      mssWriteInt(outKeyCnt,fpw)      ; mssWriteDlm(fpw);
      mssWriteInt(i+1,fpw)            ; mssWriteDlm(fpw);
      mssWriteStr( (*(dasg->alpTbl+*(alphaList+i)))->str,fpw); mssWriteDlm(fpw);
      mssWriteInt(statePnt,fpw)      ; mssWriteDlm(fpw); /*深さ*/
      mssWriteInt(*(dasg->match+state),fpw)        ; mssWriteRet(fpw);
      mssGV.outCnt++;
    }
    outKeyCnt++;
  }

  for(j=0; j<dasg->alpCnt; j++){
    *(stateList+statePnt)=state;
    *(alphaList+statePnt)=j;
    statePnt++;
    DASGwriteXtSub(dasg,Direction(state,j),minDepth,maxDepth,support,fpw);
    statePnt--;
  }
}

void DASGwriteXt(struct DASG *dasg,int minDepth,int maxDepth,int support, char **pnt,MssOptKEY *key,struct mssFPW *fpw){

  stateList=mssCalloc(sizeof(int)*dasg->stateCnt,"DASGwriteXt");
  alphaList=mssCalloc(sizeof(int)*dasg->stateCnt,"DASGwriteXt");
  outKeyCnt=1;
  gPnt=pnt;
  gKey=key;
  DASGwriteXtSub(dasg,0,minDepth,maxDepth,support,fpw);
  mssFree(stateList);
  mssFree(alphaList);
}

