/*============================================================================*/
/* 変更履歴                                                                   */
/*----------------------------------------------------------------------------*/
/* 1.0 : 新しいAPIに対応 2003/07/19                                           */
/* 1.1 : -i は必須オプションに 2003/11/05                                     */
/*============================================================================*/

#include <musashi.h>
#include <xtkmeanHelp.h>
#include <stdlib.h>
#include <limits.h>
#include <float.h>

struct mssComHelp comHelp={
  "xtkmean",      /* コマンド名       */
  "1.1",          /* バージョン       */
  HELPT,          /* コマンドタイトル */
  HELPS,          /* 要約             */
  HELPE,          /* 利用例           */
  HELPR,          /* 参照コマンド     */
  HELPA,          /* 作者情報         */
  HELPB,          /* バグレポート情報 */
  HELPH           /* ホームページ     */
};


struct CalNum {
  MssValue accum;
  int   cnt;
};

/*入力データ関連*/
struct DataInfo {
  int    cnt;    /*データ行数*/
  MssValue *maxNum; /*数値項目の最大値*/
  MssValue *minNum; /*数値項目の最小値*/
  MssValue *rngNum; /*数値項目の最大値-最小値*/
  MssValue *sumNum; /*数値項目の合計値*/
  int   *cntNum; /*数値項目の件数(nullを省く)*/
  MssValue *avgNum; /*数値項目の平均値*/
};

struct Cluster {
  int cnt;               /*クラスタに属するレコード数*/
  struct CalNum *calNum; /*重心の計算用*/
  MssValue      *cenNum; /*重心*/
  struct DataInfo *data;
};

struct Sample {
  struct SmpRec {
    MssValue        *num; /*項目別数値*/
  } *rec;
  int recCnt;
  int numFldCnt;
  int catFldCnt;
  struct DataInfo *data;
};

extern struct mssGlobalVariables mssGV;

int totalFldCnt=0;
MssValue distance;
int kCnt;

/*----------------------------------------------------------------------------*/
/* クラスタ数                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptINT optCNT={
    OINT,   /* オプションタイプ                                             */
    "k",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    0,      /* デフォルト(数値として指定)                                   */
    2,      /* 最小値                                                       */
    50,     /* 最大値                                                       */
    CNTT,   /* このオプションのタイトル(Helpで表示)                         */
    CNTC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 数値項目                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLD optNUM={
    OFLD,   /* オプションタイプ                                             */
    "n",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    MssFieldMaxCnt, /* 指定可能な最大項目数                                 */
    "i",    /* 対象とする入力データのキーワード(GUIで利用)                  */
    1,      /* 正規表現を許可するかどうか(0:不可,1:可)                      */
    0,      /* 新項目名を指定できるかどうか(0:不可,1:可)                    */
    NULL,   /* 項目オプション(%以下)で指定可能な文字                        */
            /* ex) 指定不可の場合はNULL, "nr": "-f 項目名%rn"の指定可能     */
    NUMT,   /* このオプションのタイトル(Helpで表示)                         */
    NUMC,   /* このオプションのコメント(Helpで表示)                         */
    NUMF    /* フラグについての説明(Helpで表示)複数の場合はカンマで区切る   */
  };

/*----------------------------------------------------------------------------*/
/* 乱数の種                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptINT optSED={
    OINT,   /* オプションタイプ                                             */
    "S",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    -1,     /* デフォルト(数値として指定)                                   */
    -1,     /* 最小値                                                       */
    INT_MAX,/* 最大値                                                       */
    SEDT,   /* このオプションのタイトル(Helpで表示)                         */
    SEDC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 新項目名                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptSLS optFNM={
    OSLS,   /* オプションタイプ                                             */
    "a",    /* キーワード(複数文字は不可)                                   */
    2,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    NULL,   /* デフォルト(文字列)                                           */
    1,      /* カンマで区切られる要素数の最大値                             */
    1,      /* 各要素の文字列長の最小値                                     */
    MssFieldNameMaxLen,/* 各要素の文字列長の最大値                          */
    0,      /* 1:要素にコロンを指定できる,0:不可  ex) aaaa:xxxxx            */
    FNMT,   /* このオプションのタイトル(Helpで表示)                         */
    FNMC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 初期シードの計算方法                                                       */
/*----------------------------------------------------------------------------*/
  MssOptINT optINI={
    OINT,   /* オプションタイプ                                             */
    "d",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    2,      /* デフォルト(数値として指定)                                   */
    0,      /* 最小値                                                       */
    2,      /* 最大値                                                       */
    INIT,   /* このオプションのタイトル(Helpで表示)                         */
    INIC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* -s2の時のサンプル個数                                                      */
/*----------------------------------------------------------------------------*/
  MssOptINT optMUL={
    OINT,   /* オプションタイプ                                             */
    "m",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須, 2:XMLtableでのみ必須(txtでは無視)      */
    10,     /* デフォルト(数値として指定)                                   */
    0,      /* 最小値                                                       */
    100,    /* 最大値                                                       */
    MULT,   /* このオプションのタイトル(Helpで表示)                         */
    MULC    /* このオプションのコメント(Helpで表示)                         */
  };


/*----------------------------------------------------------------------------*/
/* 入力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptINF optINF={
    OINF,   /* オプションタイプ                                             */
    "i",    /* キーワード(複数文字は不可)                                   */
    1,      /* 0:オプション, 1:必須                                         */
    1,      /* 指定可能の最大ファイル数                                     */
    0,      /*1:file not foundのエラーで終了しない 0:する                   */
    INFT,   /* このオプションのタイトル(Helpで表示)                         */
    INFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 出力ファイル                                                               */
/*----------------------------------------------------------------------------*/
  MssOptOTF optOTF={
    OOTF,   /* オプションタイプ                                             */
    "o",    /* キーワード(複数文字は不可)                                   */
    0,      /* 0:オプション, 1:必須                                         */
    OTFT,   /* このオプションのタイトル(Helpで表示)                         */
    OTFC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* 圧縮出力                                                                   */
/*----------------------------------------------------------------------------*/
  MssOptFLG optZIP={
    OFLG,   /* オプションタイプ                                             */
    "z",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    ZIPT,   /* このオプションのタイトル(Helpで表示)                         */
    ZIPC    /* このオプションのコメント(Helpで表示)                         */
  };

/*----------------------------------------------------------------------------*/
/* plain text                                                                 */
/*----------------------------------------------------------------------------*/
  MssOptFLG optTXT={
    OFLG,   /* オプションタイプ                                             */
    "t",    /* キーワード(複数文字は不可)                                   */
    0,      /* デフォルト(基本的には0) 常にonにしたいときは1にする          */
    TXTT,   /* このオプションのタイトル(Helpで表示)                         */
    TXTC    /* このオプションのコメント(Helpで表示)                         */
  };

void *opt[]={&optCNT,&optNUM,&optSED,&optINI,&optMUL,&optFNM,
             &optINF,&optOTF,&optZIP,&optTXT,NULL};

struct mssFields *fnum; /*項目構造体*/


/*============================================================================*/
/* 関数                                                                       */
/*============================================================================*/
static void valPrint(MssValue a){

  if(a.nul){
    printf("NULL");
    return;
  }

  switch(a.vType){
    case DBL: printf("%g",a.v.d); break;
    case INT: printf("%d",a.v.i); break;
    default:
    break;
  }
 
}

/*----------------------------------------------------------------------------*/
/*Cluster構造体の表示                                                         */
/*----------------------------------------------------------------------------*/
void showCluster(struct Cluster *cluster){
  int i,j;

  printf("----------------- showCluster\n");
  for(i=0; i<kCnt; i++){
    printf("c[%d] cnt=%d : ",i,(cluster+i)->cnt);
    for(j=0; j<fnum->cnt; j++){
      valPrint(*((cluster+i)->cenNum+j));
      printf("(");
      valPrint(((cluster+i)->calNum+j)->accum);
      printf(",%d)",((cluster+i)->calNum+j)->cnt);
      printf(")");
    }
    printf("\n");
  }
}
 
/*----------------------------------------------------------------------------*/
/*Sample構造体の表示                                                          */
/*----------------------------------------------------------------------------*/
void showSample(struct Sample *sample){
  int i,j;

  printf("---------------- showSample\n");
  for(i=0; i<sample->recCnt; i++){
    printf("%4d : ",i);
    for(j=0; j<sample->numFldCnt; j++){
      if( (*((sample->rec+i)->num+j)).nul ){
        printf( "* " );
      }else{
        printf( "%g ",(*((sample->rec+i)->num+j)).v.d );
      }
    }
    printf("\n");
  }
  printf("----------------\n");
}



/*----------------------------------------------------------------------------*/
/* 数値間の距離を計算                                                         */
/*----------------------------------------------------------------------------*/
MssValue disNum(MssValue x, MssValue y){
  MssValue rsl;
  double sub;

  mssVinit(&rsl,DBL);

  if(x.nul || y.nul ){
    rsl.nul=1;
  }else{
    rsl.nul=0;
    /*ユークリッド距離絶対値*/
    sub=x.v.d - y.v.d;
    if(sub<0) rsl.v.d=sub*(-1);
    else      rsl.v.d=sub;

    /*ユークリッド距離２乗*/
    /*rsl.v.d = (x.v.d - y.v.d)*(x.v.d - y.v.d);*/
  }
  return(rsl);
}

/*----------------------------------------------------------------------------*/
/* 各項目の距離の平均を計算(NULLは計算に入れない)                             */
/*----------------------------------------------------------------------------*/
MssValue disAvg(MssValue *disFld, int cnt){
  double sum=0;
  int j=0;
  MssValue rsl;
  int i;

  mssVinit(&rsl,DBL);

  /*総和*/
  sum=0;
  for(i=0;i<cnt;i++){
    if(! (*(disFld+i)).nul){
      sum+=(*(disFld+i)).v.d;
      j++;
    }
  }

  if(j==0){
    rsl.nul=1;
  }else{
    rsl.nul=0;
    rsl.v.d=sum/(double)j;
  }
  return(rsl);
}

static MssValue norm(MssValue num, MssValue min, MssValue rng){

  return( mssVdiv(mssVsub(num,min),rng) );

}

/*----------------------------------------------------------------------------*/
/* m番目のサンプル と n番目のサンプルとの距離                                */
/*----------------------------------------------------------------------------*/
MssValue calDistanceSmpSmp( struct Sample *sample, int m, int n ){

  MssValue *disFld;
  MssValue  sum;
  int i,j;
  MssValue a,b;
  MssValue *minNum;
  MssValue *rngNum;

  mssVinit(&sum,DBL);
  mssVinit(&a  ,DBL);
  mssVinit(&b  ,DBL);

  minNum=sample->data->minNum;
  rngNum=sample->data->rngNum;

  disFld=mssCalloc(sizeof(MssValue)*fnum->cnt,"near");
  for(i=0; i<fnum->cnt; i++) mssVinit(disFld+i,DBL);

  j=0;
  /*数値項目*/
  for(i=0; i<fnum->cnt; i++){
    a=norm(*((sample->rec+m)->num+i),*(minNum+i),*(rngNum+i));
    b=norm(*((sample->rec+n)->num+i),*(minNum+i),*(rngNum+i));
    *(disFld+j++)=disNum(a,b);
  }

  /*カテゴリ項目*/

  /*結合距離の計算*/
  sum=disAvg(disFld,j);
  mssFree(disFld);

  return( sum );
}


/*----------------------------------------------------------------------------*/
/* k番目のcluster と n 番名のサンプルとの距離                                 */
/*----------------------------------------------------------------------------*/
MssValue calDistanceClsSmp(
  struct Cluster *cluster, int k,
  struct Sample  *sample,  int n){

  MssValue *disFld;
  MssValue  sum;
  int i,j;
  MssValue a,b;
  MssValue *minNum;
  MssValue *rngNum;

  mssVinit(&sum,DBL);
  mssVinit(&a  ,DBL);
  mssVinit(&b  ,DBL);

  minNum=sample->data->minNum;
  rngNum=sample->data->rngNum;

  disFld=mssCalloc(sizeof(MssValue)*fnum->cnt,"near");
  for(i=0; i<fnum->cnt; i++) mssVinit(disFld+i,DBL);

  j=0;
  /*数値項目*/
  for(i=0; i<fnum->cnt; i++){
    a=norm(*((cluster+k)->cenNum+i) ,*(minNum+i),*(rngNum+i));
    b=norm(*((sample->rec+n)->num+i),*(minNum+i),*(rngNum+i));
    *(disFld+j++)=disNum(a,b);
  }

  /*カテゴリ項目*/

  /*結合距離の計算*/
  sum=disAvg(disFld,j);
  mssFree(disFld);

  return( sum );
}

/*----------------------------------------------------------------------------*/
/*与えられた行が最も近いクラスタ番号を返す                                    */
/*  クラスタが全てNULLの場合,もしくはデータ項目が全てNULLの場合,もしくは      */
/*  クラスタの項目とデータの項目がうまくNULLでかみあった場合は-1を返すことに  */
/*  なる。すなわち、近いクラスタは分からないということになる                  */
/*----------------------------------------------------------------------------*/
int nearestCluster(struct Cluster *cluster, struct mssFldRec *fr){
  MssValue  distanceTmp;
  MssValue *disFld;
  int  c,j,i,k;
  char *s;
  MssValue a,b;
  MssValue *minNum;
  MssValue *rngNum;
  MssValue *avgNum;

  k=-1;

  mssVinit(&distanceTmp,DBL);
  mssVinit(&distance,DBL);
  distance.v.d=9999;
  mssVinit(&a  ,DBL);
  mssVinit(&b  ,DBL);

  minNum=cluster->data->minNum;
  rngNum=cluster->data->rngNum;
  avgNum=cluster->data->avgNum;

  disFld=mssCalloc(sizeof(MssValue)*fnum->cnt,"near");
  for(i=0; i<fnum->cnt; i++) mssVinit(disFld+i,DBL);

  for(c=0; c<kCnt; c++){
    j=0;

    /*数値項目*/
    for(i=0; i<fnum->cnt; i++){
      a=norm(*((cluster+c)->cenNum+i),*(minNum+i),*(rngNum+i));
      s=*(fr->pnt+MssFlds2num(fnum,i));
      if(*s=='*') { b=*(avgNum+i); }
      else        {b.nul=0;b.v.d=atof(s);}
      b=norm(b,*(minNum+i),*(rngNum+i));
      *(disFld+j++)=disNum(a,b);
    }

    /*カテゴリ項目*/

    /*結合距離の計算*/
    distanceTmp=disAvg(disFld,j);

    /*距離が小さければ更新*/
    if( mssVcmpOpe(distanceTmp, OPE_LT, distance) ){
      distance=distanceTmp;
      k=c;
    }
  }

  mssFree(disFld);
  return(k);
}

/*----------------------------------------------------------------------------*/
/*与えられた行が最も近いクラスタ番号を返す(sample用)                          */
/*  クラスタが全てNULLの場合,もしくはデータ項目が全てNULLの場合,もしくは      */
/*  クラスタの項目とデータの項目がうまくNULLでかみあった場合は-1を返すことに  */
/*  なる。すなわち、近いクラスタは分からないということになる                  */
/*----------------------------------------------------------------------------*/
int nearestClusterSmp(struct Cluster *cluster, struct SmpRec *rec){
  MssValue  distanceTmp;
  MssValue *disFld;
  int  c,j,i,k;
  MssValue a,b;
  MssValue *minNum;
  MssValue *rngNum;

  mssVinit(&distanceTmp,DBL);
  mssVinit(&distance,DBL);
  mssVinit(&a  ,DBL);
  mssVinit(&b  ,DBL);

  k=-1;
  distance.v.d=9999;

  minNum=cluster->data->minNum;
  rngNum=cluster->data->rngNum;

  disFld=mssCalloc(sizeof(MssValue)*fnum->cnt,"near");
  for(i=0; i<fnum->cnt; i++) mssVinit(disFld+i,DBL);

  for(c=0; c<kCnt; c++){
    j=0;

    /*数値項目*/
    for(i=0; i<fnum->cnt; i++){
      a=norm(*((cluster+c)->cenNum+i) ,*(minNum+i),*(rngNum+i));
      b=norm(*(rec->num+i)            ,*(minNum+i),*(rngNum+i));
      *(disFld+j++)= disNum(a,b);
    }
    /*カテゴリ項目*/

    /*結合距離の計算*/
    distanceTmp=disAvg(disFld,j);

    /*距離が小さければ更新*/
    if( mssVcmpOpe(distanceTmp, OPE_LT, distance) ){
      distance=distanceTmp;
      k=c;
    }
  }
  mssFree(disFld);
  return(k);
}

/*----------------------------------------------------------------------------*/
/*全データを各クラスタに振り分け、重心計算のための値を更新していく            */
/*----------------------------------------------------------------------------*/
void setCluster(struct Cluster *cluster, struct mssHeader *hdi, struct mssFPR *fpr){
  struct mssFldRec *fr;  /*項目-行バッファ構造体*/
  int i,k;
  char *s;

  for(k=0; k<kCnt; k++){
    (cluster+k)->cnt=0; /*クラスタに属するレコード数*/
    for(i=0; i<fnum->cnt; i++){
      mssVinit( &((cluster+k)->calNum+i)->accum, DBL );
      ((cluster+k)->calNum+i)->cnt=0;
    }
  }

  fr=mssInitFldRec(hdi->flds->cnt);
  mssSeekTopFPR(fpr);
  while( EOF != mssReadFldRec(fpr,fr) ){

    k=nearestCluster(cluster,fr);
    if(k!=-1){
      (cluster+k)->cnt++; /*クラスタに属するレコード数*/

      /*数値項目*/
      for(i=0; i<fnum->cnt; i++){
        s=*(fr->pnt+MssFlds2num(fnum,i));
        if(*s=='*'){
          ((cluster+k)->calNum+i)->accum.v.d += (*(cluster->data->avgNum+i)).v.d;
        }else{
          ((cluster+k)->calNum+i)->accum.v.d += atof(s);
        }
        ((cluster+k)->calNum+i)->cnt++;
      }

    /*カテゴリ項目*/

    }
  }
  mssFreeFldRec(fr);
}

/*----------------------------------------------------------------------------*/
/*サンプリングデータを各クラスタに振り分け、重心計算のための値を更新していく  */
/*----------------------------------------------------------------------------*/
void setClusterSmp(struct Cluster *cluster, struct Sample *sample){
  int i,k;
  int smp;
  MssValue v;

  for(k=0; k<kCnt; k++){
    (cluster+k)->cnt=0; /*クラスタに属するレコード数*/
    for(i=0; i<fnum->cnt; i++){
      mssVinit( &((cluster+k)->calNum+i)->accum, DBL );
      ((cluster+k)->calNum+i)->cnt=0;
    }
  }

  for(smp=0; smp<sample->recCnt; smp++){
    k=nearestClusterSmp(cluster,sample->rec+smp);
    (cluster+k)->cnt++; /*クラスタに属するレコード数*/

    /*数値項目*/
    for(i=0; i<fnum->cnt; i++){
      v=*((sample->rec+smp)->num+i);
      if(!v.nul){
        ((cluster+k)->calNum+i)->accum.v.d += v.v.d;
        ((cluster+k)->calNum+i)->cnt++;
      }
    }
    /*カテゴリ項目*/

  }
}

/*----------------------------------------------------------------------------*/
/* 各クラスタの重心を計算し、重心が変われば1変わらなければ0を返す             */
/*----------------------------------------------------------------------------*/
int movCenter(struct Cluster *cluster){
  int mov=0; /*重心が動いたフラグ*/
  MssValue newCen;
  int i,k;

  mssVinit(&newCen,DBL);

  for(k=0; k<kCnt; k++){
    /*数値項目*/
    for(i=0; i<fnum->cnt; i++){
      if(((cluster+k)->calNum+i)->cnt != 0){
        newCen.v.d =((cluster+k)->calNum+i)->accum.v.d /
            (double)((cluster+k)->calNum+i)->cnt;
        /*cnt==0の時はcenterを動かさない */
        if( mssVcmpOpe(newCen,OPE_NE,*((cluster+k)->cenNum+i)) ){
          *((cluster+k)->cenNum+i)=newCen;
          mov=1;
        }
      }
    }
  }
  return(mov);
}

/*----------------------------------------------------------------------------*/
/* 入力データの各種情報を計算                                                 */
/*----------------------------------------------------------------------------*/
struct DataInfo *getDatInfo(
  struct mssHeader  *hdi,  /*ヘッダー構造体*/
  struct mssFPR     *fpr){ /*ファイルポインタ*/

  struct DataInfo *data;    /*入力データ情報構造体*/
  struct mssFldRec *fr;    /*項目-行バッファ構造体*/
  char  *str;
  MssValue  num;
  int i;

  mssVinit(&num,DBL);

  data        =mssMalloc(sizeof(struct DataInfo),"gdi");
  data->maxNum=mssMalloc(sizeof(MssValue)*fnum->cnt,"gdi");
  data->minNum=mssMalloc(sizeof(MssValue)*fnum->cnt,"gdi");
  data->rngNum=mssMalloc(sizeof(MssValue)*fnum->cnt,"gdi");
  data->sumNum=mssMalloc(sizeof(MssValue)*fnum->cnt,"gdi");
  data->cntNum=mssMalloc(sizeof(int)*fnum->cnt,"gdi");
  data->avgNum=mssMalloc(sizeof(MssValue)*fnum->cnt,"gdi");
  for(i=0; i<fnum->cnt; i++){
    mssVinit(data->maxNum+i,DBL);
    mssVinit(data->minNum+i,DBL);
    mssVinit(data->rngNum+i,DBL);
    mssVinit(data->sumNum+i,DBL);
    mssVinit(data->avgNum+i,DBL);
    (*(data->maxNum+i)).v.d=-DBL_MAX;
    (*(data->minNum+i)).v.d=DBL_MAX;
    (*(data->rngNum+i)).v.d=0;
    (*(data->sumNum+i)).v.d=0;
     *(data->cntNum+i)     =0;
    (*(data->avgNum+i)).v.d=0;
  }
  data->cnt=0;

  fr=mssInitFldRec(hdi->flds->cnt);
  mssSeekTopFPR(fpr);

  while( EOF != mssReadFldRec(fpr,fr) ){

    /*数値項目*/
    for(i=0; i<fnum->cnt; i++){
      str=*(fr->pnt+MssFlds2num(fnum,i));
      if(*str!='*'){
        num.v.d=atof(str);
        (*(data->cntNum+i))++;                      /*件数*/
        (*(data->sumNum+i)).v.d+=num.v.d;           /*合計*/
        if( mssVcmpOpe(num,OPE_GT,*(data->maxNum+i)) )
          (*(data->maxNum+i)).v.d=num.v.d;          /*最大値*/
        if( mssVcmpOpe(num,OPE_LT,*(data->minNum+i)) )
          (*(data->minNum+i)).v.d=num.v.d;          /*最小値*/
      }
    }

    data->cnt++;
  }
  mssFreeFldRec(fr);

  /*データなしエラーチェック*/
  for(i=0; i<fnum->cnt; i++){
    if( *(data->cntNum+i) ==0 ) {
      mssShowErrMsg("value not found on some fields");
      exit(mssErrorNoDefault);
    }
  }

  /* 最大値-最小値,平均値を求める */
  for(i=0; i<fnum->cnt; i++){
    (*(data->rngNum+i)).v.d=(*(data->maxNum+i)).v.d-(*(data->minNum+i)).v.d;
    (*(data->avgNum+i)).v.d=(*(data->sumNum+i)).v.d/(double)(*(data->cntNum+i));
  }

  /* データ行数より指定のクラスタ数の方が多い時 */
  if(data->cnt<kCnt) kCnt=data->cnt;

/*
printf("line cnt=%d\n",data->cnt);
for(i=0; i<fnum->cnt; i++){
  printf("fld[%d] : ",i);
  printf("min=%g, ",(*(data->minNum+i)).v.d);
  printf("max=%g, ",(*(data->maxNum+i)).v.d);
  printf("rng=%g, ",(*(data->rngNum+i)).v.d);
  printf("sum=%g, ",(*(data->sumNum+i)).v.d);
  printf("cnt=%d, ",*(data->cntNum+i));
  printf("avg=%g\n",(*(data->avgNum+i)).v.d);
}
*/

  return(data);
}

void freeDataInfo(struct DataInfo *data){

  mssFree(data->maxNum);
  mssFree(data->minNum);
  mssFree(data->rngNum);
  mssFree(data->sumNum);
  mssFree(data->cntNum);
  mssFree(data->avgNum);
  mssFree(data);
}

/*----------------------------------------------------------------------------*/
/*Cluster構造体の領域確保                                                     */
/*----------------------------------------------------------------------------*/
struct Cluster *malCluster(int cnt){
  struct Cluster *cluster;
  int i,j;

  cluster=mssCalloc(sizeof(struct Cluster)*cnt,"malCluster");
  for(i=0; i<cnt; i++){ /*クラスタ数回す*/
    (cluster+i)->cenNum   = mssCalloc(sizeof(MssValue        )*fnum->cnt,"mal");
    (cluster+i)->calNum   = mssCalloc(sizeof(struct CalNum)*fnum->cnt,"mal");
    for(j=0; j<fnum->cnt; j++) mssVinit((cluster+i)->cenNum+j,DBL);
  }

  return(cluster);
}

void freeCluster(struct Cluster *cluster, int cnt){
  int i;

  for(i=0; i<cnt; i++){ /*クラスタ数回す*/
    mssFree((cluster+i)->cenNum);
    mssFree((cluster+i)->calNum);
  }
  mssFree(cluster);
}

void freeSample(struct Sample *sample){
  int i;

  for(i=0; i<sample->recCnt; i++){
    mssFree( (sample->rec+i)->num );
  }
  mssFree( sample->rec );
  mssFree( sample );
}

/*----------------------------------------------------------------------------*/
/* sampleからrecCnt件サンプリングする                                         */
/*----------------------------------------------------------------------------*/
struct Sample *samplingSmp(
  struct Sample *orgSmp,  /*入力データ*/
  int            recCnt){ /*サンプリング行数*/

  struct Sample *sample;  /*サンプリングされたデータを格納する*/
  int select;
  int remaining;
  int r;                   /*元データのwhile内行数カウンタ*/
  int j;


  /*サンプリング件数の調整*/
  if(recCnt > orgSmp->recCnt) recCnt=orgSmp->recCnt;

  /*領域確保*/
  sample     =mssMalloc(sizeof(struct Sample),       "sampling");
  sample->rec=mssMalloc(sizeof(struct SmpRec)*recCnt,"sampling");
  for(j=0; j<recCnt; j++){
    (sample->rec+j)->num=mssMalloc(sizeof(MssValue)*fnum->cnt,"sampling");
  }

  select=recCnt;
  remaining=orgSmp->recCnt;
  for(r=0; r<orgSmp->recCnt; r++){
    if( (rand()%remaining) < select ){
      /*数値項目*/
      for(j=0; j<fnum->cnt; j++){
        *((sample->rec+recCnt-select)->num+j) =*((orgSmp->rec+r)->num+j);
      }

      /*カテゴリ項目*/

      select--;
    }
    remaining--;
  }

  sample->recCnt=recCnt;
  sample->numFldCnt=fnum->cnt;

  return(sample);
}

/*----------------------------------------------------------------------------*/
/* ファイルからrecCnt件サンプリングし、sample[]にセットする                   */
/*----------------------------------------------------------------------------*/
void sampling(
  struct Sample   *sample[], /*サンプリングされたデータを格納する*/
  int              mul,      /*何セット用意するか*/
  int              recCnt,   /*サンプリング行数*/
  struct DataInfo *data,     /*入力データ情報構造体*/
  struct mssHeader   *hdi,      /*ヘッダー構造体*/
  struct mssFPR      *fpr){     /*ファイルポインタ*/

  struct mssFldRec *fr;       /*項目-行バッファ構造体*/
  int select[10];
  int remaining[10];
  int r;                   /*元データのwhile内行数カウンタ*/
  int i,j;
  char *s;
  MssValue val;

  mssVinit(&val,DBL);

  /*サンプリング件数の調整*/
  if(recCnt < 100      ) recCnt=100;
  if(recCnt > 5000     ) recCnt=5000;
  if(recCnt > data->cnt) recCnt=data->cnt;

  /*領域確保*/
  for(i=0; i<mul; i++){
    sample[i]     =mssMalloc(sizeof(struct Sample),       "sampling");
    sample[i]->rec=mssMalloc(sizeof(struct SmpRec)*recCnt,"sampling");
    for(j=0; j<recCnt; j++){
      (sample[i]->rec+j)->num=mssMalloc(sizeof(MssValue)*fnum->cnt,"sampling");
    }
    sample[i]->data=data;
  }

  fr=mssInitFldRec(hdi->flds->cnt);
  mssSeekTopFPR(fpr);
  for(i=0; i<mul; i++){
    select[i]=recCnt;
    remaining[i]=data->cnt;
  }
  r=0;
  while( EOF != mssReadFldRec(fpr,fr) ){

    for(i=0; i<mul; i++){
      if( (rand()%remaining[i]) < select[i] ){
        /*数値項目*/
        for(j=0; j<fnum->cnt; j++){
          s=*(fr->pnt+MssFlds2num(fnum,j));
          if(*s=='*'){ val=*(data->avgNum+j); }
          else       { val.nul=0; val.v.d=atof(s); }
          *((sample[i]->rec+recCnt-select[i])->num+j) = val;
        }

        /*カテゴリ項目*/

        select[i]--;
      }
      remaining[i]--;
      r++;
    }
  }
  mssFreeFldRec(fr);

  for(i=0; i<mul; i++){
    sample[i]->recCnt=recCnt;
    sample[i]->numFldCnt=fnum->cnt;
  }
}

void setSmp2Cluster(
  struct Cluster *cluster,
  int k,
  struct Sample  *sample,
  int s){

  int i;
  /*数値項目*/
  for(i=0; i<fnum->cnt; i++){
    *((cluster+k)->cenNum+i)=*((sample->rec+s)->num+i);
  }
}

/*============================================================================*/
/* 初期Seedの作成                                                             */
/*============================================================================*/
/*----------------------------------------------------------------------------*/
/*ランダム初期化                                                              */
/*----------------------------------------------------------------------------*/
struct Cluster *initClusterRA( struct Sample *sample ){

  struct Sample  *smp;
  struct Cluster *cluster; /*クラスタ構造体*/
  int k=0;                 /*クラスタ番号*/

  /* k個をランダムに選択*/
  smp=samplingSmp(sample,kCnt);

  /*領域確保*/
  cluster=malCluster(kCnt);
  cluster->data=sample->data;

  for(k=0; k<smp->recCnt; k++){
    setSmp2Cluster(cluster,k,smp,k);
  }
  freeSample(smp);
  return(cluster);
}

/*----------------------------------------------------------------------------*/
/* KAUFMAN APPROACH                                                           */
/* Reference :                                                                */
/* Lozano, "An empirical comparison of four initialization methods            */
/* for the K-Mean",p6.                                                        */
/*----------------------------------------------------------------------------*/
/*重心に最も近いレコード番号を取得*/
static int getCenInstance(struct Sample *sample){

  struct Cluster *cluster;
  MssValue disTmp;
  int i,j,k;
  MssValue val;

  cluster = malCluster(1);

  for(i=0; i<sample->recCnt; i++){
    for(j=0; j<sample->numFldCnt; j++){
      if(! (*((sample->rec+i)->num+j)).nul ){
        (cluster->calNum+j)->accum.v.d += (*((sample->rec+i)->num+j)).v.d;
        (cluster->calNum+j)->cnt++;
      }
    }
  }
  for(j=0; j<sample->numFldCnt; j++){
    mssVinit(&val,DBL);
    if( (cluster->calNum+j)->cnt ==0 ){
      val.nul=1;
    }else{
      val.v.d= (cluster->calNum+j)->accum.v.d/(double)(cluster->calNum+j)->cnt;
    }
    *(cluster->cenNum+j)=val;
  }

  k=0;
  mssVinit(&distance,DBL);
  distance.v.d=DBL_MAX;
  for(i=0; i<sample->recCnt; i++){
    disTmp=calDistanceClsSmp(cluster,0,sample,i);
    if( mssVcmpOpe(distance,OPE_GT,disTmp) ){
      distance=disTmp;
      k=i;
    }
  }
  return(k);
}


/* 二つのサンプル間の距離を返す */
static MssValue get_dji(struct Sample *sample, int i, int j){

  return(calDistanceSmpSmp(sample, i, j));

}

/*現在登録されている各クラスタとサンプルjとの最短距離を返す */
static MssValue get_Dj(
  struct Cluster *cluster,
  int k,                  /*現在登録されているクラスタ数*/
  struct Sample *sample, int j){
  MssValue dis;
  MssValue disMin;
  int s_min=0;
  int s;

  mssVinit(&disMin,DBL);
  disMin.v.d=DBL_MAX;
  for(s=0; s<k; s++){
    dis=calDistanceClsSmp(cluster,s,sample,j);
    if( mssVcmpOpe(disMin, OPE_GT, dis) ){
      disMin=dis;
      s_min=s;
    }
  }
  return(disMin);
}

/*sListのk番目までに、sがあれば１を返す*/
static int isInCluster(int s,int *sList, int k){
  int i;
  for(i=0; i<k; i++){
    if( s==*(sList+i) ) return(1);
  }
  return(0);
}

/* メインルーチン (KA) */
static struct Cluster *initClusterKA( struct Sample *sample ){

  struct Cluster *cluster; /*クラスタ構造体*/
  int k=0;                 /*クラスタ番号*/
  MssValue    Cji,Cji_max;
  MssValue    val0;
  int    i,j,i_max=0;
  int     s;
  int    *sList; /*クラスタに選択されたサンプル番号リスト*/

  mssVinit(&Cji,DBL);
  mssVinit(&Cji_max,DBL);
  mssVinit(&val0,DBL);

  /*領域確保*/
  cluster=malCluster(kCnt);
  cluster->data=sample->data;

  /*シード番号を納める領域の確保*/
  sList=mssCalloc(sizeof(int)*kCnt,"initCluster");

  k=0;
  s=getCenInstance(sample);
  *(sList+k)=s;
  setSmp2Cluster(cluster,k++,sample,s);
  while(1){
    Cji_max.v.d=-1;
    for(i=0; i<sample->recCnt; i++){
      if(isInCluster(i,sList,k)) continue;
      mssVinit(&Cji,DBL);
      for(j=0; j<sample->recCnt; j++){
        if(isInCluster(j,sList,k)) continue;
        if(i==j) continue;
        Cji = mssVadd(Cji,mssVmax(mssVsub(get_Dj(cluster,k,sample,j),get_dji(sample,i,j)),val0));
      }
      if( mssVcmpOpe(Cji, OPE_GT, Cji_max) ){
        Cji_max=Cji;
        i_max=i;
      }
    }

    *(sList+k)=i_max;
    setSmp2Cluster(cluster,k++,sample,i_max);
    if(k==kCnt) break;
  }

  mssFree(sList);
  return(cluster);
}

/*----------------------------------------------------------------------------*/
/* BRADLEY & FAYYAD APPROACH                                                  */
/* Reference :                                                                */
/* Bradley, Fayyad, "Refining Initial Points for K-Means Clustering"          */
/* CMi : CM[i] */
/* CM  : smpCM */
/* FMi : FM[i] */
/*----------------------------------------------------------------------------*/
/*k番目のクラスタから最も離れたサンプルをクラスタの中心として登録する*/
static struct Cluster *farthest(
  struct Cluster *cluster,
  int k,
  struct Sample *sample){

  struct Cluster *cls; /*新しいクラスタ*/
  MssValue dis;
  MssValue dis_max;
  int i_max;
  int i;

  mssVinit(&dis,DBL);
  mssVinit(&dis_max,DBL);

  /* start i_max の算出 */
  dis_max.v.d=-1;
  i_max=-1;
  for(i=0; i<sample->recCnt; i++){
    dis=calDistanceClsSmp(cluster, k, sample,i);
    if( mssVcmpOpe(dis, OPE_GT, dis_max) ){
      dis_max=dis;
      i_max=i;
    }
  }
  cls=malCluster(kCnt);
  cls->cnt=cluster->cnt;
  for(i=0; i<kCnt;i++){
    if(i!=k){
      (cls+i)->cnt   =(cluster+i)->cnt;
      (cls+i)->cenNum=(cluster+i)->cenNum;
      (cls+i)->calNum=(cluster+i)->calNum;
    }else{
      mssFree((cluster+k)->cenNum);
      mssFree((cluster+k)->calNum);
    }
  }
  cls->data=cluster->data;
  mssFree(cluster);
  setSmp2Cluster(cls,k,sample,i_max);
  return(cls);
}

static struct Cluster *initClusterBF( struct Sample *sample[] ){

  struct Sample  *smpCM;  /*新しいサンプル*/
  struct Cluster *CM[10]; /*クラスタ構造体*/
  struct Cluster *FM[10]; /*クラスタ構造体*/
  int i,j,s,flg;
  int minFMi=0;
  MssValue minFM;
  MssValue tmpFM;


  /* 新しいサンプルの作成 */
  for(s=0; s<optMUL.val; s++){
    /*領域確保(CM[s])*/
    CM[s]=malCluster(kCnt);

    /*入力データから初期値としてk個の重心を得る*/
    /*CM[s]=initClusterKA(sample[s]);*/
    CM[s]=initClusterRA(sample[s]);
    CM[s]->data=sample[s]->data;

    /*showCluster(CM[s]);*/

    /* CONVERGENCE (kmean-mod)*/
    for(i=0;i<2;i++){
      /* k-mean */
      while(1){
        setClusterSmp(CM[s],sample[s]);
        if(!movCenter(CM[s]))break;
      }

      /* k-mean mod*/
      flg=0;
      for(j=0; j<kCnt; j++){
        /*空のクラスタもしくはNULLのクラスタがあった*/
        if((CM[s]+j)->cnt == 0){
          CM[s]=farthest(CM[s],j,sample[s]);
          flg=1; /*空のクラスタがあったフラグ*/
          break;
        }
      }
      if(!flg) break;
      /*2回目の試行で空のクラスがあればNULLを返す*/
      if(i==1 && flg){
        return(NULL);
      }
    }
  }

  /*クラスタの重心をサンプルとして登録する*/
  /*数値項目*/
  /*領域確保(smpCM)*/
  smpCM     =mssMalloc(sizeof(struct Sample),                "sampling");
  smpCM->rec=mssMalloc(sizeof(struct SmpRec)*optMUL.val*kCnt,"sampling");
  for(j=0; j<optMUL.val*kCnt; j++){
    (smpCM->rec+j)->num=mssMalloc(sizeof(MssValue)*fnum->cnt,"sampling");
  }
  smpCM->recCnt=0;
  smpCM->numFldCnt=fnum->cnt;
  for(s=0; s<optMUL.val; s++){
    for(i=0; i<kCnt; i++){
      for(j=0; j<fnum->cnt; j++){
        *((smpCM->rec+smpCM->recCnt)->num+j) = *((CM[s]+i)->cenNum+j);
      }
      smpCM->recCnt++;
    }
  }

  /*showSample(smpCM);*/

  /* FM[i]の算出*/
  for(s=0; s<optMUL.val; s++){
    /*領域確保(CM[s])*/
    FM[s]=malCluster(kCnt);
    FM[s]->data=sample[s]->data;

    /*CM[i]をFM[i]の初期値とする*/
    /*数値項目*/
    for(i=0; i<kCnt; i++){
      for(j=0; j<fnum->cnt; j++){
        *((FM[s]+i)->cenNum+j) = *((CM[s]+i)->cenNum+j);
      }
    }
 
    /* CONVERGENCE */
    while(1){
      setClusterSmp(FM[s],smpCM);
      if(!movCenter(FM[s]))break;
    }
  }

  /* DISTORTION(FM[i],smpCM)          */
  /* もっともsmpCMにfitするFM[i]を選ぶ*/
  mssVinit(&minFM,DBL);
  minFM.v.d=DBL_MAX;
  for(s=0; s<optMUL.val; s++){
    setClusterSmp(FM[s], smpCM);
    mssVinit(&tmpFM,DBL);
    for(i=0; i<kCnt; i++){
      /*数値*/
      for(j=0; j<fnum->cnt; j++){
        tmpFM = mssVadd( tmpFM, ((FM[s]+i)->calNum+j)->accum);
      }
    }
    if( mssVcmpOpe(minFM, OPE_GT, tmpFM) ){
      minFM =tmpFM;
      minFMi=s;
    }
  }

  /*showCluster(FM[minFMi]);*/

  /*領域開放(CM[i])*/
  for(s=0; s<optMUL.val; s++){
    freeCluster(CM[s],kCnt);
    if(s!=minFMi) freeCluster(FM[s],kCnt);
  }
  for(j=0; j<optMUL.val*kCnt; j++){
    mssFree( (smpCM->rec+j)->num);
  }
  mssFree( smpCM->rec );
  mssFree( smpCM );
  FM[minFMi]->data=sample[0]->data;
  return(FM[minFMi]);
}
 
/*============================================================================*/
/* メイン                                                                     */
/*============================================================================*/
int main(int argc, char *argv[]){

  struct mssHeader *hdi; /*入力ファイル用<head>タグ格納構造体*/
  struct mssHeader *hdo; /*出力ファイル用<head>タグ格納構造体*/
  struct mssFPR    *fpr; /*入力ファイル構造体*/
  struct mssFPW    *fpw; /*出力ファイル構造体*/
  struct mssFldRec *fr; /*項目-行バッファ構造体*/

  struct Cluster *cluster=NULL;
  struct DataInfo *data;
  struct Sample   *sample[10]; /*マルチスタート＆BF用*/
  MssValue *disAccum;  /*各クラスタの各データからの累積距離*/
  MssValue  totalDis;   /*距離の総和*/
  int k;
  int i;
  int convCnt=0;

/*----------------------------------------------------------------------------*/
/* 前処理                                                                     */
/*----------------------------------------------------------------------------*/
  mssInit(argc,argv,&comHelp);       /* シグナル処理などの初期化              */
  mssHelpDoc(opt,&comHelp,argc,argv);/* ヘルプ                                */
  mssSetOption(opt,argc,argv);       /* コマンドオプションの設定              */
  fpr=mssOpenFPR(optINF.str,4);      /* 入力ファイルオープン                  */
  hdi=mssReadHeader(fpr);            /* ヘッダの読み込み                      */
  mssSetOptFld(&optNUM, hdi);        /* -n 項目をヘッダー項目に関連づける     */
  fnum=optNUM.flds;

/*mssShowOption(opt);*/
/*mssShowHeader(hdi);*/

  totalFldCnt=fnum->cnt;
  kCnt = optCNT.val; /* クラスタ数 */

/*----------------------------------------------------------------------------*/
/*出力ヘッダーの作成と出力                                                    */
/*----------------------------------------------------------------------------*/
  /*出力ヘッダーの初期化(タイトル等のコピー)*/
  hdo=mssInitCpyHeader(hdi);
    
  /*入力ヘッダの全項目を追加*/
  mssAddFieldsByFields(hdo->flds,hdi->flds);

  /*新項目名の追加*/
  mssAddFieldsByStrList(hdo->flds,optFNM.strList,optFNM.cnt);

  /*標準出力オープン+ヘッダーの出力*/
  fpw=mssOpenFPW(optOTF.str,optZIP.set,0);
  mssWriteHeader(hdo, fpw);

/*----------------------------------------------------------------------------*/
/*メインルーチン                                                              */
/*----------------------------------------------------------------------------*/
  /*入力データ情報の獲得*/
  data=getDatInfo(hdi,fpr);

  /*乱数の初期化*/
  mssInitRand(optSED.val);

  mssVinit(&distance,DBL);

  /*データのサンプリング*/
  switch(optINI.val){
  case 0: /*---------------------------- ランダム*/
    sampling(sample,1,100,data,hdi,fpr);
    break;

  case 1: /*---------------------------- Kaufman Approach*/
    sampling(sample,1,100, data,hdi,fpr);
    break;

  case 2: /*---------------------------- Bradley & Fayyad Approach*/
    sampling(sample,optMUL.val,100, data,hdi,fpr);
    break;
  }
  /*showSample(sample[0]);*/

  /*入力データから初期値としてk個の重心を得る*/
  switch(optINI.val){
  case 0: /*---------------------------- ランダム*/
    cluster = initClusterRA(sample[0]);
    freeSample(sample[0]);
    break;

  case 1: /*---------------------------- Kaufman Approach*/
    cluster = initClusterKA(sample[0]);
    freeSample(sample[0]);
    break;

  case 2: /*---------------------------- Bradley & Fayyad Approach*/
    while(1){
      cluster = initClusterBF(sample);
      if(cluster==NULL) kCnt--;
      else              break;
    }
    for(i=0; i<optMUL.val; i++) freeSample(sample[i]);
    break;
  }
  /*showCluster(cluster);*/

  /*==================*/
  /* MAIN CONVERGENCE */
  /*==================*/
  while(1){
    convCnt++;
    setCluster(cluster,hdi,fpr);

    if(!movCenter(cluster))break;
  }
  /*showCluster(cluster);*/

  /*結果の表示*/
  mssVinit(&totalDis,DBL);
  disAccum=mssCalloc(sizeof(MssValue)*kCnt,"xtkmean");
  for(i=0; i<kCnt; i++) mssVinit(disAccum+i,DBL);
  fr=mssInitFldRec(hdi->flds->cnt);
  mssSeekTopFPR(fpr);
  while( EOF != mssReadFldRec(fpr,fr) ){
    mssGV.inCnt++;

    k=nearestCluster(cluster,fr);
    if(k==-1){
      mssWriteFld(fr->pnt, fr->fldCnt, " ", fpw);
      mssWriteStr("*\n",fpw);
    }else{
      *(disAccum+k) = mssVadd(*(disAccum+k),distance);
      totalDis = mssVadd(totalDis,distance);
      mssWriteFld(fr->pnt, fr->fldCnt, " ", fpw);
      mssWriteInt(k+1,fpw);
      mssWriteRet(fpw);
    }
    mssGV.outCnt++;
  }

/*
  for(i=0; i<kCnt; i++){
    printf("distance[%d]=%g\n",i,*(disAccum+i));
  }
  printf(" totalDistance: %f ",sqrt(totalDis.v.d/(double)data->cnt));
  printf("convCnt: %d ",convCnt);
  printf("\n");
*/

  mssFree(disAccum);
  freeDataInfo(data); 
  freeCluster(cluster,kCnt);

/*----------------------------------------------------------------------------*/
/*フッター出力&終了処理                                                       */
/*----------------------------------------------------------------------------*/
  mssWriteFooter(fpw);    /* フッターの出力             */
  mssCloseFPR(fpr);       /* 入力ファイルのクローズ     */
  mssCloseFPW(fpw);       /* 出力ファイルのクローズ     */
  mssFreeHeader(hdi);     /* 入力ヘッダ領域開放         */
  mssFreeHeader(hdo);     /* 出力ヘッダ領域開放         */
  mssFreeOption(opt);     /* オプション領域開放         */
  mssShowEndMsg();        /* 完了メッセージ             */
  mssEnd(mssExitSuccess); /* 終了                       */
  return(0);              /* to avoid warning message   */
}
