/*============================================================================*/
/* 領域分割ライブラリ                                                         */
/*----------------------------------------------------------------------------*/
/* 変更履歴                                                                   */
/* 1.0 : 初期作成 2003/08/07                                                  */
/*============================================================================*/
#include <musashi.h>
#include <region.h>

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <float.h>

/*
 *------------------------------------------------------------------------------
 * 領域データを格納する件数テーブル(struct CntTbl)関連
 *------------------------------------------------------------------------------
 */

/*
 * 件数テーブルの初期化
 */
struct CntTbl *initCntTbl(int x, int y){

  struct CntTbl *cntTbl;
  int i;

  cntTbl=mssCalloc(sizeof(struct CntTbl),"initCntTbl");
  cntTbl->xCnt=x; /*X軸のサイズ 将来xSizeに変更*/
  cntTbl->yCnt=y; /*Y軸のサイズ*/

  for(i=0; i<cntTbl->xCnt; i++){
    cntTbl->rtbl[0][i].from= i+1;
    cntTbl->rtbl[0][i].to  = i+1;
  }

  for(i=0; i<cntTbl->yCnt; i++){
    cntTbl->rtbl[1][i].from= i+1;
    cntTbl->rtbl[1][i].to  = i+1;
  }

/* 将来ポインタに変更
  for(i=0; i<=x; i++){
    for(j=0; j<=y; j++){
      cntTbl->tbl[i][j].negCnt = 0;
      cntTbl->tbl[i][j].posCnt = 0;
      cntTbl->tbl[i][j].total  = 0;
      cntTbl->tbl[i][j].weight = 0;
    }
  }
*/
  return(cntTbl);
}

/**
 * X,Y座標とsupport,hit件数を与えてCntTblにデータをセット
 * 引数x,yは0 から始まるが、tblにセットするときは+1する。
 * 0は再帰処理の時の初期値として使う
 */
void setCntTblByCell(struct CntTbl *cntTbl, int x, int y, int hit, int sup)
{
  if( x >= cntTbl->xCnt ||
      y >= cntTbl->yCnt ||
      x < 0            ||
      y < 0            ){
    mssShowErrMsg("Internal Error: illigal value on X or Y axis");
    mssEnd(mssErrorNoDefault);
  }

  /*各種値のCountUp*/
  cntTbl->tbl[x+1][y+1].total =sup;     /*セルのトータル件数をセット*/
  cntTbl->tbl[x+1][y+1].posCnt=hit;     /*セルのpositive件数をセット*/
  cntTbl->tbl[x+1][y+1].negCnt=sup-hit; /*セルのnegative件数をセット*/
  cntTbl->sup+=sup;                     /*全体のサポートをCountUp*/
}

/**
 * totalが0件のセルについて、totalを１件そして、posかnegを１件とする。
 * posかnegはnulの値(optNul.str)で判断する。
 * そして、その影響が領域分割にでないようにcntTblのその他の件数を1000倍する。
 */
void adjustNulCntTbl( struct CntTbl *cntTbl, char *nul)
{
    
  int i,j;
    
  cntTbl->sup=0;
  cntTbl->hit=0;
  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
      if(cntTbl->tbl[i][j].total != 0){
        cntTbl->tbl[i][j].total  *= 1;
        cntTbl->tbl[i][j].negCnt *= 1;
        cntTbl->tbl[i][j].posCnt *= 1;
        cntTbl->sup += cntTbl->tbl[i][j].total;
        cntTbl->hit += cntTbl->tbl[i][j].posCnt; 
      }else{
        cntTbl->tbl[i][j].total  = 1;
        if(strcmp(nul,"negative")==0){ /*negativeにカウント*/
          cntTbl->tbl[i][j].negCnt = 1;
          cntTbl->tbl[i][j].posCnt = 0;
        }else{                         /*positiveにカウント*/
          cntTbl->tbl[i][j].negCnt = 0;
          cntTbl->tbl[i][j].posCnt = 1;
        }
        cntTbl->sup += cntTbl->tbl[i][j].total;
        cntTbl->hit += cntTbl->tbl[i][j].posCnt;
      }
    }
  }
} 

/**
 * ajustで変換された値を元に戻す
 */
void deajustNulCntTbl(
  struct CntTbl *cntTbl,
  struct SELREG *selReg){

  int i,j;

  cntTbl->sup=0;
  cntTbl->hit=0;
  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
      if(cntTbl->tbl[i][j].total != 1){
        cntTbl->tbl[i][j].total  /= 1;
        cntTbl->tbl[i][j].negCnt /= 1;
        cntTbl->tbl[i][j].posCnt /= 1;
        cntTbl->sup += cntTbl->tbl[i][j].total;
        cntTbl->hit += cntTbl->tbl[i][j].posCnt;
      }else{
        cntTbl->tbl[i][j].total  = 0;
        cntTbl->tbl[i][j].negCnt = 0;
        cntTbl->tbl[i][j].posCnt = 0;
        cntTbl->tbl[i][j].weight = 0;
      }
    }
  }
  selReg->sup = 0;
  selReg->hit = 0;
  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
      if( *Ar2(selReg->tbl,i,j) == 1 ){
        selReg->sup += cntTbl->tbl[i][j].total;
        selReg->hit += cntTbl->tbl[i][j].posCnt;
      }
    }
  }
}

/**
 * cntTbl配列の表示
 */
void dispCntTbl(struct CntTbl *cntTbl){
  int i,j;
  printf("------------------\n");
  printf("件数表\n");
  printf("------------------\n");
  printf("    ");
  for(j=1; j<=cntTbl->xCnt; j++){
    printf("%5d ",j);
  }
  printf("\n");
  for(i=1; i<=cntTbl->yCnt; i++){
    printf("neg ");
    for(j=1; j<=cntTbl->xCnt; j++){
      printf("%5ld ",cntTbl->tbl[j][i].negCnt);
    }
    printf("\n");
    printf("pos ");
    for(j=1; j<=cntTbl->xCnt; j++){
      printf("%5ld ",cntTbl->tbl[j][i].posCnt);
    }
    printf("\n\n");
  }
  printf("\n");
}

/**
 * cntTbl配列のwight表示
 */
void dispCntTblWeight(struct CntTbl *cntTbl){
  int i,j;

  for(i=1; i<=cntTbl->yCnt; i++){
    printf("wight ");
    for(j=1; j<=cntTbl->xCnt; j++){
      printf("%4g ",cntTbl->tbl[j][i].weight);
    }
    printf("\n");
  }
}

/**
 * cntTbl配列のhit率表示
 */
void dispCntTblHitRatio(struct CntTbl *cntTbl){
  int i,j;

  printf("------------------\n");
  printf("ヒット率表\n");
  printf("------------------\n");

  printf("    ");
  for(j=1; j<=cntTbl->xCnt; j++){
    printf("%5d ",j);
  }
  printf("\n");
  for(i=1; i<=cntTbl->yCnt; i++){
    printf("%3d ",i);
    for(j=1; j<=cntTbl->xCnt; j++){
      {
        double tmp;
        tmp=(double)cntTbl->tbl[j][i].posCnt/(double)cntTbl->tbl[j][i].total;
        if(0<=tmp && tmp<=1){
          printf("%1.3f ",tmp);
        }else{
          printf("***** ");
        }
      }
    }
    printf("\n");
  }
  printf("\n");
}

/*
 *------------------------------------------------------------------------------
 * 領域分割ルーチン
 *------------------------------------------------------------------------------
 */

#define Ar1(a,x)     (a+(x))
#define Ar2(a,x,y)   (a+(y)*cntTbl->xCnt+(x))
#define Ar3(a,x,y,z) (a+(z)*cntTbl->yCnt*cntTbl->xCnt+(y)*cntTbl->xCnt+(x))
#define Ar2p1(a,x,y) (a+(y)*((cntTbl->xCnt)+1)+(x))


/**
 * log(底=2)の計算
 */
double Log2(double x){
  if(x==0){
    return(0);
  }else{
    return(log(x)/log(2));
  }
}

/**
 * 目的関数(クラス間分散)の値を求める
 * XXOXX
 * XXOOX
 * XOOXX
 * XOOXX
 * OOOXX
 *
 * 全体の support : sup,  positive:pos  negative:neg
 * Oの    support : supO, positive:posO negative:negO
 * Xの    support : supX, positive:posX negative:negX
 */
double getObjValSub(
  double sup ,double pos ,double neg ,
  double supO,double posO,double negO,
  double supX,double posX,double negX,int flag){

  switch(flag){
    case 0:
      return(
      ((pos /sup )-(posO/supO))*((pos /sup )-(posO/supO))*supO +
      ((pos /sup )-(posX/supX))*((pos /sup )-(posX/supX))*supX   );
      break;
    case 1:
      return(
      (-1)*(pos /sup )*Log2(pos /sup ) - (neg /sup )*Log2(neg /sup ) +
      (supO/sup)*((posO/supO)*Log2(posO/supO) + (negO/supO)*Log2(negO/supO))+
      (supX/sup)*((posX/supX)*Log2(posX/supX) + (negX/supX)*Log2(negX/supX)) );
       break;
    case 2:
      return(
      (1- (pos /sup )*(pos /sup )-(neg /sup )*(neg /sup )) -
      (supO/sup)*(1- (posO/supO)*(posO/supO)-(negO/supO)*(negO/supO)) -
      (supX/sup)*(1- (posX/supX)*(posX/supX)-(negX/supX)*(negX/supX))  );
    case 3:
      return( (posO + negX) / sup );
      break;
    }
    return(0);
}

/**
 * 目的関数のランチャー
 */
double getObjVal(
  struct CntTbl *cntTbl,
  double         hit,
  double         sup,
  int flag           ){

  double rsl=0;
  double      pos ,neg ;
  double supO,posO,negO;
  double supX,posX,negX;

  supO=sup        ; posO= hit      ; negO=supO-posO;
  sup =cntTbl->sup; pos =cntTbl->hit; neg =sup -pos ;
  supX=sup-supO   ; posX=pos-posO;    negX=supX-posX;

  if(sup==0 || supO==0 || supX==0){
    rsl=0;
  }else{
    rsl=getObjValSub(sup,pos,neg,supO,posO,negO,supX,posX,negX,flag);
  }
  return(rsl);
}

/*
 * 全ての目的関数値を最適領域のデータにセットする
 */
void setAllObjVal(
  struct CntTbl *cntTbl,
  struct SELREG *selReg){

  selReg->variance = getObjVal(cntTbl,selReg->hit,selReg->sup,0);
  selReg->entropy  = getObjVal(cntTbl,selReg->hit,selReg->sup,1);
  selReg->gini     = getObjVal(cntTbl,selReg->hit,selReg->sup,2);
  selReg->accuracy = getObjVal(cntTbl,selReg->hit,selReg->sup,3);
}


/*
 * cntTbl配列の各要素における weight(h(R)-θs(R))を求める
 *  hはその要素(R)におけるPositiveの件数
 *  sはその要素(R)における総件数
 */
void calWeight(
  struct CntTbl *cntTbl,
  double         theta){

  int i,j;

  /*weightを初期化*/
  for(j=1; j<=cntTbl->yCnt; j++){
    cntTbl->tbl[0][j].weight = 0;
  }

  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
      cntTbl->tbl[i][j].weight =
         (double)(cntTbl->tbl[i][j].posCnt) -
         (double)theta * (double)(cntTbl->tbl[i][j].total);

      if( cntTbl->tbl[i][j].weight <  1.0E-10 &&
          cntTbl->tbl[i][j].weight > -1.0E-10 ){
        cntTbl->tbl[i][j].weight = 0;
      }
    }
  }
}

/*
 * wm([1,v])とwm([u,N])の計算と
 * wm([1,i])の中での最適領域, wm([j,N])の中での最適領域
 *  iを増やしていき、各iにおける最適領域のスタート位置とh-θsの値を求める
 *  jについては逆
 */
void setWSub(
  struct CntTbl *cntTbl,
  int            m,
  struct WSUB   *wSub){

  int i;

  /*初期化*/
  *Ar1(wSub->w1u,0) = *Ar1(wSub->w1u,cntTbl->yCnt+1) = 0;
  *Ar1(wSub->wvN,0) = *Ar1(wSub->wvN,cntTbl->yCnt+1) = 0;

  /*wm([1,u])の計算*/
  *Ar1(wSub->w1u,1) = cntTbl->tbl[m][1].weight;
  for(i=2; i<=cntTbl->yCnt; i++){
    *Ar1(wSub->w1u,i) = *Ar1(wSub->w1u,i-1) + cntTbl->tbl[m][i].weight;
  }

  /*wm([v,N])の計算*/
  *Ar1(wSub->wvN,cntTbl->yCnt) = cntTbl->tbl[m][cntTbl->yCnt].weight;
  for(i=cntTbl->yCnt-1; i>=1; i--){
    *Ar1(wSub->wvN,i) = *Ar1(wSub->wvN,i+1) + cntTbl->tbl[m][i].weight;
  }

  /*wm([1,N])の計算*/
  wSub->w1N = *Ar1(wSub->wvN,1);

  /* wm([1,i])の中での最適領域,*/
  Ar1(wSub->w1i,1)->start  = 1;
  Ar1(wSub->w1i,1)->weight = cntTbl->tbl[m][1].weight;
  for(i=2; i<=cntTbl->yCnt; i++){
    if(Ar1(wSub->w1i,i-1)->weight > 0){
      Ar1(wSub->w1i,i)->start  = Ar1(wSub->w1i,i-1)->start;
      Ar1(wSub->w1i,i)->weight = Ar1(wSub->w1i,i-1)->weight
                               + cntTbl->tbl[m][i].weight;
    } else {
      Ar1(wSub->w1i,i)->start  = i;
      Ar1(wSub->w1i,i)->weight = cntTbl->tbl[m][i].weight;
    }
  }

  /* wm([j,N])の中での最適領域*/
  Ar1(wSub->wjN,cntTbl->yCnt)->start  = cntTbl->yCnt;
  Ar1(wSub->wjN,cntTbl->yCnt)->weight = cntTbl->tbl[m][cntTbl->yCnt].weight;
  for(i=cntTbl->yCnt-1; i>=1; i--){
    if(Ar1(wSub->wjN,i+1)->weight > 0){
      Ar1(wSub->wjN,i)->start  = Ar1(wSub->wjN,i+1)->start;
      Ar1(wSub->wjN,i)->weight = Ar1(wSub->wjN,i+1)->weight
                               + cntTbl->tbl[m][i].weight;
    } else {
      Ar1(wSub->wjN,i)->start  = i;
      Ar1(wSub->wjN,i)->weight = cntTbl->tbl[m][i].weight;
    }
  }
}


/**
 * Coverを求める
 */
double calCover(
  struct WSUB *wSub,
  int i,
  int j){

  return(  Ar1(wSub->w1i,i)->weight
         + Ar1(wSub->wjN,j)->weight
         + wSub->w1N
         - *Ar1(wSub->w1u,i)
         - *Ar1(wSub->wvN,j) );
}

/**
 * j*(i)を求める
 */
void setjAst(
  struct CntTbl *cntTbl,
  struct WSUB   *wSub,
  struct REGION *region,
  int            m,
  int            *jAst1,
  int            *jAstN){

  double maxValue;
  double value;

  int j;

  /* jAstNを求める*/
  maxValue=Ar2(region->tbl,m-1,cntTbl->yCnt)->weight +
                   calCover(wSub, 1, cntTbl->yCnt);
  *Ar1(jAstN,cntTbl->yCnt) = cntTbl->yCnt;
  for(j=cntTbl->yCnt-1; j>=1; j--){
    value=Ar2(region->tbl,m-1,j)->weight + calCover(wSub, 1, j);
    if(maxValue<value){
      maxValue=value;
      *Ar1(jAstN,j) = j;
    } else {
      *Ar1(jAstN,j) = *Ar1(jAstN,j+1);
    }
  }

  /* jAst1を求める*/
  maxValue=Ar2(region->tbl,m-1,1)->weight + calCover(wSub, 1, cntTbl->yCnt);
  *Ar1(jAst1,1) = 1;
  for(j=2; j<=cntTbl->yCnt; j++){
    value=Ar2(region->tbl,m-1,j)->weight + calCover(wSub, j, cntTbl->yCnt);
    if(maxValue<value){
      maxValue=value;
      *Ar1(jAst1,j) = j;
    } else {
      *Ar1(jAst1,j) = *Ar1(jAst1,j-1);
    }
  }
}


/**
 * 選択された領域のサポートとヒットをカウントする
 */
void setHitSup(
  struct CntTbl *cntTbl,
  struct SELREG *selReg){

  int i,j;

  selReg->sup = 0;
  selReg->hit = 0;
  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
      if( *Ar2(selReg->tbl,i,j) == 1){
        selReg->sup += cntTbl->tbl[i][j].total;
        selReg->hit += cntTbl->tbl[i][j].posCnt;
      }
    }
  }
}

/**
 *  Pθの最適領域を求める(高速アルゴリズム版)
 */
void getOptPtheta(
  struct CntTbl *cntTbl,
  double         theta,
  struct SELREG *selReg)
{

  int m,i,j;

  int x,y;

  double cover;
  double maxCover;
  int    maxCoverStart=0;
  int    maxCoverEnd=0;

  double weight1JI=0;
  double weightIJN=0;

  struct REGION  region;
  int           *jAst1;
  int           *jAstN;
  struct WSUB    wSub;

  /*各種メモリ領域確保*/
  region.tbl= mssMalloc((cntTbl->xCnt+1)*(cntTbl->yCnt+1)*
                        sizeof(struct REGBDY), "Ptheta");
  jAst1     = mssMalloc((cntTbl->yCnt+2)*sizeof(int), "Ptheta");
  jAstN     = mssMalloc((cntTbl->yCnt+2)*sizeof(int), "Ptheta");
  wSub.w1i  = mssMalloc((cntTbl->yCnt+2)*sizeof(struct W1IJN), "Ptheta");
  wSub.wjN  = mssMalloc((cntTbl->yCnt+2)*sizeof(struct W1IJN), "Ptheta");
  wSub.w1u  = mssMalloc((cntTbl->yCnt+2)*sizeof(double), "Ptheta");
  wSub.wvN  = mssMalloc((cntTbl->yCnt+2)*sizeof(double), "Ptheta");

  /*jAst1,jAstの計算でm=1の時,region[0][i]->weightを見るため0で初期化*/
  for(i=1; i<=cntTbl->yCnt; i++){
    Ar2(region.tbl,0,i)->weight = 0;
  }

  calWeight(cntTbl,theta);
  region.maxWeight=-DBL_MAX;
  for(m=1; m<=cntTbl->xCnt; m++){
    setWSub(cntTbl,m,&wSub);
    setjAst(cntTbl,&wSub,&region,m,jAst1,jAstN);
    for(i=1; i<=cntTbl->yCnt; i++){
      maxCover = DBLMINVAL;
      for(j=1; j<=cntTbl->yCnt; j++){
        if(i<j){
          cover = calCover(&wSub, i, j);
          if(maxCover < cover){
            maxCover      = cover;
            maxCoverStart = Ar1(wSub.w1i,i)->start;
            maxCoverEnd   = Ar1(wSub.wjN,j)->start;
          }
        } else {
          cover = calCover(&wSub, j, i);
          if(maxCover < cover){
            maxCover      = cover;
            maxCoverStart = Ar1(wSub.w1i,j)->start;
            maxCoverEnd   = Ar1(wSub.wjN,i)->start;
          }
        }
      }

      weight1JI = Ar2(region.tbl,m-1,*Ar1(jAst1,i))->weight
                  + calCover(&wSub, *Ar1(jAst1,i), i);
      weightIJN = Ar2(region.tbl,m-1,*Ar1(jAstN,i))->weight
                  + calCover(&wSub, i, *Ar1(jAstN,i));

      if(maxCover >= weight1JI && maxCover >= weightIJN){
        /*maxCover が大きい*/
        Ar2(region.tbl,m,i)->start  = maxCoverStart;
        Ar2(region.tbl,m,i)->end    = maxCoverEnd;
        Ar2(region.tbl,m,i)->optJ   = 0;
        Ar2(region.tbl,m,i)->weight = maxCover;
      } else if(weight1JI > weightIJN ){
        /*max(1≦j≦i)[wθ(m-1,j) + cover(i,j)] が大きい*/
        Ar2(region.tbl,m,i)->start  = Ar1(wSub.w1i,*Ar1(jAst1,i))->start;
        Ar2(region.tbl,m,i)->end    = Ar1(wSub.wjN,i            )->start;
        Ar2(region.tbl,m,i)->optJ   = *Ar1(jAst1,i);
        Ar2(region.tbl,m,i)->weight = weight1JI;
      } else {  
        /*max(i≦j≦N)[wθ(m-1,j) + cover(i,j)] が大きい*/
        Ar2(region.tbl,m,i)->start  = Ar1(wSub.w1i,i)            ->start;
        Ar2(region.tbl,m,i)->end    = Ar1(wSub.wjN,*Ar1(jAstN,i))->start;
        Ar2(region.tbl,m,i)->optJ   = *Ar1(jAstN,i);
        Ar2(region.tbl,m,i)->weight = weightIJN;
      }

      if(region.maxWeight < Ar2(region.tbl,m,i)->weight){
        region.maxX      = m;
        region.maxY      = i;
        region.maxWeight = Ar2(region.tbl,m,i)->weight;
      }
    } /*for i*/
  } /*for m*/

  /*最適領域のヒット,サポート,目的関数値の計算*/
  x=region.maxX;
  y=region.maxY;
  selReg->hit=0;
  selReg->sup=0;
  selReg->weight=Ar2(region.tbl,region.maxX,region.maxY)->weight;
  selReg->theta=theta;

  for(m=cntTbl->xCnt; m>x; m--){
    for(i=1; i<=cntTbl->yCnt; i++){
      *Ar2(selReg->tbl,m,i) = 0;
    }
  }
  while(1){
    for(i=1; i<Ar2(region.tbl,x,y)->start; i++){
      *Ar2(selReg->tbl,x,i) = 0;
    }
    for(i=Ar2(region.tbl,x,y)->start; i<=Ar2(region.tbl,x,y)->end; i++){
      *Ar2(selReg->tbl,x,i) = 1;
    }
    for(i=Ar2(region.tbl,x,y)->end+1;i<=cntTbl->yCnt; i++){
      *Ar2(selReg->tbl,x,i) = 0;
    }
    if(Ar2(region.tbl,x,y)->optJ==0)break;
    y=Ar2(region.tbl,x,y)->optJ;
    x--;
  }
  for(m=x-1; m>=1; m--){
    for(i=1; i<=cntTbl->yCnt; i++){
      *Ar2(selReg->tbl,m,i) = 0;
    }
  }
  setHitSup(cntTbl,selReg);
  selReg->objVal = getObjVal(cntTbl,(double)selReg->hit,(double)selReg->sup,0);
  mssFree(region.tbl);
  mssFree(wSub.w1i)  ; mssFree(wSub.wjN)  ; mssFree(wSub.w1u); mssFree(wSub.wvN);
  mssFree(jAst1); mssFree(jAstN);
}

/**
 * rectlinearによる最適領域の選択
 */
int maxValue(double value[], int cnt){
  double max;
  int i;
  int maxI;

  max  = value[0];
  maxI = 0;
  for(i=1; i<cnt; i++){
    if(max<value[i]){
      max=value[i];
      maxI=i;
    }
  }
  return(maxI);
}

double maxFwis(
  struct CntTbl *cntTbl,
  struct XYTBL *xyTbl,
  int m, int from, int to, int t, int *optI){

  double max;
  int i;
  /*ここの処理はあらかじめ計算しておけばO(N)で計算可能*/
  max = Ar3(xyTbl,m,from,t)->fw.weight;
  (*optI) = from;
  for(i=from+1; i<=to; i++){
    if(max<Ar3(xyTbl,m,i,t)->fw.weight){
      max=Ar3(xyTbl,m,i,t)->fw.weight;
      (*optI)=i;
    }
  }
  return(max);
}

double maxFuis(
  struct CntTbl *cntTbl,
  struct XYTBL *xyTbl,
  int m, int from, int to, int t, int *optI){
  double max;
  int i;
  /*ここの処理はあらかじめ計算しておけばO(N)で計算可能*/
  max = Ar3(xyTbl,m,from,t)->fu.weight;
  (*optI) = from;
  for(i=from+1; i<=to; i++){
    if(max<Ar3(xyTbl,m,i,t)->fu.weight){
      max=Ar3(xyTbl,m,i,t)->fu.weight;
      (*optI)=i;
    }
  }
  return(max);
}

double maxFwti( 
  struct CntTbl *cntTbl,
  struct XYTBL *xyTbl, 
  int m, int from, int to, int s, int *optI){
  double max;
  int i;
  /*ここの処理はあらかじめ計算しておけばO(N)で計算可能*/
  max = Ar3(xyTbl,m,s,from)->fw.weight;
  (*optI) = from;
  for(i=from+1; i<=to; i++){
    if(max<Ar3(xyTbl,m,s,i)->fw.weight){
      max=Ar3(xyTbl,m,s,i)->fw.weight;
      (*optI)=i;
    }
  }
  return(max);
}

double maxFdti( 
  struct CntTbl *cntTbl,
  struct XYTBL *xyTbl, 
  int m, int from, int to, int s, int *optI){
  double max;
  int i;
  /*ここの処理はあらかじめ計算しておけばO(N)で計算可能?*/
  max = Ar3(xyTbl,m,s,from)->fd.weight;
  (*optI) = from;
  for(i=from+1; i<=to; i++){
    if(max<Ar3(xyTbl,m,s,i)->fd.weight){
      max=Ar3(xyTbl,m,s,i)->fd.weight;
      (*optI)=i;
    }
  }
  return(max);
}


void setXYtbl( 
  struct XYBDY *xyTbl, 
  int type, int m, int s, int t, double w){
  xyTbl->type   = type;
  xyTbl->m      = m;
  xyTbl->s      = s;
  xyTbl->t      = t;
  xyTbl->weight = w;
}



void getFw(
  struct CntTbl *cntTbl,
  struct WSUB   *wSub,
  struct XYTBL  *xyTbl,
  int m, int s, int t){

  double value[3];                                                            
  double gmst; /*g m,[s,t] 列mのs行からt行までのgain*/
  int    optValueNo;

  gmst = wSub->w1N - *Ar1(wSub->w1u,s-1) - *Ar1(wSub->wvN,t+1);
  if(m == 1){                  /* type m  s  t*/
    setXYtbl(&Ar3(xyTbl,1,s,t)->fw, 0, 0, 0, 0, gmst);
  } else if(s == t){
    if(Ar3(xyTbl,m-1,s,t)->fw.weight < 0){
      setXYtbl(&Ar3(xyTbl,m,s,t)->fw,0,0,0,0,cntTbl->tbl[m][s].weight);
    }else{
      setXYtbl(&Ar3(xyTbl,m,s,t)->fw,1,m-1,s,t,cntTbl->tbl[m][s].weight +
                                               Ar3(xyTbl,m-1,s,t)->fw.weight);
    }
  } else {
    value[0] = Ar3(xyTbl,m-1,s,t)->fw.weight + gmst;
    value[1] = Ar3(xyTbl,m,s+1,t)->fw.weight + cntTbl->tbl[m][s].weight;
    value[2] = Ar3(xyTbl,m,s,t-1)->fw.weight + cntTbl->tbl[m][t].weight;
    optValueNo = maxValue(value,3);
    switch(optValueNo) {
    case 0:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fw, 1, m-1, s, t, value[optValueNo]);break;
    case 1:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fw, 1, m, s+1, t, value[optValueNo]);break;
    case 2: 
      setXYtbl(&Ar3(xyTbl,m,s,t)->fw, 1, m, s, t-1, value[optValueNo]);break;
    }
  }
}

/*文献では左下が(0、0)なので実質的にはUPではなくDOWNとなる*/
void getFu(
  struct CntTbl *cntTbl,
  struct WSUB   *wSub,
  struct XYTBL  *xyTbl,
  int m, int s, int t){

  double value[4];
  double gmst;
  int    optValueNo;
  int i[2];

  gmst = wSub->w1N - *Ar1(wSub->w1u,s-1) - *Ar1(wSub->wvN,t+1);
  if(m == 1){
    setXYtbl(&Ar3(xyTbl,1,s,t)->fu, 0, 0, 0, 0, gmst);
  } else {
    if(s == t){
      value[0] = cntTbl->tbl[m][t].weight;
    } else {
      value[0] = Ar3(xyTbl,m,s,t-1)->fu.weight + cntTbl->tbl[m][t].weight;
    }
    value[1] = maxFwis(cntTbl,xyTbl, m-1, 1, s, t, &i[1]) + gmst;
    value[2] = maxFuis(cntTbl,xyTbl, m-1, 1, s, t, &i[2]) + gmst;
    optValueNo = maxValue(value,3);
    switch(optValueNo) {
    case 0:
      if(s==t){
        setXYtbl(&Ar3(xyTbl,m,s,t)->fu, 0, 0, 0, 0,   value[optValueNo]);break;
      }else{
        setXYtbl(&Ar3(xyTbl,m,s,t)->fu, 2, m, s, t-1, value[optValueNo]);break;
      }
    case 1:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fu, 1, m-1, i[1], t,value[optValueNo]);break;
    case 2:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fu, 2, m-1, i[2], t,value[optValueNo]);break;
    case 3:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fu, 2, m  , s, t-1, value[optValueNo]);break;
    }
  }
}

void getFd(
  struct CntTbl *cntTbl,
  struct WSUB   *wSub,
  struct XYTBL  *xyTbl,
  int m, int s, int t){

  double value[4];
  double gmst;
  int    optValueNo;
  int i[2];

  gmst = wSub->w1N - *Ar1(wSub->w1u,s-1) - *Ar1(wSub->wvN,t+1);
  if(m == 1){
    setXYtbl(&Ar3(xyTbl,1,s,t)->fd, 0, 0, 0, 0, gmst);
  } else {
    if(s == t){
      value[0] = cntTbl->tbl[m][s].weight;
    } else {
      value[0] = Ar3(xyTbl,m,s+1,t)->fd.weight + cntTbl->tbl[m][s].weight;
    }
    value[1] = maxFwti(cntTbl,xyTbl, m-1, t, cntTbl->yCnt, s, &i[1]) + gmst;
    value[2] = maxFdti(cntTbl,xyTbl, m-1, t, cntTbl->yCnt, s, &i[2]) + gmst;
    optValueNo = maxValue(value,3);
    switch(optValueNo) {
    case 0:
      if(s==t){
        setXYtbl(&Ar3(xyTbl,m,s,t)->fd, 0, 0, 0, 0,   value[optValueNo]);break;
      }else{
        setXYtbl(&Ar3(xyTbl,m,s,t)->fd, 3, m, s+1, t, value[optValueNo]);break;
      }
    case 1:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fd, 1, m-1, s, i[1],value[optValueNo]);break;
    case 2:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fd, 3, m-1, s, i[2],value[optValueNo]);break;
    case 3:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fd, 3, m  , s+1, t, value[optValueNo]);break;
    }
  }
}

void getFn(
  struct CntTbl *cntTbl,
  struct WSUB   *wSub,
  struct XYTBL  *xyTbl,
  int m, int s, int t){

  double value[6];
  double gmst;
  int    optValueNo;

  gmst = wSub->w1N - *Ar1(wSub->w1u,s-1) - *Ar1(wSub->wvN,t+1);
  if(m == 1){
    setXYtbl(&Ar3(xyTbl,1,s,t)->fn, 0, 0, 0, 0, gmst);
  } else if(s==1 && t==cntTbl->yCnt){
    if(Ar3(xyTbl,m-1,s,t)->fn.weight < 0){
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn,0,0,0,0,gmst);
    }else{
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn,1,m-1,s,t,gmst +
                                               Ar3(xyTbl,m-1,s,t)->fn.weight);
    }
  } else {
    value[0] = Ar3(xyTbl,m-1,s,t)->fw.weight + gmst;
    value[1] = Ar3(xyTbl,m-1,s,t)->fu.weight + gmst;
    value[2] = Ar3(xyTbl,m-1,s,t)->fd.weight + gmst;
    value[3] = Ar3(xyTbl,m-1,s,t)->fn.weight + gmst;
    if(s>1){
      value[4] = Ar3(xyTbl,m,s-1,t)->fn.weight - cntTbl->tbl[m][s-1].weight;
    }else{
      value[4] = DBLMINVAL;
    }
    if(t<cntTbl->yCnt){
      value[5] = Ar3(xyTbl,m,s,t+1)->fn.weight - cntTbl->tbl[m][t+1].weight;
    }else{
      value[5] = DBLMINVAL;
    }
    optValueNo = maxValue(value,6);
    switch(optValueNo) {
    case 0:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn, 1, m-1, s, t, value[optValueNo]);break;
    case 1:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn, 2, m-1, s, t, value[optValueNo]);break;
    case 2:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn, 3, m-1, s, t, value[optValueNo]);break;
    case 3:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn, 4, m-1, s, t, value[optValueNo]);break;
    case 4:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn, 4, m, s-1, t, value[optValueNo]);break;
    case 5:
      setXYtbl(&Ar3(xyTbl,m,s,t)->fn, 4, m, s, t+1, value[optValueNo]);break;
    }
  }
}


/**
 * XY単調でPθの最適領域を求める
 */
void getOptPthetaXY(
  struct CntTbl *cntTbl,
  double         theta,
  struct SELREG *selReg){
  
  int m,i,j; 
  
  double optWeight=0;
  int    optType=0;
  int    optM=0;
  int    optI=0;
  int    optJ=0;
  
  int    m0;
  int    s0=0,s;
  int    t0=0,t;
  int    type; 
  double w0=0;
  
  struct XYTBL *xyTbl;
  struct WSUB    wSub;
  
  /*各種メモリ領域確保*/
  xyTbl     = mssCalloc((cntTbl->xCnt+1)*(cntTbl->yCnt+1)*(cntTbl->yCnt+1)
                                     *sizeof(struct XYTBL),"optPthetaXY1");
  wSub.w1i  = mssCalloc((cntTbl->yCnt+2)*sizeof(struct W1IJN),"optPthetaXY2");
  wSub.wjN  = mssCalloc((cntTbl->yCnt+2)*sizeof(struct W1IJN),"optPthetaXY3");
  wSub.w1u  = mssCalloc((cntTbl->yCnt+2)*sizeof(double),"optPthetaXY4");
  wSub.wvN  = mssCalloc((cntTbl->yCnt+2)*sizeof(double),"optPthetaXY5");


  calWeight(cntTbl,theta); /*全セルにおけるh-Osを求める*/
  for(m=1; m<=cntTbl->xCnt; m++){
    setWSub(cntTbl,m,&wSub);

    /*Fwはsとtの距離が少ないもの順に実行する*/
    for(i=1; i<=cntTbl->yCnt; i++){
      for(j=1; j<=cntTbl->yCnt-(i-1); j++){
        getFw(cntTbl,&wSub,xyTbl,m,j,j+(i-1));
      }
    }
    for(i=1; i<=cntTbl->yCnt; i++){
      for(j=i; j<=cntTbl->yCnt; j++){
        getFu(cntTbl,&wSub,xyTbl,m,i,j);
      }
    }
    for(i=cntTbl->yCnt; i>=1; i--){
      for(j=i; j>=1; j--){
        getFd(cntTbl,&wSub,xyTbl,m,j,i);
      }
    }
    /*Fnはsとtの距離が多いもの順に実行する*/
    for(i=cntTbl->yCnt; i>=1; i--){
      for(j=1; j<=cntTbl->yCnt-(i-1); j++){
        getFn(cntTbl,&wSub,xyTbl,m,j,j+(i-1));
      }
    }
  }

  optWeight=DBLMINVAL;
  for(m=1; m<=cntTbl->xCnt; m++){
    /*ウエイトの探索*/
    for(i=1; i<=cntTbl->yCnt; i++){
      for(j=i; j<=cntTbl->yCnt; j++){
        /*ここの条件式に＝を入れるか入れないかで結果が変わることがある。*/
        if( optWeight <= Ar3(xyTbl,m,i,j)->fw.weight ){
          optWeight=Ar3(xyTbl,m,i,j)->fw.weight;
          optType=1; optM=m; optI=i; optJ=j;
        }
        if( optWeight <= Ar3(xyTbl,m,i,j)->fu.weight ){
          optWeight=Ar3(xyTbl,m,i,j)->fu.weight;
          optType=2; optM=m; optI=i; optJ=j;
        }
        if( optWeight <= Ar3(xyTbl,m,i,j)->fd.weight ){
          optWeight=Ar3(xyTbl,m,i,j)->fd.weight;
          optType=3; optM=m; optI=i; optJ=j;
        }
        if( optWeight <= Ar3(xyTbl,m,i,j)->fn.weight ){
          optWeight=Ar3(xyTbl,m,i,j)->fn.weight;
          optType=4; optM=m; optI=i; optJ=j;
        }
      }
    }
  } /*for m*/

  /*最適領域のヒット,サポート,目的関数値の計算*/
  for(m=0; m<=cntTbl->xCnt; m++){
    for(i=0; i<=cntTbl->yCnt; i++){
      *Ar2(selReg->tbl,m,i) = 0;
    }
  }

  selReg->theta=theta;
  selReg->weight=optWeight;

  selReg->hit=0;
  selReg->sup=0;

  s=optI; t=optJ; type=optType;
  w0 = optWeight;
  m0 = optM;
  m  = 0;
  while(1){
    if(s==0 && t==0)break;
    /*列が変わったとき*/
    if(m0 != m){
      m=m0;

      for(i=s; i<=t; i++){
        *Ar2(selReg->tbl,m,i) = 1;
      }
    }

    switch(type){
    case 1:
      s0   = Ar3(xyTbl,m,s,t)->fw.s;
      t0   = Ar3(xyTbl,m,s,t)->fw.t;
      m0   = Ar3(xyTbl,m,s,t)->fw.m;
      w0   = Ar3(xyTbl,m,s,t)->fw.weight;
      type = Ar3(xyTbl,m,s,t)->fw.type;
      break;
    case 2:
      s0   = Ar3(xyTbl,m,s,t)->fu.s;
      t0   = Ar3(xyTbl,m,s,t)->fu.t;
      m0   = Ar3(xyTbl,m,s,t)->fu.m;
      w0   = Ar3(xyTbl,m,s,t)->fu.weight;
      type = Ar3(xyTbl,m,s,t)->fu.type;
      break;
    case 3:
      s0   = Ar3(xyTbl,m,s,t)->fd.s;
      t0   = Ar3(xyTbl,m,s,t)->fd.t;
      m0   = Ar3(xyTbl,m,s,t)->fd.m;
      w0   = Ar3(xyTbl,m,s,t)->fd.weight;
      type = Ar3(xyTbl,m,s,t)->fd.type;
      break;
    case 4:
      s0   = Ar3(xyTbl,m,s,t)->fn.s;
      t0   = Ar3(xyTbl,m,s,t)->fn.t;
      m0   = Ar3(xyTbl,m,s,t)->fn.m;
      w0   = Ar3(xyTbl,m,s,t)->fn.weight;
      type = Ar3(xyTbl,m,s,t)->fn.type;
      break;
    }
    s=s0; t=t0;
  }

  setHitSup(cntTbl,selReg);
  selReg->objVal = getObjVal(cntTbl,(double)selReg->hit,(double)selReg->sup, 0);

  mssFree(xyTbl);
  mssFree(wSub.w1i)  ; mssFree(wSub.wjN)  ; mssFree(wSub.w1u); mssFree(wSub.wvN);
}

/**
 * 最適領域の更新
 */
void cpSelReg(
  struct CntTbl *cntTbl,
  struct SELREG *selRegTo,
  struct SELREG *selRegFrom){

  int i,j;

  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
    *Ar2(selRegTo->tbl,i,j) = *Ar2(selRegFrom->tbl,i,j);
    }
  }

  selRegTo->theta     = selRegFrom->theta;
  selRegTo->weight    = selRegFrom->weight;
  selRegTo->sup       = selRegFrom->sup;
  selRegTo->hit       = selRegFrom->hit;
  selRegTo->objVal    = selRegFrom->objVal;
}

/**
 * 最適領域の保存
 */
void cpSelReg2(
  struct CntTbl *cntTbl,
  struct SELREG *selRegTo,
  struct SELREG *selRegFrom){

  int i,j;

  selRegTo->tbl=mssMalloc((cntTbl->xCnt+1)*(cntTbl->yCnt+1)*sizeof(char),"cpReg2");

  for(j=1; j<=cntTbl->yCnt; j++){
    for(i=1; i<=cntTbl->xCnt; i++){
      *(selRegTo->tbl+j*cntTbl->xCnt+i) = *Ar2(selRegFrom->tbl,i,j);
    }
  }

  for(i=0; i<cntTbl->xCnt; i++){
    selRegTo->rtbl[0][i].from = cntTbl->rtbl[0][i].from;
    selRegTo->rtbl[0][i].to   = cntTbl->rtbl[0][i].to;
  }

  for(i=0; i<cntTbl->yCnt; i++){
    selRegTo->rtbl[1][i].from = cntTbl->rtbl[1][i].from;
    selRegTo->rtbl[1][i].to   = cntTbl->rtbl[1][i].to;
  }

  selRegTo->theta     = selRegFrom->theta;
  selRegTo->xCnt      = cntTbl->xCnt;
  selRegTo->yCnt      = cntTbl->yCnt;
  selRegTo->theta     = selRegFrom->theta;
  selRegTo->weight    = selRegFrom->weight;
  selRegTo->sup       = selRegFrom->sup;
  selRegTo->hit       = selRegFrom->hit;
  selRegTo->objVal    = selRegFrom->objVal;
  selRegTo->variance  = selRegFrom->variance;
  selRegTo->entropy   = selRegFrom->entropy;
  selRegTo->gini      = selRegFrom->gini;
  selRegTo->accuracy  = selRegFrom->accuracy;

  selRegTo->supOX     = cntTbl->sup;
  selRegTo->posOX     = cntTbl->hit;
  selRegTo->negOX     = cntTbl->sup - cntTbl->hit;

  selRegTo->supO      = selRegFrom->sup;
  selRegTo->posO      = selRegFrom->hit;
  selRegTo->negO      = selRegFrom->sup - selRegFrom->hit;
  selRegTo->supX      = selRegTo->supOX - selRegTo->supO;
  selRegTo->posX      = selRegTo->posOX - selRegTo->posO;
  selRegTo->negX      = selRegTo->negOX - selRegTo->negO;
}

/**
 * 傾き(θmid),交点を求める
 */
double getIntersection(
  double s2,double h2,double t2,
  double s1,double h1,double t1,
  double *sup, double *hit      ){

  double b1,b2;

  b1=h1-t1*s1;
  b2=h2-t2*s2;

  *sup = (double)( (b2-b1)/(t1-t2) );
  *hit = (double)( *sup * t1 +h1 -t1*s1 );
  return( (double)(h1-h2)/(double)(s1-s2) );
}

/**
 * ConvexHull上の最適領域(目的関数最大化領域)を求める
 * 高速アルゴリズム版
 */
void convexHull(
  struct CntTbl *cntTbl,
  struct SELREG *optSelReg,
  struct SELREG *selReg,
  double thetaL,
  double thetaR){

  long supRegionL=0;
  long supRegionR=0;
  long supRegionM=0;

  long hitRegionL=0;
  long hitRegionR=0;
  long hitRegionM=0;

  double intSup; /*交点のサポート*/
  double intHit; /*交点のヒット*/

  double thetaM;

  getOptPtheta(cntTbl, thetaL, selReg);
  supRegionL = selReg->sup;
  hitRegionL = selReg->hit;

  getOptPtheta(cntTbl, thetaR, selReg);
  supRegionR = selReg->sup;
  hitRegionR = selReg->hit;

  thetaM     = getIntersection(
                     (double)supRegionL, (double)hitRegionL, (double)thetaL,
                     (double)supRegionR, (double)hitRegionR, (double)thetaR,
                     &intSup, &intHit);

  if(optSelReg->objVal > getObjVal( cntTbl, intHit, intSup, 0 ) ){
    return;
  }

  getOptPtheta(cntTbl, thetaM, selReg);
  supRegionM = selReg->sup;
  hitRegionM = selReg->hit;

/*
debug
printf("thetaLRM=%g,%g,%g\n",thetaL,thetaR,thetaM);
printf("sup  LRM=%ld,%ld,%ld\n",supRegionL,supRegionR,supRegionM);
printf("hit  LRM=%ld,%ld,%ld\n",hitRegionL,hitRegionR,hitRegionM);
printf("[%g]comp[%g]\n",selReg->objVal,optSelReg->objVal);
*/

  if( selReg->objVal > optSelReg->objVal ){
    cpSelReg(cntTbl,optSelReg,selReg);
  }

  if( supRegionM==supRegionL && hitRegionM==hitRegionL ) return;
  if( supRegionM==supRegionR && hitRegionM==hitRegionR ) return;

  if( supRegionM!=supRegionL || hitRegionM!=hitRegionL ){
    convexHull(cntTbl,optSelReg,selReg,thetaL,thetaM);
  }
  if( supRegionM!=supRegionR || hitRegionM!=hitRegionR ){
    convexHull(cntTbl,optSelReg,selReg,thetaM,thetaR);
  }
}

/**
 * ConvexHull上の最適領域(目的関数最大化領域)を求める
 * 高速アルゴリズム版
 */
void convexHullXY(
  struct CntTbl *cntTbl,
  struct SELREG *optSelReg,
  struct SELREG *selReg,
  double thetaL,
  double thetaR){

  long supRegionL=0;
  long supRegionR=0;
  long supRegionM=0;

  long hitRegionL=0;
  long hitRegionR=0;
  long hitRegionM=0;

  double intSup; /*交点のサポート*/
  double intHit; /*交点のヒット*/

  double thetaM;
  getOptPthetaXY(cntTbl, thetaL, selReg);
  supRegionL = selReg->sup;
  hitRegionL = selReg->hit;

  getOptPthetaXY(cntTbl, thetaR, selReg);
  supRegionR = selReg->sup;
  hitRegionR = selReg->hit;

  thetaM     = getIntersection(
                     (double)supRegionL, (double)hitRegionL, (double)thetaL,
                     (double)supRegionR, (double)hitRegionR, (double)thetaR,
                     &intSup, &intHit);


  if(optSelReg->objVal > getObjVal( cntTbl, intHit, intSup, 0 ) ){
    return;
  }

  getOptPthetaXY(cntTbl, thetaM, selReg);
  supRegionM = selReg->sup;
  hitRegionM = selReg->hit;

/* debug
printf("thetaLRM=%g,%g,%g\n",thetaL,thetaR,thetaM);
printf("sup  LRM=%ld,%ld,%ld\n",supRegionL,supRegionR,supRegionM);
printf("hit  LRM=%ld,%ld,%ld\n",hitRegionL,hitRegionR,hitRegionM);
printf("pTheta reg=%g ,opt=%g\n",selReg->theta,optSelReg->theta);
printf("[%g]comp[%g]\n",selReg->objVal,optSelReg->objVal);
*/

  if( selReg->objVal > optSelReg->objVal ){
    cpSelReg(cntTbl,optSelReg,selReg);
  }

  if( supRegionM==supRegionL && hitRegionM==hitRegionL )return;
  if( supRegionM==supRegionR && hitRegionM==hitRegionR )return;

  if( supRegionM!=supRegionL || hitRegionM!=hitRegionL ){
    convexHullXY(cntTbl,optSelReg,selReg,thetaL,thetaM);
  }
  if( supRegionM!=supRegionR || hitRegionM!=hitRegionR ){
    convexHullXY(cntTbl,optSelReg,selReg,thetaM,thetaR);
  }
}


/**
 * 長方形による最適領域の選択
 */
void rectangle(
  struct CntTbl *cntTbl,
  struct SELREG *optSelReg){

  struct HSSUB hs;

  int start,end;
  int startX, endX;
  int startY, endY;

  long hit,sup;
  double        objVal;
  double        maxObjVal;
  long area;
  long maxArea;

  long totalCnt;

  int i,j;

  hs.h1u=mssMalloc( (cntTbl->xCnt+1)*(cntTbl->yCnt+2)*sizeof(long),"rect");
  hs.s1u=mssMalloc( (cntTbl->xCnt+1)*(cntTbl->yCnt+2)*sizeof(long),"rect");
  hs.hvN=mssMalloc( (cntTbl->xCnt+1)*(cntTbl->yCnt+2)*sizeof(long),"rect");
  hs.svN=mssMalloc( (cntTbl->xCnt+1)*(cntTbl->yCnt+2)*sizeof(long),"rect");
  hs.h1N=mssMalloc( (cntTbl->xCnt+1)*sizeof(long),"rect");
  hs.s1N=mssMalloc( (cntTbl->xCnt+1)*sizeof(long),"rect");

  for(i=1; i<=cntTbl->xCnt; i++){
    *Ar2p1(hs.h1u,i,0)=0;
    *Ar2p1(hs.s1u,i,0)=0;
    for(j=1; j<=cntTbl->yCnt; j++){
      *Ar2p1(hs.h1u,i,j)=cntTbl->tbl[i][j].posCnt + *Ar2p1(hs.h1u,i,j-1);
      *Ar2p1(hs.s1u,i,j)=cntTbl->tbl[i][j].total  + *Ar2p1(hs.s1u,i,j-1);
    }
    *Ar2p1(hs.h1u,i,0)=0; *Ar2p1(hs.h1u,i,cntTbl->yCnt+1)=0;
    *Ar2p1(hs.s1u,i,0)=0; *Ar2p1(hs.s1u,i,cntTbl->yCnt+1)=0;
    *Ar2p1(hs.hvN,i,cntTbl->yCnt+1)=0;
    *Ar2p1(hs.svN,i,cntTbl->yCnt+1)=0;
    for(j=cntTbl->yCnt; j>=1; j--){
      *Ar2p1(hs.hvN,i,j)=cntTbl->tbl[i][j].posCnt + *Ar2p1(hs.hvN,i,j+1);
      *Ar2p1(hs.svN,i,j)=cntTbl->tbl[i][j].total  + *Ar2p1(hs.svN,i,j+1);
    }
    *Ar1(hs.h1N,i) = *Ar2p1(hs.h1u,i,cntTbl->yCnt);
    *Ar1(hs.s1N,i) = *Ar2p1(hs.s1u,i,cntTbl->yCnt);
  }


  maxObjVal = DBLMINVAL;
  maxArea   = -1;
  totalCnt = cntTbl->xCnt * cntTbl->yCnt;
  for(start=0; start<totalCnt; start++){
    for(end=start; end<totalCnt; end++){
      startX = start % cntTbl->xCnt + 1;
      startY = start / cntTbl->xCnt + 1;
      endX   = end   % cntTbl->xCnt + 1;
      endY   = end   / cntTbl->xCnt + 1;
      if(startX>endX) mssSwapInt(&startX, &endX);
      if(startY>endY) mssSwapInt(&startY, &endY);

      hit=0;
      sup=0;
      for(i=startX; i<=endX; i++){
        hit += *Ar1(hs.h1N,i)
             - *Ar2p1(hs.h1u,i,startY-1) - *Ar2p1(hs.hvN,i,endY+1);
        sup += *Ar1(hs.s1N,i)
             - *Ar2p1(hs.s1u,i,startY-1) - *Ar2p1(hs.svN,i,endY+1);
      }
      objVal = getObjVal(cntTbl,(double)hit,(double)sup, 0);
      area   = (endX-startX+1)*(endY-startY+1);

      /*もし目的関数値が最適領域より大きければ,もしくは*/
      /*目的関数値が最適領域と同じで,かつ,領域が大きければ*/
      /*optSelRegの更新 */
      if(maxObjVal <  objVal || (maxObjVal == objVal && area > maxArea) ){
        maxObjVal = objVal;
        maxArea   = area;
        for(i=1; i<=cntTbl->xCnt; i++){
          for(j=1; j<=cntTbl->yCnt; j++){
            if( i>=startX && i<=endX && j>=startY && j<=endY ){
              *Ar2(optSelReg->tbl,i,j) = 1;
            } else {
              *Ar2(optSelReg->tbl,i,j) = 0;
            }
          }
        }
        optSelReg->theta=0;
        optSelReg->weight=0;
        optSelReg->objVal = objVal;
        setHitSup(cntTbl,optSelReg);
      }
    }
  }
  free(hs.h1u); free(hs.s1u); free(hs.hvN);
  free(hs.svN); free(hs.h1N); free(hs.s1N);
}

/**
 * selRegに基づきXOの表示
 */
void dispSelReg(struct SELREG *selReg){

  /*int i,j=0;*/

  printf("------------------\n");
  printf("件数分布表\n");
  printf("------------------\n");
  printf("           領域       非領域     合計       合計比\n");
  printf("クラスYes [%8lu] [%8lu] [%8lu] [%8f]\n",
                             selReg->posO, selReg->posX, selReg->posOX,
                             (double)selReg->posOX/(double)selReg->supOX);
  printf("クラスNo  [%8lu] [%8lu] [%8lu] [%8f]\n",
                             selReg->negO, selReg->negX, selReg->negOX,
                             (double)selReg->negOX/(double)selReg->supOX);
  printf("合計      [%8lu] [%8lu] [%8lu]\n",
                             selReg->supO, selReg->supX, selReg->supOX);
  printf("合計比    [%8f] [%8f]\n",
                             (double)selReg->supO/(double)selReg->supOX,
                             (double)selReg->supX/(double)selReg->supOX);

  printf("\n");
  printf("------------------\n");
  printf("目的関数値など\n");
  printf("------------------\n");
  printf("     Theta   : [%f]\n",selReg->theta);
  printf("Max Weight   : [%f]\n",selReg->weight);
  printf("クラス間分散 : [%f]\n",selReg->variance);
  printf("エントロピー : [%f]\n",selReg->entropy);
  printf("GINI         : [%f]\n",selReg->gini);
  printf("正解率       : [%f]\n",selReg->accuracy);

  printf("\n");

  /*
  printf("------------------\n");
  printf("領域ルール\n");
  printf("------------------\n");
  for(i=1; i<=selReg->yCnt; i++){
    for(j=1; j<=selReg->xCnt; j++){
      if( *(selReg->tbl + selReg->xCnt*i+j) == 1 ){
        printf("O");
      }else{
        printf("X");
      }
    }
    printf("\n");
  }
  */
}

void copyCntTbl(struct CntCel *x, struct CntCel *y){
  x->negCnt = y->negCnt;
  x->posCnt = y->posCnt;
  x->total  = y->total;
  x->weight = y->weight;
} 
              
void swapCntTbl(struct CntCel *x, struct CntCel *y){
  mssSwapLng(&x->negCnt, &y->negCnt); 
  mssSwapLng(&x->posCnt, &y->posCnt);
  mssSwapLng(&x->total , &y->total);
  mssSwapDbl(&x->weight, &y->weight);
} 
    
/**
 * cntTbl行列の行と列を入れ換える(Y単調用)
 */
void xyConvertCntTbl(struct CntTbl *cntTbl){
  int iCnt,jCnt=0;
  int i,j;

  if(cntTbl->xCnt <= cntTbl->yCnt){
    iCnt=cntTbl->xCnt;
    jCnt=cntTbl->yCnt;
  } else {
    iCnt=cntTbl->yCnt;
    jCnt=cntTbl->xCnt;
  }

  for(i=1; i<=iCnt; i++){
    for(j=i+1; j<=jCnt; j++){
      if(j > iCnt){
        copyCntTbl(&cntTbl->tbl[j][i], &cntTbl->tbl[i][j]);
      }else{
        swapCntTbl(&cntTbl->tbl[j][i], &cntTbl->tbl[i][j]);
      }
    }
  }
  mssSwapInt(&cntTbl->xCnt, &cntTbl->yCnt);
}

/**
 * optRegion行列の行と列を入れ換える(Y単調用)
 */
void xyConvertSelReg(struct CntTbl *cntTbl, struct SELREG *selReg){
  int i,j;

  char *tblNew;

  tblNew = mssMalloc((cntTbl->yCnt+1)*(cntTbl->xCnt+1)*sizeof(char),"xyConv");

  for(i=1; i<=cntTbl->xCnt; i++){
    for(j=1; j<=cntTbl->yCnt; j++){
      /*新しいテーブルはXY逆になるのでAr2は使えない*/
      *(tblNew+i*cntTbl->yCnt+j) = *Ar2(selReg->tbl,i,j);
    }
  }
  /*古い領域テーブルを開放し,新しいテーブルを採用*/
  mssFree(selReg->tbl);
  selReg->tbl = tblNew;
}


/**
 * 領域を得る
 */
struct SELREG *segRegion(struct CntTbl *cntTbl, char *type, char *nul){
  static struct SELREG    *region;
  static struct SELREG    *selReg;
  static struct SELREG    *optSelReg;


/*----------------------------------------------------------------------------*/
/* メモリ確保                                                                 */
/*----------------------------------------------------------------------------*/
  /*X,Y,XY単調領域で使う一時的最適領域のメモリ確保(長方形領域では未使用)*/
  selReg=mssCalloc(sizeof(struct SELREG),"rseg");
  selReg->tbl=mssCalloc((cntTbl->xCnt+1)*(cntTbl->yCnt+1)*sizeof(char),"rseg");

  /*X,Y,XY単調,長方形領域で使う最終の最適領域のメモリ確保*/
  optSelReg=mssCalloc(sizeof(struct SELREG),"rseg");
  optSelReg->tbl=mssCalloc((cntTbl->xCnt+1)*(cntTbl->yCnt+1)*sizeof(char),"rseg");

  /*X,Y,XY単調領域で使う一時的最適領域のメモリ確保(長方形領域では未使用)*/
  region=mssCalloc(sizeof(struct SELREG),"rseg");

/*----------------------------------------------------------------------------*/
/* X単調領域                                                                  */
/*----------------------------------------------------------------------------*/
  if( strcmp(type,"x") == 0 ){
   /*supportがNULLのセルを調整(*1000)*/
    adjustNulCntTbl(cntTbl, nul);

    getOptPtheta(cntTbl, 9999, selReg);
    cpSelReg(cntTbl,optSelReg,selReg);
    getOptPtheta(cntTbl,-9999, selReg);
    if( selReg->objVal > optSelReg->objVal ){
      cpSelReg(cntTbl,optSelReg,selReg);
    }

    convexHull(cntTbl,optSelReg,selReg,-9999,9999);

    setAllObjVal(cntTbl, optSelReg);
    deajustNulCntTbl(cntTbl, optSelReg);
    cpSelReg2(cntTbl,region,optSelReg);
    
    /*dispCntTbl(cntTbl);*/
    /*dispCntTblHitRatio(cntTbl);*/

/*----------------------------------------------------------------------------*/
/* Y単調領域                                                                  */
/*----------------------------------------------------------------------------*/
  } else if( strcmp(type,"y") == 0 ){
   /*supportがNULLのセルを調整(*1000)*/
    adjustNulCntTbl(cntTbl, nul);

    xyConvertCntTbl(cntTbl);

    getOptPtheta(cntTbl, 9999, selReg);
    cpSelReg(cntTbl,optSelReg,selReg);
    getOptPtheta(cntTbl,-9999, selReg);
    if( selReg->objVal > optSelReg->objVal ){
      cpSelReg(cntTbl,optSelReg,selReg);
    }

    convexHull(cntTbl, optSelReg,selReg, -9999, 9999);

    xyConvertSelReg(cntTbl, optSelReg);
    xyConvertCntTbl(cntTbl);

    setAllObjVal(cntTbl, optSelReg);
    deajustNulCntTbl(cntTbl, optSelReg);
    cpSelReg2(cntTbl,region,optSelReg);

    /*dispCntTbl(cntTbl);*/
    /*dispCntTblHitRatio(cntTbl);*/

/*----------------------------------------------------------------------------*/
/* XY単調(rectlinear)領域                                                     */
/*----------------------------------------------------------------------------*/
  } else if( strcmp(type,"xy") == 0 ){
   /*supportがNULLのセルを調整(*1000)*/
    adjustNulCntTbl(cntTbl, nul);

    getOptPthetaXY(cntTbl, 9999, selReg);
    cpSelReg(cntTbl,optSelReg,selReg);
    getOptPthetaXY(cntTbl,-9999, selReg);

    if( selReg->objVal > optSelReg->objVal ){
      cpSelReg(cntTbl,optSelReg,selReg);
    }
    convexHullXY(cntTbl,optSelReg,selReg,9999,-9999);
    setAllObjVal(cntTbl, optSelReg);
    deajustNulCntTbl(cntTbl, optSelReg);
    cpSelReg2(cntTbl,region,optSelReg);

    /*dispCntTbl(cntTbl);*/
    /*dispCntTblHitRatio(cntTbl);*/

/*----------------------------------------------------------------------------*/
/* 長方形領域                                                                 */
/*----------------------------------------------------------------------------*/
  } else if( strcmp(type,"rect") == 0 ){
   /*supportがNULLのセルを調整(*1000)*/
    adjustNulCntTbl(cntTbl, nul);

    rectangle(cntTbl, optSelReg);
    cpSelReg2(cntTbl,region,optSelReg);

    /*dispCntTbl(cntTbl);*/
    /*dispCntTblHitRatio(cntTbl);*/
  }

  mssFree(selReg->tbl);
  mssFree(optSelReg->tbl);
  return(region);
}

