/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるハッシュ関連のヘッダーファイル
 * ============================================================================
 */

#include <musashi/mssHeader.h>
#include <musashi/mssValue.h>

#ifndef __MSSHASH_H
#define __MSSHASH_H 1

/**
 * # STRUCT #
 * 項目ハッシュの値を収める構造体
 * 項目-行のchar **pntへのポインタをハッシュテーブルに収める
 *
 * [0] [1] [2] [3] ...[HashValFld-1]->HashValFldに従ったHashValFld個の
 *                                                            ハッシュ配列
 * hn         ->rec : [0]
 * hn->next*1 ->rec : [0][1][2][3] ---->ここに各レコードの先頭項目アドレス
 * hn->next*2 ->rec : [0]
 *       :
 * hn->next*n ->rec : [0][1]
 *
 * 各ハッシュのバケットはhn->nextでHashNodeFldのリスト構造をとり、同一のhash値
 * を持つキーが連結されている。
 * ex) hn : キーがA,  hn->next : キーがZ,  hn->next->next : キーが001など
 * リスト終端はhn->rec==NULLで見分けている。
 * ここまでで、同一hash値をもつリスト化された各HashNodeFld構造体は、一つのキーに
 * 対応していることになる。
 * 次に、同一キー値を持ったレコードが複数存在する可能性がある。
 * それをhn->recにreallocで動的に領域を確保し、そこにそのレコードの先頭項目
 * のアドレスをセットしている。hn->recCntにはその件数が保存される。
 */
struct mssHashNodeFld {
  char ***rec;
  int     recCnt;
  struct mssHashNodeFld *next;
  int     endFlg; /*joinで使う。hashSearchFldされた時にセットされる。
                    traNullを認めるouterJoinで、このフラグが立っていない
                    レコードを書き出すことになる。*/
};

/**
 * # STRUCT #
 * 項目ハッシュの構造体
 */
struct mssHashFld {
  int hashVal;
  struct mssHashNodeFld **node;
  struct mssFields *flds;
  int fldCnt;
  int keyCnt; /*キーの値が何種類あったか==中味のあるHashNodeFldの数に等しい
                hashInsertFldでカウントアップされる。*/
  int endCnt; /*hashSearchFldされる度にカウントアップされる変数
                join処理で、マスタEndを判定させるために利用
                endCnt==keyCntで全てサーチされたこととみなす*/
};

/**
 * # STRUCT #
 * 汎用ハッシュの値を収める構造体
 */
struct mssHashNode {
  char  *str;
  MssValue  val;
  struct mssHashNode *last;
  struct mssHashNode *next;
  struct mssHashNode **top;
};

/**
 * # STRUCT #
 * 汎用ハッシュ構造体
 */
struct mssHash {
  int hashVal;
  int cnt;                 /*挿入された値の数*/
  struct mssHashNode **node; /*挿入された値へのポインタ*/
};

/**
 * # PROTOTYPE # 
 */
struct mssHashFld *	mssInitHashFld( int size, struct mssFields *flds);
void			mssFreeHashFld(struct mssHashFld *hash);
void			mssHashInsertFld(struct mssHashFld *hash, char **str);
struct mssHashNodeFld *	mssHashMemberFld(struct mssHashFld *hash,char **str, struct mssFields *flds);
void			mssShowHashFld(struct mssHashFld *hash, int fldCnt);
struct mssHash *	mssInitHash(int size);
void			mssFreeHash(struct mssHash *hash);
struct mssHashNode *	mssHashInsert(struct mssHash *hash, char *str, MssValue val);
struct mssHashNode *	mssHashInsertAdd(struct mssHash *hash, char *str, MssValue val);
struct mssHashNode *	mssHashDeleteNode(struct mssHash *hash, struct mssHashNode *node);
void			mssHashDelete(struct mssHash *hash, char *str);
struct mssHashNode *	mssHashMember(struct mssHash *hash, char *str);
char *			mssHashMemberAdd(struct mssHash *hash, char *str);
MssValue		mssHashMemberVal(struct mssHash *hash, char *str);
void			mssShowHash(struct mssHash *hash);

#endif
