/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるXMLtableデータの入力関連のヘッダーファイル
 * ============================================================================
 */

#include <stdio.h>
#include <zlib.h>

#include <musashi/mssHeader.h>
#include <musashi/mssConfig.h>
#include <musashi/mssOutput.h>
#include <musashi/mssOption.h>


#ifndef __MSSINPUT_H
#define __MSSINPUT_H 1

/**
 * # DEFINE #
 * マージソートを行う際に、何個までのファイルを同時に併合させるか
 */
#define PWayS 25

/**
 * # DEFINE #
 * 分割ソートでのデータ読み込みバッファサイズの上限値
 * この値、もしくはMssMaxLineSortのいずれかが上限値に達したときのデータが
 * 分割ソートにおける一つのバケットとなる。
 */
#define MaxMemS 2048000

/**
 * # DEFINE #
 * 分割ソートでのデータ読み込み行数の上限値
 * この値、もしくはMssMaxMemSortのいずれかが上限値に達したときのデータが
 * 分割ソートにおける一つのバケットとなる。
 */
#define MaxLineS 50000

/**
 * # DEFINE #
 * パイプのバッファサイズ
 */
#define PipeSize 4096

/**
 * # DEFINE #
 * リングバッファの一つのキューのブロック数
 */
#define ReadCnt 4

/**
 * # DEFINE #
 * FRM構造体におけるポインタバッファのメモリ量上限値
 */
#define MaxPntSize 2048000

/**
 * # STRUCT #
 * トーナメントツリーのデータ構造
 *
 *                 (1)
 *          +-------+--------+
 *          |                |
 *         (2)              (3)
 *    +-----+------+     +---+--+
 *    |            |     |      |
 *   (4)          (5)   (6)    (7)
 *  +--+--+     +--+--+
 *  |     |     |     |
 * (8)   (9)  (10)  (11)
 * 各ノード毎にTTnode構造体が張りつけられる
 * ノード番号(4)についてみると、
 * node->parent  は (2)
 * node->brother は (5)となる。
 * node->key はTTkey構造体へのポインタで実際の文字列を示すと考えればよい。
 * node->key->strが実際の文字列へのポインタを示し
 * node->key->bktはそのstr文字列がPwayマージソートで分割されたバケット番号
 *                                                          (ファイル番号)
 */
struct TTnode {
  int num;           /*ノード番号*/
  struct TTkey {
    char   *str;     /*キーの文字列*/
    int     bkt;     /*分割ソートにおけるバケット番号*/
  } *key;
  struct TTnode *parent;  /*親ノード*/
  struct TTnode *brother; /*兄弟ノード*/
};

/**
 * # STRUCT #
 * ソート用のデータ構造
 * ソートのアルゴリズムは、実装において非常に複雑になっている。
 * 複雑さの原因は、数値や逆順ソートと通常の文字ソートを同一プログラムで
 * 扱っているところになる。
 * 数値逆順ソートでは全ての処理において、キーの比較を項目単位で行っているため
 * 読み込みおよびTNtreeでは項目を区切って処理している。
 * 一方で、通常の文字ソートの場合は、最初にデータを読み込む際に、項目をキー順
 * に並べ換える。そうすることによって、後の処理では項目を区切ることなく処理
 * できる。比較は、一行単位で先頭から文字比較をすればよい。
 * このアルゴリズムの違いによって、通常の文字ソートはかなり高速になる。
 * 500万件のベンチマークにおいて、約1.3倍の高速化が実現できている。
 * 両者のソートを数値逆順ソートにあわせることもできるが、そうすると、
 * プログラムが非常にシンプルになる代わりに、処理が遅くなる。
 * 通常の利用においては、単純な文字ソートの方が利用率が圧倒的に多いと予測される
 * ので、プログラムのシンプルさより、高速性を選んだ。
 */
struct mssSortDat {
  struct TTnode  *tt;               /*トーナメントツリー(プライオリティキュー)*/
  struct mssFPR  *iFile[PWayS];     /*入力ワークファイルポインタ*/
  struct mssRec  *rec[PWayS];       /*入力ワークファイル読み込みバッファ*/
  struct mssFldRec *fr[PWayS];        /*数値や逆順ソートでの読み込みバッファ*/
  char            prefix[MssFileNameMaxLen]; /*ワークファイル名のprefix*/
  int             iStart;           /*開始入力ファイル番号*/
  int             iEnd;             /*終了入力ファイル番号*/
  int             bktCnt;           /*バケットの数(iEnd-iStart+1)*/
  struct mssFields *flds;           /*ソート項目*/
  int             fldCnt;           /*データの全項目数*/
  int             recCnt;           /*データのレコード数*/
  int             conv[MssFieldMaxCnt];/*要素:ソート項目順番 値:infileの項目番号*/
  int             cmpRevNum;        /*ソートにおけるキー比較を*/
                                    /* 0:単純な先頭からの文字列比較で行う*/
                                    /* 1:数値、逆順を考慮して比較を行う*/
                                    /* 追加(2004/08/03) */
};

/**
 * # STRUCT #
 * 入力バッファの構造体(リングバッファ)
 *
 * バッファの全体像                  １つのキューの内容(読み込み単位)
 * fp->que -----+---------------+       
 *              |MssRecordMaxLen|    / +------------+
 *              +---------------+   /  |PipeSize    |0
 *             0|fp->queSize    |  /   +------------+
 *              |               | /    |            |1
 *              +---------------+/     +------------+
 *             1|               |      |     :      |
 *              |               |      |     :      |
 *              +---------------+\     |     :      |
 *              |       :       | \    |     :      |
 *              |       :       |  \   +------------+
 *              |       :       |   \  |            |ReadCnt
 *              +---------------+    \ +------------+
 *              |               |
 *    fp->queCnt|               |
 *              +---------------+
 */
struct mssFPR {
  char *fName;  /*ファイル名へのポインタ(NULL:標準入力)*/
  char *curPnt; /*現在の読み込みポインタ*/
  char *buf;    /*リングバッファデータ本体*/
  int   queCnt; /*queueの数 2の累乗でなければならない*/
  int   queSize;/*queのサイズ=queCnt*PipeSize(毎回計算で求めるのが遅いので)*/
  int   enq;    /*次に読みこむバッファ番号*/
  int   deq;    /*有効なバッファの開始番号*/
  int   full;   /*バッファfullフラグ*/
  int   last;   /*eofを含む読み込みを行ったフラグ*/
  int   recCnt; /*これまでに読み込んだレコードのカウント*/
  int   chrCnt; /*これまでに読み込んだ文字のカウント*/

  int   zflg;   /*圧縮ファイル(1)か、通常ファイル(0)か?*/
  gzFile zfd;   /*圧縮ファイル*/
  FILE *fp;     /*通常ファイル*/

  int   sort;   /*通常ファイル(0)かソートファイル(1)か? reopenFPRsortでセット*/
  struct mssSortDat *sd; /*ソートファイルオブジェクト*/
  char *readPnt;         /*データの読み込みポイント*/
};

/**
 * # STRUCT #
 * 一行読み込み構造体
 * 項目による分割はおこなわない。
 */
struct mssRec {
  char *pnt;    /*行の先頭へのポインタ*/
  int   chrCnt; /*読みこんだ文字数*/
  int   eof;    /*ファイルを読み切った時にセットされる*/
};

/**
 * # STRUCT #
 * 項目によるトークン分割をともなった一行単位読み込み構造体
 */
struct mssFldRec {
  char **pnt;   /*各項目へのポインタ*/
  int fldCnt;   /*項目数(初期化initFldRec時にセットされる)*/
  int   chrCnt; /*読みこんだ文字数*/
  int   eof;
};

/**
 * # STRUCT #
 * 項目によるトークン分割をともなった二行単位読み込み構造体
 * キーブレーク処理を行うときに利用する。
 */
struct mssFldRecDbl {
  char **pnt[2]; /*各行各項目へのポインタ*/
  int new;       /*0 or 1: pnt[new]で現在行の項目をさす*/
  int old;       /*0 or 1: pnt[old]で前行の項目をさす*/
  int firstRead; /*最初の一行のキーブレーク時にのみ例外処理するためのフラグ*/
  int eof;       /*ファイルを読み切った時にセットされる*/
  int newChrCnt; /*読みこんだ文字数(new)*/
  int oldChrCnt; /*読みこんだ文字数(old)*/
  int fldCnt;    /*項目数(初期化initFRD時にセットされる)*/
};

/**
 * # STRUCT #
 * 項目によるトークン分割をともなったn行単位読み込み構造体
 * nの値はFPRのバッファの大きさ、もしくはMaxPntSizeで決まる。
 */
struct mssFldRecMax {
  char **pnt;      /*各行各項目へのポインタ
                    (*(pnt+fldCnt*行番号)+項目番号)で行番号の項目番号をさす*/
  int    fldCnt;   /*項目数(初期化initFldRec時にセットされる)*/
  int    chrCnt;   /*読みこんだ文字数*/
  int    eof;      /*ファイルを読み切った時にセットされる*/
  int    recCnt;   /*読み込んだ行数*/
  int    recMax;   /*MaxPntSizeから計算される、読み込み行数の上限値*/
};

/**
 * # STRUCT #
 * 項目によるトークン分割をともなったキー単位読み込み構造体
 */
struct mssFldRecKey {
  char **pnt;       /*一行単位の項目ポインタ*/
  int    fldCnt;    /*項目数(初期化initFRK時にセットされる)*/
  int    chrCnt;    /*読みこんだ文字数*/
  int    eof;       /*ファイルを読み切った時にセットされる*/
  int    keyRecCnt; /*現在のキーに含まれる行数*/
  int    keyNo;     /*現在の通しキー番号*/

  char **frPnt;     /*項目行を格納するポインタ*/
  int firstRead;    /*最初の一行のキーブレーク時にのみ例外処理するためのフラグ*/
  int    alcRecCnt; /*pntに領域確保された行数*/
  int    recPnt;    /*pntに読み込む位置(行番号)*/
  MssOptKEY *optKey;   /*キー*/
  int     byFile;   /*一時ファイルに出力されたかどうかのフラグ*/
  char   fName[MssFileNameMaxLen]; /*一時ファイル名*/
  struct mssFPW *fpw;  /*内部で利用するワークファイル用*/
  struct mssFPR *fpr;/*内部で利用するワークファイル用*/
  struct mssFldRec *fr;/*ワークファイルから行単位で読み込むため*/
  int    curRecNo;  /*現在処理中の行番号*/
};

/**
 * # STRUCT #
 * ファイルステータスの構造体
 */
struct mssFileInfo {
  int   maxCnt;    /* 0:データ部を完全に読み込む n:n行まで*/
  char *fName;     /*ファイル名*/
  int   totalSize; /*-tの時はtotalSize==bodySize*/
  int   readEnd;   /*最終行まで読み込んだかどうか*/
  int   bodySize;  /*データ本体*/
  int   fldCnt;    /*項目数*/
  int   recCnt;    /*レコード数*/
};

/**
 * # PROTOTYPE # 
 */
struct mssFileInfo * 	mssGetFileInfo(char *fName, int recMax);
struct mssRec *		mssInitRec(void);
void        		mssFreeRec(struct mssRec *rec);
int         		mssReadRec(struct mssFPR *fp, struct mssRec *rec);
struct mssFldRec *	mssInitFldRec(int fldCnt);
void           		mssFreeFldRec(struct mssFldRec* fr);
int            		mssReadFldRec(struct mssFPR *fp, struct mssFldRec *fr);
int 			mssReadRandFldRec(struct mssFPR *fp, struct mssFldRec *fr, int recNo, int datTop, int recLen);
struct mssFldRecDbl *	mssInitFRD(int fldCnt);
void              	mssFreeFRD(struct mssFldRecDbl *frd);
int               	mssReadFRD(struct mssFPR *fp, struct mssFldRecDbl *frd);
int               	mssKeyBreak(struct mssFldRecDbl *frd, MssOptKEY *optKey);
struct mssFldRecMax *	mssInitFRM(int fldCnt);
void 			mssFreeFRM(struct mssFldRecMax *frm);
int 			mssReadFRM(struct mssFPR *fp, struct mssFldRecMax *frm);
struct mssFldRecKey *	mssInitFRK(int fldCnt,MssOptKEY *optKey,char *path);
void              	mssFreeFRK(struct mssFldRecKey *frk);
int               	mssReadFRK(struct mssFPR *fp, struct mssFldRecKey *frk);
int			mssReadFldRecFRK(struct mssFldRecKey *frk);
void			mssSeekTopFRK(struct mssFldRecKey *frk);
int 			mssGetFldCntOnData(struct mssFPR *fp);
void 			mssSeekTopFPR(struct mssFPR *fp);
struct mssFPR *		mssOpenFPR(char *fileName, int bufCnt);
struct mssFPR *		mssReopenFPRsort(struct mssFPR *inFile, int queCnt, struct mssFields *flds, int fldCnt, char *tmpPath);
void			mssCloseFPR(struct mssFPR *fp);
struct mssFPR *		mssOpenFPU(char *fileName, int queCnt);
void        		mssCloseFPU(struct mssFPR *fp);
#endif /* _INPUT_H */
