/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられる集計関連の関数
 * ============================================================================
 */
#include <musashi/mssBase.h>
#include <musashi/mssAgg.h>
#include <musashi/mssValue.h>
#include <musashi/mssOption.h>
#include <musashi/mssInput.h>
#include <musashi/mssOutput.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

static MssValue *getAggValMem(enum MssAggType aggType, int fc);

/**
 * グローバル変数
 */
extern struct mssGlobalVariables mssGV;

static MssOptKEY *OptKey;           /*-kパラメータ*/
static MssOptFLD *OptFld;           /*-fパラメータ*/
static int  FldCnt;              /*集計する項目数*/
static enum MssAggType AGGType;  /*集計方法("sum","avg",...)*/
static char fname[MssFileNameMaxLen]; /*一時ファイル名*/
static char readEnd[2]={0xff,0}; /*番兵*/

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 赤黒木(Red Black Tree)
 * ----------------------------------------------------------------------------
 * 参考文献: 渡部敏正『データ構造とアルゴリズム』共立出版,2000,6章4節.
 * p.224 下から9行目は誤り!!
 * 誤) if (v == NULL ) return(1) -> 正) if (v == NULL ) return(0)
 */

/**
 * # FUNCTION #
 */
static int RBAGhasLeftNode(struct RBAGnode *v)
{
  if(v->left->rank !=0){
    return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBAGisExternalNode(struct RBAGnode *v)
{
  if(v->rank==0)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBAGisTopNode(struct RBAGnode *v)
{
  if(v->parent->rank==0){
    return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBAGisLeftNode(struct RBAGnode *v)
{
  if(v==v->parent->left)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBAGisRedNode(struct RBAGnode *v)
{
  if(v==NULL) return(0);
  if(v->rank == v->parent->rank)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static struct RBAGnode *RBAGfindMax(struct RBAGnode *v)
{
  while(!RBAGisExternalNode(v->right))
    v=v->right;
  return(v);
}

/**
 * # FUNCTION #
 */
static struct RBAGnode *RBAGfind_brother(struct RBAGnode *v)
{
  if(RBAGisLeftNode(v))
    return(v->parent->right);
  return(v->parent->left);
}

/**
 * # FUNCTION #
 */
static struct RBAGnode *RBAGmakeNode(void)
{
  struct RBAGnode *new;
  struct RBAGkey  *key;
  new=(struct RBAGnode *)mssMalloc(sizeof(struct RBAGnode),"RBAGmkNode");
  key=(struct RBAGkey  *)mssMalloc(sizeof(struct RBAGkey) ,"RBAGmkNode");
  new->key=key;
  /*new->key.str[0]='\0';*/
  new->key->str=NULL;
  new->rank = 0;
  new->left=new->right=NULL;
  return new;
}

/**
 * # FUNCTION #
 */
static void RBAGfreeKey(struct RBAGkey *key)
{
  mssFree(key->str);
  mssFree(key->fld);
  mssFree(key->val);
  mssFree(key->bkt);
  mssFree(key);
}

/**
 * # FUNCTION #
 */
static void RBAGfreeNode(struct RBAGnode *v)
{
  if(RBAGisExternalNode(v)){
    mssFree(v->key);
    mssFree(v);
    return;
  }else{
    RBAGfreeKey(v->key);
    mssFree(v);
    return;
  }
}

/**
 * # FUNCTION #
 */
static void RBAGfreeAllNode(struct RBAGnode *v)
{
  if(RBAGisExternalNode(v)){
    mssFree(v->key);
    mssFree(v);
    return;
  }else{
    RBAGfreeAllNode(v->left);
    RBAGfreeAllNode(v->right);
    RBAGfreeNode(v);
    return;
  }
}

/**
 * # FUNCTION #
 */
static int keyCmp(struct mssFldRec *fr, struct RBAGnode *v)
{
  int i;
  int cmp;

  for(i=0; i<OptKey->flds->cnt; i++){
    if( 0 != ( cmp=strcmp( *(fr->pnt+MssFlds2num(OptKey->flds,i)),
                           *(v->key->fld+MssFlds2num(OptKey->flds,i)) )) ){
      return(cmp);
    }
  }
  return(0); /*同じキー!!*/
}

/**
 * # FUNCTION #
 */
static struct RBAGnode *RBAGmember(struct mssFldRec *fr, struct RBAGnode *v)
{
  struct RBAGnode *vv;

  if(RBAGisExternalNode(v)) return(v);

  if( 0 > keyCmp(fr, v) ){
    vv=RBAGmember(fr,v->left);
  }else if( 0 < keyCmp(fr, v) ){
    vv=RBAGmember(fr,v->right);
  }else{
    return(v);
  }
  return(vv);
}

/**
 * # FUNCTION #
 */
static void RBAGaddExternalNodes(struct RBAGnode *n)
{
  n->left = RBAGmakeNode();
  n->right= RBAGmakeNode();
  n->left->parent =n;
  n->right->parent=n;
}

/**
 * # FUNCTION #
 */
static void RBAGsingleRotate(struct RBAGnode *v)
{
  struct RBAGnode *p, *pp,*ppp;

  p  =v->parent;
  pp =p->parent;
  ppp=pp->parent;

  if(RBAGisLeftNode(pp)){
    ppp->left=p;
  }else{
    ppp->right=p;
  }
  if(RBAGisLeftNode(v)){
    pp->left=p->right;
    pp->left->parent=pp;
    p->right=pp;
    pp->parent=p;
  }else{
    pp->right = p->left;
    pp->right->parent = pp;
    p->left = pp;
    pp->parent = p;
  }
  p->parent=ppp;
}

/**
 * # FUNCTION #
 */
static void RBAGdouble_rotate(struct RBAGnode *v)
{
  struct RBAGnode *p;
  p=v->parent;
  if(RBAGisLeftNode(v)){
    RBAGsingleRotate(v->left);
    RBAGsingleRotate(p);
  }else{
    RBAGsingleRotate(v->right);
    RBAGsingleRotate(p);
  }
}

/**
 * # FUNCTION #
 */
static void RBAGrebalanceOnInsert(struct RBAGnode *v)
{
  struct RBAGnode *w, *p, *pp;
  int p_is_left;
  
  p=v->parent;
  pp=p->parent;
  if(RBAGisTopNode(p)){
    return;
  }
  if(RBAGisRedNode(p)){
    p_is_left=0;   
    if((p_is_left=RBAGisLeftNode(p))){  
      w=pp->right;
    }else{
      w=pp->left;
    }
    if(RBAGisRedNode(w)){   
      (pp->rank)++;
      if(!RBAGisTopNode(pp) && RBAGisRedNode(pp->parent)){
        RBAGrebalanceOnInsert(pp);
      }
    }else{
      if(RBAGisLeftNode(v)){
        if(p_is_left){
          RBAGsingleRotate(v);
        }else{
          RBAGdouble_rotate(v);
        }
      }else{
        if(p_is_left){
          RBAGdouble_rotate(v);
        }else{
          RBAGsingleRotate(v);
        }
      }
    }
  }
}

/**
 * # FUNCTION #
 */
static void RBAGdetermineAB(struct RBAGnode **a, struct RBAGnode **b, struct RBAGnode *v, struct RBAGnode *u)
{
  if(RBAGisLeftNode(v)){
    *a=u->right;
    *b=u->left;
  }else{
    *a=u->left;
    *b=u->right;
  }
}

/**
 * # FUNCTION #
 */
static void RBAGrebalanceOnDelete(struct RBAGnode *v)
{
  struct RBAGnode *p,*u,*a,*b;
  int v_is_left, p_was_red;

  p=v->parent;
  if(v->rank+1 >= p->rank)
    return;
  u=RBAGfind_brother(v);
  RBAGdetermineAB(&a,&b,v,u);
  v_is_left=RBAGisLeftNode(v);
  p_was_red=RBAGisRedNode(p);
  if(RBAGisRedNode(u)){
    if(RBAGisRedNode(a) || RBAGisRedNode(b))
      return;
    RBAGsingleRotate(a);
    RBAGrebalanceOnDelete(v);
  }else{
    if(!RBAGisRedNode(a)){
      if(!RBAGisRedNode(b)){
        (p->rank)--;
        if(!p_was_red)
          RBAGrebalanceOnDelete(p);
      }else{
        RBAGdouble_rotate(b);
        (b->rank)++;
        (p->rank)--;
      }
    }else{
      RBAGsingleRotate(a);
      (u->rank)++;
      (u->rank)--;
    }
  }
}


/**
 * # FUNCTION #
 */
static void RBAGptree(struct RBAGnode *p,int h)
{
  int i,j;

  if(!RBAGisExternalNode(p)){
    RBAGptree(p->left, h+1);
    if(0 == strcmp(p->key->str,readEnd) ){
      for(i=1; i<=h; i++) fprintf(stderr,"    ");
      printf("key='EOF' ");
      /*printf(" bktCnt=%d bkt=",p->key->bktCnt);*/
      for(j=0; j<PWayA; j++){
        if(*(p->key->bkt+j) == 1)
        printf("%d ",j);
      }
      printf("\n");
    }else{
      for(i=0; i<h; i++) printf("  ");
      printf("key='");
      for(j=0; j<OptKey->flds->cnt; j++){
        printf("%s ",*(p->key->fld+MssFlds2num(OptKey->flds,j)));
      }
      printf("' fld=");
      for(j=0; j<OptFld->flds->cnt; j++){
        printf("%g ",(p->key->val+j)->v.d);
      }

      /*printf(" bktCnt=%d bkt=",p->key->bktCnt);*/
      for(j=0; j<PWayA; j++){
        if(*(p->key->bkt+j) == 1)
        printf("%d ",j);
      }
      /*for(j=0; j<p->key->bktCnt; j++){*/
      /*  printf("%d ",*(p->key->bkt+j));*/
      /*}*/
      printf("\n");
    }   
    RBAGptree(p->right,h+1);
  }
}

/**
 * # FUNCTION #
 * RBAGcpKeyの関数で確保されたメモリ量の計算
 */
static int RBAGcpKeyMemCnt(struct mssFldRec *fr)
{

  int memCnt=0;

  if(fr->eof==1){ /*EOF*/
    /*文字列として番兵を入れておく*/
    memCnt+=2*sizeof(char);

    /*キー項目のアドレスを全て番兵に設定する*/
    memCnt+=FldCnt*sizeof(char *);

    /*sortAGでは以下のbkt変数は利用していない。mergeRBAGで利用。*/
    memCnt+=(sizeof(int)*PWayA);

    /*ダミーで作っておく(freeのため)*/
    memCnt+=(sizeof(MssValue));

  }else{

    /*frの全項目をまるごとコピー*/
    memCnt+=fr->chrCnt*sizeof(char);

    /*キー項目のアドレスを計算によりセット*/
    memCnt+=FldCnt*sizeof(char *);

    switch(AGGType){
    case aggSUM:
    case aggCNT:
    case aggMIN:
    case aggMAX:
      memCnt=sizeof(MssValue)*(OptFld->flds->cnt)*1;
      break;
    case aggAVG:
    case aggAVG2:
      memCnt=sizeof(MssValue)*(OptFld->flds->cnt)*2;
      break;
    case aggSTD:
    case aggSTDP:
    case aggVAR:
    case aggVARP:
      memCnt=sizeof(MssValue)*(OptFld->flds->cnt)*3;
      break;
    }

    /*sortAGでは以下のbkt変数は利用していない。mergeRBAGで利用。*/
    memCnt+=(sizeof(int)*PWayA);
  }
  return(memCnt);
}

/**
 * # FUNCTION #
 */
static void RBAGcpKey(struct RBAGnode *v, struct mssFldRec *fr, int bkt, MssValue *val)
{

  int i;

  if(fr->eof==1){ /*EOF*/
    /*文字列として番兵を入れておく*/
    v->key->str = mssMalloc(2*sizeof(char),"RBAGAGtree1");
    memcpy(v->key->str, readEnd, 2);

    /*キー項目のアドレスを全て番兵に設定する*/
    v->key->fld = mssMalloc(FldCnt*sizeof(char *),"RBAGAGtree2");
    for(i=0; i<FldCnt; i++){
      *(v->key->fld+i) = v->key->str;
    }

    /*sortAGでは以下のbkt変数は利用していない。mergeRBAGで利用。*/
    v->key->bkt    = mssCalloc(sizeof(int)*PWayA, "RBAGtree");
    *(v->key->bkt+bkt) = 1;
    /**v->key->bkt    = bkt;*/
    /*v->key->bktCnt = 1;*/

    /*ダミーで作っておく(freeのため)*/
    v->key->val = mssMalloc(sizeof(MssValue),"RBAGAGtree3");

  }else{

    /*frの全項目をまるごとコピー*/
    v->key->str = mssMalloc(fr->chrCnt*sizeof(char),"RBAGAGtree4");
    memcpy(v->key->str, *fr->pnt, fr->chrCnt); 

    /*キー項目のアドレスを計算によりセット*/
    v->key->fld = mssMalloc(FldCnt*sizeof(char *),"RBAGAGtree5");
    for(i=0; i<FldCnt; i++){
      *(v->key->fld+i) = v->key->str + (*(fr->pnt+i) - *fr->pnt);
    }

    /*集計値領域の確保*/
    v->key->val=getAggValMem(AGGType, OptFld->flds->cnt);

    /*集計値の計算*/
    mssCalAggVal(v->key->val, AGGType, val);

    /*sortAGでは以下のbkt変数は利用していない。mergeRBAGで利用。*/
    v->key->bkt    = mssCalloc(sizeof(int)*PWayA, "RBAGtree");
    *(v->key->bkt+bkt) = 1;
  }
}

/**
 * # FUNCTION #
 * 指定のノードをFldRec構造体のデータで更新
 */
static void RBAGupdate(struct RBAGnode *v, struct mssFldRec *fr, int bkt, MssValue *val)
{
  /*更新したバケット番号をバケットフラグに更新する*/
  *(v->key->bkt+bkt) = 1;

  if(fr->eof!=1){ /*not EOF*/
    /*集計値の計算*/
    mssCalAggVal(v->key->val, AGGType, val);
  }
}

/**
 * # FUNCTION #
 * RBtreeにFldRec構造体のデータを追加
 * 既にキーが存在すればRBAGupdateで値を更新
 */
static int RBAGinsert(struct RBAGnode *v, struct mssFldRec *fr, int bkt, MssValue *val)
{
  int memCnt=0;

  v = RBAGmember(fr, v);

  /*キーが存在しなければ追加*/
  if(RBAGisExternalNode(v)){
    RBAGcpKey(v, fr, bkt, val);
    memCnt=RBAGcpKeyMemCnt(fr);
    v->rank=1;
    RBAGaddExternalNodes(v);
    RBAGrebalanceOnInsert(v);
    memCnt+=sizeof(struct RBAGnode)*2;
    memCnt+=sizeof(struct RBAGkey )*2;

  /*キーが存在すれば更新*/
  }else{
    RBAGupdate(v, fr, bkt,val);
  }

  return(memCnt);
}


/**
 * # FUNCTION #
 * 指定のノードを削除
 */
static void RBAGdeleteNode(struct RBAGnode *v)
{
  struct RBAGnode *w,*x;

  if(RBAGisExternalNode(v)){
    fprintf(stderr,"Not found such node\n");
    return;
  }
  if(RBAGhasLeftNode(v)){
    w=RBAGfindMax(v->left);
    if(w->parent != v){
      w->parent->right = w->left;
      w->left->parent  = w->parent;
    }
    x=w->left;
    RBAGfreeNode(w->right);
    /*free(w->right->key);*/
    /*free(w->right);*/

    if(RBAGisLeftNode(v)){
      v->parent->left=w;
    }else{
      v->parent->right=w;
    }
    if(w->parent != v){
      w->left = v->left;
      w->left->parent = w;
    }
    w->rank = v->rank;
    w->parent = v->parent;
    w->right = v->right;
    w->right->parent=w;
    RBAGfreeNode(v);
    /*free(v->key);*/
    /*free(v);*/
  }else{
    w=v->right;
    if(RBAGisLeftNode(v)){
      v->parent->left=w;
    }else{
      v->parent->right=w;
    }
    w->parent=v->parent;
    RBAGfreeNode(v->left);
    RBAGfreeNode(v);
    /*free(v->left->key);*/
    /*free(v->left);*/
    /*free(v->key);*/
    /*free(v);*/
    x=w;
  }

  RBAGrebalanceOnDelete(x);
}

/**
 * # FUNCTION #
 * RBtreeから優先行のノードアドレスを返す
 * この処理でノードを削除はしない。呼び出す本体側でRBAGdeleteNode(node)を実行
 */
static struct RBAGnode *RBAGpop(struct RBAGnode *v)
{
  struct RBAGnode *vv;

  if(RBAGisExternalNode(v)){
    return(NULL);
  }else{
    vv=v;
    while(!RBAGisExternalNode(vv)){
      vv=vv->left; /*rightを辿れば最大、leftは最小*/
    }
    return(vv->parent);
  }
}

/**
 * # FUNCTION #
 * RBtreeをツリーとして書き出す(デバッグ用)
 */
void RBAGprintTree(char *s,struct RBAGnode *pp)
{
  printf("%s\n",s);
  RBAGptree(pp,0);
}

/**
 * # FUNCTION #
 * 指定のノードを行として書き出す
 */
static void RBAGwriteNode(struct RBAGnode *p, int valCnt, struct mssFPW *fpw, FILE *valFile)
{
  int i;

  /*データテキストの出力*/
  for(i=0; i<FldCnt-1; i++){
    mssWriteStr(*(p->key->fld+i),fpw); mssWriteDlm(fpw);
  }
    mssWriteStr(*(p->key->fld+i),fpw); mssWriteRet(fpw);

  fwrite(p->key->val,sizeof(MssValue),valCnt,valFile);
}

/**
 * # FUNCTION #
 * RBtreeを全て行として書き出す
 */
static void RBAGwriteAllNode(struct RBAGnode *p, int valCnt, struct mssFPW *fpw, FILE *valFile)
{
  
  if(!RBAGisExternalNode(p)){
    RBAGwriteAllNode(p->left,valCnt,fpw,valFile);
    RBAGwriteNode(p,valCnt,fpw,valFile);
    RBAGwriteAllNode(p->right,valCnt,fpw,valFile);
  }
}

/**
 * # FUNCTION #
 * RBtreeのメモリ領域解放
 */
static void RBAGfree(struct RBAGnode *v)
{

  RBAGfreeAllNode(v->left);
  mssFree(v->key);
  mssFree(v);
}

/**
 * # FUNCTION #
 * RBnodeの初期化(ノード作成)
 */
static struct RBAGnode *RBAGinit(struct RBAGnode *rb)
{
  /*赤黒木の初期化*/
  rb               = RBAGmakeNode();
  rb->parent       = rb;
  rb->right        = NULL;
  rb->rank         = 0;
  rb->left         = RBAGmakeNode();
  rb->left->parent = rb;
  return(rb);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 集計タイプ別の演算
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 合計集計
 */
static void aggSum(MssValue *val, MssValue *val2)
{
  int i;
  for(i=0; i<OptFld->flds->cnt; i++){
    *(val+i)=mssVadd(*(val+i), *(val2+i));
  }
}

/**
 * # FUNCTION #
 * 件数集計
 */
static void aggCnt(MssValue *val, MssValue *val2)
{
  int i;

  for(i=0; i<OptFld->flds->cnt; i++){
    if((val2+i)->nul!=1){
      *(val+i)=mssVcntUp(*(val+i));
    }
  }
}

/**
 * # FUNCTION #
 * 平均集計
 */
static void aggAvg(MssValue *val, MssValue *val2)
{
  int i;
  for(i=0; i<OptFld->flds->cnt; i++){
    *(val+i)=mssVadd(*(val+i), *(val2+i));
    if((val2+i)->nul!=1){
      *(val+OptFld->flds->cnt+i)=mssVcntUp(*(val+OptFld->flds->cnt+i));
    }
  }
}

/**
 * # FUNCTION #
 * 平均集計２
 */
static void aggAvg2(MssValue *val, MssValue *val2)
{
  int i;
  for(i=0; i<OptFld->flds->cnt; i++){
    *(val+i)=mssVadd(*(val+i), *(val2+i));
    *(val+OptFld->flds->cnt+i)=mssVadd(*(val+OptFld->flds->cnt+i), *(val2+OptFld->flds->cnt+i));
  }
}

/**
 * # FUNCTION #
 * 最小値集計
 */
static void aggMin(MssValue *val, MssValue *val2)
{
  int i;
  for(i=0; i<OptFld->flds->cnt; i++){
    *(val+i)=mssVmin(*(val+i), *(val2+i));
  }
}

/**
 * # FUNCTION #
 * 最大値集計
 */
static void aggMax(MssValue *val, MssValue *val2)
{
  int i;
  for(i=0; i<OptFld->flds->cnt; i++){
    *(val+i)=mssVmax(*(val+i), *(val2+i));
  }
}

/**
 * # FUNCTION #
 * 分散集計
 */
static void aggVar(MssValue *val, MssValue *x)
{
  int i;
  MssValue *n;
  MssValue *s1;
  MssValue *s2;
  MssValue const1;

  mssVinit(&const1,DBL);
  const1.v.d=1;

  /*x   :仮平均との差(x)*/
  /*n :個数(n)*/
  /*s1:平均(s1)*/
  /*s2:平方和(s2)*/
  n =val;
  s1=val+OptFld->flds->cnt;
  s2=val+OptFld->flds->cnt*2;
  for(i=0; i<OptFld->flds->cnt; i++){
    if((x+i)->nul!=1){
      *(n+i)=mssVcntUp(*(n+i));
    }
    *(x+i)=mssVsub(*(x+i), *(s1+i));
    *(s1+i)=mssVadd(*(s1+i), mssVdiv(*(x+i), *(n+i)));
    *(s2+i)=mssVadd(
      *(s2+i),
      mssVdiv(
        mssVmul(
          mssVsub( *(n+i), const1),
          mssVmul(*(x+i),*(x+i))
        ),
        *(n+i)
      )
    );
  }
}

/**
 * # FUNCTION #
 * 併合時には集計方法が変わるものがある
 *ここに具体例を説明すること
 */
static enum MssAggType chgAggType(enum MssAggType aggType)
{
  switch(aggType){
  case aggSUM:
    return(aggSUM);
    break;
  case aggCNT:
    return(aggSUM);
    break;
  case aggMIN:
    return(aggMIN);
    break;
  case aggMAX:
    return(aggMAX);
    break;
  case aggAVG:
    return(aggAVG2);
    break;
  case aggSTD:
  case aggSTDP:     /*ありえないが警告を避けるため*/
    return(aggSTD);
    break;
  case aggVAR:
  case aggVARP:     /*ありえないが警告を避けるため*/
    return(aggVAR);
    break;
  case aggAVG2:     /*コンパイル警告を防ぐために*/
    return(aggAVG2);
    break;
  }
  return(aggType);
}


/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 集計関連関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 一時ファイル名の取得
 */
static char *getFname(char *prefix, int number)
{
  /*fnameはグローバル変数*/
  sprintf(fname,"%s%d",prefix,number);
  return(fname);
}

/**
 * # FUNCTION #
 * mssFldRec構造体に一行読み込む(番兵処理)
 */
static void readRBAGkey( struct mssFPR *fp, struct mssFldRec *fr)
{
  int i;

  if(EOF==mssReadFldRec(fp,fr)){
    for(i=0; i<fr->fldCnt; i++)
      *(fr->pnt+i)=readEnd; /*EOFの時は番兵を入れる*/
  }
}

/**
 * # FUNCTION #
 * 各バケットの最初の行をRBキューに読み込み
 * mergeRB,preSortから呼び出される
 * iFromからiToまでのファイルの先頭行をRBキューに入れる
 * ad   : iFile,rbをセットする
 * iFrom: 開始ファイル番号
 * iTo  : 終了ファイル番号
 */
static void setFirstLineRBAG(struct mssAggDat *ad, int iFrom, int iTo)
{
  int bkt;                /*現在処理中のバケット番号(0から始まる)*/
  int i;

  /*ad->bktCnt=iTo-iFrom+1; バケット数の設定*/

  /*プライオリティキューの初期化*/
  ad->rb=RBAGinit(ad->rb);
  bkt=0;
  for(i=iFrom; i<=iTo; i++){
    ad->fr[bkt]=mssInitFldRec(FldCnt);

    ad->iFile[bkt]=mssOpenFPR(getFname(ad->prefixTxt,i),4);
    ad->vFile[bkt]=fopen(getFname(ad->prefixVal,i),"rb");

    /*一行(先頭行)読み込み*/
    readRBAGkey(ad->iFile[bkt],ad->fr[bkt]);
    fread(ad->val, sizeof(MssValue), ad->valCnt, ad->vFile[bkt]);

    /*赤黒木に挿入(物理的にコピー)*/
    RBAGinsert(ad->rb->left,ad->fr[bkt],bkt, ad->val);
    bkt++;
  }
}

/**
 * # FUNCTION #
 *RBキューを用いたPWayマージ
 * 既に並べ変わっているバケット(iCnt個)を併合していく
 * 最後の一つになるまで併合はしない
 * PWayA個より少なくなったときに併合を止める
 */
static void mergeRBAG(struct mssAggDat *ad)
{
  int    bkt[PWayA];       /*Pway併合時にRBtreeからpopしたデータのバケットNo*/
  int    bktCnt;           /*そのバケット数*/
  struct RBAGnode *node;   /*popにて取りだしたノードを格納*/

  struct mssFPW *oFile;       /*出力ワークファイルポインタ*/
  FILE       *vFile;       /*出力ワークファイルポインタ*/

  int    iStart;           /*入力ワークファイルの開始番号(ファイル名の一部)*/
  int    iEnd;             /*入力ワークファイルの終了番号(ファイル名の一部)*/
  int    oStart;           /*出力ワークファイルの開始番号(ファイル名の一部)*/
  int    oEnd;             /*出力ワークファイルの終了番号(ファイル名の一部)*/
  int    iFrom,iTo;        /*併合する時のファイル番号*/
  int    k;
  int    i;

  /*次のループでin,outをswapするので、ここでは逆に定義*/
  iStart=ad->iEnd+1;
  iEnd  =ad->iEnd+1;
  oStart=ad->iStart;
  oEnd  =ad->iEnd;

  /*ファイルを併合するループ iCntがPWayA個より大きい場合まわり続ける*/
  while(1){
    mssSwapInt(&iStart,&oStart);
    mssSwapInt(&iEnd  ,&oEnd  );
    oEnd=oStart;

    /*入力ファイル数がPWayA以下ならば終了*/
    if(iEnd-iStart+1 <= PWayA){
      ad->iStart = iStart;
      ad->iEnd   = iEnd;
      break;
    }

    /*Pway個のiFileを一つのoFileに書き出す"を繰り返すループ*/
    k=0;
    while(1){
      /*各バケットの最初行をキューに入れる*/
      iFrom= k   *PWayA+iStart;    /*PWayA=3で一時ファイルが0,1,2,3,4,5,6の時*/
      iTo  =(k+1)*PWayA+iStart-1;  /* iFrom: 0 3 6*/
      if(iTo>iEnd) iTo=iEnd;       /* iTo  : 2 5 6*/
      setFirstLineRBAG(ad, iFrom,iTo);

      /*出力ファイルオープン*/
      oFile=mssOpenFPW(getFname(ad->prefixTxt,oEnd),0,0);
      vFile=fopen(getFname(ad->prefixVal,oEnd),"wb");

      /*各バケットから一行づつ読み込み書き出すループ*/
      while(1) {
        node=RBAGpop(ad->rb->left);          /*キューから一行取り出し*/
        if(strcmp(node->key->str,readEnd)==0)/*readEnd(番兵)を取り出したら終了*/
          break;
        RBAGwriteNode(node,ad->valCnt,oFile,vFile);  /*一行書き出し*/
        bktCnt=0;                   /*popで取り出したkey値のバケット数*/
        for(i=0; i<PWayA; i++){
          if(*(node->key->bkt+i) == 1)
          bkt[bktCnt++]=i;          /*バケット番号をセット*/
        } 
        RBAGdeleteNode(node);              /*取り出したノードの削除*/

        /*popしたバケットから新に読みこんでRBtreeに挿入する*/
        for(i=0; i<bktCnt; i++){
          readRBAGkey(ad->iFile[bkt[i]],ad->fr[bkt[i]]);
          fread(ad->val,sizeof(MssValue),ad->valCnt,ad->vFile[bkt[i]]);
          RBAGinsert(ad->rb->left,ad->fr[bkt[i]],bkt[i],ad->val);
        }
      }
      RBAGfree(ad->rb);                    /*キューの開放*/
      for(i=0; i<=iTo-iFrom; i++){         /*入力ファイルのクローズ*/
        mssCloseFPR(ad->iFile[i]);
        fclose(ad->vFile[i]);
        mssFreeFldRec(ad->fr[i]);
      }
      mssCloseFPW(oFile);                  /*出力ファイルのクローズ*/
      fclose(vFile);                       /*出力ファイルのクローズ*/
      if(iTo==iEnd)break;                  /*最後のバケットまでいけば終了*/
      oEnd++;                              /*出力ファイル番号カウントアップ*/
      k++;                                 /*併合回数カウントアップ*/
    }
    for(i=iStart; i<=iEnd; i++){
      unlink(getFname(ad->prefixTxt,i));      /*入力ワークファイルの削除*/
      unlink(getFname(ad->prefixVal,i));      /*入力ワークファイルの削除*/
    }
  }
}

/**
 * # FUNCTION #
 *集計しながらsortする関数(uniq,sum,avg,max,min,cnt,stdev,var...)
 *集計しながら赤黒木を作成
 * メモリ消費がMAXを越えたら一時ファイルに書き出し
 */
static void sortAG(struct mssAggDat *ad, struct mssFPR *fpr)
{
  struct mssFldRec *fr;    /*入力ファイルをキー順に並べ換えるためのバッファ*/
  int    pid;              /*プロセスID(ファイル名の一部)*/
  int    oNum;             /*ワークファイル番号のカウントアップ変数*/
  int    memCnt;           /*メモリ消費量*/

  struct RBAGnode *rb=NULL; /*赤黒木*/
  struct mssFPW *outFile;      /*出力一時ファイル*/
  FILE       *valFile;      /*出力一時ファイル*/

  fr=mssInitFldRec(FldCnt);/*FldRecの初期化*/
  rb=RBAGinit(rb);          /*RBAGtreeの初期化*/

  /*ファイル名のプレフィックスの設定*/
  pid=getpid();
  if(strlen(ad->tmpDir) > MssFileNameMaxLen - 50 ) {
    mssShowErrMsg("length of path name must be less than %d",MssFileNameMaxLen-50);
    exit(mssErrorNoDefault);
  }
  sprintf(ad->prefixTxt,"%s/xt##%d-PreAggTxt-",ad->tmpDir,pid);
  sprintf(ad->prefixVal,"%s/xt##%d-PreAggVal-",ad->tmpDir,pid);

  memCnt=0; /*消費メモリカウンタのクリア*/
  oNum=0;   /*出力ファイル番号*/
  while(1){
    /*データの読み込み*/
    mssReadFldRec(fpr,fr);
    (*ad->inCnt)++;

    /*赤黒木のサイズがMaxを越えたら もしくは 最終行を検出したら*/
    /*ワークファイルへ書き出す*/
    if( memCnt >= MaxMemA || fr->eof==1){
      outFile=mssOpenFPW(getFname(ad->prefixTxt,oNum),0,0);/*出力ファイルオープン*/
      valFile=fopen(getFname(ad->prefixVal,oNum),"wb"); /*出力ファイルオープン*/
      RBAGwriteAllNode(rb->left,ad->valCnt, outFile,valFile);/*RBtree全て出力*/
      mssCloseFPW(outFile);                          /*出力ファイルのクローズ*/
      fclose(valFile);                            /*出力ファイルのクローズ*/
      oNum++;                                     /*出力ファイル数countUp*/
      RBAGfree(rb);   /*RBAGtreeの領域解放*/
      if(fr->eof==1){ /*最終行で終了*/
        (*ad->inCnt)--;
        break;
      }
      rb=RBAGinit(rb);    /*RBAGtreeの初期化*/
      memCnt=0;           /*消費メモリカウンタのクリア*/
    }

    /*赤黒木に挿入して(物理的にコピー)、メモリ消費をカウントアップ*/
    mssSetAggVal(ad->val, fr->pnt);
    memCnt += RBAGinsert(rb->left,fr,1,ad->val);
  }
  mssFreeFldRec(fr);/*FldRecの初期化*/

  ad->iStart = 0;
  ad->iEnd   = oNum-1;
}

/**
 * # FUNCTION #
 * 文字列としての集計方法("sum","avg","min"..)をenumAGtypeの要素に変換
 */
static enum MssAggType setAggType(char *str)
{
       if( 0==strcmp("sum",str) ){
    return(aggSUM);
  }else if( 0==strcmp("avg",str) ){
    return(aggAVG);
  }else if( 0==strcmp("cnt",str) ){
    return(aggCNT);
  }else if( 0==strcmp("max",str) ){
    return(aggMAX);
  }else if( 0==strcmp("min",str) ){
    return(aggMIN);
  }else if( 0==strcmp("std",str) ){
    return(aggSTD);
  }else if( 0==strcmp("stdp",str) ){
    return(aggSTDP);
  }else if( 0==strcmp("var",str) ){
    return(aggVAR);
  }else if( 0==strcmp("varp",str) ){
    return(aggVARP);
  }

  mssShowErrMsg("Invalid aggregation method:'%s'",str);
  exit(mssErrorNoDefault);

  return(-1); /*コンパイラwarningメッセージを避けるため*/
}

/**
 * # FUNCTION #
 * 項目数(fc)と集計タイプに応じて集計値領域で確保するMssValueの数を返す。
 */
static int getAggValCnt(enum MssAggType aggType, int fc)
{
  switch(aggType){
  case aggSUM:
  case aggCNT:
  case aggMIN:
  case aggMAX:
    return(fc);
    break;
  case aggAVG:
  case aggAVG2:
    return(fc*2);
    break;
  case aggSTD:
  case aggSTDP:
  case aggVAR:
  case aggVARP:
    return(fc*3);
    break;
  }
  return(fc);
}

/**
 * # FUNCTION #
 * 項目数(fc)と集計タイプに応じて集計値領域を確保する。
 */
static MssValue *getAggValMem(enum MssAggType aggType, int fc)
{
  MssValue *val;
  switch(aggType){
  case aggSUM:
  case aggCNT:
  case aggMIN:
  case aggMAX:
    val = mssMalloc(sizeof(MssValue)*fc  ,"RBAGAGtree6");
    break;
  case aggAVG:
  case aggAVG2:
    val = mssMalloc(sizeof(MssValue)*fc*2,"RBAGAGtree6");
    break;
  case aggSTD:
  case aggSTDP:
  case aggVAR:
  case aggVARP:
    val = mssMalloc(sizeof(MssValue)*fc*3,"RBAGAGtree6");
    break;
  default:
    val = NULL;
  }
  mssClearAggVal(val, aggType, fc);
  return(val);
}

/**
 * # FUNCTION #
 * 項目数(fc)と集計タイプに応じて集計値領域(val)をクリアする。
 */
void mssClearAggVal(MssValue *val, enum MssAggType aggType, int fc)
{
  int i;
  switch(aggType){
  case aggSUM:
  case aggCNT:
    for(i=0; i<fc;   i++) mssVinit(val+i,DBL);
    break;
  case aggMIN:
    for(i=0; i<fc;   i++) {mssVinit(val+i,DBL); mssVclearMax( val+i );}
    break;
  case aggMAX:
    for(i=0; i<fc;   i++) {mssVinit(val+i,DBL); mssVclearMin( val+i );}
    break;
  case aggAVG:
  case aggAVG2:
    for(i=0; i<fc*2; i++) mssVinit(val+i,DBL);
    break;
  case aggSTD:
  case aggSTDP:
  case aggVAR:
  case aggVARP:
    for(i=0; i<fc*3; i++) mssVinit(val+i,DBL);
    break;
  }
}

/**
 * # FUNCTION #
 * 集計タイプに応じて、集計値(val)を新しい値(val2)によって更新する。
 */
void mssCalAggVal(MssValue *val, enum MssAggType aggType, MssValue *val2)
{
  /*val2:f=の値 val:計算結果*/
  switch(aggType){
  case aggSUM:
    aggSum( val ,val2);
    break;
  case aggCNT:
    aggCnt( val ,val2);
    break;
  case aggAVG:
    aggAvg( val ,val2);
    break;
  case aggAVG2:
    aggAvg2( val ,val2);
    break;
  case aggMIN:
    aggMin( val ,val2);
    break;
  case aggMAX:
    aggMax( val ,val2);
    break;
  case aggSTD:
  case aggSTDP:
  case aggVAR:
  case aggVARP:
    aggVar( val ,val2);
    break;
  }
}

/**
 * # FUNCTION #
 * 集計タイプに応じて、最終結果を計算し、一行出力する。
 */
void mssWriteAggFld( char **fld, MssValue *val, enum MssAggType aggType, struct mssFPW *fpw)
{
  MssValue const1;
  int i;

  mssVinit(&const1,DBL);
  const1.v.d=1;

  switch(aggType){
  case aggSUM:
    break;
  case aggCNT:
    break;
  case aggMIN:
    break;
  case aggMAX:
    break;
  case aggAVG: /*意味無いがコンパイルエラーを避けるため*/
  case aggAVG2:
    for(i=0; i<OptFld->flds->cnt; i++){
      *(val+i)=mssVdiv(*(val+i), *(val+OptFld->flds->cnt+i) );
    }
    break;
  case aggVAR:
    for(i=0; i<OptFld->flds->cnt; i++){
      *(val+i)=mssVdiv(*(val+OptFld->flds->cnt*2+i), mssVsub(*(val+i),const1) );
    }
    break;
  case aggVARP:
    for(i=0; i<OptFld->flds->cnt; i++){
      *(val+i)=mssVdiv(*(val+OptFld->flds->cnt*2+i), *(val+i) );
    }
    break;
  case aggSTD:
    for(i=0; i<OptFld->flds->cnt; i++){
      *(val+i)=mssVsqrt(mssVdiv(*(val+OptFld->flds->cnt*2+i), mssVsub(*(val+i),const1)));
    }
    break;
  case aggSTDP:
    for(i=0; i<OptFld->flds->cnt; i++){
      *(val+i)=mssVsqrt(mssVdiv(*(val+OptFld->flds->cnt*2+i), *(val+i)));
    }
    break;
  }

  for(i=0; i<FldCnt-1; i++){
    if(*(OptFld->fldNo2optNo+i)==-1){
      mssWriteStr(*(fld+i),fpw);
    }else{
      mssVwriteDbl(*(val+*(OptFld->fldNo2optNo+i)), fpw);
    }
    mssWriteDlm(fpw);
  }
    if(*(OptFld->fldNo2optNo+i)==-1){
      mssWriteStr(*(fld+i),fpw);
    }else{
      mssVwriteDbl(*(val+*(OptFld->fldNo2optNo+i)), fpw);
    }
    mssWriteRet(fpw);
}

/**
 * # FUNCTION #
 * 集計タイプに応じて、最終結果を計算し、一行出力する。
 *文字列としての項目をvalに変換する関数
 *valはrbtreeに入れるための一時的なもの
 */
void mssSetAggVal(MssValue *val, char **str)
{
  int i;
  for(i=0; i<OptFld->flds->cnt; i++){
    if( MssIsNull(*(str+MssFlds2num(OptFld->flds,i))) ){
      MssVnull(*(val+i));
    }else{
      (val+i)->nul=0;
      (val+i)->v.d=atof(*(str+MssFlds2num(OptFld->flds,i)));
    }
  }
}

/**
 * # FUNCTION #
 * RBキューを用いた項目切り分け型行読み込み
 * 既に並べ変わっているバケット(ad->iStartからad->iEnd)を集計しながら
 * 併合していく
 * ファイルの個数は既にPWayA以下になっていることが前提
 */
int mssReadWriteAgg(struct mssAggDat *ad, struct mssFPW *oFile)
{
  struct RBAGnode *node;     /*popにて取りだしたノードを格納*/
  int              bkt[PWayA];/*Pway併合時にRBtreeからpopしたデータのバケットNo*/
  int              bktCnt;   /*そのバケット数*/
  int i;

  node=RBAGpop(ad->rb->left);           /*キューから一行取り出し*/
  if(strcmp(node->key->str,readEnd)==0) /*readEnd(番兵)を取り出したら終了*/
    return(EOF);

  /*一行書き出し*/
  mssWriteAggFld(node->key->fld,node->key->val,AGGType,oFile);

  bktCnt=0;                      /*popで取り出したkey値のバケット数*/
  for(i=0; i<PWayA; i++){
    if(*(node->key->bkt+i) == 1)
      bkt[bktCnt++]=i;           /*バケットフラグをセット*/
  } 
  RBAGdeleteNode(node);          /*取り出したノードの削除*/

  /*popしたバケットから新に読みこんでRBtreeに挿入する*/
  for(i=0; i<bktCnt; i++){
    readRBAGkey(ad->iFile[bkt[i]],ad->fr[bkt[i]]);
    fread(ad->val,sizeof(MssValue),ad->valCnt,ad->vFile[bkt[i]]);
    RBAGinsert(ad->rb->left,ad->fr[bkt[i]],bkt[i],ad->val);
  }
  return(1); /*とりあえず*/
}

/**
 * # FUNCTION #
 *赤黒木を作りながら集計＋集計しながらPWayマージソート
 */
void mssPreAgg(struct mssAggDat *ad, struct mssFPR *iFile)
{
  /*シグナルハンドラの設定*/
  mssGV.usedTempFileFlg=1;
  mssSetSignalHandler();

  sortAG(ad,iFile);
  AGGType=chgAggType(AGGType);
  mergeRBAG(ad);
  setFirstLineRBAG(ad, ad->iStart,ad->iEnd);
}

/**
 * # FUNCTION #
 * mssAggDat構造体を初期化し、そのポインタを返す。
 */
struct mssAggDat *mssInitAggDat(int fldCnt, MssOptKEY *optKey, MssOptFLD *optFld, char *aggType, char *tmpDir, int *inCnt)
{
  struct mssAggDat *aggDat;
  aggDat=mssMalloc(sizeof(struct mssAggDat),"initAggDat");

  aggDat->fldCnt  = fldCnt;
  aggDat->optKey  = optKey;
  aggDat->optFld  = optFld;
  aggDat->aggType = setAggType(aggType);
  aggDat->tmpDir  = tmpDir;
  aggDat->inCnt   = inCnt;
  aggDat->val     = getAggValMem(aggDat->aggType, aggDat->optFld->flds->cnt);
  aggDat->valCnt  = getAggValCnt(aggDat->aggType, aggDat->optFld->flds->cnt);
  aggDat->aggVal  = getAggValMem(aggDat->aggType, aggDat->optFld->flds->cnt);

  /*グローバル変数*/
  FldCnt  = aggDat->fldCnt;
  OptKey  = aggDat->optKey;
  OptFld  = aggDat->optFld;
  AGGType = aggDat->aggType;
  
  return(aggDat);
}

/**
 * # FUNCTION #
 * mssAggDat構造体の開放。
 */
void mssFreeAggDat(struct mssAggDat *ad)
{
  int i;

  if(ad->procType==1){
    /*バッファ管理のために、入力ファイルはここで初めてクローズする*/
    for(i=0; i<=ad->iEnd-ad->iStart; i++){
      mssCloseFPR(ad->iFile[i]);
      mssFreeFldRec(ad->fr[i]);
    }

    /*ファイルの削除*/
    for(i=ad->iStart; i<=ad->iEnd; i++){
      unlink(getFname(ad->prefixTxt,i));
      unlink(getFname(ad->prefixVal,i));
    }

    /*領域開放*/
    RBAGfree(ad->rb);
  }
  mssFree(ad->val);
  mssFree(ad->aggVal);
  mssFree(ad);
}
