/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられる基本的な関数
 * 20040129 mktime関数のマイナーbug fix
 *          tmpパスと有効桁数を環境変数で変更できるように
 * 20040420 AllocInf関連関数見直し(reallocの対応関係改善)
 * 20040428 mssPRand()ポアソン分布乱数追加
 * 20040609 mssSort() ソート関数追加
 * 20041004 mssGV.argvをargv[]からmssStrdupするように変更
 * ============================================================================
 */

#include <musashi/mssBase.h>
#include <musashi/mssConfig.h>
#include <musashi/mssOutput.h>
#include <musashi/random.h>
#include <musashi/qs5.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <dirent.h>
#include <signal.h>
#include <float.h>
#include <limits.h>
#include <math.h>

/* ############ グローバル変数 ##############*/
struct mssGlobalVariables mssGV;

extern double (*_func[_M])(void);
extern void (*initRND[_M])(long s);

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * シグナル処理、メッセージ処理
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 異常終了時の一時ファイル削除
 */
static void delTmpFile()
{
  DIR *dp;
  struct dirent *dir;
  int len;
  char prefix[MssFileNameMaxLen];
  char fname[MssFileNameMaxLen];

  sprintf(prefix,"xt##%d-",(int)getpid());

  len=strlen(prefix);

  if((dp = opendir(mssGV.tmpPath)) == NULL) {
    mssShowErrMsg("can't open directory %s\n", mssGV.tmpPath);
    mssEnd(mssErrorNoDefault);
  }
  while ((dir = readdir(dp)) != NULL) {
    if (dir->d_ino == 0) continue; /*removedファイルのスキップ*/
    if(0==strncmp(dir->d_name,prefix,len)){
      /*fprintf(stderr,"%s\n", dir->d_name);*/
      strcpy(fname,mssGV.tmpPath);
      strcat(fname,"/");
      strcat(fname,dir->d_name);
      unlink(fname);
    }
  }
  closedir(dp);
}
 
/**
 * # FUNCTION #
 * シグナルによるエラー終了
 */
static void sigErrEnd(int sig)
{
  if(sig==2){
    mssShowErrMsg("interrupt by user (CTR^C)");
  }else{
    mssShowErrMsg("end by signal (%d)",sig);
  }
  mssEnd(mssErrorNoDefault);
}

/**
 * # FUNCTION #
 * シグナルによる正常終了
 */
static void sigEnd(int sig)
{
  mssShowEndMsg();
  mssEnd(mssExitSuccess);
}

/**
 * # FUNCTION #
 *シグナル補足の登録(一時ファイルを利用する時)
 */
void mssSetSignalHandler()
{
  signal( SIGSEGV, (void (*)())sigErrEnd );
  signal( SIGPIPE, (void (*)())sigEnd );
  signal( SIGKILL, (void (*)())sigErrEnd );
  signal( SIGINT , (void (*)())sigErrEnd );
}

/**
 * # FUNCTION #
 * メッセージを出すかどうかを判定
 * コマンドラインで -Q を指定したらQuietモード
 * 環境変数 mssQuiet の値が1ならQuietモード
 * その他の場合は NoQuietモード
 * extern変数QuietはsetOptionでセットされる
 * 返値: 1:Quietモード, 0:NoQuietモード
 */
static int isQuiet()
{
  char *envVal;

  if(mssGV.quiet) return 1;
  envVal=getenv("mssQuiet");
  if(envVal==NULL) return 0;
  if(*envVal=='1') return 1;
  return 0;
}

/**
 * # FUNCTION #
 * グローバル変数(mssGV構造体)のセットおよびシグナル補足の登録
 * 通常、コマンドの最初に呼び出される関数
 */
void mssInit(int argc, char **argv, struct mssComHelp *comHelp)
{

  char *envStr;
  int sigDigits;
  int i;
  char buf[30];

  /*グローバル変数の初期化および設定*/
  mssGV.comHelp=comHelp;
  mssGV.inCnt=0;
  mssGV.outCnt=0;
  mssGV.keyNo=0;
  mssGV.keyCnt=0;
  mssGV.keyLine=0;
  mssGV.inFldCnt=0;
  mssGV.quiet=0;
  mssGV.argc=argc;
  mssGV.argv=mssMalloc(sizeof(char *)*argc,"mssInit");
  for(i=0; i<argc; i++){
    *(mssGV.argv+i)=mssStrdup(*(argv+i));
  }
  mssGV.txtFlg=0;
  mssGV.allocDeb=0;
/*  mssGV.allocInf=NULL;*/
  mssGV.allocFreeCnt=0;
  mssGV.allocCnt=0;
  mssGV.writeHeaderFlg=0;
  mssGV.writeFooterFlg=0;
  mssGV.usedTempFileFlg=0;

  if( (envStr=getenv("MssTmpPath")) != NULL ){
    mssGV.tmpPath=mssMalloc(sizeof(char)*(strlen(envStr)+1),"mssInit");
    strcpy(mssGV.tmpPath, envStr);
  }else{
    mssGV.tmpPath=MssTempDir;
  }

  if( (envStr=getenv("MssSigDigits")) != NULL ){
    sigDigits=atoi(envStr);
  }else{
    sigDigits=MssSigDigits;
  }
  /* sprintf(mssGV.sigDigitsForm,"%%.%dg",sigDigits); */
  sprintf(buf,"%%.%dg",sigDigits);
  mssGV.sigDigitsForm=mssStrdup(buf);

  /*シグナル処理の登録*/
  mssSetSignalHandler();
}

/**
 * # FUNCTION #
 * 終了時に呼び出される関数
 */
void mssEnd(int status)
{
  int i;

  if(mssGV.usedTempFileFlg) delTmpFile();
  signal( SIGSEGV, SIG_IGN );
  signal( SIGPIPE, SIG_IGN );
  signal( SIGKILL, SIG_IGN );
  signal( SIGINT , SIG_IGN );

  for(i=0; i<mssGV.argc; i++){
    mssFree(*(mssGV.argv+i));
  }
  mssFree(mssGV.argv);
  mssFree(mssGV.sigDigitsForm); /* 追加 2004/10/31 */
  exit(status);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * メモリ領域確保、開放関連
 * ----------------------------------------------------------------------------
 */

/**
 * # DOCUMENT #
 * メモリ領域確保、開放に関するデバッグツールの使い方 
 * 1: ソースで "extern struct MssGlobalVariable mssGV;"と宣言する。
 * 2: メモリ領域確保と開放の対応関係を調べたいブロックの前で
 *    "mssGV.allocDeb=1;"とする。
 *    それ以降のmssMallocやmssFreeの呼び出し内容がmssGV.allocInf構造体に
 *    記録されることになる。
 * 3: ブロックの後でmssGV.allocDeb=0に戻し、記録をそこまでとする。
 * 4:その後にmssShowAllocInf()を呼び出せば、どのmssMallocがどのmssFreeで開放され
 * たかの対応関係リストが出力される。出力される結果表の２列目が-1であれば
 * 対応関係が見付からなかったこと(開放しわすれ)を意味する。
 */

/**
 * # FUNCTION #
 * メモリ領域確保、開放の情報をグローバル変数に追加する
 */
static void addAllocInf(int func, void *add, int size,char *msg)
{
  int i;

  if(func==0) mssGV.allocFreeCnt--;
  else        mssGV.allocFreeCnt++;

  mssGV.allocInf[mssGV.allocCnt].func=func;
  mssGV.allocInf[mssGV.allocCnt].add =add;
  mssGV.allocInf[mssGV.allocCnt].cnt =mssGV.allocFreeCnt;
  mssGV.allocInf[mssGV.allocCnt].size =size;
  mssGV.allocInf[mssGV.allocCnt].refNo=-1;
  strcpy(mssGV.allocInf[mssGV.allocCnt].msg,msg);

  if(func==0){
    for(i=mssGV.allocCnt; i>=0; i--){
      if( mssGV.allocInf[i].func!=0 && mssGV.allocInf[i].refNo==-1 ){
        if( mssGV.allocInf[i].add==mssGV.allocInf[mssGV.allocCnt].add ){
          mssGV.allocInf[i].refNo=mssGV.allocCnt;
          mssGV.allocInf[mssGV.allocCnt].refNo=i;
          break;
        }
      }
    }
  }

  mssGV.allocCnt++;
}

/**
 * # FUNCTION #
 * mssReallocでメモリ領域が確保された時、新旧のアドレスを見て更新する
 */
static void updAllocInf(int func, void *newAdd, void *oldAdd, int size, char *msg){
  int i;
  char buf[256];

  if(newAdd==oldAdd){
/*
    sprintf(buf,"%s--same(%p,%p)",msg,newAdd,oldAdd);
    addAllocInf(func, 0, 0, buf);
*/
    return;
  }

  if(oldAdd==NULL){
    sprintf(buf,"%s--new",msg);
    addAllocInf(func, newAdd, size, buf);
    return;
  }

  for(i=mssGV.allocCnt-1; i>0; i--){
    if( mssGV.allocInf[i].add==oldAdd && mssGV.allocInf[i].func!=0
                                      && mssGV.allocInf[i].refNo==-1 ){
      if(mssGV.allocInf[i].func!=func){
        sprintf(buf,"%s--mismatched (%p,%p:%s)",mssGV.allocInf[i].msg,newAdd,oldAdd,msg);
        strcpy(mssGV.allocInf[i].msg,buf);
      }
      mssGV.allocInf[i].func=func;
      mssGV.allocInf[i].add =newAdd;
      /*addAllocInf(func, 0, 0, msg);*/
      return;
    }
  }
  sprintf(buf,"%s--unmatched (%p,%p)",msg,newAdd,oldAdd);
  addAllocInf(func, newAdd, size, buf);
}

/**
 * # FUNCTION #
 * メモリ領域確保、開放の情報をグローバル変数に追加する
 */
void mssAddAllocLabel(char *msg)
{
  if(mssGV.allocDeb){
    mssGV.allocInf[mssGV.allocCnt].func=6;
    mssGV.allocInf[mssGV.allocCnt].add =NULL;
    mssGV.allocInf[mssGV.allocCnt].cnt =0;
    mssGV.allocInf[mssGV.allocCnt].size=0;
    mssGV.allocInf[mssGV.allocCnt].refNo=0;
    strcpy(mssGV.allocInf[mssGV.allocCnt].msg,msg);
    mssGV.allocCnt++;
  }
}

/**
 * # FUNCTION #
 * メモリ領域確保、開放の一覧表を表示する
 */
void mssShowAllocInf(void)
{
  int i;

/*
  for(i=mssGV.allocCnt-1; i>0; i--){
    if( mssGV.allocInf[i].func==0 && mssGV.allocInf[i].refNo==-1 ){
      for(j=i-1; j>=0; j--){
        if( mssGV.allocInf[i].add==mssGV.allocInf[j].add &&
            mssGV.allocInf[j].refNo==-1 &&
            mssGV.allocInf[j].func!=0   ){
          mssGV.allocInf[i].refNo=j;
          mssGV.allocInf[j].refNo=i;
          break;
        }
      }
    }
  }
*/

  fprintf(stderr,"========================================================\n");
  fprintf(stderr,"動的メモリ領域の確保＆解放対応表\n");
  fprintf(stderr,"--------------------------------------------------------\n");
  fprintf(stderr,"項目1: シーケンスNo. (確保or解放された順番)\n");
  fprintf(stderr,"項目2: メモリ関数種別\n");
  fprintf(stderr,"      0:mssFree\n");
  fprintf(stderr,"      1:mssMalloc\n");
  fprintf(stderr,"      2:mssCalloc\n");
  fprintf(stderr,"      3:mssRealloc\n");
  fprintf(stderr,"      4:mssreallocLim\n");
  fprintf(stderr,"      5:mssStrdup\n");
  fprintf(stderr,"項目3: 確保(解放)された先頭アドレス\n");
  fprintf(stderr,"項目4: 確保されたサイズ\n");
  fprintf(stderr,"項目5: 確保-解放の対応シーケンスNo.\n");
  fprintf(stderr,"項目6: その時点でのメモリ確保数(最終行がは0のはず)\n");
  fprintf(stderr,"       (最終行が0でなければ解放忘れがある)\n");
  fprintf(stderr,"項目7: mssFree,mssStrdup以外について、呼び出しメッセージ\n");
  fprintf(stderr,"========================================================\n");
  for(i=0; i<mssGV.allocCnt; i++){
    if( mssGV.allocInf[i].func==6){
      fprintf(stderr,"%3d : Label : %s\n", i, mssGV.allocInf[i].msg );
    }else{
      fprintf(stderr,"%3d : %d %p %d %d %d %s\n",
      i,
      mssGV.allocInf[i].func,
      mssGV.allocInf[i].add ,
      mssGV.allocInf[i].size ,
      mssGV.allocInf[i].refNo,
      mssGV.allocInf[i].cnt ,
      mssGV.allocInf[i].msg );
    }
  }
}

/**
 * # FUNCTION #
 * 領域開放
 *   アドレスが NULLなら何もしない
 */
void mssFree(void *add)
{
  if(add!=NULL){
    if(mssGV.allocDeb){
      addAllocInf(0,add,0,"");
/*      mssGV.allocFreeCnt--;*/
    }
    free(add);
    /*add=NULL;*/
  }
}

/**
 * # FUNCTION #
 * callocによる領域確保
 */
void *mssCalloc(int size, char *errMsg)
{
  void *p;
  p=calloc(size,1);
  if(p==NULL){
    mssShowErrMsg("memory allocation error in [%s]",errMsg);
    mssEnd(mssErrorNoDefault);
  }
  if(mssGV.allocDeb) {
/*    mssGV.allocFreeCnt++;*/
    addAllocInf(1,p,size,errMsg);
  }
  return(p);
}

/**
 * # FUNCTION #
 * mallocによる領域確保
 */
void *mssMalloc(int size, char *errMsg)
{
  void *p;
  p=malloc(size);
  if(p==NULL){
    mssShowErrMsg("memory allocation error in [%s]",errMsg);
    mssEnd(mssErrorNoDefault);
  }
  if(mssGV.allocDeb){
/*    mssGV.allocFreeCnt++;*/
    addAllocInf(2,p,size,errMsg);
  }
  return(p);
}

/**
 * # FUNCTION #
 * reallocによる領域確保
 */
void *mssRealloc(void *ptr, int size, char *errMsg)
{
  void *p;
  p=realloc(ptr,size);
  if(p==NULL){
    mssShowErrMsg("memory allocation error in [%s]",errMsg);
    mssEnd(mssErrorNoDefault);
  }
  if(mssGV.allocDeb){
/*    mssGV.allocFreeCnt++;*/
    updAllocInf(3,p,ptr,size,errMsg);
  }
  return(p);
}

/**
 * # FUNCTION #
 * 確保するサイズ指定ができるreallocによる領域確保
 */
void *mssReallocLim(void *ptr, int size,int lim, char *errMsg)
{
  void *p;

  if(size>lim){
    mssShowErrMsg("exceed memory limitation of %d byte in [%s]",lim,errMsg);
    mssEnd(mssErrorNoDefault);
  }
  p=realloc(ptr,size);
  if(p==NULL){
    mssShowErrMsg("memory allocation error in [%s]",errMsg);
    mssEnd(mssErrorNoDefault);
  }
  if(mssGV.allocDeb && ptr==NULL) {
/*    mssGV.allocFreeCnt++;*/
    updAllocInf(4,p,ptr,size,errMsg);
  }
  return(p);
}

/**
 * # FUNCTION #
 * strdupによる文字列コピー
 */
char *mssStrdup(char *s)
{
  char *S;
  if(s==NULL) return(NULL);
  S=strdup(s);
  if(mssGV.allocDeb) {
/*    mssGV.allocFreeCnt++;*/
    addAllocInf(5,S,strlen(S),"");
  }
  return(S);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * mssStrings構造体関連の関数群
 * ----------------------------------------------------------------------------
 */

/**
 * # DOCUMENT #
 * ここの関数群の目的は、文字列を併合(Concatenate)し新しい文字列領域を
 * 作成することを容易に実現することにある。
 *
 * 利用例)
 *   1: struct mssStrings *s;
 *   2: s=mssInitStrings();  mssStrings構造体の初期化
 *   3: catStrings(s,"abc"); "abc"という文字列をセット
 *   4: catStrings(s,"xyz"); "xyz"という文字列を追加
 *   5: catStrings(s,"opq"); "opq"という文字列を追加
 *   6: freeStrings(s);      mssStrings構造体領域を開放
 *   上記の5:番目までの処理にて、s->str="abcxyzopq", s->cnt=9、となる。
 *
 * 併合される各文字列に対してアクセスしたい場合は、StrList関連の関数群を使う。
 */

/**
 * # FUNCTION #
 * mssStrings構造体の初期化
 */
struct mssStrings *mssInitStrings()
{
  struct mssStrings *s;
  s=mssMalloc(sizeof(struct mssStrings),"initStrings");
  s->str=NULL;
  s->cnt=0;
  return(s);
}

/**
 * # FUNCTION #
 * mssStrings構造体領域の開放
 */
void mssFreeStrings(struct mssStrings *s)
{
  mssFree(s->str);
  mssFree(s);
}

/**
 * # FUNCTION #
 * mssStrings構造体に文字列を追加する
 */
void mssCatStrings(struct mssStrings *s, char *str)
{
  int cnt=s->cnt;
  s->cnt += strlen(str);
  s->str=mssRealloc(s->str,sizeof(char)* (s->cnt+1),"catStrings");
  if(cnt==0) strcpy(s->str,str);
  else       strcat(s->str,str);
}

/**
 * # FUNCTION #
 * mssStrings構造体に文字列をn文字追加する
 */
void mssCatnStrings(struct mssStrings *s, char *str, int n)
{
  int cnt=s->cnt;
  s->cnt += n;
  s->str=mssRealloc(s->str,sizeof(char)* (s->cnt+1),"catnStrings");
  if(cnt==0) strncpy(s->str,str,n);
  else       strncat(s->str,str,n);
  *(s->str+s->cnt)='\0';
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * mssStrList構造体関連の関数群
 * ----------------------------------------------------------------------------
 */

/**
 * # DOCUMENT #
 * この関数群の目的は、可変長文字列の一次元配列を効率的に作成することにある。
 * 内部的には、文字列を一つのメモリ領域に併合(Concatenate)し
 * 新しい一つの文字列領域を作成し、そして各文字列へのアクセスを
 * 文字位置ポインタ(文字列領域の先頭からのバイト数)によって参照する。
 * 配列の要素は0から始まる。
 *
 * 利用例)
 *   1: struct mssStrList *s;
 *   2: s=mssInitStrList();  mssStrList構造体の初期化
 *   3: mssPutStrList(s,"abc"); "abc"という文字列を要素0にセット
 *   4: mssPutStrList(s,"xyz"); "xyz"という文字列を要素1にセット
 *   5: mssPutStrList(s,"opq"); "opq"という文字列を要素2にセット
 *   6: mssFreeStrList(s);      mssStrList構造体領域を開放
 *   上記の5:番目までの処理にて記録された文字列は以下のようにmssGetStrList関数
 *   にて参照できる。
 *     mssGetStrList(s,0)=="abc"
 *     mssGetStrList(s,1)=="xyz"
 *     mssGetStrList(s,2)=="opq"
 *   要素の数は s->cntにて参照できる。上記の例では3。
 *   セットされた文字列全体の総文字数は s->chrCntで参照できる。上記の例では9。
 */

/**
 * # FUNCTION #
 * mssStrList構造体の初期化
 */
struct mssStrList *mssInitStrList()
{
  struct mssStrList *s;
  s=mssMalloc(sizeof(struct mssStrList),"initStrList");
  s->chrCnt=0;
  s->cnt=0;
  s->str=NULL;
  s->pointer=NULL;
  return(s);
}

/**
 * # FUNCTION #
 * mssStrList構造体領域の開放
 */
void mssFreeStrList(struct mssStrList *s)
{
  if(s==NULL)return;
  mssFree(s->str);
  mssFree(s->pointer);
  mssFree(s);
}

/**
 * # FUNCTION #
 * mssStrList構造体に文字列を追加する
 */
void mssPutStrList(struct mssStrList *s, char *str)
{

  int len;

  /*行の文字列のコピー*/
  len = strlen(str)+1; /*+1は'\0'のため*/
  s->chrCnt += len;
  s->str=mssRealloc(s->str,sizeof(char)* s->chrCnt,"putStrList");

  strncpy(s->str+s->chrCnt-len,str,len-1); /*文字のみをコピー*/
  *(s->str+s->chrCnt-1)='\0';              /*'\0'をケツにつける*/

  /*行の文字数配列の更新(文字列領域の先頭からの文字数をポインタとして利用*/
  s->pointer=mssRealloc(s->pointer,sizeof(int)*(s->cnt+1),"putStrList");
  *(s->pointer+s->cnt)=s->chrCnt-len;

  s->cnt++; /*レコード数をカウントアップ*/
}

/**
 * # FUNCTION #
 * mssStrList構造体のn番目の要素へのポインタを返す。
 * 指定した要素番号が要素数より大きい場合はNULLを返す。
 */
char *mssGetStrList(struct mssStrList *s,int n)
{
  if( n>=s->cnt ) return(NULL);
  return(s->str+*(s->pointer+n));
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 二変数のswap
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * int型変数のswap
 */
void mssSwapInt(int *x, int *y)
{
  int tmp;
  tmp=*x; *x=*y; *y=tmp;
}

/**
 * # FUNCTION #
 * long型変数のswap
 */
void mssSwapLng(long *x, long *y)
{
  long tmp;
  tmp=*x; *x=*y; *y=tmp;
}

/**
 * # FUNCTION #
 * double型変数のswap
 */
void mssSwapDbl(double *x, double *y)
{
  double tmp;
  tmp=*x; *x=*y; *y=tmp;
}


/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 型変換の関数群
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 文字列の日付からtime_t型日付値に変換
 */
time_t mssStrToTime(char *str)
{
  char buf[20];
  time_t caltime;
  struct tm worktm;

  strncpy(buf, str, 4); /*最初の文字から4文字コピーする,添字のから格納*/
  buf[4] = '\0';

  /*0から数えて4つ目に空文字を代入*/
  worktm.tm_year = atoi(buf) - 1900; /*暦時間にするために1900引く*/

  strncpy(buf, str+4, 2);
  buf[2] = '\0';
  worktm.tm_mon = atoi(buf) - 1;    /*暦時間にするために1900引く*/

  strncpy(buf, str+6, 2);
  buf[2] = '\0';
  worktm.tm_mday = atoi(buf);
  worktm.tm_hour = 0;
  worktm.tm_min = 0;
  worktm.tm_sec = 0;
  worktm.tm_isdst = -1;

  if((caltime = mktime(&worktm)) == -1){ /*mktimeにより暦時間に変換*/
    return(-1);
  }
  return(caltime);
}

/**
 * # FUNCTION #
 * int型数値を文字列に変換
 */
char *mssItoA(int num)
{
  char buf[100];
  sprintf(buf,"%d",num);
  return(mssStrdup(buf));
}

/**
 * # FUNCTION #
 * double型数値を文字列に変換
 * 小数点以下の無意味な０は取り除かれる。(ex. 1.560000 -> 1.56, 1.0 -> 1)
 */
char *mssFtoA(double num)
{
/*  int i = 0;*/
  char buf[2048];

  sprintf(buf,mssGV.sigDigitsForm,num);
/*
  sprintf(buf,"%g",num);
  while(buf[i]!='\0') i++;
  while(1) {
    i--;
    if(buf[i]=='.') {buf[i]='\0';break;}
    if(buf[i]=='0') {buf[i]='\0';}
    else        break;
  }
*/
  return(mssStrdup(buf));
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 乱数関連の関数群
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 乱数シードの設定
 * 乱数シードを設定し、そのシード(数値)を返す
 * 引数seedに-1を与えたときは、現在時刻をシードとして用いる。
 */
int mssInitRand(int seed)
{
  time_t  long_time;       /*乱数の種用の時間構造体*/                         
  struct tm     *nt;       /*同上*/
  int usedSeed;

  if(seed==-1){
    time(&long_time);
    nt = localtime(&long_time);
    usedSeed=nt->tm_hour*10000+nt->tm_min*100+nt->tm_sec;
  }else{
    usedSeed=seed;
  }
  init_urnd0((long)usedSeed);
  init_irnd(initRND[0],_func[0],(long)usedSeed);
  init_gaussd(initRND[0],_func[0],(long)usedSeed);
  return(usedSeed);
}     

/**
 * # FUNCTION #
 * 0以上1以下の実数乱数を発生させる。
 */
double mssRand()
{
  return( urnd0() );
}

/**
 * # FUNCTION #
 * 与えられた整数範囲の乱数を生成し、その値を返す。
 * 整数範囲のチェックはしていないので、アプリケーション側でチェックすること。
 */
int mssIRand(int from, int to)
{
  return( irnd(to, from) );
}

/**
 * # FUNCTION #
 * 正規分布乱数(平均m,標準偏差s)。
 */
double mssNRand(double m, double s)
{
  return( gaussd(m,s) );
}

/**
 * # FUNCTION #
 * ポアソン分布乱数(平均m)。
 */
int mssPRand(double m)
{
  int k;
  m=exp(m)*urnd0();
  for(k=0; m>1.0;k++){
    m*=urnd0();
  }
  return( k );
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 文字列操作のための各種関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 二つの文字列を連結し新しい文字列を返す。
 * もとの二つの文字列には一切変更を加えない。
 * 新たな領域を確保する。
 */
char *mssCatTwoStrings(char *str1, char *str2)
{
  char *str;
  int len;
  int len1=0;
  int len2=0;

  if(str1!=NULL){
    len1=strlen(str1);
  }

  if(str2!=NULL){
    len2=strlen(str2);
  }
  len=len1+len2;

  if(len==0) return(mssStrdup(""));

  str=mssMalloc(sizeof(char)*(len+1),"catTwoStrings");
  *str='\0';
  if(len1!=0) strcat(str,str1);
  if(len2!=0) strcat(str,str2);
  return(str);
}
 
/**
 * # FUNCTION #
 * 括弧などで囲まれた文字列ブロックをスキップし、その次の位置を返す。
 * 指定されたstartChr,endCharがなかったり、括弧の対応関係がおかしければNULLを
 * 返す。
 * 括弧の種類は開始括弧文字と終了括弧文字で指定する。
 * 開始括弧文字と終了括弧文字が同じ場合は、ネストを考慮できないので、
 * 次の括弧文字までをスキップすることになる。
 * 例1)                      例2)                    例3)
 * abc(aaa*(aaa-ccc))xyz     abc"aaa*"aaa-ccc""xyz   ab(aaa)
 *    ^              ^          ^     ^              ^      ^
 *   str          返す位置     str    返す位置       str    返す位置
 */
char *mssSkipStrBlock(char *str, char startChr, char endChr)
{
  int cnt=0;
  int same=0;

  if(startChr==endChr) same=1;

  while(1) {
    if(*str=='\0'){
      return(NULL);
    }else if(*str==startChr){
      cnt++;
    }else if(*str==endChr){
      cnt--;
    }
    str++;
    if( cnt==0        ) break;
    if( cnt==2 && same) break;
  }
  return(str);
}

/**
 * # FUNCTION #
 * 文字列のトークン分割を行い、文字列ポインタ(char **)を返す。
 * 引数dupを1とした時は、オリジナルの文字列をmssStrdupしてからトークン分割を
 * 行うのでオリジナルの文字列は一切変更されない。
 * 一方で、引数dupに0(1以外)を指定すると、オリジナルの文字列に対してトークン
 * 分割を行うので、オリジナルの文字は変更される。
 * トークン分割された文字列の数は引数cntポインタで示された
 * 整数変数にセットされる。
 *
 * なお、引数dupを1としてこの関数を用いてトークン分割した場合、
 * 新たに確保された文字列領域を開放するには、文字列ポインタの最初の
 * 要素(アドレス)のみ開放すればよい。
 *  -> 変更2004/10/28：全ての要素を解放しなければならない。
 * 利用例)
 *   char **s;
 *   int cnt;
 *   s=mssTokByChr("abc,efg,hij", ',', &cnt, 1);
 *   Free(*s);
 *
 *   結果 *(s+0)=="abc"
 *        *(s+1)=="efg"
 *        *(s+2)=="hij"
 *           cnt==3;
 */
char **mssTokByChr(char *string, char tok, int *cnt, int dup)
{
  char *orgStr;
  char *str;
  char *pos;
  char **list=NULL;
  int i;

  *cnt=0;
  if(dup) str=mssStrdup(string);
  else    str=string;
  orgStr=str;

  pos=str;
  while(1){
    if(*str==tok || *str=='\0'){
      list=mssRealloc(list,sizeof(char *)*(*cnt+1),"tokByChr");
      *(list+*cnt)=pos;
      (*cnt)++;
      pos=str+1;
      if(*str=='\0') break;
      *str='\0';
    }
    str++;
  }

  /* dupの指定があればlistの各要素もstrdupする 追加 2004/10/28 */
  if(dup){
    for(i=0; i<*cnt; i++){
      *(list+i)=mssStrdup(*(list+i));
    }
    mssFree(orgStr);
  }

  return(list);
}

/**
 * # FUNCTION #
 * 妥当な文字列かどうかチェックする。
 * 妥当な文字列とは、NULLもしくは長さ０の文字列のこと。
 */
int mssIsValidStr(char *str)
{
  if(str==NULL) return(0);
  if(*str=='\0') return(0);
  return(1);
}

/**
 * # FUNCTION #
 * mssSort用文字昇順比較ルーチン
 */
static int cmpStrUp(const void **s1,const void **s2)
{
  return (strcmp(*(char **)s1,*(char **)s2));
}

/**
 * # FUNCTION #
 * mssSort用文字降順比較ルーチン
 */
static int cmpStrDown(const void **s1,const void **s2)
{
  return (strcmp(*(char **)s1,*(char **)s2) * (-1) );
}

/**
 * # FUNCTION #
 * mssSort用数値昇順比較ルーチン
 */
static int cmpNumUp(const void **s1,const void **s2)
{
  double n1,n2;

  if(MssIsNull(*(char **)s1)) n1=-DBL_MAX;
  else                        n1=atof(*s1);
  if(MssIsNull(*(char **)s2)) n2=-DBL_MAX;
  else                        n2=atof(*s2);
  if(n1 > n2){
    return(1);
  }else if(n1 < n2){
    return(-1);
  }else{
    return(0);
  }
}

/**
 * # FUNCTION #
 * mssSort用数値降順比較ルーチン
 */
static int cmpNumDown(const void **s1,const void **s2)
{
  double n1,n2;

  if(MssIsNull(*(char **)s1)) n1=-DBL_MAX;
  else                        n1=atof(*s1);
  if(MssIsNull(*(char **)s2)) n2=-DBL_MAX;
  else                        n2=atof(*s2);
  if(n1 > n2){
    return(-1);
  }else if(n1 < n2){
    return(1);
  }else{
    return(0);
  }
}

/**
 * # FUNCTION #
 * メモリ内文字列データのソート(数値としてソートすることも可能)
 * flagで指定する数値の内容
 *     0:文字昇順
 *     1:文字降順
 *     2:数値降順
 *     3:数値昇順
 */
void mssSort(char **data, int cnt, int flag){
  switch(flag){
  case 0:
    qsort5((char *)data, cnt, sizeof(char *), (int (*))cmpStrUp);
    break;
  case 1:
    qsort5((char *)data, cnt, sizeof(char *), (int (*))cmpStrDown);
    break;
  case 2:
    qsort5((char *)data, cnt, sizeof(char *), (int (*))cmpNumUp);
    break;
  case 3:
    qsort5((char *)data, cnt, sizeof(char *), (int (*))cmpNumDown);
    break;
  }
}


/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * メッセージの出力関連の関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * コマンドラインを書き出す
 * argvに改行が含まれていればスペースに変換して出力する
 */
static void printCmdLine(struct mssFPW *fp)
{
  int i,j,len;
  char *str;


  mssWriteStr("\"",fp);
  for(i=0; i<mssGV.argc; i++){
    len=strlen(*(mssGV.argv+i));
    str=mssMalloc(sizeof(char)*(len+1),"printCmdLine");
    for(j=0; j<len; j++){
      if('\n'!=*(*(mssGV.argv+i)+j)){
        *(str+j)=*(*(mssGV.argv+i)+j);
      }else{
        *(str+j)=' ';
      }
    }
    *(str+j)='\0';
    mssWriteStr(str,fp);
    mssFree(str);

    if(i!=mssGV.argc-1){
      mssWriteStr(" ",fp);
    }else{
      mssWriteStr("\"",fp);
    }
  }
} 

/**
 * # FUNCTION #
 * 開始メッセージ
 * 出力例) #START# 2996 2003/04/11 21:56:37 "xtcut -f customer -i dat.xt"
 *   #START#に続いて、プロセス番号,日付,時刻,コマンドライン
 */
void mssShowStartMsg()
{
  struct mssFPW *fp;

  if(!isQuiet()){
    fp=mssOpenFPW((char *)1,0,0); /*標準エラー出力オープン*/
    mssWriteStr("#START# ",fp);
    mssWriteInt(getpid(),fp);
    mssWriteStr(" ",fp);
    mssWriteDate(fp,1);
    mssWriteStr(" ",fp);
    mssWriteTime(fp,1);
    mssWriteStr(" ",fp);
    printCmdLine(fp);
    mssWriteRet(fp);
    mssCloseFPW(fp);
  }
}

/**
 * # FUNCTION #
 * 正常終了メッセージ
 * 出力例)
 *   #END# 2999 2003/04/11 21:59:21 "xtcut -f customer -i xxcc.xt" in=25 out=25
 *   #END#に続いて、プロセス番号,日付,時刻,コマンドライン,入力件数,出力件数
 */
void mssShowEndMsg()
{
  struct mssFPW *fp;

  if(!isQuiet()){
    fp=mssOpenFPW((char *)1,0,0); /*標準エラー出力オープン*/
    mssWriteStr("#END# ",fp);
    mssWriteInt(getpid(),fp);
    mssWriteStr(" ",fp);
    mssWriteDate(fp,1);
    mssWriteStr(" ",fp);
    mssWriteTime(fp,1);
    mssWriteStr(" ",fp);
    printCmdLine(fp);
    mssWriteStr(" in="  ,fp);
    mssWriteInt(mssGV.inCnt  ,fp);
    mssWriteStr(" out=",fp);
    mssWriteInt(mssGV.outCnt ,fp);
    mssWriteRet(fp);
    mssCloseFPW(fp);
  }
}

/**
 * # FUNCTION #
 * エラー終了メッセージ
 * メッセージに含まれる改行は全て"_"に変換される。
 * 出力例) #ERROR# 2996 "xtcut" "option -f is mandatory"
 *   #ERROR#に続いて,プロセス番号,コマンドライン,エラーメッセージ
 */
void mssShowErrMsg(char *msg, ...)
{
  char tmp[1024];
  va_list args;
  struct mssFPW *fp;
  int i;

  if(isQuiet()) return;

  va_start(args,msg);
  vsprintf(tmp,msg,args);

  /*改行を_に変換*/
  for(i=0;;i++){
    if(tmp[i]=='\0') break;
    if(tmp[i]=='\n') tmp[i]='_';
  }

  fp=mssOpenFPW((char *)1,0,0); /*標準エラー出力オープン*/
  mssWriteStr("#ERROR# ",fp);
  mssWriteInt(getpid(),fp);
  mssWriteStr(" ",fp);
  printCmdLine(fp);
  mssWriteStr(" \"",fp);
  mssWriteStr(tmp,fp);
  mssWriteStr("\"",fp);
  mssWriteRet(fp);
  mssCloseFPW(fp);
  va_end(args);
}

/**
 * # FUNCTION #
 * 単なるメッセージ出力
 *   #MSG#に続いて,プロセス番号,メッセージ
 */
void mssShowMsg(char *msg, ...)
{
  char tmp[1024];
  va_list args;
  struct mssFPW *fp;
  int i;

  if(isQuiet()) return;

  va_start(args,msg);
  vsprintf(tmp,msg,args);

  /*改行を_に変換*/
  for(i=0;;i++){
    if(tmp[i]=='\0') break;
    if(tmp[i]=='\n') tmp[i]='_';
  }

  fp=mssOpenFPW((char *)1,0,0); /*標準エラー出力オープン*/
  mssWriteStr("#MSG# ",fp);
  mssWriteInt(getpid(),fp);
  mssWriteStr(" \"",fp);
  mssWriteStr(tmp,fp);
  mssWriteStr("\"",fp);
  mssWriteRet(fp);
  mssCloseFPW(fp);
  va_end(args);
}
