/**
 * # CHAPTER #
 * ============================================================================
 * xtcal等で利用される計算式に関する関数
 * 2003/09/26 : デリミタ文字を汎用に
 * 2004/02/04 : datefmt関数の曜日をmssEncmsg.hのMssEMdowに変更(encoding対応)
 * 2004/02/09 : 数値演算範囲エラーのチェック方法変更
 * 2004/02/14 : 数値演算範囲エラーのチェック方法変更(finite)
 * 2004/04/28 : prand関数追加
 * 2004/09/11 : uppercase,lowercase,capitalize関数追加
 * 2004/10/31 : メモリリーク修正
 * ============================================================================
 * 【扱うデータ型】
 * 内部的には数値型(double型)と文字列型(char *)のみを扱う。
 *
 * 【型変換】
 * 原則的に、各演算子および関数には引数と返値にデフォルトの型が定められている。
 * 定数や項目値については、各関数の指定するデフォルトの型に変換される。
 *
 * 次に、関数の引数の型と、その引数で関数が指定されていたときの、その戻り値に
 * 型の不一致が起こった場合を考える。例えば"length(sqrt($a))"を例にすると、
 * length関数は、デフォルトで引数として文字列型をとり、sqrt関数は数値を返す。
 * この時、sqrt関数の計算結果を文字列型に変換し、length関数に渡す。
 * 文字列->数値変換はatof関数を用いている。
 * この型変換は、原則的にcal_calculate関数にてコマンド実行時に行われる。
 * ただし例外として、関数の引数として未定型を定義できる。
 * 例えば、if関数を考える。ifは第一引数の条件式が真ならば第二引数、偽ならば
 * 第三引数を返す。この時、第二引数と第三引数の型は、ユーザの指定に依存する。
 * 例えば、"if($a>1,sqrt($a),today()"の計算式を考えると、第二引数のsqrtは
 * 数値を返し、第三引数のtoday関数は文字列を返す。
 * このようなケースでは、関数のデフォルトの型を未定義型にしておく。
 * 未定義型とその他の固定された型との違いは、未定義型の場合はcal_calculate関数
 * にて型変換を行わないことにある。
 * よって、関数を実装する場合、引数の型をS,Nにすれば、その関数には、
 * 必ず指定の型の値が引数として渡されることが保証される。
 * 
 * 【定数の与え方】
 * 定数は文字列であってもダブルクオーテーション(")で囲う必要はない。
 * ex. cat(abc,def,ghi,_) sqrt(10)
 * 
 * 【項目名の指定方法】
 * "$項目名"のように'$'記号を先頭につける。これは定数との区別をするため。
 * 可変引数をとる関数(ex. sum,cat)については、ワイルドカードを用いることが可能。
 * 例えば、"sum($数量[ab])"と指定すると、"sum($数量a,$数量b)"と指定したと
 * 等しくなる。但し、内部的には、"sum($数量[ab])"と"sum($数量a,$数量b)"の
 * mssCal構造体での格納方法は異なる。前者はcal->arg[0]->fldsに二つの項目がセット
 * され、後者は、cal->arg[0]->fldsに"$数量a"が、cal-arg[1]->fldsに"$数量b"が
 * それぞれセットされる。
 * また万能マッチの'*'の利用は注意が必要である。もし、"sum($数量*)"と指定した
 * 場合、'*'記号を乗算演算子として解釈すべきか、ワイルドカードとして解釈すべき
 * か、判断がつかない。現在のところ乗算演算子として解釈してしまう。
 * そこで'*'のワイルドカードを利用する時は、sum($(数量*))のように、
 * 項目名識別括弧を利用する。
 *
 * 【各関数で確保した文字列領域の開放】
 * 各ユーザ定義関数で確保された文字列領域は、setStrMalAdd(char *)関数を用いて、
 * そのアドレスを登録しておく。そのアドレスは、cal_calculate関数を実行するたび
 * に開放される。
 *
 * 【日付について】
 * 日付は８桁の文字列(yyyymmdd)で与える。
 * グレゴリオ歴の初日(1582年10月15日)より有効。
 *
 */

#include <musashi/mssCal.h>
#include <musashi/mssConfig.h>
#include <musashi/mssBase.h>
#include <musashi/mssValue.h>
#include <musashi/mssEncmsg.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/time.h>
#include <regex.h>
#include <math.h>
#include <float.h>
#include <errno.h>

/* 内部の関数が用いるデータタイプ*/
#define N 1   /*数値型*/
#define S 2   /*文字列型*/
#define M 64  /*可変引数*/
#define X 128 /*未定義型*/
#define E 256 /*終端*/

/* 構造体calのタイプ*/
#define NODE  0
#define LEAF  1
#define CONST 2
#define FIELD 3

/**
 * # STRUCT #
 * 優先順位の高い関数、演算子を見つけるための構造体
 */
struct OpeTbl {
  char *add; /*アドレス*/
  int   num; /*出現順序*/
  int   pri; /*演算子優先順位*/
  struct Func *func;
};

struct OpeLst {
  struct OpeTbl *tbl;
  int            cnt;
};

extern struct mssGlobalVariables mssGV;

static int UsedTopResult=0;
static int UsedPrvResult=0;
static int UsedTopField=0;
static int UsedPrvField=0;
static struct mssFldRec    *Fr=NULL;
static struct mssFldRecDbl *Frd=NULL;
static struct mssFldRecKey *Frk=NULL;

static struct PrvRslStruct {
  MssValue val;
  char  buf[MssRecordMaxLen];
} PrvRsl,TopRsl;

static struct PrvFldStruct {
  char *pnt[MssFieldMaxCnt];
  char  buf[MssRecordMaxLen];
} PrvFld,TopFld;


/* 文字列操作処理をおこなう演算子や関数で確保されたメモリ領域を後でfreeするために記憶させる領域 */
char       *strMalAdd[256]; /*実際の各関数でstr処理をしてmallocした*/
                            /*アドレス一覧*/
int         strMalCnt;      /*その件数*/

/**
 * # FUNCTION #
 * 文字列操作処理で確保されたメモリ領域をクリア
 */
static void cal_freeStrMalAdd(void)
{
  int i;
  for(i=0; i<strMalCnt; i++){
    mssFree(strMalAdd[i]);
  }
  strMalCnt=0;
}

/**
 * # FUNCTION #
 * 文字列操作処理で確保されたメモリ領域を登録
 */
static void setStrMalAdd(char *str)
{
  if(strMalCnt==100){
    mssShowErrMsg("too many string operation");
    mssEnd(mssErrorNoDefault);
  }
  strMalAdd[strMalCnt++]=str;
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 演算子および関数の定義
 * ----------------------------------------------------------------------------
 * 各演算子や関数での引数の型の扱いに関するルール
 * 1. 関数が扱う型はMssValue型で、内部的には"double"(数値と論理値)と
 *    "char *"(文字列)のみで処理する。
 * 2. 引数の型に合わないMssValueが引数に渡された時、各関数の実行時に型変換される。
 */

/**
 * # FUNCTION #
 * +(加算)演算子
 * 与え方: 数値1 + 数値2
 * 返値: 上記式演算結果
 */
static MssValue cal_addOperator(struct mssCal *cal)
{
  return(mssVadd(cal->arg[0].val[0],cal->arg[1].val[0]));
}

/**
 * # FUNCTION #
 * -(減算)演算子
 * 与え方: 数値1 - 数値2
 * 返値: 上記式演算結果
 */
static MssValue cal_subOperator(struct mssCal *cal)
{
  return(mssVsub(cal->arg[0].val[0],cal->arg[1].val[0]));
}

/**
 * # FUNCTION #
 * *(乗算)演算子
 * 与え方: 数値1 * 数値2
 * 返値: 上記式演算結果
 */
static MssValue cal_mulOperator(struct mssCal *cal)
{
  return(mssVmul(cal->arg[0].val[0],cal->arg[1].val[0]));
}

/**
 * # FUNCTION #
 * /(除算)演算子
 * 与え方: 数値1 / 数値2
 * 返値: 上記式演算結果
 */
static MssValue cal_divOperator(struct mssCal *cal)
{
  return(mssVdiv(cal->arg[0].val[0],cal->arg[1].val[0]));
}

/**
 * # FUNCTION #
 * %(剰余)演算子
 * 与え方: 数値1 % 数値2
 * 返値: 上記式演算結果
 */
static MssValue cal_modOperator(struct mssCal *cal)
{
  return(mssVmod(cal->arg[0].val[0],cal->arg[1].val[0]));
}

/**
 * # FUNCTION #
 * ==(数値比較:equal)演算子
 * 与え方: 数値1 == 数値2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_eqNOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_EQ, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * <>(数値比較:not equal)演算子
 * 与え方: 数値1 <> 数値2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_neNOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_NE, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * >=(数値比較:greater or equal)演算子
 * 与え方: 数値1 >= 数値2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_geNOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_GE, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * <=(数値比較:less or equal)演算子
 * 与え方: 数値1 <= 数値2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_leNOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_LE, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * >(数値比較:greater)演算子
 * 与え方: 数値1 > 数値2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_gtNOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_GT, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * <(数値比較:less)演算子
 * 与え方: 数値1 < 数値2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_ltNOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_LT, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * ==(文字列比較:equal)演算子
 * 与え方: 文字列1 -eq 文字列2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_eqSOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_EQ, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * !=(文字列比較:not equal)演算子
 * 与え方: 文字列1 -ne 文字列2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_neSOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_NE, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * >=(文字列比較:greater or equal)演算子
 * 与え方: 文字列1 -ge 文字列2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_geSOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_GE, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * <=(文字列比較:less or equal)演算子
 * 与え方: 文字列1 -le 文字列2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_leSOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_LE, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * >(文字列比較:greater)演算子
 * 与え方: 文字列1 -gt 文字列2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_gtSOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_GT, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * <(文字列比較:less)演算子
 * 与え方: 文字列1 -lt 文字列2
 * 返値: 0:条件不一致, 1:条件一致
 */
static MssValue cal_ltSOperator(struct mssCal *cal)
{
  int rsl;
  MssValue v;
  mssVinit(&v,DBL);

  rsl=mssVcmpOpe(cal->arg[0].val[0], OPE_LT, cal->arg[1].val[0]);

  if(rsl==-1) MssVnull(v);
  else        v.v.d=(double)rsl;

  return(v);
}

/**
 * # FUNCTION #
 * ||(論理和)演算子
 * 与え方: 数値1 || 数値2
 * 数値1、数値2のいずれかが１の時１を返し、それ以外の場合は０を返す。
 */
static MssValue cal_orOperator(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul ){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d==1 || cal->arg[1].val[0].v.d==1){
      v.v.d=1;
    }else{
      v.v.d=0;
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * &&(論理積)演算子
 * 与え方: 数値1 && 数値2
 * 数値1、数値2ともに１の時１を返し、それ以外の場合は０を返す。
 */
static MssValue cal_andOperator(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul ){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d==1 && cal->arg[1].val[0].v.d==1){
      v.v.d=1;
    }else{
      v.v.d=0;
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * if関数
 * 与え方: if(数値,value1,value2)
 * 数値が１の時はvalue1を、それ以外の時はvalue2を返す。
 * value1,value2の型変換は行わず、そのまま返す。
 * 数値がNULLの場合、STR型のNULL値を返す。
 */
static MssValue cal_if(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,STR);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d==1)
      v=cal->arg[1].val[0];
    else
      v=cal->arg[2].val[0];
  }
  return(v);
}

/**
 * # FUNCTION #
 * not関数
 * 与え方: not(数値)
 * 数値が０の時は１を、それ以外の時は０を返す。
 * その他の場合は、NULL値を返す。
 */
static MssValue cal_not(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d ==0 ){
      v.v.d = 1;
    }else{
      v.v.d = 0;
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * sum関数
 * 与え方: sum(数値1,数値2,...,数値n)
 * 数値1から数値nまでを合計し、その値を返す。
 * その中に一つでもNULL値があると、NULL値を返す。
 * sum関数では、項目名の指定にワイルドカードが利用できる。
 */
static MssValue cal_sum(struct mssCal *cal)
{
  MssValue v;
  int i,j;
  mssVinit(&v,DBL);

  for(i=0; i<cal->argCnt; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      if(cal->arg[i].val[j].nul){
        MssVnull(v); return(v);
      }else{
        v.v.d += cal->arg[i].val[j].v.d;
      }
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * min関数
 * 与え方: min(数値1,数値2,...,数値n)
 * 数値1から数値nの中で最小値を返す。
 * その中に一つでもNULL値があると、NULL値を返す。
 * min関数では、項目名の指定にワイルドカードが利用できる。
 */
static MssValue cal_min(struct mssCal *cal)
{
  MssValue v;
  int i,j;
  mssVinit(&v,DBL);

  mssVclearMax(&v);
  for(i=0; i<cal->argCnt; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      if(cal->arg[i].val[j].nul){
        MssVnull(v); return(v);
      }else{
        if(v.v.d > cal->arg[i].val[j].v.d){
          v.v.d = cal->arg[i].val[j].v.d;
        }
      }
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * max関数
 * 与え方: max(数値1,数値2,...,数値n)
 * 数値1から数値nの中で最小値を返す。
 * その中に一つでもNULL値があると、NULL値を返す。
 * maxn関数では、項目名の指定にワイルドカードが利用できる。
 */
static MssValue cal_max(struct mssCal *cal)
{
  MssValue v;
  int i,j;
  mssVinit(&v,DBL);

  mssVclearMin(&v);
  for(i=0; i<cal->argCnt; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      if(cal->arg[i].val[j].nul){
        MssVnull(v); return(v);
      }else{
        if(v.v.d < cal->arg[i].val[j].v.d){
          v.v.d = cal->arg[i].val[j].v.d;
        }
      }
    }
  }
  return(v);
}


/**
 * # FUNCTION #
 * log関数
 * 与え方: log(数値1,数値2)
 * 数値2を底とする数値1の対数を返す。
 * 数値1もしくは数値2が0以下の場合はNULL値を返す。
 */
static MssValue cal_log(struct mssCal *cal)
{
  MssValue v;
  double rsl1=0;
  double rsl2=0;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul ){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d <=0 || cal->arg[1].val[0].v.d <=0){
      MssVnull(v);
    }else{
      rsl1=log(cal->arg[0].val[0].v.d);
      rsl2=log(cal->arg[1].val[0].v.d);

      if(rsl2==0){
        MssVnull(v);
      }else{
        v.v.d = rsl1/rsl2;
      }
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * log2関数
 * 与え方: log2(数値)
 * 2を底とする数値の対数を返す。
 * 数値が0以下の場合はNULL値を返す。
 */
static MssValue cal_log2(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d <=0 ){
      MssVnull(v);
    }else{
      v.v.d=log(cal->arg[0].val[0].v.d)/log(2);
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * log関数
 * 与え方: log10(数値)
 * 10を底とする数値の対数を返す。
 * 数値が0以下の場合はNULL値を返す。
 */
static MssValue cal_log10(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d <=0 ){
      MssVnull(v);
    }else{
      v.v.d = log10(cal->arg[0].val[0].v.d);
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * ln関数
 * 与え方: ln(数値)
 * 数値の自然対数を返す。
 * 数値が0以下の場合はNULL値を返す。
 */
static MssValue cal_ln(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
  }else{
    if(cal->arg[0].val[0].v.d <=0 ){
      MssVnull(v); return(v);
    }else{
      v.v.d = log(cal->arg[0].val[0].v.d);
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * rand01関数
 * 与え方: rand01(乱数の種)
 * 0から1までの一様疑似乱数を返す。
 */
static MssValue cal_rand01(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    /*乱数の種*/
    mssInitRand((int)cal->arg[0].val[0].v.d);
  }

  v.v.d = (double)mssRand();
  return(v);
}

/**
 * # FUNCTION #
 * rand関数
 * 与え方: rand(最小値,種類数,乱数の種)
 * 最小値以上の連続する整数から、種類数の疑似乱数を返す。
 * すなわち、最小値から"最小値+種類数-1"の整数の疑似乱数を生成する。
 * 乱数の種が-1の時、時間に応じた乱数の種を用いる。
 * ex) rand(2,3,1): 2,3,4 のいずれかの値を返す。
 * 種類数が1以下ならエラー＆終了。
 */
static MssValue cal_rand(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    if(cal->arg[1].val[0].v.d<=1) {
      mssShowErrMsg("the second parameter must be more than 2.");
      mssEnd(mssErrorNoDefault);
    }
    /*乱数の種*/
    mssInitRand((int)cal->arg[2].val[0].v.d);
  }

  v.v.d = (double)mssIRand((int)cal->arg[0].val[0].v.d, (int)cal->arg[0].val[0].v.d+(int)cal->arg[1].val[0].v.d-1);
  return(v);
}

/**
 * # FUNCTION #
 * nrand関数
 * 与え方: nrand(平均値,標準偏差,乱数の種)
 * 正規分布に基づいた疑似乱数を返す。
 * 正規分布は、平均値、標準偏差を与えることによって決まる。
 * 乱数の種が-1の時、時間に応じた乱数の種を用いる。
 *
 * 【求め方】
 * 平均0、標準偏差1の正規分布関数：y=1/sqrt(2*pi) * e^(-x^2/2)
 * x=0の時、最大値y=1/sqrt(2*pi)となる。
 * ここでa=1/sqrt(2*pi)と置くと
 * y=a*e^(-x^2/2)
 * またe^(-x^2/2)=sqrt(1/e)^(x^2)で、sqrt(1/e)は定数なのでesとおくと
 * y=a*es^(x^2)となる ---- (1)
 * yの最大値が1、最小値が0となるように基準化すると
 * y=(1)式/最大値a = es^(x^2) となる
 *
 * ここで正規分布に従った乱数を求める手順は
 * 1) xを乱数で与えyの値を求める
 * 2) 再び乱数rを求める
 * 3) r<=y ならば、その時のxの値を乱数侯補とし、そうでなければ1)に戻る
 * 4) 乱数侯補xがパラメータで指定された乱数の種類数に収まらなければ1)に戻る
 * 5) 乱数侯補xを乱数として採用する
 * ただし平均を中心としてマイナスとプラスがあるので、乱数を求める毎に切替える。
 */
static MssValue cal_nrand(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    /*乱数の種*/
    mssInitRand((int)cal->arg[2].val[0].v.d);
  }
  v.v.d = mssNRand(cal->arg[0].val[0].v.d,cal->arg[1].val[0].v.d);
  return(v);
}

/**
 * # FUNCTION #
 * prand関数
 * 与え方: prand(平均値)
 * 平均値mのポアソン分布に基づいた疑似乱数を返す。
 */
static MssValue cal_prand(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    /*乱数の種*/
    mssInitRand((int)cal->arg[1].val[0].v.d);
  }
  v.v.d = mssPRand(cal->arg[0].val[0].v.d);
  return(v);
}

/**
 * # FUNCTION #
 * cal->arg[i]の中に一つでもNULL値があれば１を返す。それ以外は０を返す。
 */
static int argHaveNull(struct mssCal *cal)
{
  int i,j;
  for(i=0; i<cal->argCnt; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      if(cal->arg[i].val[j].nul){
        return(1);
      }
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * cat関数
 * 与え方: cat(文字列1,文字列2,...,文字列n,区切り文字)
 * 文字列1から文字列2を区切り文字をはさんで結合し、その結果を返す。
 * その中に一つでもNULL値があると、NULL値を返す。
 * cat関数では、項目名の指定にワイルドカードが利用できる。
 * 区切り文字にスペースがある場合は"_"に変換される。
 */
static MssValue cal_cat(struct mssCal *cal)
{
  int i,j;
  MssValue v;
  char *token;
  int   tokenLen;
  int   len;
  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  /*デリミタの設定*/
  token=cal->arg[cal->argCnt-1].val[0].v.s;
  tokenLen=strlen(token);
  for(i=0;i<tokenLen; i++){
    if(*(token+i)==MssFieldDelim) *(token+i)='_';
  }

  len=strlen(cal->arg[0].val[0].v.s)+1;
  v.v.s=mssMalloc(len,"cal_cat");
  strcpy(v.v.s, cal->arg[0].val[0].v.s);

  for(j=1; j<cal->arg[0].valCnt; j++){
    len += strlen(cal->arg[0].val[j].v.s)+tokenLen;
    v.v.s=mssReallocLim(v.v.s,len,MssFieldMaxLen,"cal_cat");
    strcat(v.v.s, token);
    strcat(v.v.s, cal->arg[0].val[j].v.s);
  }

  for(i=1; i<cal->argCnt-1; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      len += strlen(cal->arg[i].val[j].v.s)+tokenLen;
      v.v.s=mssReallocLim(v.v.s,len,MssFieldMaxLen,"cal_cat");
      strcat(v.v.s, token);
      strcat(v.v.s, cal->arg[i].val[j].v.s);
    }
  }
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * length関数
 * 与え方: length(文字列)
 * 文字列の長さをバイト数で返す。
 */
static MssValue cal_length(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  v.v.d=(double)strlen(cal->arg[0].val[0].v.s);
  return(v);
}

/**
 * # FUNCTION #
 * fix関数
 * 与え方： fix(文字列,文字数,位置,詰める文字)
 * 「文字列」を「文字数」の固定長に変換した文字列を返す。
 * 文字数はバイト数で指定する。
 * 固定長にする場合、文字列の長さが「文字数」より短い場合、
 * パディング(詰める)文字を「詰める文字」で指定する。
 * パディング文字にスペースは指定できない。
 * また、パディングする際に、文字列を左右のどちらにそろえるかを
 * 「位置」に"R"(右)もしくは"L"(左)を与えることによって指定する。
 * 「文字列」の長さが「文字数」より長い場合、「位置」がLの時は「文字列」の右側
 * の超過分が削られ、Rの時は左側が削られる。
 */
static MssValue cal_fix(struct mssCal *cal)
{
  int i;
  MssValue v;
  char *str;
  char  rl;
  unsigned char  pad;
  int   len;
  int   sl;
  int   padLen;
  int   strLen;
  int   pos;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  str =      cal->arg[0].val[0].v.s;
  len = (int)cal->arg[1].val[0].v.d;
  rl  =     *cal->arg[2].val[0].v.s;
  pad =     *cal->arg[3].val[0].v.s;
  sl=strlen(str);

  if(isspace(pad)) {
    mssShowErrMsg("padding charactor cannot be a space");
    mssEnd(mssErrorNoDefault);
  }

  if(len>=MssFieldMaxLen) {
    mssShowErrMsg("second parameter in [fix] function must be in a rage of 1-%d",MssFieldMaxLen-1);
    mssEnd(mssErrorNoDefault);
  }

  v.v.s=mssMalloc(len+1,"cal_fix");
  if(sl>len){ padLen=0     ; strLen=len; }
  else      { padLen=len-sl; strLen=sl ; }

  switch(rl){
  case 'L':
    pos=0;
    for(i=0; i<strLen; i++) {
      *(v.v.s+pos)=*(str+i);
      pos++;
    }
    for(i=0; i<padLen; i++) {
      *(v.v.s+pos)=pad;
      pos++;
    }
    *(v.v.s+pos)='\0';
    break;
  case 'R':
    pos=len-1;
    for(i=sl-1; i>=sl-strLen; i--) {
      *(v.v.s+pos)=*(str+i);
      pos--;
    }
    for(i=0; i<padLen; i++) {
      *(v.v.s+pos)=pad;
      pos--;
    }
    *(v.v.s+len)='\0';
    break;
  default:
    mssShowErrMsg("specify [R] or [L] in the third parameter in [fix] function.");
    mssEnd(mssErrorNoDefault);
    break;
  }

  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * left関数
 * 与え方： left(文字列,文字数)
 * 「文字列」の左から「文字数」の文字列を切りだし、その値を返す。
 * 文字数はバイト数で指定する。
 * 「文字数」が「文字列」の長さより長ければ、「文字列」をそのまま返す。
 */
static MssValue cal_left(struct mssCal *cal)
{
  MssValue v;
  char *str;
  int   len;
  int   sl;
  int   strLen;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  str =      cal->arg[0].val[0].v.s;
  len = (int)cal->arg[1].val[0].v.d;
  sl=strlen(str);

  if(len>=MssFieldMaxLen) {
    mssShowErrMsg("invalid length specified in [left] function");
    mssEnd(mssErrorNoDefault);
  }

  if(sl>len) strLen=len;
  else       strLen=sl ;
  v.v.s=mssMalloc(strLen+1,"cal_left");
  strncpy(v.v.s,str,strLen);
  *(v.v.s+strLen)='\0';
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * right関数
 * 与え方： right(文字列,文字数)
 * 「文字列」の右から「文字数」の文字列を切りだし、その値を返す。
 * 文字数はバイト数で指定する。
 * 「文字数」が「文字列」の長さより長ければ、「文字列」をそのまま返す。
 */
static MssValue cal_right(struct mssCal *cal)
{
  MssValue v;
  char *str;
  int   len;
  int   sl;
  int   strLen;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  str =      cal->arg[0].val[0].v.s;
  len = (int)cal->arg[1].val[0].v.d;
  sl=strlen(str);

  if(len>=MssFieldMaxLen || len <=0) {
    mssShowErrMsg("invalid length specified in [right] function");
    mssEnd(mssErrorNoDefault);
  }

  if(sl>len) strLen=len;
  else       strLen=sl ;
  v.v.s=mssMalloc(strLen+1,"cal_left");
  strncpy(v.v.s,str+(sl-strLen),strLen);
  *(v.v.s+strLen)='\0';
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * substr関数
 * 与え方： substr(文字列,開始位置,終了位置)
 * 「文字列」の「開始位置」から「終了位置」の文字列を切りだし、その値を返す。
 * 「終了位置」が「文字列」の末尾を超えた場合は、終了位置が末尾に変更される。
 * 「開始位置」、「終了位置」ともに末尾を超えた場合は、NULL値を返す。
 * 「開始位置」、「終了位置」が０以下の数字の場合はエラー終了する。
 */
static MssValue cal_substr(struct mssCal *cal)
{
  MssValue v;
  char *str;
  int   from;
  int   to;
  int   len;
  int   sl;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  str  =      cal->arg[0].val[0].v.s;
  from = (int)cal->arg[1].val[0].v.d;
  to   = (int)cal->arg[2].val[0].v.d;
  sl=strlen(str);

  if(from>to) mssSwapInt(&from,&to);
  len=to-from+1;

  if(len>=MssFieldMaxLen || len<=0 || from<=0 || to<=0) {
    mssShowErrMsg("invalid length specified in [substr] function");
    mssEnd(mssErrorNoDefault);
  }

  /*スタート位置が文字列長を超えていたらNULL*/
  if(from>sl) {
    MssVnull(v);
    return(v);
  }

  /*エンド位置が文字列長を超えていたら、文字列の最後に変える*/
  if(to>sl) to=sl;

  len=to-from+1;

  v.v.s=mssMalloc(len+1,"cal_left");
  strncpy(v.v.s,str+from-1,len);
  *(v.v.s+len)='\0';
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * =~比較演算子
 * 与え方： 文字列=~正規表現
 * 文字列が正規表現にマッチすれば１、しなければ０を返す。
 * ただし、正規表現のコンパイルは、データの一行目を処理の時のみ行われる。
 * 正規表現として、Extended Regular Expression(ERE)を用いている。
 */
static MssValue cal_regOperator(struct mssCal *cal)
{

  MssValue v;

  int rsl;
  char *str;
  char *reg;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    reg=cal->arg[1].val[0].v.s;
    cal->regex=mssMalloc(sizeof(regex_t),"comp regex"); /* 追加 2004/10/31 */
    if(regcomp(cal->regex,reg,REG_EXTENDED) ){
      mssShowErrMsg("error in compiling regex");
      mssEnd(mssErrorNoDefault);
    }
  }

  if(cal->arg[0].val[0].nul) {
    MssVnull(v);
    return(v);
  }

  str=cal->arg[0].val[0].v.s;

  rsl=regexec(cal->regex, str, 0, NULL,0);
  if(rsl!=0){
    v.v.d=0;
  }else{
    v.v.d=1;
  }
  return(v);
}

/**
 * # FUNCTION #
 * regexEnd関数
 * 与え方： regexEnd(文字列,正規表現)
 * 文字列の中で、正規表現にマッチする終端位置を返す。
 * ただし、正規表現のコンパイルは、データの一行目を処理の時のみ行われる。
 * 正規表現として、Extended Regular Expression(ERE)を用いている。
 */
static MssValue cal_regexEnd(struct mssCal *cal)
{

  MssValue v;
  regmatch_t matchPnt;

  int rsl;
  char *str;
  char *reg;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    reg=cal->arg[1].val[0].v.s;
    cal->regex=mssMalloc(sizeof(regex_t),"comp regex"); /* 追加 2004/10/31 */
    if(regcomp(cal->regex,reg,REG_EXTENDED) ){
      mssShowErrMsg("error in compiling regex");
      mssEnd(mssErrorNoDefault);
    }
  }

  if(cal->arg[0].val[0].nul) {
    MssVnull(v);
    return(v);
  }

  str=cal->arg[0].val[0].v.s;

  rsl=regexec(cal->regex, str, 1, &matchPnt,0);
  if(rsl!=0){
    MssVnull(v);
  }else{
    v.v.d=matchPnt.rm_eo;
  }
  return(v);
}

/**
 * # FUNCTION #
 * regexStart関数
 * 与え方： regexStart(文字列,正規表現)
 * 文字列の中で、正規表現にマッチする開始位置を返す。
 * ただし、正規表現のコンパイルは、データの一行目を処理の時のみ行われる。
 * 正規表現として、Extended Regular Expression(ERE)を用いている。
 */
static MssValue cal_regexStart(struct mssCal *cal)
{

  MssValue v;
  regmatch_t matchPnt;

  int rsl;
  char *str;
  char *reg;
  mssVinit(&v,DBL);

  if(mssGV.inCnt==1){
    reg=cal->arg[1].val[0].v.s;
    cal->regex=mssMalloc(sizeof(regex_t),"comp regex"); /* 追加 2004/10/31 */
    if(regcomp(cal->regex,reg,REG_EXTENDED) ){
      mssShowErrMsg("error in compiling regex");
      mssEnd(mssErrorNoDefault);
    }
  }

  if(cal->arg[0].val[0].nul) {
    MssVnull(v);
    return(v);
  }

  str=cal->arg[0].val[0].v.s;

  rsl=regexec(cal->regex, str, 1, &matchPnt,0);
  if(rsl!=0){
    MssVnull(v);
  }else{
    v.v.d=matchPnt.rm_so+1;
  }
  return(v);
}

/**
 * # FUNCTION #
 * regexLen関数
 * 与え方： regexStart(文字列,正規表現)
 * 文字列の中で、正規表現にマッチする部分文字列の長さをバイト数として返す。
 * ただし、正規表現のコンパイルは、データの一行目を処理の時のみ行われる。
 * 正規表現として、Extended Regular Expression(ERE)を用いている。
 */
static MssValue cal_regexLen(struct mssCal *cal)
{

  MssValue v;
  regmatch_t matchPnt;

  int rsl;
  char *str;
  char *reg;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul) {
    MssVnull(v);
    return(v);
  }

  str=cal->arg[0].val[0].v.s;
  if(mssGV.inCnt==1){
    reg=cal->arg[1].val[0].v.s;
    cal->regex=mssMalloc(sizeof(regex_t),"comp regex"); /* 追加 2004/10/31 */
    if(regcomp(cal->regex,reg,REG_EXTENDED) ){
      mssShowErrMsg("error in compiling regex");
      mssEnd(mssErrorNoDefault);
    }
  }

  rsl=regexec(cal->regex, str, 1, &matchPnt,0);
  if(rsl!=0){
    MssVnull(v);
  }else{
    v.v.d=matchPnt.rm_eo-matchPnt.rm_so;
  }
  return(v);
}

/**
 * # FUNCTION #
 * isIn関数
 * 与え方: isIn(検索文字列,文字列1,...,文字列n)
 * 検索文字列が文字列1から文字列nの中にあれば１を返す。
 * 文字列1から文字列nの中にNULL値があっても無視される。
 * isIn関数では、項目名の指定にワイルドカードが利用できる。
 */
static MssValue cal_isIn(struct mssCal *cal)
{
  MssValue v;
  int i,j;
  mssVinit(&v,DBL);

  for(i=1; i<cal->argCnt; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      if(!cal->arg[i].val[j].nul){
        /*一致すれば１を返す*/
        if(strcmp(cal->arg[i].val[j].v.s,cal->arg[0].val[0].v.s)==0){
          v.v.d=1;
          return(v);
        }
      }
    }
  }
  /*一致するものがなければここまでくる。そして０を返す*/
  return(v);
}


/**
 * # FUNCTION #
 * isNull関数
 * 与え方: isNull(項目値)
 * 項目値がNULLならば１を返す。
 */
static MssValue cal_isNull(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if( cal->arg[0].val[0].nul ) v.v.d=1;
  else                         v.v.d=0;
  return(v);
}

/**
 * # FUNCTION #
 * uppercase関数
 * 与え方: uppercase(項目値)
 * 項目値の小文字アルファベットを大文字に変換しその値を返す。
 */
static MssValue cal_uppercase(struct mssCal *cal)
{
  MssValue v;
  char *str;
  char *top;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  top=str=mssStrdup(cal->arg[0].val[0].v.s);

  while(*str){
    *str=toupper(*str);
    str++;
  }
  v.v.s=top;
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * lowercase関数
 * 与え方: lowercase(項目値)
 * 項目値の小文字アルファベットを大文字に変換しその値を返す。
 */
static MssValue cal_lowercase(struct mssCal *cal)
{
  MssValue v;
  char *str;
  char *top;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  top=str=mssStrdup(cal->arg[0].val[0].v.s);

  while(*str){
    *str=tolower(*str);
    str++;
  }
  v.v.s=top;
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * capitalize関数
 * 与え方: capitalize(項目値)
 * 項目値の最初のアルファベットを大文字に変換しその値を返す。
 */
static MssValue cal_capitalize(struct mssCal *cal)
{
  MssValue v;
  char *str;
  char *top;

  mssVinit(&v,STR);

  if(argHaveNull(cal)) {
    MssVnull(v);
    return(v);
  }

  top=str=mssStrdup(cal->arg[0].val[0].v.s);

  *str=toupper(*str);
  v.v.s=top;
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * pi関数
 * 与え方： pi()
 * πの値を返す。
 */
static MssValue cal_pi(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  v.v.d=M_PI;
  return(v);
}

/**
 * # FUNCTION #
 * sqrt関数
 * 与え方： sqrt(数値)
 * 数値の平方根を返す。数値がマイナスの時はNULL値を返す。
 */
static MssValue cal_Sqrt(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[0].val[0].v.d<0){
    MssVnull(v);
  }else{
    v.v.d = sqrt(cal->arg[0].val[0].v.d);
  }

  if(finite(v.v.d)==0){
    MssVnull(v);
  }

  return(v);
}

/**
 * # FUNCTION #
 * exp関数
 * 与え方： exp(数値)
 * e(ネイピア数)底とする指数「数値」の累乗(e^x)を返す。
 * 計算結果が大きすぎる場合はNULL値を返す。
 */
static MssValue cal_Exp(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  errno=0;
  if(cal->arg[0].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v.v.d = exp(cal->arg[0].val[0].v.d);

/*
  if(v.v.d == HUGE_VAL) {
    MssVnull(v);
  }
  if(errno==ERANGE || errno==EDOM){
*/
  if(finite(v.v.d)==0){
    MssVnull(v);
  }

  return(v);
}

/**
 * # FUNCTION #
 * power関数
 * 与え方： power(数値1,数値2)
 * 「数値1」の「数値2」乗を計算し、その値を返す。
 * 計算結果が大きすぎたり、小さすぎた場合はNULL値を返す。
 */
static MssValue cal_power(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  errno=0;
  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v.v.d = pow(cal->arg[0].val[0].v.d,cal->arg[1].val[0].v.d);

/*
  if(v.v.d == HUGE_VAL || v.v.d == ERANGE) {
    MssVnull(v);
  } 
  if(errno==ERANGE || errno==EDOM){
*/
  if(finite(v.v.d)==0){
    MssVnull(v);
  }
  return(v);
}

/**
 * # FUNCTION #
 * abs関数
 * 与え方： abs(数値)
 * 「数値」の絶対値を返す。
 */
static MssValue cal_abs(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
    return(v);
  }
  if(cal->arg[0].val[0].v.d<0){
    v.v.d=cal->arg[0].val[0].v.d * (-1);
  }else{
    v.v.d=cal->arg[0].val[0].v.d;
  }
  return(v);
}

/**
 * # FUNCTION #
 * up関数
 * 与え方： up(数値,基準値)
 * 数値を基準値にしたがって切り上げ、その値を返す。
 * 基準値には、切り上げた結果数値の単位を指定する。
 * 例えば、123.456という数値に対する計算結果は以下のとおり。
 * 基準値 結果
 * 1      124   -> 結果は１の倍数
 * 0.1    123.5 -> 結果は0.1の倍数
 * 10     130
 * 0.5    123.5
 */
static MssValue cal_up(struct mssCal *cal)
{
  MssValue v;
  double v0;
  double v1;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul ||
     cal->arg[1].val[0].nul ||
     cal->arg[1].val[0].v.d<=0 ){
    MssVnull(v);
    return(v);
  }

  v0=cal->arg[0].val[0].v.d;
  v1=cal->arg[1].val[0].v.d;

  if(v0<0){
    if( modf(v0/v1,&v.v.d)!=0 ) v.v.d=v.v.d*v1;
    else                        v.v.d=v0;
  }else{
    if( modf(v0/v1,&v.v.d)!=0 ) v.v.d=(v.v.d+1)*v1;
    else                        v.v.d=v0;
  }
  return(v);
}

/**
 * # FUNCTION #
 * down関数
 * 与え方： down(数値,基準値)
 * 数値を基準値にしたがって切り下げ、その値を返す。
 * 基準値には、切り下げた結果数値の単位を指定する。
 * 例えば、123.456という数値に対する計算結果は以下のとおり。
 * 基準値 結果
 * 1      123   -> 結果は１の倍数
 * 0.1    123.4 -> 結果は0.1の倍数
 * 10     120
 * 0.5    123.0
 */
static MssValue cal_down(struct mssCal *cal)
{
  MssValue v;
  double v0;
  double v1;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul ||
     cal->arg[1].val[0].nul ||
     cal->arg[1].val[0].v.d<=0 ){
    MssVnull(v);
    return(v);
  }

  v0=cal->arg[0].val[0].v.d;
  v1=cal->arg[1].val[0].v.d;

  if(v0<0){
    if( modf(v0/v1,&v.v.d)!=0 ) v.v.d=(v.v.d-1)*v1;
    else                        v.v.d=v0;
  }else{
    if( modf(v0/v1,&v.v.d)!=0 ) v.v.d=v.v.d*v1;
    else                        v.v.d=v0;
  }

  return(v);
}

/**
 * # FUNCTION #
 * round関数
 * 与え方： round(数値,基準値)
 * 数値を基準値にしたがって四捨五入し、その値を返す。
 * 基準値には、四捨五入した結果数値の単位を指定する。
 * 正確にいえば、四捨五入ではなく、基準値の整数倍の数値集合の中で最も近い数値
 * を返す。
 * 例えば、123.456という数値に対する計算結果は以下のとおり。
 * 基準値 結果
 * 1      123   -> 結果は１の倍数
 * 0.1    123.5 -> 結果は0.1の倍数
 * 10     120
 * 0.5    123.5
 */
static MssValue cal_round(struct mssCal *cal)
{
  MssValue v;
  double v0;
  double v1;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul ||
     cal->arg[1].val[0].nul ||
     cal->arg[1].val[0].v.d<=0 ){
    MssVnull(v);
    return(v);
  }

  v0=cal->arg[0].val[0].v.d;
  v1=cal->arg[1].val[0].v.d;

  if( modf(v0/v1,&v.v.d)==0 ){
    v.v.d=v0;
  }else{
    if(v0<0) modf((v0-(v1/2))/v1,&v.v.d);
    else     modf((v0+(v1/2))/v1,&v.v.d);
    v.v.d*=v1;
  }

  return(v);
}

/**
 * # FUNCTION #
 * fixdec関数
 * 与え方： up(数値,小数点以下桁数)
 * 数値の小数点以下の桁数を、指定した値でそろえ、文字列として返す。
 */
static MssValue cal_fixdec(struct mssCal *cal)
{
  char buf[256];
  char fmt[20];
  MssValue v;
  mssVinit(&v,STR);

  if(cal->arg[0].val[0].nul ||
     cal->arg[1].val[0].nul ||
     cal->arg[1].val[0].v.d<0 ||
     cal->arg[1].val[0].v.d>6 ){
    MssVnull(v);
    return(v);
  }

  fmt[0]='%';
  fmt[1]='.';
  sprintf(&fmt[2],"%df",(int)cal->arg[1].val[0].v.d);

  sprintf(buf,fmt,cal->arg[0].val[0].v.d);

  v.v.s=mssStrdup(buf);

  setStrMalAdd(v.v.s);
  return(v);
}


/**
 * # FUNCTION #
 * 有効な日付なら１をそうでなければ０を返す。
 */
static int validDate(char *d)
{
  /*長さチェック*/
  if(strlen(d) != 8) return(0);

  /*グレゴリオ歴初日より小さければ無効*/
  if(strcmp(d,"15821015")<0) return(0);

  return(1);
}

/**
 * # FUNCTION #
 * 有効な時刻なら１をそうでなければ０を返す。
 */
static int validTime(char *d)
{
  /*長さチェック*/
  if(strlen(d) != 6) return(0);

  return(1);
}

/**
 * # FUNCTION #
 *年月日時分秒からユリウス日を求める。
 * ただしグレゴリオ歴であることが前提
 * 文字列として与えられた、日付、時刻を引数にとり、日数.時刻のdouble型を返す。
 * http://alfin.mine.utsunomiya-u.ac.jp/~niy/algo/u/ut2jd.html を修正して利用
 */
static double ut2jd(char *date, char *time)
{
  int year,month,day,hour,min,sec;
  char buf[5];
  double d;
  long j;

  strncpy(buf,date  ,4); buf[4]='\0'; year =atoi(buf);
  strncpy(buf,date+4,2); buf[2]='\0'; month=atoi(buf);
  strncpy(buf,date+6,2); buf[2]='\0'; day  =atoi(buf);
  strncpy(buf,time  ,2); buf[2]='\0'; hour =atoi(buf);
  strncpy(buf,time+2,2); buf[2]='\0'; min  =atoi(buf);
  strncpy(buf,time+4,2); buf[2]='\0'; sec  =atoi(buf);
    
  if (month <= 2) {
    year--;
    month += 12;
  }
    
  if (hour < 12) {
      j = 0;
      d = 0.5;
  } else {
      j = 1;
      d = -0.5;
  }
  d += (hour*3600L + min*60 + sec)/86400.0;
   
  j += year/4;
  j+= 2 - year/100 + year/400;
  j += 1720994L + year*365L + (month+1)*30 + (month+1)*3/5 + day;

  d=d+(double)j;
  return(d);
}

/**
 * # FUNCTION #
 * ユリウス日から年月日時分秒を求める。
 * ただしグレゴリオ歴であることが前提
 * 日数.時刻のdouble型で与えたユリウス日から、日付、時刻の文字列ポインタの
 * 位置に年月日、時分秒をセットする。
 * 計算結果がグリゴリオ歴でなければ-1を返す。
 * http://alfin.mine.utsunomiya-u.ac.jp/~niy/algo/j/jd2ut.html を修正して利用
 */
static int jd2ut(char *date, char *time, double d)
{
  long c, k, e, s;
  int year,month,day,hour,min,sec;
  long j;

  j=(long)floor(d);
  d=d-(double)j;

  if (d >= 0.5) {
      j++;
      d -= 0.5;
  } else d += 0.5;

  if (j >= 2299161) j = j + 1 + (j-1867216)/36524 - (j-1867216)/146096;
  j += 1524;

  c = (j - 122.1)/365.25;
  k = 365L*c + c/4;
  e = (j - k)/30.6001;

  year = c - 4716;
  month = e - 1;
  if (month > 12) {
     (month) -= 12;
     (year)++;
  }
  day = j - k - (long)(30.6*e);

  s = d * 86400 + 0.5;
  hour = s / 3600;
  min = (s % 3600) / 60;
  sec = s % 60;

  sprintf(date,"%04d%02d%02d",year,month,day);
  sprintf(time,"%02d%02d%02d",hour,min,sec);

  /*グレゴリオ歴かどうか判定*/
  if( (year)*366+(month)*31+(day) < 1582*366+10*31+15 ) {
    return(-1);
  }else{
    return(0);
  }
}

/**
 * # FUNCTION #
 * 日付から曜日番号を求める(ツェラーの公式)。
 * 0:日, 1:月, 2:火, 3:水, 4:木, 5:金, 6:土
 * 出所:平林雅英「C言語によるプログラム辞典第１巻」,技術評論社,平成4年,p292.
 */
static int dayweek(char *date)
{
  char buf[5];
  int year,month,day;

  strncpy(buf,date  ,4); buf[4]='\0'; year =atoi(buf);
  strncpy(buf,date+4,2); buf[2]='\0'; month=atoi(buf);
  strncpy(buf,date+6,2); buf[2]='\0'; day  =atoi(buf);

  if(month<=2) year--,month+=12;
  return( (year+year/4-year/100+year/400+(13*month+8)/5+day)%7 );
}

/**
 * # FUNCTION #
 * 日付から、その年の１月１日からの経過日数求める。
 */
static int day365(char *date)
{
  char base[9];
  double fJD;
  double tJD;

  strncpy(base,date,4); base[4]='\0';
  strcat(base,"0101");
  fJD=ut2jd(base,"000000");
  tJD=ut2jd(date,"000000");

  return( (int)tJD-(int)fJD );
}


/**
 * # FUNCTION #
 * ut2jd関数
 * 与え方： ud2jd(日付,時刻)
 * 「日付」の「時刻」をユリウス通日に変換し、その値を返す。
 * ユリウス通日とは、整数部にBC4712年1月1日正午からの経過日数をもち、
 * 小数部にあまりの時間をもった実数である。
 * ただし、グレゴリオ歴の範囲内でない日付は無効としNULLを返す。
 * ex. 1991年6月15日0時0分0秒 -> 2448422.5
 */
static MssValue cal_dt2jd(struct mssCal *cal)
{
  MssValue v;

  char *v0;
  char *v1;

  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0 =cal->arg[0].val[0].v.s;
  v1 =cal->arg[1].val[0].v.s;

  if(!validDate(v0) || !validTime(v1)){
    MssVnull(v);
    return(v);
  }

  v.v.d=ut2jd(v0,v1);
  return(v);
}

/**
 * # FUNCTION #
 * jd2date関数
 * 与え方： jd2date(ユリウス通日)
 * 「ユリウス通日」から日付を計算しその値を返す。
 * ex. 2448422.5 -> 19910615
 */
static MssValue cal_jd2date(struct mssCal *cal)
{
  MssValue v;
  char bufD[9];
  char bufT[7];

  mssVinit(&v,STR);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
    return(v);
  }

  if(-1 == jd2ut(bufD,bufT,cal->arg[0].val[0].v.d)){
    MssVnull(v);
    return(v);
  }

  v.v.s=mssStrdup(bufD);
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * jd2time関数
 * 与え方： jd2time(ユリウス通日)
 * 「ユリウス通日」から時刻を計算しその値を返す。
 * ex. 2448422.5 -> 19910615
 */
static MssValue cal_jd2time(struct mssCal *cal)
{
  MssValue v;
  char bufD[9];
  char bufT[7];

  mssVinit(&v,STR);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
    return(v);
  }

  if(-1 == jd2ut(bufD,bufT,cal->arg[0].val[0].v.d)){
    MssVnull(v);
    return(v);
  }

  v.v.s=mssStrdup(bufT);
  setStrMalAdd(v.v.s);
  return(v);
}


/**
 * # FUNCTION #
 * leapyear関数
 * 与え方： leapyear(日付)
 * 「日付」の年が閏年であれば１を、そうでなければ０を返す。
 */
static MssValue cal_leapyear(struct mssCal *cal)
{
  MssValue v;

  char buf[5];
  char *v0;
  int year;

  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0=cal->arg[0].val[0].v.s;

  if(!validDate(v0)){
    MssVnull(v);
    return(v);
  }

  strncpy(buf,v0,4);
  buf[4]='\0';
  year=atoi(buf);

  v.v.d=(double)( (year%4==0 && year%100!= 0) || year%400==0 );
  return(v);
}

/**
 * # FUNCTION #
 * datefmt関数
 * 与え方： datefmt(日付,フォーマット)
 * 「日付」文字列をフォーマットに従い変換し、その値を返す。
 * 日付は８桁の文字列(ex.20030520)で与えられる。
 * フォーマットには任意の文字列を与えることができる。
 * ただし、フォーマットの中の空白文字類は全て"_"に変換される。
 * 以下の特殊記号列は、日付を特定の文字列に変換する。
 *
 * 記号	意味					例(20030514)	例(20030105)
 * %Y	年					2003		2003
 * %M	月					05		01
 * %m	英語月名				May		Jan
 * %D	日					20		05
 * %d	１月１日からの経過日数			123		4
 * %w	週番号(1月1日を第1週開始日とする)	20		1
 * %s	その週の開始日付			20030514	20030101
 * %W	曜日番号				2		0
 *      (0:日,1:月,2:火,3:水,4:木,5:金,6:土)
 *      曜日の算出にはツェラーの公式を用いている
 * %e	英語曜日(Sun,Mon,Tsu,Wed,Thu,Fri,Sat)	Tue		Sun
 * %E	曜日番号+英語曜日			2Tue		0Sun
 * %j	日本語曜日(日月火水木金土)		火		日
 * %J	曜日番号+日本語曜日			2火		0日
 * %%	%を出力					%		%
 *
 * 例)
 * datefmt("20030514","%Y年%M月%D日") -> 2003年05月14日
 */
static MssValue cal_datefmt(struct mssCal *cal)
{
  MssValue v;
  struct mssStrings *hs;
  struct tm *lct;
  time_t caltime;
  int   fmtlen;
  char buf[10];
  char date[9];
  char time[7];
  char *v0;
  unsigned char *v1;
  int dow;
  int i;
  int tmp;
  char *wday1[]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
  char *wday2[]=MssEMdow;
  char *emon[]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
  mssVinit(&v,STR);


  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0 =cal->arg[0].val[0].v.s;
  v1 =cal->arg[1].val[0].v.s;

  if(!validDate(v0)){
    MssVnull(v);
    return(v);
  }

  fmtlen=strlen(v1);
  if(fmtlen>=MssFieldMaxLen){
    MssVnull(v);
    return(v);
  }
  for(i=0; i<fmtlen; i++){
    if(isspace(*(v1+i))) *(v1+i)='_';
  }

  caltime = mssStrToTime(cal->arg[0].val[0].v.s);
  if(caltime == -1){
    MssVnull(v);
    return(v);
  }else{
    lct = localtime(&caltime);
  v.v.d = lct->tm_wday;
  } 

  i=0;
  hs=mssInitStrings(); /* 位置変更 2004/10/31 */
  while(*v1!='\0' && i<MssFieldMaxLen-4){
    if(*v1=='%'){
      switch(*(v1+1)){
        case 'Y':
          strncpy(buf,v0,4); buf[4]='\0';
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'M':
          strncpy(buf,v0+4,2); buf[2]='\0';
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'm':
          strncpy(buf,v0+4,2); buf[2]='\0';
          mssCatStrings(hs,emon[atoi(buf)-1]);
          v1=v1+2;
          break;
        case 'D':
          strncpy(buf,v0+6,2); buf[2]='\0';
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'd': /*1月1日からの経過日数 */
          sprintf(buf,"%d",day365(v0));
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'w': /*週番号*/
          sprintf(buf,"%d",day365(v0)/7);
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 's': /*週開始日付*/
          tmp=day365(v0)%7;
          jd2ut(date,time,ut2jd(v0,"000000")-(double)tmp);
          mssCatStrings(hs,date);
          v1=v1+2;
          break;
        case 'W': /*曜日番号*/
          dow=(int)dayweek(v0);
          sprintf(buf,"%d",dow);
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'e': /*英語曜日*/
          dow=(int)dayweek(v0);
          mssCatStrings(hs,wday1[dow]);
          v1=v1+2;
          break;
        case 'E': /*曜日番号+英語曜日*/
          dow=(int)dayweek(v0);
          sprintf(buf,"%d%s",dow,wday1[dow]);
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'j': /*日本語曜日*/
          dow=(int)dayweek(v0);
          mssCatStrings(hs,wday2[dow]);
          v1=v1+2;
          break;
        case 'J': /*曜日番号+日本語曜日 */
          dow=(int)dayweek(v0);
          sprintf(buf,"%d%s",dow,wday2[dow]);
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case '%': /*%*/
          buf[0]='%'; buf[1]='\0';
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        default:
          mssCatnStrings(hs,v1,1);
          v1++;
          break;
      }
    }else{
      mssCatnStrings(hs,v1,1);
      v1++;
    }
  }
  v.v.s=mssStrdup(hs->str);
  setStrMalAdd(v.v.s);
  mssFreeStrings(hs);
  return(v);
}

/**
 * # FUNCTION #
 * timefmt関数
 * 与え方： timefmt(時刻,フォーマット)
 * 「時刻」文字列をフォーマットに従い変換し、その値を返す。
 * 時刻は６桁の文字列(ex.150257)で与えられる。
 * フォーマットには任意の文字列を与えることができる。
 * ただし、フォーマットの中の空白文字類は全て"_"に変換される。
 * 以下の特殊記号列は、時刻を特定の文字列に変換する。
 *
 * 記号	意味			例(150257)	例(101101)
 * %H	時(24時間表記)		15		10
 * %h	時(12時間表記)		3		10
 * %e	AM or PM		PM		AM
 * %j	午前 or 午後		午後		午前
 * %M	分			02		11
 * %S	秒			57		01
 * %%	%を出力			%		%
 *
 * 例)
 * timefmt("101101","%H:%M:%S") -> 10:11:01
 * timefmt("150257","%j%h:%M:%S") -> 午後03:92:57
 */
static MssValue cal_timefmt(struct mssCal *cal)
{
  MssValue v;
  struct mssStrings *hs;
  int   fmtlen;
  char buf[20];
  char hour[3];
  char minute[3];
  char second[3];
  char *v0;
  unsigned char *v1;
  int i;
  mssVinit(&v,STR);


  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0 =cal->arg[0].val[0].v.s;
  v1 =cal->arg[1].val[0].v.s;

  if(!validTime(v0)){
    MssVnull(v);
    return(v);
  }

  fmtlen=strlen(v1);
  if(fmtlen>=MssFieldMaxLen){
    MssVnull(v);
    return(v);
  }
  for(i=0; i<fmtlen; i++){
    if(isspace(*(v1+i))) *(v1+i)='_';
  }

  strncpy(hour  ,v0  ,2); hour[2]  ='\0';
  strncpy(minute,v0+2,2); minute[2]='\0';
  strncpy(second,v0+4,2); second[2]='\0';
  i=0;
  hs=mssInitStrings(); /* 位置変更 2004/10/31 */
  while(*v1!='\0' && i<MssFieldMaxLen-2){
    if(*v1=='%'){
      switch(*(v1+1)){
        case 'H':
          mssCatStrings(hs,hour);
          v1=v1+2;
          break;
        case 'h':
          sprintf(buf,"%02d",(atoi(hour)-1)%12+1);
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'e':
          if(atoi(hour)<12) sprintf(buf,"AM");
          else              sprintf(buf,"PM");
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'j':
          if(atoi(hour)<12) sprintf(buf,"午前");
          else              sprintf(buf,"午後");
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        case 'M':
          mssCatStrings(hs,minute);
          v1=v1+2;
          break;
        case 'S':
          mssCatStrings(hs,second);
          v1=v1+2;
          break;
        case '%': /*%*/
          buf[0]='%'; buf[1]='\0';
          mssCatStrings(hs,buf);
          v1=v1+2;
          break;
        default:
          mssCatnStrings(hs,v1,1);
          v1++;
          break;
      }
    }else{
      mssCatnStrings(hs,v1,1);
      v1++;
    }
  }
  v.v.s=mssStrdup(hs->str);
  setStrMalAdd(v.v.s);
  mssFreeStrings(hs);
  return(v);
}

/**
 * # FUNCTION #
 * day関数
 * 与え方： day(日付1,日付2)
 * 「日付1」から「日付2」までの日数を返す。
 * 扱うことのできる日付は1582年10月15日〜9999年12月31日まで。
 */
static MssValue cal_day(struct mssCal *cal)
{
  MssValue v;

  char *v0;
  char *v1;
  double JD0;
  double JD1;

  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0 =cal->arg[0].val[0].v.s;
  v1 =cal->arg[1].val[0].v.s;

  if(!validDate(v0) || !validDate(v1)){
    MssVnull(v);
    return(v);
  }

  JD0=ut2jd(v0,"000000");
  JD1=ut2jd(v1,"000000");

  v.v.d = JD0-JD1;
  return(v);
}

/**
 * # FUNCTION #
 * time関数
 * 与え方： time(時刻1,時刻2)
 * 「時刻1」から「時刻2」までの秒数を返す。
 */
static MssValue cal_time(struct mssCal *cal)
{
  MssValue v;

  time_t    fSec;
  time_t    tSec;
  struct tm fTm;
  struct tm tTm;

  char fBuf[20];
  char tBuf[20];
  char *v0;
  char *v1;
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0 =cal->arg[0].val[0].v.s;
  v1 =cal->arg[1].val[0].v.s;

  if(!validTime(v0) || !validTime(v1)){
    MssVnull(v);
    return(v);
  }

  fTm.tm_year =-1900+2001 ; tTm.tm_year =-1900+2001;
  fTm.tm_mon  =-1+1       ; tTm.tm_mon  =-1+1;
  fTm.tm_mday =1          ; tTm.tm_mday =1;
  fTm.tm_isdst=-1         ; tTm.tm_isdst=-1;

  strncpy(fBuf,v0,2); fBuf[2]='\0';
  fTm.tm_hour=atoi(fBuf);

  strncpy(tBuf,v1,2); tBuf[2]='\0';
  tTm.tm_hour=atoi(tBuf);

  strncpy(fBuf,v0+2,2); fBuf[2]='\0';
  fTm.tm_min=atoi(fBuf);

  strncpy(tBuf,v1+2,2); tBuf[2]='\0';
  tTm.tm_min=atoi(tBuf);

  strncpy(fBuf,v0+4,2); fBuf[2]='\0';
  fTm.tm_sec=atoi(fBuf);

  strncpy(tBuf,v1+4,2); tBuf[2]='\0';
  tTm.tm_sec=atoi(tBuf);

  fSec=mktime(&fTm);
  tSec=mktime(&tTm);
  if( fSec==(time_t)-1 || tSec==(time_t)-1 ){
    MssVnull(v);
    return(v);
  }
  v.v.d = (double)difftime(fSec,tSec);
  return(v);
}

/**
 * # FUNCTION #
 * today()
 * 現在の日付を返す。
 */
static MssValue cal_today(struct mssCal *cal)
{
  MssValue v;
  struct tm *tp;
  long Tim;
  mssVinit(&v,STR);

  time(&Tim);
  tp=localtime(&Tim);
  v.v.s=mssMalloc(9,"cal_today");
  sprintf(v.v.s,"%04d%02d%02d",tp->tm_year+1900,tp->tm_mon+1,tp->tm_mday);
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * now関数
 * 与え方： now()
 * 現在の時刻を返す。
 */
static MssValue cal_now(struct mssCal *cal)
{
  MssValue v;
  struct tm *tp;
  long Tim;
  mssVinit(&v,STR);

  time(&Tim);
  tp=localtime(&Tim);
  v.v.s=mssMalloc(7,"cal_now");
  sprintf(v.v.s,"%02d%02d%02d",tp->tm_hour,tp->tm_min,tp->tm_sec);
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * age関数
 * 与え方： age(日付1,日付2)
 * 年齢を返す。言い替えれば、日付1から日付2までの経過年数を返す。
 */
static MssValue cal_age(struct mssCal *cal)
{
  MssValue v;
  int age;
  int y1,y2,m1,m2,d1,d2;
  char *v0;
  char *v1;
  char buf[20];
  char buf2[20];
  mssVinit(&v,DBL);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  v0 =cal->arg[0].val[0].v.s;
  v1 =cal->arg[1].val[0].v.s;

  if(!validDate(v0) || !validDate(v1)){
    MssVnull(v);
    return(v);
  }

  strncpy(buf,v0,4);
  buf[4]='\0';
  y1 = atoi(buf);

  strncpy(buf2,v1,4);
  buf2[4]='\0';
  y2 = atoi(buf2);

  strncpy(buf,v0+4,2);
  buf[2]='\0';
  m1 = atoi(buf);

  strncpy(buf2,v1+4,2);
  buf2[2]='\0';
  m2 = atoi(buf2);

  strncpy(buf,v0+6,2);
  buf[2]='\0';
  d1 = atoi(buf);

  strncpy(buf2,v1+6,2);
  buf2[2]='\0';
  d2 = atoi(buf2);

  age = y1 - y2;
  if(m1 < m2){
    age--;
  }else{    
    if(m1 == m2){
      if(d1 < d2){
        age--;
      }
    }
  }
  v.v.d = (double)age;
  return(v);
}

/**
 * # FUNCTION #
 * dayadd関数
 * 与え方： dayadd(日付,日数)
 * 「日付」に「日数」を加えた日付を返す。
 */
static MssValue cal_dayadd(struct mssCal *cal)
{

  MssValue v;
  char bufD[10];
  char bufT[10];
  char *v0;
  double jd;
  mssVinit(&v,STR);

  if(cal->arg[0].val[0].nul || cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }

  v0 =cal->arg[0].val[0].v.s;
  if(!validDate(v0)){
    MssVnull(v);
    return(v);
  }

  jd=ut2jd(v0,"000000");
  jd+=cal->arg[1].val[0].v.d;
  jd2ut(bufD,bufT,jd);
  v.v.s=mssStrdup(bufD);
  setStrMalAdd(v.v.s);
  return(v);
}

/**
 * # FUNCTION #
 * timeadd関数
 * 与え方： timeadd(時刻,秒数)
 * 「時刻」に秒数を加えた時刻を返す。
 */
static MssValue cal_timeadd(struct mssCal *cal)
{

  MssValue v;
  char buf[3];
  char *vt;
  struct tm time;
  struct tm *timeR;
  time_t t;
  mssVinit(&v,STR);

  if(cal->arg[0].val[0].nul||cal->arg[1].val[0].nul){
    MssVnull(v);
    return(v);
  }
  vt =cal->arg[0].val[0].v.s;

  if(!validTime(vt)){
    MssVnull(v);
    return(v);
  }

  time.tm_year = -1900+2001;
  time.tm_mon  = -1+1;
  time.tm_mday = 1;

  /*時*/
  strncpy(buf,vt+0,2); buf[2]='\0';
  time.tm_hour = atoi(buf);

  /*分*/
  strncpy(buf,vt+2,2); buf[2]='\0';
  time.tm_min  = atoi(buf);

  /*秒*/
  strncpy(buf,vt+4,2); buf[2]='\0';
  time.tm_sec  = atoi(buf);

  /*季節調整無効*/
  time.tm_isdst=-1;

  /*暦時間計算*/
  t=mktime(&time);

  t += (int)cal->arg[1].val[0].v.d;
  timeR=localtime(&t);

  v.v.s= mssMalloc(sizeof(char)*16,"timeadd");
  sprintf(v.v.s,"%02d%02d%02d", timeR->tm_hour, timeR->tm_min, timeR->tm_sec);
  setStrMalAdd(v.v.s);
  return(v);
}


/**
 * # FUNCTION #
 * line関数
 * 与え方： line()
 * 現在処理中の行番号を返す。
 */
static MssValue cal_line(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);
  v.v.d = (double)mssGV.inCnt;
  return(v);
}

/**
 * # FUNCTION #
 * keyLine関数
 * 与え方： keyLine()
 * 現在処理中のキー内行番号を返す。
 * キー内行番号とは、キー項目の値が同じ行の中での行番号のことで、異るキー値が
 * 始まると、再び１にもどる。
 * この関数は"-k"パラメータでキー項目を指定しなければならない。
 * また、同じキー値での並びを考慮したければ、"-s"パラメータを利用する。
 */
static MssValue cal_keyLine(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);
  v.v.d = (double)mssGV.keyLine;
  return(v);
}

/**
 * # FUNCTION #
 * keyCnt関数
 * 与え方： keyCnt()
 * 現在処理中のキーの行数を返す。
 * この関数は"-k"パラメータでキー項目を指定しなければならない。
 */
static MssValue cal_keyCnt(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);
  v.v.d = (double)mssGV.keyCnt;
  return(v);
}

/**
 * # FUNCTION #
 * keyNo関数
 * 与え方： keyNo()
 * 現在処理中のキー番号を返す。
 * キー番号は、異るキー値が出現するたびにインクリメントされる。
 * この関数は"-k"パラメータでキー項目を指定しなければならない。
 */
static MssValue cal_keyNo(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);
  v.v.d = (double)mssGV.keyNo;
  return(v);
}

/**
 * # FUNCTION #
 * fldCnt関数
 * 与え方： fldCnt()
 * 現在処理注の項目数を返す。
 * 全ての行において、同じ値を返すことになる。
 */
static MssValue cal_fldCnt(struct mssCal *cal)
{
  MssValue v;
  mssVinit(&v,DBL);
  v.v.d = (double)mssGV.inFldCnt;
  return(v);
}

/**
 * # FUNCTION #
 * prvResult関数
 * 与え方： prvResult()
 * 現在処理中の行の一行前の結果を返す(文字列としてでなくMssValueとして)
 * mssReadFRKが読み込み関数の場合: 同一キーにおける先頭行が現在の場合はNULL
 * mssReadFldRec,mssReadFRDが読み込み関数の場合: 先頭行の場合はNULL
 */
static MssValue cal_prvResult(struct mssCal *cal)
{
  MssValue v;

  v=PrvRsl.val;
  if(v.vType==STR && !v.nul){
    v.v.s=mssStrdup(PrvRsl.val.v.s);
    setStrMalAdd(v.v.s);
  }

  return(v);
}

/**
 * # FUNCTION #
 * topResult関数
 * 与え方： topResult()
 * 先頭行の結果を返す(文字列としてでなくMssValueとして)
 * mssReadFRKが読み込み関数の場合: 同一キーにおける先頭行の結果
 * mssReadFldRec,mssReadFRDが読み込み関数の場合: データの一行目の結果
 */
static MssValue cal_topResult(struct mssCal *cal)
{
  MssValue v;

  v=TopRsl.val;
  if(v.vType==STR && !v.nul){
    v.v.s=mssStrdup(TopRsl.val.v.s);
    setStrMalAdd(v.v.s);
  }
  return(v);
}

/**
 * # FUNCTION #
 * prvField関数
 * 与え方： prvField(項目名)
 * 現在処理中の行の一行前の指定の「項目名」の値を返す。
 * (文字列としてでなくMssValueとして)
 * mssReadFRKが読み込み関数の場合: 同一キーにおける先頭行が現在の場合はNULL
 * mssReadFldRec,mssReadFRDが読み込み関数の場合: 先頭行の場合はNULL
 */
static MssValue cal_prvField(struct mssCal *cal)
{
  MssValue v;
  char *str;

  mssVinit(&v,STR);
  if(PrvFld.pnt[0]==NULL){
    MssVnull(v);
  }else{
    str=PrvFld.pnt[(*(cal->arg[0].flds->fi))->num];
    if(*str==MssNullChr){
      MssVnull(v);
    }else{
      v.v.s=mssStrdup(str);
      setStrMalAdd(v.v.s);
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * topField関数
 * 与え方： topField(項目名)
 * 先頭行の指定の「項目名」の値を返す(文字列としてでなくMssValueとして)
 * mssReadFRKが読み込み関数の場合: 同一キーにおける先頭行の結果
 * mssReadFldRec,mssReadFRDが読み込み関数の場合: データの一行目の結果
 */
static MssValue cal_topField(struct mssCal *cal)
{
  MssValue v;
  char *str;

  mssVinit(&v,STR);
  if(TopFld.pnt[0]==NULL){
    MssVnull(v);
  }else{
    str=TopFld.pnt[(*(cal->arg[0].flds->fi))->num];
    if(*str==MssNullChr){
      MssVnull(v);
    }else{
      v.v.s=mssStrdup(str);
      setStrMalAdd(v.v.s);
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * lineCmp関数
 * 与え方： lineCmp(項目名1,項目名2,...,項目名n)
 * 指定されたn個の項目について、現在処理中の行と一行前の行の値を比較し、
 * 全て同じなら0を返す。現在行<前行なら-1,現在行>前行なら1を返す。
 * この関数は読み込み関数の種類に関係なく同じ動作をする。
 * 比較する項目にNULL値が含まれていればNULL値を返す。
 * 先頭行は、現在行>前行として扱う(すなわち１を返す)。
 */
static MssValue cal_lineCmp(struct mssCal *cal)
{

  int i,j;
  int cmp;
  int fldNo;
  MssValue v;

  mssVinit(&v,DBL);

  v.v.d=0;

  /*最初行の時は異なる行として返す*/
  if(mssGV.inCnt==1){
    for(i=0; i<cal->argCnt; i++){
      for(j=0; j<cal->arg[i].valCnt; j++){
        if(cal->arg[i].val[j].nul){
          MssVnull(v);
          return(v);
        }
      }
    }
    v.v.d=1;
    return(v);
  }

  for(i=0; i<cal->argCnt; i++){
    for(j=0; j<cal->arg[i].valCnt; j++){
      fldNo=MssFlds2num(cal->arg[i].flds,i);
      if(cal->arg[i].val[j].nul || MssIsNull(PrvFld.pnt[fldNo]) ){
        MssVnull(v);
        return(v);
      }
      cmp=strcmp(cal->arg[i].val[j].v.s,PrvFld.pnt[fldNo]);
      if(cmp<0) {
        v.v.d=-1;
        return(v);
      }
      if(cmp>0) {
        v.v.d=1;
        return(v);
      }
    }
  }
  v.v.d=0;
  return(v);
}

/**
 * 利用可能関数の設定
 * 引数の最大個数はMaxCalArgで定義
 * name : 関数名もしくは演算記号
 * type : 0:関数 1:演算子
 * priority : 演算子の演算優先順位(数字が低い方が優先順位が高い)
 *            関数の優先順位は最も高いので、常に"1"とする。
 * argType: 引数の型 N:数値,S:文字列,X:不定,M:前引数型の0個以上の繰り返し
 * retType: 返す値の型 N:数値, S:文字列, X:不定
 * procType : 利用可能なデータの読み込みタイプ
 *            現在のところ、readFldRec,readFRD,readFRKの三つのみ対応している。
 *            ３ビット列の右からreadFldRec,readFRD,readFRKの順で、
 *            対応する箇所にビットをたてる。
 *            1 : readFldRec
 *            2 : readFRD
 *            4 : readFRK
 *            もし、readFRKの読み込み関数を利用したときのみ利用可能にしたければ
 *            procTypeは4とする。全ての読み込み関数で利用可能であれば7(1+2+4)
 *            とする。通常、一行単位の処理のみを行うのであれば7となる。
 * functionAddress: 実際の関数アドレス
 * priorityは演算子のみ影響する。関数の優先順位もとりあえずここで定義しているが
 *
 * 注1) "-"演算子と"-eq"演算子。
 *   コマンドの実行時に指定された計算式の演算子を探す際には
 *   以下のリストに出現順序で文字列マッチングをしている。もし、"-"演算子
 *   が"-eq"演算子より先に登録されていれば、たとえ引数で"-eq"を指定していても
 *   "-"演算子としてマッチングされてしまう。そこで、以下のリスト登録時には
 *   同じ先頭文字列を持つ演算子については、全体の文字列長が長い
 *   方を先に登録すること。
 *   また、関数に関してはそのようなことを気づかう必要なし。
 *
 * 注2) 最後のdummyは、終端の検出のため。実際にはtype==-1で判断している。
 * 注3) 演算子の演算優先順位の値(priority)は以下の文献に基づく。
 *      平林雅秀「ANCI C/C++事典」,共立出版,1996,p.708.
 */
struct Func func[]={
/* name         ,type
                  ,priority
                     ,argType[],retType
                                    ,procType
                                       ,functionAddress*/
  {"*"          ,1, 4,{N,N,E}  ,N   ,7 ,cal_mulOperator},
  {"/"          ,1, 4,{N,N,E}  ,N   ,7 ,cal_divOperator},
  {"%"          ,1, 4,{N,N,E}  ,N   ,7 ,cal_modOperator},
  {"<>"         ,1, 7,{N,N,E}  ,N   ,7 ,cal_neNOperator},
  {"<="         ,1, 7,{N,N,E}  ,N   ,7 ,cal_leNOperator},
  {"<"          ,1, 7,{N,N,E}  ,N   ,7 ,cal_ltNOperator},
  {">="         ,1, 7,{N,N,E}  ,N   ,7 ,cal_geNOperator},
  {">"          ,1, 7,{N,N,E}  ,N   ,7 ,cal_gtNOperator},
  {"=="         ,1, 7,{N,N,E}  ,N   ,7 ,cal_eqNOperator},
  {"-le"        ,1, 7,{S,S,E}  ,N   ,7 ,cal_leSOperator},
  {"-lt"        ,1, 7,{S,S,E}  ,N   ,7 ,cal_ltSOperator},
  {"-ge"        ,1, 7,{S,S,E}  ,N   ,7 ,cal_geSOperator},
  {"-gt"        ,1, 7,{S,S,E}  ,N   ,7 ,cal_gtSOperator},
  {"-ne"        ,1, 7,{S,S,E}  ,N   ,7 ,cal_neSOperator},
  {"-eq"        ,1, 7,{S,S,E}  ,N   ,7 ,cal_eqSOperator},
  {"=~"         ,1, 7,{S,S,E}  ,N   ,7 ,cal_regOperator},
  {"+"          ,1, 5,{N,N,E}  ,N   ,7 ,cal_addOperator},
  {"-"          ,1, 5,{N,N,E}  ,N   ,7 ,cal_subOperator},
  {"||"         ,1,13,{N,N,E}  ,N   ,7 ,cal_orOperator},
  {"&&"         ,1,12,{N,N,E}  ,N   ,7 ,cal_andOperator},

  {"age"        ,0, 1,{S,S,E}  ,N   ,7 ,cal_age},
  {"datefmt"    ,0, 1,{S,S,E}  ,S   ,7 ,cal_datefmt},
  {"dayadd"     ,0, 1,{S,N,E}  ,S   ,7 ,cal_dayadd},
  {"day"        ,0, 1,{S,S,E}  ,N   ,7 ,cal_day},
  {"timefmt"    ,0, 1,{S,S,E}  ,S   ,7 ,cal_timefmt},
  {"timeadd"    ,0, 1,{S,N,E}  ,S   ,7 ,cal_timeadd},
  {"time"       ,0, 1,{S,S,E}  ,N   ,7 ,cal_time},
  {"now"        ,0, 1,{E}      ,S   ,7 ,cal_now},
  {"today"      ,0, 1,{E}      ,S   ,7 ,cal_today},
  {"dt2jd"      ,0, 1,{S,S,E}  ,N   ,7 ,cal_dt2jd},
  {"jd2date"    ,0, 1,{N,E}    ,S   ,7 ,cal_jd2date},
  {"jd2time"    ,0, 1,{N,E}    ,S   ,7 ,cal_jd2time},
  {"leapyear"   ,0, 1,{S,E}    ,N   ,7 ,cal_leapyear},

  {"uppercase"  ,0, 1,{S,E}    ,S   ,7 ,cal_uppercase},
  {"lowercase"  ,0, 1,{S,E}    ,S   ,7 ,cal_lowercase},
  {"capitalize" ,0, 1,{S,E}    ,S   ,7 ,cal_capitalize},

  {"sum"        ,0, 1,{N,M,E}  ,N   ,7 ,cal_sum},
  {"min"        ,0, 1,{N,M,E}  ,N   ,7 ,cal_min},
  {"max"        ,0, 1,{N,M,E}  ,N   ,7 ,cal_max},
  {"sqrt"       ,0, 1,{N,E}    ,N   ,7 ,cal_Sqrt},
  {"power"      ,0, 1,{N,N,E}  ,N   ,7 ,cal_power},
  {"abs"        ,0, 1,{N,E}    ,N   ,7 ,cal_abs},
  {"up"         ,0, 1,{N,N,E}  ,N   ,7 ,cal_up},
  {"down"       ,0, 1,{N,N,E}  ,N   ,7 ,cal_down},
  {"round"      ,0, 1,{N,N,E}  ,N   ,7 ,cal_round},
  {"fixdec"     ,0, 1,{N,N,E}  ,S   ,7 ,cal_fixdec},
  {"pi"         ,0, 1,{E}      ,N   ,7 ,cal_pi},
  {"exp"        ,0, 1,{N,E}    ,N   ,7 ,cal_Exp},
  {"ln"         ,0, 1,{N,E}    ,N   ,7 ,cal_ln},
  {"log10"      ,0, 1,{N,E}    ,N   ,7 ,cal_log10},
  {"log2"       ,0, 1,{N,E}    ,N   ,7 ,cal_log2},
  {"log"        ,0, 1,{N,N,E}  ,N   ,7 ,cal_log},

  {"rand01"     ,0, 1,{N,E}    ,N   ,7 ,cal_rand01},
  {"rand"       ,0, 1,{N,N,N,E},N   ,7 ,cal_rand},
  {"nrand"      ,0, 1,{N,N,N,E},N   ,7 ,cal_nrand},
  {"prand"      ,0, 1,{N,N,E}  ,N   ,7 ,cal_prand},

  {"cat"        ,0, 1,{S,M,S,E},S   ,7 ,cal_cat},
  {"length"     ,0, 1,{S,E}    ,N   ,7 ,cal_length},
  {"fix"        ,0, 1,{S,N,S,S,E},S ,7 ,cal_fix},
  {"right"      ,0, 1,{S,N,E}  ,S   ,7 ,cal_right},
  {"left"       ,0, 1,{S,N,E}  ,S   ,7 ,cal_left},
  {"substr"     ,0, 1,{S,N,N,E},S   ,7 ,cal_substr},
  {"regexStart" ,0, 1,{S,S,E}  ,N   ,7 ,cal_regexStart},
  {"regexEnd"   ,0, 1,{S,S,E}  ,N   ,7 ,cal_regexEnd},
  {"regexLen"   ,0, 1,{S,S,E}  ,N   ,7 ,cal_regexLen},
  {"isIn"       ,0, 1,{S,S,M,E},N   ,7 ,cal_isIn},
  {"isNull"     ,0, 1,{S,E}    ,N   ,7 ,cal_isNull},

  {"not"        ,0, 1,{N,E}    ,N   ,7 ,cal_not},
  {"if"         ,0, 1,{N,X,X,E},X   ,7 ,cal_if},
  {"isnull"     ,0, 1,{S,E}    ,N   ,7 ,cal_not},

  {"lineCmp"    ,0, 1,{S,M,E}  ,N   ,7 ,cal_lineCmp},

  {"prvField"   ,0, 1,{N,E}    ,S   ,7 ,cal_prvField},
  {"topField"   ,0, 1,{N,E}    ,S   ,7 ,cal_topField},
  {"prvResult"  ,0, 1,{E}      ,X   ,7 ,cal_prvResult},
  {"topResult"  ,0, 1,{E}      ,X   ,7 ,cal_topResult},
  {"keyLine"    ,0, 1,{E}      ,N   ,4 ,cal_keyLine},
  {"keyCnt"     ,0, 1,{E}      ,N   ,4 ,cal_keyCnt},
  {"keyNo"      ,0, 1,{E}      ,N   ,4 ,cal_keyNo},
  {"line"       ,0, 1,{E}      ,N   ,7 ,cal_line},
  {"fldCnt"     ,0, 1,{E}      ,N   ,7 ,cal_fldCnt},
  {"dummy"      ,-1,-1,{E}     ,E   ,0 ,NULL}
};

/**
 * # FUNCTION #
 * Func構造体のargType[]をチェックし、argCnt,argMul,mulBcnt,mulAcnt変数を
 * セットする。
 * 可変引数を表すMは一つしかないことが前提である。
 * argCnt : 最低限必要な引数の数
 * argMul : 可変引数を含むかどうか
 * mulBcnt: 可変引数の前に何個引数があるか
 * mulAcnt: 可変引数の後に何個引数があるか
 *
 * 例えば、argType[]={S,N,S,N,S,M,S,S,S} の時(可変引数を含むとき)
 * argCnt =8
 * argMul =1
 * mulBcnt=4
 * mulAcnt=3
 *
 * 例えば、argType[]={S,N,S} の時(可変引数を含まないとき)
 * argCnt =3
 * argMul =0
 * mulBcnt=0
 * mulAcnt=0
 */
static void initCalFunc(void)
{
  int i,j;

  /*初期化*/
  i=0;
  while(func[i].type!=-1){
    func[i].argCnt =0;
    func[i].argMul =0;
    func[i].mulBcnt=0;
    func[i].mulAcnt=0;
    i++;
  }

  i=0;
  while(func[i].type!=-1){
    j=0;
    while(func[i].argType[j]!=E){
      if(func[i].argType[j]==M){
        func[i].argMul  =1;
        func[i].mulBcnt =j-1;
      }
      j++;
    }

    /*func->argCntは、最低限必要な引数の数 */
    if(func[i].argMul) func[i].argCnt=j-1;
    else               func[i].argCnt=j;

    if(func[i].argMul==1){
      func[i].mulAcnt=j-func[i].mulBcnt-2;
    }
    i++;
  }
}

/**
 * # FUNCTION #
 * 与えられた文字(文字列)が演算子かどうかを判断する。
 * 登録された演算子記号にマッチすれば真(1)、そうでなければ偽(0)を返す。
 */
static int cal_isOpe(char *str)
{
  int i;

  i=0;
  /*演算子のチェック*/
  while(func[i].type!=-1){ /*func配列の終端チェック*/
    if(func[i].type==1){   /*演算子ならば*/
      if( 0==strncmp(func[i].name,str,strlen(func[i].name)) )
        return(1);
    }
    i++;
  }
  return(0);
}

/**
 * # FUNCTION #
 * 与えられた先頭文字列が登録された関数名かどうかを判断する。
 * 括弧以外の文字列+"("で関数と判断する(ex. "sum(" )。
 * 関数と判断されても、登録されていない関数であればエラー終了する。
 * 関数であれば真(1)、そうでなければ偽(0)を返す。
 */
static int cal_isFunc(char *str)
{
  int i;
  int len;
  char *parPos; /*括弧の位置*/
  char *orgStr;
  orgStr=str;

  /*最初の一文字が括弧ということは、それは関数でない*/
  if(*str=='(')return(0);

  /*strを走査し'('が出てきたら関数型である。演算子が出て来れば関数でない。*/
  while(1){
    if(*str=='('){
      parPos=str;
      break;
    }
    if(cal_isOpe(str)) return(0);
    if(*str=='\0')     return(0);
    str++;
  }

  /*しかし$(の時に限って関数でない*/
  if(*(str-1)=='$') return(0);

  /*関数のチェック*/
  str=orgStr;
  i=0;
  while(func[i].type!=-1){ /*func配列の終端チェック*/
    if(func[i].type==0){   /*関数ならば*/
      len=strlen(func[i].name);
      if( 0==strncmp(func[i].name,str,len) ){
        if(*(str+len)=='(') return(1);
      }
    }
    i++;
  }
  *parPos='\0';
  mssShowErrMsg("I don't know such a function : %s",str);
  mssEnd(mssErrorNoDefault);
  return(0);
}

/**
 * # FUNCTION #
 * strで与えられた名前の関数、演算子のC関数アドレスを返す。
 * 登録がなければエラー終了
 */
static struct Func *cal_setFuncAdd(char *str)
{
  int len;
  int i=0;

  while(1){

    /*関数リストを最後まで読みきったらエラー*/
    if(func[i].type==-1)break;

    len=strlen(func[i].name);
    if(0==strncmp(str,func[i].name,len)){
      return(&func[i]);
    }
    i++;
  }
  mssShowErrMsg("I don't know the function:%s\n",str);
  mssEnd(mssErrorNoDefault);
  return(0);
}

/**
 * # FUNCTION #
 * 文字列としての計算式のチェックおよび修正
 * チェック事項は、以下のとおり。
 *   1) 括弧の対応関係(括弧の数)が正しいか
 *
 * 修正事項は、以下のとおり。
 *   1) 不必要なスペースもしくは改行の除去
 * 
 */
static char *cal_chkCal(char *str)
{
  int i;
  int parCnt; /*括弧を飛ばすときに括弧のレベルをカウントする変数*/
  char *buf;  /*チェック後の文字列*/
  char *bufTop;

  /*括弧の数をチェック*/
  parCnt=0;
  for(i=0; i<strlen(str); i++){
    if(*(str+i)=='(') parCnt++;
    if(*(str+i)==')') parCnt--;
  }
  if(parCnt!=0){
    mssShowErrMsg("mismatch of '(' and ')'\n");
    mssEnd(mssErrorNoDefault);
  }

  /* 不必要なスペースの除去*/
  buf=mssMalloc(sizeof(char)*strlen(str)+1,"chkCal");
  bufTop=buf;
  while(*str!='\0'){
    /* '"'が出現したら次の'"'まで無条件でコピーする*/
    if(*str=='"'){
      *buf=*str;
       buf++; str++;
      while(*str!='"' && *str!='\0'){
        if(*str!='\n'){
          *buf=*str;
          buf++; str++;
        }
      }
      if(*str=='\0') {
        mssShowErrMsg("mismatch of \"\"");
        mssEnd(mssErrorNoDefault);
      }
    }

    /* '$('が出現したら次の')'まで無条件でコピーする*/
    if(*str=='$' && *(str+1)=='('){
      *buf=*str; buf++; str++;
      *buf=*str; buf++; str++;
      while(*str!=')' && *str!='\0'){
        if(*str!='\n'){
          *buf=*str;
          buf++; str++;
        }
      }
      if(*str=='\0') {
        mssShowErrMsg("mismatch of '(' and ')'");
        mssEnd(mssErrorNoDefault);
      }
    }

    /*スペースもしくは改行は詰める*/
    /*if(*str!=' ' && *str!='\n'){*/
    if(!isspace((unsigned char)*str)){
      *buf=*str;
       buf++;
    }
    str++;
  }
  *buf='\0';
/*printf("after removing redundant spaces : %s\n",bufTop);*/
  return(bufTop);
}


/**
 * # FUNCTION #
 * 演算子もしくは関数のシンボル(str)およびそのアドレスをopeLstに登録する。
 */
static char *cal_setOpeTok(struct OpeLst *opeLst, char *str)
{

  opeLst->tbl=mssRealloc(opeLst->tbl, sizeof(struct OpeTbl)*(opeLst->cnt+1),"OpeTok");
  (opeLst->tbl+opeLst->cnt)->add=str;
  (opeLst->tbl+opeLst->cnt)->num=opeLst->cnt;
  (opeLst->tbl+opeLst->cnt)->func=cal_setFuncAdd(str);
  str+=strlen((opeLst->tbl+opeLst->cnt)->func->name);
  opeLst->cnt++;
  return(str);
}

/**
 * # FUNCTION #
 * strで与えられた計算式の、演算子もしくは関数のアドレスへのリストを作成する。
 * 括弧で囲まれた中の演算子もしくは関数は、対象外とする。
 * 例えば、str="($a+$b)/10"の時、括弧内の"+"は対象とされない。
 * また、関数の引数に現れる演算子や関数も、対象外とする。
 * str="sum($a,$b,sqrt($c),$d+$e)"の時、"sqrt($c)","$d+$e"は評価されない。
 * 関数はstr文字列に一切手を加えない
 *  $a+$b/$c
 *    +  +--- (opeLst->opeTbl+1)->add
 *    |       (opeLst->opeTbl+1)->func
 *   (opeLst->opeTbl+0)->add : アドレス
 *   (opeLst->opeTbl+0)->func : 関数へのポインタ
 * opeLst->cnt = 2 "+"と"/"で2つの演算子を使っているという意味
 */
static void cal_sepOpeTok( struct OpeLst *opeLst, char *str)
{
  int opeCnt; /*演算子の数*/
  char *orgStr;
  char *tmpStr;
  int i;
  opeLst->cnt=0;
  opeLst->tbl=NULL;

  orgStr=str;
  /*まず演算子をさがす*/
  while(*str!='\0'){

    if(*str=='(') {
      str=mssSkipStrBlock(str,'(',')'); /*括弧を飛ばす*/
      continue;
    }

    if(*str=='"') {
      str=mssSkipStrBlock(str,'"','"'); /*"を飛ばす*/
      if(str==NULL) {
        mssShowErrMsg("mismatch of \"\"");
        mssEnd(mssErrorNoDefault);
      }
      continue;
    }
    if(cal_isOpe(str)){
      str=cal_setOpeTok(opeLst,str); /*opeLstに登録&strを演算子長分増す*/
    }else{
      str++;
    }
  }

  /*次に関数をさがす*/
  /*関数は先頭かもしくは演算子の次のみ*/
  str=orgStr;
  opeCnt=opeLst->cnt;

  if(cal_isFunc(str)){
    str=cal_setOpeTok(opeLst,str); /*opeLstに登録&strを演算子長分増す*/
  }

  for(i=0; i<opeCnt; i++){
    tmpStr=(opeLst->tbl+i)->add+strlen((opeLst->tbl+i)->func->name);
    if( cal_isFunc(tmpStr) ){
      str=cal_setOpeTok(opeLst,tmpStr); /*opeLstに登録&strを演算子長分増す*/
    }
  }
}

/**
 * # FUNCTION #
 * 外側の括弧を外す (a+b) -> a+b , ((a+b)) -> a+b
 * 新たに文字列領域を確保することなく、括弧を外した後の文字列先頭アドレスを
 * 返す。
 */
static char *removePar(char *str)
{

  char *orgStr=str;

  if(*str!='(')return(str);
  str=mssSkipStrBlock(str,'(',')');
  if(*str=='\0'){
    *orgStr='\0';
    *(str-1)='\0';
    orgStr=removePar(++orgStr);
  }
  return(orgStr);
}

/**
 * # FUNCTION #
 * 優先順位の最も低い演算子もしくは関数を見付ける。
 * 優先順位が同じなら出現順序が後の演算子もしくは関数
 */
static struct OpeTbl *getLowPriOpe(struct OpeLst *opeLst)
{
  int minPri=-1;
  int minNum=0;
  int i;

  for(i=0; i<opeLst->cnt; i++){
    if(minPri<=(opeLst->tbl+i)->func->priority){  /*優先順位が低ければ*/
      minPri=(opeLst->tbl+i)->func->priority;
      minNum=i;
    }
  }
  return(opeLst->tbl+minNum);
}

/**
 * # FUNCTION #
 * 数式を評価し、優先順位に従って、cal構造体のツリーを構築する
 * 構築の際には、一つの演算子もしくは関数ごとに再帰呼び出しにて。
 * ツリーのレベルの深いノードが優先順位の高い演算もしくは関数である。
 * 関数もしくは演算子が一つもなければNULLを返す。
 *
 * 例1)
 * str="$val+10/$unit"
 * cal->func=addOperator
 *    ->argCnt=2
 *    ->arg[0].str="$val"
 *    ->arg[1].str="10/$unit"
 *    ->arg[1].node->func=divOperator
 *                 ->argCnt=2
 *                 ->arrg[0].str="10"
 *                 ->arrg[1].str="$unit"
 *
 * 例2)
 * str="($val+10)/sum($a,$b,$c)"
 * cal->func=divOperator
 *    ->argCnt=2
 *    ->arg[0].str="$val+10"
 *    ->arg[0].node->func=addOperator
 *                 ->argCnt=2
 *                 ->arrg[0].str="$val"
 *                 ->arrg[1].str="10"
 *    ->arg[1].str="sum($a,$b,$c)"
 *    ->arg[1].node->func=divOperator
 *                 ->argCnt=3
 *                 ->arrg[0].str="$a"
 *                 ->arrg[1].str="$b"
 *                 ->arrg[2].str="$c"
 *
 * またこの関数では、計算式の関数の引数の数のチェック、また二項演算子であれば、
 * 左右に値があるかどうかのチェックも行う。
 */
static struct mssCal *cal_sepCal(char *str, struct mssCal *parent)
{

  struct OpeLst opeLst;
  struct mssCal    *cal;
  struct mssCal    *argCal;
  struct OpeTbl *ope;
  char          *tmpPos;
  char          *tmpStr;
  char          *prePos;
  char           buf[1024];
  int            i;

  /*外側の括弧は外す (a+b) -> a+b*/
  str=removePar(str);

  /*現在の文字列に関して、並列な演算子を探索し、それらをopeLst構造体にセット*/
  cal_sepOpeTok(&opeLst,str);

  /*演算子もしくは関数が無ければNULLでリターン*/
  if(opeLst.cnt==0) return(NULL);

  /*優先順位の最も低い演算子を見付ける*/
  ope=getLowPriOpe(&opeLst);
/*printf("ope=%s %d\n",ope->add,ope->num);*/

  /*メモリ確保+親ノードとの結合*/
  cal=mssCalloc(sizeof(struct mssCal),"sepCal");
  cal->parent=parent; /*親ノード*/

  /*最も低い優先順位の演算子によって文字列を2つに分ける*/
  /* (a+b)/c  ->  (a+b)'\0'c*/
  /*              +     +*/
  /*              |     |*/
  /*              str   ope->add*/

  /*&&& 関数の場合
    関数の引数をセット
     X='\0'とすると
      sum(abc,(a+b),sum(a+b,c))
     --->>>
      sum(abcX(a+b)Xsum(a+b,c)X
      |   |   |     +arg[2]
      |   +   +arg[1]
      +   arg[0]
      ope->add or str
                  cal->argCnt=3 */
  if( ope->func->type == 0 ){
    cal->func=ope->func; /*関数のアドレスをセット*/

    /*関数名(をスキップ*/
    tmpPos = ope->add;
    while( *(tmpPos++)!='(' );

    /* ','によるトークン分解*/
    prePos=tmpPos; 
    cal->argCnt=0;

    if( *prePos != ')' ){ /* 'abc()'引数なしでない場合*/
      while(1){
        if(*tmpPos=='(') tmpPos=mssSkipStrBlock(tmpPos,'(',')');
        if(*tmpPos=='"'){
          tmpPos=mssSkipStrBlock(tmpPos,'"','"');
          if(tmpPos==NULL) {
            mssShowErrMsg("mismatch of \"\"");
            mssEnd(mssErrorNoDefault);
          }
        }

        if(*tmpPos==',' || *tmpPos==')'){
          cal->arg[cal->argCnt].str=prePos; /*引数のセット*/
          prePos=tmpPos+1;
          cal->argCnt++;                    /*引数の数のインクリメント*/
          if(*tmpPos==')'){
            *tmpPos='\0';
            break;
          }else{
            *tmpPos='\0';
          }
        }
        tmpPos++; 
      }
    }

    /*引数のチェック*/
    if(cal->func->argMul == 0){ /*可変引数でない場合*/
      if(cal->func->argCnt != cal->argCnt){
        mssShowErrMsg("the number of arguments is wrong in [%s]\n",cal->func->name);
        mssEnd(mssErrorNoDefault);
      }
    }else{
      if(cal->argCnt>MaxCalArg){
        mssShowErrMsg("the number of arguments must be less than %d in [%s]\n",MaxCalArg,cal->func->name);
        mssEnd(mssErrorNoDefault);
      }
      if(cal->func->argCnt > cal->argCnt){
        mssShowErrMsg("the number of arguments is wrong [%s]\n",cal->func->name);
        mssEnd(mssErrorNoDefault);
      }
    }

  /*
    &&& 演算子の場合
     X='\0'とすると
     (a+b)/cX
      ->
     (a+b)XcX
     +    |+
     arg[0]|
     |    |arg[1]
     str  +
          ope->add
     cal->argCnt=2*/
  }else{
    /*演算子のアドレスをセット*/
    cal->func=ope->func;

    /*一項演算子*/
    if(cal->func->argCnt == 1){
      /*左側の引数があればエラー*/
      if(ope->add!=str){
        mssShowErrMsg("一項演算子\"%s\"の左に引数がある\n",cal->func->name);
        mssEnd(mssErrorNoDefault);
      }

      /*右の引数*/
      cal->arg[0].str=ope->add+strlen(cal->func->name);/*演算子の長さ分ずらす */

      /* 引数があるかどうかチェック*/
      if(*cal->arg[0].str=='\0'){
        mssShowErrMsg("演算子\"%s\"の右の引数がない\n",cal->func->name);
        mssEnd(mssErrorNoDefault);
      }
      cal->argCnt=1;
      *ope->add='\0';

    /*二項演算子(二項までしか対応していない)*/
    }else{
      /*左の引数*/
      cal->arg[0].str=str;

      /*右の引数*/
      cal->arg[1].str=ope->add+strlen(cal->func->name); /*演算子の長さ分ずらす*/

      /* 引数があるかどうかチェック*/
      if(*cal->arg[0].str=='\0'){
        mssShowErrMsg("演算子\"%s\"の左の引数がない\n",cal->func->name);
        mssEnd(mssErrorNoDefault);
      }
      if(*cal->arg[1].str=='\0'){
        mssShowErrMsg("演算子\"%s\"の右の引数がない\n",cal->func->name);
        mssEnd(mssErrorNoDefault);
      }
      cal->argCnt=2;
      *ope->add='\0';
    }
  }

  /*引数を物理的にコピーする(opeLst.tblをfreeするため)*/
  for(i=0; i<cal->argCnt; i++){
    tmpStr=cal->arg[i].str;
    cal->arg[i].str=mssStrdup(tmpStr);
  }
  mssFree(opeLst.tbl);

  /*再帰呼出でreallocされてアドレスが変わる可能性があるので*/
  /*アドレスを固定するのではなく、現在の番号をローカルに利用する*/
  for(i=0; i<cal->argCnt; i++){
    strcpy(buf,cal->arg[i].str);
    argCal=cal_sepCal(buf,cal); /*再帰Call*/
    if(argCal==NULL){
      cal->arg[i].type=LEAF;
    }else{
      cal->arg[i].type=NODE;
      cal->arg[i].node=argCal;
    }
  }
  return(cal);
}

/**
 * # FUNCTION #
 * 計算式をツリー構造で表示(debug用)。
 */
void mssCalShowTree(struct mssCal *cal,int level){

  int i,j;

  for(i=0; i<level; i++){
    printf("  ");
  }
  printf("%s ",cal->func->name);
  for(i=0; i<cal->argCnt; i++){
    printf("%s",cal->arg[i].str);
    switch(cal->arg[i].type){
    case FIELD:
      printf("[fld:");
      for(j=0; j<cal->arg[i].flds->cnt; j++){
        printf("%d(%d) ",(*(cal->arg[i].flds->fi+j))->num,cal->arg[i].datType);
      }
      printf("]");
      break;
    case CONST:
      switch(cal->arg[i].datType){
      case N:
        printf("[cstnum:%g]",cal->arg[i].constant.v.d); break;
      case S:
        printf("[cstStr:%s]",cal->arg[i].constant.v.s); break;
      case X:
        printf("[unknown]"); break;
      }
    }
    printf(" ");
  }
  printf("\n");

  level++;
  for(i=0; i<cal->argCnt; i++){
    if(cal->arg[i].type==NODE)
      mssCalShowTree(cal->arg[i].node,level);
  }
}


/**
 * # FUNCTION #
 * 実引数の位置番号から、その引数のデータ型を返す。
 * (可変引数のためにややこしい)
 */
static int funcArgType(int argNum,struct mssCal *cal){
  if(cal->func->argMul){ /*可変引数が含まれる場合*/
    /* S,N,S,N,S,M,S,S,S の時    */
    /* ~~~~~~~                   */
    /* argNumがここの場合 */
    if(argNum<cal->func->mulBcnt){
      return(cal->func->argType[argNum]);
    }
    /* S,N,S,N,S,M,S,S,S の時    */
    /*             ~~~~~         */
    /* argNumがここの場合 */
    if(argNum>=cal->argCnt - cal->func->mulAcnt){
      return(cal->func->argType[argNum-(cal->argCnt-cal->func->argCnt)+1]);
    }
    /* S,N,S,N,S,M,S,S,S の時    */
    /*         ~~~               */
    /* argNumがここの場合        */
    return(cal->func->argType[cal->func->mulBcnt]);
  }else{                 /*可変引数が含まれていない場合*/
    return(cal->func->argType[argNum]);
  }
}

/**
 * # FUNCTION #
 * 定数と項目変数の評価
 * 項目名にワイルドカードが指定されていれば、それも評価する。
 * 評価された複数の項目名はcal->arg[i]->fldsにセットされる。
 * リーフからトップへ評価し、返す値の型は上位の引数の型に合わせる)
 */
static void cal_evalFldsConst(
  struct mssCal    *cal,
  struct mssHeader *hd) {  /*項目を番号で指定するかどうか*/

  int i;
  char *str;
  char *tmpStr;
  char *s;
  struct mssFldInfo **fi;

  /*NODEの再帰処理*/
  for(i=0; i<cal->argCnt; i++){
    if( cal->arg[i].type ==  NODE){
      cal_evalFldsConst(cal->arg[i].node, hd);
    }
  }

  /*----------------------------------------*/
  /*引数の処理                              */
  /*----------------------------------------*/
  for(i=0; i<cal->argCnt; i++){

    /*データタイプをcal->funcからもってくる*/
    cal->arg[i].datType=funcArgType(i,cal);
    switch(cal->arg[i].type){

    /*引数が関数や演算子を含まない場合*/
    case LEAF:
      /*項目の場合*/
      if( *cal->arg[i].str == '$' ){
        if( *(cal->arg[i].str+1) == '(' ){ /*$(Date*)の時*/
          tmpStr=mssSkipStrBlock(cal->arg[i].str+1,'(',')');
          *(tmpStr-1)='\0'; 
          str=cal->arg[i].str+2;
        }else{
          str=cal->arg[i].str+1;
        }
        cal->arg[i].type = FIELD;
        cal->arg[i].flds = mssInitFields();
        fi=mssFldGlbNam2Add(hd->flds,str);
        mssAddFieldsByFldInfoList(cal->arg[i].flds, fi);
	mssFree(fi); /* 追加 2004/10/31 */

      /*定数の場合*/
      }else{
        /*実際の定数の型と関数が要求する引数の型が合わない場合は
          関数の型に合わせる*/
        cal->arg[i].type = CONST;
        s=cal->arg[i].str;
        if(*s == '"'){ /*実際の定数はS*/
          tmpStr=mssSkipStrBlock(s,'"','"');
          if(tmpStr==NULL) {
            mssShowErrMsg("\"の数がおかしい");
            mssEnd(mssErrorNoDefault);
          }
          *(tmpStr-1) = '\0';
          s++;
        }
        switch(cal->arg[i].datType) {
        case N:
          mssVinit(&cal->arg[i].constant,DBL);
          cal->arg[i].constant.v.d = atof(s);
          break;
        case S:
        case X: /*未定義型の場合は、文字列として値をセットする*/
          mssVinit(&cal->arg[i].constant,STR);
          cal->arg[i].constant.v.s = s;
          break;
        }
      }
      break;

    } /* switch*/
  } /*for i<argCnt*/

  /*返値タイプの設定*/
  cal->retType=cal->func->retType;
}

/**
 * # FUNCTION #
 * mssCal構造体に納められた全関数(演算子)のうち、利用しているデータ読込関数
 * で可能でない関数(演算子)が含まれていないかチェックする。
 * 含まれていれば、エラーメッセージを表示して終了する。
 */
static void cal_chkProcType(struct mssCal *cal, unsigned char procType)
{
  int i;

  if( (cal->func->procType & procType) == 0 ){
    mssShowErrMsg("cannot use the function [%s] in this command, or try it again with -k option if possible",cal->func->name);
    mssEnd(mssErrorNoDefault);
  }
  for(i=0; i<cal->argCnt; i++){
    if(cal->arg[i].type==NODE)
      cal_chkProcType(cal->arg[i].node,procType);
  }
}

/**
 * # FUNCTION #
 * 前行、トップ行、前行結果、トップ行結果を利用する関数が含まれているかチェック
 * し、含まれていれば、グローバル変数である、
 * UsedPrvResult,UsedTopResult,UsedPrvField,UsedTopFieldフラグをセットする。
 */
static void cal_chkUsedPrvFldRsl(struct mssCal *cal)
{
  int i;

  if( 0==strcmp(cal->func->name,"prvResult") ){
    UsedPrvResult=1;
    return;
  }
  if( 0==strcmp(cal->func->name,"topResult") ){
    UsedTopResult=1;
    return;
  }
  if( 0==strcmp(cal->func->name,"prvField") ){
    UsedPrvField=1;
    return;
  }
  if( 0==strcmp(cal->func->name,"lineCmp") ){
    UsedPrvField=1;
    return;
  }
  if( 0==strcmp(cal->func->name,"topField") ){
    UsedTopField=1;
    return;
  }
  for(i=0; i<cal->argCnt; i++){
    if(cal->arg[i].type==NODE){
      cal_chkUsedPrvFldRsl(cal->arg[i].node);
    }
  }
}

/**
 * # FUNCTION #
 * データ読込関数をreadFldRecに設定登録する。
 */
void mssCalReadFuncIsFldRec(struct mssCal *cal,struct mssFldRec *fr)
{
  Fr=fr;
  cal_chkProcType(cal, 1);
  cal_chkUsedPrvFldRsl(cal);
}

/**
 * # FUNCTION #
 * データ読込関数をreadFRDに設定登録する。
 */
void mssCalReadFuncIsFRD(struct mssCal *cal,struct mssFldRecDbl *frd)
{
  Frd=frd;
  cal_chkProcType(cal, 2);
  cal_chkUsedPrvFldRsl(cal);
}

/**
 * # FUNCTION #
 * データ読込関数をreadFRKに設定登録する。
 */
void mssCalReadFuncIsFRK(struct mssCal *cal,struct mssFldRecKey *frk)
{
  Frk=frk;
  cal_chkProcType(cal, 4);
  cal_chkUsedPrvFldRsl(cal);
}

/**
 * # FUNCTION #
 * カレント行(pnt)において計算を再帰的に実行し、その結果を返す。
 * 計算結果の値の型が呼び出し側の型に一致しない場合は型変換を行う。
 */
static MssValue cal_calculate_sub(struct mssCal *cal, char **pnt)
{

  int i,j;
  int fldNo;
  MssValue val;
  for(i=0; i<cal->argCnt; i++){
    switch(cal->arg[i].type){
    case NODE:
      cal->arg[i].val[0] = cal_calculate_sub(cal->arg[i].node,pnt);
      cal->arg[i].valCnt = 1;
      break;
    case CONST:
      cal->arg[i].val[0] = cal->arg[i].constant;
      cal->arg[i].valCnt = 1;
      break;
    case FIELD:
      for(j=0; j<cal->arg[i].flds->cnt; j++){
        fldNo=MssFlds2num(cal->arg[i].flds,j);

        cal->arg[i].val[j].nul =0;/*nul初期化*/
        if( MssIsNull(*(pnt+fldNo)) ){
          MssVnull(cal->arg[i].val[j]);
          continue;
        }

        switch(cal->arg[i].datType){
        case N:
          mssVinit(&cal->arg[i].val[j],DBL);
          cal->arg[i].val[j].v.d =atof(*(pnt+fldNo));
          break;
        case S:
        case X: /*未定義型の場合は、文字列として値をセットする*/
          mssVinit(&cal->arg[i].val[j],STR);
          cal->arg[i].val[j].v.s =*(pnt+fldNo);
          break;
        default:
          mssShowErrMsg("internal error\n");
          mssEnd(mssErrorNoDefault);
        }
      }
      cal->arg[i].valCnt = j;
      break;
    }
  }

  /*型変換*/
  for(i=0; i<cal->argCnt; i++){
    /*文字列型→数値型変換*/
    if(cal->arg[i].datType==N && cal->arg[i].val[0].vType==STR){
      for(j=0; j<cal->arg[i].valCnt; j++){
        cal->arg[i].val[j]=mssVs2d(cal->arg[i].val[j]);
      }
      continue;
    }

    /*数値型→文字列型変換*/
    if(cal->arg[i].datType==S && cal->arg[i].val[0].vType==DBL){
      for(j=0; j<cal->arg[i].valCnt; j++){
        cal->arg[i].val[j]=mssVd2s(cal->arg[i].val[j]);
      }
      continue;
    }
  }

  /* 実際の関数呼び出し */
  val=(cal->func->pnt)(cal);

  return( val );
}

static void setRslVal(struct PrvRslStruct *prv, MssValue *rsl)
{
  prv->val=*rsl;
  if(rsl->vType==STR && !rsl->nul){
    strcpy(prv->buf,rsl->v.s);
    prv->val.v.s=prv->buf;
  }
}

static void clearRslVal(struct PrvRslStruct *prv)
{
  MssVnull(prv->val); /*先頭行結果を入れる変数のクリア*/
}

static void setFldVal(struct PrvFldStruct *prv, char **pnt, int fldCnt)
{
  int chrCnt=0;
  char *str;
  int i;

  for(i=0; i<fldCnt; i++){
    prv->pnt[i]=&prv->buf[chrCnt];
    if(pnt==NULL){
      prv->buf[chrCnt++]=MssNullChr;
      prv->buf[chrCnt++]='\0';
    }else{
      str=*(pnt+i);
      while(*str!='\0'){
        prv->buf[chrCnt++]=*str++;
      }
      prv->buf[chrCnt++]='\0';
    }
  }
}

static void clearFldVal(struct PrvFldStruct *prv)
{
  prv->pnt[0]=NULL;  /*前行項目を入れる変数のクリア*/
}

static void setTopField(void)
{
  if(Frk!=NULL){
    if(Frk->curRecNo==1) setFldVal(&TopFld,Frk->pnt,Frk->fldCnt);
  }else if(Frd!=NULL){
    if(mssGV.inCnt  ==1) setFldVal(&TopFld,Frd->pnt[Frd->new],Frd->fldCnt);
  }else if(Fr!=NULL){
    if(mssGV.inCnt  ==1) setFldVal(&TopFld,Fr->pnt,Fr->fldCnt);
  }
}

static void setPrvField(void)
{
  if(Frk!=NULL){
    if(Frk->curRecNo==Frk->keyRecCnt){
      clearFldVal(&PrvFld);
    }else{
      setFldVal(&PrvFld,Frk->pnt,Frk->fldCnt);
    }
  }else if(Frd!=NULL){
    setFldVal(&PrvFld,Frd->pnt[Frd->new],Frd->fldCnt);
  }else if(Fr!=NULL){
    setFldVal(&PrvFld,Fr->pnt,Fr->fldCnt);
  }
}

static void setTopResult(MssValue rsl)
{
  if(Frk!=NULL){
    if(Frk->curRecNo==1) setRslVal(&TopRsl,&rsl);
    if(Frk->curRecNo==Frk->keyRecCnt){
      clearRslVal(&PrvRsl);
    }
  }else if(Frd!=NULL){
    if(mssGV.inCnt  ==1) setRslVal(&TopRsl,&rsl);
  }else if(Fr!=NULL){
    if(mssGV.inCnt  ==1) setRslVal(&TopRsl,&rsl);
  }
}

static void setPrvResult(MssValue rsl)
{
  if(Frk!=NULL){
    if(Frk->curRecNo==Frk->keyRecCnt){
      clearRslVal(&PrvRsl);
    }else{
      setRslVal(&PrvRsl,&rsl);
    }
  }else if(Frd!=NULL){
    setRslVal(&PrvRsl,&rsl);
  }else if(Fr!=NULL){
    setRslVal(&PrvRsl,&rsl);
  }
}

/**
 * # FUNCTION #
 * カレント行(pnt)において計算を実行し、その結果を返す。
 * 現在行や結果を保存しておく必要があれば(prvField(),prvResult()など)、
 * 保存しておく。
 */
MssValue mssCalculate(struct mssCal *cal, char **pnt)
{
  MssValue rsl;

  /*先頭行項目のセット*/
  if(UsedTopField) setTopField();

  /*計算式の中で確保されたstr領域を開放*/
  cal_freeStrMalAdd();

  /*計算式評価*/
  rsl=cal_calculate_sub(cal,pnt);

  /*前行項目のセット*/
  if(UsedPrvField !=0) setPrvField();

  /*先頭行計算結果のセット*/
  if(UsedTopResult!=0) setTopResult(rsl);

  /*前行計算結果のセット*/
  if(UsedPrvResult!=0) setPrvResult(rsl);

  return(rsl);
}

/**
 * # FUNCTION #
 * 計算式(str)を評価し、Cal構造体(ツリー構造による式の表現)を設定する。
 * ツリーのトップノードへのポインタを返す。
 * ヘッダー(hd)は、計算式内の項目名を解決するために利用する。
 */
struct mssCal *mssCalCompile(char *str, struct mssHeader *hd)
{
  struct mssCal *cal=NULL;

  initCalFunc();             /*func[]の未定義変数のセット*/
  str=cal_chkCal(str);       /*数式事前チェック(余分スペースの除去)*/
  cal=cal_sepCal(str,cal);   /*数式をトークン分割しツリー構造にする*/

  if(cal==NULL){
    mssShowErrMsg("cannot find any operators or functions");
    mssEnd(mssErrorNoDefault);
  } 
  cal_evalFldsConst(cal, hd); /*引数の評価(変数や定数を展開)*/
  /*mssCalShowTree(cal,0);*/

  mssGV.inFldCnt=hd->flds->cnt; /*入力データの項目数をグローバル変数に登録*/

  MssVnull(PrvRsl.val); /*前行結果を入れる変数のクリア*/
  MssVnull(TopRsl.val); /*先頭行結果を入れる変数のクリア*/
  PrvFld.pnt[0]=NULL;   /*前行項目を入れる変数のクリア*/
  TopFld.pnt[0]=NULL;   /*先頭行項目を入れる変数のクリア*/

  mssFree(str); /* 追加 2004/10/31 */
  return(cal);
}

/**
 * # FUNCTION #
 * 式評価ツリーの領域開放
 */
void mssCalFree(struct mssCal *cal)
{
  int i;

  cal_freeStrMalAdd();

  for(i=0; i<cal->argCnt; i++){
    if(cal->arg[i].type==NODE){
      mssCalFree(cal->arg[i].node);
      mssFree(cal->arg[i].str);
    }else{
      if(cal->arg[i].type == FIELD){
        mssFreeFields(cal->arg[i].flds);
      }
      mssFree(cal->arg[i].node);
      mssFree(cal->arg[i].str);
    }
  }
  /* 追加 2004/10/31 */
  if(cal->regex!=NULL){
    regfree(cal->regex);
    mssFree(cal->regex);
  }
  mssFree(cal);
}
