/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるハッシュ関連の関数
 * ============================================================================
 */

#include <musashi/mssHash.h>
#include <musashi/mssValue.h>
#include <musashi/mssHeader.h>

#include <string.h>

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 行-項目を扱うハッシュ
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 行-項目Hash関数：複数項目文字列からのHash値の計算
 */
static int getHashValFld(char **str, struct mssFields *flds, int hv)
{
  int i,h=0;
  char *v;

  for(i=0; i<flds->cnt; i++){
    v=*(str+MssFlds2num(flds,i));
    while(*v != '\0'){
      h=(64*h + (unsigned char)*v) % hv;
      v++;
    }
  }
  return h;
}

/**
 * # FUNCTION #
 * 二つの行(s1,s2)の指定項目(flds1,flds2)によるの比較
 * strcmp()関数と同様の値を返す。
 */
static int keyCmp(char **s1,struct mssFields *flds1,char **s2,struct mssFields *flds2)
{
  int i;
  int cmp;

  if( flds1->cnt != flds2->cnt ){
    mssShowErrMsg("internal error : mismatch of the number of key fields");
    exit(mssErrorNoDefault);
  }

  for(i=0; i<flds1->cnt; i++){
    if( 0 != (cmp=strcmp(*(s1+MssFlds2num(flds1,i)),
                         *(s2+MssFlds2num(flds2,i)))) ){
      return(cmp);
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * 行-項目ハッシュ構造体(HashFld)を初期化し、そのポインタを返す。
 * sizeには、ハッシュ表のサイズ(スロットの数)を指定する。
 * fldsは、ハッシュ値を計算する元になる項目を示したmssFields構造体。
 * この値は、２の巾乗に近くない素数を選ぶべきである。
 * チェイン法を用いている。
 * ハッシュ関数には除算法を用いている。
 */
struct mssHashFld *mssInitHashFld( int size, struct mssFields *flds)
{

  struct mssHashFld *hash;

  hash=mssMalloc(sizeof(struct mssHashFld),"initHashFld");
  hash->hashVal = size;
  hash->flds    = flds;
  hash->keyCnt  = 0;
  hash->endCnt  = 0;
  hash->node=mssCalloc(sizeof(struct mssHashNodeFld *)*hash->hashVal,"initHashFld");
  return(hash);
}

/**
 * # FUNCTION #
 * 行-項目ハッシュ構造体(HashFld)領域の開放。
 */
void mssFreeHashFld(struct mssHashFld *hash){
  int i;
  struct mssHashNodeFld *node;
  struct mssHashNodeFld *tmp;

  if(hash==NULL) return;
  for(i=0; i<hash->hashVal; i++){
    if( NULL != (node=*(hash->node+i)) ){
      while(node!=NULL){
        tmp=node;
        mssFree(node->rec);
        node=node->next;
        mssFree(tmp);
      }
    }
  }
  mssFree(hash->node);
  mssFree(hash);
}

/**
 * # FUNCTION #
 * 行-項目ハッシュテーブルに、新しいデータを追加する。
 */
void mssHashInsertFld(struct mssHashFld *hash, char **str)
{
  int hv;

  struct mssHashNodeFld *newNode;
  struct mssHashNodeFld *node;
  struct mssHashNodeFld *last;

  hv=getHashValFld(str,hash->flds,hash->hashVal);
  node=*(hash->node+hv);

  newNode=mssCalloc(sizeof(struct mssHashNodeFld),"hashInsertFld");

  if(node==NULL){
    *(hash->node+hv)=newNode;
  }else{
    while(node!=NULL){
      if(0==keyCmp(*node->rec,hash->flds,str,hash->flds)){
        node->rec=mssRealloc(node->rec,
                          sizeof(char **)*(node->recCnt+1),"hashInsertFld");
        *(node->rec+node->recCnt)=str;
        node->recCnt++;
        mssFree(newNode);
        return;
      }else{
        last=node;
        node=node->next;
      }
    }
    last->next=newNode;
  }
  newNode->rec=mssMalloc(sizeof(char **),"hashInsertFld");
  *newNode->rec=str;
  newNode->recCnt++;
  newNode->next=NULL; 
  hash->keyCnt++; /*キーの種類をカウントアップ*/
}

/**
 * # FUNCTION #
 * 行-項目ハッシュテーブルから指定の項目値を持った値を検索しする。
 * 見つかれば、そのHashNodeFldへのポインタを返す。なければNULLを返す。
 * 項目値はstrとfldsによって指定する。
 */
struct mssHashNodeFld *mssHashMemberFld( struct mssHashFld *hash, char **str, struct mssFields *flds)
{
  int hv;
  struct mssHashNodeFld *node;

  hv=getHashValFld(str,flds,hash->hashVal);
  node=*(hash->node+hv);

  while(node!=NULL){
    if(0==keyCmp(*node->rec,hash->flds,str,flds)){
      node->endFlg=1;
      hash->endCnt++;
      return(node);
    }else{
      node=node->next;
    }
  }
  return(NULL);
}

/**
 * # FUNCTION #
 * 行-項目ハッシュテーブルの出力(デバッグ用)。
 */
void mssShowHashFld(struct mssHashFld *hash, int fldCnt)
{
  int i,j,k;
  struct mssHashNodeFld *node;

  printf("Key    :");
  for(i=0; i<hash->flds->cnt; i++){
    printf("%d ", (*(hash->flds->fi+i))->num);
  }
  printf("\n");
  printf("keyCnt :%d\n",hash->keyCnt);
  printf("endCnt :%d\n",hash->endCnt);
  printf("fldCnt :%d\n",hash->fldCnt);
  printf("hashVal:%d\n",hash->hashVal);
  for(i=0; i<hash->hashVal; i++){
    if( NULL!=(node=*(hash->node+i)) ){
      while(node!=NULL){
        printf("========== HashNo: [%d]\n",i);
        for(j=0; j<node->recCnt; j++){
          for(k=0; k<fldCnt; k++){
            printf("|%s |",*(*(node->rec+j)+k));
          }
          printf("\n");
        }
        node=node->next;
      }
    }
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * MssValueを扱うハッシュ
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * MssValue Hash関数：複数項目文字列からのHash値の計算
 */
static int getHashVal(char *str,int hv){
  int v=0;

  while(*str != '\0') v+=(unsigned char)*str++;
  return(v % hv);
}
  
/**
 * # FUNCTION #
 * MssValueハッシュ構造体(Hash)を初期化し、そのポインタを返す。
 * sizeには、ハッシュ表のサイズ(スロットの数)を指定する。
 * fldsは、ハッシュ値を計算する元になる項目を示したmssFields構造体。
 * この値は、２の巾乗に近くない素数を選ぶべきである。
 * チェイン法を用いている。
 * ハッシュ関数には除算法を用いている。
 */
struct mssHash *mssInitHash(int size)
{

  struct mssHash *hash;

  hash=mssMalloc(sizeof(struct mssHash),"initHash");
  hash->hashVal = size;
  hash->cnt = 0;
  hash->node=mssCalloc(sizeof(struct mssHashNode *)*hash->hashVal,"initHash");
  return(hash);
}

/**
 * # FUNCTION #
 * MssValueハッシュ構造体(HashFld)領域の開放。
 */
void mssFreeHash(struct mssHash *hash)
{
  int i;
  struct mssHashNode *node;
  struct mssHashNode *tmp;

  if(hash==NULL) return;
  for(i=0; i<hash->hashVal; i++){
    if( NULL != (node=*(hash->node+i)) ){
      while(node!=NULL){
        tmp=node;
        mssFree(node->str);
        node=node->next;
        mssFree(tmp);
      }
    }
  }
  mssFree(hash->node);
  mssFree(hash);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルに、新しいデータ(val)を追加する。
 * 新しい値としてinsertできればノードのアドレスを返す。
 * 既に値が存在すれば、NULLを返す。
 */
struct mssHashNode *mssHashInsert( struct mssHash *hash, char *str, MssValue val)
{
  int hv;
  struct mssHashNode *newNode;
  struct mssHashNode *node;
  struct mssHashNode *last=NULL;
  struct mssHashNode **top=NULL;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);

  newNode=mssMalloc(sizeof(struct mssHashNode),"hashInsert");

  if(node==NULL){
    *(hash->node+hv)=newNode;
    top=hash->node+hv;
  }else{
    while(node!=NULL){
      if( 0==strcmp(node->str,str) ){
        mssFree(newNode);
        return(NULL);
      }else{
        last=node;
        node=node->next;
      }
    }
    last->next=newNode;
  }
  newNode->str=mssStrdup(str);
  newNode->val=val;
  newNode->last=last;
  newNode->next=NULL;
  newNode->top=top;
  hash->cnt++;
  return(newNode);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルに、新しいデータ(val)を追加する。
 * 新しい値としてinsertできればノードのアドレスを返す。
 * 既に値が存在すれば、そのノードのアドレスを返す。
 */
struct mssHashNode *mssHashInsertAdd(struct mssHash *hash, char *str, MssValue val)
{
  int hv;
  struct mssHashNode *newNode;
  struct mssHashNode *node;
  struct mssHashNode *last=NULL;
  struct mssHashNode **top=NULL;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);

  newNode=mssMalloc(sizeof(struct mssHashNode),"hashInsertAdd");

  if(node==NULL){
    *(hash->node+hv)=newNode;
    top=hash->node+hv;
  }else{
    while(node!=NULL){
      if( 0==strcmp(node->str,str) ){
        mssFree(newNode);
        return(node);
      }else{
        last=node;
        node=node->next;
      }
    }
    last->next=newNode;
  }
  newNode->str=mssStrdup(str);
  newNode->val=val;
  newNode->last=last;
  newNode->next=NULL;
  newNode->top=top;
  hash->cnt++;
  return(newNode);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルから、指定のノードを削除する。
 * 削除したノードの次のノードを返す。
 * 次のノードがない場合はNULLを返す。
 */
struct mssHashNode *mssHashDeleteNode(struct mssHash *hash,struct mssHashNode *node)
{
  struct mssHashNode *retNode;

  if(node->top==NULL){
    if(node->next!=NULL){             /* OOO -> ### -> OOO */
      node->next->last=node->last;
      node->last->next=node->next;
      retNode=node->next;
    }else{                            /* OOO -> ### */
      node->last->next=NULL;
      retNode=NULL;
    }
  }else{
    if(node->next!=NULL){             /* ### -> OOO */
      node->next->top=node->top;
      node->next->last=NULL;
      *node->top=node->next;
      retNode=node->next;
    }else{                            /* ### */
      *node->top=NULL;
      retNode=NULL;
    }
  }

  mssFree(node->str);
  mssFree(node);

  hash->cnt--;

  return(retNode);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルから、指定の文字列ノードを削除する。
 */
void mssHashDelete(struct mssHash *hash, char *str)
{
  int hv;
  struct mssHashNode *node;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);
  while(node!=NULL){
    if( 0==strcmp(node->str,str) ){
      mssHashDeleteNode(hash,node);
    }else{
      node=node->next;
    }
  }
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルに、メンバー(str)があるか検索する。
 * あればばそのノードへのポインタを返し、なければNULLを返す。
 */
struct mssHashNode *mssHashMember(struct mssHash *hash, char *str)
{
  int hv;
  struct mssHashNode *node;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);
  while(node!=NULL){
    if( 0==strcmp(node->str,str) ){
      return(node);
    }else{
      node=node->next;
    }
  }
  return(NULL);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルに、メンバー(str)があるか検索する。
 * あればそのメンバーの文字列(node->str)へのポインタを返し、なければNULLを返す。
 */
char *mssHashMemberAdd(struct mssHash *hash, char *str)
{
  int hv;
  struct mssHashNode *node;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);
  while(node!=NULL){
    if( 0==strcmp(node->str,str) ){
      return(node->str);
    }else{
      node=node->next;
    }
  }
  return(NULL);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルに、メンバー(str)があるか検索する。
 * あればそのメンバーのMssValue(node->value)を返し、なければNULLのMssValueを返す。
 */
MssValue mssHashMemberVal(struct mssHash *hash, char *str)
{
  int hv;
  struct mssHashNode *node;
  MssValue v;

  hv=getHashVal(str,hash->hashVal);
  node=*(hash->node+hv);
  while(node!=NULL){
    if( 0==strcmp(node->str,str) ){
      return(node->val);
    }else{
      node=node->next;
    }
  }
  v.nul=1;
  return(v);
}

/**
 * # FUNCTION #
 * MssValueハッシュテーブルの出力(デバッグ用)。
 */
void mssShowHash(struct mssHash *hash)
{
  int i;
  struct mssHashNode *node;

  for(i=0; i<hash->hashVal; i++){
    if( NULL != (node=*(hash->node+i)) ){
      while(node!=NULL){
        printf("hashVal=%d str=",i);
        if(node->last!=NULL) printf("%s->",node->last->str);
        else                 printf("NULL->");
        printf("[%s]",node->str);
        if(node->next!=NULL) printf("->%s  ",node->next->str);
        else                 printf("->NULL  ");
        switch(node->val.vType){
        case INT : /* int    */
          printf("val=%d\n",node->val.v.i);
          break;
        case DBL : /* double */
          printf("val=%g\n",node->val.v.d);
          break;
        case STR : /* char * */
          printf("val=%s\n",node->val.v.s);
          break;
        case ADD : /* void * */
          printf("val=%x\n",(int)node->val.v.a);
          break;
        case USI : /* unsigned short int */
          printf("val=%d\n",(int)node->val.v.usi);
          break;
        }
        node=node->next;
      }
    }
  }
}

