/**
 * # CHAPTER #
 * ============================================================================
 * XMLtableのヘッダーを扱う関数
 * 2002/08/07 mssSetOptKey: numFlg,revFlgをクリア追加
 * 2004/01/15 mssFields(sf)に同じ項目が２つ以上使われていればエラー終了する
 * 2004/08/09 mssFreeHeader 引数がNULLでリターン
 * ============================================================================
 */

#include <musashi/mssHeader.h>
#include <musashi/mssXml.h>
#include <musashi/mssXtTag.h>
#include <musashi/qs5.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>

/* ############ グローバル変数 ##############*/
/*テキストフラグ*/
extern struct mssGlobalVariables mssGV;

/**
 * # SECTION #
 *----------------------------------------------------------------------------- 
 * mssFldInfo構造体の項目名、項目番号、アドレス、などの相互変換
 *----------------------------------------------------------------------------- 
 */

/**
 * # DOCUMENT #
 * ここでの関数はreadHeader関数にてHeader構造体をセットした後に利用可能となる。
 */

/**
 * # FUNCTION #
 * mssFldInfo構造体の領域確保＆初期化
 */
struct mssFldInfo *mssInitFldInfo()
{
  struct mssFldInfo *fi;

  fi=mssMalloc(sizeof(struct mssFldInfo),"initFldInfo");

  fi->num=0;
  fi->name=NULL;
  fi->priority=0;
  fi->revFlg=0;
  fi->numFlg=0;
  fi->length=0;
  fi->comment=NULL;

  return(fi);
}

/**
 * # FUNCTION #
 * mssFldInfo構造体の領域開放
 */
void mssFreeFldInfo( struct mssFldInfo *fi)
{

  if(fi!=NULL){
    mssFree(fi->name);
    mssFree(fi->comment);
  }
  mssFree(fi);
}

/**
 * # FUNCTION #
 * mssFldInfoの出力(デバッグ用)
 */
void mssShowFldInfo(struct mssFldInfo *fi)
{
  printf("\tnum=%d name=%s priority=%d revFlg=%d numFlg=%d comment=%s\n",
       fi->num, fi->name, fi->priority, fi->revFlg, fi->numFlg, fi->comment);
}

/**
 * # FUNCTION #
 * numで指定された番号の項目をmssFields構造体のmssFldInfo構造体(flds->fi)から
 * 探し、見つかればそのポインタを返す。
 * 見つからなければエラーを表示して終了する。
 * ここで指定されるnumは内部での項目番号で、０から始まる。
 */
struct mssFldInfo *mssFldNum2Add(struct mssFields *flds, int num)
{
  int i;

  for(i=0; i<flds->cnt; i++){
    if( num==MssFlds2num(flds,i) ){
      return( *(flds->fi+i) );
    }
  }
  mssShowErrMsg("internal error field number not found : [%d]",num);
  mssEnd(mssErrorNoDefault);
  return(NULL); /*コンパイル警告を避けるため*/
}

/**
 * # FUNCTION #
 * nameで指定された名前の項目をmssFields構造体のmssFldInfo構造体(flds->fi)から
 * 探し、見つかればそのポインタを返す。
 * 見つからなければエラーを表示して終了する。
 */
struct mssFldInfo *mssFldNam2Add(struct mssFields *flds, char *name)
{
  int i;

  for(i=0; i<flds->cnt; i++){
    if( 0==strcmp(MssFlds2name(flds,i), name) ){
      return( *(flds->fi+i) );
    }
  }
  mssShowErrMsg("field name or number not found : [%s]",name);
  mssEnd(mssErrorNoDefault);
  return(NULL); /*コンパイル警告を避けるため*/
}

/**
 * # FUNCTION #
 * nameで指定されたワイルドカード(*,?など)を含む項目名をmssFields構造体の
 * mssFldInfo構造体(flds->fi)から探し、マッチしたポインタ全てを返す。
 * 一つもマッチしなければエラーを表示して終了する。
 */
struct mssFldInfo **mssFldGlbNam2Add(struct mssFields *flds, char *name)
{
  int i;
  int cnt=0;
  struct mssFldInfo **fiList=NULL;

  for(i=0; i<flds->cnt; i++){

    /*項目名のワイルドカードマッチング*/
    if( 0==fnmatch(name,(*(flds->fi+i))->name,0) ){

      /*領域確保*/
      fiList = mssRealloc(fiList,sizeof(struct mssFldInfo *)*(cnt+1),
        "FldGlbNam2Add");
      *(fiList+cnt) = *(flds->fi+i);
      cnt++;
    }
  }
  if( cnt == 0 ) {
    mssShowErrMsg("field name not found : [%s]",name);
    mssEnd(mssErrorNoDefault);
  }

  /*終端のNULLを追加*/
  fiList = mssRealloc(fiList,sizeof(struct mssFldInfo *)*(cnt+1),"fldRegN2A");
  *(fiList+cnt) = NULL;

  return(fiList);
}

/**
 * # FUNCTION #
 * nameで指定された名前の項目をmssFields構造体のmssFldInfo構造体(flds->fi)から
 * 探し、見つかればその項目番号を返す。
 * 見つからなければエラーを表示して終了する。
 * 注) 項目番号とは、XMLtableの<field no="1">要素のno属性の値から１を引いた数。
 */
int mssFldNam2Num(struct mssFields *flds, char *name)
{
  int i;
  int namLen;

  namLen=strlen(name);
  for(i=0; i<flds->cnt; i++){
    if( 0==strcmp((*(flds->fi+i))->name, name) ){
      return( (*(flds->fi+i))->num );
    }
  }
  mssShowErrMsg("field name or number not found : [%s]",name);
  mssEnd(mssErrorNoDefault);
  return(0); /*とりあえず*/
}

/**
 * # FUNCTION #
 * numで指定された項目番号をmssFields構造体のmssFldInfo構造体(flds->fi)から
 * 探し、ソート優先順位番号の値を返す。
 * ありえない項目番号が指定されたときは、エラーを表示して終了する。
 * 注1) 項目番号とは、XMLtableの<field no="1">要素のno属性の値から１を引いた数。
 * 注2) ソート優先順位番号とは、XMLtableの<sort priority="1">要素のpriority
 *      属性の値のこと。
 */
int mssFldNum2SrtPri(struct mssFields *flds, int num)
{
  int i;

  if( num<0 || num>flds->cnt-1 ) {
    mssShowErrMsg("Internal Error: field number not found : [%d]",num);
    mssEnd(mssErrorNoDefault);
  }

  for(i=0; i<flds->cnt; i++){
    if( (*(flds->fi+i))->num == num ){
      return( (*(flds->fi+i))->priority );
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * numで指定された項目番号をmssFields構造体のmssFldInfo構造体(flds->fi)から
 * 探し、その項目が逆順ソートの指定があれば１、そうでなければ０を返す。
 * ありえない項目番号が指定されたときは、エラーを表示して終了する。
 * 注1) 項目番号とは、XMLtableの<field no="1">要素のno属性の値から１を引いた数。
 * 注2) 逆順ソート指定とは、XMLtableの<reverse/>要素のこと。
 */
int mssFldNum2SrtRev(struct mssFields *flds, int num)
{
  int i;

  if( num<0 || num>flds->cnt-1 ) {
    mssShowErrMsg("Internal Error: field number not found : [%d]",num);
    mssEnd(mssErrorNoDefault);
  }

  for(i=0; i<flds->cnt; i++){
    if( (*(flds->fi+i))->num == num ){
      return( (*(flds->fi+i))->revFlg );
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * numで指定された項目番号をmssFields構造体のmssFldInfo構造体(flds->fi)から
 * 探し、その項目が数値ソートの指定があれば１、そうでなければ０を返す。
 * ありえない項目番号が指定されたときは、エラーを表示して終了する。
 * 注1) 項目番号とは、XMLtableの<field no="1">要素のno属性の値から１を引いた数。
 * 注2) 数値ソート指定とは、XMLtableの<numeric/>要素のこと。
 */
int mssFldNum2SrtNum(struct mssFields *flds, int num)
{
  int i;

  if( num<0 || num>flds->cnt-1 ) {
    mssShowErrMsg("Internal Error: field number not found : [%d]",num);
    mssEnd(mssErrorNoDefault);
  }

  for(i=0; i<flds->cnt; i++){
    if( (*(flds->fi+i))->num == num ){
      return( (*(flds->fi+i))->numFlg );
    }
  }
  return(0);
}

/**
 * # SECTION #
 *----------------------------------------------------------------------------- 
 * mssHeader構造体を扱う関数
 *----------------------------------------------------------------------------- 
 */

/**
 * # FUNCTION #
 * ヘッダ構造体の領域開放
 */
void mssFreeHeader(struct mssHeader *hd)
{
  if(hd==NULL) return;
  mssFree(hd->title  );
  mssFree(hd->comment);
  mssFreeFields(hd->flds);
  mssFree(hd->xmlver);
  mssFree(hd->xmlenc);
  mssFree(hd);
}

/**
 * # FUNCTION #
 * タイトル(title)とコメント(comment)を引数にしてHeader構造体の生成および
 * 初期化を行う。
 */
struct mssHeader *mssInitHeader(char *title, char *comment)
{

  struct mssHeader *hd;

  hd=mssMalloc( sizeof(struct mssHeader),"initHeader" );

  hd->title  = mssStrdup(title);
  hd->comment= mssStrdup(comment);
  hd->xmlver = mssStrdup(MssXmlDefVer);
  hd->xmlenc = mssStrdup(MssXmlDefEnc);
  hd->version= MssXtDefVer;
  hd->flds   = mssInitFields();
  hd->recCnt = -1;

  return(hd);
}

/**
 * # FUNCTION #
 * mssHeader構造体を生成し、oldで指定した既存のヘッダー情報(項目情報は除く)を
 * コピーする。
 * コピーする情報は、title,comment,xmlver,xmlencで、その他は初期化する。
 * version はMssXtDefVerで初期化される。
 * コピーの際はmssStrdupを利用するので、
 * 既存のヘッダー情報(文字列)をポインタで共有することはない。
 */
struct mssHeader *mssInitCpyHeader(struct mssHeader *old)
{
  struct mssHeader *hd;

  hd=mssMalloc( sizeof(struct mssHeader),"initCpyHeader" );

  if(NULL!=old->title)   hd->title  = mssStrdup(old->title);
  else                   hd->title  = NULL;

  if(NULL!=old->comment) hd->comment= mssStrdup(old->comment);
  else                   hd->comment= NULL;

  if(NULL!=old->xmlver)  hd->xmlver = mssStrdup(old->xmlver);
  else                   hd->xmlver = NULL;
  
  if(NULL!=old->xmlenc)  hd->xmlenc = mssStrdup(old->xmlenc);
  else                   hd->xmlenc = NULL;

  hd->version = MssXtDefVer;
  hd->flds    = mssInitFields();
  hd->recCnt  = -1;

  return(hd);
}

/**
 * # FUNCTION #
 * mssHeader構造体を生成し、引数で指定した文字列でヘッダー情報を設定する。
 * 設定する情報は、title,comment,xmlver,xmlenc,versionで、その他は初期化する。
 * 設定の際はmssStrdupを利用するので、
 * 引数の文字列をポインタで共有することはない。
 */
struct mssHeader *mssInitSetHeader( char *title, char *comment, char *xmlver, char *xmlenc, int version)
{
  struct mssHeader *hd;

  hd=mssMalloc( sizeof(struct mssHeader),"initCpyHeader" );

  if(NULL!=title)   hd->title  = mssStrdup(title);
  else              hd->title  = NULL;

  if(NULL!=comment) hd->comment= mssStrdup(comment);
  else              hd->comment= NULL;

  if(NULL!=xmlver)  hd->xmlver = mssStrdup(xmlver);
  else              hd->xmlver = NULL;
  
  if(NULL!=xmlenc)  hd->xmlenc = mssStrdup(xmlenc);
  else              hd->xmlenc = NULL;

  hd->version= version;

  hd->flds    = mssInitFields();
  hd->recCnt = -1;

  return(hd);
}

/**
 * # FUNCTION #
 * mssHeader構造体のレコード数(recCnt)をセットする。
 */
void mssSetHeaderRecCnt(struct mssHeader *hd, int recCnt)
{
  hd->recCnt=recCnt;
}

/**
 * # SECTION #
 *----------------------------------------------------------------------------- 
 * mssFields,mssFldInfo構造体を扱う関数
 *----------------------------------------------------------------------------- 
 */

/**
 * # FUNCTION #
 * mssFields構造体の初期化
 */
struct mssFields *mssInitFields()
{
  struct mssFields *flds;
  flds=mssMalloc(sizeof(struct mssFields),"initFields");
  flds->fi=NULL;
  flds->cnt=0;
  return(flds);
}

/**
 * # FUNCTION #
 * mssFields構造体の領域開放
 */
void mssFreeFields(struct mssFields *flds)
{
  int i;
  if(flds!=NULL){
    for(i=0; i<flds->cnt; i++){
      mssFreeFldInfo(*(flds->fi+i));
    }
    mssFree(flds->fi);
  }
  mssFree(flds);
}

/**
 * # FUNCTION #
 * ヘッダー項目情報の出力(デバッグ用)
 */
void mssShowFields(struct mssFields *flds)
{
  int i;
  for(i=0; i<flds->cnt; i++){
    printf("fld[%d]:",i);
    mssShowFldInfo(*(flds->fi+i));
  }
}


/**
 * # FUNCTION #
 * mssUpdateSortPriority関数で利用するソートの比較関数
 */
static int cmpUpdHeader(const void **a, const void **b)
{
  if((int)*a > (int)*b) return(1);
  else                  return(-1);
}

/**
 * # FUNCTION #
 * mssFields構造体のソート情報を検査し、妥当でないソート情報を削除する。
 * この関数が使われる状況
 *   入力データの一部のヘッダー項目情報を新しいヘッダー項目情報として作成した時
 *   元のソートpriorityが正しくなくなる可能性がある。例えば、入力データに
 *   a,b,c,dの４つの項目があったとして、ソートのpriorityはa,b,cの順であったと
 *   する(a=1,b=2,c=3)。すなわち、項目a,b,cでソートされているとする。
 *   この時、a,c,dの３項目を新しい項目として抜きだしたとき、ソートのpriorityは
 *   正しくなくなる。項目aではソートされているが、cではソートされていない。
 *   そこで、この関数は、項目aのpriorityは1のままで、項目cのソートpriority
 *   を消去する。
 */
void mssUpdateFieldsSortPriority(struct mssFields *flds)
{
  int list[MssFieldMaxCnt];
  int listCnt;
  int i;
  int until;

  if(mssGV.txtFlg) return;

  /*ソートされている項目のpriority番号をlist配列にセット*/
  listCnt=0;
  for(i=0; i<flds->cnt; i++){
    if((*(flds->fi+i))->priority != 0){
      list[listCnt++]=(*(flds->fi+i))->priority;
    }
  }

  /*list配列を並べ変えて、priority番号が断絶or重複している箇所をuntilに入れる*/
  /*list[]=1,2,3,5 の時、3まではOKだがそれ以降は断絶。そこでuntil=3 */
  /*list[]=1,2,2,3 の時、最初の2まではOKだがそれ以降は重複。そこでuntil=2 */
  until=0;
  if(listCnt!=0){
/*
    qsort(list,listCnt,sizeof(int),
               (int (*)(const void *,const void *))cmpUpdHeader);
*/
    qsort5(list,listCnt,sizeof(int), (int (*))cmpUpdHeader);

    for(i=0; i<listCnt; i++){
      if((i+1)!=list[i]) break;
    }
    until=i;
  }

  /*断絶(until)より上のpriorityを消す*/
  for(i=0; i<flds->cnt; i++){
    if((*(flds->fi+i))->priority > until){
      (*(flds->fi+i))->priority = 0;
      (*(flds->fi+i))->revFlg   = 0;
      (*(flds->fi+i))->numFlg   = 0;
    }
  }
}

/**
 * # FUNCTION #
 * mssFldInfo構造体のソート情報(priority,revFlg,numFlg)をセットする。
 */
void mssSetFldInfoSort(struct mssFldInfo *fi, int prioriy, int revFlg,int numFlg)
{
  fi->priority = prioriy;
  fi->revFlg   = revFlg;
  fi->numFlg   = numFlg;
}

/**
 * # FUNCTION #
 * fldsのソート情報をクリアする。
 */
void mssClearFieldsSort(struct mssFields *flds)
{
  int i;

  if(mssGV.txtFlg) return;

  for(i=0; i<flds->cnt; i++){
    mssSetFldInfoSort(*(flds->fi+i),0,0,0);
  }
}

/**
 * # FUNCTION #
 * fldsのソート情報を一旦クリアし、sfの全ソート情報をコピーする。
 */
void mssSetFieldsSort(struct mssFields *flds, struct mssFields *sortFlds)
{
  int i;
  int no;

  /*一旦、flds上の全てのソート情報をクリアする*/
  mssClearFieldsSort(flds);

  for(i=0; i<sortFlds->cnt; i++){
    no = MssFlds2num(sortFlds,i);
    MssFlds2priority(flds,no)= i+1;
    MssFlds2revFlg(flds,no)  = MssFlds2revFlg(sortFlds,i);
    MssFlds2numFlg(flds,no)  = MssFlds2numFlg(sortFlds,i);
  }
}

/**
 * # FUNCTION #
 * fldsに含まれる全項目について、登録された順番にソート優先順位番号をふる。
 */
void mssSetFieldsSortPriority(struct mssFields *flds)
{
  int i;

  for(i=0; i<flds->cnt; i++){
    MssFlds2priority(flds,i)= i+1;
  }
}

/**
 * # FUNCTION #
 * mssFldInfo構造体の項目長情報(lenght)をセットする。
 * この情報は、固定長項目にする時にセットする。
 */
void mssSetFldInfoLength(struct mssFldInfo *fi, int length)
{
  fi->length = length;
}

/**
 * # FUNCTION #
 * mssFldInfo構造体の項目番号を、numによってセットする。
 * データ上では1から始まる番号だが、内部的には０から始まる番号。
 * そこでここでは、numには内部の番号として与えること。
 */
void mssSetFldInfoNum(struct mssFldInfo *fi, int num)
{
  fi->num      = num;
}

/**
 * # FUNCTION #
 * mssFldInfo構造体の項目名を、name によってセットする。
 */
void mssSetFldInfoName(struct mssFldInfo *fi, char *name)
{
  fi->name     = mssStrdup(name);
}

/**
 * # FUNCTION #
 * mssFldInfo構造体のコメントを、commentによってセットする。
 */
void mssSetFldInfoComment(struct mssFldInfo *fi, char *comment)
{
  fi->comment = mssStrdup(comment);
}

/**
 * # FUNCTION #
 * mssFields構造体の全FldInfo構造体のnum(項目番号)に、0から始まる連番にセット
 * する。
 */
void mssSetFieldsSequenceNum(struct mssFields *flds)
{
  int i;

  for(i=0; i<flds->cnt; i++){
    MssFlds2num(flds,i)=i;
  }
}

/**
 * # FUNCTION #
 * mssFields構造体に、文字列として指定された項目名を追加する。
 * ソート情報などはクリアされる。
 */
void mssAddFieldsByStr(struct mssFields *flds, char *str)
{
  flds->fi=mssRealloc(flds->fi,sizeof(struct mssFldInfo *)*(flds->cnt+1),
                        "AddFieldsByStr");
  *(flds->fi+flds->cnt) = mssInitFldInfo();

  mssSetFldInfoNum(*(flds->fi+flds->cnt),flds->cnt);
  mssSetFldInfoName(*(flds->fi+flds->cnt),str);
  mssSetFldInfoSort(*(flds->fi+flds->cnt),0,0,0);
  mssSetFldInfoLength(*(flds->fi+flds->cnt),0);
  mssSetFldInfoComment(*(flds->fi+flds->cnt),NULL);

  flds->cnt++;
}

/**
 * # FUNCTION #
 * mssFields構造体に、文字列リストとして指定された項目名を追加する。
 * ソート情報などはクリアされる。
 */
void mssAddFieldsByStrList(struct mssFields *flds, char **str, int cnt)
{
  int i;

  for(i=0; i<cnt; i++){
    mssAddFieldsByStr(flds,*(str+i));
  }
}

/**
 * # FUNCTION #
 * mssFields構造体に、指定されたmssFldInfoを追加する。
 */
void mssAddFieldsByFldInfo(struct mssFields *flds, struct mssFldInfo *fi)
{
  struct mssFldInfo *ofi;

  flds->fi=mssRealloc(flds->fi,sizeof(struct mssFldInfo *)*(flds->cnt+1),
                      "AddFldInfo");
  *(flds->fi+flds->cnt) = mssInitFldInfo();
  ofi=*(flds->fi+flds->cnt);

  mssSetFldInfoNum(ofi,fi->num);
  mssSetFldInfoName(ofi,fi->name);
  mssSetFldInfoSort(ofi,fi->priority,fi->revFlg,fi->numFlg);
  mssSetFldInfoLength(ofi,fi->length);
  mssSetFldInfoComment(ofi,fi->comment);

  flds->cnt++;
}

/**
 * # FUNCTION #
 * mssFields構造体fldsに、指定された複数のmssFldInfo構造体fiを追加する。
 * fiの最後の要素はNULLでなければならない。
 * 例えばmssFldGlbNam2Addで取得した複数のmssFldInfoリストは最後の要素はNULL
 * で終っているので、そのまま利用できる。
 */
void mssAddFieldsByFldInfoList(struct mssFields *flds, struct mssFldInfo **fi)
{
  int i;

  for(i=0;;i++){
    if( *(fi+i) == NULL ) break;
    mssAddFieldsByFldInfo(flds, *(fi+i));
  }
}

/**
 * # FUNCTION #
 * mssFields構造体に、指定されたmssFields構造体の全てのmssFldInfoを追加する。
 * 項目番号、項目名、ソート情報など全てが、そのまま追加コピーされる。
 */
void mssAddFieldsByFields(struct mssFields *flds, struct mssFields *addFlds)
{
  int i;

  for(i=0; i<addFlds->cnt; i++){
    mssAddFieldsByFldInfo(flds, *(addFlds->fi+i));
  }
}

/**
 * # FUNCTION #
 * mssFields構造体に、オプション上で指定されたmssFields構造体とヘッダー上の
 * mssFields構造体とを追加する。この時、オプションで指定されている項目は、
 * 新項目名を追加し、そうでない項目はヘッダーの項目を追加する。
 */
void mssAddHeadOrOptFields(struct mssFields *flds, struct mssHeader *hd, MssOptFLD *optFld)
{
  int i;
  int num;

  for(i=0; i<hd->flds->cnt; i++){
    mssAddFieldsByFldInfo(flds, *(hd->flds->fi+i));

    /*オプションで指定されなかった項目*/
    num=*(optFld->fldNo2optNo+i);
    if(*(optFld->fldNo2optNo+i)!=-1){
      mssSetFldInfoNum(*(flds->fi+flds->cnt-1), i);
      mssFree((*(flds->fi+flds->cnt-1))->name); /* 追加 2004/10/31 */
      mssSetFldInfoName(*(flds->fi+flds->cnt-1), *(optFld->newNam+num));
      mssSetFldInfoSort(*(flds->fi+flds->cnt-1),0,0,0);
      mssSetFldInfoLength(*(flds->fi+flds->cnt-1),0);
      mssSetFldInfoComment(*(flds->fi+flds->cnt-1), NULL);
    }
  }
}

/**
 * # SECTION #
 *----------------------------------------------------------------------------- 
 * mssHeader構造体の入出力関数
 *----------------------------------------------------------------------------- 
 */

/**
 * # FUNCTION #
 * 入力ファイルの項目名チェック
 */
static void chkInfFldName(struct mssHeader *hd, struct mssFPR *fp )
{
  char *fName;
  int i,j;

  /*ファイル名セット*/
  if(fp->fName==NULL) fName="stdin";
  else                fName=fp->fName;

  /*項目名NULLチェック*/
  for(i=0; i<hd->flds->cnt; i++){
    if(NULL==(*(hd->flds->fi+i))->name){
      mssShowErrMsg("name of %dth field is null in %s",i+1,fName);
      mssEnd(mssErrorNoDefault);
    }
  }

  /*項目名重複チェック*/
  for(i=0; i<hd->flds->cnt-1; i++){
    for(j=i+1; j<hd->flds->cnt; j++){
      if(0==strcmp((*(hd->flds->fi+i))->name,(*(hd->flds->fi+j))->name)){
        mssShowErrMsg("same field names in %s: %s",fName,(*(hd->flds->fi+j))->name);
        mssEnd(mssErrorNoDefault);
      }
    }
  }
}

/**
 * # FUNCTION #
 * 出力ファイルの項目名チェック
 */
static void chkOtfFldName(struct mssHeader *hd, struct mssFPW *fp )
{
  char *fName;
  int i,j;

  /*ファイル名セット*/
  if(fp->fName==NULL)           fName="stdout";
  else if(fp->fName==(char *)1) fName="stderr";
  else                          fName=fp->fName;

  /*項目名NULLチェック*/
  for(i=0; i<hd->flds->cnt; i++){
    if(NULL==(*(hd->flds->fi+i))->name){
      mssShowErrMsg("name of %dth field is null in %s",i+1,fName);
      mssEnd(mssErrorNoDefault);
    }
  }

  /*項目名重複チェック*/
  for(i=0; i<hd->flds->cnt-1; i++){
    for(j=i+1; j<hd->flds->cnt; j++){
      if(0==strcmp((*(hd->flds->fi+i))->name,(*(hd->flds->fi+j))->name)){
        mssShowErrMsg("same field names in %s: %s",fName,(*(hd->flds->fi+j))->name);
        mssEnd(mssErrorNoDefault);
      }
    }
  }
}


/**
 * # FUNCTION #
 * mssFldInfo構造体情報をヘッダー情報として書き出す。
 */
void mssWriteFldInfo( struct mssFldInfo *fi, int version, struct mssFPW *fp)
{
  struct mssXmlTag *fieldTag;
  struct mssXmlTag *sortTag;
  struct mssXmlTag *revTag;
  struct mssXmlTag *numTag;
  char *str;

  switch(version){
  /*XMLtable version 1.0*/
  case 10:
    /* <field no="1"> */
    fieldTag=mssInitXmlTag("field",NULL);
    mssAddXmlTagAttributeInt(fieldTag,"no",fi->num+1,NULL);
    str=mssXmlTag2startTag(fieldTag, NULL);
    mssWriteStr(str,fp); mssFree(str);

    /* <name>fieldName</name> */
    mssWriteXmlTagStr(0,"name",fi->name,0,NULL,fp);

    /* <sort priority="1"><numeric/><reverse/></sort>*/
    if(0 != fi->priority){
      sortTag=mssInitXmlTag("sort",NULL);
      mssAddXmlTagAttributeInt(sortTag,"priority",fi->priority,NULL);
      str=mssXmlTag2startTag(sortTag, NULL);
      mssWriteStr(str,fp); mssFree(str);
      if(fi->revFlg){
        revTag=mssInitXmlTag("reverse",NULL);
        str=mssXmlTag2emptyTag(revTag, NULL);
        mssWriteStr(str,fp); mssFree(str);
        mssFreeXmlTag(revTag);
      }
      if(fi->numFlg){
        numTag=mssInitXmlTag("numeric",NULL);
        str=mssXmlTag2emptyTag(numTag, NULL);
        mssWriteStr(str,fp); mssFree(str);
        mssFreeXmlTag(numTag);
      }
      str=mssXmlTag2endTag(sortTag, NULL);
      mssWriteStr(str,fp); mssFree(str);
      mssFreeXmlTag(sortTag);
    }

    /*<length>10</length>*/
    if(fi->length!=0){
      mssWriteXmlTagInt(0,"length",fi->length,0,NULL,fp);
    }

    /*</field>*/
    str=mssXmlTag2endTag(fieldTag, NULL);
    mssWriteStr(str,fp); mssFree(str);
    mssFreeXmlTag(fieldTag);
    mssWriteRet(fp);
    break;

    /*XMLtable version 1.1*/
    /*<field num="1" name="項目名" sort="1" numeric="1" reverse="1" length="10">項目コメント<field> */
  case 11:
    fieldTag=mssInitXmlTag("field",NULL);
    mssAddXmlTagAttributeInt(fieldTag,"no"  ,fi->num+1,NULL);
    mssAddXmlTagAttributeStr(fieldTag,"name",fi->name ,NULL);
    if(0 != fi->priority){
      mssAddXmlTagAttributeInt(fieldTag,"sort",fi->priority ,NULL);
      if(fi->numFlg){
        mssAddXmlTagAttributeInt(fieldTag,"numeric",1 ,NULL);
      }
      if(fi->revFlg){
        mssAddXmlTagAttributeInt(fieldTag,"reverse",1 ,NULL);
      }
    }
    if(fi->length!=0){
      mssAddXmlTagAttributeInt(fieldTag,"length",fi->length ,NULL);
    }
    mssWriteXmlStartTag(fieldTag, NULL, fp);
    if(fi->comment!=NULL){
      mssWriteStr(fi->comment,fp);
    }
    mssWriteXmlEndTag(fieldTag, NULL, fp);
    mssWriteRet(fp);
    mssFreeXmlTag(fieldTag);
    break;

  default:
    mssShowErrMsg("Internal Error: invalid XMLtable version");
    mssEnd(mssErrorNoDefault);
  }
}

/**
 * # FUNCTION #
 * ヘッダーの書き出し。
 */
void mssWriteHeader( struct mssHeader *hd, struct mssFPW *fp)
{
  struct mssXmlTag *xtTag;
  struct mssXmlTag *hdTag;
  int i;
  char *version=NULL;
  char verStr[2][4]={"1.0","1.1"};

  if(mssGV.txtFlg)return;

  switch(hd->version){
  case 10:
    version=verStr[0];
    break;
  case 11:
    version=verStr[1];
    break;
  default:
    mssShowErrMsg("Internal Error: invalid XMLtable version");
    mssEnd(mssErrorNoDefault);
  }

  chkOtfFldName(hd,fp); /*出力項目名のチェック*/

  mssWriteXmlDeclaration( hd->xmlver, hd->xmlenc ,fp);

  /*xmlTableヘッダーの書き出し<xmltbl version="1.1">*/
  xtTag=mssInitXmlTag("xmltbl",NULL);
  mssAddXmlTagAttributeStr(xtTag,"version",version,NULL);
  mssWriteXmlStartTag(xtTag,NULL,fp);
  mssFreeXmlTag(xtTag);
  mssWriteRet(fp);

  /*<header>*/
  hdTag=mssInitXmlTag("header",NULL);
  mssWriteXmlStartTag(hdTag,NULL,fp);
  mssWriteRet(fp);

  /*<title>*/
  mssWriteXmlTagStr(0,"title",hd->title,1,NULL,fp);

  /*<comment>*/
  mssWriteXmlTagStr(0,"comment",hd->comment,1,NULL,fp);

  /*<field>*/
  for(i=0; i<hd->flds->cnt; i++){    /*ヘッダーの項目タグを出力*/
    mssWriteFldInfo(*(hd->flds->fi+i), hd->version, fp);
  }

  /*<recCnt>*/
  if(-1 != hd->recCnt){
    mssWriteXmlTagInt(0,"recCnt",hd->recCnt,1,NULL,fp);
  }

  /*</header>*/
  mssWriteXmlEndTag(hdTag,NULL,fp);
  mssWriteRet(fp);
  mssFreeXmlTag(hdTag);

  /*<body><![CDATA[*/
  mssWriteStr(MssBeginBodyString, fp) ; mssWriteRet(fp);
  mssGV.writeHeaderFlg=1;
}

/**
 * # FUNCTION #
 * フッターの書き出し。
 */
void mssWriteFooter(struct mssFPW *fp)
{

  if(fp==NULL) return;
  if(mssGV.txtFlg) return;
  mssWriteStr(MssEndBodyString, fp) ; mssWriteRet(fp);
  mssWriteStr("</xmltbl>", fp)  ; mssWriteRet(fp);

  mssGV.writeFooterFlg=1;
}

/**
 * # FUNCTION #
 * 空データのxmlTableを作成する。
 * 与えられたfldCnt個の項目名fldNameから、xmlTableヘッダーの項目を作成する。
 * この関数は、xtmkdata,xtcalenderコマンドのように、新しくデータを作成する際に
 * 利用される。既に必要な項目名を伴った空データがあれば、入力データを伴った
 * 処理フローがそのまま利用可能となる。
 */
void mssCreateBlankXtFile(char **fldName, int fldCnt, char *fname)
{
  struct mssHeader *hd;
  struct mssFPW *fpw;
  hd=mssMalloc( sizeof(struct mssHeader),"mkBlankXtFile" );

  hd->title  = NULL;
  hd->comment= NULL;
  hd->xmlver = mssStrdup(MssXmlDefVer);
  hd->xmlenc = mssStrdup(MssXmlDefEnc);
  hd->version= MssXtDefVer;
  hd->flds   = mssInitFields();
  hd->recCnt = -1;

  mssAddFieldsByStrList(hd->flds,fldName,fldCnt);

  /*標準出力オープン+ヘッダー、フッターの出力*/
  fpw=mssOpenFPW(fname, 0, 0);
  mssWriteHeader(hd, fpw);
  mssWriteFooter(fpw);
  mssCloseFPW(fpw);
  mssFreeHeader(hd);
}



static char numChr[MssFieldMaxCnt][4]; /*項目番号としての項目名用*/

/**
 * # FUNCTION #
 * ヘッダー情報の読み込み。
 */
struct mssHeader *mssReadHeader( struct mssFPR *fp)
{

  struct mssHeader *hd;
  struct mssStrings *head;
  struct mssStrings *title;
  struct mssStrings *comment;
  struct mssStrings *recCnt;
  struct mssStrings *field;
  struct mssStrings *sort;

  char *tmpStr;
  char *pos;
  char *str;
  int i;
  int datFldCnt;  /*データ上の項目数*/
  int fldNum;
  int priority;
  int revFlg;
  int numFlg;
  int length;
  hd=mssMalloc( sizeof(struct mssHeader),"readHeader" );
  hd->flds=mssInitFields();

  /*---------------------------- plainTextの場合*/
  if(mssGV.txtFlg){
    datFldCnt = mssGetFldCntOnData(fp); /*データ上の項目数をカウント*/
    hd->title   = NULL;        /*タイトル*/
    hd->comment = NULL;        /*コメント*/
    hd->recCnt  = -1;          /*レコード数*/
    hd->xmlver  = NULL;        /*xml version*/
    hd->xmlenc  = NULL;        /*xml encoding*/
    /*項目のメモリの確保*/
    hd->flds->cnt = 0;   /*項目数*/
    for(i=0; i<datFldCnt; i++){
      sprintf(numChr[i+1],"%d",i+1);
      /*mssAddFieldsByStr(hd->flds, mssStrdup(numChr[i+1]));*/
      mssAddFieldsByStr(hd->flds, numChr[i+1]); /* 変更 2004/10/31 */
    }
  /*---------------------------- xmlTblの場合*/
  }else{

    mssReadSetTop(hd, fp); /*xmlのバージョンとencもセットする*/

    /*全ヘッダー(<header>....</header>)をファイルからメモリにセット*/
    head=mssReadTag("header", fp);
    if(head==NULL){
      mssShowErrMsg("cannot find header tag");
      mssEnd(mssErrorNoDefault);
    }

    /*タイトルの取得*/
    title=mssGetTag("title", head->str, &pos);
    if(title!=NULL){
      hd->title=mssGetTagCont(title->str,"title",1); /*titleタグ内の内容を取得*/
      mssFreeStrings(title);
    }else{
      hd->title=NULL;
    }

    /*コメントの取得*/
    comment=mssGetTag("comment", head->str, &pos);
    if(comment!=NULL){
      hd->comment=mssGetTagCont(comment->str,"comment",0);
      mssFreeStrings(comment);
    }else{
      hd->comment=NULL;
    }

    /*レコード数の取得*/
    hd->recCnt=-1;
    recCnt=mssGetTag("recCnt", head->str, &pos);
    if(recCnt!=NULL){
      str=mssGetTagCont(recCnt->str,"recCnt",1);
      if( NULL!= str ) hd->recCnt=atoi(str);
      mssFree(str);
      mssFreeStrings(recCnt);
    }else{
      hd->recCnt=-1;
    }

    /*項目情報の取得*/
    tmpStr=head->str;
    while(1) {
      /*fieldタグのセット*/
      field=mssGetTag("field", tmpStr, &pos);
      if( NULL == field ){
         break;
      }

      /*検索対象を一文字ずらすことで、次のfieldの検索に備える*/
      tmpStr=pos+1;

      switch(hd->version){
      case 10:
        /*項目名*/
        str = mssGetTagCont(field->str, "name",1);
        if(str!=NULL){
          mssAddFieldsByStr(hd->flds, str);
          mssFree(str);
        }else{
          mssShowErrMsg("Internal Error: invalid filed number");
          mssEnd(mssErrorNoDefault);
        }

        /*項目番号*/
        str = mssGetTagAtt(field->str, "field", "no");
        if(str!=NULL){
          fldNum=atoi(str);
          if(fldNum>0 && fldNum<MssFieldMaxCnt){
            mssSetFldInfoNum(*(hd->flds->fi+hd->flds->cnt-1), fldNum-1);
          }else{
            mssShowErrMsg("Internal Error: invalid filed number");
            mssEnd(mssErrorNoDefault);
          }
          mssFree(str);
        }else{
          mssShowErrMsg("Internal Error: invalid filed number");
          mssEnd(mssErrorNoDefault);
        }

        /*sortタグのセット (このタグがなければ0がセットされる)*/
        priority = 0;
        revFlg   = 0;
        numFlg   = 0;
        sort=mssGetTag("sort", field->str, &pos);
        if( NULL != sort ){
          str = mssGetTagAtt(sort->str, "sort", "priority");
          priority = atoi(str);
          revFlg   = mssGetNullTag(sort->str, "reverse");
          numFlg   = mssGetNullTag(sort->str, "numeric");
          mssFree(str);
          mssFreeStrings(sort);
        }
        mssSetFldInfoSort(*(hd->flds->fi+hd->flds->cnt-1),
                                                      priority,revFlg,numFlg);

        /*lenghtタグのセット*/
        str = mssGetTagCont(field->str, "length",1);
        if(str!=NULL) length = atoi(str);
        else          length = 0;
        mssFree(str);
        mssSetFldInfoLength(*(hd->flds->fi+hd->flds->cnt-1),length);
        break;

      case 11:
        /*項目名*/
        str = mssGetTagAtt(field->str, "field", "name");
        if(str!=NULL){
          mssAddFieldsByStr(hd->flds, str);
          mssFree(str);
        }else{
          mssShowErrMsg("Internal Error: invalid filed number");
          mssEnd(mssErrorNoDefault);
        }

        /*項目番号*/
        str = mssGetTagAtt(field->str, "field", "no");
        if(str!=NULL){
          fldNum=atoi(str);
          if(fldNum>0 && fldNum<MssFieldMaxCnt){
            mssSetFldInfoNum(*(hd->flds->fi+hd->flds->cnt-1), fldNum-1);
          }else{
            mssShowErrMsg("Internal Error: invalid filed number");
            mssEnd(mssErrorNoDefault);
          }
          mssFree(str);
        }else{
          mssShowErrMsg("Internal Error: invalid filed number");
          mssEnd(mssErrorNoDefault);
        }

        /*ソート情報*/
        priority = 0;
        revFlg   = 0;
        numFlg   = 0;
        str = mssGetTagAtt(field->str, "field", "sort");
        if(str!=NULL){
          priority=atoi(str);
          mssFree(str);
          str = mssGetTagAtt(field->str, "field", "numeric");
          if(str!=NULL){
            numFlg=atoi(str);
            if(numFlg!=0 && numFlg!=1){
              mssShowErrMsg("Internal Error: invalid numeric value");
              mssEnd(mssErrorNoDefault);
            }
            mssFree(str);
          }

          str = mssGetTagAtt(field->str, "field", "reverse");
          if(str!=NULL){
            revFlg=atoi(str);
            if(revFlg!=0 && revFlg!=1){
              mssShowErrMsg("Internal Error: invalid reverse value");
              mssEnd(mssErrorNoDefault);
            }
            mssFree(str);
          }
          mssSetFldInfoSort(*(hd->flds->fi+hd->flds->cnt-1),
                                                      priority,revFlg,numFlg);
        }

        /*lenghtタグのセット*/
        str = mssGetTagAtt(field->str, "field","length");
        if(str!=NULL) length = atoi(str);
        else          length = 0;
        mssFree(str);
        mssSetFldInfoLength(*(hd->flds->fi+hd->flds->cnt-1),length);
        break;

        /*コメント*/
        str=mssGetTagCont(field->str,"field",0);
        mssSetFldInfoComment(*(hd->flds->fi+hd->flds->cnt-1), str);
        mssFree(str);
        break;

      default:
        mssShowErrMsg("Internal Error: invalid XMLtable version");
        mssEnd(mssErrorNoDefault);
      }
      mssFreeStrings(field);
    }
    mssFreeStrings(head);

    mssSkipToBody(fp);
  }

  chkInfFldName(hd,fp); /*入力項目名のチェック*/
  return(hd);
}

/**
 * # FUNCTION #
 * ヘッダー情報の出力(デバッグ用)
 */
void mssShowHeader(struct mssHeader *hd)
{
  printf("====================================\n");
  printf("         Header Information         \n");
  printf("====================================\n");
  printf("XML Table version : %d\n",hd->version);
  printf("title : %s\n",hd->title);
  printf("comment : %s\n",hd->comment);
  printf("fldCnt : %d\n",hd->flds->cnt);
  mssShowFields(hd->flds);
}

/**
 * # FUNCTION #
 * mssFields構造体に格納されているソート情報とheaderのソート情報を比較し
 * mssFieldsのソート情報から見て全て一致していればソート済とみなす。
 * numeric,reverseなしで、ソートの優先順位が一致していれば1を返す。
 * numeric,reverseありで、ソートの優先順位が一致していれば2を返す。
 * mssFields(sf)に同じ項目が２つ以上使われていればエラー終了する
 */
int mssChkSorted( struct mssFields *sf, struct mssHeader *hd)
{

  int i;

  /*キーが指定されていない場合(same,diff)はソートされているとみなす*/
  if(sf->cnt<1) return(1);

  /*テキストの場合はソートされているとみなす*/
  if(mssGV.txtFlg) return(1);

  /*sfに同じ項目がないかどうかチェックする*/
  for(i=0; i<sf->cnt-1; i++){
    if( (*(sf->fi+i))->num == (*(sf->fi+i+1))->num ){
      mssShowErrMsg("same field name is used two or more times on -k(-s)");
      mssEnd(mssErrorNoDefault);
    }
  }

  /*sfの各項目とhdの各項目を比較する*/
  for(i=0; i<sf->cnt; i++){
    if(mssFldNum2SrtRev(hd->flds,(*(sf->fi+i))->num) != (*(sf->fi+i))->revFlg){
      return(0);
    }
    if(mssFldNum2SrtNum(hd->flds,(*(sf->fi+i))->num) != (*(sf->fi+i))->numFlg){
      return(0);
    }
    if(mssFldNum2SrtPri(hd->flds,(*(sf->fi+i))->num)-1 != i ) {
      return(0);
    }
  }

  /*次にnumeric,reverseが指定されているかを検査*/
  for(i=0; i<sf->cnt; i++){
    if( (*(sf->fi+i))->revFlg) return(2);
    if( (*(sf->fi+i))->numFlg) return(2);
  }
  return(1);
}

/**
 * # SECTION #
 *----------------------------------------------------------------------------- 
 * MssOptFLD,OptKey 関連関数
 *----------------------------------------------------------------------------- 
 */

/**
 * # FUNCTION #
 * 入力データの何番目の項目が、オプションとして指定された何番目の項目かについて
 * の情報を返す。
 *   ex.) 
 *     入力データ項目名:a  b  c  d  e
 *     -f c,e,a の時
 *     配列{2,-1,0,-1,1}を返す。
 */
static int *setFldNo2optNo(struct mssFields *flds, int headFldCnt)
{
  int i,j;
  int *fldNo2optNo;

  fldNo2optNo=mssMalloc(sizeof(int)*headFldCnt,"setFldNo2optNo");
  for(i=0; i<headFldCnt; i++){
    *(fldNo2optNo+i)=-1;
  }
  j=0;
  for(i=0; i<flds->cnt; i++){
    *(fldNo2optNo+ MssFlds2num(flds,i))=j++;
  }
  return(fldNo2optNo);
}

/**
 * # FUNCTION #
 * 新項目名を作成し、その文字列を返す。
 * 通常、新項目名は "-f nam:newNam"の形式で指定される。
 * このとき、namとnewNameを引数にして新しい項目名を生成し、その文字列を返す
 * この関数では次の事項に基づいて、新項目名を生成する。
 * 1. newNamに'&'文字が含まれていれば、その文字をnamに置換する。
 * 2. 1で置換した結果新項目名の長さがMssFieldMaxLenを超えるとエラー&停止
 * 3. newNamがNULLならば、namを返す。
 * 4. 新項目名のために新たな文字列領域を確保するので、nam,newNameとの干渉
 *    はない。
 *   ex.) 
 *     newNam = NULL        , nam="Jan" -> return("Jan")
 *     newNam = "amount(&)" , nam="Jan" -> return("amount(Jan)")
 */
static char *setNewFldNam(char *newNam,char *nam)
{
  char retStr[MssFieldMaxLen]; /*展開される新しい項目名の一時バッファ*/
  char *namPnt;
  int  i;

  i=0;
  if(newNam==NULL){
    namPnt=nam;
    while(*namPnt!='\0'){
      retStr[i++]=*namPnt++;
      if(i>=MssFieldMaxLen-1){
        mssShowErrMsg("new field name exceed maximum length:%s",nam);
        mssEnd(mssErrorNoDefault);
      }
    }

  }else{
    while(*newNam!='\0'){
      if(*newNam=='&'){
        namPnt=nam;
        while(*namPnt!='\0'){
          retStr[i++]=*namPnt++;
          if(i>=MssFieldMaxLen-1){
            mssShowErrMsg("new field name exceed maximum length:%s",nam);
            mssEnd(mssErrorNoDefault);
          }
        }
        newNam++;
      }else{
        retStr[i++]=*newNam++;
        if(i>=MssFieldMaxLen-1){
          mssShowErrMsg("new field name exceed maximum length:%s",nam);
         mssEnd(mssErrorNoDefault);
        }
      }
    }
  }
  retStr[i]='\0';
  return(mssStrdup(retStr));
}

/**
 * # FUNCTION #
 * MssOptFLD構造体についてoptFld->namに指定されたワイルドカードを評価し、
 * optFld->nam,newNam,fldOptを更新する。そのときワイルドカードの評価対象は
 * flds項目とする。
 * オプションをセットする関数mssSetOptionでは、ワイルドカードの展開はしない。
 * そこでこの関数にてワイルドカード評価対象項目を指定して初めて展開が可能と
 * なる。
 *   ex.)
 *    flds->nam={"customer","date","amount1","amount2"}
 *    optFld->flds->nam={"amount*"}
 *    -->> optFld->flds->nam={"amount1","amount2"}
 */
static void evalOptFldNameGlob(MssOptFLD *optFLD, struct mssFields *flds)
{
  struct mssFldInfo **fiList;
  char **name=NULL;
  char **newName=NULL;
  char **fldOpt=NULL;

  int fiCnt=0;
  int cnt=0;
  int i;

  for(i=0 ; i<optFLD->cnt ; i++) { /*オプションで指定された各項目毎に*/
    /*WCにマッチする項目リストがfiListに代入される(最後はNULLで終る)*/
    fiCnt=0;

    /*glob指定OKなら展開する*/
    if(optFLD->globFlg) {
      fiList = mssFldGlbNam2Add(flds,*(optFLD->nam+i));

    /*glob指定NGなら単に項目名のAddressを取得する*/
    }else{
      fiList = mssCalloc(sizeof(struct mssFldInfo *) * 2,"evalGlob");
      *fiList = mssFldNam2Add(flds,*(optFLD->nam+i));
    }

    while( *(fiList+fiCnt) != NULL ){

      /*ワイルドカード展開後、optFld->maxCntを越えたらエラーのチェック*/
      if(cnt>=optFLD->maxCnt){
        mssShowErrMsg("the max number of fields in the option is %d: -%s %s",
          optFLD->maxCnt,optFLD->keyWord,optFLD->str);
        mssEnd(mssErrorNoDefault);
      }

      /*項目名*/
      name=mssRealloc(name,sizeof(char *)*(cnt+1), "evalOptFldNameGlob");
      *(name+cnt) = mssStrdup((*(fiList+fiCnt))->name);

      /*新項目名*/
      newName=mssRealloc(newName,sizeof(char *)*(cnt+1), "evalOptFldNameGlob");
      *(newName+cnt)=setNewFldNam(*(optFLD->newNam+i),*(name+cnt));

      /*項目オプション*/
      fldOpt=mssRealloc(fldOpt,sizeof(char *)*(cnt+1), "evalOptFldNameGlob");
      *(fldOpt+cnt)=mssStrdup(*(optFLD->fldOpt+i));

      fiCnt++;
      cnt++;
    }
    mssFree(fiList);
  }

  for(i=0; i<optFLD->cnt; i++){
    mssFree(*(optFLD->nam+i)); /* 追加 2004/10/28 */
  }
  mssFree(optFLD->nam   );
  mssFree(optFLD->newNam);
  mssFree(optFLD->fldOpt);

  optFLD->nam   =name;
  optFLD->newNam=newName;
  optFLD->fldOpt=fldOpt;
  optFLD->cnt   =cnt;
}

/**
 * # FUNCTION #
 * optFLD構造体の項目をヘッダーの項目と突き合わせて以下の３つの設定をおこなう
 * 1.MssOptFLDで指定された項目名(nam)に含まれるワイルドカードを評価し、
 *   optFld->nam,newNam,fldOptを更新する。
 * 2.optFLD->fldsに、ヘッダーの対応する項目情報をコピーする。
 * 3.ヘッダーの項目番号->オプションの指定位置の配列を設定。
 *   ex.) -f customer%r,date:DATE%rn,amt*:new(&)
 *             optFLD->nam   ={"customer","date","amt*"}
 *             optFLD->newNam={NULL,"DATE","new(&)"}
 *             optFLD->fldOpt={"r","rn",NULL}
 *        入力データのヘッダー={"customer","class","date","time","amt1","amt2"}
 *           -->>
 *        optFLD->nam   ={"customer,"date","amt1","amt2"}
 *        optFLD->newNam={"customer,"DATE","new(amt1)","new(amt2)"}
 *        optFLD->fldOpt={"r","rn",NULL,NULL}
 *        optFLD->fldNo2optNo={0,-1,1,-1,2,3}
 *        optFLD->flds->fi
 *                 ->num ={0,2,4,5}
 *                 ->name={"customer","date","amt1","amt2"}
 */
void mssSetOptFld(MssOptFLD *optFLD, struct mssHeader *hd)
{
  int i;
  int rev,num;

  optFLD->flds=mssInitFields();

  /*ワイルドカードの評価*/
  evalOptFldNameGlob(optFLD, hd->flds);

  /*optFLDで指定された項目名のヘッダー項目へのポインタを登録*/
  for(i=0 ; i<optFLD->cnt ; i++) {
    mssAddFieldsByFldInfo(optFLD->flds, mssFldNam2Add(hd->flds,*(optFLD->nam+i)));

    /*新項目名があれば登録*/
    if(optFLD->newFlg){
      if(0 != strcmp(*(optFLD->nam+i), *(optFLD->newNam+i)) ){
        mssFree((*(optFLD->flds->fi+optFLD->flds->cnt-1))->name); /* 追加2004/10/31 */
        mssSetFldInfoName(*(optFLD->flds->fi+optFLD->flds->cnt-1),
                          *(optFLD->newNam+i));
      }
    }

    /* 昇順／降順、文字順／数値順のフラグを設定 */
    rev=mssIsFldOptOn(optFLD,i,'r');
    num=mssIsFldOptOn(optFLD,i,'n');
    mssSetFldInfoSort(*(optFLD->flds->fi+i),0,rev,num);
  }

  /*項目番号flagの設定*/
  optFLD->fldNo2optNo=setFldNo2optNo(optFLD->flds,hd->flds->cnt);
}

/**
 * # FUNCTION #
 * MssOptKEY構造体の項目をヘッダーの項目と突き合わせて以下の3つのことをおこなう
 * 1.optFLD->fldsに、ヘッダーの対応する項目情報をコピーする。
 * 2.ヘッダーの項目番号->オプションの指定位置の配列を設定。
 * 3.同一項目を複数指定していればエラーで終了する
 *   ex.) -f customer,date
 *             optFLD->nam   ={"customer","date"}
 *        入力データのヘッダー={"customer","class","date","time","amt1","amt2"}
 *           -->>
 *        optFLD->fldNo2optNo={0,-1,1,-1,-1,-1}
 *        optFLD->flds->fi
 *                 ->num ={0,2}
 *                 ->name={"customer","date"}
 */
void mssSetOptKey(MssOptKEY *optKEY, struct mssHeader *hd)
{
  int i,j;

  optKEY->flds=mssInitFields();

  /*optKEYで指定された項目名のヘッダー項目へのポインタを登録*/
  for(i=0 ; i<optKEY->cnt ; i++) {
    mssAddFieldsByFldInfo(optKEY->flds, mssFldNam2Add(hd->flds,*(optKEY->nam+i)));
    mssSetFldInfoSort(*(optKEY->flds->fi+i),0,0,0);
  }

  /*optKEYで同じ項目が指定されたらエラー*/
  for(i=0 ; i<optKEY->cnt-1 ; i++) {
    for(j=i+1 ; j<optKEY->cnt ; j++) {
      if( MssFlds2num(optKEY->flds,i)==MssFlds2num(optKEY->flds,j) ){
        mssShowErrMsg("can not specify a same field in key field(s)");
        mssEnd(mssErrorNoDefault);
      }
    }
  }

  /*項目番号flagの設定*/
  optKEY->fldNo2optNo=setFldNo2optNo(optKEY->flds,hd->flds->cnt);
}
