/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるXMLtableデータの入力関連の関数
 * 20040123 mssReadFRK関数のメモリ初期化バグ修正
 * 20040519 -k -q 同時指定時のバグ修正
 * 20040803 数値順or逆順ソートを示すフラグCmpRevNumをmssSortDat構造体に移す。
 *          例えばTraとMstで、それぞれ通常ソート、数値ソートをしたとすると、
 *          これまでグローバルで持っていたCmpRevNumでは対処できなくなるため。
 * 20040824 ファイル名によりgzopenとfopenを使いわける仕様に変更
 * 20041031 メモリリーク解決
 * ============================================================================
 */

#include <musashi/mssHeader.h>
#include <musashi/mssXtTag.h>
#include <musashi/mssInput.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <math.h>
#include <errno.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <float.h>
#include <zlib.h>

#include <fnmatch.h>
#include <sys/time.h>

/*quick sort関連*/
void qsort5(char *base, int nel, int size, int (*cmp));
unsigned int ass_cnt;

/**
 * # MACRO #
 * リングバッファにおいて、指定のキュー(a)の次のキュー番号を計算する。
 * bにはキューの数を指定する。bは２の累乗でなければならない。
 */
#define NextQue(a,b) ((a+1)&(b-1))

/**
 * # MACRO #
 * リングバッファにおいて、指定のキュー(a)の前のキュー番号を計算する。
 * bにはキューの数を指定する。bは２の累乗でなければならない。
 */
#define PrevQue(a,b) ((a-1)&(b-1))

/* ############ グローバル変数 ##############*/
/*ファイルからの読み込みがEOFとなったときの番兵*/
/*実体ではなくアドレスとして利用している。*/
static char readEnd[2]={0xff,0};

/* qsort5で一時的に利用する sd->cmpRevNumがセットされる */
static struct mssSortDat *sdTmp;

/*一時ファイル名*/
static char TFname[MssFileNameMaxLen];

extern struct mssGlobalVariables mssGV;

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * ソート関連
 * ----------------------------------------------------------------------------
 * あるコマンドでxmlTableを処理する際に事前にソートすることが必要な時、
 * mssReopenFPR関数を呼び出すだけで、裏で時動的に並べ換えることができる。
 * その際に裏で実行されるソート関連の関数。全てstatic変数で、ライブラリ利用者
 * はこれらの関数を意識する必要はない。
 * mssReopenFPR関数が呼び出されると、大まかに以下に示すような処理が行われる。
 * 1. 入力データをある一定件数読み込む
 * 2. その部分データをquickソートする
 * 3. 入力データを全て読み込むまで1,2の処理を繰り返し、
 *    結果としてn個のソート済ファイルが出来上がる。
 * 4. n個のファイルを、ファイル数が"PWayS"個以下になるまでマージする。
 *    一回のマージでPWayS個のファイルをマージする。
 *    その際、トーナメントツリーをプライオリティキューとして利用する。
 * 上記の処理を行った後、ユーザは、mssReadFldRecなどのデータ読み込み関数を利用
 * でき、ソートを意識せず、通常ファイルと全く同様に扱うことができるようになる。
 *
 * 【トーナメントツリーの考え方】
 * Algorithm by N.Katoh in Maui 2001/06/16-22
 *
 * 下図では、６つのファイルをマージすることを想定している。
 * 1. リーフには各ファイルのカレント行のキーの値がセットされる(図a)。
 *    初期状態ではノードには値はセットされていない。
 * 2. 次に、トーナメント戦を行い、優勝者を決める(図b)。
 * 3. 優勝者のデータを書き出す。
 * 4. 優勝者のリーフに次の行のキー値をセットする(図c)。
 * 5. 新たにセットされたリーフのみトーナメント戦を行い優勝者を決める(図d)。
 * 6. 全データを読み切るまで3-5の処理を繰り返す。
 *
 *        ( )             (a)             (a)             (b)
 *     +---+---+       +---+---+       +---+---+       +---+---+
 *     |       |       |       |       |       |       |       |
 *    ( )     ( )     (a)     (b)     (a)     (b)     (e)     (b)
 *   +-+-+   +-+-+   +-+-+   +-+-+   +-+-+   +-+-+   +-+-+   +-+-+
 *   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |
 *   g   a   c   b   g   a   c   b   g  [e]  c   b   g   e   c   b
 *       [図a]           [図b]           [図d]           [図d]
 */

/**
 * # FUNCTION #
 * キー比較ルーチン
 * sd->cmpRevNumがOnの時はs1,s2は(char **)にキャストして利用している。
 * s1が勝ちの場合-1 s2が勝ちの場合1を返す 同点の場合は0を返す
 * トーナメントツリーにおいて利用される。
 */
static int cmpKeyStr(char *s1, char *s2, struct mssSortDat *sd)
{
  int i;
  int no; /*項目番号*/
  int cmp; /*strcmpの結果*/
  double n1,n2;

  if(!sd->cmpRevNum){
    /*通常の場合はstrcmpの結果が勝ち負けと等しくなる*/
    /*s1<s2 -> -1 , s1>s2 -> 1, s1==s2 -> 0*/
    return(strcmp(s1,s2));
  }else{
    if(s1==readEnd) return(1);  /*s1が終了ならs2の勝ち*/
    if(s2==readEnd) return(-1); /*s2が終了ならs1の勝ち*/
    for(i=0; i<sd->flds->cnt; i++){
      no=MssFlds2num(sd->flds,i);
      if(MssFlds2numFlg(sd->flds,i)){
        if(MssIsNull(*((char **)s1+no))) n1=-DBL_MAX;
        else                         n1=atof(*((char **)s1+no));
        if(MssIsNull(*((char **)s2+no))) n2=-DBL_MAX;
        else                         n2=atof(*((char **)s2+no));

        if(n1 > n2){
          /*数値大きい順の時 n1>n2 -> n1の勝ち*/
          if(MssFlds2revFlg(sd->flds,i)) return(-1);
          /*数値小さい順の時 n1>n2 -> n2の勝ち*/
          else                          return(1);
        }else if(n1 < n2){
          /*数値大きい順の時 n1<n2 -> n2の勝ち*/
          if(MssFlds2revFlg(sd->flds,i)) return(1);
          /*数値小さい順の時 n1<n2 -> n1の勝ち*/
          else                           return(-1);
        }
      }else{
        /*文字の場合はstrcmpの結果が勝ち負けと等しくなる*/
        if(0 != (cmp=strcmp( *((char **)s1+no),*((char **)s2+no))) ){
          if(MssFlds2revFlg(sd->flds,i)) return(cmp*(-1));
          else                          return(cmp);
        }
      }
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * キー比較ルーチン(quickソート関数用)
 * sd->cmpRevNumがOnの時はs1,s2は(char **)にキャストして利用している。
 * s1が勝ちの場合-1 s2が勝ちの場合1を返す 同点の場合は0を返す
 */
static int cmpKeyQst(const void **s1,const void **s2)
{
  register int i,no,cmp;
  double n1,n2;

  if(!sdTmp->cmpRevNum){
    return (strcmp( *s1,*s2));
  }else{
    for(i=0; i<sdTmp->flds->cnt; i++){
      no=MssFlds2num(sdTmp->flds,i);
      if(MssFlds2numFlg(sdTmp->flds,i)){
        if(MssIsNull(*((char **)s1+no))) n1=-DBL_MAX;
        else                         n1=atof(*((char **)s1+no));
        if(MssIsNull(*((char **)s2+no))) n2=-DBL_MAX;
        else                         n2=atof(*((char **)s2+no));
        if(n1 > n2){
          /*数値大きい順の時 n1>n2 -> n1の勝ち*/
          if(MssFlds2revFlg(sdTmp->flds,i)) return(-1);

          /*数値小さい順の時 n1>n2 -> n2の勝ち*/
          else                    return(1);
        }else if(n1 < n2){
          /*数値大きい順の時 n1<n2 -> n2の勝ち*/
          if(MssFlds2revFlg(sdTmp->flds,i)) return(1);
          /*数値小さい順の時 n1<n2 -> n1の勝ち*/
          else                    return(-1);
        }
      }else{
        /*文字の場合はstrcmpの結果が勝ち負けと等しくなる*/
        if(0 != (cmp=strcmp( *((char **)s1+no),*((char **)s2+no))) ){
          if(MssFlds2revFlg(sdTmp->flds,i)) return(cmp*(-1));
          else                    return(cmp);
        }
      }
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * トーナメントツリー(ヒープ)の初期化
 * トーナメントツリー(以下TT)の各ノードはTTnode構造体。
 * リーフの下図を引数cntで与え、TTを生成し、そのトップノードを返す。
 * TTの構造は以下の通り。
 *
 *                   a(1)
 *           +-------+--------+     
 *           |                |     
 *           a(2)             k(3)  
 *     +-----+------+     +---+--+   
 *     |            |     |      |   
 *     a(4)         b(5)  z(6)   k(7)
 *   +--+--+     +--+--+
 *   |     |     |     |
 *   d(8)  a(9)  c(10) b(11)
 * 親子兄弟の関係
 * 本人の番号 = n とすると
 *   親の番号   = int(n/2)  ex) 5の親=int(5/2)=2 8の親=int(8/2)=4
 *   兄弟の番号 = もしnが偶数ならば n+1 基数ならば n-1
 * トーナメントはリーフから比較していくので、子から親をたどることとなる。
 * よって親から見た子どもの番号は知っている必用はない。
 */
static struct TTnode *TTinit(int cnt)
{
  int i;

  struct TTnode *top; /*ツリーのトップとなるアドレス*/

  /*ノードとキーのメモリ確保*/
  top     =mssMalloc(sizeof(struct TTnode)*(2*cnt),"ttInitNode");
  top->key=mssMalloc(sizeof(struct TTkey) *(2*cnt),"ttInitKey");

  /*トップノード*/
  (top+1)->parent   = NULL;
  (top+1)->brother  = NULL; 
  (top+1)->num      = 1;
  (top+1)->key      = top->key+1;
  (top+1)->key->str = NULL;
  (top+1)->key->bkt = 0;

  for(i=2; i<=2*cnt-1; i++){
    /*--------初期化*/
    (top+i)->num      = i;
    (top+i)->key      = top->key+i;
    (top+i)->key->str = NULL;
    (top+i)->key->bkt = 0;

    /*--------親ノード*/
    (top+i)->parent = top+i/2;

    /*--------兄弟ノード*/
    if(i==i/2*2){ /*偶数*/
      (top+i)->brother=top+i+1; 
    }else{        /*奇数*/
      (top+i)->brother=top+i-1; 
    }
  }

  return(top);
}

/**
 * # FUNCTION #
 * TTnode構造体のメモリ領域開放*
 * TTツリーのトップとなるアドレスを引数で与える
 */
static void TTfree(struct TTnode *top)
{
  mssFree(top->key);
  mssFree(top);
}

/**
 * # FUNCTION #
 *トーナメントツリーのリーフにデータをセットする。
 *  ポインタ操作を行うだけで、データの実体は別の場所にあることが前提。
 */
static void TTsetLeaf(
  struct TTkey  *key, /*セットされる文字列などの情報*/
  struct TTnode *node,/*トップノード*/
  int num){           /*ツリー全体で何番目のノードにセットするか*/

  (node+num)->key->bkt=key->bkt;
  (node+num)->key->str=key->str;
}

/**
 * # FUNCTION #
 *トーナメントツリーに新しいデータを入れる
 *  insertとあるが実際には既存のあるリーフ(num)に新しいデータをセットし、
 *  そのリーフから始まる試合を行うだけで、新しいリーフが挿入されるのではない。
 *  新しいリーフが負けた時点で、それ以上試合を行うことは無意味であり、
 *  試合を行わないことにより、弱冠の高速化が期待できるが、今回は未実装。
 */
static void TTinsert(
  struct TTkey  *key, /*セットされる文字列などの情報*/
  struct TTnode *node,/*トップノード*/
  int num,            /*ツリー全体で何番目のノード(リーフ)にセットするか*/
  struct mssSortDat *sd){

  TTsetLeaf(key,node,num);

  while(num>=2){
    /*試合を行い、小さい方が勝ち(親ノードに登録する)。*/
    /*左が勝ち(-1) 右が勝ち(0or1)*/
    if( 0>cmpKeyStr((node+num)->key->str, (node+num)->brother->key->str,sd) ){
      (node+num)->parent->key=(node+num)->key;
    }else{
      (node+num)->parent->key=(node+num)->brother->key;
    }
    num=(node+num)->parent->num;
  }
}

/**
 * # FUNCTION #
 *最初にリーフにデータをセットした時に、全てのトーナメント戦を行う関数
 *  TTinsertの時には、そのインサートされたリーフからのみ試合を行えばよい
 *  のでこの関数は使われない
 */
static void TTevaluate(
  struct TTnode *node,/*トップノード*/
  int cnt,            /*葉っぱの数*/
  struct mssSortDat *sd){
  int i;

  /*試合を行い、小さい方が勝ち(親ノードに登録する)。*/
  for(i=cnt*2-1; i>=3; i=i-2){
    if( 0>cmpKeyStr((node+i)->key->str,(node+i)->brother->key->str,sd) ){
      (node+i)->parent->key=(node+i)->key;
    }else{
      (node+i-1)->parent->key=(node+i)->brother->key;
    }
  }
}

/**
 * # FUNCTION #
 * TTnodeの表示(デバッグ用)
 * TTprintTree関数から呼び出される。
 */
static void TTptree(
  struct TTnode *p,
  int num,   /*ノード番号*/
  int h,     /*木の深さ*/
  int max,   /*ノード番号の最大値(不変)*/
  struct mssSortDat *sd){
  int i;

  if(num<=max){
    TTptree(p, num*2, h+1, max, sd);

    for(i=1; i<=h; i++) printf("   ");
    if( (p+num)->key->str == readEnd ){
      printf("%d NULL\n",(p+num)->num);
    }else{
      if(sd->cmpRevNum){
        printf("%d ", (p+num)->num);
        for(i=0; i<4; i++){
          printf("'%s'",*((char **)(p+num)->key->str+i));
        }
        printf(" %d\n", (p+num)->key->bkt);
      }else{
        printf("%d '%s' %d\n",
               (p+num)->num,(p+num)->key->str,(p+num)->key->bkt);
      }
    }

    TTptree(p, num*2+1, h+1, max, sd);
  }
}

/**
 * # FUNCTION #
 * TTツリーの表示(デバッグ用)
 */
void TTprintTree(
  char *s,
  struct TTnode *top,
  int cnt,  /*葉っぱの数*/
  struct mssSortDat *sd){
  printf("%s\n",s);
  TTptree(top,1,0,cnt*2-1,sd);
  printf("------end\n");
}

/**
 * # FUNCTION #
 * データを全て読み切った時には、各リーフに番兵がセットされる。
 * 引数で与えた文字列が番兵かどうかを検出するための関数。
 */
static int isReadEndTT(char *str)
{
  if(str==readEnd) return(1);
  else             return(0);
}

/**
 * # FUNCTION #
 *末尾に付ける文字を指定できるstrcpy(返値はコピー後の次のアドレス)
 * 新たに領域は確保しないので、文字列toには文字列fromをコピーするに十分な領域
 * があることが前提。
 */
static char *strChrCpy(
  char *to,   /*コピーされるメモリ領域*/
  char *from, /*コピーする文字*/
  char end){  /*末尾に付ける文字*/
  
  while(*from!='\0'){
    *to++=*from++;
  }
  *to++=end;
  return(to);
} 

/**
 * # FUNCTION #
 *一時ファイル名の取得
 */
static char *getFname(char *prefix, int num)
{
  sprintf(TFname,"%s%d",prefix,num);
  return(TFname);
}

/**
 * # FUNCTION #
 * qsortの結果を一時ファイルに書き込む関数
 * wrkPntに各行の先頭アドレス、そしてwrkCnt行
 */
static void writeBuf(char **wrkPnt,int wrkCnt, char *fname)
{
  int i;
  struct mssFPW      *outFile;

  outFile=mssOpenFPW(fname,0,0);
  for(i=0; i<wrkCnt; i++){
    mssWriteStr(*(wrkPnt+i),outFile);
    mssWriteRet(outFile);
  }
  mssCloseFPW(outFile);
}

/**
 * # FUNCTION #
 * ttKeyに一行読み込む
 * 複数のバケットのうちbkt番のファイルから rec構造体に読み込み
 * そのアドレスをTTkey構造体にセットする。
 *  in setFirstLineTT,mergeTT,readFPRfile
 */
static void readTTkey(
  struct TTkey *ttKey,
  struct mssSortDat *sd,
  int bkt){

  /*逆順、数値ソートの場合*/
  if(sd->cmpRevNum){
    if(EOF==mssReadFldRec(sd->iFile[bkt],sd->fr[bkt])){
      ttKey->str=readEnd;           /*EOFの時は番兵を入れる*/
      ttKey->bkt=bkt;               /*バケット番号のセット*/
    }else{
      /*ここではコンパイラの警告を避けるために一旦char **からchar **/
      /*にキャストしておく。実際の比較ルーチン(cmpKeyStr)では、*/
      /*このアドレスを再びchar **にキャストして、項目ごとの比較を行っている。*/
      ttKey->str=(char *)sd->fr[bkt]->pnt; /*読みこんだ文字の先頭アドレス*/
      ttKey->bkt=bkt;                      /*バケット番号のセット*/
    }
  /*文字列ソートの場合*/
  }else{
    if(EOF==mssReadRec(sd->iFile[bkt],sd->rec[bkt])){
      ttKey->str=readEnd;           /*EOFの時は番兵を入れる*/
      ttKey->bkt=bkt;               /*バケット番号のセット*/
    }else{
      ttKey->str=sd->rec[bkt]->pnt; /*読みこんだ文字の先頭アドレス*/
      ttKey->bkt=bkt;               /*バケット番号のセット*/
    }
  }
}


/**
 * # FUNCTION #
 * 項目を並べ換える時の、元の並びと新しい並びの対応リストを作成する関数
 * データ項目名:a b c d    k=b,d の時
 *       項目を b d a c に項目を並べ換える、そこで
 *
 * conv[0..3]
 * 0 -> b-> 1
 * 1 -> d-> 3
 * 2 -> a-> 0
 * 3 -> c-> 2
 */
static void setConv(struct mssSortDat *sd)
{
  int i,j;
  int tmp[MssFieldMaxLen];

  for(i=0; i<sd->fldCnt; i++) tmp[i]=0;
  for(i=0; i<sd->flds->cnt; i++){
    tmp[MssFlds2num(sd->flds,i)]=1;
  } 

  for(i=0; i<sd->fldCnt;  i++){
    if(i<sd->flds->cnt){
      sd->conv[i]=MssFlds2num(sd->flds,i);
    }else{
      for(j=0; j<sd->fldCnt; j++){
        if(tmp[j]==0){
          tmp[j]=1;
          break;
        }
      }
      sd->conv[i]=j;
    }
  }
}

/**
 * # FUNCTION #
 * 各バケットの最初の行をTTキューに読み込み
 * mergeTT,preSortから呼び出される
 * iFromからiToまでのファイルの先頭行をTTキューに入れる
 */
static void setFirstLineTT(
  struct mssSortDat *sd, /*iFile,ttをセットする*/
  int iFrom,          /*開始ファイル番号*/
  int iTo){           /*終了ファイル番号*/

  struct TTkey ttKey;     /*TTキューに入れるデータの構造体*/
  int bkt;                /*現在処理中のバケット番号(0から始まる)*/
  int i;

  /*バケット数の設定*/
  sd->bktCnt=iTo-iFrom+1;

  /*トーナメントツリーの初期化*/
  sd->tt=TTinit(sd->bktCnt);

  bkt=0;
  for(i=iFrom; i<=iTo; i++){
    /*i番目のバケットのオープン*/
    sd->iFile[bkt]=mssOpenFPR(getFname(sd->prefix,i),4);
    /*バケットから一行読み込みttKeyにセット*/
    readTTkey(&ttKey, sd, bkt);

    /*セットしたttKeyをトーナメントツリーにリーフとしてセット*/
    /*全ノード数:2*sd->bktCnt-1, 葉っぱ数:sd->bktCnt;*/
    /*よって葉っぱの最初のノード番号:2*sd->bktCnt-1-sd->bktCnt+1=sd->bktCnt*/
    TTsetLeaf(&ttKey,sd->tt,sd->bktCnt+bkt);
    bkt++;
  }

  /*トーナメントツリーで、優勝者を決める*/
  TTevaluate(sd->tt,sd->bktCnt,sd);

  /*TTprintTree("############ before",sd->tt,sd->bktCnt,sd);*/
}

/**
 * # FUNCTION #
 * TTキューを用いたPWayマージ
 * 既に並べ変わっているバケット(iCnt個)を併合していく
 * 最後の一つになるまで併合はしない
 * PWayS個より少なくなったときに併合を止める
 */
static void mergeTT( struct mssSortDat *sd )
{

  int    bkt;              /*PWay併合のバケット番号*/
  struct TTkey   ttKey;    /*プライオリティキューのデータ構造体*/
  struct TTnode *node;     /*popにて取りだしたノードを格納*/

  struct mssFPW *oFile;       /*出力ワークファイルポインタ*/

  int    iStart;           /*入力ワークファイルの開始番号(ファイル名の一部)*/
  int    iEnd;             /*入力ワークファイルの終了番号(ファイル名の一部)*/
  int    oStart;           /*出力ワークファイルの開始番号(ファイル名の一部)*/
  int    oEnd;             /*出力ワークファイルの終了番号(ファイル名の一部)*/
  int    iFrom,iTo;        /*併合する時のファイル番号*/
  int    k;
  int    i;

  /*次のループでin,outをswapするので、ここでは逆に定義*/
  iStart=sd->iEnd+1;
  iEnd  =sd->iEnd+1;
  oStart=sd->iStart;
  oEnd  =sd->iEnd;

  /*ファイルを併合するループ iCntがPWayS個より大きい場合まわり続ける*/
  while(1){
    mssSwapInt(&iStart,&oStart);
    mssSwapInt(&iEnd  ,&oEnd  );
    oEnd=oStart;

    /*入力ファイル数がPWayS以下ならば終了*/
    if(iEnd-iStart+1 <= PWayS){
      sd->iStart = iStart;
      sd->iEnd   = iEnd;
      break;
    }

    /* "Pway個のiFileを一つのoFileに書き出す"を繰り返すループ*/
    k=0;
    while(1){

      /*各バケットの最初行をキューに入れる*/
      iFrom= k   *PWayS+iStart;
      iTo  =(k+1)*PWayS+iStart-1;
      if(iTo>iEnd) iTo=iEnd;
      setFirstLineTT(sd, iFrom,iTo);

      /*出力ファイルオープン*/
      oFile=mssOpenFPW(getFname(sd->prefix,oEnd),0,0);

      /*各バケットから一行づつ読み込み書き出すループ*/
      while(1) {
        node=sd->tt+1;                           /*キューから一行取り出し*/
        if(isReadEndTT(node->key->str)) break;   /*readEnd(番兵)なら終了*/
        bkt=node->key->bkt;                      /*取り出し元のバケット番号*/
        /*一行書き出し*/
        if(sd->cmpRevNum){
          mssWriteFld((char **)node->key->str,sd->fldCnt,"\n",oFile);
        }else{
          mssWriteStr(node->key->str,oFile);
          mssWriteRet(oFile);
        }
        readTTkey(&ttKey, sd, bkt);              /*次の行を読み込む*/
        TTinsert(&ttKey,sd->tt,sd->bktCnt+bkt,sd); /*それをキューに追加*/
        /*TTprintTree("############ before",sd->tt,sd->bktCnt,sd);*/
      }

      TTfree(sd->tt);             /*キューの開放*/
      for(i=0; i<=iTo-iFrom; i++) /*入力ファイルのクローズ*/
        mssCloseFPR(sd->iFile[i]);
      mssCloseFPW(oFile);            /*出力ファイルのクローズ*/
      if(iTo==iEnd)break;         /*最後のバケットまでいけば終了*/
      oEnd++;                     /*出力ファイル番号カウントアップ*/
      k++;                        /*併合回数カウントアップ*/
    }

    /*入力ワークファイルの削除*/
    for(i=iStart; i<=iEnd; i++){
      unlink(getFname(sd->prefix,i));
    }
  }
}

/**
 * # FUNCTION #
 * iCntのバケットを各々quick sortする。
 * strChrCpyでバッファ領域にコピーせずに、ファイルバッファをそのまま使えば
 * 高速化できる可能性がある。しかし項目を並べ換えるのが問題。
 */
static void sort(
  struct mssSortDat *sd,   /*convは入力として使う、その他は終了状態の記録用*/
  struct mssFPR *inFile){

  struct mssFldRec *fr;       /*入力ファイルをキー順に並べ換えるためのバッファ*/
  int            rf;       /*入力ファイルからの読み込み文字数*/
  int    iCnt;             /*入力ワークファイルのカウントアップ用変数*/
  int    wrkCnt;           /*buffer行数*/
  char  *wrkBuf;           /*データ本体のbuffer*/
  char **wrkPnt;           /*データ本体に対するポインタ配列*/
  char  *curBuf;           /*入力ファイルをwrkBufにコピーする際のcurrent位置*/
  int    i;

  /*データ本体用メモリ領域確保*/
  wrkBuf=mssMalloc((MaxMemS+MssRecordMaxLen)*sizeof(char  ),"sorting error");
  /*データへのポインタ用メモリ領域確保*/
  wrkPnt=mssMalloc( MaxLineS * sizeof(char *),"sorting error");

  /*---------------------------------------------*/
  /*各バケットのソート(qsort)*/
  /*---------------------------------------------*/
  /*データをバケット分割し、各バケット内で並べ換える*/
  /*iCntにファイル数が入る*/
  curBuf=wrkBuf;
  iCnt=0;
  wrkCnt=0;
  fr=mssInitFldRec(sd->fldCnt); /*FldRecの初期化*/
  while(1){
    /*データの読み込み*/
    rf = mssReadFldRec(inFile,fr);

    /*ワークファイルへの書き出し*/
    if( curBuf-wrkBuf >= MaxMemS || wrkCnt >= MaxLineS || rf==EOF){
/*
      qsort(wrkPnt,wrkCnt,sizeof(char *),
             (int (*)(const void *,const void *))cmpKeyQst);
*/
      sdTmp=sd;
      qsort5((char *)wrkPnt,wrkCnt,sizeof(char *),
             (int (*))cmpKeyQst);
      writeBuf(wrkPnt,wrkCnt,getFname(sd->prefix,iCnt));
      iCnt++;
      wrkCnt=0; curBuf=wrkBuf;
      if(rf==EOF) break;
    }

    sd->recCnt++; /*レコード数カウントアップ*/

    /*バッファへのコピー*/
    *(wrkPnt+wrkCnt)=curBuf;
    wrkCnt++;

    for(i=0; i<sd->fldCnt-1; i++){
      curBuf=strChrCpy(curBuf,*(fr->pnt+sd->conv[i]),MssFieldDelim);
    }
    curBuf=strChrCpy(curBuf,*(fr->pnt+sd->conv[i]),'\0');
  }
  mssFreeFldRec(fr);
  mssFree(wrkBuf);
  mssFree(wrkPnt);

  sd->iStart = 0;
  sd->iEnd   = iCnt-1;
}

/**
 * # FUNCTION #
 *iCntのバケットを各々quick sortする。(数値、逆順指定バージョン)
 *strChrCpyでバッファ領域にコピーせずに、ファイルバッファをそのまま使えば
 *高速化できる可能性がある。しかし項目を並べ換えるのが問題。
 */
static void sort2(
  struct mssSortDat *sd,   /*convは入力として使う、その他は終了状態の記録用*/
  struct mssFPR *inFile){

  struct mssFPW *fpw;
  struct mssFldRecMax *frm;   /*入力ファイルをキー順に並べ換えるためのバッファ*/
  int    iCnt;             /*入力ワークファイルのカウントアップ用変数*/
  int    i;

  /*---------------------------------------------*/
  /*各バケットのソート(qsort)*/
  /*---------------------------------------------*/
  /*データをバケット分割し、各バケット内で並べ換える*/
  /*iCntにファイル数が入る*/
  iCnt=0;
  frm=mssInitFRM(sd->fldCnt); /*FldRecの初期化*/
  while(EOF != mssReadFRM(inFile,frm) ){

    sd->recCnt+=frm->recCnt; /*レコード数カウントアップ*/

/*
    qsort(frm->pnt, frm->recCnt, sizeof(char *)*frm->fldCnt,
           (int (*)(const void *,const void *))cmpKeyQst);
*/
    sdTmp=sd;
    qsort5((char *)frm->pnt, frm->recCnt, sizeof(char *)*frm->fldCnt,
             (int (*))cmpKeyQst);

    fpw=mssOpenFPW(getFname(sd->prefix,iCnt),0,0);
    for(i=0; i<frm->recCnt; i++){
      mssWriteFld(frm->pnt+i*frm->fldCnt,frm->fldCnt,"\n",fpw);
    }
    mssCloseFPW(fpw);
    iCnt++;
  }
  mssFreeFRM(frm);

  sd->iStart = 0;
  sd->iEnd   = iCnt-1;
}

/*
同一コマンドで複数のmssReopenFPRsortを実行するとpidでワークファイルを識別
できなくなる。そのため、initSDを呼び出す毎にsecNumをカウントアップし、
その番号をワークファイル名の一部に含め、この問題を回避する。
*/
int secNum=0;

/**
 * # FUNCTION #
 * mssSortDat構造体の初期化
 */
static struct mssSortDat *initSD(struct mssFields *flds, int fldCnt, char *tmpDir)
{
  struct mssSortDat *sd;
  int i;
  int    pid;              /*プロセスID(ファイル名の一部)*/

  /* 領域確保 */
  sd=mssCalloc(sizeof(struct mssSortDat),"initSD");

  /*逆順、数値ソート指定があればOn(Global)*/
  sd->cmpRevNum=0;
  for(i=0; i<flds->cnt; i++){
    if( MssFlds2revFlg(flds,i) || MssFlds2numFlg(flds,i) ){
      sd->cmpRevNum=1;
    }
  }

  /*グローバル変数にセット*/
  /* SortFld=flds; */

  sd->flds=flds; /*ソート項目構造体のセット*/

  /*逆順、数値ソートの場合はFldRec構造体を使う*/
  if(sd->cmpRevNum){
    for(i=0; i<PWayS; i++) sd->fr[i]=mssInitFldRec(fldCnt);

  /*文字列ソートの場合はRec構造体を使う*/
  }else{
    for(i=0; i<PWayS; i++) sd->rec[i]=mssInitRec();
  }

  sd->fldCnt=fldCnt;

  /*ファイル名のプレフィックスの設定*/
  pid=getpid();
  if(tmpDir==NULL){
    mssShowErrMsg("path name of work directory is null");
    mssEnd(mssErrorNoDefault);
  }
  if(strlen(tmpDir) > MssFileNameMaxLen - 50 ) {
    mssShowErrMsg("length of path name must be less than %d",MssFileNameMaxLen-50);
    mssEnd(mssErrorNoDefault);
  }
  sprintf(sd->prefix,"%s/xt##%d-%d-PreSrt-",tmpDir,pid,secNum);

  /*シグナルハンドラの設定*/
  mssGV.usedTempFileFlg=1;
  mssSetSignalHandler();

  secNum++;
  return(sd);
}

/**
 * # FUNCTION #
 *mssSortDat構造体の開放
 */
static void freeSD(struct mssSortDat *sd)
{
  int i;

  TTfree(sd->tt); /*トーナメントツリーのキューの開放*/

  if(sd->cmpRevNum){
    for(i=0; i<PWayS; i++) mssFreeFldRec(sd->fr[i]);
  }else{
    for(i=0; i<PWayS; i++) mssFreeRec(sd->rec[i]);
  }

  mssFree(sd);
}

/**
 * # FUNCTION #
 * マージソートで使った一時ファイルの削除
 */
static void endPreSort(struct mssSortDat *sd)
{
  int i;

  /*バッファ管理のために、入力ファイルはここで初めてクローズする*/
  for(i=0; i<=sd->iEnd-sd->iStart; i++){
    mssCloseFPR(sd->iFile[i]);
  }

  /*ファイルの削除*/
  for(i=sd->iStart; i<=sd->iEnd; i++){
    unlink(getFname(sd->prefix,i));
  }
}


/**
 * # FUNCTION #
 * バケット毎のqsort+トーナメントツリーによるPWayマージソート
 */
static void preSort(
  struct mssSortDat *sd,
  struct mssFPR     *inFile){


  if(sd->cmpRevNum){/*逆順、数値ソート指定があれば initSDでセット*/
    sort2(sd,inFile);
  }else{
    setConv(sd);
    sort(sd,inFile);
  }
  mergeTT(sd);
  setFirstLineTT(sd, sd->iStart,sd->iEnd);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 入力ファイル関連の関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * ファイル情報の取得
 * fNameで与えられたxmlTableファイルについて、maxCnt行を読み込み
 * mssFileInfo構造体の行数や項目数などをセットし、その構造体のアドレスを返す。
 */
struct mssFileInfo *mssGetFileInfo(char *fName, int maxCnt)
{

  int cnt;     /*レコード数をカウント*/
  struct mssFileInfo *fi;
  struct stat fStat;
  struct mssFPR   *fpr;      /*入力ファイル構造体*/
  struct mssHeader *hdi; /*入力ファイル用headerタグ格納構造体*/
  struct mssRec    *rec=NULL; /*行バッファ構造体*/

  fi=mssMalloc(sizeof(struct mssFileInfo),"chkFile");
  if(-1 == stat(fName,&fStat)){
    mssShowErrMsg("cannot get file status:%s",fName);
    mssEnd(mssErrorNoDefault);
  }

  fi->maxCnt    = maxCnt;
  fi->fName     = fName;
  fi->totalSize = fStat.st_size;
  fi->readEnd   = 1;

  fpr=mssOpenFPR(fName,4); /*ファイルオープン*/
  hdi=mssReadHeader(fpr);  /*項目名情報の読み込み*/
  fi->fldCnt   = hdi->flds->cnt;
  mssFreeHeader(hdi);     /*入力ヘッダ領域開放*/

  rec=mssInitRec();
  cnt=0;
  fi->bodySize=0;
  while(EOF != mssReadRec(fpr,rec)){
    fi->bodySize += rec->chrCnt;
    if(maxCnt<=++cnt && maxCnt!=0){
      fi->readEnd = 0;
      break;
    }
  }
  fi->recCnt =cnt;
  mssFreeRec(rec);
  mssCloseFPR(fpr);

  return(fi);
}

/**
 * # FUNCTION #
 * mssFPR構造体fpの入力データを全て読み込んだかどうか(EOF)を検出する。
 * 返値:1=EOF, 0:not EOF
 */
static int isEOF(struct mssFPR *fp)
{
  if( fp->last ){ /*eofを含む読み込みをしたか?*/
    if(*fp->curPnt=='\0' || 0==strncmp(fp->curPnt,MssEndBodyString,10)){
      return(1);
    }
  }
  return(0);
}

/**
 * # FUNCTION #
 * 項目数が異なるerrorメッセージの表示&終了
 */
static void fldCntErr(char **pnt, int fldCnt, int j, struct mssFPR *fp)
{
  if(fp->fName==NULL){
    mssShowErrMsg("detected different number of fields in line %d(header has %d, but a record has %d) : %s",mssGV.inCnt+1,fldCnt,j,"stdin");
    mssEnd(mssErrorNoDefault);
  }else{
    mssShowErrMsg("detected different number of fields in line %d(header has %d, but a record has %d) : %s",mssGV.inCnt+1,fldCnt,j,fp->fName);
    mssEnd(mssErrorNoDefault);
  }
}

/**
 * # FUNCTION #
 * mssFPRのリングバッファの現在位置の行でスペースをdelimitorとしてトークン分割
 * トークン分割された項目値(文字列)は、*(pnt+i)に格納される。
 * よって、pnt変数は、この関数を呼び出す前に、領域確保しておく必要がある。
 * 文字列の最後の文字は'\n'でも'\0'でもOK
 *     -> '\0'はやめるべき。(連続'\0'をスキップするのは危険だから)
 * なお、連続する'\n'もしくは'\0'はスキップされる。
 * 項目数が異なればエラー終了
 * 次の行の先頭位置を返す
 */
int hamCnt=0;
static char *strToken(char **pnt, int fldCnt, int *chrCnt, struct mssFPR *fp)
{
  char *str=fp->curPnt;
  int j;
  int flg;

  if(fp->sd==NULL)              flg=1;
  else if(fp->sd->cmpRevNum==1) flg=1;
  else                          flg=0;

  /*通常ファイルの場合もしくはソートファイルで数値、逆順ソートの場合*/
  if(flg){
    j=0; /*項目数カウンタ*/
    while(1){
      switch(*str){
      case MssFieldDelim:
        *str='\0';
        str++;
        break;
      case '\n':
      case '\0':
        if(fldCnt != j){
          fldCntErr(pnt,fldCnt,j,fp);
        }
        *str='\0';
        str++;
        *chrCnt+=str-fp->curPnt; /*文字数の更新*/
        return(str);
        break;
      default:
        /*項目位置を登録*/
        if(j<fldCnt) *(pnt+j)=str;
        j++;
        /*次のMssFieldDelim(もしくはEOL)までスキップ*/
        while(*str!=MssFieldDelim && *str!='\n' && *str!='\0') str++;
        break;
      }
    }

  /*ソートファイルの場合*/
  }else{
    j=0; /*ソートファイル上の項目番号 conv[j]で元のファイル番号*/
    while(1){
      switch(*str){
      case MssFieldDelim:
        *str='\0';
        str++;
        break;
      case '\n':
      case '\0':
        if(fldCnt != j){
          fldCntErr(pnt,fldCnt,j,fp);
        }
        *str='\0';
        str++;
        *chrCnt+=str-fp->curPnt; /*文字数の更新*/
        return(str);
        break;
      default:
        /*項目位置を登録*/
        if(j<fldCnt) *(pnt+fp->sd->conv[j]) = str;
        j++;
        /*次のMssFieldDelim(もしくはEOL)までスキップ*/
        while(*str!=MssFieldDelim && *str!='\n' && *str!='\0') str++;
        break;
      }
    }
  }

  /*このreturnはあり得ないがコンパイル警告を防ぐため*/
  return(str);
}

/**
 * # FUNCTION #
 * リングバッファの表示(デバッグ用)
 */
void showQue(struct mssFPR *fp,char *title){
  int i,j,k;
  char tmp;

  fprintf(stderr,"-[QUE start %s]====================================\n",title);
  fprintf(stderr,"enq=%d\n",fp->enq);
  fprintf(stderr,"deq=%d\n",fp->deq);
  fprintf(stderr,"full=%d\n",fp->full);

  fprintf(stderr,"----------------------------------------\n");
  for(i=0; i<MssRecordMaxLen; i++){
   tmp=*(fp->buf+i);
         if(tmp == '\0') fprintf(stderr,"@");
    else if(tmp == '\n') fprintf(stderr,"$");
    else                 fprintf(stderr,"%c",tmp);
  }
  fprintf(stderr,"\n");

  for(i=0; i<fp->queCnt; i++){
    fprintf(stderr,"[QUE %d]\n",i);
    for(j=0; j<ReadCnt; j++){
      fprintf(stderr,"[BUF %d] ",j);
      for(k=0; k<PipeSize; k++){
        tmp=*(fp->buf + i*fp->queSize + PipeSize*j+k+MssRecordMaxLen);
             if(tmp == '\0') fprintf(stderr,"@");
        else if(tmp == '\n') fprintf(stderr,"$");
        else                 fprintf(stderr,"%c",tmp);
      }
      fprintf(stderr,"\n");
    }
  }
}

/**
 * # FUNCTION #
 * リングバッファへの入力データ読み込み
 * ringBuffer
 *
 *     buf[0]        buf[1]      .... buf[n]
 *    +------+      +------+
 *   A|      |  +-->|   x  |
 *    +------+  |   +------+  +
 *   B|      | copy |      |  |
 *    |      |  |   |      |  |  新に読み込み
 *    +------+  |   +------+  |
 *   C|   x  |--+   |      |  |
 *    +------+      +------+  +
 *       x:fp->curPnt
 *
 *  read関数にて読み込む領域 : B+C -> BktSize * BktCnt = ReadSize
 *  fp->curPntがCの領域にあれば、次のバッファに移る。
 *  その際、Cを次のバッファのAにコピーし、同時にcurPntも移動する。
 *
 *                     buf[0]
 *                buf[n]     buf[1] +---->fp->deq
 *              buf[.]          buf[2]
 *                buf[.]     buf[3]+---->fp->cuq
 *                     buf[4]+------>fp->enq
 * 
 *  deq : 保存しておくべきバッファの開始位置
 *  enq : 次にデータを読み込むバッファ位置
 *  cuq : 現在対象となっているバッファ位置(curPntが含まれる領域)
 *
 *  この関数でeofの検出は行わない
 *  サイズがReadSizeより少ない時に'\0'を追加
 *  バッファがfull状態の時に読み込もうとしたら-1を返す。その他の場合は1を返す。
 */
static int readFPRfile(struct mssFPR *fp)
{

  /*通常ファイルで利用*/
  register int size; /*read関数で読みこんだサイズ*/
  register int i;

  /*sortファイルで利用*/
  struct TTnode *node;     /*popにて取りだしたノードを格納*/
  struct TTkey   ttKey;    /*プライオリティキューのデータ構造体*/
  int bkt;

  /*キューがfull*/
  if(fp->full) return(-1);

//showQue(fp,"BEFORE");
  /*通常ファイルの場合*/
  if(!fp->sort){
    /*enqが先頭ならばバッファ最後のMssRecordMaxLen文字をバッファの先頭にコピー*/
    if(fp->enq==0){
      memcpy(fp->buf, fp->buf+fp->queSize*fp->queCnt, MssRecordMaxLen);
      fp->curPnt=fp->curPnt-fp->queSize*fp->queCnt; /*curPntの移動*/
    }
    /*ファイルの読み込み*/
    for(i=0; i<ReadCnt; i++){
      if(fp->zflg){
        size = gzread(fp->zfd,
                      fp->buf+fp->enq*fp->queSize+PipeSize*i+MssRecordMaxLen,
                      PipeSize);
      }else{
        size = fread(fp->buf+fp->enq*fp->queSize+PipeSize*i+MssRecordMaxLen,
                     sizeof(char) ,PipeSize, fp->fp);
      }

      /*eofを含む読み込みの場合、eofの印のために'\0'を追加*/
      if(size<PipeSize){
        *(fp->buf+fp->enq*fp->queSize+PipeSize*i+MssRecordMaxLen+size) = '\0';
        fp->last=1;
        break;

      /*sizeが0より小さいときは、読み取りエラー*/
      }else if(size<0){
        mssShowErrMsg("file read error");
        mssEnd(mssErrorNoDefault);
      }
    }

    /*enqueue対象バッファを１つ進める*/
    fp->enq=NextQue(fp->enq,fp->queCnt);
  
    /*バッファfullの判定*/
    if(fp->enq == fp->deq) fp->full=1;
//showQue(fp,"AFTER");

  /*sortファイルの場合*/
  /*通常ファイルと同等の働きをする*/
  /*異なる点は、行単位での読み込みのため、queSizeを越えたデータは次のque*/
  /*に読みこまれる。そのため、bufferIsFullの判定はdeq+2==enqの時になる。*/
  /*また、bufferの最後のqueの次にはoverFlowBufferがあることが前提となる。*/
  }else{
    /*enqが先頭ならばバッファ最後のMssRecordMaxLen*2文字をバッファ先頭にコピー*/
    if(fp->enq==0){
      /*MssRecordMaxLen*2の2はoverFlowBufferの分*/
      memcpy(fp->buf, fp->buf+fp->queSize*fp->queCnt, MssRecordMaxLen*2);
      fp->curPnt =fp->curPnt -fp->queSize*fp->queCnt; /*curPntの移動*/
      fp->readPnt=fp->readPnt-fp->queSize*fp->queCnt; /*curPntの移動*/
    }

    /*トーナメントツリーからの読み込み(queSizeを越えるまでstrcpyする)*/
    while(1){
      /*readPntがenq領域を越えたら読み込み終了*/
      if(fp->readPnt>=fp->buf+(fp->enq+1)*fp->queSize+MssRecordMaxLen){
        break;
      }

      node=fp->sd->tt+1;              /*キューから一行取り出し*/
      if(isReadEndTT(node->key->str)){  /*readEnd(番兵)を取り出したら終了*/
        *fp->readPnt='\0';
        fp->last=1;
        break;
        return(1); /*取りあえず*/
      }
      bkt=node->key->bkt;     /*取り出し元のバケット番号*/

      /*数値、逆順ソートの場合はtnTreeが既に項目を区切っているので戻す*/
      /*この処理は無駄のように思えるが、mssReadRec,mssReadFldRecなどの*/
      /*上位のread関数を通常と同等に扱えるのでメンテナンスが楽。*/
      if(fp->sd->cmpRevNum){
        for(i=0; i<fp->sd->fldCnt-1; i++){
          fp->readPnt=strChrCpy(fp->readPnt,*((char **)node->key->str+i),
                                MssFieldDelim);
        }
        fp->readPnt=strChrCpy(fp->readPnt,*((char **)node->key->str+i),'\n');
      }else{
        fp->readPnt=strChrCpy(fp->readPnt,node->key->str,'\n');
      }
      readTTkey(&ttKey, fp->sd, bkt);                /*次の行を読み込む*/
      TTinsert(&ttKey,fp->sd->tt,fp->sd->bktCnt+bkt,fp->sd);/*キューに追加*/
    }
    /*enqueue対象バッファを１つ進める*/
    fp->enq=NextQue(fp->enq,fp->queCnt);

    /*バッファfullの判定(enqの次のバッファにもデータがかかっているので)*/
    if(NextQue(fp->enq,fp->queCnt) == fp->deq) fp->full=1;
  }
  return(1); /*とりあえず*/
}

/**
 * # FUNCTION #
 * readFileする必用があるかどうか
 */
static int needFileRead(struct mssFPR *fp)
{

  if(fp->last)return(0);

  /*MssRecordMaxLenは先頭の共通領域があるので、+-で消える*/
  if(fp->curPnt >= fp->buf+(PrevQue(fp->enq,fp->queCnt)+1)*fp->queSize)
    return(1);
  else
    return(0);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 行単位の読み込み関数関連
 * ----------------------------------------------------------------------------
 * xmlTableのデータを読み込む方法を以下の５種類用意している。
 * 1.一行単位での読み込み(mssReadRec)
 *   一行ごとに読み込む。項目のトークン分割は行わないので高速である。
 * 2.一行項目単位の読み込み(mssReadFldRec)
 *   一行ごとに読み込み、項目もトークン分割される。
 * 3.二行項目単位の読み込み(mssReadFRD)
 *   常にデータ行を二行キープしながら読み込む。キーブレーク処理に便利。
 * 4.キー単位の読み込み(mssReadFRK)
 *   同じキー値のデータを一旦全て読み込む。
 *   データ量が大きければ一時ファイルを使う。
 * 5.指定されたバイト数まで読み込む(mssReadFRM)
 *   リングバッファがいっぱいになるまで読み込まれる。
 */

/**
 * # FUNCTION #
 * mssRec構造体の初期化
 */
struct mssRec *mssInitRec(void)
{
  struct mssRec *rec;
  rec=mssMalloc(sizeof(struct mssRec), "initRec");
  rec->chrCnt=0;
  rec->eof   =0;
  return(rec);
}

/**
 * # FUNCTION #
 * mssRec構造体の開放
 */
void mssFreeRec(struct mssRec *rec)
{
  mssFree(rec);
}

/**
 * # FUNCTION #
 * mssRec構造体へデータを一行読み込む
 * 読み込まれたデータへは、rec->pntで参照可能。
 * 行末の改行は省かれる。
 */
int mssReadRec(struct mssFPR *fp, struct mssRec *rec)
{
  register int i; /*高速化のためにregister変数を利用*/

  /*現在位置がバッファ最後のMssRecordMaxLen文字内にかかっていたらファイル読み込み*/
  if(needFileRead(fp)){
    readFPRfile(fp);
    /*行単位の読み込みの場合は常にdeqはenqの１つ前*/
    fp->deq=PrevQue(fp->enq,fp->queCnt);
  }
  /*eofの検知*/
  if(isEOF(fp)){
    rec->eof=1;
    return(EOF);
  }

  /* '\n'をdelimitorとして一行を切り出す*/
  rec->pnt=fp->curPnt;
  i=0;
  while( '\n' != *fp->curPnt++ ){
    if(i++>=MssRecordMaxLen){
      mssShowErrMsg("exceed %d characters per record\n",MssRecordMaxLen);
      mssEnd(mssErrorNoDefault);
    }
  }
  *(fp->curPnt-1)='\0';

  fp->chrCnt+=i+1;
  fp->recCnt++;
  rec->chrCnt=fp->curPnt-rec->pnt;

  return(1); /*とりあえず*/
}

/**
 * # FUNCTION #
 * mssFldRec構造体の初期化
 * 引数に項目数を与える。実際に読み込む際に項目数が異なればエラー終了する。
 */
struct mssFldRec *mssInitFldRec(int fldCnt)
{
  struct mssFldRec *fr;
  fr=mssMalloc(sizeof(struct mssFldRec), "initFldRec");
  fr->pnt=mssCalloc(sizeof(char *)*fldCnt,"initFldRec");
  fr->fldCnt=fldCnt;
  fr->chrCnt=0;
  fr->eof   =0;
  return(fr);
}

/**
 * # FUNCTION #
 * mssFldRec構造体の開放
 */
void mssFreeFldRec(struct mssFldRec* fr)
{
  if(fr==NULL) return;
  mssFree(fr->pnt);
  mssFree(fr);
}

/**
 * # FUNCTION #
 * mssFldRec構造体にデータを一行読み込む
 * 読み込まれたデータへは、*(rec->pnt+i)で参照可能(iは0から始まる項目番号)。
 * 行末の改行は省かれる。
 */
int mssReadFldRec(struct mssFPR *fp, struct mssFldRec *fr)
{

  /*現在位置がバッファ最後のMssRecordMaxLen文字内にかかっていたら
    ファイル読み込み*/
  if(needFileRead(fp)){
    readFPRfile(fp);
    /*行単位の読み込みの場合は常にdeqはenqの１つ前*/
    fp->deq=PrevQue(fp->enq,fp->queCnt);
  }

  /*eofの検知*/
  if(isEOF(fp)){
    fr->eof=1;
    return(EOF);
  }

  /*スペースをdelimitorとしてトークン分割*/
  fp->recCnt++;
  fp->curPnt=strToken(fr->pnt, fr->fldCnt, &fp->chrCnt, fp);

  /*文字数をセット(最後の\0含む:コピー用)*/
  fr->chrCnt=fp->curPnt-*fr->pnt;
  return(1); /*とりあえず*/
}

/**
 * # FUNCTION #
 * 固定長ファイルの指定された一行をFldRec構造体に読み込む
 * いわゆるランダムアクセス(FPR構造体に固定長アクセスフラグをつくるべき？)
 */
int mssReadRandFldRec(
  struct mssFPR *fp,
  struct mssFldRec *fr,
  int recNo,  /*seekするレコード番号*/
  int datTop, /*データの先頭位置*/
  int recLen) /*レコード長*/
{    /*レコード長*/

  char *buf;
  int cnt;

  buf=mssMalloc(sizeof(char)*MssRecordMaxLen,"readRandFldRec");

  fp->curPnt=buf; /*あまりよくない方法(呼び出し側でmssFreeする)*/
  fseek(fp->fp,datTop+recLen*recNo,SEEK_SET);
  fread(buf,sizeof(char),MssRecordMaxLen,fp->fp);

  strToken(fr->pnt, fr->fldCnt, &cnt, fp);

  return(1);
}

/**
 * # FUNCTION #
 * mssFldRecDbl構造体の初期化
 */
struct mssFldRecDbl *mssInitFRD(int fldCnt)
{

  struct mssFldRecDbl *frd;

  frd=mssMalloc(sizeof(struct mssFldRecDbl),"initFRD");
  frd->pnt[0]=mssCalloc(sizeof(char *)*fldCnt,"initFRD");
  frd->pnt[1]=mssCalloc(sizeof(char *)*fldCnt,"initFRD");

  frd->eof=0;

  frd->new=0;
  frd->old=1;

  frd->newChrCnt=0;
  frd->oldChrCnt=0;

  /*最初の行を読みこむときだけセットされる*/
  frd->firstRead=1;

  frd->fldCnt=fldCnt;

  return(frd);
}

/**
 * # FUNCTION #
 * mssFldRecDbl構造体の開放
 */
void mssFreeFRD(struct mssFldRecDbl *frd)
{

  mssFree(frd->pnt[0]);
  mssFree(frd->pnt[1]);
  mssFree(frd);
}

/**
 * # FUNCTION #
 * mssFldRecDbl構造体によるデータ読み込みのキーブレイク判定
 * 返値 1:キーブレーク 0:キーブレークでない
 * 先頭行の読み込み時はキーブレークと判定しない。
 * EOFの読み込み時はキーブレークと判定しする。
 * opeKey->diffSame が1の場合(全行異なるキー値と見る場合)は常に1を返す。
 * opeKey->diffSame が0の場合(全行同じキー値と見る場合)は常に0を返す。
 */
int mssKeyBreak(struct mssFldRecDbl *frd, MssOptKEY *optKey)
{
  int i,fn;

  /*最初行の時はキーブレークでない*/
  if(frd->firstRead){
    /*multi keyに対応するために以下のif文を追加*/
    if(frd->new){
      return(0);
    }else{
      frd->firstRead=0;
    }
  }

  /*最終行の時はキーブレーク*/
  if(frd->eof == 1) return(1);

  switch(optKey->diffSame){

  /*キーを全て異なる値と見る場合はキーブレーク*/
  case 1:
    return(1);

  /*キー全て同じ値と見る場合はキーブレーク*/
  case 2:
    return(0);

  default:
    for(i=0; i<optKey->flds->cnt; i++){
      fn=MssFlds2num(optKey->flds,i);
      if( 0 != strcmp(*(frd->pnt[frd->new]+fn),*(frd->pnt[frd->old]+fn)) )
         return(1);
    }

    /*上記の条件をクリアしたということはキーブレークでない*/
    /*(key->cntが0ということは、-kの指定がないのでキーブレークでない)*/
    return(0);
  }
}


/**
 * # FUNCTION #
 * mssFldRecDbl構造体にデータを一行読み込む
 * 読み込まれたデータのカレント行へは、*(frd->pnt[frd->new]+i)
 * 一行前の行へは*(frd->pnt[frd->old]+i) で参照可能(iは0から始まる項目番号)。
 * 先頭行を読み込んだとき、一行前は存在しないので*(frd->pnt[frd->old]+i)は、
 * NULLとなる。
 * EOFを読み込んだときは、EOFを返す。
 */
int mssReadFRD(struct mssFPR *fp, struct mssFldRecDbl *frd)
{
  char *start;

  if(frd->eof) return(EOF);

  /*現在位置がバッファ最後のMssRecordMaxLen文字内にかかっていたらファイル読み込み*/
  if(needFileRead(fp)){
    readFPRfile(fp);
    /*二行単位の読み込みでも、queCntは最低4つ必用*/
    fp->deq=PrevQue(fp->enq,fp->queCnt);
  }

  /*eofの検知 最初にサイズ比較するのは、strcmpの比較を減らすため*/
  if(isEOF(fp)){ /* 現在の読み込みポインタが</body>ならば*/
    if(frd->newChrCnt==0)return(EOF); /*データ行がないときに引っかかる*/
    frd->eof=1;
    mssSwapInt(&frd->new,&frd->old);
    return(1);
  }

  /*newに新しい行の項目位置をいれるために、newとoldを入れ換える*/
  mssSwapInt(&frd->new,&frd->old);

  /*スペースをdelimitorとしてトークン分割*/
  start=fp->curPnt; /*現在の位置を確保*/
  fp->recCnt++;
  fp->curPnt=strToken(frd->pnt[frd->new], frd->fldCnt,&fp->chrCnt,fp);

  /*new,old行の文字数をセット(最後の\0含む:コピー用)*/
  frd->oldChrCnt=frd->newChrCnt;
  frd->newChrCnt=start-*frd->pnt[frd->new]+1;

  return(1);     /*とりあえず*/
}

/**
 * # FUNCTION #
 * mssFldRecMax構造体の初期化
 */
struct mssFldRecMax *mssInitFRM(int fldCnt)
{
  struct mssFldRecMax *frm;
  frm=mssMalloc(sizeof(struct mssFldRecMax),"initFRM");
  frm->fldCnt=fldCnt;
  frm->chrCnt=0;
  frm->eof   =0;
  frm->recCnt=0;
  frm->recMax=MaxPntSize/(fldCnt*sizeof(char *));

  /*ポインタ領域の確保(freeのために最低一行分のみ確保)*/
  frm->pnt=mssMalloc(sizeof(char *)*frm->fldCnt*frm->recMax,"initFRM");
  return(frm);
}

/**
 * # FUNCTION #
 * mssFldRecMax構造体の開放
 */
void mssFreeFRM(struct mssFldRecMax *frm)
{
  mssFree(frm->pnt);
  mssFree(frm);
}

/**
 * # FUNCTION #
 * mssFldRecMax構造体に次のいずれの条件も満たしている間、データを読み込み続ける。
 *  1. 指定された最大レコード数(MaxPntSize/(項目数*sizeof(char *)))を超えない。
 *  2. リングバッファ(サイズはmssOpenFPR関数で決まる)がFULLでない。
 * 読み込まれたデータは、項目単位でfrm->pntに蓄積されていく。
 * 例えば "a b c","x y z"の２行のデータを読み込んだ場合、
 * *(pnt+0)="a",*(pnt+1)="b",*(pnt+2)="c",*(pnt+3)="x",*(pnt+4)="y",*(pnt+5)="z"
 * となる。そこで、n行目(0から始まる)のm項目目(0から始まる)にアクセスしたい
 * 場合は、*(frm->pnt+frm->fldCnt*n+m)とする。
 * 先頭行としてEOFを読み込んだときは、EOFを返す。
 */
int mssReadFRM(struct mssFPR *fp, struct mssFldRecMax *frm)
{

  if(frm->eof){
    return(EOF);
  }

  frm->recCnt=0;

  while(1){

    /*ポインタバッファがfullであればリターン*/
    if(frm->recCnt>=frm->recMax){
      fp->deq = PrevQue(fp->enq,fp->queCnt);
      fp->full=0;
      return(1);
    }

    /*現在位置がバッファ最後のMssRecordMaxLen文字内にかかっていたらファイル読み込み*/
    if(needFileRead(fp)){
      /*データバッファがfullであれば終了*/
      if(-1==readFPRfile(fp)){
        fp->deq = PrevQue(fp->enq,fp->queCnt);
        fp ->full=0;
        return(1);
      }
    }

    /*データのeofの検知*/
    if(isEOF(fp)){
      frm->eof=1;
      return(1);
    }else{
      /*スペースをdelimitorとしてトークン分割*/
      fp->recCnt++;
      fp->curPnt=strToken(frm->pnt+frm->recCnt*frm->fldCnt, frm->fldCnt,&fp->chrCnt,fp);
      frm->recCnt++;
    }
  }
  return(1);
}

/**
 * # FUNCTION #
 * mssFldRecKey構造体の初期化
 */
struct mssFldRecKey *mssInitFRK(int fldCnt, MssOptKEY *optKey, char *path)
{

  struct mssFldRecKey *frk;
  int pid;

  /*FRK構造体領域の確保*/
  frk=mssMalloc(sizeof(struct mssFldRecKey),"initFRK");

  /*初期化*/
  frk->frPnt     = NULL;
  frk->pnt       = mssMalloc(sizeof(char *)*fldCnt,"readFldRecFRK");
  frk->fldCnt    = fldCnt;
  frk->chrCnt    = 0;
  frk->eof       = 0;
  frk->keyRecCnt = 0;
  frk->keyNo     = 0;
  frk->firstRead = 1;
  frk->alcRecCnt = 0;
  frk->recPnt    = 0;
  frk->optKey    = optKey;
  frk->byFile    = 0;
  frk->curRecNo  = 0;
  frk->fpw       = NULL;
  frk->fpr       = NULL;
  frk->fr        = NULL;
  
  pid=getpid();
  sprintf(frk->fName,"%s/xt##%d-FRK",path,pid);

  /*シグナルハンドラの設定*/
  mssGV.usedTempFileFlg=1;
  mssSetSignalHandler();

  return(frk);
}

/**
 * # FUNCTION #
 * mssFldRecKey構造体の開放
 */
void mssFreeFRK(struct mssFldRecKey *frk)
{
  if(frk==NULL) return;
  if(0==access(frk->fName,R_OK & W_OK)) unlink(frk->fName);
  mssFree(frk->pnt);
  mssFree(frk->frPnt);
  mssFreeFldRec(frk->fr);
  mssCloseFPR(frk->fpr);
  mssFree(frk); /* 変更 2004/10/31 */
}

/**
 * # FUNCTION #
 * readFRK内部で利用するキーブレーク判定
 * 同じキーの値を持つ最初の行と現在読み込んだ行を比較する。
 */
static int keyBreakFRK(struct mssFldRecKey *frk)
{
  int i,fn;

  /*初回の読み込みはキーブレークでない*/
  if(frk->firstRead){
    frk->firstRead=0;
    return(0);
  }

  /*最終行の時はキーブレーク*/
  if(frk->eof == 1) return(1);

  switch(frk->optKey->diffSame){

  /*k=#allDiff#の場合はキーブレーク*/
  case 1:
    return(1);

  /*k=#allSame#の場合はnotキーブレーク*/
  case 2:
    return(0);

  default:
    for(i=0; i<frk->optKey->flds->cnt; i++){
      fn=MssFlds2num(frk->optKey->flds,i);
      if( 0 != strcmp(*(frk->frPnt+(frk->recPnt-1)*frk->fldCnt+fn) ,
                      *(frk->frPnt+(frk->recPnt  )*frk->fldCnt+fn)) )
         return(1);
    }

    /*上記の条件をクリアしたということはキーブレークでない*/
    return(0);
  }
}

/**
 * # FUNCTION #
 * mssFldRecKey構造体のfrPntバッファにおける,recNo番目のレコードを書き出す。
 */
static void writeRecFRK(struct mssFldRecKey *frk,int recNo)
{
  int j;

  for(j=0; j<frk->fldCnt; j++){
    mssWriteStr(*(frk->frPnt+frk->fldCnt*recNo+j),frk->fpw);
    if(frk->fldCnt-1 == j) mssWriteRet(frk->fpw);
    else                   mssWriteDlm(frk->fpw);
  }
}

/**
 * # FUNCTION #
 * mssFldRecKey構造体に指定のキー項目が同じ値の行を全て読み込む
 * ファイルバッファがFULLになれば、一時ファイルに書き出す。
 * よって、一時ファイルを利用するかどうかは、openFPRで指定したqueサイズに依存。
 * 読み込まれたデータへは、mssReadFldRecFRKにてアクセスできる。
 * EOFを読み込んだときは、EOFを返す。
 *
 * frPntへのデータの蓄積について
 * 例えば "a b c","x y z"の２行のデータを読み込んだ場合、
 * *(frPnt+0)="a",*(frPnt+1)="b",*(frPnt+2)="c",*(frPnt+3)="x",*(frPnt+4)="y",*(frPnt+5)="z"
 * となる。
 * この関数では同じキーの値を持つ行を全て読み込むため、行数の上限は事前に
 * わからない。そこで、realloc関数にて動的に領域を確保する。
 * frk->alcRecCntに現在alloc関数で確保した行数を記録している。
 * alcRecCntは、この関数の呼び出し毎にクリアされることはない。
 */
int mssReadFRK(struct mssFPR *fp, struct mssFldRecKey *frk)
{
  int i;
  char *curTop;
/*
frkFlg=1;
fprintf(stderr,"mssReadFRK start e,d=%d,%d\n",fp->enq,fp->deq);
*/
  if(frk->eof) return(EOF);

  /*初期化*/
  frk->keyRecCnt=0;
  frk->byFile=0;
  frk->curRecNo=0;
  mssCloseFPR(frk->fpr);
  frk->fpr=NULL;
  mssFreeFldRec(frk->fr);

 /* 2004/01/23 上のmssFreeFldRecの後に再びmssFreeFldRecを呼び出す可能性があるため、frk->frをNULLクリアする */
  frk->fr=NULL;

  /* 前回の最終レコードをfrPntの先頭にコピーする。*/
  /* 前回の最終レコードは今回のキーと同じである。 */
  if(!frk->firstRead){
    for(i=0; i<frk->fldCnt; i++){
      *(frk->frPnt+i)=*(frk->frPnt+frk->fldCnt*frk->recPnt+i);
    }
    frk->keyRecCnt=1;

    /*読み込み位置はfrPnt+1から*/
    frk->recPnt=1;
  }

  while(1){

    /*データの読み込み*/
    if(needFileRead(fp)){
      /*データバッファがfull(-1)であればファイル書き出し*/
      if(-1==readFPRfile(fp)){
        /*初回のwrite時にファイルオープン*/
        if(!frk->byFile) {
          frk->fpw=mssOpenFPW(frk->fName,0,0);
        }
        /*frPntの内容を最終行を除いて一時ファイルにwrite*/
        for(i=0; i<frk->recPnt-1; i++){
          writeRecFRK(frk, i);
        }

        /*frPnt最終行はfrPnt先頭行にコピー(keyBreakのため)*/
        for(i=0; i<frk->fldCnt; i++){
          *(frk->frPnt+i)=*(frk->frPnt+frk->fldCnt*(frk->recPnt-1)+i);
        }
        fp->deq = PrevQue(fp->enq,fp->queCnt);
        frk->byFile=1;
        fp->full=0;
        frk->recPnt=1; /*次の読み込みはfrPntの２行目から*/
        continue;
      }
    }

    curTop=fp->curPnt;

    /*eofの検知*/
    if(isEOF(fp)){
      frk->eof=1;
      if(frk->firstRead) return(EOF); /*NULL データの場合*/
    }else{
      /*スペースをdelimitorとしてトークン分割*/
      if(frk->alcRecCnt<=frk->recPnt){
        frk->frPnt=mssRealloc(frk->frPnt,
                            sizeof(char *)*(frk->recPnt+1)*frk->fldCnt,
                            "readFRK");
        frk->alcRecCnt++;
      }
      fp->curPnt=strToken(frk->frPnt+frk->recPnt*frk->fldCnt, frk->fldCnt,
                          &fp->chrCnt,fp);
      fp->recCnt++;
    }

    if(keyBreakFRK(frk)){
      if(frk->byFile){
        for(i=0; i<frk->recPnt; i++){
          writeRecFRK(frk, i);
        }
        mssCloseFPW(frk->fpw);
        frk->fpr=mssOpenFPR(frk->fName,4);
        frk->fr=mssInitFldRec(frk->fldCnt);
      }

      /* -k -q 指定時のバグ修正 2004/05/19                        */
      /* キーブレークした後のdeqは、enqの一つ前とは限らない。     */
      /* 二つ前になる可能性もある。そこで次のようにdeqを求める    */
      /* 上でstrTokenした行(次のキーの先頭行)の先頭位置(curTop)が */
      /* 属するQUE番号をdeqとする。                               */
      /* ただし、bufの先頭+MssRecordMaxLen上にある場合は0とする。 */
      fp->deq = (curTop-fp->buf-MssRecordMaxLen)/fp->queSize;
      if(fp->deq<0) fp->deq=0;
/*      fp->deq = PrevQue(fp->enq,fp->queCnt);*/
      frk->keyNo++; /*キーの通し番号をカウントアップ*/
      frk->curRecNo=0;
      return(1);
    }

    /*frk->frPntの読み込み行位置をカウントアップ*/
    frk->recPnt++;

    /*キー内のレコード数をカウントアップ*/
    frk->keyRecCnt++;
  }
  return(1);
}

/**
 * # FUNCTION #
 * mssFldRecKey構造体において、一行読み込みpntに各項目文字列をセットする。
 * 読み込むデータがなければEOFを返す。
 */
int mssReadFldRecFRK(struct mssFldRecKey *frk){
  int i;

  if(frk->curRecNo>=frk->keyRecCnt)return(EOF);

  if(frk->byFile){
    if(EOF==mssReadFldRec(frk->fpr,frk->fr)){
      return(EOF);
    }else{
      for(i=0; i<frk->fldCnt; i++){
        *(frk->pnt+i)=*(frk->fr->pnt+i);
      }
    }
   
  }else{
    for(i=0; i<frk->fldCnt; i++){
      *(frk->pnt+i)=*(frk->frPnt+frk->curRecNo*frk->fldCnt+i);
    }
  }
  frk->curRecNo++;
  return(1);
}

/**
 * # FUNCTION #
 * mssFldRecKey構造体において、先頭にシークする
 */
void mssSeekTopFRK(struct mssFldRecKey *frk)
{
  frk->curRecNo=0;
  if(frk->byFile){
    mssCloseFPR(frk->fpr);
    frk->fpr=mssOpenFPR(frk->fName,4);
    mssFreeFldRec(frk->fr);
    frk->fr=mssInitFldRec(frk->fldCnt);
  }
}

/**
 * # FUNCTION #
 * データの先頭行を読み込み、項目数を返す。EOFを読み込んだときはEOFを返す。
 * データの読み込みをおこなうがfp->curを動かさない
 * よってこの関数の次にmssReadFldRecなどで読みこめば、もう一度トップ行から
 * 読みこまれることになる
 * fpのバッファはいっさい変更しない。
 * そのかわり、新にメモリをmallocで確保し、そこへのポインタをrec構造体
 * にセットする。
 */
int mssGetFldCntOnData(struct mssFPR *fp)
{
           char *start;
           int spcCnt;
  register int i; /*高速化のためにregister変数を利用*/

  /*現在位置がバッファ最後のMssRecordMaxLen文字内にかかっていたらファイル読み込み*/
  if(needFileRead(fp)){
    readFPRfile(fp);
    /*行単位の読み込みの場合は常にdeqはenqの１つ前*/
    fp->deq=PrevQue(fp->enq,fp->queCnt);
  }

  /*eofの検知*/
  if(isEOF(fp)) return(EOF);

  start=fp->curPnt; /*fp->curは動かせないので*/
  spcCnt=0;
  i=1;
  while( *start != '\n' ){
    if(*start==MssFieldDelim) spcCnt++;
    start++; i++;
    if(i>=MssRecordMaxLen){
      mssShowErrMsg("exceed %d characters per record\n",MssRecordMaxLen);
      mssEnd(mssErrorNoDefault);
    }
  }

  return(++spcCnt); /*スペースの数+1が項目数*/
}

/**
 * # FUNCTION #
 * 先頭行にシークする(ファイルの先頭ではない)。
 * 標準入力に対しては、この関数を利用するとエラーとなる。
 * ただし、mssReopenFPRsort関数の利用後であれば利用可能。
 */
void mssSeekTopFPR(struct mssFPR *fp)
{
  int i;

  if(0==fp->fName && !fp->sort){
    mssShowErrMsg("can not seek on stdin");
    mssEnd(mssErrorNoDefault);
  }
  fp->curPnt = 0;

  /*エンキュー、デキューポインタの初期化*/
  /*enq==deq then 空, deq-1=enq then full*/
  fp->deq =0;
  fp->enq =0;
  fp->full=0;
  fp->last=0;

  fp->recCnt = 0;
  fp->chrCnt = 0;

  /*カレントポインターのセット*/
  fp->curPnt=fp->buf+fp->queSize*fp->queCnt+MssRecordMaxLen;

  if(fp->sort){
    fp->readPnt=fp->curPnt;
    /*キューの解放&入力ファイルのクローズ (追加 2004/10/31) */
    TTfree(fp->sd->tt);
    for(i=0; i<=fp->sd->iEnd-fp->sd->iStart; i++){
      mssCloseFPR(fp->sd->iFile[i]);
    }
    
    setFirstLineTT(fp->sd, fp->sd->iStart,fp->sd->iEnd);
  }else{
    if(fp->zflg) gzseek(fp->zfd, 0, 0);
    else          fseek(fp->fp , 0, 0);
    mssSkipToBody(fp);
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * ファイルオープンクローズ関連
 * ----------------------------------------------------------------------------
 * MUSASHIでは、XMLtableの扱いに関して、以下の３つの機能を実現している。
 * 1. 大量データを高速に扱うためのリングバッファの実装
 * 2. 圧縮ファイルの自動解凍(この機能により、圧縮ファイルであっても、
 *    通常ファイルと同様の扱いが可能となる。)
 * 3. キーブレーク処理やファイル間のマッチング処理のために、あるキー項目
 *    による並べ換えを裏で時動的に実行する。
 * これらの処理は全て、各種open関数で初期化もしくは実行される。
 * mssOpenFPR、mssOloseFPRは、1,2の機能を実装しており、mssReopenFPRsortは3の
 * 機能を担う。
 *
 * さらに、MUSASHIでは、mssOpenFPU関数で更新ファイルの扱いもサポートしている。
 */

/**
 * # FUNCTION #
 * ファイルのオープンエラーの表示および停止。
 */
static void fileOpenErr(char *fName)
{
  mssShowErrMsg("file open error :\"%s\"",fName);
  mssEnd(mssErrorNoDefault);
}

/**
 * # FUNCTION #
 * 読み込みオープン
 * ファイル名(fileName)がNULLならば、標準入力のオープンとなる。
 * queCnt: リングバッファにおけるキューの数
 *         キューのサイズは、ReadCnt(default:4)*PipeSize(default:4096)で決まる。
 *         すなわち、デフォルトでは16K*queCntバイトのバッファとなる。
 *         queCntは2の巾乗でなければならない,ex. 2,4,8,16...
 * queCntを4と設定すれば、デフォルトで16K*4=64Kバイトのデータをバッファ
 * を確保することになる。
 *
 * ファイル名の拡張子が.gzならばfzopen、その他の場合はfopenを利用(2004/08/24)
 */
struct mssFPR *mssOpenFPR(char *fileName,int queCnt)
{
  struct mssFPR *fp;
  int len;
  char *pos;

  fp=mssMalloc(sizeof(struct mssFPR), "openFPR");

  /*gzopen*/
  fp->zflg=0;
  if(fileName == NULL){  /*stdin open*/
    fp->fp=stdin; //fopen("/dev/stdin","rb");
    //if(fp->fp  == NULL) fileOpenErr(fileName);
    fp->fName=mssStrdup("stdin");

  }else{                 /* file open */
    len=strlen(fileName);
    if(len>=3){
      pos=fileName+len-3;
      if(*pos=='.' && *(pos+1)=='g' && *(pos+2)=='z'){
        fp->zflg=1;
      }
    }
    if(fp->zflg){
      fp->zfd=gzopen(fileName,"rb");
      if(fp->zfd == NULL) fileOpenErr(fileName);
    }else{
      fp->fp =fopen(fileName,"rb");
      if(fp->fp  == NULL) fileOpenErr(fileName);
    }
    fp->fName=mssStrdup(fileName);
  }

  /*エンキュー、デキューポインタの初期化*/
  fp->sort=0;
  fp->deq =0;
  fp->enq =0;
  fp->full=0;
  fp->last=0;
  fp->queCnt =queCnt;
  fp->queSize=ReadCnt*PipeSize;
/*
  if(fileName==NULL){
    fp->fName=mssStrdup("stdin");
  }else{
    fp->fName=mssStrdup(fileName);
  }
*/

  /*バッファ領域の確保*/
  fp->buf=mssCalloc(sizeof(char)*(fp->queSize*fp->queCnt+MssRecordMaxLen+1), "openFPR");

  /*カレントポインタをバッファの最後+1にもっていく(初回のreadFileのため)*/
  fp->curPnt=fp->buf+fp->queSize*fp->queCnt+MssRecordMaxLen;

  fp->readPnt=NULL;
  fp->sd     =NULL;
  /*fp->zflg   =1;*/

  fp->recCnt=0;
  fp->chrCnt=0;
  return(fp);
}

/**
 * # FUNCTION #
 * データをソートしてファイル読み込み再オープン
 * 通常ファイルとしてオープンしたinFileのデータを、指定のソート項目fldsで
 * 内部的にソーティングし、読み込みファイルとして再オープンする。
 * この関数を呼び出した後は、通常のmssOpenFPRでオープンしたと同様に扱うことが
 * 可能。
 * 引数fldCntには、データの全項目数を指定する。
 * queCnt: 新たに再オープンされるリングバッファにおけるキューの数
 *         キューのサイズは、ReadCnt(default:4)*PipeSize(default:4096)で決まる。
 *         すなわち、デフォルトでは16K*queCntバイトのバッファとなる。
 *         queCntは2の巾乗でなければならない,ex. 2,4,8,16...
 * queCntを4と設定すれば、デフォルトで16K*4=64Kバイトのデータをバッファ
 * を確保することになる。
 */
struct mssFPR *mssReopenFPRsort(
  struct mssFPR *inFile,
  int queCnt,
  struct mssFields *flds, /*ソート項目*/
  int fldCnt,       /*項目数*/
  char *tmpPath)    /*ワークファイルパス名*/
{     /*ワークファイルパス名*/

  struct mssFPR *fp;

  /*エンキュー、デキューポインタの初期化(mssOpenFPRと同じことをする)*/
  fp=mssMalloc(sizeof(struct mssFPR), "openFPR");
  fp->recCnt=0;
  fp->chrCnt=0;
  fp->deq =0;
  fp->enq =0;
  fp->full=0;
  fp->last=0;
  fp->queCnt =queCnt;
  fp->queSize=ReadCnt*PipeSize; /*FPRのqueSizeと同じにしておく*/

  fp->fName=mssMalloc(sizeof(char)*(strlen(inFile->fName)+1),"reopenFPRsort");
  strcpy(fp->fName,inFile->fName);

  /*バッファ領域の確保*/
  fp->buf=mssMalloc(sizeof(char)*fp->queSize*fp->queCnt+MssRecordMaxLen*2,"initDAT");

  /*カレントポインタをバッファの最後+1にもっていく(初回のreadFileのため)*/
  fp->curPnt =fp->buf+fp->queSize*fp->queCnt+MssRecordMaxLen;

  /*--------------------------- 以下のパートはmssOpenFPRにない*/
  /*読み込みポインタのセット*/
  fp->readPnt=fp->curPnt;

  /*ソートデータ構造の初期化*/
  fp->sd=initSD(flds,fldCnt,tmpPath);
  /*preSortの実行*/
  preSort(fp->sd,inFile);

  /*preSort以降は、readFPRfileでこのフラグによって、*/
  /*tnTreeから読み込むことになる*/
  fp->sort=1;

  /*元の入力ファイルを閉じる*/

  mssCloseFPR(inFile);

  return(fp);
}

/**
 * # FUNCTION #
 * 読み込みファイルクローズ
 */
void mssCloseFPR(struct mssFPR *fp)
{

  if(fp==NULL)return;

  if(fp->sort){
    endPreSort(fp->sd);
    freeSD(fp->sd);
    /*close(fp->fd);*/
  }else{

    if(fp->zflg){
      gzclose(fp->zfd);
    }else{ 
      if(fp->fp != stdin){
        fclose(fp->fp);
      }
    }
  }
  mssFree(fp->fName);
  mssFree(fp->buf);
  mssFree(fp);
}

/**
 * # FUNCTION #
 * 更新ファイルオープン
 * 標準入力および圧縮ファイルは更新ファイルとしてオープンできない。
 * queCnt: リングバッファにおけるキューの数
 *         キューのサイズは、ReadCnt(default:4)*PipeSize(default:4096)で決まる。
 *         すなわち、デフォルトでは16K*queCntバイトのバッファとなる。
 *         queCntは2の巾乗でなければならない,ex. 2,4,8,16...
 * queCntを4と設定すれば、デフォルトで16K*4=64Kバイトのデータをバッファ
 * を確保することになる。
 */
struct mssFPR *mssOpenFPU(char *fileName, int queCnt)
{
  struct mssFPR *fp;

  fp=mssMalloc(sizeof(struct mssFPR), "openFPR");

  if(fileName == NULL){
    mssShowErrMsg("cannot open stdin as update file"); /*stdin*/
    mssEnd(mssErrorNoDefault);
  }else{
    fp->fp=fopen(fileName,"rb+");
  }
  if(fp->fp == NULL) fileOpenErr(fileName);

  /*エンキュー、デキューポインタの初期化*/
  fp->sort=0;
  fp->deq =0;
  fp->enq =0;
  fp->full=0;
  fp->last=0;
  fp->queCnt =queCnt;
  fp->queSize=ReadCnt*PipeSize;
  if(fileName==NULL){
    fp->fName=mssMalloc(sizeof(char)*10,"openFPU");
    strcpy(fp->fName,"stdin");
  }else{
    fp->fName=mssMalloc(sizeof(char)*(strlen(fileName)+1),"openFPU");
    strcpy(fp->fName,fileName);
  }

  /*バッファ領域の確保*/
  fp->buf=mssMalloc(sizeof(char)*fp->queSize*fp->queCnt+MssRecordMaxLen, "openFPU");

  /*カレントポインタをバッファの最後+1にもっていく(初回のreadFileのため)*/
  fp->curPnt=fp->buf+fp->queSize*fp->queCnt+MssRecordMaxLen;

  fp->readPnt=NULL;
  fp->sd     =NULL;
  fp->zflg   =0;

  fp->recCnt=0;
  fp->chrCnt=0;
  return(fp);
}

/**
 * # FUNCTION #
 * 更新ファイルクローズ
 */
void mssCloseFPU(struct mssFPR *fp)
{

  fclose(fp->fp);
  mssFree(fp->fName);
  mssFree(fp->buf);
  mssFree(fp);
}
