/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるその他の関数
 * ============================================================================
 */

#include <musashi/mssMisc.h>
#include <musashi/mssBase.h>
#include <musashi/mssValue.h>
#include <musashi/mssMisc.h>
#include <musashi/mssInput.h>
#include <musashi/qs5.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <float.h>

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 赤黒木(Red Black Tree)
 * ----------------------------------------------------------------------------
 * 参考文献: 渡部敏正『データ構造とアルゴリズム』共立出版,2000,6章4節.
 * p.224 下から9行目は誤り!!
 * 誤) if (v == NULL ) return(1) -> 正) if (v == NULL ) return(0)
 */

/**
 * # FUNCTION #
 */
static int RBhasLeftNode(struct mssRBnode *v)
{
  if(v->left->rank !=0){
    return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 */
int mssRBisExternalNode(struct mssRBnode *v)
{
  if(v->rank==0)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBisTopNode(struct mssRBnode *v)
{
  if(v->parent->rank==0){
    return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBisLeftNode(struct mssRBnode *v)
{
  if(v==v->parent->left)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBisRedNode(struct mssRBnode *v)
{
  if(v==NULL) return(0);
  if(v->rank == v->parent->rank)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
struct mssRBnode *mssRBfindMax(struct mssRBnode *v)
{
  while(!mssRBisExternalNode(v->right))
    v=v->right;
  return(v);
}

/**
 * # FUNCTION #
 */
struct mssRBnode *mssRBfindMin(struct mssRBnode *v)
{
  while(!mssRBisExternalNode(v->left))
    v=v->left;
  return(v);
}

/**
 * # FUNCTION #
 */
static struct mssRBnode *RBfind_brother(struct mssRBnode *v)
{
  if(RBisLeftNode(v))
    return(v->parent->right);
  return(v->parent->left);
}

/**
 * # FUNCTION #
 */
static struct mssRBnode *RBmakeNode(int keyType)
{
  struct mssRBnode *new;
  struct mssRBkey  *key;
  new=(struct mssRBnode *)mssMalloc(sizeof(struct mssRBnode),"RBmkNode");
  key=(struct mssRBkey  *)mssMalloc(sizeof(struct mssRBkey ),"RBmkNode");
  new->key=key;
  mssVinit(&new->key->v,keyType);
  new->rank = 0;
  new->left=new->right=NULL;
  return new;
}

/**
 * # FUNCTION #
 */
static void RBfreeNode(struct mssRBnode *v)
{
  if(mssRBisExternalNode(v)){
    mssFree(v->key);
    mssFree(v);
    return;
  }else{
    mssFree(v->key);
    mssFree(v);
    return;
  }
}

/**
 * # FUNCTION #
 */
static void RBfreeAllNode(struct mssRBnode *v)
{
  if(mssRBisExternalNode(v)){
    mssFree(v->key);
    mssFree(v);
    return;
  }else{
    RBfreeAllNode(v->left);
    RBfreeAllNode(v->right);
    RBfreeNode(v);
    return;
  }
}

/**
 * # FUNCTION #
 */
struct mssRBnode *RBmemberNode(struct mssRBnode *n, MssValue value)
{
  struct mssRBnode *vv;
  int cmp;

  if(mssRBisExternalNode(n)) return(n);

  cmp=mssVcmp(value,n->key->v);
  if( 0 > cmp ){
    vv=RBmemberNode(n->left,value);
  }else if( 0 < cmp ){
    vv=RBmemberNode(n->right,value);
  }else{
    return(n);
  }
  return(vv);
}

/**
 * # FUNCTION #
 */
static void RBaddExternalNodes(struct mssRBnode *n)
{
  n->left = RBmakeNode(n->key->v.vType);
  n->right= RBmakeNode(n->key->v.vType);
  n->left->parent =n;
  n->right->parent=n;
}

/**
 * # FUNCTION #
 */
static void RBsingleRotate(struct mssRBnode *v)
{
  struct mssRBnode *p, *pp,*ppp;

  p  =v->parent;
  pp =p->parent;
  ppp=pp->parent;

  if(RBisLeftNode(pp)){
    ppp->left=p;
  }else{
    ppp->right=p;
  }
  if(RBisLeftNode(v)){
    pp->left=p->right;
    pp->left->parent=pp;
    p->right=pp;
    pp->parent=p;
  }else{
    pp->right = p->left;
    pp->right->parent = pp;
    p->left = pp;
    pp->parent = p;
  }
  p->parent=ppp;
}

/**
 * # FUNCTION #
 */
static void RBdouble_rotate(struct mssRBnode *v)
{
  struct mssRBnode *p;
  p=v->parent;
  if(RBisLeftNode(v)){
    RBsingleRotate(v->left);
    RBsingleRotate(p);
  }else{
    RBsingleRotate(v->right);
    RBsingleRotate(p);
  }
}

/**
 * # FUNCTION #
 */
static void RBrebalanceOnInsert(struct mssRBnode *v)
{
  struct mssRBnode *w, *p, *pp;
  int p_is_left;
  
  p=v->parent;
  pp=p->parent;
  if(RBisTopNode(p)){
    return;
  }
  if(RBisRedNode(p)){
    p_is_left=0;   
    if((p_is_left=RBisLeftNode(p))){  
      w=pp->right;
    }else{
      w=pp->left;
    }
    if(RBisRedNode(w)){   
      (pp->rank)++;
      if(!RBisTopNode(pp) && RBisRedNode(pp->parent)){
        RBrebalanceOnInsert(pp);
      }
    }else{
      if(RBisLeftNode(v)){
        if(p_is_left){
          RBsingleRotate(v);
        }else{
          RBdouble_rotate(v);
        }
      }else{
        if(p_is_left){
          RBdouble_rotate(v);
        }else{
          RBsingleRotate(v);
        }
      }
    }
  }
}

/**
 * # FUNCTION #
 */
static void RBdetermineAB(struct mssRBnode **a, struct mssRBnode **b, struct mssRBnode *v, struct mssRBnode *u)
{
  if(RBisLeftNode(v)){
    *a=u->right;
    *b=u->left;
  }else{
    *a=u->left;
    *b=u->right;
  }
}

/**
 * # FUNCTION #
 */
static void RBrebalanceOnDelete(struct mssRBnode *v)
{
  struct mssRBnode *p,*u,*a,*b;
  int v_is_left, p_was_red;

  p=v->parent;
  if(v->rank+1 >= p->rank)
    return;
  u=RBfind_brother(v);
  RBdetermineAB(&a,&b,v,u);
  v_is_left=RBisLeftNode(v);
  p_was_red=RBisRedNode(p);
  if(RBisRedNode(u)){
    if(RBisRedNode(a) || RBisRedNode(b))
      return;
    RBsingleRotate(a);
    RBrebalanceOnDelete(v);
  }else{
    if(!RBisRedNode(a)){
      if(!RBisRedNode(b)){
        (p->rank)--;
        if(!p_was_red)
          RBrebalanceOnDelete(p);
      }else{
        RBdouble_rotate(b);
        (b->rank)++;
        (p->rank)--;
      }
    }else{
      RBsingleRotate(a);
      (u->rank)++;
      (u->rank)--;
    }
  }
}

/**
 * # FUNCTION #
 */
static void RBptree(struct mssRBnode *p,int h)
{
  int i;

  if(!mssRBisExternalNode(p)){
    RBptree(p->left, h+1);
    for(i=1; i<=h; i++) fprintf(stderr,"    ");
    switch(p->key->v.vType){
    case DBL:
      printf("key=%g ",p->key->v.v.d); break;
    case INT:
      printf("key=%d ",p->key->v.v.i); break;
    case STR:
      printf("key=%s ",p->key->v.v.s); break;
    case ADD:
      printf("key=%p ",p->key->v.v.a); break;
    case USI:
      printf("key=%hu ",p->key->v.v.usi); break;
    }
    printf("address=%p\n",p->key->add);
    RBptree(p->right,h+1);
  }
}

/**
 * # FUNCTION #
 * RBtreeから指定のvalueのノードを返す。
 * 存在しなければNULLを返す。
 */
struct mssRBnode *mssRBmember( struct mssRBnode *v, MssValue value)
{

  v = RBmemberNode(v, value);

  if(mssRBisExternalNode(v)) return(v);
  else                    return(NULL);
}


/**
 * # FUNCTION #
 * RBtreeにデータを追加
 * 既にキーが存在すれば1を返す。挿入が成功すれば0を返す。
 */
int mssRBinsert( struct mssRBnode *v, MssValue value, void *add)
{

  v = RBmemberNode(v, value);

  /*キーが存在しなければ追加*/
  if(mssRBisExternalNode(v)){
    v->key->v=value;
    v->key->add=add;
    v->rank=1;
    RBaddExternalNodes(v);
    RBrebalanceOnInsert(v);

  /*キーが存在すれば、読み込みフラグをON*/
  }else{
    return(1);
  }

  return(0);
}

/**
 * # FUNCTION #
 * RBtreeにデータを追加
 * 同一キーがあればそのノードアドレスを返す
 * 挿入に成功すればNULLを返す
 */
struct mssRBnode *mssRBinsertAdd( struct mssRBnode *v, MssValue value, void *add)
{

  v = RBmemberNode(v, value);

  /*キーが存在しなければ追加*/
  if(mssRBisExternalNode(v)){
    v->key->v=value;
    v->key->add=add;
    v->rank=1;
    RBaddExternalNodes(v);
    RBrebalanceOnInsert(v);

  /*キーが存在すれば、そのアドレスを返す*/
  }else{
    return(v);
  }

  return(NULL);
}

/**
 * # FUNCTION #
 *指定のキーを持つノードを削除
 * 既にキーが存在しなければ1を返す。削除が成功すれば0を返す。
 */
int mssRBdelete(struct mssRBnode *v, MssValue value)
{
  struct mssRBnode *w,*x;

  v=RBmemberNode(v, value);

  if(mssRBisExternalNode(v)){
    return(1);
  }
  if(RBhasLeftNode(v)){
    w=mssRBfindMax(v->left);
    if(w->parent != v){
      w->parent->right = w->left;
      w->left->parent  = w->parent;
    }
    x=w->left;
    RBfreeNode(w->right);
    /*free(w->right->key);*/
    /*free(w->right);*/

    if(RBisLeftNode(v)){
      v->parent->left=w;
    }else{
      v->parent->right=w;
    }
    if(w->parent != v){
      w->left = v->left;
      w->left->parent = w;
    }
    w->rank = v->rank;
    w->parent = v->parent;
    w->right = v->right;
    w->right->parent=w;
    RBfreeNode(v);
    /*free(v->key);*/
    /*free(v);*/
  }else{
    w=v->right;
    if(RBisLeftNode(v)){
      v->parent->left=w;
    }else{
      v->parent->right=w;
    }
    w->parent=v->parent;
    RBfreeNode(v->left);
    RBfreeNode(v);
    /*free(v->left->key);*/
    /*free(v->left);*/
    /*free(v->key);*/
    /*free(v);*/
    x=w;
  }

  RBrebalanceOnDelete(x);
  return(0);
}

/**
 * # FUNCTION #
 * RBtreeをツリーとして書き出す(デバッグ用)
 */
void mssRBshowTree(char *s,struct mssRBnode *pp)
{
  fprintf(stderr,"%s\n",s);
  RBptree(pp,0);
}

/**
 * # FUNCTION #
 * RBtreeのメモリ領域解放
 */
void mssRBfree(struct mssRBnode *v)
{
  RBfreeAllNode(v->left);
  mssFree(v->key);
  mssFree(v);
}

/**
 * # FUNCTION #
 * mssRBnodeの初期化(ノード作成)
 */
struct mssRBnode *mssRBinit(int keyType)
{
  struct mssRBnode *rb;
  rb               = RBmakeNode(keyType);
  rb->parent       = rb;
  rb->right        = NULL;
  rb->rank         = 0;
  rb->left         = RBmakeNode(keyType);
  rb->left->parent = rb;
  return(rb);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 範囲インデックスの作成
 * ----------------------------------------------------------------------------
 * ex) 以下に示す4つの範囲集合がある場合を考える。
 *    1: (10_20]
 *    2: (12_15)
 *    3: [13_22)
 *    4: [22_*]
 * 以下の範囲インデックスを作成する。
 * nodeNo v   LT LE GE GT
 *      1 10   -  -  -  1
 *      2 12   -  -  -  2
 *      3 13   -  -  3  -
 *      4 15   2  -  -  -
 *      5 20   -  1  -  -
 *      6 22   3  -  4  -
 *      7 *    -  4  -  -
 */

/**
 * # FUNCTION #
 * 範囲(st,ed)をキーに行データをRange構造体にセットし、そのアドレスをかえす
 */
static struct Range *setRange( char *rec, char *st, char *ed, int nsFlg, int stEQflg, int edEQflg){
  struct Range *range;
  int invalid=0;
  int cmp;

  range=mssMalloc(sizeof(struct Range),"setRange");

  range->rec=rec;

  if( nsFlg==1 ){
    mssVinit(&range->st,DBL);
    mssVinit(&range->ed,DBL);
    if(MssIsNull(st)) range->st.v.d = -DBL_MAX;
    else              range->st.v.d = atof(st);
    if(MssIsNull(ed)) range->ed.v.d = DBL_MAX;
    else              range->ed.v.d = atof(ed);
    /*st>ed or st==ed && 以上、以下でないときは無効*/
    if(range->st.v.d>=range->ed.v.d){
      if(range->st.v.d>range->ed.v.d) invalid=1;
      else if(stEQflg!=1 || edEQflg!=1) invalid=1;
    }
  }else{
    mssVinit(&range->st,STR);
    mssVinit(&range->ed,STR);
    if(MssIsNull(st)) range->st.v.s = mssStrdup("\x00\x00\x00\x00");
    else              range->st.v.s = mssStrdup(st);
    if(MssIsNull(ed)) range->ed.v.s = mssStrdup("\xFF\xFF\xFF\xFF");
    else              range->ed.v.s = mssStrdup(ed);
    /*st>ed or st==ed && 以上、以下でないときは無効*/
    cmp=strcmp(range->st.v.s,range->ed.v.s);
    if(cmp>0)                                   invalid=1;
    else if(cmp==0 && (stEQflg!=1 || edEQflg!=1)) invalid=1;
  }

  if(invalid){
    mssFree(range->st.v.s); /* 追加 2004/10/31 */
    mssFree(range->ed.v.s); /* 追加 2004/10/31 */
    mssFree(range->rec);    /* 追加 2004/10/31 */
    mssFree(range);
    range=NULL;
  }else{
    range->stEq=stEQflg;
    range->edEq=edEQflg;
  }
  return(range);
}

/**
 * # FUNCTION #
 * 範囲テーブル構造体の領域解放
 */
static void freeRangeTbl(struct RangeTbl *rangeTbl, int nsFlg){
  int i;
  for(i=0; i<rangeTbl->cnt; i++){
    if(nsFlg==0){ /*文字*/
      mssFree( (*(rangeTbl->range+i))->st.v.s);
      mssFree( (*(rangeTbl->range+i))->ed.v.s);
    }
    mssFree( (*(rangeTbl->range+i))->rec);
    mssFree( *(rangeTbl->range+i));
  }
  mssFree(rangeTbl->range);
  mssFree(rangeTbl);
}


/**
 * # FUNCTION #
 * 範囲開始条件での並べ換え用比較ルーチン
 */
static int qscmpST(const void **a, const void **b){
  return( mssVcmp( (*(struct Range **)a)->st,
                   (*(struct Range **)b)->st) );
}

/**
 * # FUNCTION #
 * 範囲終了条件での並べ換え用比較ルーチン
 */
static int qscmpED(const void **a, const void **b){
  return( mssVcmp( (*(struct Range **)a)->ed,
                   (*(struct Range **)b)->ed) );
}

/**
 * # FUNCTION #
 * RangeをFrom条件で並べ換える
 */
static void sortByFrom(struct RangeTbl *rangeTbl){
  qsort5(rangeTbl->range,rangeTbl->cnt,sizeof(struct Range *),
                     (int (*))qscmpST);
/*
  qsort(rangeTbl->range,rangeTbl->cnt,sizeof(struct Range *),
                     (int (*)(const void *,const void *))qscmpST);
*/
}

/**
 * # FUNCTION #
 * RangeをTo条件で並べ換える
 */
static void sortByTo(struct RangeTbl *rangeTbl){
    qsort5(rangeTbl->range,rangeTbl->cnt,sizeof(struct Range *),
                       (int (*))qscmpED);
/*
    qsort(rangeTbl->range,rangeTbl->cnt,sizeof(struct Range *),
                       (int (*)(const void *,const void *))qscmpED);
*/
}

/**
 * # FUNCTION #
 * 範囲テーブル表構造体(RangeTbl)の表示
 * for debug
 */
static void showRangeTbl(struct RangeTbl *rangeTbl, int nsFlg){
  int i;

  printf("----------- RangeTbl\n");
  for(i=0; i<rangeTbl->cnt; i++){
    printf("rec[%d] = %s  ",i,(*(rangeTbl->range+i))->rec);
    if( nsFlg==1 ) printf("  val(d)=%g(%d)-%g(%d)\n",
      (*(rangeTbl->range+i))->st.v.d, (*(rangeTbl->range+i))->stEq,
      (*(rangeTbl->range+i))->ed.v.d, (*(rangeTbl->range+i))->edEq);
    else           printf("  val(s)=%s(%d)-%s(%d)\n",
      (*(rangeTbl->range+i))->st.v.s, (*(rangeTbl->range+i))->stEq,
      (*(rangeTbl->range+i))->ed.v.s, (*(rangeTbl->range+i))->edEq);
  }
}

/**
 * # FUNCTION #
 * frkMのデータ全てを範囲テーブル表構造体(RangeTbl)にセットする
 */
static struct RangeTbl *setRangeTbl(
  struct mssFldRecKey *frkM,
  struct mssFields *flds, /*-f オプション項目*/
  int stFldNo,
  int edFldNo,
  int nsFlg,
  int stEQflg,
  int edEQflg){

  struct RangeTbl *rangeTbl=NULL;
  struct Range **range=NULL;
  struct Range *r;

  int mstCnt;
  int size;
  char *str;
  char *fld;
  char *rngS;
  char *rngE;
  int i;

  /*----------------------------------*/
  /*出力文字列と範囲インデックスの作成*/
  /*----------------------------------*/

  mssSeekTopFRK(frkM);
  mstCnt=0;
  while( EOF != mssReadFldRecFRK(frkM) ){
    /* 出力文字列のセット(str) */
    if(flds!=NULL){
      size=sizeof(char); /* for '\0' */
      str=mssCalloc(size,"setMstRngIdx");
      for(i=0; i<flds->cnt; i++){
        fld=*(frkM->pnt+MssFlds2num(flds,i));
        size += sizeof(char)*(strlen(fld)+1); /*+1:デリミタ*/
        str=mssRealloc(str,size,"setMstRngIdx");
        strcat(str,MssFieldDelimStr);
        strcat(str,fld);
      }
    }else{ /* セレクト目的で利用される*/
      str=mssStrdup("dummy");
    }

    rngS = *(frkM->pnt+stFldNo);
    rngE = *(frkM->pnt+edFldNo);

    /* rngS〜rngEをrangeリストにセットする*/
    r=setRange(str,rngS,rngE,nsFlg,stEQflg,edEQflg);
    if(r!=NULL){
      range=mssRealloc(range,sizeof(struct Range *)*(mstCnt+1),"setRangeTbl");
      *(range+mstCnt)=r;
      mstCnt++;
    }
  }

  rangeTbl=mssMalloc(sizeof(struct RangeTbl),"setMstRngIdx2");
  rangeTbl->cnt=mstCnt;
  rangeTbl->range=range;

  /* for debug */
  /* showRangeTbl(rangeTbl,nsFlg);*/

  return(rangeTbl);
}

/**
 * # FUNCTION #
 * RangeNodeのリストを表示する(for debug)
 */
static void showRangeNode(struct RangeNode *node, int nsFlg){
  int i;
  int no=0;

  printf("----------- RangeIndexList\n");
  if(node==NULL) return;

  while(1){
    if(nsFlg==1) printf("node[%d] value=%g\n",no,node->v.v.d);
    else         printf("node[%d] value=%s\n",no,node->v.v.s);

    printf("LT: ");
    for(i=0; i<node->cntLT; i++) printf("%s,",(*(node->recLT+i)));
    printf("\n");

    printf("LE: ");
    for(i=0; i<node->cntLE; i++) printf("%s,",(*(node->recLE+i)));
    printf("\n");

    printf("IN: ");
    for(i=0; i<node->cntIN; i++) printf("%s,",(*(node->recIN+i)));
    printf("\n");

    printf("GE: ");
    for(i=0; i<node->cntGE; i++) printf("%s,",(*(node->recGE+i)));
    printf("\n");

    printf("GT: ");
    for(i=0; i<node->cntGT; i++) printf("%s,",(*(node->recGT+i)));
    printf("\n");

    if(node->next==NULL) break;
    no++;
    node=node->next;
  }
}
/**
 * # FUNCTION #
 * RangeNodeに開始条件を登録する
 */
static struct RangeNode *setRangeNodeST(struct RangeTbl *rangeTbl){
  struct RangeNode *rangeNode;
  struct RangeNode *topNode=NULL;
  struct RangeNode *curNode=NULL;
  struct Range *idxPnt;
  int i;


  for(i=0; i<rangeTbl->cnt; i++){
    idxPnt=(*(rangeTbl->range+i));

          if(curNode==NULL){
      curNode=mssCalloc(sizeof(struct RangeNode),"INST");
      curNode->v=idxPnt->st;
      topNode=curNode;
    }else if(mssVcmpOpe(curNode->v,OPE_NE,idxPnt->st)){
      rangeNode=mssCalloc(sizeof(struct RangeNode),"INST");
      curNode->next=rangeNode;
      curNode=rangeNode;
      curNode->v=idxPnt->st;
    }

    /* rangeTblのrec(JOINされるデータ)をRangeNodeに登録*/

    /*以上(GE)*/
    if(idxPnt->stEq){
      curNode->recGE=mssRealloc(curNode->recGE,
                                sizeof(char *)*(curNode->cntGE+1),"INST");
      *(curNode->recGE+curNode->cntGE)=idxPnt->rec;
      curNode->cntGE++;

    /*より大きい(GT)*/
    }else{
      curNode->recGT=mssRealloc(curNode->recGT,
                                sizeof(char *)*(curNode->cntGT+1),"INST");
      *(curNode->recGT+curNode->cntGT)=idxPnt->rec;
      curNode->cntGT++;
    }
  }
  return(topNode);
}

/**
 * # FUNCTION #
 * RangeNodeに終了条件を登録する
 */
static struct RangeNode *setRangeNodeED(
  struct RangeNode *node,
  struct RangeTbl *rangeTbl){

  struct RangeNode *rangeNode;
  struct RangeNode *topNode=node;
  struct RangeNode *curNode=node;
  struct RangeNode *prvNode=NULL;
  struct Range *idxPnt;
  int i;


  for(i=0; i<rangeTbl->cnt; i++){
    idxPnt=(*(rangeTbl->range+i));

Loop1:
    if(curNode==NULL){
      curNode=mssCalloc(sizeof(struct RangeNode),"INST");
      curNode->v=idxPnt->ed;
      if(prvNode!=NULL) prvNode->next=curNode;
    }else if(mssVcmpOpe(idxPnt->ed, OPE_GT, curNode->v)){
      prvNode=curNode;
      curNode=curNode->next;
      goto Loop1;
    }else if(mssVcmpOpe(idxPnt->ed, OPE_LT, curNode->v)){
      rangeNode=mssCalloc(sizeof(struct RangeNode),"INST");
      rangeNode->next=curNode;
      curNode=rangeNode;
      curNode->v=idxPnt->ed;
      if(prvNode!=NULL) prvNode->next=curNode;
    }

    if(prvNode==NULL) topNode=curNode;

    /* rangeTblのrec(JOINされるデータ)をRangeNodeに登録*/

    /*以下(LE)*/
    if(idxPnt->edEq){
      curNode->recLE=mssRealloc(curNode->recLE,
                              sizeof(char *)*(curNode->cntLE+1),"INST");
      *(curNode->recLE+curNode->cntLE)=idxPnt->rec;
      curNode->cntLE++;

    /*より小さい(LT)*/
    }else{
      curNode->recLT=mssRealloc(curNode->recLT,
                            sizeof(char *)*(curNode->cntLT+1),"INST");
      *(curNode->recLT+curNode->cntLT)=idxPnt->rec;
      curNode->cntLT++;
    }
  }
  return(topNode);
}

void mssShowRangeIndex(struct RangeIndex *rIndex){
   showRangeTbl(rIndex->tbl,rIndex->nsFlg);
   showRangeNode(rIndex->list,rIndex->nsFlg);
}

void mssFreeRangeIndex(struct RangeIndex *rIndex){
  struct RangeNode *node=rIndex->list;
  struct RangeNode *next;

  if(node==NULL) return;
  while(1){
    mssFree(node->recLT);
    mssFree(node->recLE);
    mssFree(node->recIN);
    mssFree(node->recGE);
    mssFree(node->recGT);
    next=node->next;
    mssFree(node);
    node=next;
    if(node==NULL) break;
  }
  freeRangeTbl(rIndex->tbl, rIndex->nsFlg);
  mssFree(rIndex);
}

struct RangeIndex *mssSetRangeIndexFrk(
  struct mssFldRecKey *frkM,
  struct mssFields *flds,
  int stFldNo,
  int edFldNo,
  int nsFlg,
  int stEQflg,
  int edEQflg){
  struct RangeIndex *rangeIndex;

  rangeIndex=mssCalloc(sizeof(struct RangeIndex),"setRangeIndex");

  rangeIndex->nsFlg=nsFlg;
  rangeIndex->tbl=setRangeTbl(frkM,flds,stFldNo,edFldNo,nsFlg,stEQflg,edEQflg);

  /*開始条件をrangeNodeにセット*/
  sortByFrom(rangeIndex->tbl);
  rangeIndex->list=setRangeNodeST(rangeIndex->tbl);

  /* showRangeTbl(rangeIndex->tbl,nsFlg); */
  /* showRangeNode(rangeIndex->list,nsFlg);*/

  /*終了条件をrangeNodeにセット*/
  sortByTo(rangeIndex->tbl);
  rangeIndex->list=setRangeNodeED(rangeIndex->list,rangeIndex->tbl);

  /* showRangeTbl(rangeIndex->tbl,nsFlg);*/
  /*showRangeNode(rangeIndex->list,nsFlg);*/

  return(rangeIndex);
}

/**
 * # FUNCTION #
 * LT条件のレコードをRBツリーより削除
 * GE条件のレコードをRBツリーに挿入
 */
void mssProcLTGE(struct mssRBnode *rb, struct RangeNode *node){
  int stat,i;
  MssValue key;

  /* すでにこのノードに対する処理が終っていれば何もせずreturn */
  if(node->procLTGE==1) return;

  mssVinit(&key,ADD);
  /* LT条件のレコードをRBツリーより削除*/
  for(i=0; i<node->cntLT; i++){
    key.v.a=*(node->recLT+i);
    stat=mssRBdelete( rb->left, key );
    if(stat==1){
      mssShowErrMsg("internal error1");
      mssEnd(mssErrorNoDefault);
    }
  }

  /* GE条件のレコードをRBツリーに挿入*/
  for(i=0; i<node->cntGE; i++){
    key.v.a=*(node->recGE+i);
    stat=mssRBinsert( rb->left, key, NULL );
    if(stat==1){
      mssShowErrMsg("internal error2");
      mssEnd(mssErrorNoDefault);
    }
  }
  node->procLTGE=1;
}

/**
 * # FUNCTION #
 * LE条件のレコードをRBツリーより削除
 * GT条件のレコードをRBツリーに挿入
 */
void mssProcLEGT(struct mssRBnode *rb, struct RangeNode *node){
  int stat,i;
  MssValue key;
  
  /* すでにこのノードに対する処理が終っていれば何もせずreturn */
  if(node->procLEGT==1) return;
  
  mssVinit(&key,ADD);
  
  /* LE条件のレコードをRBツリーより削除*/
  for(i=0; i<node->cntLE; i++){
    key.v.a=*(node->recLE+i);
    stat=mssRBdelete( rb->left, key );
    if(stat==1){
      mssShowErrMsg("internal error3");
      mssEnd(mssErrorNoDefault);
    }
  }
  
  /* GT条件のレコードをRBツリーに挿入*/
  for(i=0; i<node->cntGT; i++){
    key.v.a=*(node->recGT+i);
    stat=mssRBinsert( rb->left, key, NULL );
    if(stat==1){
      mssShowErrMsg("internal error4");
      mssEnd(mssErrorNoDefault);
    }
  }
  node->procLEGT=1;
}
