/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるオプション関連の関数
 * Change Log
 * 20040214 manページ出力において２行目の空白を除去
 * 20040608 -x を -XMLに、-Xを-MANに変更
 * 20040730 optINFのwild cardチェックの方法を変更
 * 20040804 mssCpyOptFld関数追加
 * 20040809 mssSetOption switchのbreakをgotoに変更
 * 20040818 optionキーワードの複数文字対応 (ex --field1)
 * 20041004 mssCpyOptFld関数,nam,newNam,fldOptの初期化バグフィックス
 * 20041004 get_optを独自で実装
 * 20041006 mssSetOptionのOINF(入力ファイル)のワイルドカードにおいて
 *          GLOB_NOMAGICが定義されていない環境では、その機能を利用しない
 *          ように変更
 * 20041006 mssSetのargStrの初期化(warningを避けるため)
 * 20041028 メモリリークの修正
 * 20041031 メモリリークの修正
 * ============================================================================
 */

#include <musashi/mssOption.h>
#include <musashi/mssHeader.h>
#include <musashi/mssBase.h>
#include <musashi/mssXml.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <float.h>
#include <limits.h>
#include <glob.h>

extern struct mssGlobalVariables mssGV;

/**
 * # FUNCTION #
 * 文字列リストstrListのcnt個の各文字列について、一つ目に出現するトークン(tok)
 * で分割し、その後の文字列リストを返す。
 *
 * ex.)
 * strList={"aaa:bbb","ccc:ddd","eee:fff"}
 *   ->
 * strList={"aaa","ccc","eee"}  返値={"bbb","ddd","fff"}
 */
static char **getNextStr(char **strList, int cnt, char tok)
{
  char *colPos;
  char **list=NULL;
  int i;

  if(strList==NULL) return(NULL);

  list=mssMalloc(sizeof(char *)*cnt,"tokCol");
  for(i=0; i<cnt; i++){
    if( *(strList+i)!=NULL ){
      colPos=strchr(*(strList+i),tok);
      if(colPos!=NULL){
        *colPos='\0';
        *(list+i)=++colPos;
      }else{
        *(list+i)=NULL;
      }
    }else{
      *(list+i)=NULL;
    }
  }
  return(list);
}

/**
 * # FUNCTION #
 * 各種オプションの構造体の領域を開放する。
 * 異るオプションの構造体をvoidポインタで与え、第一要素のタイプで判断して
 * 各オプションに応じた開放を実現している。
 */
void mssFreeOption(void *opt[])
{
  int i,j,k;

  MssOptSTR *optSTR;
  MssOptSEL *optSEL;
  MssOptSLS *optSLS;
  MssOptSRL *optSRL;
  MssOptINT *optINT;
  MssOptILS *optILS;
  MssOptRNG *optRNG;
  MssOptRAG *optRAG;
  MssOptDBL *optDBL;
  MssOptDLS *optDLS;
  MssOptFLD *optFLD;
  MssOptKEY *optKEY;
  MssOptINF *optINF;
  MssOptOTF *optOTF;

  i=0;
  while(opt[i]!=NULL){
    switch( ((MssOptKEY *)opt[i])->type ){
      case OSTR:
        optSTR=(MssOptSTR *)opt[i];
        mssFree(optSTR->str);
        break;

      case OSEL:
        optSEL=(MssOptSEL *)opt[i];
        mssFree(optSEL->str);
        break;

      case OSLS:
        optSLS=(MssOptSLS *)opt[i];
        mssFree(optSLS->str    );
        if(optSLS->strList!=NULL){
	  /* 全要素解放に変更 2004/10/28 */
	  for(j=0; j<optSLS->cnt; j++){
            mssFree(*(optSLS->strList+j));
	  }
        }
        mssFree(optSLS->strList);
        mssFree(optSLS->colList);
        break;

      case OSRL:
        optSRL=(MssOptSRL *)opt[i];
        mssFree(optSRL->str    );
        if(optSRL->from!=NULL){
          for(j=0; j<optSRL->cnt; j++){
            if(*(optSRL->from+j)!=NULL){
	      /* 全要素解放に変更 2004/10/28 */
              for(k=0; k<*(optSRL->valCnt+j); k++){
                mssFree(*(*(optSRL->from+j)+k));
	      }
              mssFree(*(optSRL->from+j));
            }
          }
        }
        if(optSRL->to!=NULL){
          for(j=0; j<optSRL->cnt; j++){
            mssFree(*(optSRL->to+j));
          }
        }
        mssFree(optSRL->valCnt); /* 追加 2004/10/31 */
        mssFree(optSRL->from);
        mssFree(optSRL->to);
        break;

      case OINT:
        optINT=(MssOptINT *)opt[i];
        mssFree(optINT->str );
        break;

      case OILS:
        optILS=(MssOptILS *)opt[i];
        mssFree(optILS->str );
        mssFree(optILS->val );
        break;

      case ORNG:
        optRNG=(MssOptRNG *)opt[i];
        mssFree(optRNG->str );
        mssFree(optRNG->from);
        mssFree(optRNG->to  );
        break;

      case ORAG:
        optRAG=(MssOptRAG *)opt[i];
        mssFree(optRAG->str );
        if(optRAG->st!=NULL){
          for(j=0; j<optRAG->cnt; j++){
            mssFree( *(optRAG->st+j));
          }
          mssFree(optRAG->st);
        }
        if(optRAG->ed!=NULL){
          for(j=0; j<optRAG->cnt; j++){
            mssFree( *(optRAG->ed+j));
          }
          mssFree(optRAG->ed);
        }
        mssFree(optRAG->stEq);
        mssFree(optRAG->edEq);
        if(optRAG->val!=NULL){
          for(j=0; j<optRAG->cnt; j++){
            mssFree( *(optRAG->val+j));
          }
          mssFree(optRAG->val);
        }
        if(optRAG->opt!=NULL){
          for(j=0; j<optRAG->cnt; j++){
            mssFree( *(optRAG->opt+j));
          }
          mssFree(optRAG->opt);
        }
        break;

      case ODBL:
        optDBL=(MssOptDBL *)opt[i];
        mssFree(optDBL->str);
        break;

      case ODLS:
        optDLS=(MssOptDLS *)opt[i];
        mssFree(optDLS->str );
        mssFree(optDLS->val );
        break;

      case OFLD:
        optFLD=(MssOptFLD *)opt[i];
        mssFree(optFLD->str);
        if(optFLD->nam!=NULL){
	  /* 全要素の領域解放に変更 2004/10/28 */
	  for(j=0; j<optFLD->cnt; j++){
            mssFree(*(optFLD->nam+j));
            mssFree(*(optFLD->newNam+j));
            mssFree(*(optFLD->fldOpt+j)); /* 追加 2004/10/31 */
	  }
        }
        mssFree(optFLD->nam   );
        mssFree(optFLD->newNam);
        mssFree(optFLD->fldOpt);
	mssFreeFields(optFLD->flds);
	mssFree(optFLD->fldNo2optNo);
        break;

      case OKEY:
        optKEY=(MssOptKEY *)opt[i];
        mssFree(optKEY->str);
        if(optKEY->nam!=NULL){
	  /* 全要素の領域解放に変更 2004/10/28 */
	  for(j=0; j<optKEY->cnt; j++){
            mssFree(*(optKEY->nam+j));
	  }
        }
        mssFree(optKEY->nam   );
	mssFreeFields(optKEY->flds);
	mssFree(optKEY->fldNo2optNo);
        break;

      case OINF:
        optINF=(MssOptINF *)opt[i];
        mssFree(optINF->str);
        if(optINF->strList!=NULL){
          for(j=0; j<optINF->cnt; j++){
            mssFree(*(optINF->strList+j));
          }
          mssFree(optINF->strList);
        }
        break;

      case OOTF:
        optOTF=(MssOptOTF *)opt[i];
        mssFree(optOTF->str);
        break;

      case OFLG:
        break;
    }
    i++;
  }
}

/**
 * # FUNCTION #
 * オプション構造体のsetフラグとmustフラグから、必須指定パラメータが
 * 指定されているかどうかチェックする。
 * 指定されていなければ、エラーメッセージを出力して終了する。
 * xmlTableの場合は、must==1 && set!=1でエラー(必須指定パラメータなし)となる。
 * plain textの場合は、xmlTableの場合の条件に加えて、
 * must==2 && set==1でもエラー(そのパラメータは必要ない)となる。
 */
static void chkMust(char *keyWord, int set, int must)
{

  if(!mssGV.txtFlg){
    if(!set && must){
      mssShowErrMsg("option -%s is mandatory",keyWord);
      exit(mssErrorNoDefault);
    }
  }else{
    if(set && must==2){
      mssShowErrMsg("option -%s is not needed with -t option",keyWord);
      exit(mssErrorNoDefault);
    }
    if(!set && must==1){
      mssShowErrMsg("option -%s is mandatory",keyWord);
      exit(mssErrorNoDefault);
    }
  }
}


/**
 * # FUNCTION #
 * keyWordをoptから検索し、opt[]->setを1にセットする。
 * opt[]の要素番号を返す。
 * keyWordが見付からなければ-1を返す。
 */
int setOptionSet(int *isFlag, char *keyWord, void *opt[], int optCnt){
  int i;

  *isFlag=0;

  if(*keyWord=='\0') return(-1);

  /*コマンドライン指定フラグ(set)とパラメータ文字のセット(str)*/
  for(i=0; i<optCnt; i++){
    if(0==strcmp(keyWord,((MssOptKEY *)opt[i])->keyWord)) break;
  }
  if(i>=optCnt){
    return(-1);
  }

  switch( ((MssOptKEY *)opt[i])->type ){
    case OSTR: ((MssOptSTR *)opt[i])->set=1; break;
    case OSEL: ((MssOptSEL *)opt[i])->set=1; break;
    case OSLS: ((MssOptSLS *)opt[i])->set=1; break;
    case OSRL: ((MssOptSRL *)opt[i])->set=1; break;
    case OINT: ((MssOptINT *)opt[i])->set=1; break;
    case OILS: ((MssOptILS *)opt[i])->set=1; break;
    case ORNG: ((MssOptRNG *)opt[i])->set=1; break;
    case ORAG: ((MssOptRAG *)opt[i])->set=1; break;
    case ODBL: ((MssOptDBL *)opt[i])->set=1; break;
    case ODLS: ((MssOptDLS *)opt[i])->set=1; break;
    case OFLD: ((MssOptFLD *)opt[i])->set=1; break;
    case OKEY: ((MssOptKEY *)opt[i])->set=1; break;
    case OINF: ((MssOptINF *)opt[i])->set=1; break;
    case OOTF: ((MssOptOTF *)opt[i])->set=1; break;
    case OFLG: ((MssOptFLG *)opt[i])->set=1; *isFlag=1; break;
  }
  return(i);
}

/**
 * # FUNCTION #
 * コマンド側で定義した複数のオプション構造体(opt[])の未定義メンバー変数
 * をコマンドライン引数を用いて設定する。
 * ただし、コマンドライン引数のうち、以下のパラメータ指定シンボルは特殊な意味
 * として解釈される。
 * -Q : 終了メッセージを表示しない。
 * -t : xmlTableでなく、plainTextを入出力データとして用いる。
 * また、通常mssHelpDoc関数を呼び出すことにより-h,-x,-Xも特殊な意味として解釈
 * される。
 */
void mssSetOption(void *opt[], int argc, char *argv[])
{
  int argPos;
  char *argStr=NULL;
  char *keyWord;
  char buf[10];
  int optNo;
  int isFlag;
  int isLongKeyWord;
  int optCnt=-1;
  int i,j,k;
  int err;
  int len;
  char **list1;
  char **list2;
  char **list3;
  char **list4;
  int cnt;
  glob_t fname;

  MssOptSTR *optSTR;
  MssOptSEL *optSEL;
  MssOptSLS *optSLS;
  MssOptSRL *optSRL;
  MssOptINT *optINT;
  MssOptILS *optILS;
  MssOptRNG *optRNG;
  MssOptRAG *optRAG;
  MssOptDBL *optDBL;
  MssOptDLS *optDLS;
  MssOptFLD *optFLD;
  MssOptKEY *optKEY;
  MssOptINF *optINF;
  MssOptOTF *optOTF;
  MssOptFLG *optFLG;

  /* getopt関数にエラーメッセージを出力させない */
  opterr=0;

  /*コマンドライン指定フラグの初期化*/
  /* オプションの数をカウントする */
  optCnt=-1;
  while( opt[++optCnt]!=NULL ){

    switch( ((MssOptKEY *)opt[optCnt])->type ){
      case OSTR: ((MssOptSTR *)opt[optCnt])->set=0; break;
      case OSEL: ((MssOptSEL *)opt[optCnt])->set=0; break;
      case OSLS: ((MssOptSLS *)opt[optCnt])->set=0; break;
      case OSRL: ((MssOptSRL *)opt[optCnt])->set=0; break;
      case OINT: ((MssOptINT *)opt[optCnt])->set=0; break;
      case OILS: ((MssOptILS *)opt[optCnt])->set=0; break;
      case ORNG: ((MssOptRNG *)opt[optCnt])->set=0; break;
      case ORAG: ((MssOptRAG *)opt[optCnt])->set=0; break;
      case ODBL: ((MssOptDBL *)opt[optCnt])->set=0; break;
      case ODLS: ((MssOptDLS *)opt[optCnt])->set=0; break;
      case OFLD: ((MssOptFLD *)opt[optCnt])->set=0; break;
      case OKEY: ((MssOptKEY *)opt[optCnt])->set=0; break;
      case OINF: ((MssOptINF *)opt[optCnt])->set=0; break;
      case OOTF: ((MssOptOTF *)opt[optCnt])->set=0; break;
      case OFLG: ((MssOptFLG *)opt[optCnt])->set=0; break;
    }
  }

  /* getoptでオプションをセットする */
  argPos=1;

  /* 最初現在走査している引数をargStrにセット */
  if(argPos<argc){
    argStr=argv[argPos];
  }

  while(argPos<argc){

    isLongKeyWord=0;

    /* キーワード(-x)でなければならない */
    if(*argStr!='-'){
      mssShowErrMsg("non-option argument: %s",argv[argPos]);
      exit(mssErrorNoDefault);
    }
    argStr++;

    /* longNameキーワードかどうかチェック */
    if(*argStr=='-'){
      isLongKeyWord=1;
      argStr++;
      keyWord=mssStrdup(argStr);
    }else{
      if(*argStr=='Q'){ /* メッセージ表示無しモード*/
        mssGV.quiet=1;
        *argStr='-';
	if(*(argStr+1)=='\0'){
          argPos++;
          if(argPos>=argc) break;
	  argStr=argv[argPos];
        }
        continue;
      }
      if(*argStr=='t'){ /* plain textモード*/
        mssGV.txtFlg=1;
        *argStr='-';
	if(*(argStr+1)=='\0'){
          argPos++;
          if(argPos>=argc) break;
	  argStr=argv[argPos];
        }
        continue;
      }
      buf[0]=*argStr; buf[1]='\0';
      keyWord=mssStrdup(buf);
    }

    optNo=setOptionSet(&isFlag, keyWord,opt,optCnt);
    if(optNo==-1){
      if(isLongKeyWord) mssShowErrMsg("unknown option: --%s",keyWord);
      else              mssShowErrMsg("unknown option: -%s" ,keyWord);
      exit(mssErrorNoDefault);
    }
//printf("keyWord=%s long=%d optNo=%d isFlag=%d\n",keyWord,isLongKeyWord,optNo,isFlag);

    if(isFlag){
      if(isLongKeyWord){
        argPos++;
        if(argPos>=argc){
          mssFree(keyWord); /* 追加 2004/10/31 */
	  break;
	}
	argStr=argv[argPos];
        mssFree(keyWord); /* 追加 2004/10/31 */
	continue;
      }else{
        *argStr='-';
	if(*(argStr+1)=='\0'){
          argPos++;
          if(argPos>=argc){
            mssFree(keyWord); /* 追加 2004/10/31 */
	    break;
	  }
	  argStr=argv[argPos];
        }
        mssFree(keyWord); /* 追加 2004/10/31 */
	continue;
      }
    }else{
      if(isLongKeyWord){
        argPos++;
        if(argPos>=argc){
          mssShowErrMsg("option --%s needs a value",keyWord);
          exit(mssErrorNoDefault);
	}
	argStr=argv[argPos];
      }else{
	argStr++;
	if(*argStr=='\0'){
          argPos++;
          if(argPos>=argc){
            mssShowErrMsg("option -%s needs a value",keyWord);
            exit(mssErrorNoDefault);
	  }
	  argStr=argv[argPos];
        }
      }
    }
    mssFree(keyWord);
//printf("argStr=%s\n",argStr);

    /*コマンドライン指定フラグ(set)とパラメータ文字のセット(str)*/
    switch( ((MssOptKEY *)opt[optNo])->type ){
      case OSTR: ((MssOptSTR *)opt[optNo])->str=mssStrdup(argStr); break;
      case OSEL: ((MssOptSEL *)opt[optNo])->str=mssStrdup(argStr); break;
      case OSLS: ((MssOptSLS *)opt[optNo])->str=mssStrdup(argStr); break;
      case OSRL: ((MssOptSRL *)opt[optNo])->str=mssStrdup(argStr); break;
      case OINT: ((MssOptINT *)opt[optNo])->str=mssStrdup(argStr); break;
      case OILS: ((MssOptILS *)opt[optNo])->str=mssStrdup(argStr); break;
      case ORNG: ((MssOptRNG *)opt[optNo])->str=mssStrdup(argStr); break;
      case ORAG: ((MssOptRAG *)opt[optNo])->str=mssStrdup(argStr); break;
      case ODBL: ((MssOptDBL *)opt[optNo])->str=mssStrdup(argStr); break;
      case ODLS: ((MssOptDLS *)opt[optNo])->str=mssStrdup(argStr); break;
      case OFLD: ((MssOptFLD *)opt[optNo])->str=mssStrdup(argStr); break;
      case OKEY: ((MssOptKEY *)opt[optNo])->str=mssStrdup(argStr); break;
      case OINF: ((MssOptINF *)opt[optNo])->str=mssStrdup(argStr); break;
      case OOTF: ((MssOptOTF *)opt[optNo])->str=mssStrdup(argStr); break;
      case OFLG:                                                   break;
    }

    argPos++;
//printf("argPos=%d, argc=%d\n",argPos,argc);
    if(argPos>=argc) break;
    argStr=argv[argPos];
  }
//exit(1);

  /* 1: mustによるチェック                                        */
  /* 2: 指定のなかったオプションについてデフォルト値をstrにセット */
  /*      MssOptINF,MssOptOTF,MssOptFLDはデフォルトなしなのでNULLをセット  */
  /*      MssOptKEYはdefが1or2をとるが、strにはNULLをセット。        */
  /*      MssOptFLGはdefをsetとしてセットする。                      */
  /* 3: 各型に応じて、strの内容を解析し、各種変数にセットする     */
  err=0;
  for(i=0; i<optCnt; i++){
    switch( ((MssOptKEY *)opt[i])->type ){
      case OSTR:
        optSTR=(MssOptSTR *)opt[i];
        chkMust(optSTR->keyWord,optSTR->set,optSTR->must);
        if(!optSTR->set && optSTR->def!=NULL) optSTR->str=mssStrdup(optSTR->def);

        /*チェック*/
        if(optSTR->set){
          len=strlen(optSTR->str);
          if(len<optSTR->minLen || len >optSTR->maxLen){
            mssShowErrMsg("length of option value is out of range : -%s %s (%d,%d)", optSTR->keyWord,optSTR->str,optSTR->minLen,optSTR->maxLen);
            exit(mssErrorNoDefault);
          }
        }
        break;

      case OSEL:
        optSEL=(MssOptSEL *)opt[i];
        chkMust(optSEL->keyWord,optSEL->set,optSEL->must);
        if(!optSEL->set && optSEL->def!=NULL) optSEL->str=mssStrdup(optSEL->def);

        /*チェック*/
        if(optSEL->set){
          cnt=0;
          list1=mssTokByChr(optSEL->selList,',',&cnt,1);
          for(j=0;j<cnt;j++){
            if(0==strcmp(*(list1+j),optSEL->str)) break;
          }
          if(j==cnt) {
            mssShowErrMsg("argument must be one of \"%s\" on option -%s",optSEL->selList,optSEL->keyWord);
            exit(mssErrorNoDefault);
          }
	  /* 全要素解放に変更 2004/10/28 */
	  for(j=0; j<cnt; j++){
            mssFree(*(list1+j));
	  }
          mssFree(list1);
        }
        break;

      case OSLS:
        optSLS=(MssOptSLS *)opt[i];
        chkMust(optSLS->keyWord,optSLS->set,optSLS->must);
        if(!optSLS->set && optSLS->def!=NULL) optSLS->str=mssStrdup(optSLS->def);

        /*初期化*/
        optSLS->strList=NULL;
        optSLS->colList=NULL;
        optSLS->cnt=0;

        /*strがNULLならリターン*/
        if( optSLS->str==NULL ) break;

        /*カンマによるトークン分割*/
        optSLS->strList=mssTokByChr(optSLS->str,',',&optSLS->cnt,1);

        /*コロンによるトークン分割*/
        optSLS->colList=getNextStr(optSLS->strList,optSLS->cnt,':');

        /*チェック*/
        if(optSLS->set){
          if(optSLS->cnt>optSLS->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optSLS->maxCnt,optSLS->keyWord,optSLS->str);
            exit(mssErrorNoDefault);
          } 
          for(j=0; j<optSLS->cnt; j++){
            if(*(optSLS->strList+j)==NULL || **(optSLS->strList+j)=='\0')
              len=0;
            else
              len=strlen(*(optSLS->strList+j));
            if(len<optSLS->minLen || len >optSLS->maxLen){
              mssShowErrMsg("length of %dth element of the option is out of range : -%s %s (%d,%d)", j+1,optSLS->keyWord,optSLS->str,optSLS->minLen,optSLS->maxLen);
              exit(mssErrorNoDefault);
            }
          }
          if(!optSLS->colFlg){
            for(j=0; j<optSLS->cnt; j++){
              if(*(optSLS->colList+j)!=NULL){
              mssShowErrMsg("can not use ':' in the %dth element : -%s %s)",
                j+1,optSLS->keyWord,optSLS->str);
              exit(mssErrorNoDefault);
              }
            }
          }
        }
        break;

      case OSRL:
        optSRL=(MssOptSRL *)opt[i];
        chkMust(optSRL->keyWord,optSRL->set,optSRL->must);
        if(!optSRL->set && optSRL->def!=NULL)
          optSRL->str=mssStrdup(optSRL->def);

        /*初期化*/
        optSRL->from  =NULL;
        optSRL->to    =NULL;
        optSRL->valCnt=NULL;
        optSRL->cnt   =0;

        /*strがNULLならリターン*/
        if( optSRL->str==NULL ) break;

        /*カンマによるトークン分割*/
        list1=mssTokByChr(optSRL->str,',',&optSRL->cnt,1);
        optSRL->valCnt=mssMalloc(sizeof(int)*optSRL->cnt    ,"setOpt");
        optSRL->from  =mssMalloc(sizeof(char **)*optSRL->cnt,"setOpt");
        optSRL->to    =mssMalloc(sizeof(char **)*optSRL->cnt,"setOpt");

        for(j=0; j<optSRL->cnt; j++){
          /*コロンによるトークン分割*/
          *(optSRL->from+j)=mssTokByChr( *(list1+j),':',optSRL->valCnt+j,1);
          *(optSRL->to  +j)=getNextStr(*(optSRL->from+j),*(optSRL->valCnt+j),'~');
        }

        /*チェック*/
        if(optSRL->set){
          if(optSRL->cnt>optSRL->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optSRL->maxCnt,optSRL->keyWord,optSRL->str);
            exit(mssErrorNoDefault);
          } 
          for(j=0; j<optSRL->cnt; j++){
            if(*(optSRL->valCnt+j)>optSRL->maxValCnt){
              mssShowErrMsg("number of value in %dth element (%s) must be less than %d", j+1,optSRL->keyWord,optSRL->maxValCnt);
              exit(mssErrorNoDefault);
            }
            for(k=0; k<*(optSRL->valCnt+j); k++){
              if( mssIsValidStr(*(*(optSRL->from+j)+k)) ){
                len=strlen(*(*(optSRL->from+j)+k));
              }else{
                len=0;
              }
              if(len<optSRL->minValLen || len >optSRL->maxValLen){
                mssShowErrMsg("length of %dth value in %dth element of the option is out of range : -%s %s (%d,%d)", k+1,j+1,optSRL->keyWord,optSRL->str,optSRL->minValLen,optSRL->maxValLen);
                exit(mssErrorNoDefault);
              }
            }
            if(!optSRL->rngFlg){
              for(k=0; k<*(optSRL->valCnt+j); k++){
                if(*(optSRL->from+k) != NULL){
                  mssShowErrMsg("can not use range (~) in th option -%s)",
                    j+1,optSRL->keyWord);
                  exit(mssErrorNoDefault);
                }
              }
            }
          }
        }
	/* 全要素解放に変更 2004/10/28 */
	for(j=0; j<optSRL->cnt; j++){
          mssFree(*(list1+j));
	}
        mssFree(list1);
        break;

      case OINT:
        optINT=(MssOptINT *)opt[i];
        chkMust(optINT->keyWord,optINT->set,optINT->must);
        if(!optINT->set){
          optINT->str=NULL;
          optINT->val=optINT->def;
        }else{
          optINT->val=atoi(optINT->str);
        }

        /*チェック*/
        if(optINT->min>optINT->val || optINT->max<optINT->val){
          mssShowErrMsg("the value of option -%s is out of range : %d(%d,%d)",
            optINT->keyWord,optINT->val,optINT->min,optINT->max);
          exit(mssErrorNoDefault);
        }
        break;

      case OILS:
        optILS=(MssOptILS *)opt[i];
        chkMust(optILS->keyWord,optILS->set,optILS->must);
        if(!optILS->set && optILS->def!=NULL) optILS->str=mssStrdup(optILS->def);

        /*初期化*/
        optILS->val=NULL;
        optILS->cnt=0;

        /*strがNULLならリターン*/
        if( optILS->str==NULL ) break;

        /*カンマによるトークン分割*/
        list1=mssTokByChr(optILS->str,',',&optILS->cnt,1);

        /*list1の文字列を数値リストに変換*/
        optILS->val=mssMalloc(sizeof(int)*optILS->cnt,"option");
        for(j=0; j<optILS->cnt; j++){
                if(0==strcmp(*(list1+j),"MIN")){
            *(optILS->val+j)=optILS->min;
          }else if(0==strcmp(*(list1+j),"MAX")){
            *(optILS->val+j)=optILS->max;
          }else{
            *(optILS->val+j)=atoi(*(list1+j));
          }
        }

        /*チェック*/
        if(optILS->set){
          if(optILS->cnt>optILS->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optILS->maxCnt,optILS->keyWord,optILS->str);
            exit(mssErrorNoDefault);
          }
          for(j=0; j<optILS->cnt; j++){
            if(*(list1+j)==NULL || **(list1+j)=='\0'){
              mssShowErrMsg("%dth element is empty : -%s %s",
                j+1,optILS->keyWord,optILS->str);
              exit(mssErrorNoDefault);
            }
          }
          for(j=0; j<optILS->cnt; j++){
            if(*(optILS->val+j)<optILS->min || *(optILS->val+j)>optILS->max ){
              mssShowErrMsg("%dth element value is out of range : -%s %s (%d,%d)", j+1,optILS->keyWord,optILS->str,optILS->min,optILS->max);
              exit(mssErrorNoDefault);
            }
          }
        }
	/* 全要素解放に変更 2004/10/28 */
	for(j=0; j<optILS->cnt; j++){
          mssFree(*(list1+j));
	}
        mssFree(list1);
        break;

      case ORNG:
        optRNG=(MssOptRNG *)opt[i];
        chkMust(optRNG->keyWord,optRNG->set,optRNG->must);
        if(!optRNG->set && optRNG->def!=NULL) optRNG->str=mssStrdup(optRNG->def);

        /*初期化*/
        optRNG->from=NULL;
        optRNG->to  =NULL;
        optRNG->cnt =0;

        /*strがNULLならリターン*/
        if( optRNG->str==NULL ) break;

        /*カンマによるトークン分割*/
        list1=mssTokByChr(optRNG->str,',',&optRNG->cnt,1);

        /*アンダーバーによるトークン分割*/
        list2=getNextStr(list1,optRNG->cnt,'_');

        /*list1,list2の文字列を数値範囲に変換*/
        optRNG->from=mssMalloc(sizeof(int)*optRNG->cnt,"option");
        optRNG->to  =mssMalloc(sizeof(int)*optRNG->cnt,"option");
        for(j=0; j<optRNG->cnt; j++){
                if(0==strcmp(*(list1+j),"MIN")){
            *(optRNG->from+j)=optRNG->min;
          }else if(0==strcmp(*(list1+j),"MAX")){
            *(optRNG->from+j)=optRNG->max;
          }else{
            *(optRNG->from+j)=atoi(*(list1+j));
          }
          if(*(list2+j) == NULL){
            *(optRNG->to+j)=*(optRNG->from+j);
          }else if(0==strcmp(*(list2+j),"MIN")){
            *(optRNG->to+j)=optRNG->min;
          }else if(0==strcmp(*(list2+j),"MAX")){
            *(optRNG->to+j)=optRNG->max;
          }else{
            *(optRNG->to+j)=atoi(*(list2+j));
          }
        }

        /*チェック*/
        if(optRNG->set){
          if(optRNG->cnt>optRNG->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optRNG->maxCnt,optRNG->keyWord,optRNG->str);
            exit(mssErrorNoDefault);
          }
          for(j=0; j<optRNG->cnt; j++){
            if(*(list1+j)==NULL || **(list1+j)=='\0'){
              mssShowErrMsg("%dth element is empty : -%s %s",
                j+1,optRNG->keyWord,optRNG->str);
              exit(mssErrorNoDefault);
            }
          }
          for(j=0; j<optRNG->cnt; j++){
            if(*(optRNG->from+j)<optRNG->min || *(optRNG->to+j)>optRNG->max ||
               *(optRNG->from+j)>optRNG->max || *(optRNG->to+j)<optRNG->min ){
              mssShowErrMsg("%dth element value is out of range : -%s %s (%d,%d)", j+1,optRNG->keyWord,optRNG->str,optRNG->min,optRNG->max);
              exit(mssErrorNoDefault);
            }
          }
        }

	/* 全要素解放に変更 2004/10/28 */
	for(j=0; j<optRNG->cnt; j++){
          mssFree(*(list1+j));
	}
        mssFree(list1);
        mssFree(list2);
        break;

      /* -R [12_13):abc%nm st=12 ed=13 stEq=1 edEq=0 val=abc  opt=nm    */
      /* -R 12_13:abc%n    st=12 ed=13 stEq=1 edEq=1 val=abc  opt=n     */
      /* -R 12:abc%n       st=12 ed=12 stEq=1 edEq=1 val=abc  opt=n     */
      /* -R 12             st=12 ed=12 stEq=1 edEq=1 val=NULL opt=NULL  */
      /* -R 12_            st=12 ed=*  stEq=1 edEq=1 val=NULL opt=NULL  */
      /* -R _12            st=*  ed=12 stEq=1 edEq=1 val=NULL opt=NULL  */
      case ORAG:
        optRAG=(MssOptRAG *)opt[i];
        chkMust(optRAG->keyWord,optRAG->set,optRAG->must);
        if(!optRAG->set && optRAG->def!=NULL)
          optRAG->str=mssStrdup(optRAG->def);

        /*初期化*/
        optRAG->st    = NULL;
        optRAG->ed    = NULL;
        optRAG->stEq  = NULL;
        optRAG->edEq  = NULL;
        optRAG->val   = NULL;
        optRAG->opt   = NULL;
        optRAG->cnt   = 0;

        /*strがNULLならリターン*/
        if( optRAG->str==NULL ) break;

        /*カンマによるトークン分割*/
        list1=mssTokByChr(optRAG->str,',',&optRAG->cnt,1);
        optRAG->st   = mssCalloc(sizeof(char *)*optRAG->cnt,"setOpt");
        optRAG->ed   = mssCalloc(sizeof(char *)*optRAG->cnt,"setOpt");
        optRAG->stEq = mssCalloc(sizeof(int   )*optRAG->cnt,"setOpt");
        optRAG->edEq = mssCalloc(sizeof(int   )*optRAG->cnt,"setOpt");
        optRAG->val  = mssCalloc(sizeof(char *)*optRAG->cnt,"setOpt");
        optRAG->opt  = mssCalloc(sizeof(char *)*optRAG->cnt,"setOpt");

        /*カンマによるトークン分割 (st)*/
        /* list1=mssTokByChr(optRAG->str,',',&optRAG->cnt,1); 削除 2004/10/28 */

        /*%によるトークン分割 (opt)*/
        list2=getNextStr(list1,optRAG->cnt,'%');

        /*:によるトークン分割 (val)*/
        list3=getNextStr(list1,optRAG->cnt,':');

        /*_によるトークン分割(ed)*/
        list4=getNextStr(list1,optRAG->cnt,'_');

        for(j=0; j<optRAG->cnt; j++){
          /* st */
          if(*(list1+j)!=NULL){
            if(**(list1+j)=='['){       /*以上*/
              *(optRAG->stEq+j)=1;
              *(optRAG->st+j)=mssStrdup(*(list1+j)+1);
            }else if(**(list1+j)=='('){ /*より大きい*/
              *(optRAG->stEq+j)=0;
              *(optRAG->st+j)=mssStrdup(*(list1+j)+1);
            }else if(**(list1+j)=='\0'){ /*_12*/
              *(optRAG->stEq+j)=1;
              *(optRAG->st+j)=mssStrdup("*");
            }else{                      /*デフォルト(以上)*/
              *(optRAG->stEq+j)=1;
              *(optRAG->st+j)=mssStrdup(*(list1+j));
            }
          }else{
            *(optRAG->stEq+j)=1;
            *(optRAG->st+j)=NULL;
          }

          /* ed */
          if(*(list4+j)!=NULL){
            len=strlen(*(list4+j));
            if(*(*(list4+j)+len-1)==']'){  /*以下*/
              *(*(list4+j)+len-1)='\0';
              *(optRAG->edEq+j)=1;
              *(optRAG->ed+j)=mssStrdup(*(list4+j));
            }else if(*(*(list4+j)+len-1)==')'){ /*より小さい*/
              *(*(list4+j)+len-1)='\0';
              *(optRAG->edEq+j)=0;
              *(optRAG->ed+j)=mssStrdup(*(list4+j));
            }else if(*(*(list4+j)+len-1)=='\0'){ /* 12_ */
              *(optRAG->edEq+j)=1;
              *(optRAG->ed+j)=mssStrdup("*");
            }else{                      /*デフォルト(以下)*/
              *(optRAG->edEq+j)=1;
              *(optRAG->ed+j)=mssStrdup(*(list4+j));
            }
          }else{
            *(optRAG->edEq+j)=1;
            *(optRAG->ed+j)=mssStrdup(*(optRAG->ed+j));
          }

          /* val */
          if(*(list3+j)!=NULL){
            *(optRAG->val+j)=mssStrdup(*(list3+j));
          }

          /* opt */
          if(*(list2+j)!=NULL){
            *(optRAG->opt+j)=mssStrdup(*(list2+j));
          }
        }

        /*チェック*/
        if(optRAG->set){
          if(optRAG->cnt>optRAG->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optRAG->maxCnt,optRAG->keyWord,optRAG->str);
            exit(mssErrorNoDefault);
          } 
        }
        if(optRAG->colFlg==0){
          for(j=0; j<optRAG->cnt; j++){
            if(*(optRAG->val+j)!=NULL){
              mssShowErrMsg("can not use colon in the option: -%s %s", optRAG->keyWord,optRAG->str);
            exit(mssErrorNoDefault);
            }
          }
        }
        if(optRAG->optFlg==0){
          for(j=0; j<optRAG->cnt; j++){
            if(*(optRAG->opt+j)!=NULL){
              mssShowErrMsg("can not use option(%%) in the option: -%s %s", optRAG->keyWord,optRAG->str);
            exit(mssErrorNoDefault);
            }
          }
        }
	/* 全要素解放に変更 2004/10/28 */
	for(j=0; j<optRAG->cnt; j++){
          mssFree(*(list1+j));
	}
        mssFree(list1);
        mssFree(*list2);
        mssFree(list2);
        mssFree(*list3);
        mssFree(list3);
        mssFree(*list4);
        mssFree(list4);
        break;

      case ODBL:
        optDBL=(MssOptDBL *)opt[i];
        chkMust(optDBL->keyWord,optDBL->set,optDBL->must);
        if(!optDBL->set){
          optDBL->str=NULL;
          optDBL->val=optDBL->def;
        }else{
          optDBL->val=atof(optDBL->str);
        }

        /*チェック*/
        if(optDBL->min>optDBL->val || optDBL->max<optDBL->val){
          mssShowErrMsg("the value of option -%s is out of range : %g(%g,%g)",
            optDBL->keyWord,optDBL->val,optDBL->min,optDBL->max);
          exit(mssErrorNoDefault);
        }
        break;

      case ODLS:
        optDLS=(MssOptDLS *)opt[i];
        chkMust(optDLS->keyWord,optDLS->set,optDLS->must);
        if(!optDLS->set && optDLS->def!=NULL) optDLS->str=mssStrdup(optDLS->def);

        /*初期化*/
        optDLS->val=NULL;
        optDLS->cnt=0;

        /*strがNULLならリターン*/
        if( optDLS->str==NULL ) break;

        /*カンマによるトークン分割*/
        list1=mssTokByChr(optDLS->str,',',&optDLS->cnt,1);

        /*list1の文字列を数値リストに変換*/
        optDLS->val=mssMalloc(sizeof(double)*optDLS->cnt,"option");
        for(j=0; j<optDLS->cnt; j++){
                if(0==strcmp(*(list1+j),"MIN")){
            *(optDLS->val+j)=optDLS->min;
          }else if(0==strcmp(*(list1+j),"MAX")){
            *(optDLS->val+j)=optDLS->max;
          }else{
            *(optDLS->val+j)=atof(*(list1+j));
          }
        }

        /*チェック*/
        if(optDLS->set){
          if(optDLS->cnt>optDLS->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optDLS->maxCnt,optDLS->keyWord,optDLS->str);
             exit(mssErrorNoDefault);
          }
          for(j=0; j<optDLS->cnt; j++){
            if(*(list1+j)==NULL || **(list1+j)=='\0'){
              mssShowErrMsg("%dth element is empty : -%s %s",
                j+1,optDLS->keyWord,optDLS->str);
              exit(mssErrorNoDefault);
            }
          }
          for(j=0; j<optDLS->cnt; j++){
            if(*(optDLS->val+j)<optDLS->min || *(optDLS->val+j)>optDLS->max ){
              mssShowErrMsg("%dth element value is out of range : -%s %s (%g,%g)", j+1,optDLS->keyWord,optDLS->str,optDLS->min,optDLS->max);
              exit(mssErrorNoDefault);
            }
          }
        }
	/* 全要素解放に変更 2004/10/28 */
	for(j=0; j<optDLS->cnt; j++){
          mssFree(*(list1+j));
	}
        mssFree(list1);
        break;

      case OFLD:
        optFLD=(MssOptFLD *)opt[i];
        chkMust(optFLD->keyWord,optFLD->set,optFLD->must);
        if(!optFLD->set) optFLD->str=NULL;

        /*初期化*/
        optFLD=(MssOptFLD *)opt[i];
        optFLD->nam=NULL;
        optFLD->newNam=NULL;
        optFLD->fldOpt=NULL;
        optFLD->cnt=0;
        optFLD->flds=NULL;
        optFLD->fldNo2optNo=NULL;

        /*strがNULLならリターン*/
        if( optFLD->str==NULL ) break;

        /*カンマによるトークン分割*/
        optFLD->nam=mssTokByChr(optFLD->str,',',&optFLD->cnt,1);
        /*:によるトークン分割*/
        optFLD->newNam=getNextStr(optFLD->nam,optFLD->cnt,':');

        /*%によるトークン分割*/
        optFLD->fldOpt=getNextStr(optFLD->nam,optFLD->cnt,'%');

        /*チェック*/
        /*maxCntを超えていればエラー*/
        if(optFLD->set){
          if(optFLD->cnt>optFLD->maxCnt){
            mssShowErrMsg("the max number of fields in the option is %d: -%s %s", optFLD->maxCnt,optFLD->keyWord,optFLD->str);
            exit(mssErrorNoDefault);
          }
        }
        /*新項目名が指定できないのに指定していればエラー*/
        if(!optFLD->newFlg){
          for(j=0; j<optFLD->cnt; j++){
            if(*(optFLD->newNam+j)!=NULL){
              mssShowErrMsg("can't use new field name : -%s %s",optFLD->keyWord,optFLD->str);
              exit(mssErrorNoDefault);
            }
          }
        }
        break;

      case OKEY:
        optKEY=(MssOptKEY *)opt[i];
        chkMust(optKEY->keyWord,optKEY->set,optKEY->must);
        if(!optKEY->set) optKEY->str=NULL;

        /*初期化*/
        optKEY=(MssOptKEY *)opt[i];
        optKEY->nam=NULL;
        optKEY->diffSame=0;
        optKEY->cnt=0;
        optKEY->flds=NULL;
        optKEY->fldNo2optNo=NULL;

        /*strがNULLならdiffSameをdefからセットしてリターン*/
        if( optKEY->str==NULL ){
          optKEY->diffSame=optKEY->def;
          break;

        /*strが#diff#,#same#ならそれぞれ1,2をセット */
        /* set=0,str=NULL すなわち-kで指定しなかったこととする */
        }else{
          if( 0==strcmp(optKEY->str,"#diff#") ){
            optKEY->diffSame=1;
            optKEY->set=0;
            mssFree(optKEY->str);
            optKEY->str=NULL;
            break;
          }
          if( 0==strcmp(optKEY->str,"#same#") ){
            optKEY->diffSame=2;
            optKEY->set=0;
            mssFree(optKEY->str);
            optKEY->str=NULL;
            break;
          }
        }

        /*カンマによるトークン分割*/
        optKEY->nam=mssTokByChr(optKEY->str,',',&optKEY->cnt,1);

        /*チェック*/
        if(optKEY->set){
          if(optKEY->cnt>optKEY->maxCnt){
            mssShowErrMsg("the max number of fields in the option is %d: -%s %s", optKEY->maxCnt,optKEY->keyWord,optKEY->str);
            exit(mssErrorNoDefault);
          }
        }
        break;

      case OINF:
        optINF=(MssOptINF *)opt[i];
        chkMust(optINF->keyWord,optINF->set,optINF->must);
        if(!optINF->set) optINF->str=NULL;

        /*初期化*/
        optINF->strList=NULL;
        optINF->cnt=0;

        /*strがNULLならリターン*/
        if(optINF->str==NULL) break;

        /*カンマによるトークン分割*/
        optINF->strList=mssTokByChr(optINF->str,',',&optINF->cnt,1);

        /*チェック*/
        if(optINF->set){
          if(optINF->cnt>optINF->maxCnt){
            mssShowErrMsg("the max number of elements in the option is %d: -%s %s", optINF->maxCnt,optINF->keyWord,optINF->str);
            exit(mssErrorNoDefault);
          }
        }

        /*ワイルドカードの展開*/
        fname.gl_pathc=0;
        for(j=0; j<optINF->cnt; j++){
          if(fname.gl_pathc==0){
#ifdef GLOB_NOMAGIC
            glob(*(optINF->strList+j),GLOB_NOMAGIC, NULL, &fname);
#else
            glob(*(optINF->strList+j),0, NULL, &fname);
#endif
          }else{
#ifdef GLOB_NOMAGIC
            glob(*(optINF->strList+j),GLOB_APPEND | GLOB_NOMAGIC, NULL, &fname);
#else
            glob(*(optINF->strList+j),GLOB_APPEND, NULL, &fname);
#endif
          }
        }

	/* 全領域解放に変更 2004/10/28 */
	for(j=0; j<optINF->cnt; j++){
          mssFree(*(optINF->strList+j));
	}
	mssFree(optINF->strList);
        optINF->cnt=fname.gl_pathc;
        if(optINF->cnt!=0){
          optINF->strList=mssMalloc(sizeof(char *)*optINF->cnt,"option");
          for(j=0; j<optINF->cnt; j++){
            *(optINF->strList+j)=mssStrdup(*(fname.gl_pathv+j));
          }
        }else{
          optINF->strList=NULL;
          if(!optINF->fnf) {
            mssShowErrMsg("file not found : -%s %s",optINF->keyWord,optINF->str);
            exit(mssErrorNoDefault);
          }
        }
        globfree(&fname);
        break;

      case OOTF:
        optOTF=(MssOptOTF *)opt[i];
        chkMust(optOTF->keyWord,optOTF->set,optOTF->must);
        if(!optOTF->set) optOTF->str=NULL;
        break;

      case OFLG:
        optFLG=(MssOptFLG *)opt[i];
        if(!optFLG->set) optFLG->set=optFLG->def;
        break;
    }
  }
}

/**
 * # FUNCTION #
 * オプション構造体の内容を表示する(debug用)。
 */
void mssShowOption(void *opt[])
{
  int i,j,k;
  MssOptSTR *optSTR;
  MssOptSEL *optSEL;
  MssOptSLS *optSLS;
  MssOptSRL *optSRL;
  MssOptINT *optINT;
  MssOptILS *optILS;
  MssOptRNG *optRNG;
  MssOptRAG *optRAG;
  MssOptDBL *optDBL;
  MssOptDLS *optDLS;
  MssOptFLD *optFLD;
  MssOptKEY *optKEY;
  MssOptINF *optINF;
  MssOptOTF *optOTF;
  MssOptFLG *optFLG;

  i=0;
  while(opt[i]!=NULL){
    fprintf(stderr,"option=%s ",((MssOptSLS *)opt[i])->keyWord);

    switch( ((MssOptKEY *)opt[i])->type ){
      case OSTR:
        optSTR=(MssOptSTR *)opt[i];
        fprintf(stderr,"(STR) ");
        /* ####### set */
        if( optSTR->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optSTR->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optSTR->str);
        break;

      case OSEL:
        optSEL=(MssOptSEL *)opt[i];
        fprintf(stderr,"(SEL) ");
        /* ####### set */
        if( optSEL->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optSEL->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optSEL->str);
        break;

      case OSLS:
        optSLS=(MssOptSLS *)opt[i];
        fprintf(stderr,"(SLS) ");
        /* ####### set */
        if( optSLS->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optSLS->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optSLS->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optSLS->cnt);

        /* ####### strList,colList */
        fprintf(stderr,"  strList,colList:\n");
        for(j=0; j<optSLS->cnt; j++){
          fprintf(stderr,"    [%d]=\"%s\",\"%s\"\n",j,
            *(optSLS->strList+j),
            *(optSLS->colList+j));
        }
        break;

      case OSRL:
        optSRL=(MssOptSRL *)opt[i];
        fprintf(stderr,"(SRL) ");
        /* ####### set */
        if( optSRL->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optSRL->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optSRL->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optSRL->cnt);

        /* ####### strList,colList */
        fprintf(stderr,"  strList,colList:\n");
        for(j=0; j<optSRL->cnt; j++){
          for(k=0; k<*(optSRL->valCnt+j); k++){
            fprintf(stderr,"    [%d,%d]=\"%s\"~\"%s\"\n",j,k,
            *(*(optSRL->from+j)+k),
            *(*(optSRL->to  +j)+k));
          }
        }
        break;

      case OINT:
        optINT=(MssOptINT *)opt[i];
        fprintf(stderr,"(INT) ");
        /* ####### set */
        if( optINT->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### val */
        fprintf(stderr,"  val=%d\n",optINT->val );
        break;

      case OILS:
        optILS=(MssOptILS *)opt[i];
        fprintf(stderr,"(ILS) ");
        /* ####### set */
        if( optILS->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optILS->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optILS->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optILS->cnt);

        /* ####### val */
        fprintf(stderr,"  val:\n");
        for(j=0; j<optILS->cnt; j++){
          fprintf(stderr,"    [%d]=%d\n",j,*(optILS->val+j));
        }
        break;

      case ORNG:
        optRNG=(MssOptRNG *)opt[i];
        fprintf(stderr,"(RNG) ");
        /* ####### set */
        if( optRNG->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optRNG->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optRNG->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optRNG->cnt);

        /* ####### from,to */
        fprintf(stderr,"  from,to:\n");
        for(j=0; j<optRNG->cnt; j++){
          fprintf(stderr,"    [%d]=%d,%d\n",j,
            *(optRNG->from+j),
            *(optRNG->to  +j));
        }
        break;

      case ORAG:
        optRAG=(MssOptRAG *)opt[i];
        fprintf(stderr,"(RAG) ");
        /* ####### set */
        if( optRAG->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optRAG->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optRAG->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optRAG->cnt);

        /* ####### strList,colList */
        fprintf(stderr,"  strList,colList:\n");
        for(j=0; j<optRAG->cnt; j++){
          fprintf(stderr,"    ele[%d] st=%s ed=%s stEq=%d edEq=%d val=%s opt=%s\n",j,
            *(optRAG->st+j),
            *(optRAG->ed+j),
            *(optRAG->stEq+j),
            *(optRAG->edEq+j),
            *(optRAG->val+j),
            *(optRAG->opt+j));
        }
        break;


      case ODBL:
        optDBL=(MssOptDBL *)opt[i];
        fprintf(stderr,"(DBL) ");
        /* ####### set */
        if( optDBL->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### val */
        fprintf(stderr,"  val=%g\n",optDBL->val );
        break;

      case ODLS:
        optDLS=(MssOptDLS *)opt[i];
        fprintf(stderr,"(DLS) ");
        /* ####### set */
        if( optDLS->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optDLS->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optDLS->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optDLS->cnt);

        /* ####### val */
        fprintf(stderr,"  val:\n");
        for(j=0; j<optDLS->cnt; j++){
          fprintf(stderr,"    [%d]=%g\n",j,*(optDLS->val+j));
        }
        break;

      case OFLD:
        optFLD=(MssOptFLD *)opt[i];
        fprintf(stderr,"(FLD) ");
        /* ####### set */
        if( optFLD->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optFLD->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optFLD->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optFLD->cnt);

        /* ####### nam,fldOpt,newNam */
        fprintf(stderr,"  nam,fldOpt,newNam:\n");
        for(j=0; j<optFLD->cnt; j++){
          fprintf(stderr,"    [%d]=%s,%s,%s\n",j, *(optFLD->nam   +j),
                                                  *(optFLD->fldOpt+j),
                                                  *(optFLD->newNam+j));
        }

        break;

      case OKEY:
        optKEY=(MssOptKEY *)opt[i];
        fprintf(stderr,"(KEY) ");
        /* ####### set */
        if( optKEY->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optKEY->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optKEY->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optKEY->cnt);

        /* ####### nam,fldOpt,newNam */
        for(j=0; j<optKEY->cnt; j++){
          fprintf(stderr,"  nam[%d]=%s\n",j, *(optKEY->nam   +j));
        }

        break;

      case OINF:
        optINF=(MssOptINF *)opt[i];
        fprintf(stderr,"(INF) ");
        /* ####### set */
        if( optINF->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optINF->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optINF->str);

        /* ####### cnt */
        fprintf(stderr,"  cnt=%d\n",optINF->cnt);

        /* ####### strList */
        fprintf(stderr,"  strList:\n");
        for(j=0; j<optINF->cnt; j++){
          fprintf(stderr,"    [%d]=%s\n",j, *(optINF->strList+j));
        }

        break;

      case OOTF:
        optOTF=(MssOptOTF *)opt[i];
        fprintf(stderr,"(OTF) ");
        /* ####### set */
        if( optOTF->set ) fprintf(stderr,"--> set in command line\n");
        else              fprintf(stderr,"--> not set in command line\n");

        /* ####### str */
        if(optOTF->str==NULL) break;
        fprintf(stderr,"  str=\"%s\"\n",optOTF->str);
        break;

      case OFLG:
        optFLG=(MssOptFLG *)opt[i];
        fprintf(stderr,"(FLG) ");
        /* ####### set */
        if( optFLG->set )
          fprintf(stderr,"--> set in command line\n");
        else
          fprintf(stderr,"--> not set in command line\n");
        break;
    }

    i++;
  }
}

/**
 * # FUNCTION #
 * mssComHelp構造体の内容からコマンドヘルプを表示する。
 */
static void prnComHlp(void *opt[], struct mssComHelp *comHelp)
{
  int optCnt=-1;
  int i,j;
  char *tmp;
  char *tmp2;
  char **list1;
  int    list1Cnt;
  struct mssFPW *fpw;
  char *symbol="-";
  MssOptSTR *optSTR;
  MssOptSEL *optSEL;
  MssOptSLS *optSLS;
  MssOptSRL *optSRL;
  MssOptINT *optINT;
  MssOptILS *optILS;
  MssOptRNG *optRNG;
  MssOptRAG *optRAG;
  MssOptDBL *optDBL;
  MssOptDLS *optDLS;
  MssOptFLD *optFLD;
  MssOptKEY *optKEY;
  MssOptINF *optINF;
  MssOptOTF *optOTF;
  MssOptFLG *optFLG;

  fpw=mssOpenFPW((char *)1,0,0);

  /* オプションの数をカウントする */
  while( opt[++optCnt]!=NULL );

  /* ####### name */
  mssWriteStr("NAME:\n",fpw);
  mssWriteStr(comHelp->name,fpw);
  mssWriteStr(" ", fpw);

  /* ####### version */
  mssWriteStr("version: ",fpw);
  mssWriteStr(comHelp->version,fpw);
  mssWriteStr(" ", fpw);

  /* ####### title */
  if(mssIsValidStr(comHelp->title)){
    mssWriteStr(comHelp->title,fpw);
  }
  mssWriteRet(fpw);
  mssWriteRet(fpw);

  /* ####### USAGE: */
  mssWriteStr("USAGE:\n",fpw);
  mssWriteStr(comHelp->name,fpw);
  mssWriteStr(" ", fpw);

  /* ####### OPTION */
  for(i=0; i<optCnt; i++){
    if( strlen( ((MssOptKEY *)opt[i])->keyWord) > 1){
      symbol="--";
    }else{
      symbol="-";
    }
    switch( ((MssOptKEY *)opt[i])->type ){
      case OSTR:
        optSTR=(MssOptSTR *)opt[i];
        if(!optSTR->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSTR->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSTR->title,fpw);
        if(!optSTR->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OSEL:
        optSEL=(MssOptSEL *)opt[i];
        if(!optSEL->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSEL->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSEL->title,fpw);
        mssWriteStr("{",fpw);
        tmp=mssStrdup(optSEL->selList);
        tmp2=tmp;
        while(*tmp2!='\0'){
          if(*tmp2==',')*tmp2='|';
          tmp2++;
        }
        mssWriteStr(tmp,fpw);
        mssFree(tmp);
        mssWriteStr("}",fpw);
        if(!optSEL->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OSLS:
        optSLS=(MssOptSLS *)opt[i];
        if(!optSLS->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSLS->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSLS->title,fpw);
        if(!optSLS->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OSRL:
        optSRL=(MssOptSRL *)opt[i];
        if(!optSRL->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSRL->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSRL->title,fpw);
        if(!optSRL->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OINT:
        optINT=(MssOptINT *)opt[i];
        if(!optINT->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optINT->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optINT->title,fpw);
        if(!optINT->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OILS:
        optILS=(MssOptILS *)opt[i];
        if(!optILS->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optILS->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optILS->title,fpw);
        if(!optILS->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ORNG:
        optRNG=(MssOptRNG *)opt[i];
        if(!optRNG->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optRNG->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optRNG->title,fpw);
        if(!optRNG->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ORAG:
        optRAG=(MssOptRAG *)opt[i];
        if(!optRAG->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optRAG->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optRAG->title,fpw);
        if(!optRAG->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ODBL:
        optDBL=(MssOptDBL *)opt[i];
        if(!optDBL->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optDBL->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optDBL->title,fpw);
        if(!optDBL->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ODLS:
        optDLS=(MssOptDLS *)opt[i];
        if(!optDLS->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optDLS->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optDLS->title,fpw);
        if(!optDLS->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OFLD:
        optFLD=(MssOptFLD *)opt[i];
        if(!optFLD->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optFLD->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optFLD->title,fpw);
        if(mssIsValidStr(optFLD->fldOptSel)){
          list1Cnt=0;
          list1=mssTokByChr(optFLD->fldOptSel,',',&list1Cnt,1);
          mssWriteStr("[%{",fpw);
          for(j=0; j<list1Cnt-1; j++){
            mssWriteStr(*(list1+j),fpw);
            mssWriteStr("|",fpw);
          }
          mssWriteStr(*(list1+j),fpw);
          mssWriteStr("}]",fpw);
	  /* 全領域解放に変更 2004/10/28 */
	  for(j=0; j<list1Cnt; j++){
            mssFree(*(list1+j));
	  }
          mssFree(list1);
        } 
        if(!optFLD->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OKEY:
        optKEY=(MssOptKEY *)opt[i];
        if(!optKEY->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optKEY->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optKEY->title,fpw);
        if(!optKEY->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OINF:
        optINF=(MssOptINF *)opt[i];
        if(!optINF->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optINF->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optINF->title,fpw);
        if(!optINF->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OOTF:
        optOTF=(MssOptOTF *)opt[i];
        if(!optOTF->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optOTF->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optOTF->title,fpw);
        if(!optOTF->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OFLG:
        optFLG=(MssOptFLG *)opt[i];
        mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optFLG->keyWord,fpw);
        mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;
    }
  }
  mssWriteRet(fpw);
  mssWriteRet(fpw);

  /* ####### summary */
  if(mssIsValidStr(comHelp->summary)){
    mssWriteStr("SUMMARY:\n",fpw);
    mssWriteStr(comHelp->summary,fpw);
    mssWriteRet(fpw);
    mssWriteRet(fpw);
  }

  mssWriteStr("OPTION:\n",fpw);
  /* ####### OPTION */
  for(i=0; i<optCnt; i++){
    if( ((MssOptKEY *)opt[i])->type == OFLG ){
      if( strlen( ((MssOptKEY *)opt[i])->keyWord) > 1){
        symbol="--";
      }else{
        symbol="-";
      }
      optFLG=(MssOptFLG *)opt[i];
      mssWriteStr(symbol,fpw);
      mssWriteStr(optFLG->keyWord,fpw);
      mssWriteStr(": ",fpw);
      mssWriteStr(optFLG->title,fpw);
      mssWriteRet(fpw);
    }
  }
  mssWriteRet(fpw);

  /* ####### example */
  if(mssIsValidStr(comHelp->example)){
    mssWriteStr("EXAMPLES:\n",fpw);
    mssWriteStr(comHelp->example,fpw);
    mssWriteRet(fpw);
  }

  /* ####### homepage */
  if(mssIsValidStr(comHelp->homepage)){
    mssWriteStr("INFORMATION: <",fpw);
    mssWriteStr(comHelp->homepage,fpw);
    mssWriteStr(">\n",fpw);
  }

  /* ####### bugReport */
  if(mssIsValidStr(comHelp->bugReport)){
    mssWriteStr("BUG  REPORT: <",fpw);
    mssWriteStr(comHelp->bugReport,fpw);
    mssWriteStr(">\n",fpw);
  }

  mssWriteRet(fpw);
}

/**
 * # FUNCTION #
 * mssComHelp構造体の内容からmanドキュメント(groff)を出力する。
 */
static void prnMan(void *opt[], struct mssComHelp *comHelp)
{
  int optCnt=-1;
  int i,j;
  char *tmp;
  char *tmp2;
  char **list1;
  int    list1Cnt;
  struct mssFPW *fpw;
  char *symbol="-";
  MssOptSTR *optSTR;
  MssOptSEL *optSEL;
  MssOptSLS *optSLS;
  MssOptSRL *optSRL;
  MssOptINT *optINT;
  MssOptILS *optILS;
  MssOptRNG *optRNG;
  MssOptRAG *optRAG;
  MssOptDBL *optDBL;
  MssOptDLS *optDLS;
  MssOptFLD *optFLD;
  MssOptKEY *optKEY;
  MssOptINF *optINF;
  MssOptOTF *optOTF;
  MssOptFLG *optFLG;

  time_t  long_time;
  struct tm     *nt;
  char msg[100];


  fpw=mssOpenFPW((char *)1,0,0);

  /* オプションの数をカウントする */
  while( opt[++optCnt]!=NULL );

  /* ####### comment */
  mssWriteStr(".\\Generated by xtcmd(",fpw);
  mssWriteStr("). Modify the source.\n",fpw);

  /* ####### .TH*/
  time(&long_time);
  nt = localtime(&long_time);
  mssWriteStr(".TH ",fpw);
  mssWriteStr(comHelp->name,fpw);
  mssWriteStr(" 1 \"",fpw);
  sprintf(msg, "%04d-%02d-%02d", nt->tm_year+1900,nt->tm_mon+1,nt->tm_mday);
  mssWriteStr(msg,fpw);
  mssWriteStr("\" ",fpw);
  mssWriteStr("MUSASHI",fpw);
  mssWriteRet(fpw); mssWriteRet(fpw);

  /* ####### .SH 名前 */
  /*mssWriteStr(".SH 名前\n",fpw);*/
  mssWriteStr(".SH NAME\n",fpw);
  mssWriteStr(comHelp->name,fpw);
  mssWriteStr(" \\- ",fpw);
  if(mssIsValidStr(comHelp->title)){
    mssWriteStr(comHelp->title,fpw);
    mssWriteRet(fpw);
  }else{
    mssWriteStr("noData\n",fpw);
  }
  mssWriteRet(fpw);

  /* ####### .SH 書式: */
  /*mssWriteStr(".SH 書式\n",fpw);*/
  mssWriteStr(".SH SYNOPSIS\n",fpw);
  mssWriteStr(comHelp->name,fpw);
  mssWriteStr(" ", fpw);

  for(i=0; i<optCnt; i++){
    if( strlen( ((MssOptKEY *)opt[i])->keyWord) > 1){
      symbol="--";
    }else{
      symbol="-";
    }
    switch( ((MssOptKEY *)opt[i])->type ){
      case OSTR:
        optSTR=(MssOptSTR *)opt[i];
        if(!optSTR->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSTR->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSTR->title,fpw);
        if(!optSTR->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OSEL:
        optSEL=(MssOptSEL *)opt[i];
        if(!optSEL->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSEL->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSEL->title,fpw);
        mssWriteStr("{",fpw);
        tmp=mssStrdup(optSEL->selList);
        tmp2=tmp;
        while(*tmp2!='\0'){
          if(*tmp2==',')*tmp2='|';
          tmp2++;
        }
        mssWriteStr(tmp,fpw);
        mssFree(tmp);
        mssWriteStr("}",fpw);
        if(!optSEL->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OSLS:
        optSLS=(MssOptSLS *)opt[i];
        if(!optSLS->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSLS->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSLS->title,fpw);
        if(!optSLS->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OSRL:
        optSRL=(MssOptSRL *)opt[i];
        if(!optSRL->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optSRL->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optSRL->title,fpw);
        if(!optSRL->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OINT:
        optINT=(MssOptINT *)opt[i];
        if(!optINT->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optINT->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optINT->title,fpw);
        if(!optINT->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OILS:
        optILS=(MssOptILS *)opt[i];
        if(!optILS->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optILS->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optILS->title,fpw);
        if(!optILS->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ORNG:
        optRNG=(MssOptRNG *)opt[i];
        if(!optRNG->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optRNG->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optRNG->title,fpw);
        if(!optRNG->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ORAG:
        optRAG=(MssOptRAG *)opt[i];
        if(!optRAG->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optRAG->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optRAG->title,fpw);
        if(!optRAG->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;


      case ODBL:
        optDBL=(MssOptDBL *)opt[i];
        if(!optDBL->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optDBL->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optDBL->title,fpw);
        if(!optDBL->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case ODLS:
        optDLS=(MssOptDLS *)opt[i];
        if(!optDLS->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optDLS->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optDLS->title,fpw);
        if(!optDLS->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OFLD:
        optFLD=(MssOptFLD *)opt[i];
        if(!optFLD->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optFLD->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optFLD->title,fpw);
        if(mssIsValidStr(optFLD->fldOptSel)){
          list1Cnt=0;
          list1=mssTokByChr(optFLD->fldOptSel,',',&list1Cnt,1);
          mssWriteStr("[%{",fpw);
          for(j=0; j<list1Cnt-1; j++){
            mssWriteStr(*(list1+j),fpw);
            mssWriteStr("|",fpw);
          }
          mssWriteStr(*(list1+j),fpw);
          mssWriteStr("}]",fpw);
	  /* 全領域解放に変更 2004/10/28 */
	  for(j=0; j<list1Cnt; j++){
            mssFree(*(list1+j));
	  }
          mssFree(list1);
        } 
        if(!optFLD->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OKEY:
        optKEY=(MssOptKEY *)opt[i];
        if(!optKEY->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optKEY->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optKEY->title,fpw);
        if(!optKEY->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OINF:
        optINF=(MssOptINF *)opt[i];
        if(!optINF->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optINF->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optINF->title,fpw);
        if(!optINF->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OOTF:
        optOTF=(MssOptOTF *)opt[i];
        if(!optOTF->must ) mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optOTF->keyWord,fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(optOTF->title,fpw);
        if(!optOTF->must ) mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;

      case OFLG:
        optFLG=(MssOptFLG *)opt[i];
        mssWriteStr("[",fpw);
        mssWriteStr(symbol,fpw);
        mssWriteStr(optFLG->keyWord,fpw);
        mssWriteStr("]",fpw);
        mssWriteStr(" ", fpw);
        break;
    }
  }
  mssWriteRet(fpw);
  mssWriteRet(fpw);

  /* ####### .SH 説明 */
  if(mssIsValidStr(comHelp->summary)){
    /*mssWriteStr(".SH 説明\n",fpw);*/
    mssWriteStr(".SH DESCRIPTION\n",fpw);
    mssWriteStr(comHelp->summary,fpw);
    mssWriteRet(fpw);
    mssWriteRet(fpw);
  }


  /* ####### .SH パラメータ */
  /*mssWriteStr(".SH パラメータ\n",fpw);*/
  mssWriteStr(".SH PARAMETERS\n",fpw);
  for(i=0; i<optCnt; i++){
    if( strlen( ((MssOptKEY *)opt[i])->keyWord) > 1){
      symbol="--";
    }else{
      symbol="-";
    }
    if( ((MssOptKEY *)opt[i])->type != OFLG ){
      switch( ((MssOptKEY *)opt[i])->type ){
        case OSTR:
          optSTR=(MssOptSTR *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optSTR->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optSTR->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optSTR->comment,fpw); mssWriteRet(fpw);
          break;
        case OSEL:
          optSEL=(MssOptSEL *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optSEL->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optSEL->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optSEL->comment,fpw); mssWriteRet(fpw);
          break;
        case OSLS:
          optSLS=(MssOptSLS *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optSLS->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optSLS->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optSLS->comment,fpw); mssWriteRet(fpw);
          break;
        case OSRL:
          optSRL=(MssOptSRL *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optSRL->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optSRL->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optSRL->comment,fpw); mssWriteRet(fpw);
          break;
        case OINT:
          optINT=(MssOptINT *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optINT->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optINT->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optINT->comment,fpw); mssWriteRet(fpw);
          break;
        case OILS:
          optILS=(MssOptILS *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optILS->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optILS->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optILS->comment,fpw); mssWriteRet(fpw);
          break;
        case ORNG:
          optRNG=(MssOptRNG *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optRNG->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optRNG->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optRNG->comment,fpw); mssWriteRet(fpw);
          break;
        case ORAG:
          optRAG=(MssOptRAG *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optRAG->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optRAG->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optRAG->comment,fpw); mssWriteRet(fpw);
          break;
        case ODBL:
          optDBL=(MssOptDBL *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optDBL->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optDBL->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optDBL->comment,fpw); mssWriteRet(fpw);
          break;
        case ODLS:
          optDLS=(MssOptDLS *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optDLS->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optDLS->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optDLS->comment,fpw); mssWriteRet(fpw);
          break;
        case OFLD:
          optFLD=(MssOptFLD *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optFLD->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optFLD->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optFLD->comment,fpw); mssWriteRet(fpw);
          break;
        case OKEY:
          optKEY=(MssOptKEY *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optKEY->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optKEY->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optKEY->comment,fpw); mssWriteRet(fpw);
          break;
        case OINF:
          optINF=(MssOptINF *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optINF->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optINF->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optINF->comment,fpw); mssWriteRet(fpw);
          break;
        case OOTF:
          optOTF=(MssOptOTF *)opt[i];
          mssWriteStr(".IP ",fpw);
          mssWriteStr(symbol,fpw);
          mssWriteStr(optOTF->keyWord,fpw); mssWriteStr("\n",fpw);
          /*mssWriteStr(optOTF->title,fpw)  ; mssWriteStr("\"\n",fpw);*/
          mssWriteStr(optOTF->comment,fpw); mssWriteRet(fpw);
          break;
        case OFLG:
          break;
      }
    }
  }
  mssWriteRet(fpw);

  /* ####### .SH オプション */
  /*mssWriteStr(".SH オプション\n",fpw);*/
  mssWriteStr(".SH OPTIONS\n",fpw);
  for(i=0; i<optCnt; i++){
    if( ((MssOptKEY *)opt[i])->type == OFLG ){
      if( strlen( ((MssOptKEY *)opt[i])->keyWord) > 1){
        symbol="--";
      }else{
        symbol="-";
      }
      optFLG=(MssOptFLG *)opt[i];
      mssWriteStr(".IP ",fpw);
      mssWriteStr(symbol,fpw);
      mssWriteStr(optFLG->keyWord,fpw);
      mssWriteStr("\n",fpw);
      mssWriteStr(optFLG->title,fpw);
      mssWriteStr(": ",fpw);
      mssWriteStr(optFLG->comment,fpw);
      mssWriteRet(fpw);
    }
  }
  mssWriteRet(fpw);

  /* ####### .SH 利用例 */
  if(mssIsValidStr(comHelp->example)){
    /*mssWriteStr(".SH 利用例\n",fpw);*/
    mssWriteStr(".SH EXAMPLE\n",fpw);
    list1Cnt=0;
    list1=mssTokByChr(comHelp->example,'\n',&list1Cnt,1);
    for(j=0; j<list1Cnt; j++){
      if(mssIsValidStr(*(list1+j)) ){
        mssWriteStr(".P\n",fpw);
        mssWriteStr(" ", fpw);
        mssWriteStr(*(list1+j),fpw);
        mssWriteRet(fpw);
      }
    }
    /* 全領域解放に変更 2004/10/28 */
    for(j=0; j<list1Cnt; j++){
      mssFree(*(list1+j));
    }
    mssFree(list1);
    mssWriteRet(fpw);
  }

  /* ####### .SH 参照 */
  if(mssIsValidStr(comHelp->example)){
    /*mssWriteStr(".SH 参照\n",fpw);*/
    mssWriteStr(".SH SEE ALSO\n",fpw);
    list1Cnt=0;
    list1=mssTokByChr(comHelp->reference,',',&list1Cnt,1);
    for(j=0; j<list1Cnt; j++){
      if(mssIsValidStr(*(list1+j)) ){
        mssWriteStr(".BR ",fpw);
        mssWriteStr(*(list1+j),fpw);
        if(j==list1Cnt-1){
          mssWriteStr(" (1)\n",fpw);
        }else{
          mssWriteStr(" (1),\n",fpw);
        }
      }
    }
    /* 全領域解放に変更 2004/10/28 */
    for(j=0; j<list1Cnt; j++){
      mssFree(*(list1));
    }
    mssFree(list1);
    mssWriteRet(fpw);
  }


  /* ####### .SH webページ */
  if(mssIsValidStr(comHelp->homepage)){
    /*mssWriteStr(".SH webページ\n",fpw);*/
    mssWriteStr(".SH WEB PAGE\n",fpw);
    mssWriteStr(comHelp->homepage,fpw);
    mssWriteRet(fpw); mssWriteRet(fpw);
  }

  /* ####### .SH バグ報告 */
  if(mssIsValidStr(comHelp->bugReport)){
    /*mssWriteStr(".SH バグ報告\n",fpw);*/
    mssWriteStr(".SH BUG REPORT\n",fpw);
    mssWriteStr(comHelp->bugReport,fpw);
    mssWriteRet(fpw); mssWriteRet(fpw);
  }

  /* ####### .SH 作者 */
  if(mssIsValidStr(comHelp->example)){
    /*mssWriteStr(".SH 作者\n",fpw);*/
    mssWriteStr(".SH AUTHORS\n",fpw);
    mssWriteStr(comHelp->author,fpw);
    mssWriteRet(fpw);
  }
}

/**
 * # FUNCTION #
 * mssComHelp構造体の内容からXMLドキュメントを出力する。
 */
static void prnComXml(void *opt[], struct mssComHelp *comHelp)
{
  int optCnt=-1;
  int i,j;
  struct mssFPW *fpw;
  char **list1;
  int    list1Cnt;
  char **list2;
  int    list2Cnt;
  MssOptSTR *optSTR;
  MssOptSEL *optSEL;
  MssOptSLS *optSLS;
  MssOptSRL *optSRL;
  MssOptINT *optINT;
  MssOptILS *optILS;
  MssOptRNG *optRNG;
  MssOptRAG *optRAG;
  MssOptDBL *optDBL;
  MssOptDLS *optDLS;
  MssOptFLD *optFLD;
  MssOptKEY *optKEY;
  MssOptINF *optINF;
  MssOptOTF *optOTF;
  MssOptFLG *optFLG;

  struct mssXmlTag *commandTag;
  struct mssXmlTag *referTag;
  struct mssXmlTag *authorTag;
  struct mssXmlTag *exampleTag;
  struct mssXmlTag *optionTag;
  struct mssXmlTag *xmlTag;
  struct mssXmlTag *manTag;

  fpw=mssOpenFPW((char *)1,0,0);
  manTag=mssInitXmlTag("mandatory",NULL);

  /* オプションの数をカウントする */
  while( opt[++optCnt]!=NULL );

  /* <?xml version="1.0" encoding="EUC-JP"?> */
  mssWriteXmlDeclaration( MssXmlDefVer, MssXmlDefEnc ,fpw);

  /* <command name="xtcut" version="3.00"> */
  commandTag=mssInitXmlTag("command",NULL);
  mssAddXmlTagAttributeStr(commandTag,"name"   ,comHelp->name,NULL);
  mssAddXmlTagAttributeStr(commandTag,"version",comHelp->version,NULL);
  mssWriteXmlStartTag(commandTag,NULL,fpw);
  mssWriteRet(fpw);

  /* ####### title */
  if(mssIsValidStr(comHelp->title)){
    mssWriteXmlTagStr(1,"title",comHelp->title,1,NULL,fpw);
  }

  /* ####### summary */
  if(mssIsValidStr(comHelp->summary)){
    mssWriteXmlTagStr(1,"summary",comHelp->summary,1,NULL,fpw);
  }

  /* ####### reference */
  if(mssIsValidStr(comHelp->reference)){
    list1Cnt=0;
    list1=mssTokByChr(comHelp->reference,',',&list1Cnt,1);
    for(j=0; j<list1Cnt; j++){
      mssWriteXmlIndent(1,fpw);
      referTag=mssInitXmlTag("reference",NULL);
      mssAddXmlTagAttributeInt(referTag,"no", j+1,NULL);
      mssWriteXmlStartTag(referTag,NULL,fpw);
      mssWriteXmlContent(*(list1+j),NULL,fpw);
      mssWriteXmlEndTag(referTag,NULL,fpw);
      mssWriteRet(fpw);
      mssFreeXmlTag(referTag);
    }
    /* 全領域解放に変更 2004/10/28 */
    for(j=0; j<list1Cnt; j++){
      mssFree(*(list1+j));
    }
    mssFree(list1);
  }
 
  /* ####### author */
  if(mssIsValidStr(comHelp->author)){
    list1Cnt=0;
    list1=mssTokByChr(comHelp->author,',',&list1Cnt,1);
    for(j=0; j<list1Cnt; j++){
      mssWriteXmlIndent(1,fpw);
      authorTag=mssInitXmlTag("author",NULL);
      mssAddXmlTagAttributeInt(authorTag,"no", j+1,NULL);
      mssWriteXmlStartTag(authorTag,NULL,fpw);
      mssWriteXmlContent(*(list1+j),NULL,fpw);
      mssWriteXmlEndTag(authorTag,NULL,fpw);
      mssWriteRet(fpw);
      mssFreeXmlTag(authorTag);
    }
    /* 全領域解放に変更 2004/10/28 */
    for(j=0; j<list1Cnt; j++){
      mssFree(*(list1+j));
    }
    mssFree(list1);
  }

  /* ####### example */
  if(mssIsValidStr(comHelp->example)){
    list1Cnt=0;
    list1=mssTokByChr(comHelp->example,',',&list1Cnt,1);
    for(j=0; j<list1Cnt; j++){
      mssWriteXmlIndent(1,fpw);
      exampleTag=mssInitXmlTag("example",NULL);
      mssAddXmlTagAttributeInt(exampleTag,"no", j+1,NULL);
      mssWriteXmlStartTag(exampleTag,NULL,fpw);
      mssWriteXmlContent(*(list1+j),NULL,fpw);
      mssWriteXmlEndTag(exampleTag,NULL,fpw);
      mssWriteRet(fpw);
      mssFreeXmlTag(exampleTag);
    }
    /* 全領域解放に変更 2004/10/28 */
    for(j=0; j<list1Cnt; j++){
      mssFree(*(list1+j));
    }
    mssFree(list1);
  }

  /* ####### bugReport */
  if(mssIsValidStr(comHelp->bugReport)){
    mssWriteXmlTagStr(1,"bugReport",comHelp->bugReport,1,NULL,fpw);
  }

  /* ####### homepage */
  if(mssIsValidStr(comHelp->homepage)){
    mssWriteXmlTagStr(1,"homepage",comHelp->homepage,1,NULL,fpw);
  }

  /* ####### OPTION */
  for(i=0; i<optCnt; i++){
    mssWriteXmlIndent(1,fpw);
    optionTag=mssInitXmlTag("option",NULL);
    mssAddXmlTagAttributeStr(optionTag,"keyword", ((MssOptKEY *)opt[i])->keyWord,NULL);
    mssWriteXmlStartTag(optionTag,NULL,fpw);
    mssWriteRet(fpw);

    switch( ((MssOptKEY *)opt[i])->type ){
      case OSTR:
        optSTR=(MssOptSTR *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"STR"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optSTR->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optSTR->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"minLen"   ,optSTR->minLen ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxLen"   ,optSTR->maxLen ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optSTR->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optSTR->comment,1,NULL,fpw);
        break;

      case OSEL:
        optSEL=(MssOptSEL *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"SEL"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optSEL->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optSEL->def    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optSEL->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optSEL->comment,1,NULL,fpw);

        /* ####### selList,selListTitle */
        if(mssIsValidStr(optSEL->selList)){
          list1Cnt=0;
          list1=mssTokByChr(optSEL->selList,',',&list1Cnt,1);
          list2Cnt=0;
          list2=mssTokByChr(optSEL->selListTitle,',',&list2Cnt,1);
          for(j=0; j<list1Cnt; j++){
            mssWriteXmlIndent(2,fpw);
            xmlTag=mssInitXmlTag("selectList",NULL);
            mssAddXmlTagAttributeInt(xmlTag,"no"     , j+1,NULL);
            mssAddXmlTagAttributeStr(xmlTag,"keyword", *(list1+j),NULL);
            mssWriteXmlStartTag(xmlTag,NULL,fpw);
            if(j<list2Cnt) mssWriteXmlContent(*(list2+j),NULL,fpw);
            mssWriteXmlEndTag(xmlTag,NULL,fpw);
            mssWriteRet(fpw);
            mssFreeXmlTag(xmlTag);
          }
          /* 全領域解放に変更 2004/10/28 */
          for(j=0; j<list1Cnt; j++){
            mssFree(*(list1+j));
	  }
          mssFree(list1);
          /* 全領域解放に変更 2004/10/28 */
          for(j=0; j<list2Cnt; j++){
            mssFree(*(list2+j));
	  }
	  mssFree(list2);
        }
        break;


      case OSLS:
        optSLS=(MssOptSLS *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"SLS"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optSLS->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optSLS->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optSLS->maxCnt ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"minLen"   ,optSLS->minLen ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxLen"   ,optSLS->maxLen ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"colFlg"   ,optSLS->colFlg ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optSLS->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optSLS->comment,1,NULL,fpw);
        break;

      case OSRL:
        optSRL=(MssOptSRL *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"SRL"             ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optSRL->must      ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optSRL->def       ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optSRL->maxCnt    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxValCnt",optSRL->maxValCnt ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"minValLen",optSRL->minValLen ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxValLen",optSRL->maxValLen ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"rngFlg"   ,optSRL->rngFlg ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optSRL->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optSRL->comment,1,NULL,fpw);
        break;

      case OINT:
        optINT=(MssOptINT *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"INT"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optINT->must   ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"default"  ,optINT->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"min"      ,optINT->min    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"max"      ,optINT->max    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optINT->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optINT->comment,1,NULL,fpw);
        break;

      case OILS:
        optILS=(MssOptILS *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"ILS"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optILS->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optILS->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optILS->maxCnt ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"min"      ,optILS->min    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"max"      ,optILS->max    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optILS->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optILS->comment,1,NULL,fpw);
        break;

      case ORNG:
        optRNG=(MssOptRNG *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"RNG"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optRNG->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optRNG->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optRNG->maxCnt ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"min"      ,optRNG->min    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"max"      ,optRNG->max    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optRNG->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optRNG->comment,1,NULL,fpw);
        break;

      case ORAG:
        optRAG=(MssOptRAG *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"RAG"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optRAG->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optRAG->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optRAG->maxCnt ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optRAG->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optRAG->comment,1,NULL,fpw);
        break;

      case ODBL:
        optDBL=(MssOptDBL *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"DBL"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optDBL->must   ,1,NULL,fpw);
        mssWriteXmlTagDbl(2,"default"  ,optDBL->def    ,1,NULL,fpw);
        mssWriteXmlTagDbl(2,"min"      ,optDBL->min    ,1,NULL,fpw);
        mssWriteXmlTagDbl(2,"max"      ,optDBL->max    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optDBL->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optDBL->comment,1,NULL,fpw);
        break;

      case ODLS:
        optDLS=(MssOptDLS *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"DLS"          ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optDLS->must   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"default"  ,optDLS->def    ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optDLS->maxCnt ,1,NULL,fpw);
        mssWriteXmlTagDbl(2,"min"      ,optDLS->min    ,1,NULL,fpw);
        mssWriteXmlTagDbl(2,"max"      ,optDLS->max    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optDLS->title  ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optDLS->comment,1,NULL,fpw);
        break;

      case OFLD:
        optFLD=(MssOptFLD *)opt[i];
        mssWriteXmlTagStr(2,"type"           ,"FLD"            ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory"      ,optFLD->must     ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"         ,optFLD->maxCnt   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"inKeyword"      ,optFLD->inKeyWord,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"newFieldNameFlg",optFLD->newFlg   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"          ,optFLD->title    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"        ,optFLD->comment  ,1,NULL,fpw);

        /* ####### fldOptTitle */
        if(mssIsValidStr(optFLD->fldOptSel)){
          list1Cnt=0;
          list1=mssTokByChr(optFLD->fldOptTitle,',',&list1Cnt,1);
          list2Cnt=0;
          list2=mssTokByChr(optFLD->fldOptSel,',',&list2Cnt,1);
          if(list1Cnt>list2Cnt) list1Cnt=list2Cnt;
          for(j=0; j<list1Cnt; j++){
            mssWriteXmlIndent(2,fpw);
            xmlTag=mssInitXmlTag("fieldOption",NULL);
            mssAddXmlTagAttributeInt(xmlTag,"no"     , j+1,NULL);
            mssAddXmlTagAttributeStr(xmlTag,"keyword", *(list2+j),NULL);
            mssWriteXmlStartTag(xmlTag,NULL,fpw);
            mssWriteXmlContent(*(list1+j),NULL,fpw);
            mssWriteXmlEndTag(xmlTag,NULL,fpw);
            mssWriteRet(fpw);
            mssFreeXmlTag(xmlTag);
          }
          /* 全領域解放に変更 2004/10/28 */
          for(j=0; j<list1Cnt; j++){
            mssFree(*(list1+j));
	  }
          mssFree(list1);
          /* 全領域解放に変更 2004/10/28 */
          for(j=0; j<list2Cnt; j++){
            mssFree(*(list2+j));
	  }
	  mssFree(list2);
        }
        break;

      case OKEY:
        optKEY=(MssOptKEY *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"KEY"            ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optKEY->must     ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"default"  ,optKEY->def      ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optKEY->maxCnt   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"inKeyword",optKEY->inKeyWord,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optKEY->title    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optKEY->comment  ,1,NULL,fpw);
        break;

      case OINF:
        optINF=(MssOptINF *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"INF"            ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optINF->must     ,1,NULL,fpw);
        mssWriteXmlTagInt(2,"maxCnt"   ,optINF->maxCnt   ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"inout"    ,"in"             ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optINF->title    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optINF->comment  ,1,NULL,fpw);
        break;

      case OOTF:
        optOTF=(MssOptOTF *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"OTF"            ,1,NULL,fpw);
        mssWriteXmlTagFlg(2,"mandatory",optOTF->must     ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"inout"    ,"out"            ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optOTF->title    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optOTF->comment  ,1,NULL,fpw);
        break;

      case OFLG:
        optFLG=(MssOptFLG *)opt[i];
        mssWriteXmlTagStr(2,"type"     ,"FLG"            ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"title"    ,optFLG->title    ,1,NULL,fpw);
        mssWriteXmlTagStr(2,"comment"  ,optFLG->comment  ,1,NULL,fpw);
        break;
    }
    /* </option> */
    mssWriteXmlIndent(1,fpw);
    mssWriteXmlEndTag(optionTag,NULL,fpw);
    mssWriteRet(fpw);
    mssFreeXmlTag(optionTag);
  }
  /* </command> */
  mssWriteXmlEndTag(commandTag,NULL,fpw);
  mssFreeXmlTag(commandTag);
  mssWriteRet(fpw);
}

/**
 * # FUNCTION #
 * コマンドヘルプドキュメントを出力する。
 * 第一パラメータの記号により以下に示す３種類のドキュメントを出力する。
 *   -h : 簡易ヘルプの表示
 *   -x : XMLドキュメントによる表示
 *   -X : manドキュメント(troff)による表示
 */
void mssHelpDoc(void *opt[],struct mssComHelp *comHelp, int argc, char **argv)
{

  if( argc>1 ){
    if( 0==strncmp(argv[1],"-h",2) ){
      prnComHlp(opt,comHelp);
      exit(EXIT_FAILURE);
    }
    if( 0==strncmp(argv[1],"-XML",4) ){
      prnComXml(opt,comHelp);
      exit(EXIT_FAILURE);
    }
    if( 0==strncmp(argv[1],"-MAN",4) ){
      prnMan(opt,comHelp);
      exit(EXIT_FAILURE);
    }
  }
}

/**
 * # FUNCTION #
 * optFldのi番目の項目に項目オプションcが指定されているかどうかを返す。
 * ex)
 *  "-k 日付%nr"の時、c=='r'もしくはc=='n'の時１を返し、それ以外の時０を返す。
 */
int mssIsFldOptOn(MssOptFLD *optFld,int i, char c )
{

  if(optFld==NULL) return(0);
  if(optFld->fldOpt==NULL) return(0);
  if(*(optFld->fldOpt+i)==NULL) return(0);
  if( NULL == strchr(*(optFld->fldOpt+i),c) ) return(0);

  return(1);
}

/**
 * # FUNCTION #
 * キー項目オプション構造体(MssOptKEY)のstr以下の情報を実体としてコピーする。
 */
void mssCpyOptKey(MssOptKEY *to,MssOptKEY *from)
{
  int i;

  to->str=mssStrdup(from->str);
  /*初期化*/
  to->diffSame=from->diffSame;
  to->cnt     =from->cnt;

  if(to->cnt==0){
    to->nam   =NULL;
  }else{
    to->nam   =mssMalloc(sizeof(char *)*to->cnt,"cpyOptKey");

    for(i=0; i<to->cnt; i++){
      if( *(from->nam+i)!=NULL ) *(to->nam+i)=mssStrdup(*(from->nam+i));
    }
  }
}

/**
 * # FUNCTION #
 * 項目オプション構造体(MssOptFLD)のstr以下の情報を実体としてコピーする。
 */
void mssCpyOptFld(MssOptFLD *to,MssOptFLD *from)
{
  int i;

  to->str=mssStrdup(from->str);
  to->cnt     =from->cnt;

  if(to->cnt==0){
    to->nam   =NULL;
    to->newNam=NULL;
    to->fldOpt=NULL;
  }else{
    to->nam   =mssCalloc(sizeof(char *)*to->cnt,"cpyOptFld");
    to->newNam=mssCalloc(sizeof(char *)*to->cnt,"cpyOptFld");
    to->fldOpt=mssCalloc(sizeof(char *)*to->cnt,"cpyOptFld");

    for(i=0; i<to->cnt; i++){
      if( *(from->nam   +i)!=NULL )*(to->nam   +i)=mssStrdup(*(from->nam   +i));
      if( *(from->newNam+i)!=NULL )*(to->newNam+i)=mssStrdup(*(from->newNam+i));
      if( *(from->fldOpt+i)!=NULL )*(to->fldOpt+i)=mssStrdup(*(from->fldOpt+i));
    }
  }
}

/**
 * # FUNCTION #
 * 与えられた数値(val)が数値範囲構造体の範囲にあれば１を返し、
 * そうでなければ０を返す。
 */
int mssIsInRange(MssOptRNG *optRNG,int val)
{
  int i;

  for(i=0; i<optRNG->cnt; i++){
    if( *(optRNG->from+i) <= val && val<=*(optRNG->to+i) ) return(1);
  }
  return(0);
}

