/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるデータの出力関連の関数
 * 20040824 -z オプションを無向化 (標準出力は圧縮無し)に仕様変更
 *             (mssOpenFPW関数を大幅に変更)
 * ============================================================================
 */

#include <musashi/mssConfig.h>
#include <musashi/mssOutput.h>
#include <musashi/mssBase.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <zlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

/**
 * # FUNCTION #
 * 圧縮ファイルの書き込みオープンエラーの表示＆終了。
 */
static void zwOpenErr(char *fName)
{
  mssShowErrMsg("gz file write open error :\"%s\"",fName);
  mssEnd(mssErrorNoDefault);
}

/**
 * # FUNCTION #
 * 通常ファイルの書き込みオープンエラーの表示＆終了。
 */
static void fwOpenErr(char *fName)
{
  mssShowErrMsg("file write open error :\"%s\"",fName);
  mssEnd(mssErrorNoDefault);
}

/**
 * # FUNCTION #
 * ファイル名から必要なディレクトリを作成する。
 * ex.)
 * fileName="abc/def/ghi/dat.xt"
 *  -> カレントディレクトリにおいて、abc/dec/ghiのディレクトリを作成。
 *     最後の/以下は通常ファイルとみなす。
 */
static void mkDir(char *fileName)
{
  char fn[MssFileNameMaxLen];
  char *slaPos;
  char *dirTbl[100];
  int cnt=0;
  int i;
  int rt=0; /*ルートが指定されているかフラグ'/tmp'*/
  int err;
  char dirStr[MssFileNameMaxLen];

  if(*fileName=='/'){
    fileName++;
    rt=1;
  }

  strcpy(fn,fileName);

  slaPos = strtok(fn,"/");
  while(slaPos != NULL) {
    dirTbl[cnt]=slaPos;    
    slaPos = strtok(NULL,"/");
    cnt++;
    if(cnt>=100) {
      mssShowErrMsg("too deep");
      mssEnd(mssErrorNoDefault);
    }
  }
  if(cnt<=1) return;

  if(rt){
    dirStr[0]='/'; dirStr[1]='\0';
  }else{
    dirStr[0]='\0';
  }

  for(i=0; i<cnt-1; i++){
    strcat(dirStr,dirTbl[i]);
    strcat(dirStr,"/");
    err=mkdir(dirStr,S_IRWXU|S_IRWXG|S_IRWXO);
    if(err==-1 && errno==EEXIST) continue;
    if(err==0) continue;
    mssShowErrMsg("can not make directory(%d) : \"%s\"",errno,dirStr);
    mssEnd(mssErrorNoDefault);
  }
}

/**
 * # FUNCTION #
 * 書き込みファイルをオープンする。
 * ファイル名(fileName)がNULLの場合は標準出力、１の場合は標準エラー出力として
 * オープンする。
 * 第二引数が１の場合は、圧縮ファイルとしてオープンし、書き込み結果は全て
 * gzip圧縮される。
 * 第三引数が１の場合は、ファイル名に含まれるディレクトリを強制的に作成する。
 *
 * ファイル名を判断して圧縮するかどうかを決める。
 * 第二引数の効力はなくす。(1.1.0では廃止の方向で)
 */
struct mssFPW *mssOpenFPW( char *fileName, int z, int d)
{
  struct mssFPW *fp;
  int len;
  char *pos;

  if(d) mkDir(fileName);

  fp=mssMalloc(sizeof(struct mssFPW),"openFPW");

  if(fileName == NULL){          /*stdout*/
    fp->fp=stdout;
    //fp->fp=fopen("/dev/stdout","wb");//stdout;
    fp->fName=mssStrdup("stdout");
    fp->zflg=0;
  }else if(fileName==(char *)1){ /*stderr*/
    fp->fp=stderr;
    //fp->fp=fopen("/dev/stderr","wb");//stderr;
    fp->fName=mssStrdup("stderr");
    fp->zflg=0;
  }else{
    len=strlen(fileName);
    fp->zflg=0;
    if(len>=3){
      pos=fileName+len-3;
      if(*pos=='.' && *(pos+1)=='g' && *(pos+2)=='z'){
        fp->zflg=1;
      }
    }
    if(fp->zflg){
      fp->zfd=gzopen(fileName,"wb");
      if(fp->zfd == NULL) zwOpenErr(fileName);
    }else{
      fp->fp=fopen(fileName,"wb");
      if(fp->fp == NULL) fwOpenErr(fileName);
    }
    fp->fName=mssStrdup(fileName);
  }
  return(fp);
}

/**
 * # FUNCTION #
 * 書き込みファイルをクローズする。
 */
void mssCloseFPW(struct mssFPW *fp){

  if(fp==NULL) return;

  if(fp->zflg){
    gzclose(fp->zfd);
  }else{
    if(fp->fp != stdout && fp->fp != stderr){
      fclose(fp->fp);
    }
  }

  mssFree(fp->fName);
  mssFree(fp);
}

/**
 * # FUNCTION #
 * xmlTableの項目デリミター文字(MssFieldDelim)の出力。
 */
void mssWriteDlm(struct mssFPW *fp){
  if(fp->zflg){
    gzputc(fp->zfd,MssFieldDelim);
  }else{
    fputc(MssFieldDelim,fp->fp);
  }
}

/**
 * # FUNCTION #
 * 改行文字の出力。
 */
void mssWriteRet(struct mssFPW *fp){
  if(fp->zflg){
    gzputc(fp->zfd,'\n');
  }else{
    fputc('\n',fp->fp);
  }
}

/**
 * # FUNCTION #
 * 文字列の出力。
 */
void mssWriteStr(char *str,struct mssFPW *fp){
  if(fp->zflg){
    gzputs(fp->zfd, str);
  }else{
    fputs(str, fp->fp);
  }
}

/**
 * # FUNCTION #
 * 文字の出力。
 */
void mssWriteChr(char chr,struct mssFPW *fp){
  if(fp->zflg){
    gzputc(fp->zfd, chr);
  }else{
    fputc(chr, fp->fp);
  }
}

/**
 * # FUNCTION #
 * 整数(int)の出力。
 */
void mssWriteInt(int num,struct mssFPW *fp){
  if(fp->zflg){
    gzprintf(fp->zfd,"%d",num);
  }else{
    fprintf(fp->fp,"%d",num);
  }
}

/**
 * # FUNCTION #
 * Long整数(long)の出力。
 */
void mssWriteLong(long num,struct mssFPW *fp){
  if(fp->zflg){
    gzprintf(fp->zfd,"%ld",num);
  }else{
    fprintf(fp->fp,"%ld",num);
  }
}

/**
 * # FUNCTION #
 * Double型実数(double)を全て１０進浮動小数点表記("整数部.小数部")の書式で出力。
 */
void mssWriteDbl(double num, struct mssFPW *fp){
  char *buf;

  buf=mssFtoA(num);
  mssWriteStr(buf,fp);
  mssFree(buf);
}

/**
 * # FUNCTION #
 * Double型実数(double)を数の大きさに応じて１０進浮動小数点表記もしくは
 * １０進指数表記(ex.3.14152e+10)の書式で出力。
 */
void mssWriteDbe(double num, struct mssFPW *fp){
  char buf[100];

  sprintf(buf,"%g",num);
  mssWriteStr(buf,fp);
}

/**
 * # FUNCTION #
 * NULL値(MssNullChr)の出力。
 */
void mssWriteNull(struct mssFPW *fp){
  mssWriteChr(MssNullChr,fp);
}

/**
 * # FUNCTION #
 * 現在時刻の出力。
 * sepFlgを１に指定すると、"HH:MM:SS"の書式で出力。
 * それ以外は"HHMMSS"の書式で出力。
 */
void mssWriteTime(struct mssFPW *fp,int sepFlg){
  time_t  long_time;
  struct tm     *nt;
  char msg[100];

  time(&long_time);
  nt = localtime(&long_time);

  if(sepFlg){
    sprintf(msg, "%02d:%02d:%02d",
          nt->tm_hour,
          nt->tm_min,
          nt->tm_sec);
  }else{
    sprintf(msg, "%02d%02d%02d",
          nt->tm_hour,
          nt->tm_min,
          nt->tm_sec);
  }
  mssWriteStr(msg,fp);
}

/**
 * # FUNCTION #
 * 現在日付の出力。
 * sepFlgを１に指定すると、"YYYY/MM/DD"の書式で出力。
 * それ以外は"YYYYMMDD"の書式で出力。
 */
void mssWriteDate(struct mssFPW *fp,int sepFlg){
  time_t  long_time;
  struct tm     *nt;
  char msg[100];

  time(&long_time);
  nt = localtime(&long_time);

  /*日付*/
  if(sepFlg){
    sprintf(msg, "%04d/%02d/%02d",
          nt->tm_year + 1900,
          nt->tm_mon + 1,
          nt->tm_mday);
  }else{
    sprintf(msg, "%04d%02d%02d",
          nt->tm_year + 1900,
          nt->tm_mon + 1,
          nt->tm_mday);
  }
  mssWriteStr(msg,fp);
}

/**
 * # FUNCTION #
 * 文字列配列のcnt個の文字列を出力し、末尾文字列としてendStrを出力する。
 * 出力される文字列と文字列の間にはMssFieldDelimが出力される。
 * この関数は、通常、データ一行を出力するために利用する。
 * 
 */
void mssWriteFld(char **str, int cnt, char *endStr, struct mssFPW *fp){
  int i;
  for(i=0; i<cnt-1; i++){
    mssWriteStr(*(str+i),fp);
    mssWriteDlm(fp);
  }
  mssWriteStr(*(str+i),fp);
  mssWriteStr(endStr,fp);
}
