/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるPMML関連の関数
 *
 * 多くの関数で引数のコメントの形式を統一している。
 * 引数名 引数の説明 [必須 | optional] (選択リスト) | CDATA
 * [必須]とコメントされている引数がNULLの場合はError終了する。
 *   CDATAの場合は自由記述を意味する。
 *   (選択リスト)の場合は、その中の一つが指定されていなければエラー終了する。
 * [optional]とコメントされている引数がNULLの場合は属性として何も出力しない。
 *   なお[*]がついた要素はデフォルトを意味している。
 * ============================================================================
 */
#include <stdarg.h>
#include <string.h>
#include <musashi/mssOutput.h>
#include <musashi/mssHeader.h>
#include <musashi/mssXml.h>
#include <musashi/mssPMML.h>

/**
 * グローバル変数
 */
struct mssPMMLvariables mssPV;

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * PMML共通関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * PMML出力のための初期化
 */
void mssPMMLinit(){
  mssPV.indentLevel=0;
  mssPV.outRetCnt=0;
}

/**
 * # FUNCTION #
 * インデントの出力
 * ユーザが設定したグローバル変数mssPMML.indentLevelによってインデント幅が決まる
 */
void mssPMMLindent(struct mssFPW *fpw){
  int i;
  for(i=0; i<mssPV.indentLevel; i++){
    mssWriteStr("  ",fpw);
  }
}

/**
 * # FUNCTION #
 * 属性があるエンティティの条件にあっているかをチェックする。
 * 文字列(一つ目の引数)が、指定の引数リスト(二つめ以降の引数)になければ
 * エラー終了。
 */
void checkOneInListAttribute(char *attName, char *elementName, char *att, ...){
  va_list args;
  char *s;
  int stat=0;
  int msgLen=0;
  char *msg;
  char *msg0="attribute %s of %s tag must be one in {";

  va_start(args,att);
  while(1){
    s=va_arg(args, char *);
    if(s==NULL) break;
    msgLen+=strlen(s)+1; /* comma */
    if(0==strcmp(att,s)){
      return;
    }
  }
  va_end(args);

  if(stat==0){
    msgLen+=strlen(msg0)+2; /* }\0 の２文字分追加*/
    msg=mssMalloc(sizeof(char)*msgLen,"checkMustOne");
    *msg='\0';
    strcat(msg,msg0);
    va_start(args,att);
    while(1){
      s=va_arg(args, char *);
      if(s==NULL) break;
      strcat(msg,s);
      strcat(msg,",");
    }
    va_end(args);
    strcat(msg,"}");
    mssShowErrMsg(msg,attName,elementName);
    mssEnd(mssErrorNoDefault);
  }
}

/**
 * # FUNCTION #
 * 属性がフラグ(0 or 1)であるかをチェックする。
 * 0もしくは1でなければエラー終了。
 */
void checkIsFlagAttribute(char *attName, char *elementName, int *attInt){
  if(attInt==NULL)return;
  if(*attInt!=0 && *attInt!=0){
    mssShowErrMsg("attribute %s of %s tag must be 0 or 1",attName,elementName);
    mssEnd(mssErrorNoDefault);
  }
}

/**
 * # FUNCTION #
 * 必須の属性が指定されているかどうかをチェックする。
 * 文字列(att)がNULLの場合、エラー終了。
 */
void checkMandatoryAttribute(char *attName, char *elementName,void *att){

  if(att==NULL){
    mssShowErrMsg("Internal Error: %s attribute is mandatory in %s tag",attName,elementName);
    mssEnd(mssErrorNoDefault);
  }

}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * General 関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * Extension の開始タグの出力
 * ex) <Extension extender="MUSASHI" name="optype" value="pattern">
 */
void mssPMMLextensionStart( char *extender, char *name, char *value, int   emptyFlg, struct mssFPW *fpw)
{
  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="Extension";
  tag=mssInitXmlTag(eleNam,NULL);

  /* extender属性 (Optional)*/
  attNam="extender";
  attStr= extender;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* name属性 (Optional)*/
  attNam="name";
  attStr= name;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* value属性 (Optional)*/
  attNam="value";
  attStr= value;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* 書き出し*/
  if(emptyFlg){
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlEmptyTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
    mssPV.indentLevel--;
  }else{
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlStartTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
  }

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Extension の終了タグの出力
 * ex) </Extension>
 */
void mssPMMLextensionEnd(struct mssFPW *fpw)
{
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="Extension";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}


/**
 * # FUNCTION #
 * Array(文字列) の要素の出力
 * ex) <Array n="3" type="string">a aa aaa</Array>
 */
void mssPMMLarrayStr(int *n, char **str, struct mssFPW *fpw){

  struct mssXmlTag *tag;
  int i;

  char *eleNam;
  char *attNam;
  char *attStr;
  int  *attInt;

  eleNam="Array";
  tag=mssInitXmlTag(eleNam,NULL);

  /* n属性 (Optional)*/
  attNam="n";
  attInt= n;
  if(n!=NULL){
    mssAddXmlTagAttributeInt(tag,attNam,*attInt,NULL);
  }

  /* type属性 (Optional)*/
  attNam="type";
  attStr="string";
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  for(i=0; i<*n; i++){
    mssWriteXmlContent(*(str+i),NULL,fpw);
    if(i!=*n-1) mssWriteXmlContent(" ",NULL,fpw);
  }
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * False の空タグの出力
 * ex) <True/>
 */
void mssPMMLfalseEmpty(struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;

  eleNam="False";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlEmptyTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * True の空タグの出力
 * ex) <True/>
 */
void mssPMMLtrueEmpty(struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;

  eleNam="True";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlEmptyTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}


/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * PMML 関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * PMML の開始タグの出力
 * ex) <PMML version="2.0">
 */
void mssPMMLpmmlStart(
  char *version,
  struct mssFPW *fpw){
  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="PMML";
  tag=mssInitXmlTag(eleNam,NULL);

  /* version属性 (必須)*/
  attNam="version";
  attStr= version;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  checkOneInListAttribute(attNam, eleNam, attStr, "2.0",NULL);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * PMML の終了タグの出力
 * ex) </PMML>
 */
void mssPMMLpmmlEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="PMML";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * Header 関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * Header の開始タグの出力
 * ex) <Header copyright="MUSASHI" description="customer loyalty">
 */
void mssPMMLheaderStart(
  char *copyright,
  char *description,
  struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="Header";
  tag=mssInitXmlTag(eleNam,NULL);

  /* copyright属性 (必須)*/
  attNam="copyright";
  attStr= copyright;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* description属性 (Optional)*/
  attNam="description";
  attStr= description;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Header の終了タグの出力
 * ex) </Header>
 */
void mssPMMLheaderEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="Header";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Application の空タグの出力
 * ex) <Application name="xtclassify" version="1.1"/>
 */
void mssPMMLapplicationEmpty(
  char *name,
  char *version,
  struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="Application";
  tag=mssInitXmlTag(eleNam,NULL);

  /* name属性 (必須)*/
  attNam="name";
  attStr= name;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* version属性 (Optional)*/
  attNam="version";
  attStr= version;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlEmptyTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Timestamp の出力
 * ex) <Timestamp>10 JUN 2003 13:16:15</Timestamp>
 */
void mssPMMLtimestamp(struct mssFPW *fpw){
  char *mon[]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
  time_t  long_time;
  struct tm     *nt;
  char msg[100];
  struct mssXmlTag *tag;

  time(&long_time);
  nt = localtime(&long_time);

  sprintf(msg, "%d %s %04d %02d:%02d:%02d",
          nt->tm_mday,
          mon[nt->tm_mon],
          nt->tm_year + 1900,
          nt->tm_hour,
          nt->tm_min,
          nt->tm_sec);

  tag=mssInitXmlTag("Timestamp",NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteXmlContent(msg,NULL,fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * Data Dictionary 関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * DataDictionary の開始タグの出力
 * ex) <DataDictionary numberOfFields="5">
 */
void mssPMMLdataDictionaryStart(int *numberOfFields, struct mssFPW *fpw)
{
  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  int  *attInt;

  eleNam="DataDictionary";
  tag=mssInitXmlTag(eleNam,NULL);

  /* numberOfFields属性 (必須)*/
  attNam="numberOfFields";
  attInt= numberOfFields;
  checkMandatoryAttribute(attNam, eleNam, attInt);
  mssAddXmlTagAttributeInt(tag,attNam,*attInt,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * DataDictionary の終了タグの出力
 * ex) </PMML>
 */
void mssPMMLdataDictionaryEnd(struct mssFPW *fpw)
{
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="DataDictionary";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * DataField の開始タグの出力
 * ex) <DataField name="温度" optype="continuous">
 */
void mssPMMLdataFieldStart(char *name, char *displayName, char *optype, char *taxonomy, int  *isCyclic, int emptyFlg, struct mssFPW *fpw)
{
  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;
  int  *attInt;

  eleNam="DataField";
  tag=mssInitXmlTag(eleNam,NULL);

  /* name属性 (必須)*/
  attNam="name";
  attStr= name;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* displayName属性 (Optional)*/
  attNam="displayName";
  attStr= displayName;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* optype属性 (必須)*/
  attNam="optype";
  attStr= optype;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  checkOneInListAttribute(attNam, eleNam, attStr,
                          "categorical", "ordinal", "continuous",NULL);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* taxonomy属性 (Optional)*/
  attNam="taxonomy";
  attStr= taxonomy;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* isCyclic属性 (Optional)*/
  attNam="isCyclic";
  attInt= isCyclic;
  if(attInt!=NULL){
    checkIsFlagAttribute(attNam, eleNam, attInt);
    mssAddXmlTagAttributeInt(tag,attNam,*attInt,NULL);
  }

  /* 書き出し*/
  if(emptyFlg){
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlEmptyTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
    mssPV.indentLevel--;
  }else{
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlStartTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
  }

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * DataField の終了タグの出力
 * ex) </DataField>
 */
void mssPMMLdataFieldEnd(struct mssFPW *fpw)
{
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="DataField";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Value の開始タグの出力
 * ex) <Value value="*" property="missing">
 */
void mssPMMLvalueStart( char *value, char *displayValue, char *property, int   emptyFlg, struct mssFPW *fpw)
{
  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="Value";
  tag=mssInitXmlTag(eleNam,NULL);

  /* value属性 (必須)*/
  attNam="value";
  attStr= value;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* displayValue属性 (Optional)*/
  attNam="displayValue";
  attStr= displayValue;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* property属性 (Optional)*/
  attNam="property";
  attStr= property;
  if(attStr!=NULL){
    checkOneInListAttribute(attNam, eleNam, attStr,
                            "valid", "invalid", "missing",NULL);
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* 書き出し*/
  if(emptyFlg){
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlEmptyTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
    mssPV.indentLevel--;
  }else{
    mssPV.indentLevel++;
    mssWriteXmlStartTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
  }

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Value の終了タグの出力
 * ex) </Value>
 */
void mssPMMLvalueEnd(struct mssFPW *fpw)
{
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="Value";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * Mining Schema 関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * Mining Schema の開始タグの出力
 * ex) <MiningSchema>
 */
void mssPMMLminingSchemaStart(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="MiningSchema";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}   
    
/**
 * # FUNCTION #
 * Mining Schema の終了タグの出力
 * ex) </MiningSchema>
 */
void mssPMMLminingSchemaEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="MiningSchema";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * MiningField の開始タグの出力
 * ex) <MiningField name="ゴルフ" usageType="predicted"/>
 * 引数)
 * char *name:                    項目名        [必須]     CDATA
 * char *usageType:               利用タイプ    [Optional] ("active",
 *                                                          "predicted",
 *                                                          "supplimentary")
 * char *outliers:                異常値の扱い  [optional] ("asIs",
                                                            "asMissingValues",
                                                            "asExtremeValues")
 * double *lowValue:              異常値の範囲(最小値) [optional] CDATA
 * double *highValue:             異常値の範囲(最大値) [optional] CDATA
 * char *missingValueReplacement: NULL値を変換する値 [optional] CDATA
 * char *missingValueTreatment:   NULL値の扱い [optional] ("asIs,asMean",
 *                                                         "asMode",
 *                                                         "asMedian",
 *                                                         "asValue")
 * int empty: 1:空タグとして出力 0:開始タグとして出力
 */
void mssPMMLminingFieldStart(char *name, char *usageType, char *outliers, double *lowValue, double *highValue, char *missingValueReplacement, char *missingValueTreatment, int emptyFlg, struct mssFPW *fpw)
{

  struct mssXmlTag *tag;

  char   *eleNam;
  char   *attNam;
  char   *attStr;
  double *attDbl;

  eleNam="MiningField";
  tag=mssInitXmlTag(eleNam,NULL);

  /* name属性 (必須)*/
  attNam="name";
  attStr= name;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* usageType属性 (Optional)*/
  attNam="usageType";
  attStr= usageType;
  if(attStr!=NULL){
    checkOneInListAttribute(attNam, eleNam, attStr,
                            "active", "predicted", "supplementary",NULL);
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* outliers属性 (Optional)*/
  attNam="outliers";
  attStr= outliers;
  if(attStr!=NULL){
    checkOneInListAttribute(attNam, eleNam, attStr,
                            "asIs", "asMissingValues", "asExtremeValues",NULL);
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* lowValue属性 (Optional)*/
  attNam="lowValue";
  attDbl= lowValue;
  if(attDbl!=NULL){
    mssAddXmlTagAttributeDbl(tag,attNam,*attDbl,NULL);
  }

  /* highValue属性 (Optional)*/
  attNam="highValue";
  attDbl= highValue;
  if(attDbl!=NULL){
    mssAddXmlTagAttributeDbl(tag,attNam,*attDbl,NULL);
  }

  /* missingValueReplacement属性 (Optional)*/
  attNam="missingValueReplacement";
  attStr= missingValueReplacement;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* missingValueTreatment属性 (Optional)*/
  attNam="missingValueTreatment";
  attStr= missingValueTreatment;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* 書き出し*/
  if(emptyFlg){
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlEmptyTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
    mssPV.indentLevel--;
  }else{
    mssPV.indentLevel++;
    mssPMMLindent(fpw);
    mssWriteXmlStartTag(tag,NULL,fpw);
    mssWriteRet(fpw); mssPV.outRetCnt++;
  }

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Mining Field の終了タグの出力
 * ex) </MiningField>
 */
void mssPMMLminingFieldEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="MiningField";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}


/**
 * # FUNCTION #
 * MiningField タグを空タグとして、mssFieldsに登録された複数項目について出力
 */
void mssPMMLminingFields(struct mssFields *fld, char *usageType, char *outliers, double *lowValue, double *highValue, char *missingValueReplacement, char *missingValueTreatment, struct mssFPW *fpw)
{
  int i;

  for(i=0; i<fld->cnt; i++){
    mssPMMLminingFieldStart(MssFlds2name(fld,i), usageType, outliers,
                            lowValue, highValue, missingValueReplacement,
                            missingValueTreatment, 1, fpw);
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * Trees 関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * Tree の開始タグの出力
 * ex) <TreeModel modelName="ゴルフ" functionName="classification">
 * 引数)
 * char *modelName:           モデル名        [optional] CDATA
 * char *functionName:        MINING-FUNCTION [必須]     ("associationRules",
 *                                                        "sequnces",
 *                                                        "classification",
 *                                                        "regression",
 *                                                        "clustering")
 * char *algorithmName:       アルゴリズム名  [optional] CDATA
 * char *splitCharacteristic: ノード分岐方法  [optional] ("binarySplit",
 *                                                        "multiSplit"[*])
 */
void mssPMMLtreeModelStart(
  char *modelName,
  char *functionName,
  char *algorithmName,
  char *splitCharacteristic,
  struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="TreeModel";
  tag=mssInitXmlTag(eleNam,NULL);

  /* modelName属性 (Optional)*/
  attNam="modelName";
  attStr= modelName;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* functionName属性 (必須)*/
  attNam="functionName";
  attStr= functionName;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  checkOneInListAttribute(attNam, eleNam, attStr,
                          "associationRules", "sequences", "classification",
                          "regression","clustering",NULL);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* algorithmName属性 (Optional)*/
  attNam="algorithmName";
  attStr= algorithmName;
  if(attStr!=NULL){
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* splitCharacteristic属性 (Optional)*/
  attNam="splitCharacteristic";
  attStr= splitCharacteristic;
  if(attStr!=NULL){
    checkOneInListAttribute(attNam, eleNam, attStr,
                            "binarySplit", "multiSplit",NULL);
    mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);
  }

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}


/**
 * # FUNCTION #
 * Tree の終了タグの出力
 * ex) </TreeModel>
 */
void mssPMMLtreeModelEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="TreeModel";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Node の開始タグの出力
 * ex) <Node score="will play" recordCount="14">
 * 引数)
 * char *score:           結果属性の値           [必須]     CDATA
 * int  *recordCount:     そのノードに属する件数 [Optional] CDATA
 */
void mssPMMLnodeStart( char *score, double *recordCount, struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;
  double *attDbl;

  eleNam="Node";
  tag=mssInitXmlTag(eleNam,NULL);

  /* score属性 (必須)*/
  attNam="score";
  attStr= score;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* recordCount属性 (Optional)*/
  attNam="recordCount";
  attDbl= recordCount;
  if(attDbl!=NULL){
    mssAddXmlTagAttributeDbl(tag,attNam,*attDbl,NULL);
  }

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * Node の終了タグの出力
 * ex) </TreeModel>
 */
void mssPMMLnodeEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="Node";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * SimplePredicate(数値) の空タグの出力
 * ex) <SimplePredicate field="数量" operator="lessThan",value=10.5/>
 */
void mssPMMLsimplePredicateNumEmpty(char *field,char *operator, double *value, struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;
  double *attDbl;

  eleNam="SimplePredicate";
  tag=mssInitXmlTag(eleNam,NULL);

  /* field属性 (必須)*/
  attNam="field";
  attStr= field;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* operator属性 (必須)*/
  attNam="operator";
  attStr= operator;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  checkOneInListAttribute(attNam, eleNam, attStr, "equal",
                                                  "notEqual",
                                                  "lessThan",
                                                  "lessOrEqual",
                                                  "greaterThan",
                                                  "greaterOrEqual");
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* value属性 (必須)*/
  attNam="value";
  attDbl= value;
  checkMandatoryAttribute(attNam, eleNam, attDbl);
  mssAddXmlTagAttributeDbl(tag,attNam,*attDbl,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlEmptyTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * SimplePredicate(文字列) の空タグの出力
 * ex) <SimplePredicate field="天気" operator="equal",value="晴れ"/>
 */
void mssPMMLsimplePredicateStrEmpty(char *field,char *operator, char *value, struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="SimplePredicate";
  tag=mssInitXmlTag(eleNam,NULL);

  /* field属性 (必須)*/
  attNam="field";
  attStr= field;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* operator属性 (必須)*/
  attNam="operator";
  attStr= operator;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  checkOneInListAttribute(attNam, eleNam, attStr, "equal",
                                                  "notEqual",
                                                  "lessThan",
                                                  "lessOrEqual",
                                                  "greaterThan",
                                                  "greaterOrEqual");
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* value属性 (必須)*/
  attNam="value";
  attStr= value;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlEmptyTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * SimpleSetPredicate の開始タグの出力
 * ex) <SimpleSetPredicate field="天気" booleanOperator="isIn">
 */
void mssPMMLsimpleSetPredicateStart(char *field,char *booleanOperator, struct mssFPW *fpw){

  struct mssXmlTag *tag;

  char *eleNam;
  char *attNam;
  char *attStr;

  eleNam="SimpleSetPredicate";
  tag=mssInitXmlTag(eleNam,NULL);

  /* field属性 (必須)*/
  attNam="field";
  attStr= field;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* booleanOperator属性 (必須)*/
  attNam="booleanOperator";
  attStr= booleanOperator;
  checkMandatoryAttribute(attNam, eleNam, attStr);
  checkOneInListAttribute(attNam, eleNam, attStr, "isIn", "isNotIn");
  mssAddXmlTagAttributeStr(tag,attNam,attStr,NULL);

  /* 書き出し*/
  mssPV.indentLevel++;
  mssPMMLindent(fpw);
  mssWriteXmlStartTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;

  mssFreeXmlTag(tag);
}

/**
 * # FUNCTION #
 * SimpleSetPredicate の終了タグの出力
 * ex) </SimpleSetPredicate>
 */
void mssPMMLsimpleSetPredicateEnd(struct mssFPW *fpw){
  struct mssXmlTag *tag;
  char *eleNam;

  eleNam="SimpleSetPredicate";
  tag=mssInitXmlTag(eleNam,NULL);

  /* 書き出し*/
  mssPMMLindent(fpw);
  mssWriteXmlEndTag(tag,NULL,fpw);
  mssWriteRet(fpw); mssPV.outRetCnt++;
  mssPV.indentLevel--;

  mssFreeXmlTag(tag);
}



