/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられる集計関連の関数
 * ============================================================================
 */

#include <musashi/mssBase.h>
#include <musashi/mssUniq.h>
#include <musashi/mssHeader.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

/**
 * グローバル変数
 */
extern struct mssGlobalVariables mssGV;

static MssOptKEY  *OptKey;               /*-kパラメータ*/
static int  FldCnt;                   /*集計する項目数*/
static char readEnd[2]={0xff,0};      /*番兵*/
static char fname[MssFileNameMaxLen]; /*一時ファイル名*/

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 赤黒木(Red Black Tree)
 * ----------------------------------------------------------------------------
 * 参考文献: 渡部敏正『データ構造とアルゴリズム』共立出版,2000,6章4節.
 * p.224 下から9行目は誤り!!
 * 誤) if (v == NULL ) return(1) -> 正) if (v == NULL ) return(0)
 */

/**
 * # FUNCTION #
 */
static int RBUQhasLeftNode(struct RBUQnode *v)
{
  if(v->left->rank !=0){
    return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBUQisExternalNode(struct RBUQnode *v)
{
  if(v->rank==0)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBUQisTopNode(struct RBUQnode *v)
{
  if(v->parent->rank==0){
    return(1);
  }
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBUQisLeftNode(struct RBUQnode *v)
{
  if(v==v->parent->left)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static int RBUQisRedNode(struct RBUQnode *v)
{
  if(v==NULL) return(0);
  if(v->rank == v->parent->rank)
    return(1);
  return(0);
}

/**
 * # FUNCTION #
 */
static struct RBUQnode *RBUQfindMax(struct RBUQnode *v)
{
  while(!RBUQisExternalNode(v->right))
    v=v->right;
  return(v);
}

/**
 * # FUNCTION #
 */
static struct RBUQnode *RBUQfind_brother(struct RBUQnode *v)
{
  if(RBUQisLeftNode(v))
    return(v->parent->right);
  return(v->parent->left);
}

/**
 * # FUNCTION #
 */
static struct RBUQnode *RBUQmakeNode(void)
{
  struct RBUQnode *new;
  struct RBUQkey  *key;
  new=(struct RBUQnode *)mssMalloc(sizeof(struct RBUQnode),"RBUQmkNode");
  key=(struct RBUQkey  *)mssMalloc(sizeof(struct RBUQkey ),"RBUQmkNode");
  new->key=key;
  /*new->key.str[0]='\0';*/
  new->key->str=NULL;
  new->rank = 0;
  new->left=new->right=NULL;
  return new;
}

/**
 * # FUNCTION #
 */
static void RBUQfreeKey(struct RBUQkey *key)
{
  mssFree(key->str);
  mssFree(key->fld);
  mssFree(key->bkt);
  mssFree(key);
}

/**
 * # FUNCTION #
 */
static void RBUQfreeNode(struct RBUQnode *v)
{
  if(RBUQisExternalNode(v)){
    mssFree(v->key);
    mssFree(v);
    return;
  }else{
    RBUQfreeKey(v->key);
    mssFree(v);
    return;
  }
}

/**
 * # FUNCTION #
 */
static void RBUQfreeAllNode(struct RBUQnode *v)
{
  if(RBUQisExternalNode(v)){
    mssFree(v->key);
    mssFree(v);
    return;
  }else{
    RBUQfreeAllNode(v->left);
    RBUQfreeAllNode(v->right);
    RBUQfreeNode(v);
    return;
  }
}

/**
 * # FUNCTION #
 */
static int keyCmp(struct mssFldRec *fr, struct RBUQnode *v)
{
  int i;
  int cmp;

  for(i=0; i<OptKey->flds->cnt; i++){
    if( 0 != ( cmp=strcmp( *(fr->pnt+MssFlds2num(OptKey->flds,i)),
                           *(v->key->fld+MssFlds2num(OptKey->flds,i)) )) ){
      return(cmp);
    }
  }
  return(0); /*同じキー!!*/
}

/**
 * # FUNCTION #
 */
static struct RBUQnode *RBUQmember(struct mssFldRec *fr, struct RBUQnode *v)
{
  struct RBUQnode *vv;

  if(RBUQisExternalNode(v)) return(v);

  if( 0 > keyCmp(fr, v) ){
    vv=RBUQmember(fr,v->left);
  }else if( 0 < keyCmp(fr, v) ){
    vv=RBUQmember(fr,v->right);
  }else{
    return(v);
  }
  return(vv);
}

/**
 * # FUNCTION #
 */
static void RBUQaddExternalNodes(struct RBUQnode *n)
{
  n->left = RBUQmakeNode();
  n->right= RBUQmakeNode();
  n->left->parent =n;
  n->right->parent=n;
}

/**
 * # FUNCTION #
 */
static void RBUQsingleRotate(struct RBUQnode *v)
{
  struct RBUQnode *p, *pp,*ppp;

  p  =v->parent;
  pp =p->parent;
  ppp=pp->parent;

  if(RBUQisLeftNode(pp)){
    ppp->left=p;
  }else{
    ppp->right=p;
  }
  if(RBUQisLeftNode(v)){
    pp->left=p->right;
    pp->left->parent=pp;
    p->right=pp;
    pp->parent=p;
  }else{
    pp->right = p->left;
    pp->right->parent = pp;
    p->left = pp;
    pp->parent = p;
  }
  p->parent=ppp;
}

/**
 * # FUNCTION #
 */
static void RBUQdouble_rotate(struct RBUQnode *v)
{
  struct RBUQnode *p;
  p=v->parent;
  if(RBUQisLeftNode(v)){
    RBUQsingleRotate(v->left);
    RBUQsingleRotate(p);
  }else{
    RBUQsingleRotate(v->right);
    RBUQsingleRotate(p);
  }
}

/**
 * # FUNCTION #
 */
static void RBUQrebalanceOnInsert(struct RBUQnode *v)
{
  struct RBUQnode *w, *p, *pp;
  int p_is_left;
  
  p=v->parent;
  pp=p->parent;
  if(RBUQisTopNode(p)){
    return;
  }
  if(RBUQisRedNode(p)){
    p_is_left=0;   
    if((p_is_left=RBUQisLeftNode(p))){  
      w=pp->right;
    }else{
      w=pp->left;
    }
    if(RBUQisRedNode(w)){   
      (pp->rank)++;
      if(!RBUQisTopNode(pp) && RBUQisRedNode(pp->parent)){
        RBUQrebalanceOnInsert(pp);
      }
    }else{
      if(RBUQisLeftNode(v)){
        if(p_is_left){
          RBUQsingleRotate(v);
        }else{
          RBUQdouble_rotate(v);
        }
      }else{
        if(p_is_left){
          RBUQdouble_rotate(v);
        }else{
          RBUQsingleRotate(v);
        }
      }
    }
  }
}

/**
 * # FUNCTION #
 */
static void RBUQdetermineAB(struct RBUQnode **a, struct RBUQnode **b, struct RBUQnode *v, struct RBUQnode *u)
{
  if(RBUQisLeftNode(v)){
    *a=u->right;
    *b=u->left;
  }else{
    *a=u->left;
    *b=u->right;
  }
}

/**
 * # FUNCTION #
 */
static void RBUQrebalanceOnDelete(struct RBUQnode *v)
{
  struct RBUQnode *p,*u,*a,*b;
  int v_is_left, p_was_red;

  p=v->parent;
  if(v->rank+1 >= p->rank)
    return;
  u=RBUQfind_brother(v);
  RBUQdetermineAB(&a,&b,v,u);
  v_is_left=RBUQisLeftNode(v);
  p_was_red=RBUQisRedNode(p);
  if(RBUQisRedNode(u)){
    if(RBUQisRedNode(a) || RBUQisRedNode(b))
      return;
    RBUQsingleRotate(a);
    RBUQrebalanceOnDelete(v);
  }else{
    if(!RBUQisRedNode(a)){
      if(!RBUQisRedNode(b)){
        (p->rank)--;
        if(!p_was_red)
          RBUQrebalanceOnDelete(p);
      }else{
        RBUQdouble_rotate(b);
        (b->rank)++;
        (p->rank)--;
      }
    }else{
      RBUQsingleRotate(a);
      (u->rank)++;
      (u->rank)--;
    }
  }
}

/**
 * # FUNCTION #
 */
static void RBUQptree(struct RBUQnode *p,int h)
{
  int i,j;

  if(!RBUQisExternalNode(p)){
    RBUQptree(p->left, h+1);
    if(0 == strcmp(p->key->str,readEnd) ){
      for(i=1; i<=h; i++) fprintf(stderr,"    ");
      printf("key='EOF' ");
      /*printf(" bktCnt=%d bkt=",p->key->bktCnt);*/
      for(j=0; j<PWayU; j++){
        if(*(p->key->bkt+j) == 1)
        printf("%d ",j);
      }
      printf("\n");
    }else{
      for(i=1; i<=h; i++) fprintf(stderr,"    ");
      printf("key='");
      for(j=0; j<OptKey->flds->cnt; j++){
        printf("%s ",*(p->key->fld+MssFlds2num(OptKey->flds,j)+1));
      }

      /*printf(" bktCnt=%d bkt=",p->key->bktCnt);*/
      for(j=0; j<PWayU; j++){
        if(*(p->key->bkt+j) == 1)
        printf("%d ",j);
      }
      /*for(j=0; j<p->key->bktCnt; j++){*/
      /*  printf("%d ",*(p->key->bkt+j));*/
      /*}*/
      printf("\n");
    }   
    RBUQptree(p->right,h+1);
  }
}

/**
 * # FUNCTION #
 * RBUQcpKeyの関数で確保されたメモリ量の計算
 */
static int RBUQcpKeyMemCnt( struct mssFldRec *fr)
{

  int memCnt=0;

  if(fr->eof==1){ /*EOF*/
    /*文字列として番兵を入れておく*/
    memCnt+=2*sizeof(char);

    /*キー項目のアドレスを全て番兵に設定する*/
    memCnt+=FldCnt*sizeof(char *);

    /*sortUQでは以下のbkt変数は利用していない。mergeRBUQで利用。*/
    memCnt+=(sizeof(int)*PWayU);

  }else{

    /*frの全項目をまるごとコピー*/
    memCnt+=fr->chrCnt*sizeof(char);

    /*キー項目のアドレスを計算によりセット*/
    memCnt+=FldCnt*sizeof(char *);

    /*sortUQでは以下のbkt変数は利用していない。mergeRBUQで利用。*/
    memCnt+=(sizeof(int)*PWayU);
  }
  return(memCnt);
}

/**
 * # FUNCTION #
 */
static void RBUQcpKey( struct RBUQnode  *v, struct mssFldRec *fr, int bkt)
{
  int i;

  if(fr->eof==1){ /*EOF*/
    /*文字列として番兵を入れておく*/
    v->key->str = mssMalloc(2*sizeof(char),"RBUQUQtree1");
    memcpy(v->key->str, readEnd, 2);

    /*キー項目のアドレスを全て番兵に設定する*/
    v->key->fld = mssMalloc(FldCnt*sizeof(char *),"RBUQUQtree2");
    for(i=0; i<FldCnt; i++){
      *(v->key->fld+i) = v->key->str;
    }

    /*sortUQでは以下のbkt変数は利用していない。mergeRBUQで利用。*/
    v->key->bkt    = mssCalloc(sizeof(int)*PWayU, "RBUQtree");
    *(v->key->bkt+bkt) = 1;
    /**v->key->bkt    = bkt;*/
    /*v->key->bktCnt = 1;*/

  }else{

    /*frの全項目をまるごとコピー*/
    v->key->str = mssMalloc(fr->chrCnt*sizeof(char),"RBUQUQtree4");
    memcpy(v->key->str, *fr->pnt, fr->chrCnt); 

    /*キー項目のアドレスを計算によりセット*/
    v->key->fld = mssMalloc(FldCnt*sizeof(char *),"RBUQUQtree5");
    for(i=0; i<FldCnt; i++){
      *(v->key->fld+i) = v->key->str + (*(fr->pnt+i) - *fr->pnt);
    }

    /*sortUQでは以下のbkt変数は利用していない。mergeRBUQで利用。*/
    v->key->bkt    = mssCalloc(sizeof(int)*PWayU, "RBUQtree");
    *(v->key->bkt+bkt) = 1;
  }
}

/**
 * # FUNCTION #
 *指定のノードをFldRec構造体のデータで更新
 */
static void RBUQupdate( struct RBUQnode *v, int bkt)
{
  /*更新したバケット番号をバケットフラグに更新する*/
  *(v->key->bkt+bkt) = 1;
}

/**
 * # FUNCTION #
 * RBtreeにFldRec構造体のデータを追加
 * 既にキーが存在すればRBUQupdateで値を更新
 */
static int RBUQinsert( struct RBUQnode *v, struct mssFldRec *fr, int bkt)
{
  int memCnt=0;

  v = RBUQmember(fr, v);

  /*キーが存在しなければ追加*/
  if(RBUQisExternalNode(v)){
    RBUQcpKey(v, fr, bkt);
    memCnt=RBUQcpKeyMemCnt(fr);
    v->rank=1;
    RBUQaddExternalNodes(v);
    RBUQrebalanceOnInsert(v);
    memCnt+=sizeof(struct RBUQnode)*2;
    memCnt+=sizeof(struct RBUQkey )*2;

  /*キーが存在すれば、読み込みフラグをON*/
  }else{
    RBUQupdate(v, bkt);
  }

  return(memCnt);
}

/**
 * # FUNCTION #
 *指定のノードを削除
 */
static void RBUQdeleteNode(struct RBUQnode *v)
{
  struct RBUQnode *w,*x;

  if(RBUQisExternalNode(v)){
    fprintf(stderr,"Not found such node\n");
    return;
  }
  if(RBUQhasLeftNode(v)){
    w=RBUQfindMax(v->left);
    if(w->parent != v){
      w->parent->right = w->left;
      w->left->parent  = w->parent;
    }
    x=w->left;
    RBUQfreeNode(w->right);
    /*free(w->right->key);*/
    /*free(w->right);*/

    if(RBUQisLeftNode(v)){
      v->parent->left=w;
    }else{
      v->parent->right=w;
    }
    if(w->parent != v){
      w->left = v->left;
      w->left->parent = w;
    }
    w->rank = v->rank;
    w->parent = v->parent;
    w->right = v->right;
    w->right->parent=w;
    RBUQfreeNode(v);
    /*free(v->key);*/
    /*free(v);*/
  }else{
    w=v->right;
    if(RBUQisLeftNode(v)){
      v->parent->left=w;
    }else{
      v->parent->right=w;
    }
    w->parent=v->parent;
    RBUQfreeNode(v->left);
    RBUQfreeNode(v);
    /*free(v->left->key);*/
    /*free(v->left);*/
    /*free(v->key);*/
    /*free(v);*/
    x=w;
  }

  RBUQrebalanceOnDelete(x);
}

/**
 * # FUNCTION #
 *指定のノードを削除
 *RBtreeから優先行のノードアドレスを返す
 *この処理でノードを削除はしない。呼び出す本体側でRBUQdeleteNode(node)を実行
 */
static struct RBUQnode *RBUQpop(struct RBUQnode *v)
{
  struct RBUQnode *vv;

  if(RBUQisExternalNode(v)){
    return(NULL);
  }else{
    vv=v;
    while(!RBUQisExternalNode(vv)){
      vv=vv->left; /*rightを辿れば最大、leftは最小*/
    }
    return(vv->parent);
  }
}

/**
 * # FUNCTION #
 * RBtreeをツリーとして書き出す(デバッグ用)
 */
void RBUQprintTree(char *s,struct RBUQnode *pp)
{
  fprintf(stderr,"%s\n",s);
  RBUQptree(pp,0);
}

/**
 * # FUNCTION #
 * 指定のノードを行として書き出す
 */
static void RBUQwriteNode( struct RBUQnode *p, struct mssFPW *fpw)
{
  int i;

  /*データテキストの出力*/
  for(i=0; i<FldCnt-1; i++){
    mssWriteStr(*(p->key->fld+i),fpw); mssWriteDlm(fpw);
  }
  mssWriteStr(*(p->key->fld+i),fpw); mssWriteRet(fpw);
}

/**
 * # FUNCTION #
 * RBtreeを全て行として書き出す(デバッグ用)
 */
static void RBUQwriteAllNode(struct RBUQnode *p, struct mssFPW *fpw)
{
  if(!RBUQisExternalNode(p)){
    RBUQwriteAllNode(p->left,fpw);
    RBUQwriteNode(p,fpw);
    RBUQwriteAllNode(p->right,fpw);
  }
}

/**
 * # FUNCTION #
 * RBtreeのメモリ領域解放
 */
static void RBUQfree(struct RBUQnode *v)
{
  RBUQfreeAllNode(v->left);
  mssFree(v->key);
  mssFree(v);
}

/**
 * # FUNCTION #
 * RBnodeの初期化(ノード作成)
 */
static struct RBUQnode *RBUQinit(struct RBUQnode *rb)
{
  /*赤黒木の初期化*/
  rb               = RBUQmakeNode();
  rb->parent       = rb;
  rb->right        = NULL;
  rb->rank         = 0;
  rb->left         = RBUQmakeNode();
  rb->left->parent = rb;
  return(rb);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 集計関連関数
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 一時ファイル名の取得
 */
static char *getFname(char *prefix, int number)
{
  /*fnameはグローバル変数*/
  sprintf(fname,"%s%d",prefix,number);
  return(fname);
}

/**
 * # FUNCTION #
 * mssFldRec構造体に一行読み込む(番兵処理)
 */
static void readRBUQkey( struct mssFPR *fp, struct mssFldRec *fr)
{
  int i;

  if(EOF==mssReadFldRec(fp,fr)){
    for(i=0; i<fr->fldCnt; i++)
      *(fr->pnt+i)=readEnd; /*EOFの時は番兵を入れる*/
  }
}

/**
 * # FUNCTION #
 *各バケットの最初の行をRBキューに読み込み
 *mergeRB,preSortから呼び出される
 *iFromからiToまでのファイルの先頭行をRBキューに入れる
 * ud   : iFile,rbをセットする
 * iFrom: 開始ファイル番号
 * iTo  : 終了ファイル番号
 */
static void setFirstLineRBUQ( struct mssUnqDat *ud, int iFrom, int iTo)
{
  int bkt;                /*現在処理中のバケット番号(0から始まる)*/
  int i;

  /*プライオリティキューの初期化*/
  ud->rb=RBUQinit(ud->rb);
  bkt=0;
  for(i=iFrom; i<=iTo; i++){
    ud->fr[bkt]=mssInitFldRec(FldCnt);

    ud->iFile[bkt]=mssOpenFPR(getFname(ud->prefixTxt,i),4);

    /*一行(先頭行)読み込み*/
    readRBUQkey(ud->iFile[bkt],ud->fr[bkt]);

    /*赤黒木に挿入(物理的にコピー)*/
    RBUQinsert(ud->rb->left,ud->fr[bkt],bkt);
    bkt++;
  }
}

/**
 * # FUNCTION #
 * RBキューを用いたPWayマージ
 * 既に並べ変わっているバケット(iCnt個)を併合していく
 * 最後の一つになるまで併合はしない
 * PWayU個より少なくなったときに併合を止める
 */
static void mergeRBUQ(struct mssUnqDat *ud)
{
  int    bkt[PWayU];       /*Pway併合時にRBtreeからpopしたデータのバケットNo*/
  int    bktCnt;           /*そのバケット数*/
  struct RBUQnode *node;   /*popにて取りだしたノードを格納*/

  struct mssFPW *oFile;       /*出力ワークファイルポインタ*/

  int    iStart;           /*入力ワークファイルの開始番号(ファイル名の一部)*/
  int    iEnd;             /*入力ワークファイルの終了番号(ファイル名の一部)*/
  int    oStart;           /*出力ワークファイルの開始番号(ファイル名の一部)*/
  int    oEnd;             /*出力ワークファイルの終了番号(ファイル名の一部)*/
  int    iFrom,iTo;        /*併合する時のファイル番号*/
  int    k;
  int    i;

  /*次のループでin,outをswapするので、ここでは逆に定義*/
  iStart=ud->iEnd+1;
  iEnd  =ud->iEnd+1;
  oStart=ud->iStart;
  oEnd  =ud->iEnd;

  /*ファイルを併合するループ iCntがPWayU個より大きい場合まわり続ける*/
  while(1){
    mssSwapInt(&iStart,&oStart);
    mssSwapInt(&iEnd  ,&oEnd  );
    oEnd=oStart;

    /*入力ファイル数がPWayU以下ならば終了*/
    if(iEnd-iStart+1 <= PWayU){
      ud->iStart = iStart;
      ud->iEnd   = iEnd;
      break;
    }

    /*Pway個のiFileを一つのoFileに書き出す"を繰り返すループ*/
    k=0;
    while(1){
      /*各バケットの最初行をキューに入れる*/
      iFrom= k   *PWayU+iStart;     /* PWayU=3で一時ファイルが
                                          0,1,2,3,4,5,6の時*/
      iTo  =(k+1)*PWayU+iStart-1; /* iFrom: 0 3 6*/
      if(iTo>iEnd) iTo=iEnd;        /* iTo  : 2 5 6*/

      setFirstLineRBUQ(ud, iFrom,iTo);

      /*出力ファイルオープン*/
      oFile=mssOpenFPW(getFname(ud->prefixTxt,oEnd),0,0);

      /*各バケットから一行づつ読み込み書き出すループ*/
      while(1) {
        node=RBUQpop(ud->rb->left);          /*キューから一行取り出し*/
        if(strcmp(node->key->str,readEnd)==0)/*readEnd(番兵)を取り出したら終了*/
          break;
        RBUQwriteNode(node,oFile);  /*一行書き出し*/
        bktCnt=0;                   /*popで取り出したkey値のバケット数*/
        for(i=0; i<PWayU; i++){
          if(*(node->key->bkt+i) == 1)
          bkt[bktCnt++]=i;          /*バケット番号をセット*/
        } 
        RBUQdeleteNode(node);              /*取り出したノードの削除*/

        /*popしたバケットから新に読みこんでRBtreeに挿入する*/
        for(i=0; i<bktCnt; i++){
          readRBUQkey(ud->iFile[bkt[i]],ud->fr[bkt[i]]);
          RBUQinsert(ud->rb->left,ud->fr[bkt[i]],bkt[i]);
        }
      }

      RBUQfree(ud->rb);                    /*キューの開放*/
      for(i=0; i<=iTo-iFrom; i++){          /*入力ファイルのクローズ*/
        mssCloseFPR(ud->iFile[i]);
        mssFreeFldRec(ud->fr[i]);
      }
      mssCloseFPW(oFile);                     /*出力ファイルのクローズ*/
      if(iTo==iEnd)break;                  /*最後のバケットまでいけば終了*/
      oEnd++;                              /*出力ファイル番号カウントアップ*/
      k++;                                 /*併合回数カウントアップ*/
    }
    for(i=iStart; i<=iEnd; i++){
      unlink(getFname(ud->prefixTxt,i));      /*入力ワークファイルの削除*/
    }
  }
}

/**
 * # FUNCTION #
 * 集計しながらsortする関数
 * 集計しながら赤黒木を作成
 * メモリ消費がMAXを越えたら一時ファイルに書き出し
 */
static void sortUQ( struct mssUnqDat *ud, struct mssFPR *fpr)
{

  struct mssFldRec *fr;    /*入力ファイルをキー順に並べ換えるためのバッファ*/
  int    pid;              /*プロセスID(ファイル名の一部)*/
  int    oNum;             /*ワークファイル番号のカウントアップ変数*/
  int    memCnt;           /*メモリ消費量*/

  struct RBUQnode *rb=NULL; /*赤黒木*/
  struct mssFPW *fpw;      /*出力一時ファイル*/

  fr=mssInitFldRec(FldCnt);/*FldRecの初期化*/
  rb=RBUQinit(rb);          /*RBUQtreeの初期化*/

  /*ファイル名のプレフィックスの設定*/
  pid=getpid();
  if(strlen(ud->tmpDir) > MssFileNameMaxLen - 50 ) {
    mssShowErrMsg("length of path name must be less than %d",MssFileNameMaxLen-50);
    exit(mssErrorNoDefault);
  }
  sprintf(ud->prefixTxt,"%s/xt##%d-PreUnqTxt-",ud->tmpDir,pid);

  memCnt=0; /*消費メモリカウンタのクリア*/
  oNum=0;   /*出力ファイル番号*/
  while(1){
    /*データの読み込み*/
    mssReadFldRec(fpr,fr);
    (*ud->inCnt)++;

    /*赤黒木のサイズがMaxを越えたら もしくは 最終行を検出したら*/
    /*ワークファイルへ書き出す*/
    if( memCnt >= MaxMemU || fr->eof==1){
      fpw=mssOpenFPW(getFname(ud->prefixTxt,oNum),0,0);/*出力ファイルオープン*/
      RBUQwriteAllNode(rb->left, fpw);        /*RBtree全て出力*/
      mssCloseFPW(fpw);                          /*出力ファイルのクローズ*/
      oNum++;                                     /*出力ファイル数countUp*/
      RBUQfree(rb);   /*RBUQtreeの領域解放*/
      if(fr->eof==1){ /*最終行で終了*/
        (*ud->inCnt)--;
        break;
      }
      rb=RBUQinit(rb);    /*RBUQtreeの初期化*/
      memCnt=0;           /*消費メモリカウンタのクリア*/
    }

    /*赤黒木に挿入して(物理的にコピー)、メモリ消費をカウントアップ*/
    memCnt += RBUQinsert(rb->left,fr,1);
  }
  mssFreeFldRec(fr);/*FldRecの初期化*/

  ud->iStart = 0;
  ud->iEnd   = oNum-1;
}

/**
 * # FUNCTION #
 *必用となる項目関係のグローバル変数の設定
 *preUnqとrbTreeUQとのつなぎのための変数は全てここで
 */
struct mssUnqDat *mssInitUnqDat(int fldCnt, MssOptKEY *optKey, char *tmpDir, int *inCnt)
{
  struct mssUnqDat *unqDat;
  unqDat=mssMalloc(sizeof(struct mssUnqDat),"initUnqDat");
    
  unqDat->fldCnt  = fldCnt;
  unqDat->optKey  = optKey;
  unqDat->tmpDir  = tmpDir;
  unqDat->inCnt   = inCnt;
    
  /*グローバル変数*/
  FldCnt  = unqDat->fldCnt;
  OptKey  = unqDat->optKey;
  
  return(unqDat);
}

/**
 * # FUNCTION #
 * RBキューを用いた項目切り分け型行読み込み
 * 既に並べ変わっているバケット(ud->iStartからud->iEnd)を
 * 集計しながら併合していく
 * ファイルの個数は既にPWayU以下になっていることが前提
 */
int mssReadWriteUnq(struct mssUnqDat *ud, struct mssFPW *oFile)
{
  struct RBUQnode *node;      /*popにて取りだしたノードを格納*/
  int              bkt[PWayU];/*Pway併合時にRBtreeからpopしたデータのバケットNo*/
  int              bktCnt;    /*そのバケット数*/
  int i;

  node=RBUQpop(ud->rb->left);           /*キューから一行取り出し*/
  if(strcmp(node->key->str,readEnd)==0) /*readEnd(番兵)を取り出したら終了*/
    return(EOF);

  /*一行書き出し*/
  mssWriteFld(node->key->fld,FldCnt,"\n",oFile);

  bktCnt=0;                      /*popで取り出したkey値のバケット数*/
  for(i=0; i<PWayU; i++){
    if(*(node->key->bkt+i) == 1)
      bkt[bktCnt++]=i;           /*バケットフラグをセット*/
  } 
  RBUQdeleteNode(node);          /*取り出したノードの削除*/

  /*popしたバケットから新に読みこんでRBtreeに挿入する*/
  for(i=0; i<bktCnt; i++){
    readRBUQkey(ud->iFile[bkt[i]],ud->fr[bkt[i]]);
    RBUQinsert(ud->rb->left,ud->fr[bkt[i]],bkt[i]);
  }
  return(1); /*とりあえず*/
}

/**
 * # FUNCTION #
 * mssUnqDat構造体の開放。
 */
void mssFreeUnqDat(struct mssUnqDat *ud)
{
  int i;

  if(ud->procType==1){
    /*バッファ管理のために、入力ファイルはここで初めてクローズする*/
    for(i=0; i<=ud->iEnd-ud->iStart; i++){
      mssCloseFPR(ud->iFile[i]);
      mssFreeFldRec(ud->fr[i]);
    }

    /*ファイルの削除*/
    for(i=ud->iStart; i<=ud->iEnd; i++){
      unlink(getFname(ud->prefixTxt,i));
    }

    /*領域開放*/
    RBUQfree(ud->rb);
  }
  mssFree(ud);
}

/**
 * # FUNCTION #
 * 赤黒木を作りながら集計＋集計しながらPWayマージソート
 */
void mssPreUnq( struct mssUnqDat *ud, struct mssFPR *iFile)
{
  /*シグナルハンドラの設定*/
  mssGV.usedTempFileFlg=1;
  mssSetSignalHandler();

  sortUQ(ud,iFile);
  mergeRBUQ(ud);
  setFirstLineRBUQ(ud, ud->iStart,ud->iEnd);
}
