/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるMssValue型に関する関数
 *
 * Change log
 * 2004/02/14 : 数値演算範囲エラーのチェック追加(finite)
 * 2004/08/02 : mssVcmpをmssVcmpOpeに変更+mssVcmp関数追加
 * ============================================================================
 * NULLに対する演算は全てNULL
 */


#include <musashi/mssValue.h>
#include <musashi/mssBase.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/time.h>
#include <regex.h>
#include <math.h>
#include <errno.h>
#include <limits.h>
#include <float.h>


/**
 * # FUNCTION #
 * MssValueの初期化
 * MssValueを０クリアし、指定のデータタイプをセットする。
 */
void mssVinit(MssValue *v, enum vType vType)
{
  v->vType=vType;
  mssVclear(v);
} 

/**
 * # FUNCTION #
 * MssValueの０クリア
 */
void mssVclear(MssValue *v)
{
  v->nul=0;
  v->v.d=0; /*最もサイズの大きい型(double)で初期化*/
} 

/**
 * # FUNCTION #
 * MssValueの型に応じた最小値でのクリア
 * INT型:INT_MINに設定
 * DBL型:-DBL_MAXに設定
 * その他の型:NULLをセット
 */
void mssVclearMin(MssValue *v)
{
  v->nul=0;
  switch(v->vType){
    case INT: v->v.i= INT_MIN; break;
    case DBL: v->v.d=-DBL_MAX; break;
    default : MssVnull(*v);    break;
  }
}
  
/**
 * # FUNCTION #
 * MssValueの型に応じた最大値でのクリア
 * INT型:INT_MAXに設定
 * DBL型:DBL_MAXに設定
 * その他の型:NULLをセット
 */
void mssVclearMax(MssValue *v)
{
  v->nul=0;
  switch(v->vType){
    case INT: v->v.i= INT_MAX; break;
    case DBL: v->v.d= DBL_MAX; break;
    default : MssVnull(*v);    break;
  }
}
 
/**
 * # FUNCTION #
 * int型をMssValueのINT型に変換
 */
MssValue mssVint2i(int i)
{
  MssValue v;

  mssVinit(&v,INT);
  v.v.i=i;
  return(v);
}

/**
 * # FUNCTION #
 *int型をMssValueのDBL型に変換
 */
MssValue mssVint2d(int i)
{
  MssValue v;

  mssVinit(&v,DBL);
  v.v.d=(double)i;
  return(v);
}

/**
 * # FUNCTION #
 * int型をMssValueのSTR型に変換
 * MssValueのSTR型に新たに文字列領域を確保する。
 */
MssValue mssVint2s(int i)
{
  MssValue v;

  mssVinit(&v,STR);
  v.v.s=mssItoA(i);
  return(v);
}

/**
 * # FUNCTION #
 * double型をMssValueのINT型に変換
 */
MssValue mssVdbl2i(double d)
{
  MssValue v;

  mssVinit(&v,INT);
  v.v.i=(int)d;
  return(v);
}

/**
 * # FUNCTION #
 * double型をMssValueのDBL型に変換
 */
MssValue mssVdbl2d(double d)
{
  MssValue v;

  mssVinit(&v,DBL);
  v.v.d=d;
  return(v);
}

/**
 * # FUNCTION #
 * double型をMssValueのSTR型に変換
 * MssValueのSTR型に新たに文字列領域を確保する。
 */
MssValue mssVdbl2s(double d)
{
  MssValue v;

  mssVinit(&v,STR);
  v.v.s=mssFtoA(d);
  return(v);
}

/**
 * # FUNCTION #
 * 文字列型(char *)をMssValueのINT型に変換
 * 文字列の先頭が"*"(NULL値)の時は、MssValueもNULLにセットする。
 * 変換にはatoi関数を利用している。
 */
MssValue mssVstr2i(char *str)
{
  MssValue v;

  mssVinit(&v,INT);

  if(MssIsNull(str)){
    MssVnull(v);
  }else{
    v.v.i=atoi(str);
  }
  return(v);
}

/**
 * # FUNCTION #
 * 文字列型(char *)をMssValueのDBL型に変換
 * 文字列の先頭が"*"(NULL値)の時は、MssValueもNULLにセットする。
 * 変換にはatof関数を利用している。
 */
MssValue mssVstr2d(char *str)
{
  MssValue v;

  mssVinit(&v,DBL);

  if(MssIsNull(str)){
    MssVnull(v);
  }else{
    v.v.d=atof(str);
  }
  return(v);
}

/**
 * # FUNCTION #
 * 文字列型(char *)をMssValueのSTR型に変換
 * 文字列の先頭が"*"(NULL値)の時は、MssValueもNULLにセットする。
 * 新たに文字領域をコピーすることはしない。
 * すなわち、単に引数strのアドレスをセットするだけ。
 */
MssValue mssVstr2s(char *str)
{
  MssValue v;

  mssVinit(&v,STR);

  if(MssIsNull(str)){
    MssVnull(v);
  }else{
    v.v.s=str;
  }
  return(v);
}

/**
 * # FUNCTION #
 * 引数の型が合わない場合の、エラーメッセージ＆終了
 */
static void errorInvalidType(char *msg)
{
  mssShowErrMsg("internal error: invalid type of value is assigned: %s",msg);
  mssEnd(mssErrorNoDefault);
}

/**
 * # FUNCTION #
 * MssValueのINT型をMssValueのDBL型に変換
 */
MssValue mssVi2d(MssValue v)
{

  if(v.vType!=INT) errorInvalidType("mssVi2d");

  v.vType=DBL;

  if(!v.nul){
    v.v.d=(double)v.v.i;
  }
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのINT型をMssValueのSTR型に変換
 */
MssValue mssVi2s(MssValue v)
{

  if(v.vType!=INT) errorInvalidType("mssVi2s");

  v.vType=STR;

  if(!v.nul){
    v.v.s=mssItoA(v.v.i);
  }
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのDBL型をMssValueのINT型に変換
 */
MssValue mssVd2i(MssValue v)
{

  if(v.vType!=DBL) errorInvalidType("mssVd2i");

  v.vType=INT;

  if(!v.nul){
    v.v.i=(int)v.v.d;
  }
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのDBL型をMssValueのSTR型に変換
 * MssValueのSTR型に新たに文字列領域を確保する。
 */
MssValue mssVd2s(MssValue v)
{

  if(v.vType!=DBL) errorInvalidType("mssVd2s");

  v.vType=STR;

  if(!v.nul){
    v.v.s=mssFtoA(v.v.d);
  }
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのSTR型をMssValueのINT型に変換
 * 引数で与えられたMssValueのSTR型の文字列領域は開放しない。
 * 変換にはatoi関数を利用している。
 */
MssValue mssVs2i(MssValue v)
{

  if(v.vType!=STR) errorInvalidType("mssVs2i");

  v.vType=INT;

  if(!v.nul){
    v.v.i=atoi(v.v.s);
  }
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのSTR型をMssValueのDBL型に変換
 * 引数で与えられたMssValueのSTR型の文字列領域は開放しない。
 * 変換にはatof関数を利用している。
 */
MssValue mssVs2d(MssValue v)
{

  if(v.vType!=STR) errorInvalidType("mssVs2d");

  v.vType=DBL;

  if(!v.nul){
    v.v.d=atof(v.v.s);
  }
  return(v);
}

/**
 * # FUNCTION #
 * ２つの引数の型が異る場合の、エラーメッセージ＆終了
 */
static void errorDiffType(char *msg)
{
    mssShowErrMsg("internal error: operation with different types of values");
    mssEnd(mssErrorNoDefault);
}

/**
 * # FUNCTION #
 * MssValueの足し算
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * ２つのMssValue引数(v1,v2)の型が一致しなければエラーを出力する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVadd(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    v1.nul=0;
    switch(v1.vType){
      case INT: v1.v.i += v2.v.i; break;
      case DBL: v1.v.d += v2.v.d;
                if(finite(v1.v.d)==0) MssVnull(v1);
                break;
      default : errorInvalidType("mssVadd");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * MssValueの引き算
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * ２つのMssValue引数(v1,v2)の型が一致しなければエラーを出力する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVsub(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    v1.nul=0;
    switch(v1.vType){
      case INT: v1.v.i -= v2.v.i; break;
      case DBL: v1.v.d -= v2.v.d; break;
      default : errorInvalidType("mssVadd");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * MssValueのかけ算
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * ２つのMssValue引数(v1,v2)の型が一致しなければエラーを出力する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVmul(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    v1.nul=0;
    switch(v1.vType){
      case INT: v1.v.i *= v2.v.i; break;
      case DBL: v1.v.d *= v2.v.d;
                if(finite(v1.v.d)==0) MssVnull(v1);
                break;
      default : errorInvalidType("mssVadd");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * MssValueの割り算
 * MssValueのDBL型に対してのみ利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * ２つのMssValue引数(v1,v2)の型が一致しなければエラーを出力する。
 * 返す値の型は、引数の型に同じ(DBL)。
 * v2が０の場合はNULL値を返す。
 */
MssValue mssVdiv(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    v1.nul=0;
    switch(v1.vType){
      case DBL:
        if(v2.v.d==0) MssVnull(v1);
        else          v1.v.d /= v2.v.d;
        break;
      default : errorInvalidType("mssVadd");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * MssValueの剰余算
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * ２つのMssValue引数(v1,v2)の型が一致しなければエラーを出力する。
 * 返す値の型は、引数の型に同じ。
 * v2が０の場合はNULL値を返す。
 * INTの場合は"%"演算子を使い、DBLの場合はfmod関数を利用している。
 * ex. 5 % 2 = 1, 5.2 % 1.2 = 0.3
 */
MssValue mssVmod(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    v1.nul=0;
    switch(v1.vType){
      case INT:
        if(v2.v.i==0) MssVnull(v1);
        else          v1.v.i %= v2.v.i;
        break;
      case DBL:
        if(v2.v.d==0) MssVnull(v1);
        else          v1.v.d=fmod(v1.v.d,v2.v.d);
        break;
      default : errorInvalidType("mssVadd");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * MssValueの平方根
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、引数の型に関わらずDBL型。
 * v2が負の数の場合はNULL値を返す。
 */
MssValue mssVsqrt(MssValue v)
{

  if(!v.nul){
    switch(v.vType){
      case INT:
        if(v.v.i<0) MssVnull(v);
        else        v.v.d=sqrt((double)v.v.i);
        break;
      case DBL:
        if(v.v.d<0) MssVnull(v);
        else        v.v.d=sqrt(v.v.d);
        break;
      default : errorInvalidType("mssVsqrt");
    }
  }
  v.vType=DBL;
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのカウントアップ
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVcntUp(MssValue v)
{

  if(!v.nul){
    switch(v.vType){
      case INT: v.v.i++; break;
      case DBL: v.v.d++; break;
      default : errorInvalidType("mssVcntUp");
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * MssValueのカウントダウン
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVcntDown(MssValue v)
{

  if(!v.nul){
    switch(v.vType){
      case INT: v.v.i--; break;
      case DBL: v.v.d--; break;
      default : errorInvalidType("mssVcntUp");
    }
  }
  return(v);
}

/**
 * # FUNCTION #
 * ２つのMssValue引数v1,v2で小さい値を返す
 * ２つのMssValue引数が同じ場合はv1を返す
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVmin(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    switch(v1.vType){
    case DBL:
      if(v1.v.d>v2.v.d) v1=v2;
      break;
    case INT:
      if(v1.v.i>v2.v.i) v1=v2;
      break;
    default : errorInvalidType("mssVcntUp");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * ２つのMssValue引数v1,v2で大きい値を返す
 * ２つのMssValue引数が同じ場合はv1を返す
 * MssValueのINT型およびDBL型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、引数の型に同じ。
 */
MssValue mssVmax(MssValue v1, MssValue v2)
{

  if(v1.nul || v2.nul){
    MssVnull(v1);
  }else{
    if(v1.vType!=v2.vType) errorDiffType("mssVadd");

    switch(v1.vType){
    case DBL:
      if(v1.v.d<v2.v.d) v1=v2;
      break;
    case INT:
      if(v1.v.i<v2.v.i) v1=v2;
      break;
    default : errorInvalidType("mssVcntUp");
    }
  }
  return(v1);
}

/**
 * # FUNCTION #
 * ２つのMssValue引数a,bの比較演算
 * "a ope b"を評価し、真ならば1、偽なら０を返す。
 * a,bいずれかがNULLなら-1を返す。
 * MssValueのINT型,DBL型,STR型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、常にint型。
 */
int mssVcmpOpe(MssValue a, enum OpeType ope, MssValue b)
{
  double d=0;
  int  flag=0;

  if(a.nul || b.nul) return(-1);

  switch(a.vType){
    /*Double型 ------- -DBL0 0 +DBL0 ++++++++ */
    case DBL:
    d=a.v.d-b.v.d;
    switch(ope){
      case OPE_LT: if(d<-DBL0)           flag=1; break;
      case OPE_LE: if(d< DBL0)           flag=1; break;
      case OPE_GT: if(d> DBL0)           flag=1; break;
      case OPE_GE: if(d>-DBL0)           flag=1; break;
      case OPE_EQ: if(d>=-DBL0&&d<=DBL0) flag=1; break;
      case OPE_NE: if(d< -DBL0||d >DBL0) flag=1; break;
    }
    break;

    /*INT型*/
    case INT:
    switch(ope){
      case OPE_LT: if(a.v.i <b.v.i) flag=1; break;
      case OPE_LE: if(a.v.i<=b.v.i) flag=1; break;
      case OPE_GT: if(a.v.i >b.v.i) flag=1; break;
      case OPE_GE: if(a.v.i>=b.v.i) flag=1; break;
      case OPE_EQ: if(a.v.i==b.v.i) flag=1; break;
      case OPE_NE: if(a.v.i!=b.v.i) flag=1; break;
    }
    break;

    /*STR型*/
    case STR:
    switch(ope){
      case OPE_LT: if(0> strcmp(a.v.s,b.v.s)) flag=1; break;
      case OPE_LE: if(0>=strcmp(a.v.s,b.v.s)) flag=1; break;
      case OPE_GT: if(0< strcmp(a.v.s,b.v.s)) flag=1; break;
      case OPE_GE: if(0<=strcmp(a.v.s,b.v.s)) flag=1; break;
      case OPE_EQ: if(0==strcmp(a.v.s,b.v.s)) flag=1; break;
      case OPE_NE: if(0!=strcmp(a.v.s,b.v.s)) flag=1; break;
    }
    break;

    default : errorInvalidType("mssVcmpOpe");
    break;
  }
  return(flag);
} 


/**
 * # FUNCTION #
 * ２つのMssValue引数a,bの比較演算
 * "a > b"なら+1、a<bなら-1、a=bなら0を返す。
 * a,bいずれかがNULLなら9を返す。
 * MssValueのINT型,DBL型,STR型に対して利用可能で、その他の型が指定されればエラー
 * メッセージを表示し、終了する。
 * 返す値の型は、常にint型。
 */
int mssVcmp(MssValue a, MssValue b)
{
  double d=0;
  int  flag=0;

  if(a.nul || b.nul) return(9);

  switch(a.vType){
    /*Double型 ------- -DBL0 0 +DBL0 ++++++++ */
    case DBL:
    d=a.v.d-b.v.d;
         if(d<-DBL0) flag=-1;
    else if(d> DBL0) flag=1;
    else             flag=0;
    break;

    /*INT型*/
    case INT:
         if(a.v.i<b.v.i) flag=-1;
    else if(a.v.i>b.v.i) flag=1;
    else                 flag=0;
    break;

    /*STR型*/
    case STR:
    flag=strcmp(a.v.s,b.v.s);
         if(flag<0) flag=-1;
    else if(flag>0) flag=1;
    else            flag=0;
    break;

    /*ADD型*/
    case ADD:
         if(a.v.a<b.v.a) flag=-1;
    else if(a.v.a>b.v.a) flag=1;
    else                 flag=0;
    break;

    default : errorInvalidType("mssVcmp");
    break;
  }
  return(flag);
} 

/**
 * # FUNCTION #
 * MssValueの出力
 * MssValueの型を時動的に判断し、適切に出力する。
 */
void mssVwrite(MssValue v, struct mssFPW *fp)
{
  if(v.nul){
    mssWriteNull(fp);
  }else{
    switch(v.vType){
      case INT: mssWriteInt(v.v.i,fp); break;
      case DBL: mssWriteDbl(v.v.d,fp); break;
      case STR: mssWriteStr(v.v.s,fp); break;
      case ADD: mssWriteInt((int)v.v.a,fp); break;
      case USI: mssWriteInt(v.v.usi,fp); break;
    }
  }
}

/**
 * # FUNCTION #
 * MssValueのINT型の出力
 */
void mssVwriteInt(MssValue v, struct mssFPW *fp)
{

  if(v.nul){
    mssWriteNull(fp);
  }else{
    mssWriteInt(v.v.i,fp);
  }
}

/**
 * # FUNCTION #
 * MssValueのDBL型の出力
 */
void mssVwriteDbl(MssValue v, struct mssFPW *fp)
{

  if(v.nul){
    mssWriteNull(fp);
  }else{
    mssWriteDbl(v.v.d,fp);
  }
}

/**
 * # FUNCTION #
 * MssValueのSTR型の出力
 */
void mssVwriteStr(MssValue v, struct mssFPW *fp)
{

  if(v.nul){
    mssWriteNull(fp);
  }else{
    mssWriteStr(v.v.s,fp);
  }
}

