/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるXMLデータの入力関連の関数
 * iconv関数の第二引数を(const char **)にプロトタイプ変換 2003/12/24
 * mssAddXmlTagAttribute* 関連の関数で属性名がNULLの時なにもせずに終了する
 *   ように変更 2004/08/15
 * メモリリーク修正 2004/10/31
 * ============================================================================
 */

#include <musashi/mssXml.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/time.h>
#include <libxml/parser.h>
#include <libxml/parserInternals.h>
#include <libxml/encoding.h>
#include <iconv.h>
#include <errno.h> 
#include <glob.h>
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

/*############################################################################*/
/* グローバル変数                                                             */
/*############################################################################*/
xmlParserCtxtPtr ctxt;
char mssXmlDomErrMsg[1024];

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * 文字列のエンコーディング変換
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * 文字列strをencodingする。encoding方法はグローバル変数icidで決まる。
 * 返値 : 変換後の文字列ポインタを返す。
 *        strがNULLならNULLを返す。
 *        encodingの方法が指定されていなければstrを返す。
 * icidの設定方法
 * include <iconv.h>
 * icid=iconv_open("UTF-8","EUC-JP"); EUC-JPからUTF-8への変換
 */
char *mssEncoding(char *str, iconv_t *icid)
{
  size_t  encilen; /*iconv用 入力文字列長*/
  size_t  encolen; /*iconv用 出力文字列長*/
#ifdef MssNeedLibiconv
  const char *encip;     /*iconv用 入力文字列を示すポインタ */
#else
  char *encip;     /*iconv用 入力文字列を示すポインタ */
#endif
  char *encop;     /*iconv用 出力文字列を示すポインタ */
  char *outStr;    /*出力文字列の辺値用ポインタ*/
  int  encMax;

  if(str==NULL)return(NULL);
  if(icid==NULL)return(str);
  if(*str=='\0')return(str);

  encMax=strlen(str)*4;

  /*一応、入力文字列の４倍の領域確保*/
  outStr=mssCalloc(sizeof(char)*encMax,"encoding");
  encip=str;
  encop=outStr;
  encilen=strlen(encip)+1;
  encolen=encMax;
  if(-1==iconv(icid,&encip,&encilen,&encop,&encolen)){
    mssShowErrMsg("encoding error in iconv");
    exit(mssErrorNoDefault);
  }
  return(outStr);
}

/**
 * # FUNCTION #
 * mssEncoding関数の文字数指定バージョン
 */
char *mssNencoding(char *str, int len, iconv_t *icid)
{
  size_t  encilen; /*iconv用 入力文字列長*/
  size_t  encolen; /*iconv用 出力文字列長*/
#ifdef MssNeedLibiconv
  const char *encip;     /*iconv用 入力文字列を示すポインタ */
#else
  char *encip;     /*iconv用 入力文字列を示すポインタ */
#endif
  char *encop;     /*iconv用 出力文字列を示すポインタ */
  char *outStr;    /*出力文字列の辺値用ポインタ*/
  int  encMax;

  if(str==NULL)return(NULL);
  if(icid==NULL)return(str);

  encMax=strlen(str)*4;

  /*一応、入力文字列の４倍の領域確保*/
  outStr=mssCalloc(sizeof(char)*encMax,"encoding");
  encip=str;
  encop=outStr;
  encilen=len;
  encolen=encMax;
  if(-1==iconv(icid,&encip,&encilen,&encop,&encolen)){
    mssShowErrMsg("encoding error in iconv");
    exit(mssErrorNoDefault);
  }
  return(outStr);
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * XMLタグに関する「文字列←→mssXmlTag構造体」相互変換
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * mssXmlTag構造体の初期化。
 * 要素名(element)を登録する(メモリを確保して)。
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * 属性をセットする時はmssAddXmlTagAttribute{Str,Int,Dbl}を利用する。
 * elementがNULLの時はNULLを返す。
 * elementが'\0'の場合はエラー終了。
 */
struct mssXmlTag *mssInitXmlTag(char *element, iconv_t *icid)
{
  struct mssXmlTag *xmlTag;

  if(element==NULL){
    xmlTag=NULL;
  }else if(*element=='\0'){
    mssShowErrMsg("name of XML element is NULL");
    exit(mssErrorNoDefault);
  }else{
    xmlTag=mssCalloc(sizeof(struct mssXmlTag),"initXmlTag");
    if(icid==NULL){
      xmlTag->element=mssStrdup(element);
    }else{
      xmlTag->element=mssEncoding(element,icid);
    }
  }
  return(xmlTag);
}

/**
 * # FUNCTION #
 * パラメータで与えられた要素＆属性リストをセット、mssXmlTag構造体を返す。
 * str="element:attNam1=attVal1,attNam2=attVal2"
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * 要素名が'\0'の場合はエラー終了。
 */
struct mssXmlTag *mssInitXmlTagByOpt(char *str, iconv_t *icid)
{
  struct mssXmlTag *xmlTag;
  char *tmp;
  char *pos;
  int   attCnt;
  char **attNam;
  int   i;

  if(!mssIsValidStr(str)) return(NULL);
  
  /*元の文字列を保存*/
  tmp=mssStrdup(str);

  /* 要素-属性セパレータの発見*/
  if( NULL!=(pos=strchr(tmp,':')) ){
    *pos='\0';
  }

  /*要素の登録*/
  xmlTag=mssInitXmlTag(tmp,icid);

  /*属性の登録*/
  if( pos!=NULL ){
    pos++;

    attCnt=0;
    attNam=mssTokByChr(pos,',',&attCnt,0);
    for(i=0; i<attCnt; i++){
      if( NULL!=(pos=strchr(*(attNam+i),'=')) ){
        *pos='\0'; pos++;
        mssAddXmlTagAttributeStr(xmlTag,*(attNam+i), pos, icid);
      }else{
        mssAddXmlTagAttributeStr(xmlTag,*(attNam+i), "", icid);
      }
    }
    mssFree(attNam);
  }
  mssFree(tmp);
  return(xmlTag);
}

/**
 * # FUNCTION #
 * mssXmlTag構造体領域の開放。
 */
void mssFreeXmlTag(struct mssXmlTag *xmlTag)
{
  int i;

  if(xmlTag!=NULL){
    mssFree(xmlTag->element);
    for(i=0; i<xmlTag->attCnt; i++){
      mssFree(*(xmlTag->attributes+i));
    }
    mssFree(xmlTag->attributes);
    mssFree(xmlTag);
  }
}

/**
 * # FUNCTION #
 * mssXmlTag構造体に指定の属性名(name)があるかどうかを調べ、あれば、
 * その属性名に対する値(文字列:value)を更新する。なければ、属性名と値を追加する
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * xmlTag->attributesの偶数番要素に属性名を、奇数番要素に値が保持される。
 * xmlTagがNULLの場合は何もせずにリターンする。
 */
void mssAddXmlTagAttributeFmt(struct mssXmlTag *xmlTag, char *name, iconv_t *icid, char *fmt, ... )
{
  char *buf;
  char *val;
  char *nam;
  char tmp[1024];
  va_list args;

  int i;

  if(name==NULL || xmlTag==NULL){
    return;
  }

  if(fmt==NULL){
    mssShowErrMsg("internal error in addXmlAttribute");
    exit(mssErrorNoDefault);
  }

  /*属性名*/
  if(icid==NULL){
    nam=mssStrdup(name);
  }else{
    nam=mssEncoding(name,icid);
  }

  /*属性値*/
  va_start(args,fmt);
  vsprintf(tmp,fmt,args);
  va_end(args);

  buf=mssMalloc(sizeof(char)*(strlen(tmp)+3),"updXmlTag");
  strcpy(buf,"\"");
  strcat(buf,tmp);
  strcat(buf,"\"");
  if(icid==NULL){
    val=mssStrdup(buf);
  }else{
    val=mssEncoding(buf,icid);
  }
  mssFree(buf);
 
  /*既に登録されているか調べる*/
  for(i=0; i<xmlTag->attCnt; i=i+2){
    /*一致する属性があった!!*/
    if(strcmp(*(xmlTag->attributes+i),nam)==0){
      mssFree(*(xmlTag->attributes+i+1));
      *(xmlTag->attributes+i+1)=val;
      return;
    }
  }
  
  /*valueに一致する属性名がなかったので追加*/
  xmlTag->attCnt+=2;
  xmlTag->attributes=mssRealloc(xmlTag->attributes,
                                sizeof(char *)*xmlTag->attCnt,"addXmlAtt");
  *(xmlTag->attributes+xmlTag->attCnt-2)=nam;
  *(xmlTag->attributes+xmlTag->attCnt-1)=val;
}

/**
 * # FUNCTION #
 * mssXmlTag構造体に指定の属性名(name)があるかどうかを調べ、あれば、
 * その属性名に対する値(文字列:value)を更新する。なければ、属性名と値を追加する
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * xmlTag->attributesの偶数番要素に属性名を、奇数番要素に値が保持される。
 * xmlTagがNULLの場合は何もせずにリターンする。
 */
void mssAddXmlTagAttributeStr(struct mssXmlTag *xmlTag, char *name, char *value, iconv_t *icid)
{
  char *buf;
  char *val;
  char *nam;
  int i;

  if(xmlTag==NULL){
    return;
  }

  if(name==NULL || value==NULL){
    return;
  }

  /*属性名*/
  if(icid==NULL){
    nam=mssStrdup(name);
  }else{
    nam=mssEncoding(name,icid);
  }
  /*属性値*/
  buf=mssMalloc(sizeof(char)*(strlen(value)+3),"updXmlTag");
  strcpy(buf,"\"");
  strcat(buf,value);
  strcat(buf,"\"");
  if(icid==NULL){
    val=mssStrdup(buf);
  }else{
    val=mssEncoding(buf,icid);
  }
  mssFree(buf);
 
  /*既に登録されているか調べる*/
  for(i=0; i<xmlTag->attCnt; i=i+2){
    /*一致する属性があった!!*/
    if(strcmp(*(xmlTag->attributes+i),nam)==0){
      mssFree(*(xmlTag->attributes+i+1));
      *(xmlTag->attributes+i+1)=val;
      return;
    }
  }
  
  /*valueに一致する属性名がなかったので追加*/
  xmlTag->attCnt+=2;
  xmlTag->attributes=mssRealloc(xmlTag->attributes,
                                sizeof(char *)*xmlTag->attCnt,"addXmlAtt");
  *(xmlTag->attributes+xmlTag->attCnt-2)=nam;
  *(xmlTag->attributes+xmlTag->attCnt-1)=val;
}

/**
 * # FUNCTION #
 * mssXmlTag構造体に指定の属性名(name)があるかどうかを調べ、あれば、
 * その属性名に体する値(整数:value)を更新する。なければ、属性名と値を追加する
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * xmlTag->attributesの偶数番要素に属性名を、奇数番要素に値が保持される。
 * xmlTagがNULLの場合は何もせずにリターンする。
 */
void mssAddXmlTagAttributeInt(struct mssXmlTag *xmlTag, char *name, int value, iconv_t *icid)
{
  char buf[100];
  char *tmp;
  char *val;
  char *nam;
  int i;

  if(name==NULL || xmlTag==NULL){
    return;
  }

  /*属性名*/
  if(icid==NULL){
    nam=mssStrdup(name);
  }else{
    nam=mssEncoding(name,icid);
  }

  /*属性値*/
  strcpy(buf,"\"");
  tmp=mssItoA(value);
  strcat(buf,tmp); mssFree(tmp);
  strcat(buf,"\"");
  if(icid==NULL){
    val=mssStrdup(buf);
  }else{
    val=mssEncoding(buf,icid);
  }
 
  /*既に登録されているか調べる*/
  for(i=0; i<xmlTag->attCnt; i=i+2){
    /*一致する属性があった!!*/
    if(strcmp(*(xmlTag->attributes+i),nam)==0){
      mssFree(*(xmlTag->attributes+i+1));
      *(xmlTag->attributes+i+1)=val;
      return;
    }
  }

  /*valueに一致する属性名がなかったので追加*/
  xmlTag->attCnt+=2;
  xmlTag->attributes=mssRealloc(xmlTag->attributes,
                                sizeof(char *)*xmlTag->attCnt,"addXmlAtt");
  *(xmlTag->attributes+xmlTag->attCnt-2)=nam;
  *(xmlTag->attributes+xmlTag->attCnt-1)=val;
}

/**
 * # FUNCTION #
 * mssXmlTag構造体に指定の属性名(name)があるかどうかを調べ、あれば、
 * その属性名に体する値(実数:value)を更新する。なければ、属性名と値を追加する
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * xmlTag->attributesの偶数番要素に属性名を、奇数番要素に値が保持される。
 * xmlTagがNULLの場合は何もせずにリターンする。
 */
void mssAddXmlTagAttributeDbl(struct mssXmlTag *xmlTag, char *name, double value, iconv_t *icid)
{
  char buf[256];
  char *tmp;
  char *val;
  char *nam;
  int i;

  if(name==NULL || xmlTag==NULL){
    return;
  }

  /*属性名*/
  if(icid==NULL){
    nam=mssStrdup(name);
  }else{
    nam=mssEncoding(name,icid);
  }

  /*属性値*/
  strcpy(buf,"\"");
  tmp=mssFtoA(value);
  strcat(buf,tmp); mssFree(tmp);
  strcat(buf,"\"");
  if(icid==NULL){
    val=mssStrdup(buf);
  }else{
    val=mssEncoding(buf,icid);
  }
 
  /*既に登録されているか調べる*/
  for(i=0; i<xmlTag->attCnt; i=i+2){
    /*一致する属性があった!!*/
    if(strcmp(*(xmlTag->attributes+i),nam)==0){
      mssFree(*(xmlTag->attributes+i+1));
      *(xmlTag->attributes+i+1)=val;
      return;
    }
  }

  /*valueに一致する属性名がなかったので追加*/
  xmlTag->attCnt+=2;
  xmlTag->attributes=mssRealloc(xmlTag->attributes,
                                sizeof(char *)*xmlTag->attCnt,"addXmlAtt");
  *(xmlTag->attributes+xmlTag->attCnt-2)=nam;
  *(xmlTag->attributes+xmlTag->attCnt-1)=val;
}

/**
 * # FUNCTION #
 * オプションで指定される要素-属性の指定書式をmssXmlTag構造体にセットし、
 * そのポインタを返す。
 * 与えられた文字列(str)がNULLの場合はNULLを返す。
 * 与えられた文字列(str)には一切変更を加えず、要素名、属性名、属性値
 * は新しい領域にセットされる。
 *
 * ex)
 * str="title:date=2002/01/01,time=10:34:58"
 * ->
 * xmlTag->element="title"
 * xmlTag->attributes={"date","2002/01/01","time","10:34:58"}
 * xmlTag->attCnt=4
 */
struct mssXmlTag *mssOpt2XmlTag(char *str, iconv_t *icid)
{
  char *pos;
  int   attCnt;
  char **atts;
  char *tmpStr;
  int   i;
  struct mssXmlTag *xmlTag; 
  
  if(!mssIsValidStr(str)) return(NULL);

  tmpStr=mssStrdup(str);
  pos=strchr(tmpStr,':');

  /*要素名だけの場合*/
  if(pos==NULL){
    xmlTag=mssInitXmlTag(tmpStr,icid);
    return(xmlTag);
  }

  /*属性名、属性値のセット*/
  *pos++='\0';
  xmlTag=mssInitXmlTag(tmpStr,icid);
  atts=mssTokByChr(pos,',',&attCnt,0);
  for(i=0; i<attCnt; i++){
    pos=strchr(*(atts+i),'=');
    if(pos==NULL){
      mssShowErrMsg("invalid attribute specified");
      exit(mssErrorNoDefault);
    }
    *pos++='\0';
    mssAddXmlTagAttributeStr(xmlTag,*(atts+i), pos,icid);
  }

  mssFree(atts);
  mssFree(tmpStr);
  return(xmlTag);
}   

/**
 * # FUNCTION #
 * mssXmlTag構造体から文字列としての空タグを作成し、そのポインタを返す。
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * 属性の数(xmlTag->attCnt)が奇数個の場合は、属性の最後の配列要素は無視される。
 *
 * ex.)
 * xmlTag->element="title"
 * xmlTag->attributes="date","2002/01/01","time","10:10:45"
 * xmlTag->attCnt=4
 *   ->
 * <title date="2002/01/01" time="10:10:45"/>
 */
char *mssXmlTag2emptyTag(struct mssXmlTag *xmlTag, iconv_t *icid)
{
  char buf[4096];
  char *str;
  int bc;
  int i;

  buf[0]='<';

  str=xmlTag->element;
  bc=1;
  while(*str!='\0'){
    if(bc>=4093){
      mssShowErrMsg("XML tag is too long");
      exit(mssErrorNoDefault);
    }
    buf[bc++]=*str++;
  }

  for(i=0; i<xmlTag->attCnt/2; i++){
    buf[bc++]=' ';
    str=*(xmlTag->attributes+i*2);
    while(*str!='\0'){
      if(bc>=4093){
        mssShowErrMsg("XML tag is too long");
        exit(mssErrorNoDefault);
      }
      buf[bc++]=*str++;
    }
    buf[bc++]='=';
    str=*(xmlTag->attributes+i*2+1);
    while(*str!='\0'){
      if(bc>=4093){
        mssShowErrMsg("XML tag is too long");
        exit(mssErrorNoDefault);
      }
      buf[bc++]=*str++;
    }
  }

  buf[bc++]='/';
  buf[bc++]='>';
  buf[bc++]='\0';

  if(icid!=NULL){
    return(mssEncoding(buf,icid));
  }else{
    return(mssStrdup(buf));
  }
}

/**
 * # FUNCTION #
 * mssXmlTag構造体から文字列としての開始タグを作成し、そのポインタを返す。
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 * 属性の数(xmlTag->attCnt)が奇数個の場合は、属性の最後の配列要素は無視される。
 *
 * ex.)
 * xmlTag->element="title"
 * xmlTag->attributes="date","2002/01/01","time","10:10:45"
 * xmlTag->attCnt=4
 *   ->
 * <title date="2002/01/01" time="10:10:45">
 */
char *mssXmlTag2startTag(struct mssXmlTag *xmlTag, iconv_t *icid)
{
  char buf[4096];
  char *str;
  int bc;
  int i;

  buf[0]='<';

  str=xmlTag->element;
  bc=1;
  while(*str!='\0'){
    if(bc>=4093){
      mssShowErrMsg("XML tag is too long");
      exit(mssErrorNoDefault);
    }
    buf[bc++]=*str++;
  }

  for(i=0; i<xmlTag->attCnt/2; i++){
    buf[bc++]=' ';
    str=*(xmlTag->attributes+i*2);
    while(*str!='\0'){
      if(bc>=4093){
        mssShowErrMsg("XML tag is too long");
        exit(mssErrorNoDefault);
      }
      buf[bc++]=*str++;
    }
    buf[bc++]='=';
    str=*(xmlTag->attributes+i*2+1);
    while(*str!='\0'){
      if(bc>=4093){
        mssShowErrMsg("XML tag is too long");
        exit(mssErrorNoDefault);
      }
      buf[bc++]=*str++;
    }
  }

  buf[bc++]='>';
  buf[bc++]='\0';

  if(icid!=NULL){
    return(mssEncoding(buf,icid));
  }else{
    return(mssStrdup(buf));
  }
}

/**
 * # FUNCTION #
 * mssXmlTag構造体から文字列としての終了タグを作成し、そのポインタを返す。
 * icidがセットされていれば(NULLでなければ)エンコーディングの変換をおこなう。
 *
 * ex.)
 * xmlTag->element="title"
 * xmlTag->attributes="date","2002/01/01","time","10:10:45"
 * xmlTag->attCnt=4
 *   ->
 * </title>
 */
char *mssXmlTag2endTag(struct mssXmlTag *xmlTag, iconv_t *icid)
{
  char buf[4096];
  char *str;
  int bc;

  buf[0]='<';
  buf[1]='/';

  str=xmlTag->element;
  bc=2;
  while(*str!='\0'){
    if(bc>=4093){
      mssShowErrMsg("XML tag is too long");
      exit(mssErrorNoDefault);
    }
    buf[bc++]=*str++;
  }

  buf[bc++]='>';
  buf[bc++]='\0';

  if(icid!=NULL){
    return(mssEncoding(buf,icid));
  }else{
    return(mssStrdup(buf));
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * XMLの出力関連
 * ----------------------------------------------------------------------------
 */

/**
 * # FUNCTION #
 * XML宣言の出力。
 * バージョンとエンコーディングは、それぞれver,encで指定する。
 *
 * 例)
 * <?XML version="1.0" encoding="EUC-JP"?>
 */
void mssWriteXmlDeclaration( char *ver, char *enc, struct mssFPW *fp )
{
  
  mssWriteStr("<?xml",fp);
  if(ver!=NULL){
    mssWriteStr(" version=\"", fp);
    mssWriteStr(ver, fp);
    mssWriteStr("\"", fp);
  }
  if(enc!=NULL){
    mssWriteStr(" encoding=\"", fp);
    mssWriteStr(enc, fp);
    mssWriteStr("\"",fp);
  }
  mssWriteStr("?>\n",fp);
}

/**
 * # FUNCTION #
 * mssXmlTag構造体の情報を元に、開始タグを出力する。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 */
void mssWriteXmlStartTag(struct mssXmlTag *xmlTag, iconv_t *icid, struct mssFPW *fpw)
{
  char *str;
  str=mssXmlTag2startTag(xmlTag, icid);
  mssWriteStr(str,fpw);
  mssFree(str);
}

/**
 * # FUNCTION #
 * mssXmlTag構造体の情報を元に、終了タグを出力する。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 */
void mssWriteXmlEndTag(struct mssXmlTag *xmlTag, iconv_t *icid, struct mssFPW *fpw)
{
  char *str;
  str=mssXmlTag2endTag(xmlTag, icid);
  mssWriteStr(str,fpw);
  mssFree(str);
}

/**
 * # FUNCTION #
 * mssXmlTag構造体の情報を元に、空タグを出力する。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 */
void mssWriteXmlEmptyTag(struct mssXmlTag *xmlTag, iconv_t *icid, struct mssFPW *fpw)
{
  char *str;
  str=mssXmlTag2emptyTag(xmlTag, icid);
  mssWriteStr(str,fpw);
  mssFree(str);
}

/**
 * # FUNCTION #
 * 文字列(str)を出力する。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 */
void mssWriteXmlContent( char *str, iconv_t *icid, struct mssFPW *fp )
{
  char *encStr;

  if(str==NULL) return;

  if(icid!=NULL){
    encStr=mssEncoding(str,icid);
    mssWriteStr(encStr,fp);
    mssFree(encStr);
  }else{
    mssWriteStr(str,fp);
  }
}

/**
 * # FUNCTION #
 * XMLのインデントを出力する。
 * インデントの単位は半角空白２文字で、cnt個の空白を出力する。
 */
void mssWriteXmlIndent(int cnt, struct mssFPW *fpw)
{
  char spc[1025];
  int i;
  if(cnt>512) cnt=512;
  for(i=0; i<cnt; i++){
    spc[i*2]=' ';
    spc[i*2+1]=' ';
  }
  spc[i*2]='\0';
  mssWriteStr(spc, fpw);
}

/**
 * # FUNCTION #
 * 開始タグ、文字列、終了タグをセットで出力する。
 * 属性を含む開始タグは扱えない。
 * またインデント、改行の有無も指定できる。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 * ex)
 * indCnt=2, element="title", content="this is title" ,retFlg=1
 * ->
 *     <title>this is title</title>
 */
void mssWriteXmlTagStr(int indCnt, char *element, char *content, int retFlg, iconv_t *icid, struct mssFPW *fp )
{
  struct mssXmlTag *xmlTag;

  if(content==NULL) return;

  mssWriteXmlIndent(indCnt,fp);

  xmlTag=mssInitXmlTag(element,icid);
  mssWriteXmlStartTag(xmlTag,icid,fp);
  mssWriteXmlContent(content, icid, fp);
  mssWriteXmlEndTag(xmlTag, icid,fp);
  mssFreeXmlTag(xmlTag);

  if(retFlg){
    mssWriteRet(fp);
  }
}

/**
 * # FUNCTION #
 * 開始タグ、整数、終了タグをセットで出力する。
 * 属性を含む開始タグは扱えない。
 * またインデント、改行の有無も指定できる。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 * ex)
 * indCnt=2, element="title", content=10 ,retFlg=1
 * ->
 *     <title>10</title>
 */
void mssWriteXmlTagInt(int indCnt, char *element, int content, int retFlg, iconv_t *icid, struct mssFPW *fp )
{
  char *buf;
  struct mssXmlTag *xmlTag;

  buf=mssItoA(content);
  mssWriteXmlIndent(indCnt,fp);

  xmlTag=mssInitXmlTag(element,icid);
  mssWriteXmlStartTag(xmlTag,icid,fp);
  mssWriteXmlContent(buf, icid, fp);
  mssWriteXmlEndTag(xmlTag, icid,fp);
  mssFreeXmlTag(xmlTag);

  if(retFlg){
    mssWriteRet(fp);
  }
  mssFree(buf); /* 追加 2004/10/31 */
}

/**
 * # FUNCTION #
 * 開始タグ、実数、終了タグをセットで出力する。
 * 属性を含む開始タグは扱えない。
 * またインデント、改行の有無も指定できる。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 * ex)
 * indCnt=2, element="title", content=5.24 ,retFlg=1
 * ->
 *     <title>5.24</title>
 */
void mssWriteXmlTagDbl(int indCnt, char *element, double content, int retFlg, iconv_t *icid, struct mssFPW *fp )
{
  char *buf;
  struct mssXmlTag *xmlTag;

  buf=mssFtoA(content);
  mssWriteXmlIndent(indCnt,fp);

  xmlTag=mssInitXmlTag(element,icid);
  mssWriteXmlStartTag(xmlTag,icid,fp);
  mssWriteXmlContent(buf, icid, fp);
  mssWriteXmlEndTag(xmlTag, icid,fp);
  mssFreeXmlTag(xmlTag);

  if(retFlg){
    mssWriteRet(fp);
  }
  mssFree(buf); /* 追加 2004/10/31 */
}

/**
 * # FUNCTION #
 * flgが１ならば空タグを出力する。
 * 属性を含む空タグは扱えない。
 * またインデント、改行の有無も指定できる。
 * icidがNULLでなければエンコーディング変換を行って出力する。
 * ex)
 * indCnt=2, element="title", flg=1 ,retFlg=1
 * ->
 *     <title/>
 */
void mssWriteXmlTagFlg(int indCnt, char *element, int flg, int retFlg, iconv_t *icid, struct mssFPW *fp )
{
  struct mssXmlTag *xmlTag;

  if(flg){

    mssWriteXmlIndent(indCnt,fp);

    xmlTag=mssInitXmlTag(element,icid);
    mssWriteXmlEmptyTag(xmlTag,icid,fp);
    mssFreeXmlTag(xmlTag);

    if(retFlg){
      mssWriteRet(fp);
    }
  }
}

/**
 * # SECTION #
 * ----------------------------------------------------------------------------
 * エラーハンドラ
 * ----------------------------------------------------------------------------
 */

/**
 * # GLOBAL #
 * DOMのエラーハンドラー。
 * エラーメッセージがmssXmlDomErrMsg[]に蓄積される。
 * main関数で、mssXmlDomErrMsg[0]='\0'として初期化しておくこと。
 */
void mssXmlDomErrHandler(void *ctxt, const char *msg, ...){

  char bufMsg[1024];

  /*エラーメッセージをbufMsgに格納しencoding*/
  va_list args;
  va_start(args,msg);
  vsprintf(bufMsg,msg,args);

  /*長さチェック*/
  if(strlen(bufMsg)+strlen(mssXmlDomErrMsg) >= 1023){
    return;
  }

  /*上記メッセージを連結*/
  strcat(mssXmlDomErrMsg,bufMsg);
}

