/**
 * # CHAPTER #
 * ============================================================================
 * MUSASHIで用いられるXMLtableのタグを操作する関連の関数
 * ============================================================================
 * 通常の流れ
 * 1. mssReadTagでファイルから読み込み特定のタグをStringsにセットする
 * 2. mssGetTagで必要なタグをStringsにセットする
 * 3. mssGetTagCont,mssGetTagAtt,mssGetNulTagなどで内容を取り出す
 * 4. 2で確保したStringsを開放する
 * 5. 2,3,4の処理を繰り返す
 * 6. 1で確保したStringsを開放する
 *
 * mssGetTagCont,mssGetTagAtt,mssGetNulTagはStringsの文字列単にを検索し、最初に 
 * みつけた内容を返すので、Stringsの中にタグが複数ある場合はmssGetTagで
 * あらかじめ必要なタグをStringsにセットしておく
 *
 * 20040204 MssBeginBodyTagの検査をstrncmpからstrstrに変更
 */

#include <musashi/mssInput.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* ############ グローバル変数 ##############*/
extern struct mssGlobalVariables mssGV;


/** 
 * # FUNCTION # 
 * 3つ以内の文字列から新しい一つの文字列を作成し、そのポインタを返す。
 * その際、新たな文字列領域が確保される。
 * Tagの作成などに利用
 * ex.) bgn="<", str="field", end=">"
 *      ->
 *      "<field>"
 */     
static char *mkStr(char *bgn, char *str, char *end){
  char *tag;
  
  tag=mssMalloc(sizeof(char)*(strlen(bgn)+strlen(str)+strlen(end)+1),"mkStr");
  strcpy(tag,bgn);
  strcat(tag,str);
  strcat(tag,end);
  return(tag);
}

/** 
 * # FUNCTION # 
 * ファイル(fp)から指定タグ(tag)を検索しStringsにセットし、そのポインタを返す。
 * 開始タグが無ければNULLを返す。
 * 開始タグがあり、終了タグが無ければエラー表示＆終了。
 * ex.)
 * ファイルが"<xmltable><header>....</header></xmltable>"とすると、
 * 返値strings->str="<header><title>this is title</title></header>"
 */
struct mssStrings *mssReadTag(char *tag, struct mssFPR *fp){
    
  struct mssRec *rec;
  int   len;
  char *bgnTag;
  char *endTag;
  char *tagTmp=NULL;
  struct mssStrings *strings=NULL;

  rec=mssInitRec();

  /*beginタグとendタグをそれぞれセットする*/
  bgnTag=mkStr("<" ,tag,"" );
  endTag=mkStr("</",tag,">");

  /*beginタグの検索*/
  while( EOF != (len=mssReadRec(fp, rec)) ){
    if( NULL != (tagTmp=strstr(rec->pnt, bgnTag)) ){
      strings=mssInitStrings();
      mssCatStrings(strings, tagTmp);
      mssCatStrings(strings, "\n");
      break;
    }
  }

  /*beginタグがなかった*/
  if(len==EOF){
    mssFree(bgnTag); mssFree(endTag);
    return(NULL);
  }

  /*endタグの検索*/
  while( EOF != (len=mssReadRec(fp, rec)) ){
    if( NULL != (tagTmp=strstr(rec->pnt, endTag)) ){
      mssCatnStrings(strings, rec->pnt, rec->pnt-tagTmp+strlen(endTag));
      mssCatStrings(strings, "\n");
      break;
    }else{
      mssCatStrings(strings, rec->pnt);
      mssCatStrings(strings, "\n");
    }
  }

  /*endタグがなかった*/
  if(len==EOF){
    mssShowErrMsg("there is not the end tag : %s",tag);
    mssEnd(mssErrorNoDefault);
  }

  mssFreeRec(rec);
  mssFree(bgnTag); mssFree(endTag);

  return(strings);
}

/** 
 * # FUNCTION #
 * 文字列(str)から指定タグ(tag)で囲まれた文字列をStringsにセットし、
 * そのポインタを返す。
 * 開始タグが無ければNULLを返す。
 * 開始タグがあり、終了タグが無ければエラー表示＆終了。
 * さらに*posに、検索された開始タグの先頭のポインタをセットする。
 * タグの階層は考慮していないので、同名タグが入れ子構造になっている場合は
 * うまく動作しない。
 * ex.)
 * *str="<header><title>this is title</title><header>"とすると、
 * 返値strings->str="<title>this is title</title>"
 * *posは、オリジナルの*strの"<title>...."の先頭位置がセットされる。
 */
struct mssStrings *mssGetTag(char *tag, char *str, char **pos){

  char *bgnTag;
  char *endTag;
  char *bgnTagTmp;
  char *endTagTmp;
  struct mssStrings *strings;

  /*beginタグとendタグをそれぞれセットする*/
  bgnTag=mkStr("<" ,tag,"" );
  endTag=mkStr("</",tag,">");

  /*beginタグの検索*/
  bgnTagTmp=strstr(str, bgnTag);
  if( bgnTagTmp == NULL ){
    mssFree(bgnTag); mssFree(endTag);
    return(NULL);
  }

  /*endタグの検索*/
  endTagTmp=strstr(bgnTagTmp, endTag);
  if( endTagTmp == NULL ){
    mssShowErrMsg("there is not the end tag : %s",tag);
    mssEnd(mssErrorNoDefault);
  }

  strings=mssInitStrings();
  mssCatnStrings(strings, bgnTagTmp, endTagTmp-bgnTagTmp+strlen(endTag));

  mssFree(bgnTag); mssFree(endTag);

  *pos=bgnTagTmp;
  return(strings);
}

/** 
 * # FUNCTION #
 * 文字列(str)から空要素タグを検索し、あれば1、なければ0を返す。
 * 空要素タグの例: <numeric/>
 */
int mssGetNullTag(char *str, char *tag){

  char *t;
  int   ret=0;

  /*タグを作る*/
  t=mkStr("<",tag,"/>");

  /*タグの検索*/
  if(NULL != strstr(str, t)) ret=1;

  mssFree(t);
  return(ret);
}

/** 
 * # FUNCTION #
 * 文字列(str)から指定タグ(tag)の内容を返す。
 * 指定タグがなければNULL値を返す。
 * 開始タグがあり、終了タグが無ければエラー表示＆終了。
 * strの内容は変えずに、新たにメモリを確保して、そのアドレスを返す。
 * retIgnoreが１ならば改行を無視する。
 * タグの階層は考慮していないので、同名タグが入れ子構造になっている場合は
 * うまく動作しない。
 * ex.)
 * str="<title>this is title</title>"
 * tag="title"
 * 返値="this is title"
 */
char *mssGetTagCont(char *str, char *tag, int retIgnore){

  char *bgnTag;
  char *endTag;
  char *bgnTagTmp=NULL;
  char *endTagTmp=NULL;
  char *retStr=NULL;
  char *retPnt;
  char *conPnt;
  char  tmp;

  if( str == NULL ) return(NULL);

  /*beginタグとendタグをそれぞれセットする*/
  bgnTag=mkStr("<" ,tag,"" );
  endTag=mkStr("</",tag,">");

  /*begin,endタグの検索*/
  bgnTagTmp=strstr(str, bgnTag);
  if( bgnTagTmp == NULL ){
    mssFree(bgnTag); mssFree(endTag);
    return(NULL);
  }

  endTagTmp=strstr(bgnTagTmp, endTag);
  if( endTagTmp == NULL ){
    mssShowErrMsg("there is not the end tag : %s",tag);
    mssEnd(mssErrorNoDefault);
  }

  while(*bgnTagTmp != '>') bgnTagTmp++;
  bgnTagTmp++;

  tmp=*endTagTmp;
  *endTagTmp='\0';
  if(retIgnore){
    retStr=mssMalloc(sizeof(char)*(strlen(bgnTagTmp)+1),"getTagCont");
    conPnt=bgnTagTmp;
    retPnt=retStr;
    while(conPnt!=endTagTmp){
      if(*conPnt!='\n'){
        *retPnt=*conPnt;
        retPnt++;
      }
      conPnt++;
    }
    *retPnt='\0'; 
  }else{
    retStr=mssStrdup(bgnTagTmp);
  }
  *endTagTmp=tmp;

  mssFree(bgnTag); mssFree(endTag);
  return(retStr);
}

/** 
 * # FUNCTION #
 * 文字列(str)から指定タグ(tag)の指定の属性(att)の値を返す。
 * 開始タグがあり、終了タグが無ければエラー表示＆終了。
 * strの内容は変えずに、新たにメモリを確保して、そのアドレスを返す。
 * 属性値は'"'で囲まれていることを前提。
 * ex.)
 * str="<field no="1">...</field>"
 * 返値="1"
 */
char *mssGetTagAtt(char *str, char *tag, char *att){

  char *bgnTag;
  char *bgnTagTmp;
  char *attStr;
  char *retStr=NULL;
  char *pos;
  char buf[256];
  int i;

  /*beginタグ、属性名をセットする*/
  bgnTag=mkStr("<",tag," " );
  attStr=mkStr("" ,att,"=");

  /*beginタグの検索*/
  bgnTagTmp=strstr(str, bgnTag);
  if( bgnTagTmp == NULL ){
    mssFree(bgnTag); mssFree(attStr);
    return(NULL);
  }

  /*一時的に">"を'\0'にする。*/
  pos=bgnTagTmp;
  while(*pos!='>' && *pos!='\0') pos++;
  if(*pos!='>'){
    mssShowErrMsg("not a complete tag : %s",tag);
    mssEnd(mssErrorNoDefault);
  }
  *pos='\0';

  /*属性の検索*/
  bgnTagTmp=strstr(bgnTagTmp, attStr);
  if( bgnTagTmp == NULL ){
    retStr=NULL;
  }else{

    /*属性値のセット*/
    while(*bgnTagTmp!='"' && *bgnTagTmp!='\0') bgnTagTmp++;
    if(*bgnTagTmp=='\0'){
      mssShowErrMsg("invalid attribute value : %s",att);
      mssEnd(mssErrorNoDefault);
    }
    bgnTagTmp++;
    for(i=0; *bgnTagTmp!='"' && *bgnTagTmp!='\0' && i<255; i++){
      buf[i]=*bgnTagTmp++;
    }
    if(*bgnTagTmp=='\0' || i>=255){
      mssShowErrMsg("invalid attribute value : %s",att);
      mssEnd(mssErrorNoDefault);
    }
    buf[i]='\0';
    retStr=mssStrdup(buf);
  }

  *pos='>';

  mssFree(bgnTag); mssFree(attStr);
  return(retStr);
}

/** 
 * # FUNCTION #
 * トップ行(xml宣言、<xmltbl>タグ)の読み込み
 * １行目はXML宣言とみなす。
 * xmltblタグが出現するまでデータをスキップする。
 * XMLバージョン、エンコーディングをセット。
 * xmlTableのバージョンをセット。
 */
void mssReadSetTop(struct mssHeader *header, struct mssFPR *fp){
  struct mssRec *rec;
  struct mssStrings *buf;
  int   len;
  char  *version;
  char  *pnt;

  rec=mssInitRec();
  buf=mssInitStrings();

  len=mssReadRec(fp, rec);
  /*mssReadRecでEOFということはデータなしということ*/
  if(len==EOF) {
    mssShowErrMsg("no input data");
    exit(mssErrorNoDefault);
  }

  if( 0!=strncmp(rec->pnt,"<?xml",5) ){
    mssShowErrMsg("not XML data");
    mssEnd(mssErrorNoDefault);
  }else{
    header->xmlver=mssGetTagAtt(rec->pnt, "?xml", "version");
    header->xmlenc=mssGetTagAtt(rec->pnt, "?xml", "encoding");
    len=mssReadRec(fp, rec);
    /*mssReadRecでEOFということはデータなしということ*/
    if(len==EOF) {
      mssShowErrMsg("no input data");
      exit(mssErrorNoDefault);
    }
  }

  /*xmltblタグが見つかるまで読み続ける*/
  while(1) {
    pnt=strstr(rec->pnt,"<xmltbl");
    if(pnt!=NULL)break;
    len=mssReadRec(fp, rec);
    if(len==EOF) {
      mssShowErrMsg("cannot find xmltbl tag");
      exit(mssErrorNoDefault);
    }
  }

  mssCatStrings(buf, pnt);

  version = mssGetTagAtt(buf->str, "xmltbl", "version");
  if(version!=NULL){
    header->version = (int)(atof(version)*10);
  }else{
    mssShowErrMsg("version of XMLtable is not specified");
    exit(mssErrorNoDefault);
  }
  mssFree(version); /*追加 2004/10/28 */
  mssFreeRec(rec);
  mssFreeStrings(buf);  /*20021126追加*/
}

/** 
 * # FUNCTION #
 * データ部までスキップする。
 * 具体的には"<body><![CDATA["の次の行にファイルポインタを移す
 * <body>タグは一行に書かれていることが前提
 */
void mssSkipToBody(struct mssFPR *fp){
  struct mssRec *rec=NULL;

  if(mssGV.txtFlg) return;

  rec=mssInitRec();

  while(EOF!=mssReadRec(fp, rec)){
/*    if( 0==strncmp(rec->pnt,MssBeginBodyString,15) ){*/
    if( NULL!=strstr(rec->pnt,MssBeginBodyString) ){
      mssFreeRec(rec);
      return;
    }
  }
  mssFreeRec(rec);
  mssShowErrMsg("can't find the start body tag");
  exit(mssErrorNoDefault);
}
