/*
 * Created on 2005/01/25
 *
 *
 * Copyright(c) 2005 Yoshimasa Matsumoto
 */
package netjfwatcher.engine.snmpmanager.process;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketTimeoutException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.logging.Logger;

import netjfwatcher.engine.alarm.AlarmMessageMake;
import netjfwatcher.engine.common.model.DataStringDisplay;
import netjfwatcher.snmp.messageformat.SNMPTLV;
import netjfwatcher.snmp.messageformat.SnmpBadValueException;
import netjfwatcher.snmp.messageformat.SnmpErrorStatusException;
import netjfwatcher.snmp.messageformat.SnmpUnmuchRequestIDException;
import netjfwatcher.snmp.messageformat.SnmpVarBindList;
import netjfwatcher.snmp.preference.SnmpBERCodec;
import netjfwatcher.snmp.preference.SnmpPreference;
import netjfwatcher.snmp.snmpobject.message.AbstractSnmpObject;
import netjfwatcher.snmp.snmpobject.message.SnmpMessage;
import netjfwatcher.snmp.snmpobject.message.SnmpObjectIdentifier;
import netjfwatcher.snmp.snmpobject.message.SnmpPDU;
import netjfwatcher.snmp.snmpobject.message.SnmpSequence;
import netjfwatcher.snmpmanager.SnmpManagerQueue;


/**
 * SNMPMf[^̉͂s\bhNXłB
 *
 * Mf[^SNMPbZ[WAPDUAvarListƏ͂܂B
 * ܂AMf[^̃G[Xe[^X`FbNAG[
 * ꍇɂ͓vJEgƂƂɗOX[܂B
 *
 * SNMPbZ[W
 * +----------+-----------+---------------+
 * | Version  | Community |   SNMP PDU    |
 * +----------+-----------+---------------+
 *                        |               |
 *                        |               |
 * SNMP PDU
 * +--------+--------------+-------------+------------+
 * |   ID   | Error Status | Erroe Index |   varList  |
 * +--------+--------------+-------------+------------+
 *                                       |            |
 *                                       |            |
 * SNMP varList
 * +-----+-------+-----+-------+---
 * | OID | Value | OID { Value |   .....
 * +-----+-------+-----*-------+---
 *
 * SNP Pair
 *
 * +-----+-------+
 * | OID | Value |
 * +-----+-------+
 *
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class SnmpManagerMessageReceive extends DataStringDisplay {
    /** Mobt@őTCY */
    public static final int SNMP_V3_RECEIVE_MAXSIZE = 8196;

    /* MO */
    private static Logger logger;

    /* AgentAhX */
    private InetAddress hostAddress;

    /* DatagramSocket */
    private DatagramSocket dSocket;

    /* SNMPbZ[W */
    private SnmpMessage receivedSNMPMessage = new SnmpMessage();

    /* SnmpVarBindList */
    private SnmpVarBindList retrievedVars = new SnmpVarBindList();

    /* Mf[^tH[}bg */
    private DateFormat simpleDateformat = new SimpleDateFormat("HH:mm:ss");

    /**
     * SNMPMf[^̉͂s\bhNXCX^X
     * ܂B
     */
    public SnmpManagerMessageReceive() {
        logger = Logger.getLogger(this.getClass().getName());
    }

    /**
     * SNMPMSocket̐ݒs܂B
     *
     * @param hostAddress AgentAhX
     * @param dSocket DatagramSocket
     */
    public void setSnmpReceiveSocket(
        InetAddress hostAddress, DatagramSocket dSocket) {
        this.dSocket = dSocket;
        this.hostAddress = hostAddress;
    }

    /**
     * G[WFg̃f[^M҂󂯂܂B
     * f[^MSnmpbZ[W͂Snmp VarBindListƂĕԂ܂B
     *
     * @param requestID NGXgID
     * @param requestOID OID
     * @param dSocket DatagramSocket
     * @return retrievedVars SnmpVarBindList
     * @throws SnmpUnmuchRequestIDException NGXgIDҒlƈvȂꍇ
     * @throws SnmpErrorStatusException G[Xe[^XłȂꍇ
     * @throws SnmpBadValueException f[^͂ňُ킪ꍇ
     * @throws SocketTimeoutException gC񐔈ȓSocketTimeoutꍇ
     * @throws IOException f[^Msꍇ
     */
    public SnmpVarBindList parseReceiveData(
        int requestID, String requestOID, DatagramSocket dSocket)
        throws SnmpUnmuchRequestIDException, SnmpErrorStatusException, 
            SnmpBadValueException, SocketTimeoutException, IOException {
        // logger.info("parseReceiveData requestID : " + requestID);
        // logger.info("parseReceiveData requestOID : " + requestOID);
        byte[] receiveDataByte = null;

        // logger.info("Rceive wait OID=" + requestOID);
        DatagramPacket inPacket =
            new DatagramPacket(
                new byte[SNMP_V3_RECEIVE_MAXSIZE], SNMP_V3_RECEIVE_MAXSIZE);
        dSocket.receive(inPacket);

        receiveDataByte = inPacket.getData();
        logger.fine(
            "Receive data : "
            + hexBytetoString(receiveDataByte, inPacket.getLength()));

        Date date = Calendar.getInstance().getTime();
        SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
            "Receive date : " + simpleDateformat.format(date)
            + " : Data length : " + inPacket.getLength() + "Byte");
        SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
            hexBytetoString(receiveDataByte, inPacket.getLength()));

        SnmpPDU receivedPDU = null;

        SNMPTLV nextTLV = null;

        nextTLV = receivedSNMPMessage.extractNextTLV(receiveDataByte, 0);
        receivedSNMPMessage.setTLV(nextTLV);
        receivedPDU = parseSNMPMessage(receivedSNMPMessage);

        retrievedVars = parsePDU(receivedPDU, requestID, requestOID);

        return retrievedVars;
    }

    /**
     * SNMPbZ[W͂SNMP PDUԂ܂B
     *
     * @param receivedSNMPMessage SNMPbZ[W
     * @return receivedPDU SnmpPDU
     * @throws SnmpBadValueException f[^͂ňُ킪ꍇ
     */
    public SnmpPDU parseSNMPMessage(SnmpMessage receivedSNMPMessage)
        throws SnmpBadValueException {
        SnmpPDU receivedPDU = null;

        try {
            /* o[Wcommunity𒊏o */
            byte[] versionByte = receivedSNMPMessage.getVersion();
            int version = versionByte[0];
            byte[] communityByte = receivedSNMPMessage.getCommunity();
            String snmpVersion = getVesionDiscrip(version);

            // logger.info("Version : " + version + "(" + snmpVersion + ")");
            // logger.info("Community : " + bytetoChar(communityByte));
            SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
                "Version : " + version + " (" + snmpVersion + ")");
            SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
                "Community : " + bytetoChar(communityByte));

            // PDUo
            receivedPDU = receivedSNMPMessage.getPDU();
        } catch (SnmpBadValueException e) {
            e.printStackTrace();
            logger.warning("Receive PDU parse error : " + e.getMessage());
            throw new SnmpBadValueException("Receive PDU parse error");
        }

        if (receivedPDU == null) {
            logger.warning("Receive PDU parse error for PDU null");
            throw new SnmpBadValueException(
                "Receive PDU parse error for PDU null");
        }

        return receivedPDU;
    }

    /**
     * w肳ꂽSnmp PDUf[^͂Snmp verBindlistƂĕԂ܂B
     *
     * @param receivedPDU ͂PDU
     * @param requestID NGXgID
     * @param requestOID vOID
     * @return SNMPVarBindList PDU͂SNMPVarBindList
     * @throws SnmpUnmuchRequestIDException MʎqԍMʎqԍƈvȂꍇ
     * @throws SnmpErrorStatusException G[Xe[^XłȂꍇ
     */
    public SnmpVarBindList parsePDU(
        SnmpPDU receivedPDU, int requestID, String requestOID)
        throws SnmpUnmuchRequestIDException, SnmpErrorStatusException {
        byte command = receivedPDU.getPDUType();
        String snmpCode = getCommandDescription(command);

        // logger.info("Command : " + hexByte(command) + " (" + snmpCode + ")");
        SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
            "Command : " + hexByte(command) + " (" + snmpCode + ")");

        //
        SnmpVarBindList retrievedVars = new SnmpVarBindList();

        if (receivedPDU.getRequestID() == requestID) {
            this.checkErrorStatus(receivedPDU, requestOID);

            SnmpSequence varList;
            varList = receivedPDU.getVarBindList();

            SnmpSequence receiveSnmpPair =
                (SnmpSequence) (varList.getSNMPObjectAt(0));

            retrievedVars.clearSNMPObjectList();
            retrievedVars.addSNMPObject(receiveSnmpPair);

            /*
             * Agentf[^j^[\L[ɃZbg
             */
            SnmpObjectIdentifier responseOID =
                (SnmpObjectIdentifier) receiveSnmpPair.getSNMPObjectAt(0);
            AbstractSnmpObject responseValue =
                receiveSnmpPair.getSNMPObjectAt(1);
            String responseTypeString = responseValue.getTagCode();
            SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
                "OID : " + responseOID);
            SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
                "Value : " + responseValue);
            SnmpManagerQueue.getInstance().pushSnmpManagerQueue(
                "Type : " + responseTypeString);
        } else {
            /*
             * RequestID Unmuch
             */
            SnmpSequence varList;
            varList = receivedPDU.getVarBindList();

            /* vJEgAbv */
            SnmpStatisticsManager.getInstance().addSnmpUnmuchRequestID(
                hostAddress.getHostAddress());

            varList = receivedPDU.getVarBindList();

            SnmpSequence newPair = (SnmpSequence) (varList.getSNMPObjectAt(0));

            SnmpObjectIdentifier newObjectIdentifier =
                (SnmpObjectIdentifier) (newPair.getSNMPObjectAt(0));

            logger.warning(
                "Unmuch Request ID : receive ID=" + receivedPDU.getRequestID()
                + " expected ID=" + requestID + " Address="
                + hostAddress.toString() + " receive OID="
                + newObjectIdentifier + " expected OID=" + requestOID);

            throw new SnmpUnmuchRequestIDException(
                "Unmuch Request ID : receive ID=" + receivedPDU.getRequestID()
                + " expected ID=" + requestID + " Address="
                + hostAddress.toString());
        }

        return retrievedVars;
    }

    /**
     * MPDŨG[Xe[^X`FbNāAłȂꍇ
     * SnmpErrorStatusExceptionOX[܂B
     *
     * @param pdu SNMP PDU
     * @param requestOID OID
     * @throws SnmpErrorStatusException G[Xe[^XłȂꍇ
     */
    private void checkErrorStatus(SnmpPDU pdu, String requestOID)
        throws SnmpErrorStatusException {
        if (pdu.getErrorStatus() != SnmpPreference.ERROR_STATUS_CORRECT) {
            String msg;
            AlarmMessageMake message;

            switch (pdu.getErrorStatus()) {
            case SnmpPreference.ERROR_STATUS_FORMATERROR:
                msg = "Format error " + requestOID + ".";
                message = AlarmMessageMake.getInstance();
                message.setErrorSNMPErrorStatus(
                    hostAddress.getHostAddress(), msg);

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(msg);
                throw new SnmpErrorStatusException(msg);

            case SnmpPreference.ERROR_STATUS_TOOBIG:
                msg = "Value supplied for OID " + requestOID + " too big.";
                message = AlarmMessageMake.getInstance();
                message.setErrorSNMPErrorStatus(
                    hostAddress.getHostAddress(), msg);

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(msg);
                throw new SnmpErrorStatusException(msg);

            case SnmpPreference.ERROR_STATUS_NOSUCHNAME:
                msg = "OID " + requestOID + " nosuchname.";
                message = AlarmMessageMake.getInstance();
                message.setErrorSNMPErrorStatus(
                    hostAddress.getHostAddress(), msg);

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(msg);
                throw new SnmpErrorStatusException(msg);

            case SnmpPreference.ERROR_STATUS_BADVALUE:
                msg = "Bad value supplied for OID " + requestOID + ".";
                message = AlarmMessageMake.getInstance();
                message.setErrorSNMPErrorStatus(
                    hostAddress.getHostAddress(), msg);

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(msg);
                throw new SnmpErrorStatusException(msg);

            case SnmpPreference.ERROR_STATUS_READONLY:
                msg = "OID " + requestOID + " read-only.";
                message = AlarmMessageMake.getInstance();
                message.setErrorSNMPErrorStatus(
                    hostAddress.getHostAddress(), msg);

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(msg);
                throw new SnmpErrorStatusException(msg);

            case SnmpPreference.ERROR_STATUS_GENERROR:
                msg = "GenError OID " + requestOID;
                message = AlarmMessageMake.getInstance();
                message.setErrorSNMPErrorStatus(
                    hostAddress.getHostAddress(), msg);

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(msg);
                throw new SnmpErrorStatusException(msg);

            case SnmpPreference.ERROR_STATUS_NOACCESS:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("noAccess OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "noAccess OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_WRONG_TYPE:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("wrongType OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "wrongType OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_WRONG_LENGTH:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("wrongLength OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "wrongLength OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_WRONG_VALUE:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("wrongValue OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "wrongValue OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_NO_CREATION:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("noCreation OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "noCreation OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_INCONSISTENT_VALUE:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("inconsistentValue OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "inconsistentValue OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_RESOURCE_UNAVAILABLE:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("resourceUnavailable OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "resourceUnavailable OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_COMMIT_FAILED:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("commitFailed OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "commitFailed OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_UNDO_FAILED:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("undoFailed OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "undoFailed OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_AUTHORIZATION_ERROR:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("authorizationError OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "authorizationError OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_NO_WRITABLE:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("noWritable OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "noWritable OID " + requestOID);

            case SnmpPreference.ERROR_STATUS_INCONSISTENT_NAME:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning("inconsistentName OID " + requestOID);
                throw new SnmpErrorStatusException(
                    "inconsistentName OID " + requestOID);

            default:

                /* vJEgAbv */
                SnmpStatisticsManager.getInstance().addSnmpErrorStatus(
                    hostAddress.getHostAddress());
                logger.warning(
                    "Error setting OID " + requestOID + " error status("
                    + pdu.getErrorStatus() + ").");
                throw new SnmpErrorStatusException(
                    "Error setting OID " + requestOID + " error status("
                    + pdu.getErrorStatus() + ").");
            }
        }
    }

    private String getVesionDiscrip(int version) {
        String snmpVersion = "";

        switch (version) {
        case SnmpPreference.SNMP_VERSION_1:
            snmpVersion = "SnmpV1";

            break;

        case SnmpPreference.SNMPV2C:
            snmpVersion = "SnmpV2c";

            break;

        case SnmpPreference.SNMPV3:
            snmpVersion = "SnmpV3";

            break;

        default:
            logger.warning("Bad Snmp version: " + version);
        }

        return snmpVersion;
    }

    /*
     * SNMPR}hMOs܂B
     *
     * @param code R}hR[h
     * @return R}hDescription
     */
    private String getCommandDescription(byte code) {
        String snmpCode = "";

        switch (code) {
        case SnmpBERCodec.SNMPGETREQUEST:
            snmpCode = "GetRequest";

            break;

        case SnmpBERCodec.SNMPGETNEXTREQUEST:
            snmpCode = "GetNextRequest";

            break;

        case SnmpBERCodec.SNMPSETREQUEST:
            snmpCode = "SetRequest";

            break;

        case SnmpBERCodec.SNMPGETRESPONSE:
            snmpCode = "GetResponse";

            break;

        default:
            snmpCode = "Unknown";
            logger.warning("Undefined SNMP Command code : " + code);
        }

        return snmpCode;
    }


}
