/*
 * Created on 2004/04/01
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmp.snmpobject.message;

import netjfwatcher.snmp.messageformat.SnmpBadValueException;

import netjfwatcher.snmp.preference.SnmpBERCodec;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import java.util.StringTokenizer;
import java.util.logging.Logger;


/**
 * ASN.1 Type(^O)ɂOBJECT IDENTIFIER(^Oԍ0x06)ɑΉ
 * SNMPIuWFNg𑀍삷郁\bhNXłB
 * OIDĺAhbgeintzɊi[ĕێ܂B
 *
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class SnmpObjectIdentifier extends AbstractSnmpObject {
    /* hbg`OID̃hbgAei[ */
    private int[] digits; // array of integers

    /* MO */
    private Logger logger;

    /**
     * SNMP OIDIuWFNg𐶐܂B
     *
     */
    public SnmpObjectIdentifier() {
        logger = Logger.getLogger(this.getClass().getName());
        digits = new int[0];
        tag = SnmpBERCodec.SNMPOBJECTIDENTIFIER_TAG;
        tagDescription = SnmpBERCodec.SNMPOBJECTIDENTIFIER;
    }

    /**
     * hbg`OID񂩂hbgAeNXo[
     * intzɊi[SNMP OIDIuWFNg𐶐܂B
     *
     * @param digitString hbg`OID
     * @throws SnmpBadValueException IuWFNgňُ킪ꍇ
     */
    public SnmpObjectIdentifier(String digitString)
        throws SnmpBadValueException {
        logger = Logger.getLogger(this.getClass().getName());
        this.convertDigitString(digitString);
        tag = SnmpBERCodec.SNMPOBJECTIDENTIFIER_TAG;
        tagDescription = SnmpBERCodec.SNMPOBJECTIDENTIFIER;
    }

    /**
     * OID̃hbgei[intz񂩂SNMP OID
     * IuWFNg𐶐܂B
     *
     * @param digits OID̃hbgei[intz
     * @throws SnmpBadValueException IuWFNgɎsꍇ
     */
    public SnmpObjectIdentifier(int[] digits) throws SnmpBadValueException {
        logger = Logger.getLogger(this.getClass().getName());

        for (int i = 0; i < digits.length; i++) {
            if (digits[i] < 0) {
                throw new SnmpBadValueException(
                    "Negative value supplied for SNMPObjectIdentifier.");
            }
        }

        this.digits = digits;
        tag = SnmpBERCodec.SNMPOBJECTIDENTIFIER_TAG;
        tagDescription = SnmpBERCodec.SNMPOBJECTIDENTIFIER;
    }

    /**
     * ASN.1 BERŃGR[fBOꂽoCgz񂩂SNMP OID
     * IuWFNg𐶐܂B
     *
     * @param enc ASN.1 BERŃGR[fBOꂽoCgz
     */
    public SnmpObjectIdentifier(byte[] enc) {
        logger = Logger.getLogger(this.getClass().getName());

        /*
         * ASN.1 BERŃGR[fBOꂽoCgz񂩂OIDl𒊏oA
         * CX^Xo[intzɊi[
         */
        extractFromBEREncoding(enc);
        tag = SnmpBERCodec.SNMPOBJECTIDENTIFIER_TAG;
        tagDescription = SnmpBERCodec.SNMPOBJECTIDENTIFIER;
    }

    /**
     * wOIDSNMP OIDIuWFNgf[^ƂăZbg܂B
     *
     * @param digitString OID
     * @throws SnmpBadValueException f[^ZbgɎsꍇ
     */
    public void setSNMPObjectIdentifier(String digitString)
        throws SnmpBadValueException {
        this.convertDigitString(digitString);
    }

    /**
     * OIDhbgȂŊeintzɊi[NXo[
     * ObjectƂĕԂ܂B
     *
     *@return hbg`OID̃hbgAei[z
     */
    public Object getValue() {
        return this.digits;
    }

    /**
     * OID̃hbgeintzɊi[NXo[
     * ƂăZbgĕێ܂B
     *
     * @param digits OID̃hbgeintz
     * @throws SnmpBadValueException (p[^)IuWFNgُȏꍇ
     */
    public void setValue(Object digits) throws SnmpBadValueException {
        if (digits instanceof int[]) {
            for (int i = 0; i < ((int[]) digits).length; i++) {
                if (((int[]) digits)[i] < 0) {
                    throw new SnmpBadValueException(
                        "Negative value supplied for SNMPObjectIdentifier.");
                }
            }

            this.digits = (int[]) digits;
        } else if (digits instanceof String) {
            convertDigitString((String) digits);
        } else {
            throw new SnmpBadValueException(
                " Object Identifier: bad object supplied to set value ");
        }
    }

    /**
     * OIDASN.1 BERGR[fBOoCgzƂĕԂ܂B
     *
     * @return ASN.1 BERGR[fBOoCgz
     */
    public byte[] getBEREncoding() {
        ByteArrayOutputStream outBytes = new ByteArrayOutputStream();
        byte[] outBytesArray = null;

        try {
            byte type = SnmpBERCodec.SNMPOBJECTIDENTIFIER_TAG;

            // write contents of array of values
            byte[] data = encodeArray();

            // calculate encoding for length of data
            byte[] len = this.encodeLength(data.length);

            // encode T,L,V info
            outBytes.write(type);
            outBytes.write(len, 0, len.length);
            outBytes.write(data, 0, data.length);

            outBytesArray = outBytes.toByteArray();
        } finally {
            try {
                outBytes.close();
            } catch (IOException e) {
                logger.warning(e.getMessage());
                e.printStackTrace();
            }
        }

        return outBytesArray;
    }

    /**
     * intzŕێĂOIDlASN.1 BERɂăGR[fBOoCgz
     * ĕԂ܂B
     *
     * @return intzŕێĂOIDlASN.1 BERɂăGR[fBO
     * oCgz
     */
    private byte[] encodeArray() {
        ByteArrayOutputStream outBytes = new ByteArrayOutputStream();

        byte[] outBytesArray = null;

        try {
            int numElements = digits.length;

            /*
             * encode first two identifier digits as one byte,
             * using the 40*x + y rule;
             * of course, if only one element, just use 40*x;
             * if none, do nothing
             */
            if (numElements >= 2) {
                outBytes.write((byte) ((40 * digits[0]) + digits[1]));
            } else if (numElements == 1) {
                outBytes.write((byte) (40 * digits[0]));
            }

            for (int i = 2; i < numElements; ++i) {
                byte[] nextBytes = encodeValue(digits[i]);
                outBytes.write(nextBytes, 0, nextBytes.length);
            }

            outBytesArray = outBytes.toByteArray();
        } finally {
            try {
                outBytes.close();
            } catch (IOException e) {
                logger.warning(e.getMessage());
                e.printStackTrace();
            }
        }

        return outBytesArray;
    }

    /**
     * wintf[^ASN.1 BERɂăoCgzɂĕԂ܂B
     *
     * @param intValue ASN.1 BERintf[^
     * @return oCgz
     */
    private byte[] encodeValue(int intValue) {
        /*
         * see how many bytes are needed: each value uses just
         * 7 bits of each byte, with high-order bit functioning as
         * a continuation marker
             */
        int numBytes = 0;
        int temp = intValue;

        do {
            ++numBytes;
            temp = (int) Math.floor(temp / 128);
        } while (temp > 0);

        byte[] enc = new byte[numBytes];

        // encode lowest-order byte, without setting high bit
        enc[numBytes - 1] = (byte) (intValue % 128);

        int length = (int) Math.floor(intValue / 128);

        //.encode other bytes with high bit set
        for (int i = numBytes - 2; i >= 0; --i) {
            enc[i] = (byte) ((length % 128) + 128);
            length = (int) Math.floor(length / 128);
        }

        return enc;
    }

    /**
     * hbg`ŕ\OID񂩂hbgŕeintzɊi[
     * NXo[Ƃĕێ܂B
     *
     * @param digitString hbg`ŕ\OID
     * @throws SnmpBadValueException
     */
    private void convertDigitString(String digitString)
        throws SnmpBadValueException {
        try {
            /*
             * hbg`OIDg[NɕāAeg[N
             * i[intz쐬
             */
            StringTokenizer st = new StringTokenizer(digitString, " .");
            int size = 0;

            while (st.hasMoreTokens()) {
                // figure out how many values are in string
                size++;
                st.nextToken();
            }

            int[] returnDigits = new int[size];

            st = new StringTokenizer(digitString, " .");

            for (int i = 0; i < size; i++) {
                returnDigits[i] = Integer.parseInt(st.nextToken());

                if (returnDigits[i] < 0) {
                    throw new SnmpBadValueException(
                        " Object Identifier: bad string supplied to set value ");
                }
            }

            digits = returnDigits;

            /* if(size == 0){
                    noDigits = digitString;
            } */
        } catch (NumberFormatException e) {
            throw new SnmpBadValueException(
                " Object Identifier:"
                + " bad string supplied for object identifier value ");
        }
    }

    /**
     * ASN.1 BERGR[fBOꂽoCgf[^Value𒊏o
     * āANXo[intzɊi[܂B
     *
     * @param enc ASN.1 BERGR[fBOꂽoCgf[^
     */
    private void extractFromBEREncoding(byte[] enc) {
        // note: masks must be ints; byte internal representation issue(?)
        int bitTest = 0x80; // test for leading 1
        int highBitMask = 0x7F; // mask out high bit for value

        int numInts = 0;

        for (int i = 0; i < enc.length; i++) {
            if ((enc[i] & bitTest) == 0) { //high-order bit not set; count
                numInts++;
            }
        }

        if (numInts > 0) {
            /*
             * create new int array to hold digits;
             * since first value is 40*x + y,
             * need one extra entry in array to hold this.
                 */
            digits = new int[numInts + 1];

            int currentByte = -1; // will be incremented to 0

            int value = 0;

            // read in values 'til get leading 0 in byte
            do {
                currentByte++;
                value = (value * 128) + (enc[currentByte] & highBitMask);
            } while ((enc[currentByte] & bitTest) > 0); // implies high bit set!

            // now handle 40a + b
            digits[0] = (int) Math.floor(value / 40);
            digits[1] = value % 40;

            // now read in rest!
            for (int i = 2; i < (numInts + 1); i++) {
                // read in values 'til get leading 0 in byte
                value = 0;

                do {
                    currentByte++;
                    value = (value * 128) + (enc[currentByte] & highBitMask);
                } while ((enc[currentByte] & bitTest) > 0);

                digits[i] = value;
            }
        } else {
            // no digits; create empty digit array
            digits = new int[0];
        }
    }

    /**
     * w肳ꂽSNMP OIDIuWFNgێlƂ̃NXCX^X
     * ێlrAʂԂ܂B
     *
     * @param other SnmpObjectIdentifier
     * @return r
     */
    public boolean equalsOid(SnmpObjectIdentifier other) {
        int[] otherDigits = (int[]) (other.getValue());

        boolean areEqual = true;

        if (digits.length != otherDigits.length) {
            areEqual = false;
        } else {
            for (int i = 0; i < digits.length; i++) {
                if (digits[i] != otherDigits[i]) {
                    areEqual = false;

                    break;
                }
            }
        }

        return areEqual;
    }

    /**
     * ێĂOIDldot-separatedƂĕԂ܂B
     *
     * @return OID dot-separated
     */
    public String toString() {
        String valueString = "";

        // if (digits != null && digits.length > 0) {
        valueString += digits[0];

        for (int i = 1; i < digits.length; ++i) {
            valueString += ("." + digits[i]);
        }

        return valueString;
    }
}
