/*
 * Created on 2004/04/01
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmp.snmpobject.octetstring;

import java.util.StringTokenizer;

import netjfwatcher.snmp.messageformat.SnmpBadValueException;
import netjfwatcher.snmp.preference.SnmpBERCodec;



/**
 * ASN.1 Type(^O)ɂNSAPADDRESS(^Oԍ 0x45)ɑΉSNMPIuWFNg
 * 삷郁\bhNXŁA6oCgō\镨AhXێ܂B
 *
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class SnmpNSAPAddress extends SnmpOctetString {
    // length limited to 6 octets
    private static final int NSAADDRESS_LENGTH = 6;

    /**
     * l 0.0.0.0.0.0 SNMP NSAPADDRESSIuWFNg𐶐܂B
     */
    public SnmpNSAPAddress() {
        // initialize to 0.0.0.0.0.0
        dataArray = new byte[NSAADDRESS_LENGTH];

        for (int i = 0; i < NSAADDRESS_LENGTH; i++) {
            dataArray[i] = 0;
        }

        tag = SnmpBERCodec.SNMPNSAPADDRESS_TAG;
        tagDescription = SnmpBERCodec.SNMPNSAPADDRESS;
    }

    /**
     * dot؂̃AhXSNMP NSAPADDRESSIuWFNg𐶐܂B
     *
     * @param string AhX
     * @throws SnmpBadValueException IPAhX񂪈ُȏꍇ
     */
    public SnmpNSAPAddress(String string) throws SnmpBadValueException {
        dataArray = parseNSAPAddress(string);
        tag = SnmpBERCodec.SNMPNSAPADDRESS_TAG;
        tagDescription = SnmpBERCodec.SNMPNSAPADDRESS;
    }

    /**
     * ASN.1 BER encodingf[^ɂSNMP NSAPADDRESSIuWFNg𐶐܂B
     *
     * @param enc ASN.1 BER encodingf[^
     * @throws SnmpBadValueException f[^6łȂꍇ
     */
    public SnmpNSAPAddress(byte[] enc) throws SnmpBadValueException {
        tag = SnmpBERCodec.SNMPNSAPADDRESS_TAG;

        if (enc.length == NSAADDRESS_LENGTH) {
            dataArray = enc;
        } else {
            // wrong size
            throw new SnmpBadValueException(
                " NSAPAddress: bad BER encoding supplied to set value ");
        }
    }

    /**
     * SNMP NSAPADDRESSIuWFNgɃf[^Zbg܂B
    *
    * @param newAddress ZbgIuWFNgl
    * @throws SnmpBadValueException f[^ZbgɎsꍇ
     */
    public void setValue(Object newAddress) throws SnmpBadValueException {
        if (
            (newAddress instanceof byte[])
                && (((byte[]) newAddress).length == NSAADDRESS_LENGTH)) {
            dataArray = (byte[]) newAddress;
        } else if (newAddress instanceof String) {
            dataArray = parseNSAPAddress((String) newAddress);
        } else {
            throw new SnmpBadValueException(
                " NSAPAddress: bad length byte string supplied to set value ");
        }
    }

    /**
     * ێĂAhXf[^𕶎ɕϊĕԂ܂B
     *
     * @return returnString AhXf[^
     */
    public String toString() {
        String returnString = "";

        if (dataArray.length > 0) {
            int convert = dataArray[0];

            if (convert < 0) {
                convert += 256;
            }

            returnString += Integer.toHexString(convert);

            for (int i = 1; i < dataArray.length; i++) {
                convert = dataArray[i];

                if (convert < 0) {
                    convert += 256;
                }

                returnString += ("-" + Integer.toHexString(convert));
            }
        }

        return returnString;
    }

    /**
     * dot؂̃AhXoCgf[^ɕϊĕԂ܂B
     *
     * @param addressString AhX
     * @return returnBytes oCgf[^
     * @throws SnmpBadValueException AhX񂪈ُȏꍇ
     */
    private byte[] parseNSAPAddress(String addressString)
        throws SnmpBadValueException {
        try {
            StringTokenizer st = new StringTokenizer(addressString, " .-");
            int size = 0;

            while (st.hasMoreTokens()) {
                // figure out how many values are in string
                size++;
                st.nextToken();
            }

            if (size != NSAADDRESS_LENGTH) {
                throw new SnmpBadValueException(
                    " NSAPAddress: wrong number"
                    + " of components supplied to set value ");
            }

            byte[] returnBytes = new byte[size];

            st = new StringTokenizer(addressString, " .-");

            for (int i = 0; i < size; i++) {
                int addressComponent = (Integer.parseInt(st.nextToken(), 16));

                if ((addressComponent < 0) || (addressComponent > 255)) {
                    throw new SnmpBadValueException(
                        " NSAPAddress: invalid component supplied to set value ");
                }

                returnBytes[i] = (byte) addressComponent;
            }

            return returnBytes;
        } catch (NumberFormatException e) {
            throw new SnmpBadValueException(
                " NSAPAddress: invalid component supplied to set value ");
        }
    }
}
