/*
 * Created on 2004/10/16
 *
 *
 * Copyright(c) 2004 Yoshimasa Matsumoto
 */
package netjfwatcher.snmpagent.process;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.security.GeneralSecurityException;
import java.util.Calendar;
import java.util.Date;
import java.util.logging.Logger;

import netjfwatcher.engine.resource.SnmpV3AgentConfig;
import netjfwatcher.engine.resource.SnmpV3AgentConfigInfo;
import netjfwatcher.snmp.messageformat.SNMPTLV;
import netjfwatcher.snmp.messageformat.SnmpBadValueException;
import netjfwatcher.snmp.messageformat.SnmpTrap2PDU;
import netjfwatcher.snmp.messageformat.SnmpVarBindList;
import netjfwatcher.snmp.messageformat.SnmpVariablePair;
import netjfwatcher.snmp.mibtree.AgentMibTree;
import netjfwatcher.snmp.mibtree.MibInfo;
import netjfwatcher.snmp.preference.SnmpBERCodec;
import netjfwatcher.snmp.preference.SnmpPreference;
import netjfwatcher.snmp.preference.SnmpV3Preference;
import netjfwatcher.snmp.snmpobject.integer.SnmpInteger;
import netjfwatcher.snmp.snmpobject.message.AbstractSnmpObject;
import netjfwatcher.snmp.snmpobject.message.SnmpMessage;
import netjfwatcher.snmp.snmpobject.message.SnmpNull;
import netjfwatcher.snmp.snmpobject.message.SnmpObjectIdentifier;
import netjfwatcher.snmp.snmpobject.message.SnmpPDU;
import netjfwatcher.snmp.snmpobject.message.SnmpSequence;
import netjfwatcher.snmp.snmpobject.octetstring.SnmpOctetString;
import netjfwatcher.snmp.snmpv3.SnmpEngineID;
import netjfwatcher.snmp.snmpv3.SnmpV3ConfigurationException;
import netjfwatcher.snmp.snmpv3.UsmSecuritySubsystem;
import netjfwatcher.snmp.snmpv3.UsmUserAgentManager;
import netjfwatcher.snmp.snmpv3.UsmUserEntry;
import netjfwatcher.snmpagent.model.AgentInstanceDataMap;
import netjfwatcher.snmpagent.model.AgentInstanceDataRefreshQueue;
import netjfwatcher.snmpagent.model.AgentMonitorMessageQueue;
import netjfwatcher.snmptest.SnmpErrorTest;


/**
 * SNMP V3ɑΉ^SNMPG[WFgsNXłB
 *
 * +-----------+-----------+----------------------------+
 * | Header    |           | msgVersion                 |
 * |           |           +----------------------------+
 * |           |           | msgID                      |
 * |           |           +----------------------------+
 * |           |           | msgMaxSize                 |
 * |           |           +----------------------------+
 * |           |           | msgFlags                   |
 * |           |           +----------------------------+
 * |           |           | msgSecurityModel           |
 * +-----------+-----------+----------------------------+
 * | Security  |           | msgAuthoritativeEngineID   |
 * | Params    |           +----------------------------+
 * |           |           | msgAuthoritativeBoots      |
 * |           |           +----------------------------+
 * |           |           | msgAuthoritativeEngineTime |
 * |           |           +----------------------------+
 * |           |           | msgUserName                |
 * |           |           +----------------------------+
 * |           |           | msgAuthenticationParameters|
 * |           |           +----------------------------+
 * |           |           | msgPrivacyParameters       |
 * +-----------+-----------+----------------------------+
 * | Scoped    |           | contextEngineID            |
 * | PDU Data  |           +----------------------------+
 * |           |           | contextName                |
 * |           |           +----------------------------+
 * |           |           | PDU Data                   |
 * +-----------+-----------+----------------------------+
 *
 * @author Yoshimasa Matsumoto
 * @version 1.0
 */
public class AgentV3MP extends AbstractAgentMessageProcessing {
    /* MO */
    private static Logger logger;

    /* MmsgID */
    private static AbstractSnmpObject msgID;

    /* MmsgAuthoritativeEngineID */
    private static AbstractSnmpObject msgAuthoritativeEngineID;

    /* MmsgAuthoritativeBoots */
    private AbstractSnmpObject msgAuthoritativeBoots;

    /* MmsgAuthoritativeEngineTime */
    private AbstractSnmpObject msgAuthoritativeEngineTime;

    /* UsmUserEntry(EngineIDUser Name擾) */
    private UsmUserEntry usmUserEntry = null;

    /* Agent\񂩂Auth/Priv */
    private String authPriv;

    /* Auth Flag */
    private boolean authFlag = false;

    /* Priv Flag */
    private boolean privFlag = false;

    /* SNMP V3 Agent\ */
    private SnmpV3AgentConfigInfo snmpV3AgentConfigInfo;

    /* EngineID */
    private String snmpEngineID = "";

    /* MEngineBoots */
    private String receiveCheckEngineID = "";

    /* MEngineTime */
    private long receiveCheckEngineTime = 0;

    /* SNMP V3 Agent\ɂ郆[U */
    private String userName = "";

    /* M[U */
    private String receiveCheckUserName = "";

    /* MEngineBoots */
    private long receiveCheckBoots = 0;

    /* MNGXgmsgID */
    private int receiveMsgID = 0;

    /**
     * TrapMSNMPG[WFg𐶐܂B
     *
     * @throws SocketException Socket̐Ɏsꍇ
     */
    public AgentV3MP() throws SocketException {
        logger = Logger.getLogger(this.getClass().getName());
        this.version = SnmpPreference.SNMPV3;
    }

    /**
     * SNMPR}hMSNMPG[WFg𐶐܂B
     *
     * @param dSocket DatagramSocket
     * @throws SocketException Socket̐Ɏsꍇ
     */
    public AgentV3MP(DatagramSocket dSocket) throws SocketException {
        super();
        logger = Logger.getLogger(this.getClass().getName());
        this.version = SnmpPreference.SNMPV3;
        this.resSendDatagramSocket = dSocket;

        // this.oidHashMap = AgentOidTree.getInstance().getOidHashMap();
        // this.messagesArea = dataholder.getMessagesArea();
    }

    /**
     * Mf[^͂ăG[WFgDiscover̃`FbNs܂B
     * G[WFgDiscoverł΁AEngineID
     * EngineBoots/EngineTimeADiscoverłEngineIDA
     * EngineBoots/EngineTime̐퐫`FbNASnmp}l[W
     * Ԃ܂B
     *
     * @param inPacket f[^MDatagramPacket
     * @param encodedMessage Mf[^oCg
     * @param receiveLength Mf[^
     * @throws GeneralSecurityException ZLeB֘Aňُ킪ꍇ
     * @throws SnmpV3ConfigurationException SNMP V3\Ɉُ킪ꍇ
     * @throws SnmpBadValueException Mf[^͂ňُ킪ꍇ
     * @throws IOException MɎsꍇ
     */
    public void process(
        final DatagramPacket inPacket, final byte[] encodedMessage,
        final int receiveLength)
        throws GeneralSecurityException, SnmpV3ConfigurationException, 
            SnmpBadValueException, IOException {
        this.inPacket = inPacket;

        int receiveMsgFlags = 0;

        SnmpMessage receivedMessage = null;

        AbstractSnmpObject msgVersion = null;
        AbstractSnmpObject msgMaxSixe = null;
        AbstractSnmpObject msgFlags = null;
        AbstractSnmpObject msgSecurityModel = null;
        AbstractSnmpObject msgUserName = null;
        AbstractSnmpObject msgAuthenticationParameters = null;
        AbstractSnmpObject msgPrivacyParameters = null;

        /* contextEngineID */
        AbstractSnmpObject contextEngineID = null;

        /* contextName */
        AbstractSnmpObject contextName = null;

        SnmpSequence request = null;
        SnmpSequence securityParasSequence = null;

        // MR}ho
        try {
            receivedMessage = new SnmpMessage();

            SNMPTLV nextTLV = receivedMessage.extractNextTLV(encodedMessage, 0);
            receivedMessage.setTLV(nextTLV);

            request = new SnmpSequence();

            SNMPTLV nextTLV2 = request.extractNextTLV(encodedMessage, 0);

            // pPbg̃fR[h
            try {
                request.setTLV(nextTLV2);
            } catch (SnmpBadValueException e) {
                logger.warning(e.getMessage());
                throw e;
            }

            // SNMP ̃o[W
            msgVersion =
                request.getSNMPObjectAt(SnmpV3Preference.HEADER_MSG_VERSION);

            if (!(msgVersion instanceof SnmpInteger)) {
                logger.warning("Version is not INTEGER");
                throw new SnmpBadValueException("Version is not INTEGER");
            }

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgVersion : " + msgVersion.getValue());

            SnmpSequence header =
                (SnmpSequence) request.getSNMPObjectAt(
                    SnmpV3Preference.POS_HEADER_OF_MESSAGE);
            msgID = header.getSNMPObjectAt(SnmpV3Preference.HEADER_MSG_ID);
            receiveMsgID = Integer.parseInt(msgID.toString());

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgID : " + msgID.getValue());
            msgMaxSixe =
                header.getSNMPObjectAt(SnmpV3Preference.HEADER_MSG_MAXSIZE);

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgMaxSixe : " + msgMaxSixe.getValue());
            msgFlags =
                header.getSNMPObjectAt(SnmpV3Preference.HEADER_MSG_FLAGS);

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgFlags : " + msgFlags);
            msgSecurityModel =
                header.getSNMPObjectAt(SnmpV3Preference.HEADER_MSG_SEC_MODEL);

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgSecurityModel : " + msgSecurityModel.getValue());

            /* Security Params */
            SnmpOctetString securityParaOctet =
                (SnmpOctetString) request.getSNMPObjectAt(
                    SnmpV3Preference.POS_SECURITY_PARA_MESSAGE);
            SnmpSequence securityPara = null;

            try {
                securityPara =
                    new SnmpSequence((byte[]) securityParaOctet.getValue());
            } catch (SnmpBadValueException e) {
                logger.warning(
                    "Parse error at receive data : " + e.getMessage());
                AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                    "Parse error at receive data : " + e.getMessage());
                throw e;
            }

            securityParasSequence =
                (SnmpSequence) securityPara.getSNMPObjectAt(0);

            msgAuthoritativeEngineID =
                securityParasSequence.getSNMPObjectAt(
                    SnmpV3Preference.SECURITY_PARAMS_MSG_AUTH_ENGINEID);

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgAuthoritativeEngineID : " + msgAuthoritativeEngineID);

            if (msgAuthoritativeEngineID.toString().length() >= 2) {
                receiveCheckEngineID =
                    msgAuthoritativeEngineID.toString().substring(2);
            }

            msgAuthoritativeBoots =
                securityParasSequence.getSNMPObjectAt(
                    SnmpV3Preference.SECURITY_PARAMS_MSG_AUTH_ENGINEBOOTS);

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgAuthoritativeBoots : " + msgAuthoritativeBoots);
            receiveCheckBoots =
                Long.parseLong(msgAuthoritativeBoots.toString());

            msgAuthoritativeEngineTime =
                securityParasSequence.getSNMPObjectAt(
                    SnmpV3Preference.SECURITY_PARAMS_MSG_AUTH_ENGINETIME);
            receiveCheckEngineTime =
                Long.parseLong(msgAuthoritativeEngineTime.toString());

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgAuthoritativeEngineTime : " + msgAuthoritativeEngineTime);
            msgUserName =
                securityParasSequence.getSNMPObjectAt(
                    SnmpV3Preference.SECURITY_PARAMS_MSG_USERNAME);
            receiveCheckUserName = msgUserName.toString();

            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "msgUserName : " + msgUserName);

            msgAuthenticationParameters =
                securityParasSequence.getSNMPObjectAt(
                    SnmpV3Preference.SECURITY_PARAMS_MSG_AUTH_PARAMS);

            logger.info("msgVersion : " + msgVersion.getValue());
            logger.info("msgID : " + msgID.getValue());
            logger.info("msgMaxSixe : " + msgMaxSixe.getValue());
            logger.info("msgFlags : " + msgFlags);
            logger.info("msgSecurityModel : " + msgSecurityModel.getValue());
            logger.info(
                "msgAuthoritativeEngineID : " + msgAuthoritativeEngineID);
            logger.info("msgAuthoritativeBoots : " + msgAuthoritativeBoots);
            logger.info(
                "msgAuthoritativeEngineTime : " + msgAuthoritativeEngineTime);
            logger.info("msgUserName : " + msgUserName);

            if (
                securityParasSequence.size() > SnmpV3Preference.SECURITY_PARAMS_MSG_PRIV_PARAMS) {
                msgPrivacyParameters =
                    securityParasSequence.getSNMPObjectAt(
                        SnmpV3Preference.SECURITY_PARAMS_MSG_PRIV_PARAMS);
            }
        } catch (SnmpBadValueException e) {
            logger.warning("Parse error of receive data : " + e.getMessage());
            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "Parse error of receive data : " + e.getMessage());
            throw e;
        }

        // G[WFg\[Xt@CsnmpEngineIDǂݏo
        snmpV3AgentConfigInfo =
            SnmpV3AgentConfig.getInstance().getSNMPV3AgentConfigFileInfo()
                             .getSnmpV3AgentResourceInfo();

        snmpEngineID = snmpV3AgentConfigInfo.getSnmpEngineID();
        userName = snmpV3AgentConfigInfo.getUserName();
        authPriv = snmpV3AgentConfigInfo.getAuthPriv();
        logger.info("Agent authPriv config : " + authPriv);

        try {
            receiveMsgFlags = Integer.parseInt(msgFlags.toString());
        } catch (NumberFormatException e3) {
            logger.warning("Illegal msgFlags : " + msgFlags);
            logger.warning("Illegal msgFlags : " + msgFlags.getTagCode());
            logger.warning("Illegal msgFlags : " + msgFlags.getValue());
            throw new SnmpBadValueException("Illegal msgFlags : " + msgFlags);
        }

        authFlag = false;
        privFlag = false;

        /* F؃`FbN */
        if (
            (receiveMsgFlags & SnmpV3Preference.SNMPV3_MSG_FLAGS_AUTH) == SnmpV3Preference.SNMPV3_MSG_FLAGS_AUTH) {
            authFlag = true;
        } else {
            authFlag = false;
        }

        /* Í`FbN */
        if (
            (receiveMsgFlags & SnmpV3Preference.SNMPV3_MSG_FLAGS_PRIV) == SnmpV3Preference.SNMPV3_MSG_FLAGS_PRIV) {
            privFlag = true;
        } else {
            privFlag = false;
        }

        /*
         * snmpEngineIDnullmsgFlagsReport(0x04)ł
         * Discovery snmpEngineID
         * łȂ΁Adiscovery snmpEngineBoots/Time
         * ߂ɔF؋yшÍw`FbN
         *
         */
        if (
            (receiveMsgFlags == SnmpV3Preference.SNMPV3_MSG_FLAGS_REPORT)
                && (((byte[]) msgAuthoritativeEngineID.getValue()).length <= 0)) {
            logger.info("Discovey snmpEngineID");
        } else {
            /*
             * Discovery snmpEngineIDłȂ߂ɁAsnmpEngineID
             * `FbNAF؋yшÍ`FbN
             */
            if (
                (msgAuthoritativeEngineID.toString().length() >= 2)
                    && !snmpEngineID.equals(
                        msgAuthoritativeEngineID.toString().substring(2))) {
                /*
                     * Unknown EngineID
                     */
                logger.warning(
                    "Unmuch EngineID " + "Agent EngineID=" + snmpEngineID
                    + "Receive EngineID=" + msgAuthoritativeEngineID);
                AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                    "Unmuch EngineID " + "Agent EngineID=" + snmpEngineID
                    + "Receive EngineID=" + msgAuthoritativeEngineID);

                AgentV3Report report = new AgentV3Report();
                SnmpPDU reportPDU =
                    report.sendUnknownEngineIDReport(receiveMsgID);
                this.sendReportPDU(inPacket, msgID.toString(), reportPDU);

                return;
            }

            /* F؃`FbN */
            if (authFlag) {
                // F؂͐邩
                usmUserEntry =
                    UsmUserAgentManager.getInstance().getUsmUserEntry(
                        new SnmpEngineID(snmpEngineID), userName);

                if (usmUserEntry == null) {
                    logger.warning(
                        "Notfound " + userName + "@" + snmpEngineID
                        + " usmUserEntry");
                    logger.warning(
                        "Notfound " + receiveCheckUserName + "@"
                        + msgAuthoritativeEngineID + " usmUserEntry");
                }

                /* F؃`FbN */
                boolean isCheckAuth = false;

                try {
                    isCheckAuth =
                        UsmSecuritySubsystem.getInstance().checkAuth(
                            request, securityParasSequence,
                            usmUserEntry.getUsmUserAuthKey());

                    logger.info(
                        "snmpV3AgentConfigInfo.getAuthPassPhrase : "
                        + snmpV3AgentConfigInfo.getAuthPassPhrase());
                    logger.info(
                        "Agent usmUserName : " + usmUserEntry.getUsmUserName());
                    logger.info(
                        "Agent usmUserAuthProtocol : "
                        + usmUserEntry.getUsmUserAuthProtocol());
                    logger.info(
                        "Agent usmUserAuthKey : "
                        + hexBytetoString(
                            usmUserEntry.getUsmUserAuthKey().getEncoded(),
                            usmUserEntry.getUsmUserAuthKey().getEncoded().length));
                } catch (GeneralSecurityException e) {
                    logger.warning(e.getMessage());
                    throw e;
                }

                if (isCheckAuth) {
                    logger.info("CheckAuth true : " + isCheckAuth);
                } else {
                    logger.warning("Auth error");
                    AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                        "Auth error");

                    AgentV3Report report = new AgentV3Report();
                    SnmpPDU reportPDU =
                        report.sendWrongDigestsReport(receiveMsgID);
                    this.sendReportPDU(inPacket, msgID.toString(), reportPDU);

                    return;
                }
            }
        }

        if (msgAuthenticationParameters == null) {
            logger.warning("msgAuthenticationParameters Null");
        }

        if (msgPrivacyParameters == null) {
            logger.warning("msgPrivacyParameters Null");
        }

        SnmpSequence scopedPDUData = null;

        // Í`FbN
        if (privFlag) {
            SnmpOctetString scopedPDUOctet =
                (SnmpOctetString) request.getSNMPObjectAt(3);

            try {
                /* scopedPDU */
                SnmpSequence scopedPDUDataWork = null;

                scopedPDUDataWork =
                    UsmSecuritySubsystem.getInstance().decrypt(
                        (byte[]) scopedPDUOctet.getValue(),
                        (byte[]) msgPrivacyParameters.getValue(),
                        usmUserEntry.getUsmUserPrivKey());

                if (scopedPDUDataWork == null) {
                    logger.warning("Abort decrypt scopedPDU");

                    AgentV3Report report = new AgentV3Report();
                    SnmpPDU reportPDU =
                        report.sendDecryptionErrorReport(receiveMsgID);
                    this.sendReportPDU(inPacket, msgID.toString(), reportPDU);

                    return;
                }

                try {
                    scopedPDUData =
                        (SnmpSequence) scopedPDUDataWork.getSNMPObjectAt(0);
                } catch (ClassCastException e) {
                    /*
                     * Mf[^ScopedPDUData̕G[
                     * iSnmp BERGR[fBOŃG[j
                     */
                    logger.warning(
                        "Abort Snmp BER Coding by decrypt scopedPDU");

                    AgentV3Report report = new AgentV3Report();
                    SnmpPDU reportPDU =
                        report.sendDecryptionErrorReport(receiveMsgID);
                    this.sendReportPDU(inPacket, msgID.toString(), reportPDU);

                    return;
                }

                logger.info(
                    "Decrypt scopedPDUData : "
                    + this.hexBytetoString(
                        scopedPDUData.getBEREncoding(),
                        scopedPDUData.getBEREncoding().length));
            } catch (GeneralSecurityException e4) {
                logger.warning("Decrypt scopedPDUData : " + e4.getMessage());
                e4.printStackTrace();
                throw e4;
            }
        } else {
            /* ÍĂȂꍇscopedPDU */
            scopedPDUData =
                (SnmpSequence) request.getSNMPObjectAt(
                    SnmpV3Preference.POS_SCOPED_PDU_DATA);

            logger.info(
                "Plain scopedPDUData : "
                + this.hexBytetoString(
                    scopedPDUData.getBEREncoding(),
                    scopedPDUData.getBEREncoding().length));
        }

        if (scopedPDUData == null) {
            logger.warning("Scoped PDU Data null");
            throw new SnmpBadValueException("Scoped PDU Data null");
        }

        /*
         * scopedPDU
         *
         * contextEngineID
         */
        contextEngineID =
            scopedPDUData.getSNMPObjectAt(
                SnmpV3Preference.SCOPED_PDU_CONTEXT_ENGINEID);
        logger.info("contextEngineID : " + contextEngineID);
        AgentMonitorMessageQueue.getInstance().pushAgentMessage(
            "contextEngineID : " + contextEngineID);

        /*
         * contextName
         */
        contextName =
            scopedPDUData.getSNMPObjectAt(
                SnmpV3Preference.SCOPED_PDU_CONTEXT_NAME);
        logger.info("contextName : " + contextName);
        AgentMonitorMessageQueue.getInstance().pushAgentMessage(
            "contextName : " + contextName);

        /*
         * PDU Data
         */
        SnmpPDU pduData =
            (SnmpPDU) scopedPDUData.getSNMPObjectAt(
                SnmpV3Preference.SCOPED_PDU_PDU_DATA);
        logger.info(
            "PDU-Type : " + setCommandLog(pduData.getPDUType()) + "("
            + hexByte(pduData.getPDUType()) + ")");
        logger.info(
            "PDU Data"
            + this.hexBytetoString(
                pduData.getBEREncoding(), pduData.getBEREncoding().length));

        AgentMonitorMessageQueue.getInstance().pushAgentMessage(
            "PDU-Type : " + setCommandLog(pduData.getPDUType()) + "("
            + hexByte(pduData.getPDUType()) + ")");

        Date date = Calendar.getInstance().getTime();
        AgentMonitorMessageQueue.getInstance().pushAgentMessage(
            "PDU Receive date : " + dateformat.format(date)
            + " : Data length : " + pduData.getBEREncoding().length + "Byte");

        AgentMonitorMessageQueue.getInstance().pushAgentMessage(
            this.hexBytetoString(
                pduData.getBEREncoding(), pduData.getBEREncoding().length));

        /*
         * PDUf[^
         */
        this.parsePDUData(inPacket, pduData);
    }

    /**
     * MPDUf[^͂܂B
     * iFؐݒ莞͔F؍ς݁AÍݒ莞͕ς݂̎M
     * PDUf[^͂܂Bj
     * ܂AMf[^Discoverywł邩`FbN܂B
     * EngineIDAEngineBootsyEngineSysupTimel`FbNāA
     * Discoveryłꍇ́AReportԂ܂B
     *
     *
     * @param inPacket MDatagramPacket
     * @param pduData SNMP PDU
     * @throws GeneralSecurityException ZLeBňُ킪ꍇ
     * @throws SnmpBadValueException f[^͂܂̓f[^ňُ킪ꍇ
     * @throws IOException f[^MɎsꍇ
     */
    private void parsePDUData(
        final DatagramPacket inPacket, final SnmpPDU pduData)
        throws GeneralSecurityException, SnmpBadValueException, IOException {
        //
        SnmpSequence varList = null;
        SnmpSequence responseSeq = null;

        logger.info("Receive Data Parse Start");
        logger.info("PDU-Type : " + hexByte(pduData.getPDUType()));

        if (pduData.getErrorStatus() != 0) {
            logger.warning("Error status " + pduData.getErrorStatus());

            // throw new SNMPGetException(
            //     "OID " + " not available for retrieval");
        }

        varList = pduData.getVarBindList();

        int requestID = pduData.getRequestID();

        /*
         * Discovery snmpEngienID
         */
        if (
            (msgAuthoritativeEngineID == null)
                || (((byte[]) msgAuthoritativeEngineID.getValue()).length <= 0)) {
            logger.info("Discovery snmpEngienID");

            snmpV3AgentConfigInfo.setDiscoveryLearned(false);

            AgentV3DiscoverySequence discovery = new AgentV3DiscoverySequence();

            try {
                responseSeq =
                    discovery.createDiscoverySnmpEngineIDMessage(
                        requestID, (msgID.getValue()).toString());
            } catch (SnmpBadValueException e) {
                logger.info("Discovery snmpEngienID : " + e.getMessage());
                e.printStackTrace();
                throw e;
            }

            /*
             * Discovery snmpEngienID f[^M
             */
            byte[] messageEncoding = responseSeq.getBEREncoding();
            DatagramPacket outPacket =
                new DatagramPacket(
                    messageEncoding, messageEncoding.length,
                    inPacket.getAddress(), inPacket.getPort());

            try {
                resSendDatagramSocket.send(outPacket);

                /* logger.info(
                    "Report Send :"
                    + hexBytetoString(messageEncoding, outPacket.getLength()));
                    */
            } catch (IOException e1) {
                logger.info(
                    "Abort discovery snmpEngienID send report : "
                    + e1.getMessage());
                e1.printStackTrace();
                throw e1;
            }

            return;
        }

        /*
         * }l[WsnmpEngineBootsysnmpEngineTimewKς݂
         * `FbN
         * wKŊsnmpEngineBootsysnmpEngineTime0̏ꍇ͊wK
         * ̂߂̗vƔf
         */

        // if (!snmpV3AgentConfigInfo.isDiscoveryLearned()) {
        // logger.info("!snmpV3AgentConfigInfo.isDiscoveryLearned()");
        if (
            msgAuthoritativeBoots.toString().equals("0")
                && msgAuthoritativeEngineTime.toString().equals("0")) {
            /* }l[WvƂč̎OIDw肪܂܂邩ۂ
             * snmpBooys/Time@Discoveryt@[XgOID̎
             * 𔻒f
             * OIDw肪Ȃꍇ́AsnmpEngineBootsy
             * snmpTime@Discoveryv
             */
            if (varList.size() < 1) {
                /*
                 * snmpBootsysnmpEngineTimeoV[PXƂĉf[^
                 * }l[WɕԂ
                 */
                logger.info("Discovery snmpBoots and snmpEngineTime");

                AgentV3DiscoverySequence discovery =
                    new AgentV3DiscoverySequence();

                try {
                    responseSeq =
                        discovery.createDiscoveryBootsAndTimeMessage(
                            requestID,
                            msgAuthoritativeEngineID.toString().substring(2),
                            (msgID.getValue()).toString());
                } catch (GeneralSecurityException e1) {
                    logger.warning(
                        "Discovery snmpBoots and snmpEngineTime : "
                        + e1.getMessage());
                    e1.printStackTrace();
                    throw e1;
                } catch (SnmpBadValueException e1) {
                    logger.warning(
                        "Discovery snmpBoots and snmpEngineTime : "
                        + e1.getMessage());
                    e1.printStackTrace();
                    throw e1;
                }

                byte[] messageEncoding = responseSeq.getBEREncoding();

                DatagramPacket outPacket =
                    new DatagramPacket(
                        messageEncoding, messageEncoding.length,
                        inPacket.getAddress(), inPacket.getPort());

                try {
                    resSendDatagramSocket.send(outPacket);
                    logger.info(
                        "Discover SnmpV3 Boots&Time Report Send :"
                        + hexBytetoString(
                            messageEncoding, outPacket.getLength()));
                } catch (IOException e1) {
                    logger.warning(
                        "Error send report for discovery snmpBoots and snmpEngineTime : "
                        + e1.getMessage());
                    e1.printStackTrace();
                    throw e1;
                }

                return;
            }
        }

        try {
            if (checkUsm()) {
                logger.warning("USM Check error");

                return;
            }
        } catch (SnmpBadValueException e) {
            logger.warning("Error check USM : " + e.getMessage());
            e.printStackTrace();
            throw e;
        }

        /*
         * SNMP V3bZ[W𐶐A}l[WɕԂ
         *
         */
        /* wOIDɑΉSNMPv3bZ[W𐶐 */
        responseSeq =
            this.parseMessage(
                requestID, (msgID.getValue()).toString(), pduData);

        byte[] messageEncoding = responseSeq.getBEREncoding();

        DatagramPacket outPacket =
            new DatagramPacket(
                messageEncoding, messageEncoding.length, inPacket.getAddress(),
                inPacket.getPort());

        try {
            resSendDatagramSocket.send(outPacket);
            logger.info(
                "Usual Response Data Send :"
                + hexBytetoString(messageEncoding, outPacket.getLength()));
        } catch (IOException e1) {
            logger.warning("Error send report response : " + e1.getMessage());
            e1.printStackTrace();
            throw e1;
        }

        /* snmpEngineBootsysnmpEngineTimewKς݂Zbg */
        snmpV3AgentConfigInfo.setDiscoveryLearned(true);

        return;
    }

    /**
     * SNMPV[PXbZ[W𐶐ĕԂ܂B
     *
     * @param requestID NGXgʎq
     * @param msgIDString msgID
     * @param pduData SNMP PDU
     * @return sendSNMPSequence SNMPV[PXbZ[W
     * @throws GeneralSecurityException ZLeB֘Aňُ킪ꍇ
     * @throws SnmpBadValueException bZ[Wňُ킪ꍇ
     */
    public SnmpSequence parseMessage(
        final int requestID, final String msgIDString, final SnmpPDU pduData)
        throws GeneralSecurityException, SnmpBadValueException {
        SnmpSequence varList = new SnmpSequence();
        varList = pduData.getVarBindList();

        int snmpVersion = SnmpPreference.SNMPV3;
        int securityModel = SnmpV3Preference.SECURITY_MODEL_USM;
        int msgID = Integer.parseInt(msgIDString);

        // G[WFg\[Xt@CsnmpEngineIDǂݏo
        SnmpV3AgentConfigInfo snmpV3AgentConfigInfo =
            SnmpV3AgentConfig.getInstance().getSNMPV3AgentConfigFileInfo()
                             .getSnmpV3AgentResourceInfo();
        String snmpV3AgentSnmpEngineID =
            snmpV3AgentConfigInfo.getSnmpEngineID();
        SnmpEngineID snmpEngineID = new SnmpEngineID(snmpV3AgentSnmpEngineID);
        int snmpEngineBoots = 0;

        String contextName = snmpV3AgentConfigInfo.getContextName();

        /* SNMPV[PXbZ[W */
        SnmpSequence sendSNMPSequence = new SnmpSequence();
        SnmpSequence msgSecurityParameters = null;

        try {
            // Header
            // msgVersion
            sendSNMPSequence.addSNMPObject(new SnmpInteger(snmpVersion));

            // msgGlobalData
            SnmpSequence msgGlobalData = new SnmpSequence();

            // msgID
            msgGlobalData.addSNMPObject(new SnmpInteger(msgID));

            // msgMaxSize
            msgGlobalData.addSNMPObject(
                new SnmpInteger(SnmpV3Preference.SNMP_V3_MSG_MAXSIZE));

            // msgFlagsreport flagZbg
            byte msgFlag = (byte) 0x0;

            if (authFlag) {
                msgFlag |= SnmpV3Preference.SNMPV3_MSG_FLAGS_AUTH;
            }

            if (privFlag) {
                msgFlag |= SnmpV3Preference.SNMPV3_MSG_FLAGS_PRIV;
            }

            msgGlobalData.addSNMPObject(
                new SnmpOctetString(new byte[] { msgFlag }));

            // msgSecurityModel
            msgGlobalData.addSNMPObject(new SnmpInteger(securityModel));
            sendSNMPSequence.addSNMPObject(msgGlobalData);

            logger.info(
                "Discovery Learned : "
                + snmpV3AgentConfigInfo.isDiscoveryLearned());
            logger.info("authFlag : " + authFlag);
            logger.info("privFlag : " + privFlag);
            logger.info("msgFlag : " + msgFlag);

            /* msgSecurityParameters
             *
             */
            msgSecurityParameters = new SnmpSequence();

            // msgAuthoritativeEngineID
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(snmpEngineID.getId()));

            // msgAuthoritativeBoots
            msgSecurityParameters.addSNMPObject(
                new SnmpInteger(snmpEngineBoots));

            // msgAuthoritativeEngineTime
            int snmpEngineTime =
                (int) ((System.currentTimeMillis()
                - ThreadAgent.getInstance().getStartSnmpAgentEngineTime()) / 1000);
            msgSecurityParameters.addSNMPObject(
                new SnmpInteger(snmpEngineTime));

            // msgUserName
            logger.info("userName : " + userName);
            msgSecurityParameters.addSNMPObject(new SnmpOctetString(userName));

            // msgAuthenticationParameters
            if (authFlag) {
                msgSecurityParameters.addSNMPObject(
                    new SnmpOctetString(
                        new byte[SnmpV3Preference.MSG_AUTHENTICATION_PARAM_MSG_DIGEST_LEN]));
            } else {
                msgSecurityParameters.addSNMPObject(
                    new SnmpOctetString(new byte[0]));
            }

            byte[] iv = null;

            // msgPrivacyParameters
            if (privFlag) {
                iv = UsmSecuritySubsystem.getInstance().getIV(snmpEngineBoots);
                msgSecurityParameters.addSNMPObject(
                    new SnmpOctetString(
                        UsmSecuritySubsystem.getInstance().getSalt(
                            usmUserEntry.getUsmUserPrivKey(), iv)));
            } else {
                msgSecurityParameters.addSNMPObject(
                    new SnmpOctetString(new byte[0]));
            }

            sendSNMPSequence.addSNMPObject(
                new SnmpOctetString(msgSecurityParameters.getBEREncoding()));

            // msgData
            SnmpSequence msgData = new SnmpSequence();

            // contextEngineID
            msgData.addSNMPObject(new SnmpOctetString(snmpEngineID.getId()));

            // contextName
            msgData.addSNMPObject(new SnmpOctetString(contextName));

            SnmpPDU sendPDU = null;
            byte command = 0;
            command = pduData.getPDUType();

            switch (command) {
            case SnmpBERCodec.SNMPGETREQUEST:

                /*
                 * MPDU͂ĉPDUf[^쐬
                 *
                 */
                sendPDU = createResponsePDU(requestID, varList);

                break;

            case SnmpBERCodec.SNMPGETNEXTREQUEST:

                /*
                 * MPDU͂ĉPDUf[^쐬
                 *
                 */
                sendPDU = createResponsePDU(requestID, varList);

                break;

            case SnmpBERCodec.SNMPSETREQUEST:
                sendPDU = setRequest(requestID, varList);

                break;

            default:
                logger.warning("Not defined : " + command);
            }

            /* PDU牞bZ[W𐶐 */
            msgData.addSNMPObject(sendPDU);

            logger.info("authFlag : " + authFlag);
            logger.info("privFlag : " + privFlag);
            logger.info(
                "isDiscoveryLearned : "
                + snmpV3AgentConfigInfo.isDiscoveryLearned());

            // Scoped PDU DataÍ邩
            if (privFlag) {
                try {
                    if (
                        (SnmpErrorTest.getInstance().getAgentErrorControlCode() != null)
                            && SnmpErrorTest.getInstance()
                                                .getAgentErrorControlCode()
                                                .equals(
                                SnmpPreference.DECRYPTION_ERROR_DESCRIPTION)) {
                        logger.info("Error test Decryption error");
                        sendSNMPSequence.addSNMPObject(
                            new SnmpOctetString(msgData.getBEREncoding()));
                    } else {
                        sendSNMPSequence.addSNMPObject(
                            new SnmpOctetString(
                                UsmSecuritySubsystem.encrypt(
                                    msgData, iv,
                                    usmUserEntry.getUsmUserPrivKey())));
                    }
                } catch (GeneralSecurityException e2) {
                    logger.warning(
                        "Create encrypt Scoped PDU Data : " + e2.getMessage());
                    e2.printStackTrace();
                    throw e2;
                }
            } else {
                logger.info("Create plain Scoped PDU Data");
                sendSNMPSequence.addSNMPObject(msgData);
            }

            // bZ[WF؃R[h(MAC)bZ[Wɖߍނ
            if (authFlag) {
                logger.info("Create authentication SecurityParams");

                try {
                    SnmpOctetString msgAuthenticationParametersOctet = null;

                    if (
                        (SnmpErrorTest.getInstance().getAgentErrorControlCode() != null)
                            && SnmpErrorTest.getInstance()
                                                .getAgentErrorControlCode()
                                                .equals(
                                SnmpPreference.WRONG_DIGEST_DESCRIPTION)) {
                        logger.info("Error test Wrong digest");
                        msgAuthenticationParametersOctet =
                            new SnmpOctetString();
                    } else {
                        msgAuthenticationParametersOctet =
                            UsmSecuritySubsystem.getInstance().setAuth(
                                sendSNMPSequence, msgSecurityParameters,
                                usmUserEntry.getUsmUserAuthKey());
                    }

                    msgSecurityParameters.addSNMPObject(
                        msgAuthenticationParametersOctet,
                        SnmpV3Preference.SECURITY_PARAMS_MSG_AUTH_PARAMS);
                } catch (GeneralSecurityException e2) {
                    logger.warning(
                        "Error create authentication SecurityParams : "
                        + e2.getMessage());
                    e2.printStackTrace();
                }

                sendSNMPSequence.addSNMPObject(
                    new SnmpOctetString(msgSecurityParameters.getBEREncoding()),
                    2);
            }
        } catch (SnmpBadValueException e1) {
            logger.warning("Error parse message : " + e1.getMessage());
            e1.printStackTrace();
            throw e1;
        }

        return sendSNMPSequence;
    }

    /**
     * SNMP V3Mf[^̃ZLeBe`FbN܂B
    * ZLeBeɈُoꍇɂ́AvJEgA
    * |[gSNMP}l[WɑM܂B
    *
    * @return ZLeBe`FbN
    * true :ُ팟o
    * false:ُȂ
    * @throws SnmpBadValueException |[gOIDbZ[WɎsꍇ
    * @throws IOException |[gOIDMɎsꍇ
    */
    private boolean checkUsm() throws SnmpBadValueException, IOException {
        SnmpPDU reportPduData = null;

        int snmpEngineTime =
            (int) ((System.currentTimeMillis()
            - ThreadAgent.getInstance().getStartSnmpAgentEngineTime()) / 1000);

        if (
            SnmpErrorTest.getInstance().getAgentErrorReportCode().equals(
                    SnmpPreference.WRONG_DIGEST_DESCRIPTION)) {
            logger.warning(
                SnmpPreference.WRONG_DIGEST_DESCRIPTION + " : "
                + SnmpPreference.WRONG_DIGEST);

            AgentV3Report report = new AgentV3Report();
            reportPduData = report.sendWrongDigestsReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            SnmpErrorTest.getInstance().getAgentErrorReportCode().equals(
                    SnmpPreference.DECRYPTION_ERROR_DESCRIPTION)) {
            logger.warning(
                SnmpPreference.DECRYPTION_ERROR_DESCRIPTION + " : "
                + SnmpPreference.DECRYPTION_ERROR);

            AgentV3Report report = new AgentV3Report();
            reportPduData = report.sendDecryptionErrorReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            !receiveCheckEngineID.equals(snmpEngineID)
                || SnmpErrorTest.getInstance().getAgentErrorReportCode().equals(
                    SnmpPreference.UNKNOWN_ENGINE_ID_DESCRIPTION)) {
            logger.warning(
                SnmpPreference.UNKNOWN_ENGINE_ID_DESCRIPTION + " : "
                + SnmpPreference.UNKNOWN_ENGINE_ID);
            logger.warning("Receive EngineID : " + receiveCheckEngineID);
            logger.warning("Agent EngineID : " + snmpEngineID);

            AgentV3Report report = new AgentV3Report();
            reportPduData = report.sendUnknownEngineIDReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            ((usmUserEntry =
                    UsmUserAgentManager.getInstance().getUsmUserEntry(
                        new SnmpEngineID(snmpEngineID), receiveCheckUserName)) == null)
                || SnmpErrorTest.getInstance().getAgentErrorReportCode().equals(
                    SnmpPreference.UNKNOWN_USER_NAME_DESCRIPTION)) {
            logger.warning("UnknownUserNames");
            logger.warning("receiveCheckUserName" + receiveCheckUserName);

            AgentV3Report report = new AgentV3Report();
            reportPduData = report.sendUnknownUserNamesReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);

            /* UserName`FbNiUnknownUserNamesj */

            // reportPDU = createReportPDU( new OID( "1.3.6.1.6.3.15.1.1.3.0" ), ++usmStats.usmStatsUnknownUserNames ) ;
        } else if (
            (!authFlag
                && (usmUserEntry.getUsmUserAuthProtocol() != UsmUserEntry.USM_NO_AUTH_PROTOCOL))
                || SnmpErrorTest.getInstance().getAgentErrorReportCode().equals(
                    SnmpPreference.UNSUPPORTED_SECURITY_LEVEL_DESCRIPTION)) {
            logger.warning("UnsupportedSecLevels");

            AgentV3Report report = new AgentV3Report();
            reportPduData =
                report.sendUnsupportedSecurityLevelReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            authFlag
                && (usmUserEntry.getUsmUserAuthProtocol() == UsmUserEntry.USM_NO_AUTH_PROTOCOL)) {
            logger.warning("UnsupportedSecLevels");

            AgentV3Report report = new AgentV3Report();
            reportPduData =
                report.sendUnsupportedSecurityLevelReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            !privFlag
                && (usmUserEntry.getUsmUserPrivProtocol() != UsmUserEntry.USM_NO_PRIV_PROTOCOL)) {
            logger.warning("UnsupportedSecLevels");

            AgentV3Report report = new AgentV3Report();
            reportPduData =
                report.sendUnsupportedSecurityLevelReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            privFlag
                && (usmUserEntry.getUsmUserPrivProtocol() == UsmUserEntry.USM_NO_PRIV_PROTOCOL)) {
            logger.warning("UnsupportedSecLevels");

            AgentV3Report report = new AgentV3Report();
            reportPduData =
                report.sendUnsupportedSecurityLevelReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (
            (receiveCheckBoots != Long.parseLong(
                    snmpV3AgentConfigInfo.getSnmpEngineBoots()))
                || (receiveCheckEngineTime < (snmpEngineTime
                - SnmpV3Preference.NOT_INTIME_WINDOWS))
                || (receiveCheckEngineTime > (snmpEngineTime
                + SnmpV3Preference.NOT_INTIME_WINDOWS))
                || SnmpErrorTest.getInstance().getAgentErrorReportCode().equals(
                    SnmpPreference.NOT_IN_TIME_PACKET_DESCRIPTION)) {
            logger.warning("Receive EngineBoots : " + receiveCheckBoots);
            logger.warning(
                "Agent EngineBoots : "
                + snmpV3AgentConfigInfo.getSnmpEngineBoots());
            logger.warning(
                "Receive EngineTime : " + msgAuthoritativeEngineTime.toString());
            logger.warning("Agent EngineTime : " + snmpEngineTime);

            AgentV3Report report = new AgentV3Report();
            reportPduData = report.sendNotInTimeWindowsReport(receiveMsgID);
            this.sendReportPDU(inPacket, msgID.toString(), reportPduData);
        } else if (privFlag) {
            // nothing
        }

        if (reportPduData != null) {
            return true;
        }

        return false;
    }

    /**
     * SNMP V3 SetRequestR}hMs܂B
     *
     * @param requestID NGXgID
     * @param varList MSNMPf[^
     * @throws SNMPBadValueException Mf[^͂ɎsꍇyщPFUf[^Ɏsꍇ
     */
    private SnmpPDU setRequest(int requestID, SnmpSequence varList)
        throws SnmpBadValueException {
        int errorStatus = 0;
        int errorIndex = 0;

        SnmpVarBindList varbindList = new SnmpVarBindList();
        SnmpPDU pdu = null;

        try {
            SnmpSequence newPair = (SnmpSequence) (varList.getSNMPObjectAt(0));

            SnmpObjectIdentifier setReqSnmpOID =
                (SnmpObjectIdentifier) (newPair.getSNMPObjectAt(0));
            AbstractSnmpObject setReqObject = newPair.getSNMPObjectAt(1);

            varbindList.addSNMPObject(newPair);

            errorStatus = 0;

            String setRequestOID = setReqSnmpOID.toString();

            AbstractSnmpObject responseSnmpValue = null;
            boolean found = false;

            MibInfo miboid;

            if (agentOidHashMap.containsKey(setRequestOID)) {
                // 
                miboid = (MibInfo) agentOidHashMap.get(setRequestOID);

                logger.info("retrievedID : " + setRequestOID);
                logger.info("object : " + setReqObject);
                logger.info("getInstanceNo : " + miboid.getInstanceNo());
                logger.info("getAccess : " + miboid.getAccess());

                AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                    "SetRequest OID = " + setRequestOID);
                AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                    "SetRequest Object = " + setReqObject);

                miboid.setAgentMibValue(setReqObject);
                miboid.setValue(setReqObject.toString());

                AgentMibTree.getInstance().getOidHashMap().put(
                    setRequestOID, miboid);

                miboid =
                    AgentInstanceDataMap.getInstance().getAgentInstanceDataInfo(
                        setRequestOID);
                miboid.setAgentMibValue(setReqObject);
                miboid.setValue(setReqObject.toString());

                /* G[WFgCX^XXV */
                AgentInstanceDataMap.getInstance().setAgentInstanceDataMap(
                    setRequestOID, miboid);

                /* AgentCX^X̃e[u\XVf[^L[Zbg */
                AgentInstanceDataRefreshQueue.getInstance().pushQueue(miboid);

                if (miboid.isAgentInstance()) {
                    logger.info("instance : " + miboid.getAccess());
                    logger.info(
                        "Descrip : " + miboid.getDescrip() + " access : "
                        + miboid.getAccess());
                }

                if ((miboid != null)) {
                    responseSnmpValue = setReqObject;
                    found = true;
                } else {
                    /* SetRequest OIDCX^XȂ */
                    errorStatus = 2;
                    responseSnmpValue = new SnmpNull();
                    found = true;
                }
            } else {
                /* SetRequest OID`ĂȂ */
                logger.warning("Not defined SetRequest OID : " + setRequestOID);
                errorStatus = 2;

                responseSnmpValue = new SnmpNull();
                found = true;
            }

            if (found) {
                /* PDU𐶐 */
                logger.info("message snmp oid : " + setRequestOID);

                try {
                    SnmpObjectIdentifier requestedObjectIdentifier =
                        new SnmpObjectIdentifier(setRequestOID);
                    SnmpVariablePair nextPair =
                        new SnmpVariablePair(
                            requestedObjectIdentifier, responseSnmpValue);

                    SnmpSequence varList2 = new SnmpSequence();
                    varList2.addSNMPObject(nextPair);

                    pdu = new SnmpPDU(
                            SnmpBERCodec.SNMPGETRESPONSE, requestID, errorStatus,
                            errorIndex, varList2);
                } catch (SnmpBadValueException e) {
                    logger.warning(
                        "Abort create response PDU for setRequest : "
                        + e.getMessage());
                    e.printStackTrace();
                    throw e;
                }
            }
        } catch (SnmpBadValueException e) {
            logger.warning("Abort parse SetRequest data : " + e.getMessage());
            e.printStackTrace();
            throw e;
        }

        return pdu;
    }

    /**
     * PDU𐶐܂B
     *
     * @param requestID NGXgID
     * @param varList variable binding List
     * @return SnmpPDU
     * @throws SnmpBadValueException f[^Ɏsꍇ
     */
    public SnmpPDU createResponsePDU(int requestID, SnmpSequence varList)
        throws SnmpBadValueException {
        int errorStatus = SnmpPreference.ERROR_STATUS_CORRECT;
        int errorIndex = 0;
        SnmpPDU pdu = null;

        SnmpSequence responseVarList = new SnmpSequence();
        int varListCount = varList.getSequenceContentsList().size();

        for (int contentCount = 0; contentCount < varListCount;
                contentCount++) {
            boolean isFoundOID = false;
            String requestOID = "";
            SnmpSequence newPair =
                (SnmpSequence) (varList.getSNMPObjectAt(contentCount));
            SnmpObjectIdentifier newObjectIdentifier =
                (SnmpObjectIdentifier) (newPair.getSNMPObjectAt(0));
            requestOID = newObjectIdentifier.toString();

            logger.info("Request OID = " + requestOID);
            AgentMonitorMessageQueue.getInstance().pushAgentMessage(
                "Request OID = " + requestOID);

            AbstractSnmpObject snmpValue = null;

            if (requestOID.equals("1.0")) {
                requestOID = "1.3";
            }

            String nextMibOid = null;

            if (agentOidHashMap.containsKey(requestOID)) {
                MibInfo miboid = (MibInfo) agentOidHashMap.get(requestOID);
                MibInfo nextmiboid = null;

                for (;;) {
                    miboid = (MibInfo) agentOidHashMap.get(requestOID);

                    if (miboid == null) {
                        break;
                    }

                    nextmiboid =
                        (MibInfo) agentOidHashMap.get(miboid.getNextMibOid());

                    if (nextmiboid == null) {
                        break;
                    }

                    if (nextmiboid.isAgentInstance()) {
                        // logger.info("instance : " + nextmiboid.getAccess());
                        break;
                    }

                    /* logger.info(
                        "Descrip : " + nextmiboid.getDescrip() + " access : "
                        + nextmiboid.getAccess());
                    */
                    requestOID = miboid.getNextMibOid();
                }

                // [v
                if ((miboid != null) && (nextmiboid != null)) {
                    nextMibOid = miboid.getNextMibOid();

                    // logger.info("next Mib Oid : " + nextMibOid);
                    // logger.info("message snmp value : " + snmpValue.toString());
                    snmpValue =
                        ((MibInfo) agentOidHashMap.get(nextMibOid))
                        .getAgentMibValue();
                    requestOID =
                        ((MibInfo) agentOidHashMap.get(requestOID))
                        .getNextMibOid();

                    isFoundOID = true;
                } else {
                    errorStatus = SnmpPreference.ERROR_STATUS_NOSUCHNAME;
                    snmpValue = new SnmpNull();
                    isFoundOID = true;
                }
            } else {
                errorStatus = SnmpPreference.ERROR_STATUS_NOSUCHNAME;
                logger.warning("unmuch OID : " + requestOID);

                if (nextMibOid != null) {
                    requestOID = nextMibOid;
                }

                snmpValue = new SnmpNull();
                isFoundOID = true;
            }

            if (isFoundOID) {
                logger.info("message snmp oid : " + requestOID);

                SnmpObjectIdentifier requestedObjectIdentifier =
                    new SnmpObjectIdentifier(requestOID);
                SnmpVariablePair nextPair =
                    new SnmpVariablePair(requestedObjectIdentifier, snmpValue);

                responseVarList.addSNMPObject(nextPair);
            }
        }

        try {
            pdu = new SnmpPDU(
                    SnmpBERCodec.SNMPGETRESPONSE, requestID, errorStatus,
                    errorIndex, responseVarList);
        } catch (SnmpBadValueException e) {
            logger.warning(
                "Abort create message of response PDU : " + e.getMessage());
            e.printStackTrace();
            throw e;
        }

        return pdu;
    }

    /**
      * SNMP V3 TrapMs\bhłB
      *
      * @param hostAddress MAhX
      * @param community R~jeB
      * @param pdu PDU
      * @throws GeneralSecurityException ZLeB֘Aňُ킪ꍇ
      * @throws UnknownHostException sȏꍇ
      * @throws SnmpV3ConfigurationException SNMP V3\񂪈ُȏꍇ
      * @throws SnmpBadValueException f[^ňُ킪ꍇ
      * @throws IOException MɎsꍇ
      */
    public void sendTrap(
        String hostAddress, String community, SnmpSequence pdu)
        throws GeneralSecurityException, UnknownHostException, 
            SnmpV3ConfigurationException, SnmpBadValueException, IOException {
        int msgID = 0;
        InetAddress targetAddress = null;

        try {
            targetAddress = InetAddress.getByName(hostAddress);
        } catch (UnknownHostException e) {
            logger.warning(e.getMessage());
            e.printStackTrace();
            throw e;
        }

        SnmpSequence message = createTrap3Message(msgID, (SnmpTrap2PDU) pdu);

        byte[] messageEncoding = message.getBEREncoding();
        logger.info(
            "Trap send Version3 : "
            + hexBytetoString(messageEncoding, messageEncoding.length));

        DatagramPacket outPacket =
            new DatagramPacket(
                messageEncoding, messageEncoding.length, targetAddress, trapPort);

        trapDatagramSocket.send(outPacket);
    }

    /**
     * Snmp}l[WɃ|[gOIDԐM܂B
     *
     * @param inPacket throws
     * @param requestID NGXgID
     * @param msgIDString msgID
     * @param reportPDUData |[gPDU
     * @throws IOException |[gOIDMɎsꍇ
     */
    private void sendReportPDU(
        DatagramPacket inPacket, String msgIDString, SnmpPDU reportPDUData)
        throws IOException {
        /*  */
        SnmpSequence responseSeq =
            this.createReportMessage(msgIDString, reportPDUData);

        byte[] messageEncoding = responseSeq.getBEREncoding();

        DatagramPacket outPacket =
            new DatagramPacket(
                messageEncoding, messageEncoding.length, inPacket.getAddress(),
                inPacket.getPort());

        try {
            resSendDatagramSocket.send(outPacket);
        } catch (IOException e1) {
            logger.warning("Error send report message : " + e1.getMessage());
            e1.printStackTrace();
            throw e1;
        }
    }

    /**
     * |[gOIDbZ[W𐶐܂B
     *
     * @param msgIDString msgID
     * @param reportPDUData |[gPDU
     * @return SnmpSequence
     */
    public SnmpSequence createReportMessage(
        String msgIDString, SnmpPDU reportPDUData) {
        int snmpVersion = SnmpPreference.SNMPV3;
        int securityModel = SnmpV3Preference.SECURITY_MODEL_USM;
        int msgID = Integer.parseInt(msgIDString);

        // G[WFg\[Xt@CsnmpEngineIDǂݏo
        SnmpV3AgentConfigInfo snmpV3AgentConfigInfo =
            SnmpV3AgentConfig.getInstance().getSNMPV3AgentConfigFileInfo()
                             .getSnmpV3AgentResourceInfo();
        String snmpV3AgentSnmpEngineID =
            snmpV3AgentConfigInfo.getSnmpEngineID();
        SnmpEngineID snmpEngineID = new SnmpEngineID(snmpV3AgentSnmpEngineID);
        int snmpEngineBoots = 0;

        String contextName = snmpV3AgentConfigInfo.getContextName();
        SnmpSequence sendSNMPSequence = new SnmpSequence();
        SnmpSequence msgSecurityParameters = null;

        // Header
        // msgVersion
        sendSNMPSequence.addSNMPObject(new SnmpInteger(snmpVersion));

        // msgGlobalData
        SnmpSequence msgGlobalData = new SnmpSequence();

        // msgID
        msgGlobalData.addSNMPObject(new SnmpInteger(msgID));

        // msgMaxSize
        msgGlobalData.addSNMPObject(
            new SnmpInteger(SnmpV3Preference.SNMP_V3_MSG_MAXSIZE));

        // msgFlagsreport flagZbg
        byte msgFlag = (byte) SnmpV3Preference.SNMPV3_MSG_FLAGS_REPORT;

        if (privFlag) {
            msgFlag |= SnmpV3Preference.SNMPV3_MSG_FLAGS_PRIV;
        }

        msgGlobalData.addSNMPObject(
            new SnmpOctetString(new byte[] { msgFlag }));

        // msgSecurityModel
        msgGlobalData.addSNMPObject(new SnmpInteger(securityModel));
        sendSNMPSequence.addSNMPObject(msgGlobalData);

        // msgSecurityParameters
        msgSecurityParameters = new SnmpSequence();

        // msgAuthoritativeEngineID
        msgSecurityParameters.addSNMPObject(
            new SnmpOctetString(snmpEngineID.getId()));

        // msgAuthoritativeBoots
        msgSecurityParameters.addSNMPObject(new SnmpInteger(snmpEngineBoots));

        // msgAuthoritativeEngineTime
        int snmpEngineTime =
            (int) ((System.currentTimeMillis()
            - ThreadAgent.getInstance().getStartSnmpAgentEngineTime()) / 1000);
        msgSecurityParameters.addSNMPObject(new SnmpInteger(snmpEngineTime));

        // msgUserName
        msgSecurityParameters.addSNMPObject(new SnmpOctetString(userName));

        msgSecurityParameters.addSNMPObject(new SnmpOctetString(new byte[0]));

        byte[] iv = null;

        // UsmUserEntry usmUserEntry = null;
        logger.info("usmUserEntry snmpEngineID : " + snmpEngineID);
        logger.info("usmUserEntry userName : " + userName);

        usmUserEntry =
            UsmUserAgentManager.getInstance().getUsmUserEntry(
                snmpEngineID, userName);

        // msgPrivacyParameters
        if (privFlag) {
            iv = UsmSecuritySubsystem.getInstance().getIV(snmpEngineBoots);
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(
                    UsmSecuritySubsystem.getInstance().getSalt(
                        usmUserEntry.getUsmUserPrivKey(), iv)));
        } else {
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(new byte[0]));
        }

        sendSNMPSequence.addSNMPObject(
            new SnmpOctetString(msgSecurityParameters.getBEREncoding()));

        // msgData
        SnmpSequence msgData = new SnmpSequence();

        // contextEngineID
        msgData.addSNMPObject(new SnmpOctetString(snmpEngineID.getId()));

        // contextName
        msgData.addSNMPObject(new SnmpOctetString(contextName));

        msgData.addSNMPObject(reportPDUData);

        sendSNMPSequence.addSNMPObject(msgData);

        return sendSNMPSequence;
    }

    /**
     * SNMP V3 TrapMbZ[W𐶐郁\bhłB
     *
     * @param trapMsgID msgID
     * @param trapPdu Trap PDU
     * @return sendSNMPSequence SNMPV[PXbZ[W
     * @throws GeneralSecurityException ZLeB֘ȀQꍇ
     * @throws SnmpV3ConfigurationException SNMP Agent\Ɉُ킪ꍇ
     */
    public SnmpSequence createTrap3Message(
        final int trapMsgID, final SnmpTrap2PDU trapPdu)
        throws GeneralSecurityException, SnmpV3ConfigurationException {
        int snmpVersion = SnmpPreference.SNMPV3;
        int securityModel = SnmpV3Preference.SECURITY_MODEL_USM;
        int msgID = trapMsgID;
        int snmpEngineBoots = 0;
        long agentStartTime = 0;
        String snmpV3AgentSnmpEngineID;
        SnmpEngineID snmpEngineID;
        String userName;
        String contextName;

        // \[Xt@CsnmpEngineIDǂݏo
        SnmpV3AgentConfigInfo snmpV3AgentConfigInfo =
            SnmpV3AgentConfig.getInstance().getSNMPV3AgentConfigFileInfo()
                             .getSnmpV3AgentResourceInfo();
        snmpV3AgentSnmpEngineID = snmpV3AgentConfigInfo.getSnmpEngineID();
        snmpEngineBoots =
            Integer.parseInt(snmpV3AgentConfigInfo.getSnmpEngineBoots());
        snmpEngineID = new SnmpEngineID(snmpV3AgentSnmpEngineID);
        userName = snmpV3AgentConfigInfo.getUserName();
        authPriv = snmpV3AgentConfigInfo.getAuthPriv();

        // AuthKeyyPrivKeyo߂USM UserGg𒊏o
        UsmUserEntry usmUserEntry = null;
        usmUserEntry =
            UsmUserAgentManager.getInstance().getUsmUserEntry(
                snmpEngineID, userName);

        if (usmUserEntry == null) {
            logger.info("Not found USM user entry");
            logger.info("snmpEngineID : " + snmpEngineID);
            logger.info("userName : " + userName);
            throw new SnmpV3ConfigurationException(
                "Not found USM user entry" + " snmpEngineID : " + snmpEngineID
                + " userName : " + userName);
        }

        if (authPriv.equals(SnmpV3Preference.AUTH)) {
            authFlag = true;
            privFlag = false;
        } else if (authPriv.equals(SnmpV3Preference.PRIV)) {
            authFlag = true;
            privFlag = true;
        } else if (authPriv.equals(SnmpV3Preference.NOAUTH)) {
            authFlag = false;
            privFlag = false;
        }

        contextName = snmpV3AgentConfigInfo.getContextName();

        SnmpSequence sendSNMPSequence = new SnmpSequence();
        SnmpSequence msgSecurityParameters = null;

        // Header
        // msgVersion
        sendSNMPSequence.addSNMPObject(new SnmpInteger(snmpVersion));

        // msgGlobalData
        SnmpSequence msgGlobalData = new SnmpSequence();

        // msgID
        msgGlobalData.addSNMPObject(new SnmpInteger(msgID));

        // msgMaxSize
        msgGlobalData.addSNMPObject(
            new SnmpInteger(SnmpV3Preference.SNMP_V3_MSG_MAXSIZE));

        // msgFlagsreport flagZbg
        byte msgFlag = SnmpV3Preference.SNMPV3_MSG_FLAGS_REPORT;

        if (authFlag) {
            msgFlag |= SnmpV3Preference.SNMPV3_MSG_FLAGS_AUTH;
        }

        if (privFlag) {
            msgFlag |= SnmpV3Preference.SNMPV3_MSG_FLAGS_PRIV;
        }

        msgGlobalData.addSNMPObject(
            new SnmpOctetString(new byte[] { msgFlag }));

        // msgSecurityModel
        msgGlobalData.addSNMPObject(new SnmpInteger(securityModel));
        sendSNMPSequence.addSNMPObject(msgGlobalData);

        // msgSecurityParameters
        msgSecurityParameters = new SnmpSequence();

        // msgAuthoritativeEngineID
        msgSecurityParameters.addSNMPObject(
            new SnmpOctetString(snmpEngineID.getId()));

        // msgAuthoritativeBoots
        msgSecurityParameters.addSNMPObject(new SnmpInteger(snmpEngineBoots));

        // msgAuthoritativeEngineTime
        int snmpEngineTime =
            (int) ((System.currentTimeMillis() - agentStartTime) / 1000);
        msgSecurityParameters.addSNMPObject(new SnmpInteger(snmpEngineTime));

        // msgUserName
        msgSecurityParameters.addSNMPObject(new SnmpOctetString(userName));

        // msgAuthenticationParameters
        if (authFlag) {
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(
                    new byte[SnmpV3Preference.MSG_AUTHENTICATION_PARAM_MSG_DIGEST_LEN]));
        } else {
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(new byte[0]));
        }

        byte[] iv = null;

        // msgPrivacyParameters
        if (privFlag) {
            iv = UsmSecuritySubsystem.getInstance().getIV(snmpEngineBoots);
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(
                    UsmSecuritySubsystem.getInstance().getSalt(
                        usmUserEntry.getUsmUserPrivKey(), iv)));
        } else {
            msgSecurityParameters.addSNMPObject(
                new SnmpOctetString(new byte[0]));
        }

        sendSNMPSequence.addSNMPObject(
            new SnmpOctetString(msgSecurityParameters.getBEREncoding()));

        // msgData
        SnmpSequence msgData = new SnmpSequence();

        // contextEngineID
        msgData.addSNMPObject(new SnmpOctetString(snmpEngineID.getId()));

        // contextName
        msgData.addSNMPObject(new SnmpOctetString(contextName));

        // PDU Data
        int trapRequestID = 0;
        int errorStatus = SnmpPreference.ERROR_STATUS_CORRECT;
        int errorIndex = SnmpPreference.ERROR_INDEX_CORRECT;
        trapPdu.setPDU(trapRequestID, errorStatus, errorIndex);
        msgData.addSNMPObject(trapPdu);

        // Í
        if (privFlag) {
            try {
                sendSNMPSequence.addSNMPObject(
                    new SnmpOctetString(
                        UsmSecuritySubsystem.encrypt(
                            msgData, iv, usmUserEntry.getUsmUserPrivKey())));
            } catch (GeneralSecurityException e2) {
                logger.warning(
                    "Eorror encrypt data for create trap message : "
                    + e2.getMessage());
                e2.printStackTrace();
                throw e2;
            }
        } else {
            sendSNMPSequence.addSNMPObject(msgData);
        }

        // bZ[WF؃R[h(MAC)
        if (authFlag) {
            try {
                SnmpOctetString msgAuthenticationParametersOctet = null;

                msgAuthenticationParametersOctet =
                    UsmSecuritySubsystem.getInstance().setAuth(
                        sendSNMPSequence, msgSecurityParameters,
                        usmUserEntry.getUsmUserAuthKey());

                msgSecurityParameters.addSNMPObject(
                    msgAuthenticationParametersOctet, 4);
            } catch (GeneralSecurityException e2) {
                logger.warning(
                    "Error create msgAuthenticationParameters MAC for create trap message : "
                    + e2.getMessage());
                e2.printStackTrace();
                throw e2;
            }

            sendSNMPSequence.addSNMPObject(
                new SnmpOctetString(msgSecurityParameters.getBEREncoding()), 2);
        }

        return sendSNMPSequence;
    }

    /*
     * SNMPR}hMOs܂B
     *
     * @param code R}hR[h
     */
    private String setCommandLog(final byte code) {
        String snmpCode = "";

        switch (code) {
        case SnmpBERCodec.SNMPGETREQUEST:
            snmpCode = "GetRequest";

            break;

        case SnmpBERCodec.SNMPGETNEXTREQUEST:
            snmpCode = "GetNextRequest";

            break;

        case SnmpBERCodec.SNMPSETREQUEST:
            snmpCode = "SetRequest";

            break;

        default:
            snmpCode = "Unknown";
        }

        return snmpCode;
    }
}
