package jp.sf.nikonikofw;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

import jp.sf.nikonikofw.action.DefaultActionMapping;
import jp.sf.nikonikofw.action.IActionMapping;
import jp.sf.nikonikofw.authentication.HttpSessionAuthenticationManager;
import jp.sf.nikonikofw.authentication.IAuthenticationMananger;
import jp.sf.nikonikofw.exception.InitializeException;
import jp.sf.nikonikofw.persistence.IPersistenceManager;
import jp.sf.nikonikofw.persistence.NullPersistenceManager;
import jp.sf.nikonikofw.util.StringUtils;
import jp.sf.nikonikofw.util.converter.IConverter;

/**
 * フレームワークの設定情報を保持します。
 * 
 * @author Naoki Takezoe
 */
public class Config {
	
	private static Logger logger = Logger.getLogger(Config.class.getName());
	
	public static final String CONFIG_PROP_CONVERTERS = "converters";
	
	public static final String CONFIG_PROP_PERSISTENCE_MANAGER = "persistenceManager";
	
	public static final String CONFIG_PROP_AUTHENTICATION_MANAGER = "authenticationManager";
	
	public static final String CONFIG_PROP_ROOT_PACKAGES = "rootPackages";
	
	/**
	 * URLとアクションのマッピングを行うオブジェクト
	 */
	static IActionMapping actionMapping;
	
	/**
	 * 永続マネージャ
	 */
	static IPersistenceManager persistenceManager;
	
	/**
	 * 認証マネージャ
	 */
	static IAuthenticationMananger authenticationMananger;
	
	/**
	 * ルートパッケージのリスト
	 */
	static List<String> rootPackages;
	
	/**
	 * 設定ファイルを読み込みます。
	 * {@link ControllerServlet#init(javax.servlet.ServletConfig)}から呼び出されます。
	 */
	static void initConfig(Properties props){
		// 初期値を設定
		actionMapping = new DefaultActionMapping();
		persistenceManager = new NullPersistenceManager();
		authenticationMananger = new HttpSessionAuthenticationManager();
		rootPackages = new ArrayList<String>();
		
		try {
			// コンバーターの初期化
			String converters = props.getProperty(CONFIG_PROP_CONVERTERS);
			if(StringUtils.isNotBlank(converters)){
				for(String className: converters.split(",")){
					className = className.trim();
					Class<?> clazz = Class.forName(className);
					StringUtils.registerConverter((IConverter) clazz.newInstance());
				}
				logger.info("convertersは" + converters + "で初期化されました。");
			} else {
				logger.info("convertersの設定をスキップしました。");
			}
			
			// 永続化サポートの初期化
			String persistenceManager = props.getProperty(CONFIG_PROP_PERSISTENCE_MANAGER);
			if(StringUtils.isNotBlank(persistenceManager)){
				Class<?> clazz = Class.forName(persistenceManager.trim());
				Config.persistenceManager = (IPersistenceManager) clazz.newInstance();
				Config.persistenceManager.init(props);
			}
			logger.info("persistenceManagerは" + getPersistenceManager().getClass().getName() + "で初期化されました。");
			
			// 認証サービスの初期化
			String authenticationManager = props.getProperty(CONFIG_PROP_AUTHENTICATION_MANAGER);
			if(StringUtils.isNotBlank(persistenceManager)){
				Class<?> clazz = Class.forName(authenticationManager.trim());
				IAuthenticationMananger manager = (IAuthenticationMananger) clazz.newInstance();
				Config.authenticationMananger = manager;
			}
			logger.info("authenticationManangerは" + getAuthenticationMananger().getClass().getName() + "で初期化されました。");
			
			// ルートパッケージの初期化
			String rootPackages = props.getProperty(CONFIG_PROP_ROOT_PACKAGES);
			if(StringUtils.isNotBlank(rootPackages)){
				for(String rootPackage: rootPackages.split(",")){
					rootPackage = rootPackage.trim();
					if(!rootPackage.endsWith(".")){
						rootPackage = rootPackage + ".";
					}
					Config.rootPackages.add(rootPackage);
				}
			}
			
			if(Config.rootPackages.isEmpty()){
				throw new InitializeException("ルートパッケージが指定されていません。");
			} else {
				logger.info("rootPackagesとして以下のパッケージを使用します。");
				for(String packageName: Config.rootPackages){
					logger.info(packageName);
				}
			}
			
		} catch(InitializeException ex){
			throw ex;
			
		} catch(Exception ex){
			throw new InitializeException(ex);
		}
	}
	
	public static IActionMapping getActionMapping(){
		return actionMapping;
	}
	
	public static IPersistenceManager getPersistenceManager(){
		return persistenceManager;
	}
	
	public static IAuthenticationMananger getAuthenticationMananger(){
		return authenticationMananger;
	}
	
	public static List<String> getRootPackages(){
		return rootPackages;
	}
	
}
