package jp.sf.nikonikofw.util;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 入出力関係のユーティリティメソッドを提供します。
 * 
 * @author Naoki Takezoe
 */
public class IOUtil {
	
	/**
	 * 入力ストリームから出力ストリームにデータをコピーします。
	 * 
	 * @param in コピー元の入力ストリーム。処理の終了後はクローズされます。
	 * @param out コピー先の出力ストリーム。処理の終了後はクローズされません。
	 */
	public static void copy(InputStream in, OutputStream out){
		byte[] buf = new byte[1024 * 8];
		int length = 0;
		
		try {
			while((length = in.read(buf)) != -1){
				out.write(buf, 0, length);
			}
		} catch(Exception ex){
			throw new RuntimeException(ex);
		} finally {
			closeQuietly(in);
		}
	}
	
	/**
	 * 入力ストリームからUTF-8で文字列として読み込みます。
	 * 
	 * @param in 入力ストリーム
	 * @return 入力ストリームからUTF-8で読み込んだ文字列
	 */
	public static String read(InputStream in){
		return read(in, "UTF-8");
	}
	
	/**
	 * 入力ストリームから指定した文字コードで文字列として読み込みます。
	 * 
	 * @param in 入力ストリーム
	 * @param charset 文字コード
	 * @return 入力ストリームから指定した文字コードで読み込んだ文字列
	 */
	public static String read(InputStream in, String charset){
		try {
			byte[] bytes = new byte[in.available()];
			in.read(bytes);
			
			return new String(bytes, charset);
			
		} catch(IOException ex){
			throw new RuntimeException(ex);
		} finally {
			closeQuietly(in);
		}
	}
	
	/**
	 * ストリームをクローズします。
	 * クローズに失敗した場合でも例外をスローしません。
	 * 
	 * @param closeable クローズするストリーム
	 */
	public static void closeQuietly(Closeable closeable){
		if(closeable != null){
			try {
				closeable.close();
			} catch(IOException ex){
				;
			}
		}
	}
}
