package jp.sf.nikonikofw.persistence.jdbc;

import java.lang.reflect.Modifier;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Logger;

import jp.sf.nikonikofw.Config;
import jp.sf.nikonikofw.exception.PersistenceException;

public class JdbcUtil {
	
	private static Logger logger = Logger.getLogger(JdbcUtil.class.getName());
	
	/**
	 * SQLを指定して1件のデータを取得します。
	 * 
	 * @param <T> エンティティ
	 * @param clazz エンティティクラス
	 * @param sql 検索に用いるSQL
	 * @return エンティティ（複数レコードが存在する場合は先頭のレコード、レコードが存在しない場合はnullを返します）
	 */
	public static <T> T getSingleResult(Class<T> clazz, String sql) {
		return getSingleResult(clazz, sql, (Object[]) null);
	}
	
	/**
	 * SQLを指定して1件のデータを取得します。
	 * 
	 * @param <T> エンティティ
	 * @param clazz エンティティクラス
	 * @param sql 検索に用いるSQL
	 * @param params 検索に用いるパラメータ
	 * @return エンティティ（複数レコードが存在する場合は先頭のレコード、レコードが存在しない場合はnullを返します）
	 */
	public static <T> T getSingleResult(Class<T> clazz, String sql, Object... params) {
		List<T> result = getResultList(clazz, sql, params);
		if(result.isEmpty()){
			return null;
		}
		return result.get(0);
	}
	
	/**
	 * SQLを指定して複数件のデータを取得します。
	 * 
	 * @param <T> エンティティ
	 * @param clazz エンティティクラス
	 * @param sql 検索に用いるSQL
	 * @return エンティティのリスト
	 */
	public static <T> List<T> getResultList(Class<T> clazz, String sql) {
		return getResultList(clazz, sql, (Object[]) null);
	}
	
	/**
	 * SQLを指定して複数件のデータを取得します。
	 * 
	 * @param <T> エンティティ
	 * @param clazz エンティティクラス
	 * @param sql 検索に用いるSQL
	 * @param params 検索に用いるパラメータ
	 * @return エンティティのリスト
	 */
	public static <T> List<T> getResultList(Class<T> clazz, String sql, Object... params) {
		logger.info("SQL: " + sql);
		logger.info("パラメータ: " + params);
		
		Connection conn = ((JdbcPersistenceManager) Config.getPersistenceManager()).getConnection();
		PreparedStatement stmt = null;
		ResultSet rs = null;
		List<T> result = new ArrayList<T>();
		
		TableMeta tableMeta = TableMeta.getTableMeta(clazz);
		
		try {
			stmt = conn.prepareStatement(sql);
			if(params != null){
				for(int i=0; i<params.length; i++){
					stmt.setObject(i + 1, params[i]);
				}
			}
			rs = stmt.executeQuery();
			
			List<String> columns = new ArrayList<String>();
			ResultSetMetaData meta = rs.getMetaData();
			int columnCount = meta.getColumnCount();
			for(int i=0; i<columnCount; i++){
				String columnName = meta.getColumnName(i + 1);
				columns.add(columnName);
			}
			
			while(rs.next()){
				result.add(createEntity(clazz, tableMeta, rs, columns));
			}
		} catch(SQLException ex){
			throw new PersistenceException();
		} finally {
			closeResultSet(rs);
			closeStatement(stmt);
		}
		
		return result;
	}
	
	/**
	 * 任意のSQLを実行します。
	 * 
	 * @param sql SQL
	 */
	public static void execute(String sql){
		execute(sql, (Object[]) null);
	}
	
	public static void execute(String sql, Object... params) {
		logger.info("SQL: " + sql);
		logger.info("パラメータ: " + params);
		
		Connection conn = ((JdbcPersistenceManager) Config.getPersistenceManager()).getConnection();
		PreparedStatement stmt = null;
		try {
			stmt = conn.prepareStatement(sql);
			if(params != null){
				for(int i=0; i<params.length; i++){
					if(params[i] instanceof Date){
						stmt.setTimestamp(i + 1, new Timestamp(((Date) params[i]).getTime()));
					} else {
						stmt.setObject(i + 1, params[i]);
					}
				}
			}
			stmt.executeUpdate();
		} catch(SQLException ex){
			throw new PersistenceException(ex);
		} finally {
			closeStatement(stmt);
		}
	}
	
	public static void update(Object entity) {
		TableMeta meta = TableMeta.getTableMeta(entity.getClass());
		
		List<Object> params = new ArrayList<Object>();
		StringBuilder sb = new StringBuilder();
		
		sb.append("UPDATE ").append(meta.getTableName()).append(" SET ");
		{
			int count = 0;
			for(ColumnMeta column: meta.getColumns()){
				if(!column.isPk() && !Modifier.isTransient(column.getField().getModifiers())){
					if(count != 0){
						sb.append(", ");
					}
					sb.append(column.getColumnName()).append(" = ?");
					try {
						params.add(column.getField().get(entity));
					} catch(Exception ex){
						throw new RuntimeException(ex);
					}
					count++;
				}
			}
		}
		sb.append(" WHERE ");
		{
			int count = 0;
			for(ColumnMeta column: meta.getColumns()){
				if(column.isPk()){
					if(count != 0){
						sb.append(" AND ");
					}
					sb.append(column.getColumnName()).append(" = ? ");
					try {
						params.add(column.getField().get(entity));
					} catch(Exception ex){
						throw new RuntimeException(ex);
					}
					count++;
				}
			}
			if(count == 0){
				throw new PersistenceException(
						"エンティティに主キーが設定されていません: " + entity.getClass());
			}
		}
		
		JdbcUtil.execute(sb.toString(), params.toArray());
	}
	
	public static void insert(Object entity) {
		TableMeta meta = TableMeta.getTableMeta(entity.getClass());
		
		List<Object> params = new ArrayList<Object>();
		StringBuilder sb = new StringBuilder();
		sb.append("INSERT INTO ").append(meta.getTableName()).append(" (");
		{
			int count = 0;
			for(ColumnMeta column: meta.getColumns()){
				if(!column.isPk() && !Modifier.isTransient(column.getField().getModifiers())){
					if(count != 0){
						sb.append(", ");
					}
					sb.append(column.getColumnName());
					count++;
				}
			}
		}
		sb.append(") VALUES (");
		{
			int count = 0;
			for(ColumnMeta column: meta.getColumns()){
				if(!column.isPk() && !Modifier.isTransient(column.getField().getModifiers())){
					if(count != 0){
						sb.append(", ");
					}
					sb.append("?");
					
					try {
						params.add(column.getField().get(entity));
					} catch(Exception ex){
						throw new RuntimeException(ex);
					}
					
					count++;
				}
			}
		}
		sb.append(")");
		
		JdbcUtil.execute(sb.toString(), params.toArray());
	}
	
	public static void delete(Object entity) {
		TableMeta tableMeta = TableMeta.getTableMeta(entity.getClass());
		StringBuilder sb = new StringBuilder();
		sb.append("DELETE FROM ").append(tableMeta.getTableName());
		sb.append(" WHERE ");
		
		List<Object> params = new ArrayList<Object>();
		boolean hasPrimaryKey = false;
		for(ColumnMeta columnMeta: tableMeta.getColumns()){
			if(columnMeta.isPk()){
				if(!params.isEmpty()){
					sb.append(" AND ");
				}
				sb.append(columnMeta.getColumnName()).append("=?");
				try {
					params.add(columnMeta.getField().get(entity));
				} catch(Exception ex){
					throw new PersistenceException(ex);
				}
				hasPrimaryKey = true;
			}
		}
		
		if(hasPrimaryKey == false){
			throw new PersistenceException(
					"エンティティに主キーが設定されていません: " + entity.getClass());
		}
		
		JdbcUtil.execute(sb.toString(), params.toArray());
	}
	
	private static <T> T createEntity(Class<T> clazz, 
			TableMeta tableMeta, ResultSet rs, List<String> columnNames){
		try {
			T obj = clazz.newInstance();
			
			for(ColumnMeta column: tableMeta.getColumns()){
				if(columnNames.contains(column.getColumnName())){
					Class<?> fieldType = column.getField().getType();
					if(fieldType == String.class){
						column.getField().set(obj, rs.getString(column.getColumnName()));
					} else if(fieldType == Integer.class || fieldType == Integer.TYPE){
						column.getField().set(obj, rs.getInt(column.getColumnName()));
					} else if(fieldType == Long.class || fieldType == Long.TYPE){
						column.getField().set(obj, rs.getLong(column.getColumnName()));
					} else if(fieldType == Double.class || fieldType == Double.TYPE){
						column.getField().set(obj, rs.getDouble(column.getColumnName()));
					} else if(fieldType == Date.class){
						column.getField().set(obj, rs.getTimestamp(column.getColumnName()));
					}
				}
			}
			return obj;
			
		} catch(Exception ex){
			throw new RuntimeException(ex);
		}
	}
	
	private static void closeStatement(Statement stmt){
		if(stmt != null){
			try {
				stmt.close();
			} catch(Exception ex){
			}
		}
	}
	
	private static void closeResultSet(ResultSet rs){
		if(rs != null){
			try {
				rs.close();
			} catch(Exception ex){
			}
		}
	}


}
