/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Pattern;

import net.morilib.automata.DFAState;
import net.morilib.nina.DFABuilder;
import net.morilib.nina.NinaState;
import net.morilib.nina.translate.sh.ReplaceStrangeChar;
import net.morilib.range.Interval;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/23
 */
public class NinaTranslatorJava extends AbstractNinaTranslator {

	//
	private static final Pattern RET_CONST = Pattern.compile(
			"return +([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*;");
	private static final Pattern E_CONST = Pattern.compile(
			"([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*");

	//
	private int stateNo = 0;
	private Map<DFAState<Object, ?, Void>, Integer> states =
			new IdentityHashMap
			<DFAState<Object, ?, Void>, Integer>();
	private Set<Integer> accepts = new HashSet<Integer>();
	private Stack<DFAState<Object, ?, Void>> trz =
			new Stack<DFAState<Object, ?, Void>>();

	//
	private void printintv(PrintStream out, Interval v, boolean els) {
		String s, t, a;

		s = v.isInfimumClosed() ? ">=" : ">";
		t = v.isSupremumClosed() ? "<=" : "<";
		a = els ? "\t\t\t} else if" : "\t\t\tif";
		out.format("%s($c %s %d && $c %s %d) {\n",
				a, s, ((Integer)v.getInfimumBound()).intValue(),
				t, ((Integer)v.getSupremumBound()).intValue());
	}

	//
	private void printobj(PrintStream out, Object o, boolean els) {
		String a;

		a = els ? "\t\t\t} else if" : "\t\t\tif";
		out.format("%s($c.equals(\"%s\")) {\n", a, o.toString());
	}

	//
	private void printclass(PrintStream out, Object o, boolean els) {
		String a, s;
		char c;

		a = els ? "\t\t\t} else if" : "\t\t\tif";
		if(o instanceof Character) {
			if((c = ((Character)o).charValue()) == '\\' || c == '\'') {
				s = String.format("Character.valueOf('\\%c')", c);
			} else {
				s = String.format("Character.valueOf('%c')", c);
			}
			out.format("%s(%s.equals($c)) {\n", a, s);
		} else if(E_CONST.matcher(s = o.toString()).matches()) {
			out.format("%s(%s.equals($c)) {\n", a, s);
		} else {
			out.format("%s($c instanceof %s) {\n", a, s);
		}
	}

	//
	private String outln(boolean els, PrintStream out) {
		if(els) {
			out.println("\t\t\t} else {");
			return "\t";
		} else {
			return "";
		}
	}

	//
	private void printState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		boolean els = false, el2 = true;
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa), c;
		DFABuilder.DBS a, b;
		Object o;
		String s;

		out.format("\t\tcase %d:\n", sn);
		for(Interval v : dfa.getAlphabetRanges()) {
			printintv(out, v, els);
			c = ((Integer)v.getInfimumBound()).intValue();
			if(v.isInfimumOpen())  c++;

			// print Mealy edge (DFA only)
			if((o = dfa.getLabelInt(c)) != null) {
				out.format("\t\t\t\t%s\n", o.toString());
			}

			// print next step
			d = dfa.goInt(c);
			out.format("\t\t\t\tSTATE = %d;\n", getStateNo(d));
			out.println("\t\t\t\treturn true;");
			els = true;
		}

		if(dfa instanceof DFABuilder.DBS &&
				(b = ((DFABuilder.DBS)dfa).getEnd()) != null) {
			if(els) {
				out.println("\t\t\t} else if($c < 0) {");
			} else {
				out.println("\t\t\tif($c < 0) {");
			}

			if((o = ((DFABuilder.DBS)dfa).getMealyEnd()) != null) {
				out.format("\t\t\t\t%s\n", o.toString());
			}
			out.format("\t\t\t\tSTATE = %d;\n", getStateNo(b));
			out.format("\t\t\t\treturn true;\n");
			els = true;
		}

		if(!(dfa instanceof DFABuilder.DBS)) {
			// do nothing
		} else if((b = (a = (DFABuilder.DBS)dfa).getOthers()) != null) {
			s = outln(els, out);
			if((o = a.getMealyOthers()) != null) {
				out.format("%s\t\t\t%s\n", s, o.toString());
			}
			out.format("%s\t\t\tSTATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn true;\n", s);
			el2 = false;
		} else if((b = a.getRecursive()) != null) {
			s = outln(els, out);
			out.format("%s\t\t\tSTATE = 0;\n", s);
			out.format("%s\t\t\t%s_parse(__rd, $c);\n", s,
					ReplaceStrangeChar.replace(a.getRecursiveName()));
			out.format("%s\t\t\tSTATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn true;\n", s);
			el2 = false;
		}

		if(els)  out.println("\t\t\t}");
		if(el2)  out.println("\t\t\treturn false;");
	}

	//
	private void printObjectState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		boolean els = false, el2 = true;
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa);
		DFABuilder.DBS a, b;
		Object o;
		String s;

		out.format("\t\tcase %d:\n", sn);
		for(Object p : dfa.getAlphabets()) {
			printobj(out, p, els);

			// print Mealy edge (DFA only)
			if((o = dfa.getLabel(p)) != null) {
				out.format("\t\t\t\t%s\n", o.toString());
			}

			// print next step
			d = dfa.go(p);
			out.format("\t\t\t\tSTATE = %d;\n", getStateNo(d));
			out.println("\t\t\t\treturn true;");
			els = true;
		}

		if(dfa instanceof DFABuilder.DBS &&
				(b = ((DFABuilder.DBS)dfa).getEnd()) != null) {
			if(els) {
				out.println("\t\t\t} else if($c == null) {");
			} else {
				out.println("\t\t\tif($c == null) {");
			}

			if((o = ((DFABuilder.DBS)dfa).getMealyEnd()) != null) {
				out.format("\t\t\t\t%s\n", o.toString());
			}
			out.format("\t\t\t\tSTATE = %d;\n", getStateNo(b));
			out.format("\t\t\t\treturn true;\n");
			els = true;
		}

		if(!(dfa instanceof DFABuilder.DBS)) {
			// do nothing
		} else if((b = (a = (DFABuilder.DBS)dfa).getOthers()) != null) {
			s = outln(els, out);
			if((o = ((DFABuilder.DBS)dfa).getMealyOthers()) != null) {
				out.format("%s\t\t\t%s\n", s, o.toString());
			}
			out.format("%s\t\t\tSTATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn true;\n", s);
			el2 = false;
		} else if((b = a.getRecursive()) != null) {
			s = outln(els, out);
			out.format("%s\t\t\tSTATE = 0;\n", s);
			out.format("%s\t\t\t%s_parse(__rd, $c);\n", s,
					ReplaceStrangeChar.replace(a.getRecursiveName()));
			out.format("%s\t\t\tSTATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn true;\n", s);
			el2 = false;
		}

		if(els)  out.println("\t\t\t}");
		if(el2)  out.println("\t\t\treturn false;");
	}

	//
	private void printClassState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		boolean els = false, el2 = true;
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa);
		DFABuilder.DBS a, b;
		Object o;
		String s;

		out.format("\t\tcase %d:\n", sn);
		for(Object p : dfa.getAlphabets()) {
			printclass(out, p, els);

			// print Mealy edge (DFA only)
			if((o = dfa.getLabel(p)) != null) {
				out.format("\t\t\t\t%s\n", o.toString());
			}

			// print next step
			d = dfa.go(p);
			out.format("\t\t\t\tSTATE = %d;\n", getStateNo(d));
			out.println("\t\t\t\treturn true;");
			els = true;
		}

		if(dfa instanceof DFABuilder.DBS &&
				(b = ((DFABuilder.DBS)dfa).getEnd()) != null) {
			if(els) {
				out.println("\t\t\t} else if($c == null) {");
			} else {
				out.println("\t\t\tif($c == null) {");
			}

			if((o = ((DFABuilder.DBS)dfa).getMealyEnd()) != null) {
				out.format("\t\t\t\t%s\n", o.toString());
			}
			out.format("\t\t\t\tSTATE = %d;\n", getStateNo(b));
			out.format("\t\t\t\treturn true;\n");
			els = true;
		}

		if(!(dfa instanceof DFABuilder.DBS)) {
			// do nothing
		} else if((b = (a = (DFABuilder.DBS)dfa).getOthers()) != null) {
			s = outln(els, out);
			if((o = ((DFABuilder.DBS)dfa).getMealyOthers()) != null) {
				out.format("%s\t\t\t%s\n", s, o.toString());
			}
			out.format("%s\t\t\tSTATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn true;\n", s);
			el2 = false;
		} else if((b = a.getRecursive()) != null) {
			s = outln(els, out);
			out.format("%s\t\t\tSTATE = 0;\n", s);
			out.format("%s\t\t\t%s_parse(__rd, $c);\n", s,
					ReplaceStrangeChar.replace(a.getRecursiveName()));
			out.format("%s\t\t\tSTATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn true;\n", s);
			el2 = false;
		}

		if(els)  out.println("\t\t\t}");
		if(el2)  out.println("\t\t\treturn false;");
	}

	/**
	 * 
	 * @param state
	 * @return
	 */
	private int getStateNo(DFAState<Object, ?, Void> state) {
		if(states.containsKey(state)) {
			return states.get(state);
		} else {
			states.put(state, stateNo);
			if(state.isAccepted())  accepts.add(stateNo);
			trz.push(state);
			return stateNo++;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printStates(java.io.PrintStream)
	 */
	@Override
	public void printStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		s = dfa.getInitialState();
		getStateNo(s);
		while(!trz.isEmpty()) {
			s = trz.pop();
			printState(out, s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printStates(java.io.PrintStream)
	 */
	@Override
	public void printObjectStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		s = dfa.getInitialState();
		states.put(s, stateNo++);
		trz.push(s);
		while(!trz.isEmpty()) {
			s = trz.pop();
			printObjectState(out, s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printObjectStates(java.io.PrintStream)
	 */
	@Override
	public void printClassStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		s = dfa.getInitialState();
		states.put(s, stateNo++);
		trz.push(s);
		while(!trz.isEmpty()) {
			s = trz.pop();
			printClassState(out, s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printAcceptStates(PrintStream out) {
		String d = "\t\treturn (";

		if(accepts.size() == 0) {
			out.println("\t\treturn false;");
		} else {
			for(Integer i : accepts) {
				out.print(d);
				out.format("STATE == %d", i);
				d = " ||\n\t\t\t\t";
			}
			out.println(");");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptToken(java.io.PrintStream)
	 */
	@Override
	public void printAcceptToken(PrintStream out) {
		int m = -1, q;
		NinaState n;
		String x, p;

		for(DFAState<Object, ?, Void> s : states.keySet()) {
			if(!s.isAccepted())  continue;
			p = null;
			out.format("\t\tcase %d:\n", states.get(s));
			for(Object a : s.getAccepted()) {
				if(a == null || !(a instanceof NinaState)) {
					// do nothing
				} else if((q = ((NinaState)a).getPriority()) < 0) {
					// do nothing
				} else if(q > m) {
					m = q;
				}
			}

			for(Object a : s.getAccepted()) {
				if(a == null || !(a instanceof NinaState)) {
					// do nothing
				} else if((x = (n = (NinaState)a).getLabel()) == null) {
					// do nothing
				} else if((x = x.trim()).equals("")) {
					// do nothing
				} else if(m >= 0) {
					if(n.getPriority() == m)  p = x;
				} else if(RET_CONST.matcher(x).matches()) {
					p = x;  break;
				} else {
					if(p != null) {
						getOptions().pwarn("ambiguousaccept");
					}
					p = x;
				}
			}
			out.format("\t\t\t%s\n", p != null ? p : "return $$;");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptToken(java.io.PrintStream)
	 */
	@Override
	public void printActions(PrintStream out) {
		boolean[] a = new boolean[1];
		String x, p;

		for(DFAState<Object, ?, Void> s : states.keySet()) {
			p = null;
			out.format("\t\tcase %d:\n", states.get(s));
			if((x = s.toString()) == null) {
				// do nothing
			} else if((x = x.trim()).equals("")) {
				// do nothing
			} else {
				p = x;
			}

			if(p == null) {
				out.println("\t\t\tbreak;");
			} else {
				out.format("\t\t\t%s\n",
						ReplaceAction.replace(p, "false", a));
				if(!a[0]) {
					out.println("\t\t\tbreak;");
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printImports(java.io.PrintStream)
	 */
	@Override
	public void printImports(List<String> imp, PrintStream out) {
		for(String s : imp) {
			out.format("import %s;\n", s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printIsEnd(PrintStream out) {
		Set<Integer> t = new HashSet<Integer>();
		String d = "\t\treturn (";

		for(DFAState<Object, ?, Void> s : states.keySet()) {
			if(!(s instanceof DFABuilder.DBS)) {
				// do nothing
			} else if(((DFABuilder.DBS)s).getEnd() != null) {
				t.add(states.get(s));
			}
		}

		if(t.size() == 0) {
			out.println("\t\treturn false;");
		} else {
			for(Integer i : t) {
				out.print(d);
				out.format("STATE == %d", i);
				d = " ||\n\t\t\t\t";
			}
			out.println(");");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#reportStatistics(java.io.PrintStream)
	 */
	@Override
	public void reportStatistics(PrintStream std) {
		getOptions().print("statheader");
		getOptions().print("statstates", states.size());
		getOptions().print("stataccept", accepts.size());
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#openScript()
	 */
	@Override
	protected InputStream openScript() {
		return NinaTranslator.class.getResourceAsStream(
				"/net/morilib/nina/translate/nina_template." +
				getMachine() +
				".java.sh");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#openOutput()
	 */
	@Override
	protected PrintStream openOutput() throws IOException {
		return new PrintStream(new FileOutputStream(
				getOptions().getJavaFile()), true);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#newPrototype()
	 */
	@Override
	protected AbstractNinaTranslator newPrototype() {
		return new NinaTranslatorJava();
	}

}
